#ifdef HAVE_LIBZIP
# include <string.h>
# include <errno.h>
# include <time.h>
# include <zip.h>
# include <gtk/gtk.h>
# include "cdialog.h"
# include "progressdialog.h"
# include "edv_types.h"
# include "libendeavour2-base/edv_utils.h"
# include "libendeavour2-base/edv_path.h"
# include "libendeavour2-base/edv_link.h"
# include "libendeavour2-base/edv_directory.h"
# include "libendeavour2-base/edv_vfs_obj.h"
# include "libendeavour2-base/edv_vfs_obj_stat.h"
# include "libendeavour2-base/edv_archive_obj.h"
# include "edv_utils_gtk.h"
# include "edv_progress.h"
# include "edv_archive_add.h"
# include "edv_archive_add_zip.h"
# include "endeavour2.h"
# include "edv_cfg_list.h"
# include "config.h"
#else
# include <stdio.h>
# include <stdlib.h>
# include <string.h>
# include <errno.h>
# include <signal.h>
# include <gtk/gtk.h>
# include "cfg.h"
# include "progressdialog.h"
# include "edv_types.h"
# include "libendeavour2-base/edv_utils.h"
# include "libendeavour2-base/edv_path.h"
# include "libendeavour2-base/edv_link.h"
# include "libendeavour2-base/edv_directory.h"
# include "libendeavour2-base/edv_stream.h"
# include "libendeavour2-base/edv_process.h"
# include "libendeavour2-base/edv_vfs_obj.h"
# include "libendeavour2-base/edv_vfs_obj_stat.h"
# include "libendeavour2-base/edv_archive_obj.h"
# include "edv_utils_gtk.h"
# include "edv_progress.h"
# include "edv_archive_add.h"
# include "edv_archive_add_zip.h"
# include "endeavour2.h"
# include "edv_cfg_list.h"
# include "config.h"
#endif


#ifdef HAVE_LIBZIP
typedef struct _EDVArchiveAddLibzipSourceData	EDVArchiveAddLibzipSourceData;
#define EDV_ARCHIVE_ADD_LIBZIP_SOURCE_DATA(p)	((EDVArchiveAddLibzipSourceData *)(p))
#endif


#ifdef HAVE_LIBZIP
static struct zip *edv_archive_add_libzip_open_write(
	const gchar *arch_path,
	gint *zip_error_code, gint *sys_error_code
);
static EDVArchiveAddLibzipSourceData *edv_archive_add_libzip_source_data_new(void);
static void edv_archive_add_libzip_source_data_delete(EDVArchiveAddLibzipSourceData *d);
static ssize_t edv_archive_libzip_source_cb(
	void *user_data, void *data,
	size_t len, enum zip_source_cmd cmd
);
static gint edv_archive_add_libzip_iterate_file(
	EDVCore *core,
	const gchar *arch_path,
	struct zip *archive,
	const gchar *password,
	const gchar *parent_path,
	const gchar *path,
	EDVVFSObject *obj,
	GList **new_paths_list_rtn,
	gulong *cur_size, const gulong total_size,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gint compression,
	gint *dstatus_rtn
);
static gint edv_archive_add_libzip_iterate_directory(
	EDVCore *core,
	const gchar *arch_path,
	struct zip *archive,
	const gchar *password,
	const gchar *parent_path,
	const gchar *path,
	EDVVFSObject *obj,
	GList **new_paths_list_rtn,
	gulong *cur_size, const gulong total_size,
	GtkWidget *toplevel,
	const gboolean show_progress,
	gint *dstatus_rtn
);
static gint edv_archive_add_libzip_iterate_link(
	EDVCore *core,
	const gchar *arch_path,
	struct zip *archive,
	const gchar *password,
	const gchar *parent_path,
	const gchar *path,
	EDVVFSObject *obj,
	GList **new_paths_list_rtn,
	gulong *cur_size, const gulong total_size,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gint compression,
	gint *dstatus_rtn
);
static gint edv_archive_add_libzip_iterate(
	EDVCore *core,
	const gchar *arch_path,
	struct zip *archive,
	const gchar *password,
	const gchar *parent_path,
	const gchar *path,
	GList **new_paths_list_rtn,
	gulong *cur_size, const gulong total_size,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive,
	const gint compression,
	const gboolean dereference_links,
	gint *dstatus_rtn
);
#endif	/* HAVE_LIBZIP */
gint edv_archive_add_zip(
	EDVCore *core,
	const gchar *arch_path,
	GList *tar_paths_list,
	GList **new_paths_list_rtn,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive,
	const gint compression,
	const gboolean dereference_links
);


#ifdef HAVE_LIBZIP
/*
 *	libzip source data:
 *
 *	Used in edv_archive_libzip_source_cb() to pass and record data
 *	relavent to the object being added to the archive.
 */
struct _EDVArchiveAddLibzipSourceData {
	EDVCore	*core;

	gint		*dstatus_rtn;		/* Dynamic status return */
	int		zip_error[2];		/* Zip error codes */

	GtkWidget	*toplevel;
	gboolean	show_progress;

	gchar		*arch_path;		/* Archive's path */

	GList		**new_paths_list_rtn;

	gulong		*cur_size,		/* Total size of all the
						 * objects added thus far in
						 * bytes */
			total_size;		/* Total size of all the
						 * objects to be added in
						 * bytes */

	gchar		*src_path,		/* Current object's path */
			*tar_path;		/* Path of object in the archive */
	EDVObjectType	type;			/* Current object's type */
	gulong		this_size;		/* Current object's size in
						 * bytes (same as stat_buf.st_size) */
	EDVVFSObject	*obj;			/* Current object's statistics */
	FILE		*fp;			/* Current object's stream */
	gulong		bytes_read;		/* Bytes read for the current
						 * object */
};
#endif	/* HAVE_LIBZIP */


#define EDV_ITERATION_SLEEP_MIN_US		8000l

#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)

#define ISBLANK(c)	(((c) == ' ') || ((c) == '\t'))
#define ISCR(c)		(((c) == '\n') || ((c) == '\r'))

#define FCLOSE(p)	(((p) != NULL) ? (gint)fclose(p) : -1)
#define INTERRUPT(p)	(((p) > 0) ? (gint)kill((int)(p), SIGINT) : -1)

#define FDISCARD(p)	{			\
 if((p) != NULL) {				\
  const gint fd = (gint)fileno(p);		\
  while(edv_poll_read(fd)) {			\
   if(fgetc(p) == EOF)				\
    break;					\
  }						\
 }						\
}


#ifdef HAVE_LIBZIP
/*
 *	Opens the PKZip Archive for writing.
 *
 *	Returns the handle to the PKZip Archive.
 */
static struct zip *edv_archive_add_libzip_open_write(
	const gchar *arch_path,
	gint *zip_error_code, gint *sys_error_code
)
{
	struct zip *archive;

	/* If the PKZip Archive exists but has a size of 0 bytes then
	 * it must be deleted first or else libzip will return an
	 * error stating that it is not a valid PKZip Archive
	 */
	EDVVFSObject *obj = edv_vfs_object_stat(arch_path);
	if(obj != NULL)
	{
		if(obj->size == 0l)
			(void)edv_unlink(arch_path);
		edv_vfs_object_delete(obj);
	}

	archive = zip_open(
		arch_path,
		ZIP_CREATE,
		zip_error_code
	);
	if(sys_error_code != NULL)
		*sys_error_code = errno;

	return(archive);
}

/*
 *	Creates the libzip source data.
 */
static EDVArchiveAddLibzipSourceData *edv_archive_add_libzip_source_data_new(void)
{
	return((EDVArchiveAddLibzipSourceData *)g_malloc0(
		sizeof(EDVArchiveAddLibzipSourceData)
	));
}

/*
 *	Deletes the libzip source data.
 *
 *	If the source file is still opened then it will be closed.
 */
static void edv_archive_add_libzip_source_data_delete(EDVArchiveAddLibzipSourceData *d)
{
	if(d == NULL)
		return;

	g_free(d->arch_path);
	g_free(d->src_path);
	g_free(d->tar_path);
	(void)FCLOSE(d->fp);
	edv_vfs_object_delete(d->obj);
	g_free(d);
}

/*
 *	libzip source callback.
 *
 *	For adding objects of any type to the PKZip Archive by
 *	responding to libzip commands.
 */
static ssize_t edv_archive_libzip_source_cb(
	void *user_data, void *data,
	size_t len, enum zip_source_cmd cmd
)
{
	EDVCore *core;
	CfgList *cfg_list;
	EDVArchiveAddLibzipSourceData *d = EDV_ARCHIVE_ADD_LIBZIP_SOURCE_DATA(user_data);
	if(d == NULL)
		return(-1);

	core = d->core;
	cfg_list = core->cfg_list;

	switch(cmd)
	{
	  case ZIP_SOURCE_OPEN:
		/* Previously aborted? */
		if(*d->dstatus_rtn == -4)
			return(0);	/* Return successfully opened (even though not) */

		/* Update the progress dialog to display the object
		 * being added to the archive?
		 */
		if(d->show_progress)
		{
			if(edv_archive_add_update_progress_dialog(
				cfg_list,
				d->src_path,
				d->arch_path,
				(d->total_size > 0l) ?
	((gfloat)(*d->cur_size) / (gfloat)d->total_size) : 0.0f,
				d->toplevel,
				FALSE
			) > 0)
			{
				*d->dstatus_rtn = -4;
				return(0);
			}
		}

		/* Open by object type */
		switch(d->type)
		{
		  case EDV_OBJECT_TYPE_UNKNOWN:
			if(d->zip_error != NULL)
			{
				int *pi = d->zip_error;
				pi[0] = ZIP_ER_OPEN;
				pi[1] = EINVAL;
			}
			core->last_error_ptr = "Unknown object type.";
			*d->dstatus_rtn = -2;
			return(-1);
			break;

		  case EDV_OBJECT_TYPE_FILE:
			/* Is the file already opened? */
			if(d->fp != NULL)
			{
				/* Rewind it */
				rewind(d->fp);
				d->bytes_read = 0l;
			}
			/* Is the source file known? */
			else if(d->src_path != NULL)
			{
				/* Open the source file for reading */
				d->fp = fopen((const char *)d->src_path, "rb");
				if(d->fp == NULL)
				{
					/* Return success to libzip and only indicate
					 * local error because otherwise libzip will
					 * not add subsequent objects and revert the
					 * entire archive
					 */
					const gint error_code = (gint)errno;
					gchar *msg = g_strdup_printf(
"Unable to add:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
%s.",
						d->src_path,
						d->arch_path,
						g_strerror(error_code)
					);
					edv_archive_add_set_error(core, msg);
					g_free(msg);
					*d->dstatus_rtn = -1;
					return(0);
				}
				d->bytes_read = 0l;
			}
			else
			{
				int *pi = d->zip_error;
				pi[0] = ZIP_ER_INTERNAL;
				pi[1] = EINVAL;
				core->last_error_ptr = "Source file not specified.";
				return(-1);
			}
			break;

		  case EDV_OBJECT_TYPE_DIRECTORY:
		  case EDV_OBJECT_TYPE_LINK:
		  case EDV_OBJECT_TYPE_DEVICE_BLOCK:
		  case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
		  case EDV_OBJECT_TYPE_FIFO:
		  case EDV_OBJECT_TYPE_SOCKET:
			d->bytes_read = 0l;
			break;

		  case EDV_OBJECT_TYPE_ERROR:
			/* Ignore error objects */
			d->bytes_read = 0l;
			break;
		}

		/* Add this object to the list of objects added */
		if(d->new_paths_list_rtn != NULL)
			*d->new_paths_list_rtn = g_list_append(
				*d->new_paths_list_rtn,
				STRDUP(d->src_path)
			);

		return(0);
		break;

	  case ZIP_SOURCE_READ:
		/* Previously aborted? */
		if(*d->dstatus_rtn == -4)
			return(0);	/* Return 0 bytes read (which is a success) */

		/* Report progress? */
		if(d->show_progress && ProgressDialogIsQuery())
		{
			if(edv_archive_add_update_progress_dialog(
				cfg_list,
				NULL,
				NULL,
				(d->total_size > 0l) ?
	((gfloat)(*d->cur_size) / (gfloat)d->total_size) : 0.0f,
				d->toplevel,
				FALSE
			) > 0)
			{
				*d->dstatus_rtn = -4;
				return(0);
			}
		}

		/* Read by object type */
		switch(d->type)
		{
		  case EDV_OBJECT_TYPE_UNKNOWN:
			if(d->zip_error != NULL)
			{
				int *pi = d->zip_error;
				pi[0] = ZIP_ER_READ;
				pi[1] = EINVAL;
			}
			core->last_error_ptr = "Unknown object type.";
			*d->dstatus_rtn = -2;
			return(-1);
			break;

		  case EDV_OBJECT_TYPE_FILE:
			/* Was the file opened? */
			if(d->fp != NULL)
			{
				gint units_read;

				/* Read the requested length of bytes from the
				 * file into the specified buffer
				 */
				if(len > 0)
					units_read = (gint)fread(
						data,
						sizeof(guint8),
						(size_t)len,
						d->fp
					);
				else
					units_read = 0;

				/* Read error? */
				if(units_read < 0)
				{
					/* Return success to libzip and only indicate
					 * local error because otherwise libzip will
					 * not add subsequent objects and revert the
					 * entire archive
					 */
					const gint error_code = (gint)errno;
					gchar *msg = g_strdup_printf(
"Unable to add:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
%s.",
						d->src_path,
						d->arch_path,
						g_strerror(error_code)
					);
					edv_archive_add_set_error(core, msg);
					g_free(msg);
					*d->dstatus_rtn = -1;
					return(0);	/* Return no bytes read */
				}

				/* Increment the number of bytes read for this
				 * object
				 */
				d->bytes_read += (gulong)(units_read * sizeof(guint8));

				/* Increment the total number of bytes read for
				 * all the files to be added to the archive
				 */
				*d->cur_size = (*d->cur_size) +
					(gulong)(units_read * sizeof(guint8));

				/* Return the number of bytes read */
				return((ssize_t)(units_read * sizeof(guint8)));
			}
			else
			{
				/* A file may not have been able to be opened but
				 * we should not return error to libzip because
				 * libzip will not add subsequent objects and
				 * revert the entire archive so instead return 0
				 * bytes read
				 */
				return(0);	/* Return no bytes read */
			}
			break;

		  case EDV_OBJECT_TYPE_DIRECTORY:
			return(0);
			break;

		  case EDV_OBJECT_TYPE_LINK:
			/* Is the path to the source file known? */
			if(d->src_path != NULL)
			{
				gint	bytes_read,
					value_len;
				const gulong prev_pos = d->bytes_read;
				gchar *value;

				/* Already read this or no need to? */
				if(prev_pos >= d->this_size)
					return(0);	/* Return no bytes read */

				/* Get the link's target value */
				value = edv_link_get_target(d->src_path);
				if(value == NULL)
				{
					const gint error_code = (gint)errno;
					gchar *msg = g_strdup_printf(
"Unable to add:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
%s.",
						d->src_path,
						d->arch_path,
						g_strerror(error_code)
					);
					edv_archive_add_set_error(core, msg);
					g_free(msg);
					*d->dstatus_rtn = -1;
					return(0);	/* Return no bytes read */
				}

				value_len = STRLEN(value);

				/* Copy the link's target value to libzip's
				 * read buffer and update bytes_read to indicate
				 * the number of bytes coppied to it
				 */
				if((data != NULL) && (len > 0))
				{
					const gchar	*src = value + prev_pos,
									*src_end = value + value_len;
					gchar	*tar = (gchar *)data,
							*tar_end = tar + len;

					bytes_read = 0;		/* Reset */

					while((src < src_end) && (tar < tar_end))
					{
						*tar = *src;
						tar++;
						src++;
						bytes_read++;
					}

					/* Increment the number of bytes read for
					 * this object
					 */
					d->bytes_read += (gulong)bytes_read;

					/* Increment the total number of bytes read
					 * for all the files to be added to the
					 * archive
					 */
					*d->cur_size = (*d->cur_size) + (gulong)bytes_read;
				}
				else
				{
					bytes_read = 0;
				}

				/* Delete the link's target value */
				g_free(value);

				/* Return the number of bytes read */
				return((ssize_t)bytes_read);
			}
			else
			{
				int *pi = d->zip_error;
				pi[0] = ZIP_ER_INTERNAL;
				pi[1] = EINVAL;
				core->last_error_ptr = "Source file not specified.";
				return(-1);
			}
			break;

		  case EDV_OBJECT_TYPE_DEVICE_BLOCK:
		  case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
		  case EDV_OBJECT_TYPE_FIFO:
		  case EDV_OBJECT_TYPE_SOCKET:
			return(0);
			break;

		  case EDV_OBJECT_TYPE_ERROR:
			/* Do not report any data read for error objects */
			return(0);
			break;
		}
		break;

	  case ZIP_SOURCE_CLOSE:
		/* Close the object as needed, regardless of if it was
		 * opened or not or if there was an error
		 */
		if(d->fp != NULL)
		{
			fclose(d->fp);
			d->fp = NULL;
		}

		/* Report progress only if there were no errors and
		 * and not user aborted
		 */
		if(d->show_progress && ProgressDialogIsQuery() &&
		   (*d->dstatus_rtn == 0)
		)
		{
			if(edv_archive_add_update_progress_dialog(
				cfg_list,
				NULL,
				NULL,
				(d->total_size > 0l) ?
	((gfloat)(*d->cur_size) / (gfloat)d->total_size) : 0.0f,
				d->toplevel,
				FALSE
			) > 0)
			{
				*d->dstatus_rtn = -4;
			}
		}
		return(0);
		break;

	  case ZIP_SOURCE_STAT:
		if((data != NULL) && (len >= sizeof(struct zip_stat)))
		{
			EDVVFSObject *obj = d->obj;
			struct zip_stat *zstat = (struct zip_stat *)data;

			/* Set the zip stat values
			 *
			 * Although not explicitly stated in the libzip
			 * documentation as of libzip 0.7, libzip's code
			 * suggests that only members; crc, mtime, size,
			 * comp_size and comp_method are to be set
			 */
			zstat->crc = 0;
			zstat->mtime = (time_t)obj->modify_time;
			switch(d->type)
			{
			  case EDV_OBJECT_TYPE_UNKNOWN:
				zstat->size = 0l;
				break;
			  case EDV_OBJECT_TYPE_FILE:
				zstat->size = (size_t)obj->size;
				break;
			  case EDV_OBJECT_TYPE_DIRECTORY:
				zstat->size = 0l;
				break;
			  case EDV_OBJECT_TYPE_LINK:
				zstat->size = (size_t)obj->size;
				break;
			  case EDV_OBJECT_TYPE_DEVICE_BLOCK:
			  case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
			  case EDV_OBJECT_TYPE_FIFO:
			  case EDV_OBJECT_TYPE_SOCKET:
				zstat->size = 0l;
				break;
			  case EDV_OBJECT_TYPE_ERROR:
				zstat->size = 0l;
				break;
			}
			zstat->comp_size = -1;
			zstat->comp_method = ZIP_CM_STORE;
			zstat->encryption_method = ZIP_EM_NONE;

			/* We return the size of struct zip_stat instead of 0
			 * to indicate success, even though the libzip
			 * documentation which states that we should return 0
			 * for success, this is due to the fact that libzip,
			 * actually, expects the size of struct zip_stat and
			 * 0 or -1 are both errors in this case
			 */
			return((ssize_t)sizeof(struct zip_stat));
		}
		else
		{
			int *pi = d->zip_error;
			pi[0] = ZIP_ER_INTERNAL;
			pi[1] = EINVAL;
			core->last_error_ptr =
"libzip requested to obtain statistics without providing a buffer.";
			*d->dstatus_rtn = -3;
			return(-1);
		}
		break;

	  case ZIP_SOURCE_ERROR:
		if((data != NULL) && (len >= (2 * sizeof(int))))
		{
			(void)memcpy(
				data,
				d->zip_error,
				2 * sizeof(int)
			);
			return((ssize_t)(2 * sizeof(int)));
		}
		else
		{
			int *pi = d->zip_error;
			pi[0] = ZIP_ER_INTERNAL;
			pi[1] = EINVAL;
			return(-1);
		}
		break;

	  case ZIP_SOURCE_FREE:
		edv_archive_add_libzip_source_data_delete(d);
		return(0);
		break;
	}

	return(-1);
}

/*
 *	Adds the file specified by path to the PKZip Archive.
 */
static gint edv_archive_add_libzip_iterate_file(
	EDVCore *core,
	const gchar *arch_path,
	struct zip *archive,
	const gchar *password,
	const gchar *parent_path,		/* Parent directory of the archive */
	const gchar *path,			/* The object to add to the archive */
	EDVVFSObject *obj,
	GList **new_paths_list_rtn,
	gulong *cur_size, const gulong total_size,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gint compression,
	gint *dstatus_rtn
)
{
	struct zip_source *zsrc;
	gint i;
	gchar *tar_path;
	EDVArchiveAddLibzipSourceData *d;

	/* Generate the path of the file within the archive */
	if(edv_path_is_parent(
		parent_path,
		path
	))
	{
		const char *s = path + strlen(parent_path);
		while(*s == G_DIR_SEPARATOR)
			s++;
		tar_path = g_strdup(s);
	}
	else
	{
		tar_path = g_strdup(path);
	}
	if(tar_path == NULL)
	{
		core->last_error_ptr = "Memory allocation error.";
		return(-3);
	}

	/* Create a new and set up the libzip source data */
	d = edv_archive_add_libzip_source_data_new();
	if(d == NULL)
	{
		g_free(tar_path);
		core->last_error_ptr = "Memory allocation error.";
		return(-3);
	}
	d->core = core;
	d->dstatus_rtn = dstatus_rtn;
	d->zip_error[0] = ZIP_ER_OK;
	d->zip_error[1] = 0;
	d->toplevel = toplevel;
	d->show_progress = show_progress;
	d->arch_path = STRDUP(arch_path);
	d->new_paths_list_rtn = new_paths_list_rtn;
	d->cur_size = cur_size;
	d->total_size = total_size;

	d->src_path = STRDUP(path);
	d->tar_path = STRDUP(tar_path);
	d->type = EDV_OBJECT_TYPE_FILE;
	d->this_size = obj->size;
	d->fp = NULL;
	d->obj = edv_vfs_object_copy(obj);
	d->bytes_read = 0l;

	/* Check if the object already exists in the PKZip Archive */
	i = zip_name_locate(
		archive,
		tar_path,
		0
	);

	/* Set the libzip source function */
	zsrc = zip_source_function(
		archive,
		edv_archive_libzip_source_cb,
		d
	);
	if(zsrc == NULL)
	{
		gchar *msg = g_strdup_printf(
"Unable to set the libzip source function.\n\
\n\
%s.",
			zip_strerror(archive)
		);
		edv_archive_add_set_error(core, msg);
		g_free(msg);
		edv_archive_add_libzip_source_data_delete(d);
		g_free(tar_path);
		return(-1);
	}

	/* Replace or add the file? */
	if(i > -1)
	{
		/* Replace the existing file */
		if(zip_replace(
			archive,
			i,
			zsrc
		) < 0)
		{
			gchar *msg = g_strdup_printf(
"Unable to replace the PKZip Archive object #%i.\n\
\n\
%s.",
				i + 1,
				zip_strerror(archive)
			);
			edv_archive_add_set_error(core, msg);
			g_free(msg);
			zip_source_free(zsrc);
			g_free(tar_path);
			return(-1);
		}
	}
	else
	{
		/* Add the file */
		if(zip_add(
			archive,
			tar_path,
			zsrc
		) < 0)
		{
			gchar *msg = g_strdup_printf(
"Unable to add a new PKZip Archive object.\n\
\n\
%s.",
				zip_strerror(archive)
			);
			edv_archive_add_set_error(core, msg);
			g_free(msg);
			zip_source_free(zsrc);
			g_free(tar_path);
			return(-1);
		}
	}

	g_free(tar_path);

	return(0);
}

/*
 *	Adds the directory specified by path to the PKZip Archive.
 *
 *	Does not add any objects within the directory into the archive.
 */
static gint edv_archive_add_libzip_iterate_directory(
	EDVCore *core,
	const gchar *arch_path,
	struct zip *archive,
	const gchar *password,
	const gchar *parent_path,		/* Parent directory of the archive */
	const gchar *path,			/* The object to add to the archive */
	EDVVFSObject *obj,
	GList **new_paths_list_rtn,
	gulong *cur_size, const gulong total_size,
	GtkWidget *toplevel,
	const gboolean show_progress,
	gint *dstatus_rtn
)
{
	struct zip_source *zsrc;
	gint i;
	gchar *tar_path;
	EDVArchiveAddLibzipSourceData *d;

	/* Generate the path of the file within the archive
	 *
	 * All directories in PKZip Archives must have a tailing
	 * deliminator character to denote that it is a directory
	 */
	if(edv_path_is_parent(parent_path, path))
	{
		const char *s = path + strlen(parent_path);
		while(*s == G_DIR_SEPARATOR)
			s++;
		tar_path = g_strconcat(
			s,
			G_DIR_SEPARATOR_S,
			NULL
		);
	}
	else
	{
		tar_path = g_strconcat(
			path,
			G_DIR_SEPARATOR_S,
			NULL
		);
	}
	if(tar_path == NULL)
	{
		core->last_error_ptr = "Memory allocation error.";
		return(-3);
	}

	/* Create a new and set up the libzip source data */
	d = edv_archive_add_libzip_source_data_new();
	if(d == NULL)
	{
		g_free(tar_path);
		core->last_error_ptr = "Memory allocation error.";
		return(-3);
	}
	d->core = core;
	d->dstatus_rtn = dstatus_rtn;
	d->zip_error[0] = ZIP_ER_OK;
	d->zip_error[1] = 0;
	d->toplevel = toplevel;
	d->show_progress = show_progress;
	d->arch_path = STRDUP(arch_path);
	d->new_paths_list_rtn = new_paths_list_rtn;
	d->cur_size = cur_size;
	d->total_size = total_size;

	d->src_path = STRDUP(path);
	d->tar_path = STRDUP(tar_path);
	d->type = EDV_OBJECT_TYPE_DIRECTORY;
	d->this_size = obj->size;
	d->fp = NULL;
	d->obj = edv_vfs_object_copy(obj);
	d->bytes_read = 0l;

	/* Check if the object already exists in the PKZip Archive */
	i = zip_name_locate(
		archive,
		tar_path,
		0
	);

	/* Set the libzip source function */
	zsrc = zip_source_function(
		archive,
		edv_archive_libzip_source_cb, d
	);
	if(zsrc == NULL)
	{
		gchar *msg = g_strdup_printf(
"Unable to set the libzip source function.\n\
\n\
%s.",
			zip_strerror(archive)
		);
		edv_archive_add_set_error(core, msg);
		g_free(msg);
		edv_archive_add_libzip_source_data_delete(d);
		g_free(tar_path);
		return(-1);
	}

	/* Does the object already exist within the archive? */
	if(i > -1)
	{
		/* Replace the object */
		if(zip_replace(
			archive,
			i,
			zsrc
		) < 0)
		{
			gchar *msg = g_strdup_printf(
"Unable to replace the PKZip Archive object #%i.\n\
\n\
%s.",
				i + 1,
				zip_strerror(archive)
			);
			edv_archive_add_set_error(core, msg);
			g_free(msg);
			zip_source_free(zsrc);
			g_free(tar_path);
			return(-1);
		}
	}
	else
	{
		/* Add the directory */
		if(zip_add(
			archive,
			tar_path,
			zsrc
		) < 0)
		{
			gchar *msg = g_strdup_printf(
"Unable to add a new PKZip Archive object.\n\
\n\
%s.",
				zip_strerror(archive)
			);
			edv_archive_add_set_error(core, msg);
			g_free(msg);
			zip_source_free(zsrc);
			g_free(tar_path);
			return(-1);
		}
	}

	g_free(tar_path);

	return(0);
}

/*
 *	Adds the link specified by path to the PKZip Archive.
 *
 *	The link is not dereferenced.
 */
static gint edv_archive_add_libzip_iterate_link(
	EDVCore *core,
	const gchar *arch_path,
	struct zip *archive,
	const gchar *password,
	const gchar *parent_path,
	const gchar *path,
	EDVVFSObject *obj,
	GList **new_paths_list_rtn,
	gulong *cur_size, const gulong total_size,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gint compression,
	gint *dstatus_rtn
)
{
	struct zip_source *zsrc;
	gint i;
	gchar *tar_path;
	EDVArchiveAddLibzipSourceData *d;

	/* Generate the path of the file within the archive
	 *
	 * All links in PKZip Archives must have a tailing '@'
	 * character to denote that it is a link
	 */
	if(edv_path_is_parent(
		parent_path,			/* Parent */
		path				/* Path */
	))
	{
		const char *s = path + strlen(parent_path);
		while(*s == G_DIR_SEPARATOR)
			s++;
		tar_path = g_strconcat(
			s,
			"@",
			NULL
		);
	}
	else
	{
		tar_path = g_strconcat(
			path,
			"@",
			NULL
		);
	}
	if(tar_path == NULL)
	{
		core->last_error_ptr = "Memory allocation error.";
		return(-3);
	}

	/* Create a new and set up the libzip source data */
	d = edv_archive_add_libzip_source_data_new();
	if(d == NULL)
	{
		g_free(tar_path);
		core->last_error_ptr = "Memory allocation error.";
		return(-3);
	}
	d->core = core;
	d->dstatus_rtn = dstatus_rtn;
	d->zip_error[0] = ZIP_ER_OK;
	d->zip_error[1] = 0;
	d->toplevel = toplevel;
	d->show_progress = show_progress;
	d->arch_path = STRDUP(arch_path);
	d->new_paths_list_rtn = new_paths_list_rtn;
	d->cur_size = cur_size;
	d->total_size = total_size;

	d->src_path = STRDUP(path);
	d->tar_path = STRDUP(tar_path);
	d->type = EDV_OBJECT_TYPE_LINK;
	d->this_size = obj->size;
	d->fp = NULL;
	d->obj = edv_vfs_object_copy(obj);
	d->bytes_read = 0l;

	/* Check if the object already exists in the PKZip Archive */
	i = zip_name_locate(
		archive,
		tar_path,
		0
	);

	/* Set the libzip source function */
	zsrc = zip_source_function(
		archive,
		edv_archive_libzip_source_cb, d
	);
	if(zsrc == NULL)
	{
		gchar *msg = g_strdup_printf(
"Unable to set the libzip source function.\n\
\n\
%s.",
			zip_strerror(archive)
		);
		edv_archive_add_set_error(core, msg);
		g_free(msg);
		edv_archive_add_libzip_source_data_delete(d);
		g_free(tar_path);
		return(-1);
	}

	/* Does the object already exist within the archive? */
	if(i > -1)
	{
		/* Replace the object */
		if(zip_replace(
			archive,
			i,
			zsrc
		) < 0)
		{
			gchar *msg = g_strdup_printf(
"Unable to replace the PKZip Archive object #%i.\n\
\n\
%s.",
				i + 1,
				zip_strerror(archive)
			);
			edv_archive_add_set_error(core, msg);
			g_free(msg);
			zip_source_free(zsrc);
			g_free(tar_path);
			return(-1);
		}
	}
	else
	{
		/* Add the link */
		if(zip_add(
			archive,
			tar_path,
			zsrc
		) < 0)
		{
			gchar *msg = g_strdup_printf(
"Unable to add a new PKZip Archive object.\n\
\n\
%s.",
				zip_strerror(archive)
			);
			edv_archive_add_set_error(core, msg);
			g_free(msg);
			zip_source_free(zsrc);
			g_free(tar_path);
			return(-1);
		}
	}

	g_free(tar_path);

	return(0);
}

/*
 *	Adds the object to the PKZip Archive.
 */
static gint edv_archive_add_libzip_iterate(
	EDVCore *core,
	const gchar *arch_path,
	struct zip *archive,
	const gchar *password,
	const gchar *parent_path,
	const gchar *path,
	GList **new_paths_list_rtn,
	gulong *cur_size, const gulong total_size,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive,
	const gint compression,
	const gboolean dereference_links,
	gint *dstatus_rtn
)
{
	gint status;
	EDVObjectType type;
	EDVVFSObject *obj = NULL;

#define CLEANUP_RETURN(_v_)	{		\
 edv_vfs_object_delete(obj);			\
						\
 return(_v_);					\
}

	if(STRISEMPTY(path))
		CLEANUP_RETURN(-2);

	/* Skip the archive itself */
	if(!strcmp((const char *)arch_path, path))
		CLEANUP_RETURN(0);

	/* Check if this object does not exist */
	obj = edv_vfs_object_lstat(path);
	if(obj == NULL)
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"Unable to add:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
%s.",
			path,
			arch_path,
			g_strerror(error_code)
		);
		edv_archive_add_set_error(core, msg);
		g_free(msg);
		CLEANUP_RETURN(-1);
	}

	type = obj->type;

/* Reports an add error to the user and queries the user to
 * continue adding objects
 */
#define REPORT_ERROR_QUERY_CONTINUE_ADDING	{	\
 if(interactive) {					\
  if(!(*yes_to_all)) {					\
   /* Query the user to continue adding */		\
   gint response;					\
   gchar *msg = g_strdup_printf(			\
"An error occured while adding the object:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
Continue adding subsequent objects?",			\
    path,						\
    arch_path						\
   );							\
   edv_play_sound_error(core);				\
   CDialogSetTransientFor(toplevel);			\
   response = CDialogGetResponse(			\
    "Add Failed",					\
    msg,						\
    NULL,						\
    CDIALOG_ICON_ERROR,					\
    CDIALOG_BTNFLAG_YES |				\
     CDIALOG_BTNFLAG_YES_TO_ALL |			\
     CDIALOG_BTNFLAG_NO,				\
    CDIALOG_BTNFLAG_YES					\
   );							\
   g_free(msg);						\
   CDialogSetTransientFor(NULL);			\
							\
   /* Stop adding? */					\
   if((response == CDIALOG_RESPONSE_NO) ||		\
      (response == CDIALOG_RESPONSE_CANCEL) ||		\
      (response == CDIALOG_RESPONSE_NOT_AVAILABLE)	\
   ) {							\
    /* Override the status value return and return	\
     * indicating user aborted				\
     */							\
    CLEANUP_RETURN(-4);					\
   }							\
							\
   if(response == CDIALOG_RESPONSE_YES_TO_ALL)		\
    *yes_to_all = TRUE;					\
  }							\
 }							\
}
	status = 0;

	/* Directory? */
	if(type == EDV_OBJECT_TYPE_DIRECTORY)
	{
		/* Add this directory to the PKZip Archive (but not the
		 * objects within it)
		 */
		status = edv_archive_add_libzip_iterate_directory(
			core,
			arch_path,
			archive,
			password,
			parent_path,		/* Parent directory of the archive */
			path,			/* The object to add to the archive */
			obj,
			new_paths_list_rtn,
			cur_size, total_size,
			toplevel,
			show_progress,
			dstatus_rtn
		);
		if(status != 0)
		{
			REPORT_ERROR_QUERY_CONTINUE_ADDING
		}

		/* Recurse into this directory? */
		if(recursive && (status != -4) && (status != -5))
		{
			gint status2;

			/* Add the objects within this directory into the
			 * PKZip Archive
			 */
			GList *names_list = edv_directory_list(
				path,
				FALSE,			/* Unsorted */
				FALSE			/* Exclude notations */
			);
			if(names_list != NULL)
			{
				const gchar *name;
				gchar *full_path;
				GList *glist;

				for(glist = names_list;
				    glist != NULL;
				    glist = g_list_next(glist)
				)
				{
					name = (const gchar *)glist->data;
					if(name == NULL)
						continue;

					full_path = edv_paths_join(
						path,
						name
					);
					status2 = edv_archive_add_libzip_iterate(
						core,
						arch_path,
						archive,
						password,
						parent_path,
						full_path,
						new_paths_list_rtn,
						cur_size, total_size,
						toplevel,
						show_progress,
						interactive,
						yes_to_all,
						recursive,
						compression,
						dereference_links,
						dstatus_rtn
					);
					g_free(full_path);
					if(status2 != 0)
						status = status2;
					if(status2 == -4)
						break;
				}

				edv_directory_list_delete(names_list);
			}
		}
	}
	/* Link? */
	else if(type == EDV_OBJECT_TYPE_LINK)
	{
		/* Dereference this link? */
		if(dereference_links)
		{
			/* Get the destination object's stats */
			edv_vfs_object_delete(obj);
			obj = edv_vfs_object_stat(path);
			if(obj == NULL)
			{
				const gint error_code = (gint)errno;
				gchar *msg = g_strdup_printf(
"Unable to add:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
%s.",
					path,
					arch_path,
					g_strerror(error_code)
				);
			        edv_archive_add_set_error(core, msg);
			        g_free(msg);
				CLEANUP_RETURN(-1);
			}
			else
			{
				type = obj->type;

				/* Destination is a directory? */
				if(type == EDV_OBJECT_TYPE_DIRECTORY)
				{
					/* Add this directory to the PKZip Archive (but
					 * not the objects within it)
					 */
					status = edv_archive_add_libzip_iterate_directory(
						core,
						arch_path,
						archive,
						password,
						parent_path,
						path,
						obj,
						new_paths_list_rtn,
						cur_size, total_size,
						toplevel,
						show_progress,
						dstatus_rtn
					);
					if(status != 0)
					{
						REPORT_ERROR_QUERY_CONTINUE_ADDING
					}

					/* Recurse into this directory? */
					if(recursive && (status != -4) && (status != -5))
					{
						gint status2;

						/* Add the objects within this directory
						 * into the PKZip Archive
						 */
						GList *names_list = edv_directory_list(
							path,
							FALSE,		/* Unsorted */
							FALSE		/* Include notations */
						);
						if(names_list != NULL)
						{
							const gchar *name;
							GList *glist;
							gchar *full_path;

							for(glist = names_list;
							    glist != NULL;
							    glist = g_list_next(glist)
							)
							{
								name = (const gchar *)glist->data;
								if(name == NULL)
									continue;

								full_path = edv_paths_join(
									path,
									name
								);
								status2 = edv_archive_add_libzip_iterate(
									core,
									arch_path,
									archive,
									password,
									parent_path,
									full_path,
									new_paths_list_rtn,
									cur_size, total_size,
									toplevel,
									show_progress,
									interactive,
									yes_to_all,
									recursive,
									compression,
									dereference_links,
									dstatus_rtn
								);
								g_free(full_path);
								if(status2 != 0)
									status = status2;
								if(status2 == -4)
									break;
							}

							edv_directory_list_delete(names_list);
						}
					}
				}
				/* Destination is a file? */
				else if(type == EDV_OBJECT_TYPE_FILE)
				{
					/* Add this file to the archive */
					status = edv_archive_add_libzip_iterate_file(
						core,
						arch_path,
						archive,
						password,
						parent_path,
						path,
						obj,
						new_paths_list_rtn,
						cur_size, total_size,
						toplevel,
						show_progress,
						compression,
						dstatus_rtn
					);
					if(status != 0)
					{
						REPORT_ERROR_QUERY_CONTINUE_ADDING
					}
				}
				/* Destination is some other type of object that
				 * is not supported by the PKZip format
				 */
				else
				{
					const gchar *error_msg =
"This type of object is not supported by the PKZip format";
					gchar *msg;
					if(type == EDV_OBJECT_TYPE_DEVICE_CHARACTER)
						error_msg = 
"Character devices are not supported by the PKZip format";
					if(type == EDV_OBJECT_TYPE_DEVICE_BLOCK)
						error_msg = 
"Block devices are not supported by the PKZip format";
					if(type == EDV_OBJECT_TYPE_FIFO)
						error_msg = 
"FIFO pipes are not supported by the PKZip format";
					if(type == EDV_OBJECT_TYPE_SOCKET)
						error_msg = 
"Sockets are not supported by the PKZip format";
					msg = g_strdup_printf(
"Unable to add:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
%s.",
						path,
						arch_path,
						error_msg
					);
					edv_archive_add_set_error(core, msg);
					g_free(msg);
					CLEANUP_RETURN(-2);
				}
			}
		}
		else
		{
			/* Add this link to the archive (do not dereference) */
			status = edv_archive_add_libzip_iterate_link(
				core,
				arch_path,
				archive,
				password,
				parent_path,
				path,
				obj,
				new_paths_list_rtn,
				cur_size, total_size,
				toplevel,
				show_progress,
				compression,
				dstatus_rtn
			);
			if(status != 0)
			{
				REPORT_ERROR_QUERY_CONTINUE_ADDING
			}
		}
	}
	/* File? */
	else if(type == EDV_OBJECT_TYPE_FILE)
	{
		/* Add this file to the archive */
		status = edv_archive_add_libzip_iterate_file(
			core,
			arch_path,
			archive,
			password,
			parent_path,
			path,
			obj,
			new_paths_list_rtn,
			cur_size, total_size,
			toplevel,
			show_progress,
			compression,
			dstatus_rtn
		);
		if(status != 0)
		{
			REPORT_ERROR_QUERY_CONTINUE_ADDING
		}
	}
	/* Destination is some other type of object not supported by
	 * the PKZIP format
	 */
	else
	{
		const gchar *error_msg =
"This type of object is not supported by the PKZip format";
		gchar *msg;
		if(type == EDV_OBJECT_TYPE_DEVICE_CHARACTER)
			error_msg = 
"Character devices are not supported by the PKZip format";
		if(type == EDV_OBJECT_TYPE_DEVICE_BLOCK)
			error_msg = 
"Block devices are not supported by the PKZip format";
		if(type == EDV_OBJECT_TYPE_FIFO)
			error_msg = 
"FIFO pipes are not supported by the PKZip format";
		if(type == EDV_OBJECT_TYPE_SOCKET)
			error_msg = 
"Sockets are not supported by the PKZip format";
		msg = g_strdup_printf(
"Unable to add:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
%s.",
			path,
			arch_path,
			error_msg
		);
		edv_archive_add_set_error(core, msg);
		g_free(msg);
		CLEANUP_RETURN(-2);
	}

#undef REPORT_ERROR_QUERY_CONTINUE_ADDING

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}
#endif	/* HAVE_LIBZIP */

/*
 *	Add object to a PKZip Archive.
 *
 *	Inputs assumed valid.
 */
gint edv_archive_add_zip(
	EDVCore *core, const gchar *arch_path,
	GList *tar_paths_list,
	GList **new_paths_list_rtn,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive, const gint compression,
	const gboolean dereference_links
)
{
#ifdef HAVE_LIBZIP
	struct zip *archive;
	gint		status,
			status2,
			dstatus,
			zip_error_code,
			sys_error_code,
			nobjs;
	gulong		cur_size,
			total_size;
	gchar *parent_path;
	GList *glist;

	/* Get the parent directory of the archive, this will be the
	 * base directory in which all objects added into the PKZip
	 * archive will be relative to (have their parent paths striped
	 * of if their parent paths match this parent path)
	 */
	parent_path = g_dirname(arch_path);
	if(parent_path == NULL)
	{
		core->last_error_ptr = "Unable to obtain the parent directory of the archive.";
		return(-1);
	}

	/* Calculate the total uncompressed size of all the objects to
	 * be added to the archive
	 */
	nobjs = 0;
	total_size = 0l;
	for(glist = tar_paths_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
		total_size += edv_archive_add_calculate_total_size(
			arch_path, (const gchar *)glist->data,
			&nobjs,
			recursive, dereference_links
		);

	/* Open the PKZip Archive for writing */
	archive = edv_archive_add_libzip_open_write(
		arch_path,
		&zip_error_code, &sys_error_code
	);
	if(archive == NULL)
	{
		gchar *msg, err_msg[1024];
		zip_error_to_str(
			err_msg, sizeof(err_msg),
			zip_error_code, sys_error_code
		);
		msg = g_strdup_printf(
"Unable to open the PKZip Archive for writing:\n\
\n\
    %s\n\
\n\
%s.",
			arch_path, err_msg
		);
		edv_archive_add_set_error(core, msg);
		g_free(msg);
		g_free(parent_path);
		return(-1);
	}

	/* Reset the dynamic status which will be set in
	 * edv_archive_libzip_source_cb() to indicate any errors that
	 * might occure
	 *
	 * edv_archive_libzip_source_cb() is called when zip_close()
	 * is called so the allocation of the dynamic status must
	 * remain allocated until zip_close() is called
	 */
	dstatus = 0;

	/* Iterate through the list of objects to add */
	status = 0;
	cur_size = 0l;
	for(glist = tar_paths_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		status2 = edv_archive_add_libzip_iterate(
			core,
			arch_path,
			archive,
			password,
			parent_path,
			(const gchar *)glist->data,	/* Full path to the object to be added */
			new_paths_list_rtn,
			&cur_size, total_size,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			recursive,
			compression,
			dereference_links,
			&dstatus
		);
		if(status2 != 0)
			status = status2;
		if(status2 == -4)
			break;
	}

	/* Write/flush changes to the PKZip Archive and close it
	 *
	 * This is where libzip actually adds the object to the archive
	 * by calling our edv_archive_libzip_source_cb() write callback function
	 *
	 * The progress will be reported and the status and cur_size
	 * will be updated by calls to edv_archive_libzip_source_cb() made in
	 * zip_close()
	 */
	if(zip_close(archive) != 0)
	{
		/* If not interrupted during the write and close then
		 * set the status to indicate error, otherwise the error
		 * was that the user aborted
		 */
		if((dstatus == 0) || (dstatus != -4))
		{
			gchar *msg = g_strdup_printf(
"Unable to close the PKZip Archive:\n\
\n\
    %s\n\
\n\
%s.",
				arch_path,
				zip_strerror(archive)
			);
			edv_archive_add_set_error(core, msg);
			g_free(msg);
		}
	}

	/* Check if an error occured by checking if dstatus was changed
	 * in edv_archive_libzip_source_cb() when zip_close() was called
	 */
	if((status == 0) && (dstatus != 0))
		status = dstatus;

	g_free(parent_path);

	return(status);
#else
	FILE		*cstdout = NULL,
			*cstderr = NULL;
	CfgList *cfg_list = core->cfg_list;
	const gchar *prog_zip = EDV_GET_S(EDV_CFG_PARM_PROG_ZIP);
	gint		status,
			pid,
			nobjs;
        gulong total_size;
	gchar		*parent_path = NULL,
			*pwd = NULL,
			*cmd = NULL;
	EDVVFSObject *arch_obj = edv_vfs_object_stat(arch_path);

#define CLEANUP_RETURN(_v_)	{		\
 (void)FCLOSE(cstdout);				\
 (void)FCLOSE(cstderr);				\
 edv_vfs_object_delete(arch_obj);		\
 g_free(parent_path);				\
 g_free(cmd);					\
						\
 /* Restore the previous working dir */		\
 if(pwd != NULL) {				\
  (void)edv_setcwd(pwd);			\
  g_free(pwd);					\
 }						\
						\
 return(_v_);					\
}

	/* Record the previous working directory and set the new
	 * working directory
	 */
	pwd = edv_getcwd();
	parent_path = g_dirname(arch_path);
	(void)edv_setcwd(parent_path);

	/* If the PKZip Archive exists and is empty then it must
	 * be removed first before adding objects to it
	 */
	if(arch_obj != NULL)
	{
		if(arch_obj == 0l)
			(void)edv_unlink(arch_path);
	}

	/* Format the add objects to a PKZip archive command */
	cmd = g_strdup_printf(
		"\"%s\" -%i %s %s \"%s\"",
		prog_zip,
		CLIP(compression * 9 / 100, 0, 9),
		recursive ? "-r" : "",
		dereference_links ? "" : "-y",
		arch_path
	);
	nobjs = 0;
	total_size = 0l;
	if(cmd != NULL)
	{
		gchar *s;
		const gchar *path;
		GList *glist;
		for(glist = tar_paths_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			path = (const gchar *)glist->data;
			if(STRISEMPTY(path))
				continue;

                        /* Count the size and number of objects */
                        total_size += edv_archive_add_calculate_total_size(
                                arch_path,
                                path,
                                &nobjs,
                                recursive,
                                dereference_links
                        );

			/* Seek tpath to the relative path as needed */
			if(edv_path_is_parent(
				parent_path,
				path
			))
			{
				path += strlen(parent_path);
				while(*path == G_DIR_SEPARATOR)
					path++;
			}

			s = g_strconcat(
				cmd,
				" \"",
				path,
				"\"",
				NULL
			);
			if(s != NULL)
			{
				g_free(cmd);
				cmd = s;
			}
		}
	}
	if(cmd == NULL)
	{
		core->last_error_ptr = "Unable to generate the add command.";
		CLEANUP_RETURN(-1);
	}

        /* Execute the add objects to archive command */
        pid = edv_archive_add_execute(
                cfg_list,
                cmd,
                NULL,
                &cstdout,
                &cstderr
        );
        if(pid < 0)
	{
		core->last_error_ptr = "Unable to execute the add command.";
		CLEANUP_RETURN(-1);
	}

	/* Delete the command */
	g_free(cmd);
	cmd = NULL;

	status = 0;

        /* Read the archive add messages from the output streams */
        if((cstdout != NULL) && (cstderr != NULL))
        {
                gint nobjs_added = 0;
                gfloat progress = 0.0f;
                gchar *buf = NULL;
                FILE *fp = cstdout;

                /* Map the progress dialog? */
                if(show_progress)
                {
                        if(edv_archive_add_update_progress_dialog(
                                cfg_list,
                                "",
                                arch_path,
                                progress,
                                toplevel,
                                FALSE
                        ) > 0)
                        {
                                (void)INTERRUPT(pid);
                                pid = 0;
                                status = -4;
                        }
                }

                do {
                        /* Check if the process is no longer running */
                        if(pid > 0)
                        {
                                if(!edv_pid_exists(pid))
                                        pid = 0;
                        }

                        /* Update the progress? */
                        if(show_progress && ProgressDialogIsQuery())
                        {
                                if(edv_archive_add_update_progress_dialog(
                                        cfg_list,
                                        NULL,
                                        NULL,
                                        progress,
                                        toplevel,
                                        FALSE
                                ) > 0)
                                {
                                        (void)INTERRUPT(pid);
                                        pid = 0;
                                        status = -4;
                                        break;
                                }
                        }

                        /* Flush any error messages */
                        FDISCARD(cstderr);

                        /* Read the next line */
                        if(edv_stream_read_lineptr(
                                fp,
                                &buf,
                                FALSE           /* Nonblocking */
                        ))
                        {
                                gchar	*s = buf,
					*s2;
				const gchar *path;
                                while(ISBLANK(*s))
                                        s++;

				/* Seek s past the first prefix */
				while(!ISBLANK(*s) && (*s != '\0'))
					s++;
				while(ISBLANK(*s))
					s++;

				/* Cap the space character after path */
				s2 = (gchar *)strpbrk((char *)s, " \t(\n\r");
				if(s2 != NULL)
					*s2 = '\0';

				path = s;

                                /* Append this path to the list of
                                 * paths added to the archive
				 */
				if(new_paths_list_rtn != NULL)
					*new_paths_list_rtn = g_list_append(
						*new_paths_list_rtn,
						g_strdup(path)
					);

				/* Update the progress dialog's label? */
				if(show_progress && !STRISEMPTY(path))
				{
					if(edv_archive_add_update_progress_dialog(
						cfg_list,
						path,
						arch_path,
                                                progress,
						toplevel,
						FALSE
					) > 0)
					{
						(void)INTERRUPT(pid);
						pid = 0;
						status = -4;
						break;
					}
				}

                                nobjs_added++;
                                progress = (nobjs > 0) ?
                                        ((gfloat)nobjs_added / (gfloat)nobjs) : 0.0f;

                                g_free(buf);
                                buf = NULL;
                        }

                        edv_usleep(EDV_ITERATION_SLEEP_MIN_US);

                } while((pid > 0) || !feof(fp));

                /* Flush any error messages */
                FDISCARD(cstderr);

                g_free(buf);
        }

        /* Close the output stremas */
        (void)FCLOSE(cstdout);
        cstdout = NULL;
        (void)FCLOSE(cstderr);
        cstderr = NULL;

        /* Report the final progress? */
        if(show_progress && (status == 0) && ProgressDialogIsQuery())
        {
                if(edv_archive_add_update_progress_dialog(
                        cfg_list,
                        NULL,
                        NULL,
                        1.0f,
                        toplevel,
                        FALSE
                ) > 0)
                        status = -4;
        }

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
#endif	/* !HAVE_LIBZIP */
}
