#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#ifdef HAVE_LIBXAR
# include <xar/xar.h>
/* The X Archive library does not allow public error callback
 * setting... yet, when it does you can uncomment the below
 */
/* # define USE_LIBXAR_ERROR_CB */
#endif
#include <gtk/gtk.h>

#include "cdialog.h"
#include "progressdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_directory.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "libendeavour2-base/edv_archive_obj.h"
#include "edv_utils_gtk.h"
#include "edv_progress.h"
#include "edv_archive_add.h"
#include "edv_archive_add_xar.h"
#include "endeavour2.h"

#include "config.h"


#ifdef HAVE_LIBXAR
# ifdef USE_LIBXAR_ERROR_CB
typedef struct _EDVArchiveAddXARErrorData	EDVArchiveAddXARErrorData;
#define EDV_ARCH_XAR_ERROR_DATA(p)	((EDVArchiveAddXARErrorData *)(p))
# endif
#endif


#ifdef HAVE_LIBXAR
# ifdef USE_LIBXAR_ERROR_CB
static int32_t edv_archive_add_xar_error_cb(
	int32_t sev, int32_t err, xar_errctx_t ctx, void *data
);
# endif
static gint edv_archive_add_xar_iterate(
	EDVCore *core,
	const gchar *arch_path,
	xar_t xar,
	const gchar *password,
	const gchar *parent_path,
	const gchar *path,
	GList **new_paths_list_rtn,
	gulong *cur_size, const gulong total_size,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive,
	const gboolean dereference_links
);
#endif	/* HAVE_LIBXAR */
gint edv_archive_add_xar(
	EDVCore *core, const gchar *arch_path,
	GList *tar_paths_list,
	GList **new_paths_list_rtn,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive, const gint compression,
	const gboolean dereference_links
);


#ifdef HAVE_LIBXAR
# ifdef USE_LIBXAR_ERROR_CB
/*
 *	Error Data:
 */
struct _EDVArchiveAddXARErrorData {
	EDVCore		*core;
	const gchar	*arch_path;
};
# endif	/* USE_LIBXAR_ERROR_CB */
#endif	/* HAVE_LIBXAR */


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


#ifdef HAVE_LIBXAR
# ifdef USE_LIBXAR_ERROR_CB
/*
 *	X Archive library error callback.
 */
static int32_t edv_archive_add_xar_error_cb(
	int32_t sev, int32_t err, xar_errctx_t ctx, void *data
)
{
	xar_file_t xar_fp;
	gint error_code;
	gchar *path;
	const gchar *error_msg;
	EDVCore *core;
	EDVArchiveAddXARErrorData *d = EDV_ARCH_XAR_ERROR_DATA(data);
	if(d == NULL)
		return(0);

	core = d->core;

	xar_fp = xar_err_get_file(ctx);
	error_msg = (const gchar *)xar_err_get_string(ctx);
	error_code = (gint)xar_err_get_errno(ctx);

	switch(sev)
	{
	  case XAR_SEVERITY_DEBUG:
	  case XAR_SEVERITY_INFO:
	  case XAR_SEVERITY_WARNING:
	  case XAR_SEVERITY_NORMAL:
		/* Ignore these */
		break;

	  case XAR_SEVERITY_NONFATAL:
	  case XAR_SEVERITY_FATAL:
		path = (xar_fp != NULL) ? (gchar *)xar_get_path(xar_fp) : NULL;
		if(path != NULL)
		{
			gchar *msg = g_strdup_printf(
"Unable to add:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
%s, %s.",
				path,
				d->arch_path,
				error_msg,
				g_strerror(error_code)
			);
			edv_archive_add_set_error(core, msg);
			g_free(msg);
			g_free(path);
		}
		else
		{
			gchar *msg = g_strdup_printf(
"Unable to add an object to the archive:\n\
\n\
    %s\n\
\n\
%s, %s.",
				d->arch_path,
				error_msg,
				g_strerror(error_code)
			);
			edv_archive_add_set_error(core, msg);
			g_free(msg);
		}
		break;
	}

	return(0);
}
# endif	/* USE_LIBXAR_ERROR_CB */

/*
 *	Adds one object to the X Archive.
 *
 *	The path specifies the full path to the object to add to the
 *	archive.
 *
 *	If recursive is TRUE and path refers to a directory then the
 *	directory and all the objects within the directory will be
 *	added to the archive.
 */
static gint edv_archive_add_xar_iterate(
	EDVCore *core,
	const gchar *arch_path,
	xar_t xar,
	const gchar *password,
	const gchar *parent_path,
	const gchar *path,
	GList **new_paths_list_rtn,
	gulong *cur_size, const gulong total_size,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive,
	const gboolean dereference_links
)
{
	gint status;
	gulong size;
	gchar *in_arch_path = NULL;
	CfgList *cfg_list = core->cfg_list;
	EDVObjectType type;
	EDVVFSObject *obj = NULL;

#define CLEANUP_RETURN(_v_)	{		\
 edv_vfs_object_delete(obj);			\
 g_free(in_arch_path);				\
						\
 return(_v_);					\
}

	if(STRISEMPTY(path))
	{
		CLEANUP_RETURN(-2);
	}

	/* Skip the archive itself */
	if(!strcmp((const char *)arch_path, path))
	{
		CLEANUP_RETURN(0);
	}

	/* Get this object's local stats */
	obj = edv_vfs_object_lstat(path);
	if(obj == NULL)
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"Unable to add:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
%s.",
			path,
			arch_path,
			g_strerror(error_code)
		);
		edv_archive_add_set_error(core, msg);
		g_free(msg);
		CLEANUP_RETURN(-1);
	}

	type = obj->type;
	size = obj->size;

	/* Generate the path of the object within the archive */
	if(edv_path_is_parent(
		parent_path,
		path
	))
	{
		const char *s = path + strlen(parent_path);
		while(*s == G_DIR_SEPARATOR)
			s++;
		in_arch_path = g_strdup(s);
	}
	else
	{
		in_arch_path = g_strdup(path);
	}
	if(in_arch_path == NULL)
	{
		core->last_error_ptr = "Memory allocation error.";
		CLEANUP_RETURN(-3);
	}

	/* Map the progress dialog? */
	if(show_progress)
	{
		if(edv_archive_add_update_progress_dialog(
			cfg_list,
			path,
			arch_path,
			(total_size > 0l) ?
				((gfloat)(*cur_size) / (gfloat)total_size) : 0.0f,
			toplevel,
			FALSE
		) > 0)
		{
			CLEANUP_RETURN(-4);
		}
	}

/* Sets the status, reports an add error to the user, and queries
 * the user to continue adding objects or return user abort
 */
#define REPORT_ERROR_QUERY_CONTINUE_ADDING(_v_)	{	\
							\
 status = (_v_);					\
							\
 if(interactive) {					\
  if(!(*yes_to_all)) {					\
   /* Query the user to continue adding */		\
   gint response;					\
   gchar *msg = g_strdup_printf(			\
"An error occured while adding the object:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
Continue adding subsequent objects?",			\
    path,						\
    arch_path						\
   );							\
   edv_play_sound_error(core);				\
   CDialogSetTransientFor(toplevel);			\
   response = CDialogGetResponse(			\
    "Add Failed",					\
    msg,						\
    NULL,						\
    CDIALOG_ICON_ERROR,					\
    CDIALOG_BTNFLAG_YES |				\
     CDIALOG_BTNFLAG_YES_TO_ALL |			\
     CDIALOG_BTNFLAG_NO,				\
    CDIALOG_BTNFLAG_YES					\
   );							\
   g_free(msg);						\
   CDialogSetTransientFor(NULL);			\
							\
   /* Stop adding? */					\
   if((response == CDIALOG_RESPONSE_NO) ||		\
      (response == CDIALOG_RESPONSE_CANCEL) ||		\
      (response == CDIALOG_RESPONSE_NOT_AVAILABLE)	\
   ) {							\
    /* Override the status value and return the value	\
     * indicating user aborted				\
     */							\
    CLEANUP_RETURN(-4);					\
   }							\
							\
   if(response == CDIALOG_RESPONSE_YES_TO_ALL)		\
    *yes_to_all = TRUE;					\
  }							\
 }							\
}

/* Reports the progress and returns if user aborted */
#define REPORT_PROGRESS	{				\
 if(show_progress && ProgressDialogIsQuery()) {		\
  if(edv_archive_add_update_progress_dialog(		\
   cfg_list,						\
   NULL,						\
   NULL,						\
   (total_size > 0l) ?					\
    ((gfloat)(*cur_size) / (gfloat)total_size) : 0.0f,	\
   toplevel,						\
   FALSE						\
  ) > 0) {						\
   CLEANUP_RETURN(-4);					\
  }							\
 }							\
}

/* Increment *cur_size with _size_inc_ and adds _path_ to
 * new_paths_list_rtn
 */
#define COUNT_OBJECT_PATH(_path_,_size_inc_)	{	\
 if((_size_inc_) > 0l)					\
  *cur_size = (*cur_size) + (_size_inc_);		\
							\
 /* Add the path to the list of objects added */	\
 if(((_path_) != NULL) && (new_paths_list_rtn != NULL))	\
  *new_paths_list_rtn = g_list_append(			\
   *new_paths_list_rtn,					\
   g_strdup(_path_)					\
  );							\
}

	status = 0;

	/* Add this object to the archive based on its type
	 *
	 * Directory?
	 */
	if(type == EDV_OBJECT_TYPE_DIRECTORY)
	{
		/* Add this directory to the X Archive */
		if(xar_add(xar, (const char *)in_arch_path) != NULL)
		{
			COUNT_OBJECT_PATH(path, 0l);
		}
		else
		{
			if(core->last_error_ptr == NULL)
			{
				gchar *msg = g_strdup_printf(
"Unable to add:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s",
					path,
					arch_path
				);
				edv_archive_add_set_error(core, msg);
				g_free(msg);
			}
			REPORT_ERROR_QUERY_CONTINUE_ADDING(-1);
		}

		REPORT_PROGRESS;

		/* Recurse into this directory? */
		if(recursive && (status != -4) && (status != -5))
		{
			/* Add the objects within this directory into the
			 * X Archive
			 */
			GList *names_list = edv_directory_list(
				path,
				FALSE,			/* Unsorted */
				FALSE			/* Exclude notations */
			);
			if(names_list != NULL)
			{
				gint status2;
				const gchar *name;
				gchar *full_path;
				GList *glist;

				for(glist = names_list;
				    glist != NULL;
				    glist = g_list_next(glist)
				)
				{
					name = (const gchar *)glist->data;
					if(name == NULL)
						continue;

					full_path = edv_paths_join(
						path,
						name
					);
					status2 = edv_archive_add_xar_iterate(
						core,
						arch_path,
						xar,
						password,
						parent_path,
						full_path,
						new_paths_list_rtn,
						cur_size, total_size,
						toplevel,
						show_progress,
						interactive,
						yes_to_all,
						recursive,
						dereference_links
					);
					g_free(full_path);
					if(status2 != 0)
						status = status2;
					if(status2 == -4)
						break;
				}

				edv_directory_list_delete(names_list);
			}
		}
	}
	/* Link? */
	else if(type == EDV_OBJECT_TYPE_LINK)
	{
		/* Dereference this link? */
		if(dereference_links)
		{
			/* Get the destination object's stats */
			edv_vfs_object_delete(obj);
			obj = edv_vfs_object_stat(path);
			if(obj == NULL)
			{
				const gint error_code = (gint)errno;
				gchar *msg = g_strdup_printf(
"Unable to add:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
%s.",
					path,
					arch_path,
					g_strerror(error_code)
			        );
			        edv_archive_add_set_error(core, msg);
			        g_free(msg);
				CLEANUP_RETURN(-1);
			}
			else
			{
				type = obj->type;
				size = obj->size;

				/* Destination is a directory? */
				if(type == EDV_OBJECT_TYPE_DIRECTORY)
				{
					/* Add this directory to the X Archive */
					if(xar_add(xar, (const char *)in_arch_path) != NULL)
					{
						COUNT_OBJECT_PATH(path, 0l);
					}
					else
					{
						if(core->last_error_ptr == NULL)
						{
							gchar *msg = g_strdup_printf(
"Unable to add:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s",
								path,
								arch_path
							);
							edv_archive_add_set_error(core, msg);
							g_free(msg);
						}
						REPORT_ERROR_QUERY_CONTINUE_ADDING(-1);
					}

					REPORT_PROGRESS

					/* Recurse into this directory? */
					if(recursive && (status != -4) && (status != -5))
					{
						/* Add the objects within this directory
						 * into the X Archive
						 */
						GList *names_list = edv_directory_list(
							path,
							FALSE,		/* Unsorted */
							FALSE		/* Exclude notations */
						);
						if(names_list != NULL)
						{
							gint status2;
							const gchar *name;
							gchar *full_path;
							GList *glist;

							for(glist = names_list;
							    glist != NULL;
							    glist = g_list_next(glist)
							)
							{
								name = (const gchar *)glist->data;
								if(name == NULL)
									continue;

								full_path = edv_paths_join(
									path,
									name
								);
								status2 = edv_archive_add_xar_iterate(
									core,
									arch_path,
									xar,
									password,
									parent_path,
									full_path,
									new_paths_list_rtn,
									cur_size, total_size,
									toplevel,
									show_progress,
									interactive,
									yes_to_all,
									recursive,
									dereference_links
								);
								g_free(full_path);
								if(status2 != 0)
									status = status2;
								if(status2 == -4)
									break;
							}

							edv_directory_list_delete(names_list);
						}
					}
				}
				/* Destination is a file or other object */
				else
				{
					/* Add this object to the archive */
					if(xar_add(xar, (const char *)in_arch_path) != NULL)
					{
						if(type == EDV_OBJECT_TYPE_FILE)
						{
							COUNT_OBJECT_PATH(path, size);
						}
						else
						{
							COUNT_OBJECT_PATH(path, 0l);
						}
					}
					else
					{
						if(core->last_error_ptr == NULL)
						{
							gchar *msg = g_strdup_printf(
"Unable to add:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s",
								path,
								arch_path
							);
							edv_archive_add_set_error(core, msg);
							g_free(msg);
						}
						REPORT_ERROR_QUERY_CONTINUE_ADDING(-1);
					}

					REPORT_PROGRESS
				}
			}
		}
		else
		{
			/* Add this link to the archive */
			if(xar_add(xar, (const char *)in_arch_path) != NULL)
			{
				COUNT_OBJECT_PATH(path, size);
			}
			else
			{
				if(core->last_error_ptr == NULL)
				{
					gchar *msg = g_strdup_printf(
"Unable to add:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s",
						path,
						arch_path
					);
					edv_archive_add_set_error(core, msg);
					g_free(msg);
				}
				REPORT_ERROR_QUERY_CONTINUE_ADDING(-1);
			}

			REPORT_PROGRESS
		}
	}
	/* File or other object */
	else if(type == EDV_OBJECT_TYPE_FILE)
	{
		/* Add this object to the archive */
		if(xar_add(xar, (const char *)in_arch_path) != NULL)
		{
			if(type == EDV_OBJECT_TYPE_FILE)
			{
				COUNT_OBJECT_PATH(path, size);
			}
			else
			{
				COUNT_OBJECT_PATH(path, 0l);
			}
		}
		else
		{
			if(core->last_error_ptr == NULL)
			{
				gchar *msg = g_strdup_printf(
"Unable to add:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s",
					path,
					arch_path
				);
				edv_archive_add_set_error(core, msg);
				g_free(msg);
			}
			REPORT_ERROR_QUERY_CONTINUE_ADDING(-1);
		}

		REPORT_PROGRESS
	}

#undef REPORT_ERROR_QUERY_CONTINUE_ADDING
#undef REPORT_PROGRESS
#undef COUNT_OBJECT_PATH

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}
#endif	/* HAVE_LIBXAR */

/*
 *	Add object to an X Archive.
 */
gint edv_archive_add_xar(
	EDVCore *core, const gchar *arch_path,
	GList *tar_paths_list,
	GList **new_paths_list_rtn,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive, const gint compression,
	const gboolean dereference_links
)
{
#ifdef HAVE_LIBXAR
	xar_t xar;
	gint		status,
			status2,
			nobjs;
	gulong		cur_size,
			total_size;
	gchar		*parent_path,
			*pwd;
	GList *glist;
	CfgList *cfg_list = core->cfg_list;
#if defined(USE_LIBXAR_ERROR_CB)
	EDVArchiveAddXARErrorData *error_data;
#endif

	/* Get the parent directory of the archive, this will be the
	 * base directory in which all objects added into the X
	 * Archive will be relative to (have their parent paths striped
	 * of if their parent paths match this parent path)
	 */
	parent_path = g_dirname(arch_path);
	if(parent_path == NULL)
	{
		core->last_error_ptr = "Unable to obtain the parent directory of the archive.";
		return(-1);
	}

	/* Calculate the total uncompressed size of all the objects to
	 * be added to the archive
	 */
	nobjs = 0;
	total_size = 0l;
	for(glist = tar_paths_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
		total_size += edv_archive_add_calculate_total_size(
			arch_path,
			(const gchar *)glist->data,
			&nobjs,
			recursive,
			dereference_links
		);

	/* Record the current working directory and then change to the
	 * parent directory of the archive
	 */
	pwd = edv_getcwd();
	(void)edv_setcwd(parent_path);

	/* Open the X Archive for writing */
	xar = xar_open(arch_path, WRITE);
	if(xar == NULL)
	{
		gchar *msg = g_strdup_printf(
"Unable to open the X Archive for writing:\n\
\n\
    %s",
			arch_path
		);
		edv_archive_add_set_error(core, msg);
		g_free(msg);
		(void)edv_setcwd(pwd);
		g_free(parent_path);
		g_free(pwd);
		return(-1);
	}

	/* Set the compression
	 *
	 *	60% - 100%	BZip2
	 *	1% - 59%	GZip
	 *	0%		None
	 */
	if(compression >= 60)
#if defined(XAR_OPT_VAL_BZIP2)
		xar_opt_set(xar, XAR_OPT_COMPRESSION, XAR_OPT_VAL_BZIP2);
#elif defined(XAR_OPT_VAL_BZIP)
		xar_opt_set(xar, XAR_OPT_COMPRESSION, XAR_OPT_VAL_BZIP);
#elif defined(XAR_OPT_VAL_GZIP)
		xar_opt_set(xar, XAR_OPT_COMPRESSION, XAR_OPT_VAL_GZIP);
#else
		xar_opt_set(xar, XAR_OPT_COMPRESSION, XAR_OPT_VAL_NONE);
#endif
	else if(compression > 0)
#if defined(XAR_OPT_VAL_GZIP)
		xar_opt_set(xar, XAR_OPT_COMPRESSION, XAR_OPT_VAL_GZIP);
#else
		xar_opt_set(xar, XAR_OPT_COMPRESSION, XAR_OPT_VAL_NONE);
#endif
	else
		xar_opt_set(xar, XAR_OPT_COMPRESSION, XAR_OPT_VAL_NONE);

	/* Set the checksum */
	xar_opt_set(xar, XAR_OPT_TOCCKSUM, XAR_OPT_VAL_SHA1);

#if defined(USE_LIBXAR_ERROR_CB)
	/* Set the error callback */
	error_data = EDV_ARCH_XAR_ERROR_DATA(g_malloc0(
		sizeof(EDVArchiveAddXARErrorData)
	));
	if(error_data != NULL)
	{
		error_data->core = core;
		error_data->arch_path = arch_path;
		xar_register_errhandler(
			xar,
			edv_archive_add_xar_error_cb, error_data
		);
	}
#endif

	/* Add each object to the X Archive */
	status = 0;
	cur_size = 0l;
	for(glist = tar_paths_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		status2 = edv_archive_add_xar_iterate(
			core,
			arch_path,
			xar,
			password,
			parent_path,
			(const gchar *)glist->data,	/* Full path to the object to be added */
			new_paths_list_rtn,
			&cur_size, total_size,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			recursive,
			dereference_links
		);
		if(status2 != 0)
			status = status2;
		if(status2 == -4)
			break;
	}

	/* Close the X Archive */
	if(xar_close(xar) != 0)
	{
		/* If not interrupted during the write and close then
		 * set the status to indicate error, otherwise the error
		 * was that the user aborted
		 */
		if((status == 0) || (status != -4))
		{
			if(core->last_error_ptr == NULL)
			{
				gchar *msg = g_strdup_printf(
"Unable to close the X Archive:\n\
\n\
    %s",
					arch_path
				);
				edv_archive_add_set_error(core, msg);
				g_free(msg);
			}
			status = -1;
		}
	}

	/* Report the final progress */
	if(show_progress && (status == 0) && ProgressDialogIsQuery())
	{
		if(edv_archive_add_update_progress_dialog(
			cfg_list,
			NULL,
			NULL,
			1.0f,
			toplevel,
			FALSE
		) > 0)
		{
			status = -4;
		}
	}

	/* Restore the previous working directory */
	if(pwd != NULL)
	{
		(void)edv_setcwd(pwd);
		g_free(pwd);
	}
	g_free(parent_path);

#if defined(USE_LIBXAR_ERROR_CB)
	/* Delete the error callback data */
	g_free(error_data);
#endif

	return(status);
#else
	core->last_error_ptr = "Unsupported archive format.";
	return(-2);
#endif
}
