#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#ifdef HAVE_LIBTAR
/* Support for adding using libtar is not supported since libtar 
 * currently does not support appending objects to a Tape Archive
 */
#endif
#include <gtk/gtk.h>

#include "cfg.h"

#include "progressdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_process.h"
#include "libendeavour2-base/edv_stream.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "libendeavour2-base/edv_archive_obj.h"
#include "edv_utils_gtk.h"
#include "edv_progress.h"
#include "edv_archive_add.h"
#include "edv_archive_add_tar.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"


gint edv_archive_add_tar(
	EDVCore *core,
	const gchar *arch_path,
	GList *tar_paths_list,
	GList **new_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive,
	const gint compression,
	const gboolean dereference_links,
	const gboolean is_compress_compressed,
	const gboolean is_gzip_compressed,
	const gboolean is_bzip2_compressed
);


#define EDV_ITERATION_SLEEP_MIN_US		8000l

#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)

#define ISBLANK(c)	(((c) == ' ') || ((c) == '\t'))
#define ISCR(c)		(((c) == '\n') || ((c) == '\r'))

#define FCLOSE(p)	(((p) != NULL) ? (gint)fclose(p) : -1)
#define INTERRUPT(p)	(((p) > 0) ? (gint)kill((int)(p), SIGINT) : -1)

#define FDISCARD(p)	{			\
 if((p) != NULL) {				\
  const gint fd = (gint)fileno(p);		\
  while(edv_poll_read(fd)) {			\
   if(fgetc(p) == EOF)				\
    break;					\
  }						\
 }						\
}



/*
 *	Add object to a Tape Archive.
 */
gint edv_archive_add_tar(
	EDVCore *core,
	const gchar *arch_path,
	GList *tar_paths_list,
	GList **new_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive,
	const gint compression,
	const gboolean dereference_links,
	const gboolean is_compress_compressed,
	const gboolean is_gzip_compressed,
	const gboolean is_bzip2_compressed
)
{
	FILE		*cstdout = NULL,
			*cstderr = NULL;
	CfgList *cfg_list = core->cfg_list;
	const gchar *prog_tar = EDV_GET_S(EDV_CFG_PARM_PROG_TAR);
	const gchar *prog_compress = EDV_GET_S(EDV_CFG_PARM_PROG_COMPRESS);
	const gchar *prog_uncompress = EDV_GET_S(EDV_CFG_PARM_PROG_UNCOMPRESS);
	const gchar *prog_gzip = EDV_GET_S(EDV_CFG_PARM_PROG_GZIP);
	const gchar *prog_gunzip = EDV_GET_S(EDV_CFG_PARM_PROG_GUNZIP);
	const gchar *prog_bzip2 = EDV_GET_S(EDV_CFG_PARM_PROG_BZIP2);
	const gchar *prog_bunzip2 = EDV_GET_S(EDV_CFG_PARM_PROG_BUNZIP2);
	gint		pid,
			status,
			nobjs;
	gulong total_size;
	gchar		*parent_path = NULL,
			*pwd = NULL,
			*cmd = NULL,
			*arch_uncompressed_path = NULL;
	GTimer *timer = g_timer_new();
	EDVVFSObject *arch_obj = edv_vfs_object_stat(arch_path);

#define CLEANUP_RETURN(_v_)	{		\
 g_timer_destroy(timer);			\
 (void)FCLOSE(cstdout);				\
 (void)FCLOSE(cstderr);				\
 g_free(parent_path);				\
 g_free(cmd);					\
 g_free(arch_uncompressed_path);		\
 edv_vfs_object_delete(arch_obj);		\
						\
 /* Restore the previous working dir */		\
 if(pwd != NULL) {				\
  (void)edv_setcwd(pwd);			\
  g_free(pwd);					\
 }						\
						\
 return(_v_);					\
}

	/* Record the previous working directory and set the new
	 * working directory
	 */
	pwd = edv_getcwd();
	parent_path = g_dirname(arch_path);
	(void)edv_setcwd(parent_path);

	/* If compression is specified then create a temporary
	 * uncompressed archive
	 */
	if(is_compress_compressed || is_gzip_compressed || is_bzip2_compressed)
	        arch_uncompressed_path = edv_tmp_name(EDV_GET_S(EDV_CFG_PARM_DIR_TMP));

	/* If the specified archive exists and is non-zero in size then
	 * format the decompress archive command
	 */
	if((arch_obj != NULL) ? (arch_obj->size > 0l) : FALSE)
	{
		if(is_compress_compressed)
			cmd = g_strdup_printf(
				"\"%s\" -c \"%s\" > \"%s\"",
				prog_uncompress,
				arch_path,
				arch_uncompressed_path
			);
		else if(is_gzip_compressed)
			cmd = g_strdup_printf(
				"\"%s\" -d -q -c \"%s\" > \"%s\"",
				prog_gunzip,
				arch_path,
				arch_uncompressed_path
			);
		else if(is_bzip2_compressed)
			cmd = g_strdup_printf(
				"\"%s\" -d -q -c \"%s\" > \"%s\"",
				prog_bunzip2,
				arch_path,
				arch_uncompressed_path
			);
		else
			cmd = NULL;
	}
	else
	{
		cmd = NULL;
	}

	/* Need to decompress the archive? */
	status = 0;
	if(cmd != NULL)
	{
		/* Execute the decompress archive command */
	        pid = edv_archive_add_execute(
	                cfg_list,
	                cmd,
	                NULL,
	                &cstdout,
	                &cstderr
	        );
		if(pid < 0)
		{
			core->last_error_ptr =
"Unable to execute the decompress command.";
			CLEANUP_RETURN(-1);
		}

		/* Delete the command */
		g_free(cmd);
		cmd = NULL;

		/* Map the progress dialog? */
		if(show_progress)
		{
			gchar	*arch_path_shortened = edv_path_shorten(
				arch_path,
				EDV_PROGRESS_DLG_PATH_MAX_CHARS - 10
			),	*msg = g_strdup_printf(
"Decompressing Archive:\n\
\n\
    %s\n",
				arch_path_shortened
			);
			g_free(arch_path_shortened);
			edv_progress_dialog_map_archive_add_animated(
				cfg_list,
				msg,
				-1.0f,
				toplevel,
				TRUE
			);
			g_free(msg);
		}

		/* Wait for the decompress archive process to finish */
		g_timer_start(timer);
		while(edv_pid_exists(pid))
		{
			/* Update the progress? */
			if(show_progress && ProgressDialogIsQuery())
			{
				/* Update the label every second */
				if(g_timer_elapsed(timer, NULL) >= 1.0)
				{
					gchar *msg;
					EDVVFSObject *obj = edv_vfs_object_stat(arch_uncompressed_path);
					if(obj != NULL)
					{
						const gulong size = obj->size;
						gchar	*arch_path_shortened = edv_path_shorten(
							arch_path,
							EDV_PROGRESS_DLG_PATH_MAX_CHARS - 10
						),
							*size_str = STRDUP(edv_str_size_delim(size));
						msg = g_strdup_printf(
"Decompressing Archive:\n\
\n\
    %s (%s %s)\n",
							arch_path_shortened,
							size_str,
							(size == 1l) ? "byte" : "bytes"
						);
						edv_vfs_object_delete(obj);
						g_free(arch_path_shortened);
						g_free(size_str);
					}
					else
					{
						msg = NULL;
					}
					ProgressDialogUpdateUnknown(
						NULL,
						msg,
						NULL,
						NULL,
						TRUE
					);
					g_timer_start(timer);
					g_free(msg);
				}
				else
				{
					ProgressDialogUpdateUnknown(
						NULL,
						NULL,
						NULL,
						NULL,
						TRUE
					);
				}
				if(ProgressDialogStopCount() > 0)
				{
					INTERRUPT(pid);
					pid = 0;
					status = -4;
					break;
				}
			}

			/* Flush the output streams */
			FDISCARD(cstdout);
			FDISCARD(cstderr);

			edv_usleep(EDV_ITERATION_SLEEP_MIN_US);
		}

		/* Unmap the progress dialog */
		if(show_progress)
		{
			ProgressDialogBreakQuery(TRUE);
			ProgressDialogSetTransientFor(NULL);
		}

		g_timer_stop(timer);

		/* Close the output streams */
		(void)FCLOSE(cstdout);
		cstdout = NULL;
		(void)FCLOSE(cstderr);
		cstderr = NULL;
	}
	/* Error or user aborted? */
	if(status != 0)
	{
		(void)edv_unlink(arch_uncompressed_path);
		CLEANUP_RETURN(status);
	}


	/* Format the add object(s) to archive command */
	cmd = g_strdup_printf(
		"\"%s\" -r%s%s -v -f \"%s\"",
		prog_tar,
		recursive ? "" : " --no-recursion",
		dereference_links ? " -h" : "",
		(arch_uncompressed_path != NULL) ?
			arch_uncompressed_path : arch_path
	);
	nobjs = 0;
	total_size = 0l;
	if(cmd != NULL)
	{
		gchar *s;
		const gchar *path;
		GList *glist;
		for(glist = tar_paths_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			path = (const gchar *)glist->data;
			if(STRISEMPTY(path))
				continue;

			/* Count the size and number of objects */
			total_size += edv_archive_add_calculate_total_size(
				arch_path,
				path,
				&nobjs,
				recursive,
				dereference_links
			);

			/* Seek tpath to the relative path as needed */
			if(edv_path_is_parent(
				parent_path,	/* Parent */
				path		/* Path */
			))
			{
				path += strlen(parent_path);
				while(*path == G_DIR_SEPARATOR)
					path++;
			}

			s = g_strconcat(
				cmd,
				" \"",
				path,
				"\"",
				NULL
			);
			if(s != NULL)
			{
				g_free(cmd);
				cmd = s;
			}
		}
	}
	if(cmd == NULL)
	{
		core->last_error_ptr = "Unable to generate the add command.";
		CLEANUP_RETURN(-1);
	}

	/* Execute the add object(s) to archive command */
	pid = edv_archive_add_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	if(pid < 0)
	{
		core->last_error_ptr = "Unable to execute the add command.";
		CLEANUP_RETURN(-1);
	}

	/* Delete the command */
	g_free(cmd);
	cmd = NULL;

	/* Read the archive add messages from the output streams */
	if((cstdout != NULL) && (cstderr != NULL))
	{
		gfloat progress = 0.0f;
		gulong nobjs_added = 0l;
		gchar *buf = NULL;
		FILE *fp = cstdout;

		/* Map the progress dialog? */
		if(show_progress)
		{
			if(edv_archive_add_update_progress_dialog(
				cfg_list,
				"",
				arch_path,
				progress,
				toplevel,
				FALSE
			) > 0)
			{
				(void)INTERRUPT(pid);
				pid = 0;
				status = -4;
			}
		}

		do {
			/* Check if the process is no longer running */
			if(pid > 0)
			{
				if(!edv_pid_exists(pid))
					pid = 0;
			}

			/* Update the progress? */
			if(show_progress && ProgressDialogIsQuery())
			{
				if(edv_archive_add_update_progress_dialog(
					cfg_list,
					NULL,
					NULL,
					progress,
					toplevel,
					FALSE
				) > 0)
				{
					(void)INTERRUPT(pid);
					pid = 0;
					status = -4;
					break;
				}
			}

			/* Flush any error messages */
			FDISCARD(cstderr);

			/* Read the next line */
			if(edv_stream_read_lineptr(
				fp,
				&buf,
				FALSE           /* Nonblocking */
			))
			{
				gchar *s = buf;
				const gchar *path;

				while(ISBLANK(*s))
					s++;

				path = s;

				/* Append path to the list of new paths added
				 * to the archive
				 */
				if(new_paths_list_rtn != NULL)
					*new_paths_list_rtn = g_list_append(
						*new_paths_list_rtn,
						g_strdup(path)
					);

				/* Update the progress dialog's label? */
				if(show_progress && !STRISEMPTY(path))
				{
	                                if(edv_archive_add_update_progress_dialog(
		                                cfg_list,
			                        path,
				                arch_path,
	                                        progress,
	                                        toplevel,
	                                        FALSE
	                                ) > 0)
	                                {
	                                        (void)INTERRUPT(pid);
	                                        pid = 0;
	                                        status = -4;
						break;
	                                }
				}

				nobjs_added++;
				progress = (nobjs > 0) ?
					((gfloat)nobjs_added / (gfloat)nobjs) : 0.0f;

				g_free(buf);
				buf = NULL;
			}

			edv_usleep(EDV_ITERATION_SLEEP_MIN_US);

		} while((pid > 0) || !feof(fp));

		/* Flush any error messages */
		FDISCARD(cstderr);

		g_free(buf);
	}
	/* Error or user aborted? */
	if(status != 0)
	{
		(void)edv_unlink(arch_uncompressed_path);
		CLEANUP_RETURN(status);
	}

	/* Close the output stremas */
	(void)FCLOSE(cstdout);
	cstdout = NULL;
	(void)FCLOSE(cstderr);
	cstderr = NULL;

	/* Report the final progress? */
	if(show_progress && (status == 0) && ProgressDialogIsQuery())
	{
		if(edv_archive_add_update_progress_dialog(
			cfg_list,
			NULL,
			NULL,
			1.0f,
			toplevel,
			FALSE
		) > 0)
			status = -4;
	}


	/* Format the compress archive command as needed */
	if(is_compress_compressed)
		cmd = g_strdup_printf(
			"\"%s\" -c \"%s\" > \"%s\"",
			prog_compress,
			arch_uncompressed_path,
			arch_path
		);
	else if(is_gzip_compressed)
		cmd = g_strdup_printf(
			"\"%s\" -%i -c \"%s\" > \"%s\"",
			prog_gzip,
			CLIP(compression * 9 / 100, 1, 9),
			arch_uncompressed_path,
			arch_path
		);
	else if(is_bzip2_compressed)
		cmd = g_strdup_printf(
			"\"%s\" -z -%i -c -q \"%s\" > \"%s\"",
			prog_bzip2,
			CLIP(compression * 9 / 100, 1, 9),
			arch_uncompressed_path,
			arch_path
		);
	else
		cmd = NULL;

	/* Need to compress the archive? */
	if(cmd != NULL)
	{
		/* Execute the compress archive command */
	        pid = edv_archive_add_execute(
	                cfg_list,
	                cmd,
		        NULL,
	                NULL,			/* Ignore standard output
						 * which has been redirected
						 * to the archive output */
	                &cstderr
	        );
		if(pid < 0)
		{
			core->last_error_ptr = "Unable to execute the compress command.";
			CLEANUP_RETURN(-1);
		}

		/* Delete the command */
		g_free(cmd);
		cmd = NULL;

		/* Unmap the progress dialog from the above operation */
		if(show_progress)
		{
			ProgressDialogBreakQuery(TRUE);
			ProgressDialogSetTransientFor(NULL);
		}

		/* Map the progress dialog? */
		if(show_progress)
		{
			gchar	*arch_path_shortened = edv_path_shorten(
				arch_path,
				EDV_PROGRESS_DLG_PATH_MAX_CHARS - 10
			),
					*msg = g_strdup_printf(
"Compressing Archive:\n\
\n\
    %s\n",
				arch_path_shortened
			);
			g_free(arch_path_shortened);
			edv_progress_dialog_map_archive_add_animated(
				cfg_list,
				msg,
				-1.0f,
				toplevel,
				TRUE
			);
			g_free(msg);
		}

		/* Wait for the decompress archive process to finish */
		g_timer_start(timer);
		while(edv_pid_exists(pid))
		{
			/* Update the progress? */
			if(show_progress && ProgressDialogIsQuery())
			{
				/* Update the label every second */
				if(g_timer_elapsed(timer, NULL) >= 1.0)
				{
					gchar *msg;
					EDVVFSObject *obj = edv_vfs_object_stat(arch_path);
					if(obj != NULL)
					{
						const gulong size = obj->size;
						gchar *arch_path_shortened = edv_path_shorten(
							arch_path,
							EDV_PROGRESS_DLG_PATH_MAX_CHARS - 10
						),
							*size_str = STRDUP(edv_str_size_delim(size));
						msg = g_strdup_printf(
"Compressing Archive:\n\
\n\
    %s (%s %s)\n",
							arch_path_shortened,
							size_str,
							(size == 1l) ? "byte" : "bytes"
						);
						edv_vfs_object_delete(obj);
						g_free(arch_path_shortened);
						g_free(size_str);
					}
					else
					{
						msg = NULL;
					}
					ProgressDialogUpdateUnknown(
						NULL,
						msg,
						NULL,
						NULL,
						TRUE
					);
					g_timer_start(timer);
					g_free(msg);
				}
				else
				{
					ProgressDialogUpdateUnknown(
						NULL,
						NULL,
						NULL,
						NULL,
						TRUE
					);
				}
				if(ProgressDialogStopCount() > 0)
				{
					INTERRUPT(pid);
					pid = 0;
					status = -4;
					break;
				}
			}

			/* Flush the output streams */
			FDISCARD(cstdout);
			FDISCARD(cstderr);

			edv_usleep(EDV_ITERATION_SLEEP_MIN_US);
		}

		g_timer_stop(timer);

		/* Remove the uncompressed version of the archive */
		(void)edv_unlink(arch_uncompressed_path);

		/* Close the output streams */
		(void)FCLOSE(cstderr);
		cstderr = NULL;
	}

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}
