#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <gtk/gtk.h>

#include "cfg.h"

#include "cdialog.h"
#include "progressdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_directory.h"
#include "libendeavour2-base/edv_stream.h"
#include "libendeavour2-base/edv_process.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "libendeavour2-base/edv_archive_obj.h"
#include "edv_utils_gtk.h"
#include "edv_progress.h"
#include "edv_archive_add.h"
#include "edv_archive_add_tar.h"
#include "edv_archive_add_xar.h"
#include "edv_archive_add_zip.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"


/*
 *	Return values legend:
 *
 *	0	Success.
 *	-1	General error.
 *	-2	Invalid value.
 *	-3	Systems error, out of memory, or out of disk space.
 *	-4	User responded with "Cancel".
 *	-5	User responded with "No" or response was not available.
 *	-6	An operation is already in progress.
 */


/* Error Message */
const gchar *edv_archive_add_get_error(EDVCore *core);
void edv_archive_add_set_error(
	EDVCore *core,
	const gchar *msg
);

/* Execute */
gint edv_archive_add_execute(
	CfgList *cfg_list,
	const gchar *cmd,
	FILE **cstdin_rtn,
	FILE **cstdout_rtn,
	FILE **cstderr_rtn
);

/* Progress Dialog */
gint edv_archive_add_update_progress_dialog(
	CfgList *cfg_list,
	const gchar *src_path,
	const gchar *tar_path,
	const gfloat progress,
	GtkWidget *toplevel,
	const gboolean force_remap
);

/* Calculate Total Size */
gulong edv_archive_add_calculate_total_size(
	const gchar *arch_path,
	const gchar *path,
	gint *nobjs,
	const gboolean recursive,
	const gboolean dereference_links
);

/* Add To Archive */
static gint edv_archive_add_arj(
	EDVCore *core,
	const gchar *arch_path,
	GList *src_paths_list,
	GList **new_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive,
	const gint compression,
	const gboolean dereference_links
);
static gint edv_archive_add_lha(
	EDVCore *core,
	const gchar *arch_path,
	GList *src_paths_list,
	GList **new_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive,
	const gint compression,
	const gboolean dereference_links
);
static gint edv_archive_add_rar(
	EDVCore *core,
	const gchar *arch_path,
	GList *src_paths_list,
	GList **new_paths_list_rtn,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive,
	const gint compression,
	const gboolean dereference_links
);
gint edv_archive_add(
	EDVCore *core,
	const gchar *arch_path,
	GList *src_paths_list,
	GList **new_paths_list_rtn,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive,
	const gint compression,
	const gboolean dereference_links
);


#define EDV_ITERATION_SLEEP_MIN_US	8000l
#define EDV_ARCHIVE_ADD_MAX_ERROR_MESSAGE_LINES	\
					40

#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)

#define ISBLANK(c)	(((c) == ' ') || ((c) == '\t'))
#define ISCR(c)		(((c) == '\n') || ((c) == '\r'))

#define STRCASEPFX(s,p)	((((s) != NULL) && ((p) != NULL)) ?	\
 !g_strncasecmp((s), (p), strlen(p)) : FALSE)

#define FCLOSE(p)	(((p) != NULL) ? (gint)fclose(p) : -1)
#define INTERRUPT(p)	(((p) > 0) ? (gint)kill((int)(p), SIGINT) : -1)

#define FDISCARD(p)	{			\
 if((p) != NULL) {				\
  const gint fd = (gint)fileno(p);		\
  while(edv_poll_read(fd)) {			\
   if(fgetc(p) == EOF)				\
    break;					\
  }						\
 }						\
}


/*
 *	Returns the last error message as a statically allocated
 *	string or NULL if there was no previous error.
 */
const gchar *edv_archive_add_get_error(EDVCore *core)
{
	return((core != NULL) ? core->last_error_ptr : NULL);
}

/*
 *	Coppies the error message to the core's last_error_buf
 *	and sets the core's last_error_ptr to point to it.
 */
void edv_archive_add_set_error(
	EDVCore *core,
	const gchar *msg
)
{
	if(core == NULL)
		return;

	g_free(core->last_error_buf);
	core->last_error_ptr = core->last_error_buf = STRDUP(msg);
}


/*
 *	Executes the command using the shell specified by the
 *	configuration and opens and returns the requested streams.
 */
gint edv_archive_add_execute(
	CfgList *cfg_list,
	const gchar *cmd,
	FILE **cstdin_rtn,
	FILE **cstdout_rtn,
	FILE **cstderr_rtn
)
{
	gchar		*shell_prog;
	const gchar	*shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL),
			*shell_args = edv_strarg(
			shell_cmd,
			&shell_prog,
			TRUE,			/* Parse escapes */
			TRUE			/* Parse quotes */
	);

	/* Execute the list archive command */
	const gint pid = edv_system_shell_streams(
		cmd,
		shell_prog,
		shell_args,
		cstdin_rtn,
		cstdout_rtn,
		cstderr_rtn
	);

	g_free(shell_prog);

	return(pid);
}


/*
 *      Updates/maps the progress dialog.
 */
gint edv_archive_add_update_progress_dialog(
	CfgList *cfg_list,
	const gchar *src_path,
	const gchar *tar_path,
	const gfloat progress,
	GtkWidget *toplevel,
	const gboolean force_remap
) 
{
	if((src_path != NULL) && (tar_path != NULL))
	{
		gchar   *src_path_shortened = edv_path_shorten(
			src_path,
			EDV_PROGRESS_DLG_PATH_MAX_CHARS
		),
			*tar_path_shortened = edv_path_shorten(
			tar_path,
			EDV_PROGRESS_DLG_PATH_MAX_CHARS
		),
			*msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Agregar:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Addition:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Hinzufgen:\n\
\n\
    %s\n\
\n\
Zu:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Aggiunta:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Toevoegen:\n\
\n\
    %s\n\
\n\
Te:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Adicionar:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Tilfying:\n\
\n\
    %s\n\
\n\
Til:\n\
\n\
    %s\n"
#else
"Adding:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n"
#endif
			,
			src_path_shortened,
			tar_path_shortened
		);
		g_free(src_path_shortened);
		g_free(tar_path_shortened);
		edv_progress_dialog_map_archive_add_animated(
			cfg_list,
			msg,
			progress,
			toplevel,
			force_remap
		);
		g_free(msg);
	}
	else
	{
		if(progress >= 0.0f)
			ProgressDialogUpdate(
				NULL,
				NULL,
				NULL,
				NULL,
				progress,
				EDV_PROGRESS_BAR_NTICKS,
				TRUE
			);
		else
			ProgressDialogUpdateUnknown(
				NULL,
				NULL,
				NULL,
				NULL,
				TRUE
			);
	}

	return(ProgressDialogStopCount());
}


/*
 *	Calculates the size and number of objects of the specified
 *	path.
 *
 *	If recursive is TRUE and the specified path is a directory
 *	then the sizes and number of all the objects within that
 *	directory will be counted.
 *
 *	If dereference_links is TRUE and the specified path is a link
 *	then the size of the link's target will be counted if it
 *	leads to a file. Otherwise the link's size is counted.
 *
 *	The *nobjs will be added with the number of objects
 *	counted.
 *
 *	Returns the total size of all the objects of the specified
 *	path. Only the sizes of files and links are counted.
 */
gulong edv_archive_add_calculate_total_size(
	const gchar *arch_path,
	const gchar *path,
	gint *nobjs,
	const gboolean recursive,
	const gboolean dereference_links
)
{
	gulong subtotal = 0l;
	EDVObjectType type;
	EDVVFSObject *obj;

	if(STRISEMPTY(path))
		return(subtotal);

	/* Skip the archive itself */
	if(!strcmp((const char *)arch_path, path))
		return(subtotal);

	obj = edv_vfs_object_lstat(path);
	if(obj == NULL)
		return(subtotal);

	/* Count this object */
	*nobjs = (*nobjs) + 1;

	type = obj->type;

	/* Directory? */
	if((type == EDV_OBJECT_TYPE_DIRECTORY) && recursive)
	{
		EDVDirectory *dp = edv_directory_open(
			path,
			FALSE,			/* Unsorted */
			FALSE			/* Exclude notations */
		);
		if(dp != NULL)
		{
			const gchar *name;
			gchar *full_path;

			for(name = edv_directory_next(dp);
			    name != NULL;
			    name = edv_directory_next(dp)
			)
			{
				full_path = g_strconcat(
					path,
					G_DIR_SEPARATOR_S,
					name,
					NULL
				);
				subtotal += edv_archive_add_calculate_total_size(
					arch_path,
					full_path,
					nobjs,
					recursive,
					dereference_links
				);
				g_free(full_path);
			}

			edv_directory_close(dp);
		}
	}
	/* Link? */
	else if(type == EDV_OBJECT_TYPE_LINK)
	{
		/* Dereference this link? */
		if(dereference_links)
		{
			/* Get the destination object's stats */
			EDVVFSObject *obj = edv_vfs_object_stat(path);
			if(obj != NULL)
			{
				const EDVObjectType type = obj->type;

				/* Destination is a directory? */
				if((type == EDV_OBJECT_TYPE_DIRECTORY) && recursive)
				{
					EDVDirectory *dp = edv_directory_open(
						path,
						FALSE,		/* Unsorted */
						FALSE		/* Exclude notations */
					);
					if(dp != NULL)
					{
						const gchar *name;
						gchar *full_path;

						for(name = edv_directory_next(dp);
						    name != NULL;
						    name = edv_directory_next(dp)
						)
						{
							full_path = g_strconcat(
								path,
								G_DIR_SEPARATOR_S,
								name,
								NULL
							);
							subtotal += edv_archive_add_calculate_total_size(
								arch_path,
								full_path,
								nobjs,
								recursive,
								dereference_links
							);
							g_free(full_path);
						}

						edv_directory_close(dp);
					}
				}
				/* Destination is a file? */
				else if(type == EDV_OBJECT_TYPE_FILE)
				{
					subtotal += obj->size;
				}

				edv_vfs_object_delete(obj);
			}
		}
		else
		{
			/* Not dereferencing links so count this link's size */
			subtotal += obj->size;
		}
	}
	/* File? */
	else if(type == EDV_OBJECT_TYPE_FILE)
	{
		subtotal += obj->size;
	}

	edv_vfs_object_delete(obj);

	return(subtotal);
}


/*
 *	Add objects to an ARJ archive.
 */
static gint edv_archive_add_arj(
	EDVCore *core,
	const gchar *arch_path,
	GList *src_paths_list,
	GList **new_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive,
	const gint compression,
	const gboolean dereference_links
)
{
	FILE            *cstdout = NULL,
			*cstderr = NULL;
	gint		status,
			pid,
			nobjs;
	gulong total_size;
	gchar		*s,
			*parent_path = NULL,
			*pwd = NULL,
			*cmd = NULL;
	CfgList *cfg_list = core->cfg_list;
	const gchar	*path,
			*prog_arj = EDV_GET_S(EDV_CFG_PARM_PROG_ARJ);
	GList *glist;
	EDVVFSObject *arch_obj;

#define CLEANUP_RETURN(_v_)	{		\
 (void)FCLOSE(cstdout);				\
 (void)FCLOSE(cstderr);				\
 g_free(parent_path);				\
 g_free(cmd);					\
						\
 /* Restore the previous working dir */		\
 if(pwd != NULL) {				\
  (void)edv_setcwd(pwd);			\
  g_free(pwd);					\
 }						\
						\
 return(_v_);					\
}

	/* Record the previous working directory and set the new
	 * working directory
	 */
	pwd = edv_getcwd();
	parent_path = g_dirname(arch_path);
	(void)edv_setcwd(parent_path);

	/* If the ARJ archive exists and it is empty then it must be
	 * removed first before adding objects to it
	 */
	arch_obj = edv_vfs_object_stat(arch_path);
	if(arch_obj != NULL)
	{
		if(arch_obj->size == 0l)
			(void)edv_unlink(arch_path);

		edv_vfs_object_delete(arch_obj);
	}

	/* Format the add to archive command, calculate the number of
	 * objects to add, and the total size of all the objects to
	 * add
	 */
	cmd = g_strdup_printf(
		"\"%s\" a -i -y -m%i %s\"%s\"",
		prog_arj,
		CLIP(compression * 4 / 100, 0, 4),
		recursive ? "-r " : "",
		arch_path
	);
	nobjs = 0;
	total_size = 0l;
	for(glist = src_paths_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		path = (const gchar *)glist->data;
		if(STRISEMPTY(path))
			continue;

		/* Count the size and number of objects */
		total_size += edv_archive_add_calculate_total_size(
			arch_path,
			path,
			&nobjs,
			recursive,
			dereference_links
		);

		/* Seek path to the relative path as needed */
		if(edv_path_is_parent(
			parent_path,
			path
		))
		{
			path += strlen(parent_path);
			while(*path == G_DIR_SEPARATOR)
				path++;
		}

		s = g_strconcat(
			cmd,
			" \"",
			path,
			"\"",
			NULL
		);
		if(s != NULL)
		{
			g_free(cmd);
			cmd = s;
		}
	}
	if(cmd == NULL)
	{
		core->last_error_ptr = "Unable to generate the add command.";
		CLEANUP_RETURN(-1);
	}

	/* Execute the add objects to archive command */
	pid = edv_archive_add_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	if(pid < 0)
	{
		core->last_error_ptr = "Unable to execute the add command.";
		CLEANUP_RETURN(-1);
	}

	/* Delete the command */
	g_free(cmd);
	cmd = NULL;

	status = 0;

	/* Read the archive add messages from the output streams */
	if((cstdout != NULL) && (cstderr != NULL))
	{
		gint nobjs_added = 0;
		gfloat progress = 0.0f;
		gchar *buf = NULL;
		FILE *fp = cstdout;

		/* Map the progress dialog? */
		if(show_progress)
		{
			if(edv_archive_add_update_progress_dialog(
				cfg_list,
				"",
				arch_path,
				progress,
				toplevel,
				FALSE
			) > 0)
			{
				(void)INTERRUPT(pid);
				pid = 0;
				status = -4;
			}
		}

		do {
			/* Check if the process is no longer running */
			if(pid > 0)
			{
				if(!edv_pid_exists(pid))
					pid = 0;
			}

			/* Update the progress? */
			if(show_progress && ProgressDialogIsQuery())
			{
				if(edv_archive_add_update_progress_dialog(
					cfg_list,
					NULL,
					NULL,
					progress,
					toplevel,
					FALSE
				) > 0)
				{
					(void)INTERRUPT(pid);
					pid = 0;
					status = -4;
					break;
				}
			}

			/* Flush any error messages */
			FDISCARD(cstderr);

			/* Read the next line */
			if(edv_stream_read_lineptr(
				fp,
				&buf,
				FALSE		/* Nonblocking */
			))
			{
				gchar *s = buf;
				while(ISBLANK(*s))
					s++;

				if(STRCASEPFX(s, "Adding") ||
				   STRCASEPFX(s, "Replacing")
				)
				{
					gchar *s_end;
					const gchar *path;

					/* Seek s past the prefix to the path value */
					while(!ISBLANK(*s) && (*s != '\0'))
						s++;
					while(ISBLANK(*s))
						s++;

					/* Cap the first blank character after path */
					for(s_end = s; *s_end != '\0'; s_end++)
					{
						if(ISBLANK(*s_end))
						{
							*s_end = '\0';
							break;
						}
					}

					path = s;

					/* Append this path to the list
					 * of paths added to the archive
					 */
					if(new_paths_list_rtn != NULL)
						*new_paths_list_rtn = g_list_append(
							*new_paths_list_rtn,
							g_strdup(path)
						);

					/* Update the progress dialog's label */
					if(show_progress &&
					   !STRISEMPTY(path)
					)
					{
						if(edv_archive_add_update_progress_dialog(
							cfg_list,
							path,
							arch_path,
							progress,
							toplevel,
							FALSE
						) > 0)
						{
							(void)INTERRUPT(pid);
							pid = 0;
							status = -4;
							break;
						}
					}

					nobjs_added++;
					progress = (nobjs > 0) ?
						((gfloat)nobjs_added / (gfloat)nobjs) : 0.0f;
			       }

				g_free(buf);
				buf = NULL;
			}

			edv_usleep(EDV_ITERATION_SLEEP_MIN_US);

		} while((pid > 0) || !feof(fp));

		/* Flush any error messages */
		FDISCARD(cstderr);

		g_free(buf);
	}

	/* Close the output stremas */
	(void)FCLOSE(cstdout);
	cstdout = NULL;
	(void)FCLOSE(cstderr);
	cstderr = NULL;

	/* Report the final progress? */
	if(show_progress && (status == 0) && ProgressDialogIsQuery())
	{
		if(edv_archive_add_update_progress_dialog(
			cfg_list,
			NULL,
			NULL,
			1.0f,
			toplevel,
			FALSE
		) > 0)
			status = -4;
	}

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Add objects to a LHA archive.
 */
static gint edv_archive_add_lha(
	EDVCore *core,
	const gchar *arch_path,
	GList *src_paths_list,
	GList **new_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive,
	const gint compression,
	const gboolean dereference_links
)
{
	FILE            *cstdout = NULL,
			*cstderr = NULL;
	gint		status,
			pid,
			nobjs;
	gulong total_size;
	gchar		*s,
			*parent_path = NULL,
			*pwd = NULL,
			*cmd = NULL;
	CfgList *cfg_list = core->cfg_list;
	const gchar	*path,
			*prog_lha = EDV_GET_S(EDV_CFG_PARM_PROG_LHA);
	GList *glist;

#define CLEANUP_RETURN(_v_)	{		\
 (void)FCLOSE(cstdout);				\
 (void)FCLOSE(cstderr);				\
 g_free(parent_path);				\
 g_free(cmd);					\
						\
 /* Restore the previous working dir */		\
 if(pwd != NULL) {				\
  (void)edv_setcwd(pwd);			\
  g_free(pwd);					\
 }						\
						\
 return(_v_);					\
}

	/* Record the previous working directory and set the new
	 * working directory
	 */
	pwd = edv_getcwd();
	parent_path = g_dirname(arch_path);
	(void)edv_setcwd(parent_path);

	/* Format the add to archive command, calculate the number of
	 * objects to add, and the total size of all the objects to
	 * add
	 */
	cmd = g_strdup_printf(
		"\"%s\" -aq1fo%i \"%s\"",
		prog_lha,
		CLIP((compression * 2 / 100) + 5, 5, 7),
		arch_path
	);
	nobjs = 0;
	total_size = 0l;
	for(glist = src_paths_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		path = (const gchar *)glist->data;
		if(STRISEMPTY(path))
			continue;

		/* Count the size and number of objects */
		total_size += edv_archive_add_calculate_total_size(
			arch_path,
			path,
			&nobjs,
			recursive,
			dereference_links
		);

		/* Seek path to the relative path as needed */
		if(edv_path_is_parent(
			parent_path,
			path
		))
		{
			path += strlen(parent_path);
			while(*path == G_DIR_SEPARATOR)
				path++;
		}

		s = g_strconcat(
			cmd,
			" \"",
			path,
			"\"",
			NULL
		);
		if(s != NULL)
		{
			g_free(cmd);
			cmd = s;
		}
	}
	if(cmd == NULL)
	{
		core->last_error_ptr = "Unable to generate the add command.";
		CLEANUP_RETURN(-1);
	}

	/* Execute the add objects to archive command */
	pid = edv_archive_add_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	if(pid < 0)
	{
		core->last_error_ptr = "Unable to execute the add command.";
		CLEANUP_RETURN(-1);
	}

	/* Delete the command */
	g_free(cmd);
	cmd = NULL;

	status = 0;

	/* Read the archive add messages from the output streams */
	if((cstdout != NULL) && (cstderr != NULL))
	{
		gint nobjs_added = 0;
		gfloat progress = 0.0f;
		gchar *buf = NULL;
		FILE *fp = cstdout;

		/* Map the progress dialog? */
		if(show_progress)
		{
			if(edv_archive_add_update_progress_dialog(
				cfg_list,
				"",
				arch_path,
				progress,
				toplevel,
				FALSE
			) > 0)
			{
				(void)INTERRUPT(pid);
				pid = 0;
				status = -4;
			}
		}

		do {
			/* Check if the process is no longer running */
			if(pid > 0)
			{
				if(!edv_pid_exists(pid))
					pid = 0;
			}

			/* Update the progress? */
			if(show_progress && ProgressDialogIsQuery())
			{
				if(edv_archive_add_update_progress_dialog(
					cfg_list,
					NULL,
					NULL,
					progress,
					toplevel,
					FALSE
				) > 0)
				{
					(void)INTERRUPT(pid);
					pid = 0;
					status = -4;
					break;
				}
			}

			/* Flush any error messages */
			FDISCARD(cstderr);

			/* Read the next line */
			if(edv_stream_read_lineptr(
				fp,
				&buf,
				FALSE		/* Nonblocking */
			))
			{
				gchar	*s = buf,
					*s_end;
				while(ISBLANK(*s))
					s++;

				/* Cap the end of the path deliminator */
				s_end = (gchar *)strstr((char *)s, " :");
				if(s_end != NULL)
				{
					const gchar *path;

					*s_end = '\0';

					path = s;

					/* Append this path to the list
					 * of paths added to the archive
					 */
					if(new_paths_list_rtn != NULL)
					    *new_paths_list_rtn = g_list_append(
							*new_paths_list_rtn,
							g_strdup(path)
						);

					/* Update the progress dialog's label */
					if(show_progress &&
					   !STRISEMPTY(path)
					)
					{
						if(edv_archive_add_update_progress_dialog(
							cfg_list,
							path,
							arch_path,
							progress,
							toplevel,
							FALSE
						) > 0)
						{
							(void)INTERRUPT(pid);
							pid = 0;
							status = -4;
							break;
						}
					}

					nobjs_added++;
					progress = (nobjs > 0) ?
						((gfloat)nobjs_added / (gfloat)nobjs) : 0.0f;
				}

				g_free(buf);
				buf = NULL;
			}

			edv_usleep(EDV_ITERATION_SLEEP_MIN_US);

		} while((pid > 0) || !feof(fp));

		/* Flush any error messages */
		FDISCARD(cstderr);

		g_free(buf);
	}

	/* Close the output stremas */
	(void)FCLOSE(cstdout);
	cstdout = NULL;
	(void)FCLOSE(cstderr);
	cstderr = NULL;

	/* Report the final progress? */
	if(show_progress && (status == 0) && ProgressDialogIsQuery())
	{
		if(edv_archive_add_update_progress_dialog(
			cfg_list,
			NULL,
			NULL,
			1.0f,
			toplevel,
			FALSE
		) > 0)
			status = -4;
	}

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Add objects to a RAR archive.
 */
static gint edv_archive_add_rar(
	EDVCore *core,
	const gchar *arch_path,
	GList *src_paths_list,
	GList **new_paths_list_rtn,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive,
	const gint compression,
	const gboolean dereference_links
)
{
	FILE            *cstdout = NULL,
			*cstderr = NULL;
	gint		status,
			pid,
			nobjs;
	gulong total_size;
	gchar		*s,
			*parent_path = NULL,
			*pwd = NULL,
			*cmd = NULL;
	CfgList *cfg_list = core->cfg_list;
	const gchar	*path,
			*prog_rar = EDV_GET_S(EDV_CFG_PARM_PROG_RAR);
	GList *glist;
	EDVVFSObject *arch_obj;

#define CLEANUP_RETURN(_v_)	{		\
 (void)FCLOSE(cstdout);				\
 (void)FCLOSE(cstderr);				\
 g_free(parent_path);				\
 g_free(cmd);					\
						\
 /* Restore the previous working dir */		\
 if(pwd != NULL) {				\
  (void)edv_setcwd(pwd);			\
  g_free(pwd);					\
 }						\
						\
 return(_v_);					\
}

	/* Record the previous working directory and set the new
	 * working directory
	 */
	pwd = edv_getcwd();
	parent_path = g_dirname(arch_path);
	(void)edv_setcwd(parent_path);

	/* If the RAR archive exists and it is empty then it must be
	 * removed first before adding objects to it
	 */
	arch_obj = edv_vfs_object_stat(arch_path);
	if(arch_obj != NULL)
	{
		if(arch_obj->size == 0l)
			(void)edv_unlink(arch_path);

		edv_vfs_object_delete(arch_obj);
	}

	/* Format the add objects to an RAR archive command */
	if(STRISEMPTY(password))
		cmd = g_strdup_printf(
			"\"%s\" a%s -m%i%s -y \"%s\"",
			prog_rar,
			dereference_links ? "" : " -ol",
			CLIP(compression * 5 / 100, 0, 5),
			recursive ? " -r" : "",
			arch_path
		);
	else
		cmd = g_strdup_printf(
			"\"%s\" a%s \"-p%s\" -m%i%s -y \"%s\"",
			prog_rar,
			dereference_links ? "" : " -ol",
			password,
			CLIP(compression * 5 / 100, 0, 5),
			recursive ? " -r" : "",
			arch_path
		);
	nobjs = 0;
	total_size = 0l;
	for(glist = src_paths_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		path = (const gchar *)glist->data;
		if(STRISEMPTY(path))
			continue;

		/* Count the size and number of objects */
		total_size += edv_archive_add_calculate_total_size(
			arch_path,
			path,
			&nobjs,
			recursive,
			dereference_links
		);

		/* Seek tpath to the relative path as needed */
		if(edv_path_is_parent(
			parent_path,
			path
		))
		{
			path += strlen(parent_path);
			while(*path == G_DIR_SEPARATOR)
				path++;
		}

		s = g_strconcat(
			cmd,
			" \"",
			path,
			"\"",
			NULL
		);
		if(s != NULL)
		{
			g_free(cmd);
			cmd = s;
		}
	}
	if(cmd == NULL)
	{
		core->last_error_ptr = "Unable to generate the add command.";
		CLEANUP_RETURN(-1);
	}

	/* Execute the add objects to archive command */
	pid = edv_archive_add_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	if(pid < 0)
	{
		core->last_error_ptr = "Unable to execute the add command.";
		CLEANUP_RETURN(-1);
	}

	/* Delete the command */
	g_free(cmd);
	cmd = NULL;

	status = 0;

	/* Read the archive add messages from the output streams */
	if((cstdout != NULL) && (cstderr != NULL))
	{
		const gint max_errors = EDV_ARCHIVE_ADD_MAX_ERROR_MESSAGE_LINES;
		gint	nerrors = 0,
			nobjs_added = 0;
		gfloat progress = 0.0f;
		gchar	*buf = NULL,
			*error_buf = NULL;
		FILE *fp = cstdout;

		/* Map the progress dialog? */
		if(show_progress)
		{
			if(edv_archive_add_update_progress_dialog(
				cfg_list,
				"",
				arch_path,
				progress,
				toplevel,
				FALSE
			) > 0)
			{
				(void)INTERRUPT(pid);
				pid = 0;
				status = -4;
			}
		}

		do {
			/* Check if the process is no longer running */
			if(pid > 0)
			{
				if(!edv_pid_exists(pid))
					pid = 0;
			}

			/* Update the progress? */
			if(show_progress && ProgressDialogIsQuery())
			{
				if(edv_archive_add_update_progress_dialog(
					cfg_list,
					NULL,
					NULL,
					progress,
					toplevel,
					FALSE
				) > 0)
				{
					(void)INTERRUPT(pid);
					pid = 0;
					status = -4;
					break;
				}
			}

			/* Read the next line */
			if(edv_stream_read_lineptr(
				fp,
				&buf,
				FALSE		/* Nonblocking */
			))
			{
				gchar *s = buf;
				while(ISBLANK(*s))
					s++;

				if(STRCASEPFX(s, "Adding  ") ||
				   STRCASEPFX(s, "Updating  ")
				)
				{
					gchar *s_end;
					const gchar *path;

					/* Seek s past the prefix to the path value */
					while(!ISBLANK(*s) && (*s != '\0'))
						s++;
					while(ISBLANK(*s))
						s++;

					/* Cap the first blank character after path */
					for(s_end = s; *s_end != '\0'; s_end++)
					{
						if(ISBLANK(*s_end))
						{
							*s_end = '\0';
							break;
						}
					}

					path = s;

					/* Append this path to the list
					 * of paths added to the archive
					 */
					if(new_paths_list_rtn != NULL)
						*new_paths_list_rtn = g_list_append(
							*new_paths_list_rtn,
							g_strdup(path)
						);

					/* Update the progress dialog's label */
					if(show_progress &&
					   !STRISEMPTY(path)
					)
					{
						if(edv_archive_add_update_progress_dialog(
							cfg_list,
							path,
							arch_path,
							progress,
							toplevel,
							FALSE
						) > 0)
						{
							(void)INTERRUPT(pid);
							pid = 0;
							status = -4;
							break;
						}
					}

					nobjs_added++;
					progress = (nobjs > 0) ?
						((gfloat)nobjs_added / (gfloat)nobjs) : 0.0f;
				}

				g_free(buf);
				buf = NULL;
			}

			/* Check for errors and read no more than
			 * max_errors error messages from the error
			 * stream
			 */
			if(nerrors < max_errors)
			{
				if(edv_stream_read_lineptr(
					cstderr,
					&error_buf,
					FALSE		/* Nonblocking */
				))
					nerrors++;
			}
			else
			{
				FDISCARD(cstderr);
			}

			edv_usleep(EDV_ITERATION_SLEEP_MIN_US);

		} while((pid > 0) || !feof(fp));

		/* Record any errors */
		if(nerrors > 0)
		{
			if(core->last_error_ptr == NULL)
				edv_archive_add_set_error(
					core,
					error_buf
				);
			if(status == 0)
				status = -1;
		}

		g_free(error_buf);
		g_free(buf);
	}

	/* Close the output stremas */
	(void)FCLOSE(cstdout);
	cstdout = NULL;
	(void)FCLOSE(cstderr);
	cstderr = NULL;

	/* Report the final progress? */
	if(show_progress && (status == 0) && ProgressDialogIsQuery())
	{
		if(edv_archive_add_update_progress_dialog(
			cfg_list,
			NULL,
			NULL,
			1.0f,
			toplevel,
			FALSE
		) > 0)
			status = -4;
	}

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Adds the object(s) to the archive.
 *
 *	The arch_path specifies the archive.
 *
 *	The src_paths_list specifies the list of objects to add to the
 *	archive.
 *
 *	If new_paths_list_rtn is not NULL then a list of paths
 *	describing the objects that have been added to the archive will
 *	be returned. The calling function must delete the returned list
 *	and each string.
 *
 *	If password is not NULL then a password will be used to
 *	encrypt the objects being added to the archive (if the archive
 *	format supports encryption).
 *
 *	If recursive is TRUE then all the objects in any directory
 *	object specified in src_paths_list will be added as well.
 *
 *	The compression specifies the compression level from 0 to
 *	100 where 0 is no compression and 100 is maximum compression.
 *	This value has different affects on different archive formats.
 *
 *	If dereference_links is TRUE then the link's target will
 *	be added instead of the link itself.
 */
gint edv_archive_add(
	EDVCore *core,
	const gchar *arch_path,
	GList *src_paths_list,
	GList **new_paths_list_rtn,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive,
	const gint compression,
	const gboolean dereference_links
)
{
	gint status;
	const gulong time_start = edv_time();
	const gchar	*s,
			*arch_name;
	gchar *path;
	GList		*glist,
			*dsrc_paths_list = NULL;

#define CLEANUP_RETURN(_v_)	{		\
 if(dsrc_paths_list != NULL) {			\
  g_list_foreach(				\
   dsrc_paths_list, (GFunc)g_free, NULL		\
  );						\
  g_list_free(dsrc_paths_list);			\
 }						\
						\
 return(_v_);					\
}

	/* Reset the last error message */
	edv_archive_add_set_error(core, NULL);

	/* Reset the return values */
	if(new_paths_list_rtn != NULL)
		*new_paths_list_rtn = NULL;

	if((core == NULL) || STRISEMPTY(arch_path) ||
	   (yes_to_all == NULL)
	)
	{
		edv_archive_add_set_error(core, "Invalid value.");
		return(-2);
	}

	/* No objects to add? */
	if(src_paths_list == NULL)
		return(0);

	arch_name = g_basename(arch_path);

	/* Make a copy of the list of objects to be added to the
	 * archive as dsrc_paths_list
	 */
	for(glist = src_paths_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		s = (const gchar *)glist->data;
		if(STRISEMPTY(s))
			continue;

		path = STRDUP(s);
		if(path == NULL)
			continue;

		/* Strip any tailing deliminators */
		edv_path_simplify(path);

		dsrc_paths_list = g_list_append(
			dsrc_paths_list,
			path
		);
	}
	if(dsrc_paths_list == NULL)
	{
		CLEANUP_RETURN(-1);
	}

	/* Add the objects to the archive based on the archive's format
	 *
	 * ARJ Archive
	 */
	if(edv_name_has_extension(arch_name, ".arj"))
	{
		status = edv_archive_add_arj(
			core,
			arch_path,
			dsrc_paths_list,
			new_paths_list_rtn,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			recursive,
			compression,
			dereference_links
		);
	}
	/* LHA Archive */
	else if(edv_name_has_extension(arch_name, ".lha"))
	{
		status = edv_archive_add_lha(
			core,
			arch_path,
			dsrc_paths_list,
			new_paths_list_rtn,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			recursive,
			compression,
			dereference_links
		);
	}
	/* RAR Archive */
	else if(edv_name_has_extension(arch_name, ".rar"))
	{
		status = edv_archive_add_rar(
			core,
			arch_path,
			dsrc_paths_list,
			new_paths_list_rtn,
			password,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			recursive,
			compression,
			dereference_links
		);
	}
	/* Tape Archive (Compressed) */
	else if(edv_name_has_extension(arch_name, ".tar.Z"))
	{
		status = edv_archive_add_tar(
			core,
			arch_path,
			dsrc_paths_list,
			new_paths_list_rtn,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			recursive,
			compression,
			dereference_links,
			TRUE,			/* Is compress compressed */
			FALSE,			/* Not gzip compressed */
			FALSE			/* Not bzip2 compressed */
		);
	}
	/* Tape Archive (GZip) */
	else if(edv_name_has_extension(arch_name, ".tgz .tar.gz"))
	{
		status = edv_archive_add_tar(
			core,
			arch_path,
			dsrc_paths_list,
			new_paths_list_rtn,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			recursive,
			compression,
			dereference_links,
			FALSE,			/* Not compress compressed */
			TRUE,			/* Is gzip compressed */
			FALSE			/* Not bzip2 compressed */
		);
	}
	/* Tape Archive (BZip2) */
	else if(edv_name_has_extension(arch_name, ".tar.bz2"))
	{
		status = edv_archive_add_tar(
			core,
			arch_path,
			dsrc_paths_list,
			new_paths_list_rtn,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			recursive,
			compression,
			dereference_links,
			FALSE,			/* Not compress compressed */
			FALSE,			/* Not gzip compressed */
			TRUE			/* Is bzip2 compressed */
		);
	}
	/* Tape Archive */
	else if(edv_name_has_extension(arch_name, ".tar"))
	{
		status = edv_archive_add_tar(
			core, arch_path,
			dsrc_paths_list,
			new_paths_list_rtn,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			recursive,
			compression,
			dereference_links,
			FALSE,			/* Not compress compressed */
			FALSE,			/* Not gzip compressed */
			FALSE			/* Not bzip2 compressed */
		);
	}
	/* X Archive */
	else if(edv_name_has_extension(arch_name, ".xar"))
	{
		status = edv_archive_add_xar(
			core,
			arch_path,
			dsrc_paths_list,
			new_paths_list_rtn,
			password,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			recursive,
			compression,
			dereference_links
		);
	}
	/* PKZip Archive */
	else if(edv_name_has_extension(arch_name, ".zip .xpi .jar"))
	{
		status = edv_archive_add_zip(
			core, arch_path,
			dsrc_paths_list,
			new_paths_list_rtn,
			password,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			recursive,
			compression,
			dereference_links
		);
	}
	else
	{
		core->last_error_ptr = "Unsupported archive format.";
		status = -2;
	}

	/* Record history */
	if(status != 0)
	{
		const gulong time_end = edv_time();
		const gchar *first_src_obj = (dsrc_paths_list != NULL) ?
			(const gchar *)dsrc_paths_list->data : NULL;

		edv_append_history(
			core,
			EDV_HISTORY_ARCHIVE_OBJECT_ADD,
			time_start,
			time_end,
			status,
			first_src_obj,		/* Source */
			arch_path,		/* Target */
			core->last_error_ptr	/* Comment */
		);
	}
	else
	{
		const gulong time_end = edv_time();
		for(glist = dsrc_paths_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
			edv_append_history(
				core,
				EDV_HISTORY_ARCHIVE_OBJECT_ADD,
				time_start,
				time_end,
				status,
				(const gchar *)glist->data,	/* Source */
				arch_path,	/* Target */
				core->last_error_ptr	/* Comment */
			);
	}

	/* Need to flush disk changes since the archive may have been
	 * modified on another process and the changes have not reached
	 * our process yet
	 */
	(void)edv_sync();

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}
