#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <gtk/gtk.h>

#include "../guiutils.h"
#include "../fprompt.h"
#include "../cdialog.h"
#include "../pdialog.h"
#include "../fb.h"
#include "../progressdialog.h"

#include "../libendeavour2-base/endeavour2.h"

#include "dlf_cfg_fio.h"
#include "config.h"


#include "../images/icon_download_file_32x32.xpm"
#include "../images/icon_download_file_48x48.xpm"


#if 0
static gulong dlf_get_avg_time_from_buffer(
	const gulong *buf,
	const gint nitems
);
#endif

static void dlf_signal_cb(int s);
static gchar *dlf_browse_target_path_cb(
	gpointer d, gpointer data, gint prompt_num
);

static void dlf_file_selector_created_cb(
	const gchar *path, gpointer data
);
static void dlf_file_selector_modified_cb(
	const gchar *old_path,
	const gchar *new_path,
	gpointer data
);
static void dlf_file_selector_delete_cb(
	const gchar *path, gpointer data
);

static gulong dlf_get_total_length_from_stream(
	FILE *fp,
	gchar **line_buf_ptr
);
static gint dlf_monitor(
	EDVContext *ctx,
	const gint pid,
	const gulong check_interval_ms,
	const gulong start_time,
	const gchar *src_path,
	const gchar *tar_path,
	FILE *cstdout,
	FILE *cstderr,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry,
	const gboolean keep_dialog
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)

#define ISBLANK(c)	(((c) == ' ') || ((c) == '\t'))
#define FCLOSE(_fp_)    (((_fp_) != NULL) ? (gint)fclose(_fp_) : -1)

#define STRPFX(s,p)	((((s) != NULL) && ((p) != NULL)) ?	\
 !strncmp((const char *)(s),(const char *)(p),strlen(p)) : FALSE)


#if 0
/*
 *	Returns the average value of the specified time buffer.
 *
 *	Note that this average is biased to the last (highest) index
 *	value.
 */
static gulong dlf_get_avg_time_from_buffer(
	const gulong *buf,
	const gint nitems 
)
{
	gint i;
	gulong l;

	if((buf == NULL) || (nitems <= 0))
		return(0l);

	if(nitems == 1)
		return(buf[0]);

	l = buf[0];
	for(i = 1; i < nitems; i++)
		l += buf[i];

	return(l / nitems);
}
#endif


/*
 *	UNIX signal callback.
 */
static void dlf_signal_cb(int s)
{
	switch(s)
	{
	  case SIGINT:
	  case SIGTERM:
	  case SIGSEGV:
		exit(1);
		break;
	}
}

/*
 *	Prompt dialog browse target path callback.
 */
static gchar *dlf_browse_target_path_cb(
	gpointer d, gpointer data, gint prompt_num
)
{
	gboolean response;
	gint		nftypes = 0,
			npaths = 0;
	gchar **paths_list = NULL;       
	GtkWidget *toplevel = NULL;
	fb_type_struct	**ftypes_list = NULL,
			*ftype_rtn = NULL;

	if(FileBrowserIsQuery())
		return(NULL);

	/* Create the file types list */
	FileBrowserTypeListNew(
		&ftypes_list, &nftypes,
		"*.*", "All Files"
	);

	/* Query the user for a path */
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"El Sendero Selecto",
"Selecto",
"Cancele",
#elif defined(PROG_LANGUAGE_FRENCH)
"Choisir le rpertoire",
"Choisir",
"Annuler",
#elif defined(PROG_LANGUAGE_GERMAN)
"Erlesener Pfad",
"Erlesen.",
"Heben",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scegliere Il Sentiero",
"Scegliere",
"Annullare",
#elif defined(PROG_LANGUAGE_DUTCH)
"Uitgezocht Pad",
"Uitgezocht",
"Annuleer",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Selecione Caminho",
"Selecione",
"Cancelamento",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Utvalgt Sti",
"Utvalgt",
"Kanseller",
#else
"Select Path",
"Select",
"Cancel",
#endif
		PDialogGetPromptValue(prompt_num),
		ftypes_list, nftypes,
		&paths_list, &npaths,
		&ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);

	/* Reset the file browser due to possible file related change */
	FileBrowserReset();

	return((response && (npaths > 0)) ? paths_list[0] : NULL);
}


/*
 *	File selector created callback.
 */
static void dlf_file_selector_created_cb(
	const gchar *path, gpointer data
) 
{
	EDVContext *ctx = EDV_CONTEXT(data);
	edv_notify_queue_vfs_object_added(ctx, path);
	edv_context_flush(ctx);
}

/*
 *	File selector changed callback.
 */
static void dlf_file_selector_modified_cb(
	const gchar *old_path,
	const gchar *new_path,
	gpointer data
)
{
	EDVContext *ctx = EDV_CONTEXT(data);
	edv_notify_queue_vfs_object_modified(ctx, old_path, new_path);
	edv_context_flush(ctx);
}

/*
 *	File selector deleted callback.
 */
static void dlf_file_selector_delete_cb(
	const gchar *path, gpointer data
) 
{
	EDVContext *ctx = EDV_CONTEXT(data);
	edv_notify_queue_vfs_object_removed(ctx, path);
	edv_context_flush(ctx);
}


/*
 *	Reads the first 1024 bytes (or less) of the file specified by
 *	fp, searching for the total length header (prefixed by the
 *	string "Length:".
 *
 *	This function may return (gulong)-1, indicating that a permanent
 *	error has occured and not to call it anymore.  It normally
 *	returns the total length (or 0 if it was not found).
 */
static gulong dlf_get_total_length_from_stream(
	FILE *fp,
	gchar **line_buf_ptr
)
{
	const gint max_lines = 5;
	gint		fd,
			nlines_read;
	gulong total_length = 0l;
	const gchar *line;

	if(fp == NULL)
		return((gulong)-1);

	fd = (gint)fileno(fp);
	nlines_read = 0;

	while(nlines_read < max_lines)
	{	
		/* Read until; a newline, return, null, or EOF is read
		 * or when no more data is available
		 *
		 * Break if a newline or return was not read
		 */
		if(!edv_stream_read_strptrbrk(
			fp,
			line_buf_ptr,
			"\n\r",
			FALSE,			/* Exclude end character */
			FALSE			/* Nonblocking */
		))
			break;

		nlines_read++;

		line = *line_buf_ptr;
		while(ISBLANK(*line))
			line++;

		/* Look for the line that starts with the "Length:" */
		if(STRPFX(line, "Length:"))
		{
			const gchar *s = line;
			gchar *arg;

			/* Seek past the "Length:" parameter */
			s = edv_strarg(
				s,
				&arg,
				TRUE,		/* Parse escapes */
				TRUE		/* Parse quotes */
			);
			g_free(arg);

			/* Get the total length (size) argument */
			s = edv_strarg(
				s,
				&arg,
				TRUE,		/* Parse escapes */
				TRUE		/* Parse quotes */
			);
			if(arg != NULL)
			{
				/* Remove any ',' characters from the number string */
				gchar *num_str = edv_strsub(arg, ",", "");

				/* Get the total length */
				total_length = (gulong)ATOL(num_str);
				g_free(num_str);

				/* Number string was 0 or contained no numbers?
				 *
				 * Then set total_length to 1, indicating that although
				 * we got the total length, it was undeciperable
				 * (which can happen if the total length is not given)
				 */
				if(total_length == 0l)
					total_length = (gulong)-1;

				g_free(arg);
			}
			g_free(*line_buf_ptr);
			*line_buf_ptr = NULL;
			break;		/* Got the value we wanted */
		}

		/* Reset the line buffer for reading of the next line */
		g_free(*line_buf_ptr);
		*line_buf_ptr = NULL;
	}

	return(total_length);
}


/*
 *	Monitors the download process p.
 *
 *	Returns:
 *
 *	0	Success
 *	-2	Process exited but object not downloaded completely
 *	-3	Object does not exist or unable to download from remote
 *		server (downloaded object will not exist)
 *	-4	User aborted
 */
static gint dlf_monitor(
	EDVContext *ctx,
	const gint pid,
	const gulong check_interval_ms,
	const gulong start_time,
	const gchar *src_path,
	const gchar *tar_path,
	FILE *cstdout,
	FILE *cstderr,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry,
	const gboolean keep_dialog
)
{
	gboolean	notified_object_added = FALSE,
			user_abort = FALSE;
	const gchar *s;
	const gint check_count_max = (gint)(
		check_interval_ms * 1000l / 20000l
	);
	gint		check_count = 0,
			notify_count = 0;
	gulong		last_length = 0l,
			current_length = 0l,
			total_length = 0l,
			last_ms = edv_time_ms();
	gchar		*line_buf,
			*animations_path,
			*title,
			*msg,
			*p1, *p2,
			*tar_name;
	GList		*start_icon_paths_list,
			*icon_paths_list,
			*end_icon_paths_list;

	/* Get just the name of the target */
	s = (const gchar *)strrchr((const char *)tar_path, '/');
	if(s != NULL)
		tar_name = g_strdup(s + 1);
	else
		tar_name = g_strdup(tar_path);

	/* Set up the icon data */
	animations_path = g_strconcat(
		edv_get_s(ctx, EDV_CFG_PARM_DIR_GLOBAL),
		G_DIR_SEPARATOR_S,
		EDV_NAME_ANIMATIONS_SUBDIR,
		NULL
	);
#define APPEND_PATH(_glist_,_xpm_file_name_)	{	\
 (_glist_) = g_list_append(				\
  (_glist_),						\
  ((_xpm_file_name_) != NULL) ? g_strconcat(		\
   animations_path,					\
   G_DIR_SEPARATOR_S,					\
   (_xpm_file_name_),					\
   NULL							\
  ) : NULL						\
 );							\
}
	start_icon_paths_list = NULL;
	APPEND_PATH(start_icon_paths_list, "planet_32x32.xpm");
	APPEND_PATH(start_icon_paths_list, "planet_32x32.xpm");
	APPEND_PATH(start_icon_paths_list, "planet_32x32.xpm");
	icon_paths_list = NULL;
	APPEND_PATH(icon_paths_list, "file01_20x20.xpm");
	APPEND_PATH(icon_paths_list, "file02_20x20.xpm");
	APPEND_PATH(icon_paths_list, "file03_20x20.xpm");
	APPEND_PATH(icon_paths_list, "file04_20x20.xpm");
	APPEND_PATH(icon_paths_list, "file05_20x20.xpm");
	APPEND_PATH(icon_paths_list, "file06_20x20.xpm");
	end_icon_paths_list = NULL;
	APPEND_PATH(end_icon_paths_list, "folder_32x32.xpm");
	APPEND_PATH(end_icon_paths_list, "folder_32x32.xpm");
	APPEND_PATH(end_icon_paths_list, "folder_file_32x32.xpm");
#undef APPEND_PATH
	g_free(animations_path);

	/* Format the progress dialog title */
	title = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Descargar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Tlcharger"
#elif defined(PROG_LANGUAGE_GERMAN)
"Laden"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scaricare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Downloaden"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Download"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Downloading"
#else
"Downloading"
#endif
		" %s", tar_name
	);

	/* Format the initial progress dialog message */
	p1 = edv_path_shorten(
		src_path,
		EDV_PROGRESS_DLG_PATH_MAX_CHARS - 6
	);
	p2 = edv_path_shorten(
		tar_path,
		EDV_PROGRESS_DLG_PATH_MAX_CHARS - 16
	);
	msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Descargar:\n\
\n\
    %s (conectar...)\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Tlcharger:\n\
\n\
    %s (connecter...)\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Laden:\n\
\n\
    %s (verbinden...)\n\
\n\
Zu:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scaricare:\n\
\n\
    %s (collegare...)\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Downloaden:\n\
\n\
    %s (verbinden...)\n\
\n\
Te:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Download:\n\
\n\
    %s (ligar...)\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Downloading:\n\
\n\
    %s (forbinding...)\n\
\n\
Til:\n\
\n\
    %s\n"
#else
"Downloading:\n\
\n\
    %s (connecting...)\n\
\n\
To:\n\
\n\
    %s\n"
#endif
		,
		p1,
		p2
	);

	/* Set the progress dialog's WM icon */
	ProgressDialogSetWMIconData(
		(const guint8 **)icon_download_file_48x48_xpm
	);

	/* Set the progress dialog's geometry */
	if(geometry != NULL)
	{
		GtkWidget *w = ProgressDialogGetToplevel();
		if(w != NULL)
		{
			gtk_widget_set_uposition(
				w,
				(geometry_flags & GDK_GEOMETRY_X) ?
					geometry->x : 0,
				(geometry_flags & GDK_GEOMETRY_Y) ?
					geometry->y : 0
			);
		}
	}

	/* Map the progress dialog */
	ProgressDialogMapAnimationFile(
		title,
		msg,
#if defined(PROG_LANGUAGE_SPANISH)
"Parada"
#elif defined(PROG_LANGUAGE_FRENCH)
"Arrt"
#elif defined(PROG_LANGUAGE_GERMAN)
"Halt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Fermata"
#elif defined(PROG_LANGUAGE_DUTCH)
"Einde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Parada"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stans"
#else
"Stop"
#endif
		,
		start_icon_paths_list,
		icon_paths_list,
		end_icon_paths_list,
		EDV_PROGRESS_DLG_ANIM_INT,
		EDV_PROGRESS_DLG_ANIM_INC
	);

	g_list_foreach(start_icon_paths_list, (GFunc)g_free, NULL);
	g_list_free(start_icon_paths_list);
	g_list_foreach(icon_paths_list, (GFunc)g_free, NULL);
	g_list_free(icon_paths_list);
	g_list_foreach(end_icon_paths_list, (GFunc)g_free, NULL);
	g_list_free(end_icon_paths_list);

	g_free(title);
	title = NULL;
	g_free(msg);
	msg = NULL;

	/* While the download process is running... */
	line_buf = NULL;
	while(cstderr != NULL)
	{
		/* User aborted? */
		if(ProgressDialogStopCount() > 0)
		{
			(void)kill((int)pid, SIGINT);
			user_abort = TRUE;
			break;
		}

		/* Time to check on the download progress? */
		if(check_count >= check_count_max)
		{
			EDVVFSObject *obj;

			/* Is the download process is done */
			if(!edv_pid_exists(pid))
				break;

			/* Read from the streams */
			if(cstdout != NULL)
			{
				const gint fd = (gint)fileno(cstdout);
				while(edv_poll_read(fd))
				{
					if(fgetc(cstdout) == EOF)
						break;
				}
			}
			if(cstderr != NULL)
			{
				/* Need to check for and get the total length? */
				if(total_length == 0)
				{
					/* Check for and get the total length from
					 * standard error
					 */
					total_length = dlf_get_total_length_from_stream(
						cstderr,
						&line_buf
					);
				}
				else
				{
					/* We have obtained all the data we need from
					 * standard error, discard any subsequent data
					 */
					const gint fd = (gint)fileno(cstderr);
					while(edv_poll_read(fd))
					{
						if(fgetc(cstderr) == EOF)
							break;
					}
				}
			}

			/* Get the target object's statistics */
			obj = edv_vfs_object_stat(tar_path);
			if(obj != NULL)
			{
				gchar time_left_str[80];
				gulong	cur_ms, delta_ms,
					time_remaining_sec,
					delta_length, remaining_length;

				/* Update the current length and time lapse
				 * between the length checks
				 */
				last_length = current_length;
				current_length = obj->size;
				delta_length = (current_length > last_length) ?
					(current_length - last_length) : 0l;
				remaining_length = total_length - current_length;

				cur_ms = edv_time_ms();
				if((cur_ms > last_ms) && (last_ms > 0l))
					delta_ms = cur_ms - last_ms;
				else
					delta_ms = 0l;
				last_ms = cur_ms;

				/* Calculate the time left in seconds
				 *
				 * If the result is 0 then it means the transfer
				 * has stalled
				 */
				if((total_length > 0l) && (total_length != (gulong)-1) &&
				   (delta_length > 0l) && (remaining_length > 0l)
				)
					time_remaining_sec = (gulong)(
						(gfloat)delta_ms / (gfloat)delta_length *
						(gfloat)remaining_length / 1000.0f
					);
				else
					time_remaining_sec = 0l;

#if 0
/* The time left calculation is a bit "jumpy", this is most likely
 * due to the fact that this process is unable to determine precisely,
 * the moment, when additional bytes are downloaded to the target object
 * in order to accurately calculate the delta_ms that is
 * needed to predict how many bytes were downloaded during this
 * interval and ultimately the time remaining
 */
				/* Append this time left value to the time left
				 * history buffer and then calculate the average
				 * time left
				 */
				if(time_remaining_sec > 0l)
				{
					const gint	m = MIN(
						(time_left_sec_nitems + 1),
						time_left_sec_max_items
					),
									im = m - 1;
					gint i;

					for(i = 0; i < im; i++)
						time_left_sec_buf[i] = time_left_sec_buf[i + 1];
					time_left_sec_buf[im] = time_remaining_sec;
					time_left_sec_nitems = m;

					time_remaining_sec = dlf_get_avg_time_from_buffer(
						time_left_sec_buf,
						time_left_sec_nitems
					);
				}
#endif

				/* Format the time left string */
				if(time_remaining_sec > 3600l)
					g_snprintf(
						time_left_str, sizeof(time_left_str),
						"%ld:%.2ld:%.2ld "
#if defined(PROG_LANGUAGE_SPANISH)
"quedndose"
#elif defined(PROG_LANGUAGE_FRENCH)
"restant"
#elif defined(PROG_LANGUAGE_GERMAN)
"bleiben"
#elif defined(PROG_LANGUAGE_ITALIAN)
"rimanere"
#elif defined(PROG_LANGUAGE_DUTCH)
"blijven"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"permanecer"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"forbliing"
#else
"left"
#endif
						, (time_remaining_sec / 3600l),
						(time_remaining_sec / 60l) % 60l,
						(time_remaining_sec / 1l) % 60l
					);
				else if(time_remaining_sec > 0l)
					g_snprintf(
						time_left_str, sizeof(time_left_str),
						"%ld:%.2ld "
#if defined(PROG_LANGUAGE_SPANISH)
"quedndose"
#elif defined(PROG_LANGUAGE_FRENCH)
"restant"
#elif defined(PROG_LANGUAGE_GERMAN)
"bleiben"
#elif defined(PROG_LANGUAGE_ITALIAN)
"rimanere"
#elif defined(PROG_LANGUAGE_DUTCH)
"blijven"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"permanecer"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"forbliing"
#else
"left"
#endif
						, (time_remaining_sec / 60l),
						(time_remaining_sec / 1l) % 60l
					);
				else
					g_snprintf(
						time_left_str, sizeof(time_left_str),
#if defined(PROG_LANGUAGE_SPANISH)
"atascado"
#elif defined(PROG_LANGUAGE_FRENCH)
"ralis"
#elif defined(PROG_LANGUAGE_GERMAN)
"hingehalten"
#elif defined(PROG_LANGUAGE_ITALIAN)
"fermato"
#elif defined(PROG_LANGUAGE_DUTCH)
"geblokkeerde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"imvel"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"stoppet"
#else
"stalled"
#endif
					);

#if 0
g_print(
 "Time left %ld seconds (%ld left) %ld %ld\n",
 time_remaining_sec, remaining_length,
 current_length, total_length
);
#endif

				/* Need to notify about the object being
				 * added?
				 *
				 * Note that since we stat() it we know it exists
				 */
				if(!notified_object_added)
				{
					edv_notify_queue_vfs_object_added(ctx, tar_path);
					edv_context_flush(ctx);
					notified_object_added = TRUE;
				}

				/* Update the title string */
				g_free(title);
				if((total_length > 0l) && (total_length != (gulong)-1))
					title = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Descargar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Tlcharger"
#elif defined(PROG_LANGUAGE_GERMAN)
"Laden"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scaricare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Downloaden"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Download"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Downloading"
#else
"Downloading"
#endif
						" %s %.0f%%",
						tar_name,
						((gfloat)current_length /
							(gfloat)total_length * 100.0f)
					);
				else
					title = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Descargar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Tlcharger"
#elif defined(PROG_LANGUAGE_GERMAN)
"Laden"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scaricare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Downloaden"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Download"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Downloading"
#else
"Downloading"
#endif
						 " %s",
						 tar_name
					);

				/* Update the message string */
				g_free(msg);
				if((total_length > 0l) && (total_length != (gulong)-1))
				{
					gchar	*total_size_str = STRDUP(edv_str_size_delim(total_length)),
						*cur_size_str = STRDUP(edv_str_size_delim(current_length));

					msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Descargar:\n\
\n\
    %s (%s bytes)\n\
\n\
A:\n\
\n\
    %s (%s bytes) %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Tlcharger:\n\
\n\
    %s (%s bytes)\n\
\n\
A:\n\
\n\
    %s (%s bytes) %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Laden:\n\
\n\
    %s (%s bytes)\n\
\n\
Zu:\n\
\n\
    %s (%s bytes) %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scaricare:\n\
\n\
    %s (%s bytes)\n\
\n\
A:\n\
\n\
    %s (%s bytes) %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Downloaden:\n\
\n\
    %s (%s bytes)\n\
\n\
Te:\n\
\n\
    %s (%s bytes) %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Download:\n\
\n\
    %s (%s bytes)\n\
\n\
A:\n\
\n\
    %s (%s bytes) %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Downloading:\n\
\n\
    %s (%s bytes)\n\
\n\
Til:\n\
\n\
    %s (%s bytes) %s\n"
#else
"Downloading:\n\
\n\
    %s (%s bytes)\n\
\n\
To:\n\
\n\
    %s (%s bytes) %s\n"
#endif
						,
						p1, total_size_str,
						p2, cur_size_str, time_left_str
					);
					g_free(total_size_str);
					g_free(cur_size_str);
				}
				else
				{
					gchar *cur_size_str = STRDUP(edv_str_size_delim(current_length));
					msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Descargar:\n\
\n\
    %s (??? bytes)\n\
\n\
A:\n\
\n\
    %s (%s bytes)\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Tlcharger:\n\
\n\
    %s (??? octets)\n\
\n\
A:\n\
\n\
    %s (%s octets)\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Laden:\n\
\n\
    %s (??? bytes)\n\
\n\
Zu:\n\
\n\
    %s (%s bytes)\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scaricare:\n\
\n\
    %s (??? bytes)\n\
\n\
A:\n\
\n\
    %s (%s bytes)\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Downloaden:\n\
\n\
    %s (??? bytes)\n\
\n\
Te:\n\
\n\
    %s (%s bytes)\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Download:\n\
\n\
    %s (??? bytes)\n\
\n\
A:\n\
\n\
    %s (%s bytes)\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Downloading:\n\
\n\
    %s (??? bytes)\n\
\n\
Til:\n\
\n\
    %s (%s bytes)\n"
#else
"Downloading:\n\
\n\
    %s (??? bytes)\n\
\n\
To:\n\
\n\
    %s (%s bytes)\n"
#endif
						,
						p1,
						p2,
						cur_size_str
					);
					g_free(cur_size_str);
				}

				edv_vfs_object_delete(obj);
			}

			check_count = 0;	/* Reset timmer */
		}
		else
		{
			/* Not time to check, so just increment the check
			 * counter
			 */
			check_count++;
		}

		/* Time to notify? (every 10 seconds or longer) */
		if(notify_count >= 500l)
		{
			if(notified_object_added)
			{
				/* Send object modified notify to Endeavour */
				edv_notify_queue_vfs_object_modified(
					ctx,
					tar_path,
					NULL
				);
				edv_context_flush(ctx);
			}
			notify_count = 0;		/* Reset counter */
		}
		else
		{
			/* Not time to notify, so just increment the notify
			 * counter
			 */
			notify_count++;
		}


		/* Manage any pending GTK+ events */
		gtk_events_process();

		/* Update progress and set message (if it is not NULL).
		 * If the total length is know then we use a known progress
		 * method, otherwise we use the unknown method.
		 */
		if((total_length > 0) && (total_length != (gulong)-1))
			ProgressDialogUpdate(
				title, msg, NULL, NULL,
				(gfloat)current_length / (gfloat)total_length,
				EDV_PROGRESS_BAR_NTICKS,
				TRUE
			);
		else
			ProgressDialogUpdateUnknown(
				title, msg, NULL, NULL,
				TRUE
			);

		/* Reset the title and message strings */
		if(title != NULL)
		{
			g_free(title);
			title = NULL;
		}
		if(msg != NULL)
		{
			g_free(msg);
			msg = NULL;
		}

		edv_usleep(20000l);
	}
	g_free(line_buf);
	line_buf = NULL;

	/* Update the progress dialog message if keeping the
	 * dialog
	 */
	if(keep_dialog)
	{
		/* Get the target object's statistics */
		EDVVFSObject *obj = edv_vfs_object_stat(tar_path);
		if(obj != NULL)
		{
			gchar	*total_size_str = ((total_length > 0) && (total_length != (gulong)-1)) ?
				STRDUP(edv_str_size_delim(total_length)) : g_strdup("???"),
				*cur_size_str = STRDUP(edv_str_size_delim(current_length));
			gulong duration_sec = (start_time > 0l) ?
				(edv_time() - start_time) : 0l;
			gchar duration_str[80];

			/* Format the title and message based on if we got the
			 * entire object or not
			 */
			current_length = obj->size;
			if(current_length == total_length)
			{
				g_free(title);
				title = g_strconcat(
					"Download ",
					tar_name,
					" Complete",
					NULL
				);
			}
			else if((total_length > 0l) && (total_length != (gulong)-1))
			{
				g_free(title);
				title = g_strconcat(
					"Download ",
					tar_name,
					" Incomplete",
					NULL
				);
			}
			else
			{
				g_free(title);
				title = g_strconcat(
					"Download ",
					tar_name,
					" Done",
					NULL
				);
			}

			if(duration_sec > 3600l)
				g_snprintf(
					duration_str, sizeof(duration_str),
					"%ld:%.2ld:%.2ld",
					(duration_sec / 3600l),
					(duration_sec / 60l) % 60l,
					(duration_sec / 1l) % 60l
				);
			else
				g_snprintf(
					duration_str, sizeof(duration_str),
					"%ld:%.2ld",
					(duration_sec / 60l),
					(duration_sec / 1l) % 60l
				);

			g_free(msg);
			msg = g_strdup_printf(
"Downloaded:\n\
\n\
    %s (%s bytes)\n\
\n\
To:\n\
\n\
    %s (%s bytes) in %s\n"
				,
				p1, total_size_str,
				p2, cur_size_str,
				duration_str
			);
			g_free(cur_size_str);
			g_free(total_size_str);

			/* Set up the icon data */
			animations_path = g_strconcat(
				edv_get_s(ctx, EDV_CFG_PARM_DIR_GLOBAL),
				G_DIR_SEPARATOR_S,
				EDV_NAME_ANIMATIONS_SUBDIR,
				NULL
			);
#define APPEND_PATH(_glist_,_xpm_file_name_)	{	\
 (_glist_) = g_list_append(				\
  (_glist_),						\
  ((_xpm_file_name_) != NULL) ? g_strconcat(		\
   animations_path,					\
   G_DIR_SEPARATOR_S,					\
   (_xpm_file_name_),					\
   NULL							\
  ) : NULL						\
 );							\
}
			start_icon_paths_list = NULL;
			APPEND_PATH(start_icon_paths_list, "planet_32x32.xpm");
			APPEND_PATH(start_icon_paths_list, "planet_32x32.xpm");
			APPEND_PATH(start_icon_paths_list, "planet_32x32.xpm");
			icon_paths_list = NULL;
			/* No intermediate icons */
			end_icon_paths_list = NULL;
			APPEND_PATH(end_icon_paths_list, "folder_file_32x32.xpm");
			APPEND_PATH(end_icon_paths_list, "folder_file_32x32.xpm");
			APPEND_PATH(end_icon_paths_list, "folder_file_32x32.xpm");
#undef APPEND_PATH
			g_free(animations_path);

			ProgressDialogMapAnimationFile(
				title,
				msg,
				"Close",
				start_icon_paths_list,
				icon_paths_list,
				end_icon_paths_list,
				EDV_PROGRESS_DLG_ANIM_INT,
				EDV_PROGRESS_DLG_ANIM_INC
			);

			g_list_foreach(start_icon_paths_list, (GFunc)g_free, NULL);
			g_list_free(start_icon_paths_list);
			g_list_foreach(icon_paths_list, (GFunc)g_free, NULL);
			g_list_free(icon_paths_list);
			g_list_foreach(end_icon_paths_list, (GFunc)g_free, NULL);
			g_list_free(end_icon_paths_list);

			edv_vfs_object_delete(obj);
		}
	}

	/* Delete the progress dialog message */
	g_free(title);
	g_free(msg);
	g_free(p1);
	g_free(p2);
	g_free(tar_name);

	/* Check if the object was downloaded by checking if the
	 * target object exists and if the user did not abort
	 */
	if(!user_abort)
	{
		EDVVFSObject *obj = edv_vfs_object_stat(tar_path);
		if(obj != NULL)
		{
			/* Target object exists */
			current_length = obj->size;

			/* Report that the target object's size or being added */
			if(notified_object_added)
				edv_notify_queue_vfs_object_modified(
					ctx,
					tar_path,
					NULL
				);
			else
				edv_notify_queue_vfs_object_added(
					ctx,
					tar_path
				);
			edv_context_flush(ctx);

			/* Target object was not downloaded completely? */
			if((total_length > 0l) && (total_length != (gulong)-1) &&
			   (total_length != current_length)
			)
			{
				gchar	*cur_size_str = STRDUP(edv_str_size_delim(current_length)),
							*total_size_str = STRDUP(edv_str_size_delim(total_length));
				gchar *msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"El objeto cargado tiene un tamao de byte de %s, pero el\n\
el tamao esperado del objeto debe ser los byte de %s.\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"L'objet charg a une taille de %s octets, mais \n\
la taille prvue de l'objet tait de %s octets.\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Das geladene objekt hat eine Gre von %s bytes, aber die\b\
erwartete gre des objekts soll %s bytes sein.\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'oggetto scaricato ha una misura di byte di %s, ma la\n\
misura aspettata dell'oggetto dovrebbe essere i byte di %s.\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Het gedownload voorwerp heeft een maat van %s bytes, maar de\n\
verwachte maat van het voorwerp zou %s bytes moeten zijn.\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O objeto baixado tem um tamanho de bytes de %s, mas o tamanho\n\
esperado do objeto deve ser bytes de %s.\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Downloaded objekt har en strrelse av %s byter, men den\n\
ventede strrelsen av objektet er %s byter.\n"
#else
"The downloaded object has a size of %s bytes, but the\n\
expected size of the object should be %s bytes.\n"
#endif
					,
					cur_size_str,
					total_size_str
				);
				g_free(cur_size_str);
				g_free(total_size_str);

				CDialogSetTransientFor(NULL);
				CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"Descargado Se Opone El Tamao Inconsistancy",
					msg,
"La descarga puede haber terminado prematuramente, se recomienda\n\
que usted reasuma la descarga descargando el objeto otra vez.\n",
#elif defined(PROG_LANGUAGE_FRENCH)
"Inconsistance de la taille de l'objet tlcharg",
					msg,
"Le chargement a pu terminer prmaturment, il est recommand\n\
que vous repreniez  nouveau le tlchargement de l'objet.\n",
#elif defined(PROG_LANGUAGE_GERMAN)
"Geladene Objekt Gre Inconsistancy",
					msg,
"Das wird verfrht, es empfohlen ldt kann beendet haben, da\n\
sie das wiederaufnehmen, durch Laden des Objekts wieder zu laden.\n",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Inconsistancy Di Misura Di Oggetto Scaricato",
					msg,
"Lo scarica pu avere finito il prematurly,  raccomandato che lei\n\
riprende lo scarica da scaricare l'oggetto ancora.\n",
#elif defined(PROG_LANGUAGE_DUTCH)
"Gedownloade Voorwerp Maat Inconsistancy",
					msg,
"Het downloadt voortijdig, het door downloaden heeft misschien\n\
beindigd wordt aangeraden dat u het hervat van het voorwerp\n\
downloadt opnieuw.\n",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Inconsistancy Baixado De Tamanho De Objeto",
					msg,
"O download pode ter acabado prematuramente,  recomendado que\n\
resume o download por baixar o objeto novamente.\n",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Downloaded Objekt Strrelse Inconsistancy",
					msg,
"Download sluttet forhastet, det anbefaler at De gjenopptar den\n\
download ved downloading objektet igjen.\n",
#else
"Downloaded Object Size Inconsistancy",
					msg,
"The download may have ended prematurly, it is recommended\n\
that you resume the download by downloading the object again.",
#endif
					CDIALOG_ICON_WARNING,
					CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
					CDIALOG_BTNFLAG_OK
				);
				g_free(msg);
				CDialogSetTransientFor(NULL);

				/* Unmap the progress dialog */
				ProgressDialogBreakQuery(TRUE);

				edv_vfs_object_delete(obj);
				return(-2);			/* Target object smaller in size */
			}

			edv_vfs_object_delete(obj);
		}
		else
		{
			/* Unable to get the destination file's statistics,
			 * this suggests that the object does not exist on
			 * the remote server or that the remote server is not
			 * responding
			 */
			gchar *msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Incapaz de descargar:\n\
\n\
    %s\n\
\n\
El objeto no existe ni el camarero remoto no responde.\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Impossible de tlcharger:\n\
\n\
    %s\n\
\n\
L'objet n'existe pas ou le serveur loign ne rpond pas.\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Unfhig zu laden:\n\
\n\
    %s\n\
\n\
Das objekt existiert nicht oder der entfernte diener antwortet nicht.\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Incapace per scaricare:\n\
\n\
    %s\n\
\n\
L'oggetto non esiste o il server remoto non risponde.\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Onbekwaam te downloaden:\n\
\n\
    %s\n\
\n\
Het voorwerp bestaat niet of de vere kelner antwoordt niet.\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Incapaz de baixar:\n\
\n\
    %s\n\
\n\
O objeto no existe nem o servidor remoto no responde.\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Maktesls til download:\n\
\n\
    %s\n\
\n\
Objektet finnes ikke eller den fjerne tjeneren responderer ikke.\n"
#else
"Unable to download:\n\
\n\
    %s\n\
\n\
The object does not exist or the remote server is not responding.\n"
#endif
				,
				src_path
			);
			CDialogSetTransientFor(NULL);
			CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"Cargue Fallado",
				msg,
"El objeto no puede existir en el camarero remoto ni el camarero\n\
remoto no responde. Verifique por favor que el URL ese reffers al\n\
objeto es correcto y que el objeto existe verdaderamente en el\n\
remoto camarero. Si el camarero remoto no responde entonces usted\n\
puede querer tratar otra vez luego.\n",
#elif defined(PROG_LANGUAGE_FRENCH)
"Charger A Echou",
				msg,
"L'objet peut ne pas exister sur le serveur distant ou le serveur\n\
distant ne rpond pas. S'il vous plat vrifier que l'URL se\n\
rfrant  l'objet est exact et que l'objet existe en fait sur\n\
 le serveur distant. Si le serveur distant ne rpond pas alors\n\
vous pouvez ressayer plus tard.\n",
#elif defined(PROG_LANGUAGE_GERMAN)
"Laden Sie Versagt",
				msg,
"Das Objekt kann auf dem entfernten Diener nicht existieren\n\
oder der entfernte Diener antwortet nicht. Beglaubigen sie bitte,\n\
da die URL jener reffers zum objekt richtig ist, und, da das\n\
objekt tatschlich auf dem entfernten diener existiert. Wenn der\n\
entfernte diener dann sie nicht antwortet, wollen knnen, wieder\n\
spter zu versuchen.\n",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scaricare Fallito",
				msg,
"L'oggetto non pu esistere sul server remoto o il server remoto\n\
non risponde. Per favore di verificare che l'URL quel reffers\n\
all'oggetto  corretto e che l'oggetto esiste effettivamente sul\n\
server remoto. Se il server remoto non risponde poi lei pu volere\n\
tentare ancora dopo.\n",
#elif defined(PROG_LANGUAGE_DUTCH)
"Download Geverzuimenene",
				msg,
"Het voorwerp zou op de vere kelner niet kunnen bestaan of de vere\n\
kelner antwoordt niet. Bevestiig alstublieft dat de URL dat reffers\n\
aan het voorwerp correct is en dat het voorwerp eigenlijk op de vere\n\
kelner bestaat. Indien de vere kelner dan u niet antwoordt kunnen\n\
willen zou om opnieuw later te proberen.\n",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Baixe Fracassado",
				msg,
"O objeto no pode existir no servidor remoto nem o servidor remoto\n\
no responde. Por favor verifique-se que o URL aquele reffers ao\n\
objeto  correto e que o objeto realmente existe no servidor remoto.\n\
Se o servidor remoto no responde ento pode querer tentar novamente\n\
mais tarde.\n",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Download Sviktet",
				msg,
"Objektet finnes ikke p den fjerne tjeneren eller den fjerne\n\
tjeneren responderer ikke. Vr s snill og bekreft at URL den\n\
reffers til objektet er riktig og at objektet faktisk finnes p\n\
den fjerne tjeneren. Om den fjerne tjeneren ikke responderer da de\n\
vil ha prve igjen senere.\n",
#else
"Download Failed",
				msg,
"The object may not exist on the remote server or\n\
the remote server is not responding.  Please verify\n\
that the URL that reffers to the object is correct\n\
and that the object actually exists on the remote\n\
server.  If the remote server is not responding then\n\
you may want to try again later.",
#endif
				CDIALOG_ICON_WARNING,
				CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
				CDIALOG_BTNFLAG_OK
			);
			g_free(msg);
			CDialogSetTransientFor(NULL);

			/* Unmap the progress dialog */
			ProgressDialogBreakQuery(TRUE);

			return(-3);			/* Target object does not exist */
		}
	}

	/* Unmap the progress dialog? */
	if(!keep_dialog || user_abort)
		ProgressDialogBreakQuery(TRUE);

	/* We know the object appears to have been downloaded,
	 * but we need to report if it was successful or the user
	 * aborted
	 */
	return(user_abort ? -4 : 0);
}

/*
 *	Returns:
 *
 *	0	Success
 *	1	General error (object probably not downloaded)
 *	2	Object not downloaded (object not exist, server problem, etc)
 *	3	Download program that will be used to download the
 *		object does not exist
 *	4	User abort
 */
int main(int argc, char *argv[])
{
	gboolean	initialized_gtk = FALSE,
			need_confirmation = FALSE,
			beep_when_complete = FALSE,
			keep_dialog = FALSE,
			original_name = FALSE,
			open_object = FALSE,
			download_last_object = FALSE;
	gulong		check_interval_ms = 1000l;
	GdkGeometryFlags geometry_flags = 0;
	GdkRectangle *geometry = NULL;
	gint		i,
			status = 0;
	const gchar *arg;
	gchar		*prog = NULL,
			*source_url = NULL,
			*destination_path = NULL,
			*destination_file = NULL;
	EDVContext *ctx = NULL;

#define CLEANUP_RETURN(_v_)	{	\
 if(ctx != NULL) {			\
  edv_context_sync(ctx);			\
  edv_context_delete(ctx);		\
  ctx = NULL;				\
 }					\
					\
 g_free(prog);				\
 prog = NULL;				\
 g_free(source_url);			\
 source_url = NULL;			\
 g_free(destination_path);		\
 destination_path = NULL;		\
 g_free(destination_file);		\
 destination_file = NULL;		\
					\
 if(initialized_gtk) {			\
  /* Shutdown dialogs */		\
  PDialogShutdown();			\
  FileBrowserShutdown();		\
  ProgressDialogShutdown();		\
  CDialogShutdown();			\
  FPromptShutdown();			\
					\
  initialized_gtk = FALSE;		\
 }					\
					\
 g_free(geometry);			\
 geometry = NULL;			\
					\
 return(_v_);				\
}

	/* Set up signal callbacks */
#ifdef SIGINT
	signal(SIGINT, dlf_signal_cb);
#endif
#ifdef SIGTERM
	signal(SIGTERM, dlf_signal_cb);
#endif
#ifdef SIGKILL
	signal(SIGKILL, dlf_signal_cb);
#endif
#ifdef SIGSEGV
	signal(SIGSEGV, dlf_signal_cb);
#endif
#ifdef SIGSTOP
	signal(SIGSTOP, dlf_signal_cb);
#endif
#ifdef SIGCONT
	signal(SIGCONT, dlf_signal_cb);
#endif
#ifdef SIGPIPE
	signal(SIGPIPE, dlf_signal_cb);
#endif


	/* Parse arguments */
	for(i = 1; i < argc; i++)
	{
		arg = argv[i];
		if(arg == NULL)
			continue;

		/* Help */
		if(!g_strcasecmp(arg, "--help") ||
		   !g_strcasecmp(arg, "-help") ||
		   !strcmp((const char *)arg, "--h") ||
		   !strcmp((const char *)arg, "-h") ||
		   !strcmp((const char *)arg, "-?")
		)
		{
			g_print(
				"%s",
				PROG_HELP_MESG
			);
			status = 0;
			CLEANUP_RETURN(status);
		}
		/* Version */
		else if(!g_strcasecmp(arg, "--version") ||
				!g_strcasecmp(arg, "-version")
		)
		{
			g_print(
				"%s %s\n%s",
				PROG_NAME,
				PROG_VERSION,
				PROG_COPYRIGHT
			);
			status = 0;
			CLEANUP_RETURN(status);
		}
		/* Beep when the download has been complete? */
		else if(!g_strcasecmp(arg, "--beep") ||
			!g_strcasecmp(arg, "-beep") ||
			!strcmp((const char *)arg, "--b") ||
			!strcmp((const char *)arg, "-b")
		)
		{
			beep_when_complete = TRUE;
		}
		/* Query the user for confirmation before downloading? */
		else if(!g_strcasecmp(arg, "--confirm") ||
			!g_strcasecmp(arg, "-confirm") ||
			!strcmp((const char *)arg, "--c") ||
			!strcmp((const char *)arg, "-c")
		)
		{
			need_confirmation = TRUE;
		}
		/* Keep dialog */
		else if(!g_strcasecmp(arg, "--keep-dialog") ||
			!g_strcasecmp(arg, "-keep-dialog") ||
			!g_strcasecmp(arg, "--keep_dialog") ||
			!g_strcasecmp(arg, "-keep_dialog") ||
			!strcmp((const char *)arg, "--k") ||
			!strcmp((const char *)arg, "-k")
		)
		{
			keep_dialog = TRUE;
		}
		/* Original name */
		else if(!g_strcasecmp(arg, "--original-name") ||
			!g_strcasecmp(arg, "-original-name") ||
			!g_strcasecmp(arg, "--original_name") ||
			!g_strcasecmp(arg, "-original_name") ||
			!strcmp((const char *)arg, "--m") ||
			!strcmp((const char *)arg, "-m")
		)
		{
			original_name = TRUE;
		}
		/* Open the object afterwards (only if the download was
		 * successful and complete)?
		 */
		else if(!g_strcasecmp(arg, "--open") ||
			!g_strcasecmp(arg, "-open") ||
			!strcmp((const char *)arg, "--o") ||
			!strcmp((const char *)arg, "-o")
		)
		{
			open_object = TRUE;
		}
		/* (Re)download the last object? */
		else if(!g_strcasecmp(arg, "--last") ||
			!g_strcasecmp(arg, "-last") ||
			!strcmp((const char *)arg, "--l") ||
			!strcmp((const char *)arg, "-l")
		)
		{
			download_last_object = TRUE;
		}
		/* Check download progress and update messages interval
		 * in milliseconds (default is 1000)?
		 */
		else if(!g_strcasecmp(arg, "--interval") ||
			!g_strcasecmp(arg, "-interval") ||
			!g_strcasecmp(arg, "--int") ||
			!g_strcasecmp(arg, "-int") ||
			!g_strcasecmp(arg, "--i") ||
			!g_strcasecmp(arg, "-i")
		)
		{
			i++;
			if(i < argc)
			{
				check_interval_ms = MAX(ATOL(argv[i]), 1l);
			}
			else
			{
				g_printerr(
"%s: Requires argument.\n",
					arg
				);
				status = 2;
				CLEANUP_RETURN(status);
			}
		}
		/* Geometry */
		else if(!g_strcasecmp(arg, "--geometry") ||
			!g_strcasecmp(arg, "-geometry")
		)
		{
			i++;
			arg = (i < argc) ? argv[i] : NULL;
			if(arg != NULL)
			{
				if(geometry == NULL)
				{
					gint	x, y,
						width, height;

					geometry = (GdkRectangle *)g_malloc0(
						sizeof(GdkRectangle)
					);
					geometry_flags = GUIParseGeometry(
						arg, &x, &y, &width, &height
					);
					geometry->x = x;
					geometry->y = y;
					geometry->width = width;
					geometry->height = height;
				}
			}
			else 
			{
				g_printerr(
"%s: Requires argument\n",
					argv[i - 1]
				);
				status = 2;
				CLEANUP_RETURN(status);
			}
		}
		/* Single character argument? */
		else if((*arg == '-') ? (arg[1] != '-') : FALSE)
		{
			const gchar *v = arg + 1;
			gchar c;

			while(*v != '\0')
			{
				c = *v;
				if(c == 'b')
				{
					beep_when_complete = TRUE;
				}
				else if(c == 'c')
				{
					 need_confirmation = TRUE;
				}
				else if(c == 'o')
				{
					 open_object = TRUE;
				}
				else if(c == 'l')
				{
					 download_last_object = TRUE;
				}
				else
				{
					g_printerr(
"-%c: Unsupported argument.\n",
						c
					);
					status = 2;
					CLEANUP_RETURN(status);
				}
				v++;
			}
		}
		/* Non-option argument? */
		else if((*arg != '-') && (*arg != '+'))
		{
			/* All else assume source url or destination path */
			if(source_url == NULL)
				source_url = g_strdup(arg);
			else if(destination_path == NULL)
				destination_path = g_strdup(arg);
		}
		else
		{
			g_printerr(
"%s: Unsupported argument.\n",
				arg
			);
			status = 2;
			CLEANUP_RETURN(status);
		}
	}

	/* Initialize GTK+ as needed */
	if(!initialized_gtk)
	{
		if(!gtk_init_check(&argc, &argv))
		{
			g_printerr("Unable to initialize GTK.\n");
			status = 1;
			CLEANUP_RETURN(status);
		}
		initialized_gtk = TRUE;

		/* Initialize GDK RGB buffers system */
		gdk_rgb_init();
	}

	/* Initialize the dialogs */
	FPromptInit();
	CDialogInit();
	PDialogInit();
	FileBrowserInit();
	ProgressDialogInit();

	/* Initialize the Endeavour context */
	ctx = edv_context_new();
	edv_context_init(ctx, NULL);

	/* Set the file selector callbacks */
	FileBrowserSetObjectCreatedCB(
		dlf_file_selector_created_cb, ctx
	);
	FileBrowserSetObjectModifiedCB(
		dlf_file_selector_modified_cb, ctx
	);
	FileBrowserSetObjectDeletedCB(
		dlf_file_selector_delete_cb, ctx
	);

	/* (Re)download the last object? */
	if(download_last_object)
	{
		/* Read last download configuration file and see if there
		 * was an object recorded, make sure that we at least have
		 * the source URL
		 */
		gint download_status = 0;
		dlf_cfg_get_last(
			ctx,
			&source_url,
			&destination_path,
			&download_status
		);
		if(source_url == NULL)
		{
			CDialogSetTransientFor(NULL);
			CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"Ningn Objeto Para Descargar",
"No hay el registro de un previamente descargado se opone\n\
para descargar.",
#elif defined(PROG_LANGUAGE_FRENCH)
"Aucun Objet  Tlcharger",
"Il n'y a pas de trace d'un objet prcdemment\n\
tlcharg  tlcharger de nouveau.",
#elif defined(PROG_LANGUAGE_GERMAN)
"Kein Objekt Zu Laden",
"Es gibt keine aufzeichnung von einem vorher geladenen objekt\n\
zu laden.",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Nessuno Oggetto Di Scaricar",
"Non ci  disco di un oggetto precedentemente scaricato di\n\
scaricare.",
#elif defined(PROG_LANGUAGE_DUTCH)
"Geen Voorwerp Te Downloaden",
"Er is geen verslag van een vroeger gedownload voorwerp te\n\
downloaden.",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Nenhum Objeto Baixar",
"No h nenhum registro de um objeto previamente baixado\n\
baixar.",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Ingen Objekt Til Download",
"Der er ikke noen protokoll av et tidligere downloaded objekt\n\
til download.",
#else
"No Object To Download",
"There is no record of a previously downloaded object\n\
to (re)download.",
#endif
				NULL,
				CDIALOG_ICON_INFO,
				CDIALOG_BTNFLAG_OK,
				CDIALOG_BTNFLAG_OK
			);
			CDialogSetTransientFor(NULL);

			status = 2;
			CLEANUP_RETURN(status);
		}
	}

	/* Use current working directory as destination path if the
	 * destination path is not obtained from the command line.
	 */
	if(STRISEMPTY(destination_path))
	{
		g_free(destination_path);
		destination_path = STRDUP(g_get_current_dir());
	}


	/* No source url given? then query user for url */
	if((source_url != NULL) ? (*source_url == '\0') : TRUE)
	{
		gchar **strv;
		gint strc;

		PDialogDeleteAllPrompts();
		PDialogAddPrompt(
			NULL, "URL:", NULL
		);
		PDialogSetPromptTip(
			-1,
#if defined(PROG_LANGUAGE_SPANISH)
"Entre el URL (comenzar con http:// o el ftp://) del objeto eso\
 usted quiere cargar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Entrer l'URL (en commencant par http:// ou ftp://) de l'objet\
 que vous voulez tlcharger"
#elif defined(PROG_LANGUAGE_GERMAN)
"Tragen sie die URL (der mit http:// oder ftp:// anfngt) vom objekt,\
 das sie laden wollen ein"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Entrare l'URL (cominciando con http:// o con ftp://) dell'oggetto che\
 lei vuole scaricare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Ga de (, die met http:// of ftp:// begint) URL van het voorwerp binnen\
 dat u downloaden wil"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Entre o URL (comeando com http:// ou ftp://) do objeto que voc quer\
 baixar"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"G inn i URL (start med http:// eller ftp://) av objektet som De vil\
 ha download"
#else
"Enter the URL (starting with http:// or ftp://) of the\
 object that you want to download"
#endif
		);
		PDialogAddPromptWithBrowse(
			NULL,
#if defined(PROG_LANGUAGE_SPANISH)
"A"
#elif defined(PROG_LANGUAGE_FRENCH)
"A"
#elif defined(PROG_LANGUAGE_GERMAN)
"Zu"
#elif defined(PROG_LANGUAGE_ITALIAN)
"A"
#elif defined(PROG_LANGUAGE_DUTCH)
"Te"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"A"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Til"
#else
"To"
#endif
			":", destination_path,
			NULL, dlf_browse_target_path_cb
		);
		PDialogSetPromptCompletePath(-1);
		PDialogSetPromptTip(
			-1,
#if defined(PROG_LANGUAGE_SPANISH)
"Entre el sendero repleto de la gua que usted quiere para\
 descargar el objeto a o le sale blanco para descargar a la\
 gua actual"
#elif defined(PROG_LANGUAGE_FRENCH)
"Entrer le nom complet du rpertoire o vous voulez\
 tlcharger l'objet ou ne rien indiquer pour tlcharger dans\
 le rpertoire actuel"
#elif defined(PROG_LANGUAGE_GERMAN)
"Tragen sie den vollen pfad des verzeichnisses, das sie das\
 objekt zu oder laden wollen, verlassen es leerstelle ein, zum\
 jetzigen verzeichnis zu laden"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Entrare il sentiero pieno dell'elenco che lei vuole scaricare\
 l'oggetto a o lasciare lo spazio vuoto di esso per scaricare\
 all'elenco attuale"
#elif defined(PROG_LANGUAGE_DUTCH)
"Ga het vol pad van de gids dat u het voorwerp te downloaden wil\
 binnen of verlaat het leegte om te de huidig gids te downloaden"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Entre o pleno caminho do guia que voc quer baixar o objeto a\
 ou sai de ele lacuna baixar ao guia atual"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"G inn i den fulle stien av katalogen som De vil ha download\
 objektet til eller forlate det tomrom til download til den\
 nvrendee katalogen"
#else
"Enter the full path of the directory that you want to download the\
 object to or leave it blank to download to the current directory"
#endif
		);
		if(geometry != NULL)
		{
			GtkWidget *w = PDialogGetToplevel();
			if(w != NULL)
			{
				gtk_widget_set_uposition(
					w,
					(geometry_flags & GDK_GEOMETRY_X) ?
						geometry->x : 0,
					(geometry_flags & GDK_GEOMETRY_Y) ?
						geometry->y : 0
				);
			}
		}
		PDialogSetSize(450, -1);
		strv = PDialogGetResponseIconData(
			PROG_NAME,
#if defined(PROG_LANGUAGE_SPANISH)
"Entre el URL (empezando con http:// o ftp://) del objeto que usted\n\
quiere descargar. Si usted sale el blanco de campo de A: entonces\n\
el objeto ser descargado a la gua actual."
#elif defined(PROG_LANGUAGE_FRENCH)
"Entrer l'URL (en commenant par http:// ou ftp://) de l'objet que\n\
vous voulez tlcharger. Si vous laissez le champ du destinataire vide A: alors\n\
l'objet sera tlcharg dans le rpertoire actuel."
#elif defined(PROG_LANGUAGE_GERMAN)
"Tragen sie die URL (der mit http:// oder ftp:// anfngt) vom objekt,\n\
das sie laden wollen ein. Wenn sie die Zu: feld leerstelle dann das\n\
objekt verlassen, zum jetzigen verzeichnis wird geladen werden."
#elif defined(PROG_LANGUAGE_ITALIAN)
"Entrare l'URL (cominciando con http:// o con ftp://) dell'oggetto\n\
che lei vuole scaricare. Se lei lascia lo spazio vuoto di campo di\n\
A: di il poi l'oggetto sar scaricato all'elenco attuale."
#elif defined(PROG_LANGUAGE_DUTCH)
"Ga de (, die met http:// of ftp:// begint) URL van het voorwerp\n\
binnen dat u downloaden wil. Indien u de Te: veld leegte dan het\n\
voorwerp verlaat te de huidig gids zal gedownload worden."
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Entre o URL (comeando com http:// ou ftp://) do objeto que voc\n\
quer baixar. Se sai da lacuna de campo de A: ento o objeto ser\n\
baixado ao guia atual."
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"G inn i URL (start med http:// eller ftp://) av objektet som\n\
de vil ha download. Om De forlater Til: felttomrom da objektet\n\
er downloaded til den nvrendee katalogen."
#else
"Enter the URL (starting with http:// or ftp://) of the object that\n\
you want to download. If you leave the To: field blank then the\n\
object will be downloaded to the current directory."
#endif
			, NULL,
			(guint8 **)icon_download_file_32x32_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Descarga",
"Cancele",
#elif defined(PROG_LANGUAGE_FRENCH)
"Chargement",
"Annuler",
#elif defined(PROG_LANGUAGE_GERMAN)
"Laden",
"Heben",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scaricare",
"Annullare",
#elif defined(PROG_LANGUAGE_DUTCH)
"Download",
"Annuleer",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Download",
"Cancelamento",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Download",
"Kanseller",
#else
"Download",
"Cancel",
#endif
			PDIALOG_BTNFLAG_SUBMIT | PDIALOG_BTNFLAG_CANCEL,
			PDIALOG_BTNFLAG_SUBMIT,
			&strc
		);
		if((strv != NULL) && (strc > 0))
		{
			/* Get source URL from first string */
			const gchar *s = strv[0];
			if(!STRISEMPTY(s))
			{
				g_free(source_url);
				source_url = STRDUP(s);
			}

			/* Get destination path from second string (if any) */
			if(strc > 1)
			{
				s = strv[1];
				if(!STRISEMPTY(s))
				{
					g_free(destination_path);
					destination_path = g_strdup(s);
				}
			}
		}

		/* If no source url was obtained than that implies the user
		 * has canceled
		 */
		if(STRISEMPTY(source_url))
		{
			status = 4;
			CLEANUP_RETURN(status);
		}
	}


	/* Use the current working directory as the destination path
	 * if the destination path is not given
	 *
	 * Note that we have to do this again since the user may have
	 * been queried above and no destination path was given
	 */
	if(STRISEMPTY(destination_path))
	{
		g_free(destination_path);
		destination_path = STRDUP(g_get_current_dir());
	}


	/* Change the working directory to the destination path only
	 * if the destination path was specified, otherwise use the
	 * current directory
	 */
	if(edv_path_is_directory(destination_path))
	{
		/* Get the destination file */
		const gchar *s = (const gchar *)strrchr(
			(const char *)source_url,
			'/'
		);
		if(s != NULL)
			destination_file = g_strconcat(
				destination_path,
				G_DIR_SEPARATOR_S,
				s + 1,
				NULL
			);
		else
			destination_file = STRDUP(destination_path);
		(void)edv_setcwd(destination_path);
	}
	else if(destination_path != NULL)
	{
		/* Destination path is given, but it does not appear to be
		 * a directory, so instead get its parent path and change
		 * to its directory
		 */
		gchar *parent_path = g_dirname(destination_path);
		if(parent_path != NULL)
		{
			(void)edv_setcwd(parent_path);
			g_free(parent_path);
		}
		destination_file = g_strdup(destination_path);
	}
	else
	{
		g_free(destination_path);
		destination_path = STRDUP(g_get_current_dir());
	}

	/* Clean up the destination file name */
	if((destination_file != NULL) && !original_name)
	{
		gchar	*s,
			*name = (gchar *)strrchr(
			(char *)destination_file,
			'/'
		);
		name = (name != NULL) ? (name + 1) : destination_file;

		/* Replace all occurances of %HH with their intended
		 * character values
		 */
		name = edv_strsubh(name);

		/* Terminate the string at any URL script argument
		 * deliminators and other characters that are not
		 * acceptable for file names
		 */
		s = (gchar *)strpbrk((char *)name, "?=*&,;><|~!@#$^");
		if(s != NULL)
			*s = '\0';
	}

	/* Need to make initial confirmation? */
	if(need_confirmation)
	{
		gint response;
		gchar *msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Descarga:\n\
\n\
    %s\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Chargement:\n\
\n\
    %s\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Laden:\n\
\n\
    %s\n\
Zu:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scaricare:\n\
\n\
    %s\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Download:\n\
\n\
    %s\n\
Te:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Download:\n\
\n\
    %s\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Download:\n\
\n\
    %s\n\
Til:\n\
\n\
    %s\n"
#else
"Download:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n"
#endif
			,
			source_url,
			destination_file
		);
		CDialogSetTransientFor(NULL);
		response = CDialogGetResponseIconData(
#if defined(PROG_LANGUAGE_SPANISH)
"Confirme La Descarga"
#elif defined(PROG_LANGUAGE_FRENCH)
"Confirme le chargement"
#elif defined(PROG_LANGUAGE_GERMAN)
"Besttigen Sie Ldt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Confermare Scaricare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bevestiig Downloadt"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Confirme Download"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Bekreft Download"
#else
"Confirm Download"
#endif
			,
			msg,
			NULL,
			(guint8 **)icon_download_file_32x32_xpm,
			CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
			CDIALOG_BTNFLAG_YES
		);
		g_free(msg);
		CDialogSetTransientFor(NULL);
		switch(response)
		{
		  case CDIALOG_RESPONSE_NO:
		  case CDIALOG_RESPONSE_NOT_AVAILABLE:
			status = 4;
			CLEANUP_RETURN(status);
			break;
		}
	}

	/* Destination file already exists? */
	if(edv_path_exists(destination_file))
	{
		gint response;
		gchar *msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Reasuma cargue de existente se opone:\n\
\n\
    %s\n\
\n\
El chasquido en el \"Si\" de para reasumir carga.\n\
El chasquido en el \"No\" de para escribir para reemplazar objeto\n\
local con objeto remoto.\n\
El chasquido en el \"Cancela\" de para cancelar el carga.\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Reprendre le chargement d'objet existant:\n\
\n\
    %s\n\
\n\
Clicquer sur \"Oui\" reprendre le chargement.\n\
Clicquer sur \"Non\" remplace l'objet local par l'objet distant.\n\
Clicquer sur \"Annule\" pour annuler le chargement.\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Lebenslauf ldt von existieren objekt:\n\
\n\
    %s\n\
\n\
Klicken auf \"Ja\" wiederaufzunehmen zu laden.\n\
Klicken auf \"Nein\", rtliches objekt mit entferntem objekt zu\n\
berschreiben.\n\
Klicken auf "Hebt", das aufzuheben, ldt.\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Riprendere scaricare di oggetto esistente:\n\
\n\
    %s\n\
\n\
Scattare su \"S\" di riprendere scaricare.\n\
Scattare su \"No\" di sovrascrivere l'oggetto locale con\n\
l'oggetto remoto.\n\
Lo scatto su \"Annulla\" per annullare lo scarica.\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Cv download van bestaan voorwerp:\n\
\n\
    %s\n\
\n\
Klik op \"Ja\" te hervatten downloadt.\n\
Klik op \"Geen\" plaatselijk voorwerp met ver voorwerp te\n\
beschrijven.\n\
Klik \"Annuleert\" op om het te annuleren downloadt.\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Resuma download de objeto existente:\n\
\n\
    %s\n\
\n\
Clique em \"Sim\" resumir download.\n\
Clique em \"No\" a overwrite objeto local com objeto remoto.\n\
Clique em \"Cancelamento\" cancelar o download.\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Gjenoppta download av  finnes objekt:\n\
\n\
    %s\n\
\n\
Klikk p \"Ja\" gjenoppta download.\n\
Klikk p \"Ingen\" overskrive lokalt objekt med fjernt objekt.\n\
Klikk p \"Kansellerer\" kansellere download.\n"
#else
"Resume download of existing object:\n\
\n\
    %s\n\
\n\
Click on \"Yes\" to resume download.\n\
Click on \"No\" to overwrite local object with remote object.\n\
Click on \"Cancel\" to cancel the download.\n"
#endif
			, destination_file
		);
		CDialogSetTransientFor(NULL);
		response = CDialogGetResponseIconData(
#if defined(PROG_LANGUAGE_SPANISH)
"Confirme Reasuma",
			msg,
"El objeto que usted tratan de cargar aparece a\n\
existe ya localmente. Si el objeto local existente\n\
era slo parcialmente cargado (e.g. debido a un interrumpido\n\
la transferencia) usted debe chasquear en \"S\" reasumir\n\
cargarlo. Si usted desea escribir para  reemplazar el\n\
existente objeto local con el objeto que usted cargan\n\
entonces el chasquido en \"No\". Si usted est no seguro lo\n\
que hacer ni usted quiere no cargar el objeto entonces\n\
chasquido en \"Cancela\".\n",
#elif defined(PROG_LANGUAGE_FRENCH)
"Confirmer Reprendre",
			msg,
"L'objet que vous essayez de charger existe dj localement.\n\
 Si l'objet local existe seulement en raison d'un tlchargement prcdent\n\
(e.g. grce  un transfert interrompu) vous devez cliquer\n\
sur \"Oui\" pour reprendre le chargement. Si vous souhaitez\n\
superposer l'objet local existant avec l'objet que vous\n\
tlchargez alors clicquer sur \"Non\". Si vous n'tes pas sr de\n\
 ce que vous devez faire, alors clicquer sur \"Annule\".\n",

#elif defined(PROG_LANGUAGE_GERMAN)
"Besttigen Sie Lebenslauf",
			msg,
"Das objekt, das sie versuchen, zu laden, erscheint schon,\n\
rtlich zu existieren. Wenn das existierende rtliche objekt nur\n\
teilweise (e.g. aufgrund einer unterbrochenen bertragung) sie\n\
geladen wurde, auf sollen \"Ja\" klicken, wieder anzufangen, es zu\n\
laden. Wenn sie wnschen, das existierende rtliche objekt mit dem\n\
objekt sie zu berschreiben, dann laden klicken auf \"Nein\".\n\
Wenn sie nicht sicher sind was, zu machen oder sie \"Aufhebt\" das\n\
objekt dann klicken auf nicht wollen laden.\n",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Confermare Riprendere",
			msg,
"L'oggetto che lei tentano scaricare appare gi esistere localmente.\n\
Se l'esistere l'oggetto locale parzialmente  stato soltanto\n\
scaricato (e.g. dovuto a un trasferimento interrotto) lei dovrebbe\n\
scattare su \"S\" di riprendere per scaricare esso. Se lei desidera\n\
sovrascrivere l'esistere l'oggetto locale con l'oggetto che lei\n\
scaricano poi lo scatto su \"No\". Se lei non sono sicuro ci che di\n\
fare o lei non vuole scaricare l'oggetto il poi scatto su \"Annulla\".\n",
#elif defined(PROG_LANGUAGE_DUTCH)
"Bevestiig Cv",
			msg,
"Het voorwerp dat u probeert te downloaden verschijnt reeds om\n\
plaatselijk te bestaan. Indien het bestaand plaatselijk voorwerp\n\
enige gedeeltelijk werd gedownload (e.g. tengevolge van een\n\
onderbrekenene overdracht) u op \"Ja\" zou moeten klikken downloaden\n\
het te hervatten. Indien u wenst het bestaand plaatselijk voorwerp\n\
met het voorwerp u te beschrijven dan downloadt klik op \"Geen\".\n\
Indien u niet zeker bent wat te doen of u \"Annuleert\" het voorwerp\n\
dan klik op niet wil downloaden.\n",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Confirme Resume",
			msg,
"O objeto que voc tentam baixar aparece j existir localmente.\n\
Se o existir objeto local s parcialmente foi baixado (e.g. devido\n\
a uma transferncia interrompida) deve clicar em \"Sim\" resumir\n\
download ele. Se deseja a overwrite o existir objeto local com o\n\
objeto que voc baixam ento estalido em \"No\". Se voc no esto\n\
seguro o que fazer nem voc no quer baixar o objeto ento estalido\n\
em \"Cancelamento\".\n",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Bekreft Fortsetter",
			msg,
"Objektet som De prver til download kommer fram allerede finnes\n\
lokalt. Om finnesingen av lokalt objekt var bare delvis downloaded\n\
(e.g. p grunn av en avbrytet overfring) De klikker p \"Ja\"\n\
gjenoppta downloading det. Om De nsker overskrive finnesingen av\n\
lokalt objekt med objektet De er downloading da klikk p \"Ingen\".\n\
Om De er ikke sikker hva gjre eller De gjr ikke vil ha download\n\
objektet da klikk p \"Kansellerer\".\n",
#else
"Confirm Resume",
			msg,
"The object that you are trying to download appears to\n\
already exist locally.  If the existing local object\n\
was only partially downloaded (e.g. due to an interrupted\n\
transfer) you should click on \"Yes\" to resume\n\
downloading it.  If you wish to overwrite the existing\n\
local object with the object you are downloading then\n\
click on \"No\".  If you are not sure what to do or you\n\
do not want to download the object then click on \"Cancel\".",
#endif
			(guint8 **)icon_download_file_32x32_xpm,
			CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO |
			CDIALOG_BTNFLAG_CANCEL | CDIALOG_BTNFLAG_HELP,
			CDIALOG_BTNFLAG_YES
		);
		g_free(msg);
		CDialogSetTransientFor(NULL);
		switch(response)
		{
		  case CDIALOG_RESPONSE_CANCEL:
		  case CDIALOG_RESPONSE_NOT_AVAILABLE:
			status = 4;
			CLEANUP_RETURN(status);
			break;

		  case CDIALOG_RESPONSE_NO:
			/* Overwrite, so remove the destination file */
			if(edv_unlink(destination_file))
			{
				const gint error_code = (gint)errno;
				gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %s",
					g_strerror(error_code),
					destination_file
				);
				CDialogSetTransientFor(NULL);
				CDialogGetResponse(
					"Remove File Failed",
					msg,
					NULL,
					CDIALOG_ICON_WARNING,
					CDIALOG_BTNFLAG_OK,
					CDIALOG_BTNFLAG_OK
				);
				g_free(msg);
				CDialogSetTransientFor(NULL);
			}
			/* Do not notify Endeavour about the removed file */
			break;
		}
	}

	/* Get the GNU WGet program */
	prog = edv_which(WGET_PROG);
	if(prog == NULL)
	{
		gchar *msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Incapaz de encontrar el programa de la descarga:\n\
\n\
    %s\n\
\n\
Verifique por favor que el programa es instalado en una\n\
ubicacion especificada por el ambiente de %s y que\n\
es ejecutable por este programa.",
#elif defined(PROG_LANGUAGE_FRENCH)
"Impossible de trouver le programme de tlchargement:\n\
\n\
    %s\n\
\n\
S'il vous plait verifier que le programme soit installe\n\
a l'emplacement spcifi par l'environnement de\n\
%s et que c'est xcutable par ce programme.",
#elif defined(PROG_LANGUAGE_GERMAN)
"Unfahig, das downloadprogramm zu finden:\n\
\n\
    %s\n\
\n\
Bitte beglaubigt, dass das programm von der %s umwelt\n\
an einem ort installiert wird, der und angegeben wird,\n\
dass es ausfuhrbar durch dieses programm ist.",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Incapace per trovare lo scarica il programma:\n\
\n\
    %s\n\
\n\
Per favore di verificare che il programma e installato a\n\
una posizione specificata dall'ambiente di %s e che\n\
e eseguibile da questo programma.",
#elif defined(PROG_LANGUAGE_DUTCH)
"Onbekwaam om het te vinden downloadt programma:\n\
\n\
    %s\n\
\n\
Bevestigt u alstublieft dat het programma door de %s\n\
omgeving aan een locatie wordt geinstalleerd die door dit\n\
programma gespecificeerd wordt en dat het uitvoerbaar is.",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Incapaz de achar o programa de download:\n\
\n\
    %s\n\
\n\
Por favor verifique-se que o programa e instalado numa\n\
situacao especificado pelo ambiente de %s e que e\n\
executable por este programa.",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Maktesl?s finne download program:\n\
\n\
    %s\n\
\n\
Ver sa snill og bekreft at programet installerer pa en\n\
plassering spesifisert ved %s miljo og at det er\n\
gjennomforbar ved dette programet.",
#else
"Unable to find the download program:\n\
\n\
    %s\n\
\n\
Please verify that the program is installed at a location\n\
specified by the %s environment and that it is\n\
executable by this program.",
#endif
			WGET_PROG, ENV_VAR_NAME_PATH
		);
		CDialogSetTransientFor(NULL);
		CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"El Programa De La Descarga No Encontr",
			msg,
"El programa de la descarga se utiliza para realizar la descarga\n\
verdadera, se debe instalar en su sistema en la ubicacin\n\
especificada encima de para permite este programa es capaz de\n\
descargar objetos.\n",
#elif defined(PROG_LANGUAGE_FRENCH)
"Le Programme de chargement n'a pas trouv",
			msg,
"Le programme de chargement est utilis pour excuter le\n\
chargement vritable, il doit tre install sur votre systme\n\
 l'emplacement spcifi afin de rendre capable au-dessus ce\n\
programme peut tlcharger des objets.\n",
#elif defined(PROG_LANGUAGE_GERMAN)
"Laden Sie Programm, Das Gefunden Wird Nicht",
			msg,
"Das ist programm benutzt ldt, das eigentliche durchzufhren,\n\
es auf ihrem system am ort ldt oben mu installiert werden hat\n\
angegeben, um zu ermglichen, dieses programm kann objekte\n\
laden.\n",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Non scaricare il Programma Trovato",
			msg,
"Lo scarica il programma  usato per eseguire il reale scarica,\n\
deve essere installato sul suo sistema alla posizione\n\
specificata sopra per permettere questo programma  in grado di\n\
scaricare gli oggetti.\n",
#elif defined(PROG_LANGUAGE_DUTCH)
"Download Niet Programma Vond",
			msg,
"Het is programma gebruikt downloadt om het eigenlijk te\n\
verrichten, downloadt het op je systeem aan de plaats, die\n\
boven om gespecificeerd is om aan de gelegenheid deze programma\n\
te geven voorwerpen moet genstalleerd worden kan downloaden.\n",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Programa De Download No Achou",
			msg,
"O programa de download  usado para executar o download real,\n\
deve ser instalado em seu sistema na localidade especificado\n\
acima para capacitar este programa  capaz de baixar objetos.\n",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Download Program Funnet Ikke",
			msg,
"Download program bruker gjennomfre den aktuelle download,\n\
installert det p Deres system p plasseringen som spesifisert\n\
over muliggjre dette programet er kyndig til download objekt.\n",
#else
"Download Program Not Found",
			msg,
"The download program is used to perform the actual download,\n\
it must be installed on your system at the location specified\n\
above in order to enable this program is able to download\n\
objects.\n",
#endif
			CDIALOG_ICON_ERROR,
			CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
			CDIALOG_BTNFLAG_OK
		);
		g_free(msg);
		CDialogSetTransientFor(NULL);

		status = 3;
		CLEANUP_RETURN(status);
	}

	/* Execute the download program and monitor the download */
	if(destination_file != NULL)
	{
		FILE	*cstdout,
					*cstderr;
		const gulong start_time = edv_time();
		gchar	*shell_prog,
					*cmd = g_strdup_printf(
"\"%s\" \"%s\"\
 --verbose --tries=1 --progress=dot\
 -c\
 --timestamping\
 --restrict-file-names=nocontrol\
 -O \"%s\"",
			prog,
			source_url,
			destination_file
		);
		const gchar	*shell_cmd = edv_get_s(ctx, EDV_CFG_PARM_PROG_SHELL),
					*shell_args = edv_strarg(
			shell_cmd,
			&shell_prog,
			TRUE,			/* Parse escapes */
			TRUE			/* Parse quotes */
		);

		/* Execute the download command
		 *
		 * Note that wget writes stdout output to stderr
		 */
		const gint pid = edv_system_shell_streams(
			cmd,
			shell_prog,
			shell_args,
			NULL,
			&cstdout,
			&cstderr		/* wget outputs to stderr and
						 * not to stdout */
		);
		if(pid < 0)
		{
			/* An error occured while executing the download command */
			const gint error_code = (gint)errno;
			gchar *msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Incapaz de ejecutar la orden de la descarga:\n\
\n\
    %s\n\
\n\
%s."
#elif defined(PROG_LANGUAGE_FRENCH)
"Impossible d'excuter l'ordre de chargement:\n\
\n\
    %s\n\
\n\
%s."
#elif defined(PROG_LANGUAGE_GERMAN)
"Unfhig, durchzufhren, befehl ldt:\n\
\n\
    %s\n\
\n\
%s."
#elif defined(PROG_LANGUAGE_ITALIAN)
"Incapace per eseguire scarica il comando:\n\
\n\
    %s\n\
\n\
%s."
#elif defined(PROG_LANGUAGE_DUTCH)
"Onbekwaam uit te voeren bevel downloadt:\n\
\n\
    %s\n\
\n\
%s."
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Incapaz de executar comando de download:\n\
\n\
    %s\n\
\n\
%s."
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Maktesls utfre download kommando:\n\
\n\
    %s\n\
\n\
%s."
#else
"An error occured while executing the download command:\n\
\n\
    %s\n\
\n\
%s."
#endif
				,
				cmd,
				g_strerror(error_code)
			);
			CDialogSetTransientFor(NULL);
			CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"Descargue Fallado",
				msg,
"La orden utiliz para descargar el objeto no se podra\n\
ejecutar, verifica por favor que el programa de la descarga\n\
existe en la ubicacin especificada encima de y es ejecutable.\n",
#elif defined(PROG_LANGUAGE_FRENCH)
"ECHEC de tlchargement",
				msg,
"La commande utilis pour tlcharger l'objet ne pourrait pas\n\
tre excut, s'il vous plat vrifier que le programme de\n\
chargement existe  l'emplacement spcifi au-dessus  et est\n\
xcutable.\n",
#elif defined(PROG_LANGUAGE_GERMAN)
"Laden Sie Versagt",
				msg,
"Der befehl knnte das objekt nicht durchgefhrt werden hatte\n\
geladen, beglaubigt bitte, da das programm ldt, am Ort ober\n\
existiert hat angegeben und ist ausfhrbar.\n",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scaricare Fallito",
				msg,
"Il comando ha usato per scaricare l'oggetto non potrebbe\n\
essere eseguito, verifica per favore che lo scarica il programma\n\
esiste alla posizione specificata e  sopra eseguibile.\n",
#elif defined(PROG_LANGUAGE_DUTCH)
"Download Geverzuimenene",
				msg,
"Het bevel zou het voorwerp niet kunnen uitgevoerd worden heeft\n\
gedownload, alstublieft bevestigt dat het programma downloadt\n\
aan de plaats bovenstaande bestaat specificeerde en uitvoerbaar\n\
is.\n",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Baixe Fracassado",
				msg,
"O comando baixava o objeto no podia ser executado, por favor\n\
verifica-se que o programa de download existe na localidade\n\
especificado acima e est executable.\n",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Download Sviktet",
				msg,
"Kommandoen som brukt til download objektet ikke utfrer, vr\n\
s snill og bekreft at download program finnes p plasseringen\n\
spesifisert ovenfor og er gjennomfrbar.\n",
#else
"Download Failed",
				msg,
"The command used to download the object could not be executed,\n\
please verify that the download program exists at the location\n\
specified above and is executable.\n",
#endif
				CDIALOG_ICON_ERROR,
				CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
				CDIALOG_BTNFLAG_OK
			);
			CDialogSetTransientFor(NULL);
			g_free(msg);
			status = 1;
		}
		else
		{
			gint edv_history_status = 0;
			gulong end_time;

			/* Download command executed, now monitor the
			 * download and handle the result
			 */
			const gint download_status = dlf_monitor(
				ctx,
				pid,
				check_interval_ms,
				start_time,
				source_url,
				destination_file,
				cstdout,
				cstderr,
				geometry_flags,
				geometry,
				keep_dialog
			);

			/* Get the ending time of the download */
			end_time = edv_time();

			/* Check the result of the download */
			switch(download_status)
			{
			  case 0:	/* Success */
				if(beep_when_complete)
					edv_play_sound_completed(ctx);
				if(open_object)
				{
					GList *paths_list = NULL;
					paths_list = g_list_append(
						paths_list,
						g_strdup(destination_file)
					);
					(void)edv_open(
						ctx,
						paths_list,
						NULL
					);
					g_list_foreach(paths_list, (GFunc)g_free, NULL);
					g_list_free(paths_list);
				}
				edv_history_status = 0;
				break;

			  case -2:	/* Process exited but object not completely downloaded */
				if(beep_when_complete)
					edv_play_sound_completed(ctx);
				status = 2;
				edv_history_status = -1;
				break;

			  case -3:	/* Object not downloaded (does not exist) */
				status = 2;
				edv_history_status = -2;
				break;

			  case -4:	/* User abort */
				status = 4;
				edv_history_status = -4;
				break;
			}

			/* Record the last downloaded URL regardless of the
			 * above result
			 */
			dlf_cfg_set_last(
				ctx,
				source_url,
				destination_path,
				download_status
			);

			/* Append history to Endeavour */
			edv_history_append(
				ctx,
				EDV_HISTORY_DISK_OBJECT_DOWNLOAD,
				start_time,
				end_time,
				edv_history_status,
				source_url,
				destination_path,
				NULL		/* No comments */
			);

			/* Wait for the user to close the progress dialog? */
			while(ProgressDialogIsQuery())
			{
				ProgressDialogUpdate(
					NULL, NULL, NULL, NULL,
					1.0f,
					EDV_PROGRESS_BAR_NTICKS,
					TRUE
				);
				if(ProgressDialogStopCount() > 0)
					break;

				gtk_events_process();
				edv_usleep(8000l);
			}
		}

		(void)FCLOSE(cstdout);
		(void)FCLOSE(cstderr);
		g_free(shell_prog);
		g_free(cmd);
	}	/* Execute the download program and monitor the download */


	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}
