#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <glib.h>

#include "../../include/fio.h"
#include "../libendeavour2-base/endeavour2.h"

#include "dlf_cfg_fio.h"
#include "config.h"

#define DOWNLOAD_LAST_FILE	"download_last.ini"


static gchar *dlf_cfg_get_path(EDVContext *ctx);

gint dlf_cfg_get_last(
	EDVContext *ctx,
	gchar **source_url_rtn,
	gchar **destination_path_rtn,
	gint *status_rtn
);
void dlf_cfg_set_last(
	EDVContext *ctx,
	const gchar *source_url,
	const gchar *destination_path,
	const gint status
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Returns a dynamically allocated string specifying the path to
 *	the DOWNLOAD_LAST_FILE configuration file.
 */
static gchar *dlf_cfg_get_path(EDVContext *ctx)
{
	const gchar *data_path;

	if(ctx == NULL)
		return(NULL);

	data_path = edv_get_s(ctx, EDV_CFG_PARM_DIR_LOCAL);
	if(data_path == NULL)
		return(NULL);

	return(edv_paths_join(
		data_path,
		DOWNLOAD_LAST_FILE
	));
}


/*
 *	Opens the last download values from the last download
 *	configuration file.
 */
gint dlf_cfg_get_last(
	EDVContext *ctx,
	gchar **source_url_rtn,
	gchar **destination_path_rtn,
	gint *status_rtn				/* Return value of DownloadMonitor() */
)
{
	FILE *fp;
	gchar		*parm,
			*cfg_file;

	if(source_url_rtn != NULL)
	{
		g_free(*source_url_rtn);
		*source_url_rtn = NULL;
	}
	if(destination_path_rtn != NULL)
	{
		g_free(*destination_path_rtn);
		*destination_path_rtn = NULL;
	}
	if(status_rtn != NULL)
		*status_rtn = -1;

	if(ctx == NULL)
		return(-1);

	/* Get download last configuration file path */
	cfg_file = dlf_cfg_get_path(ctx);
	if(cfg_file == NULL)
		return(-1);

	/* Open download last configuration file for reading */
	fp = fopen((const char *)cfg_file, "rb");
	if(fp == NULL)
	{
		g_free(cfg_file);
		return(-1);
	}

	/* Begin reading last configuration file */
	parm = NULL;
	while(1)
	{
		/* Get parameter of the next non-comment line */
		parm = (gchar *)FSeekNextParm(
			fp,
			(char *)parm,
			EDV_CFG_COMMENT_CHAR,
			EDV_CFG_DELIMINATOR_CHAR
		);
		if(parm == NULL)
			break;

		/* Check which parameter this line specifies */

		if(!strcasecmp(parm, "LastSourceURL") &&
		   (source_url_rtn != NULL)
		)
		{
			g_free(*source_url_rtn);
			*source_url_rtn = FGetString(fp);
		}
		else if(!strcasecmp(parm, "LastDestinationPath") &&
				(destination_path_rtn != NULL)
		)
		{
			g_free(*destination_path_rtn);
			*destination_path_rtn = FGetString(fp);
		}
		else if(!strcasecmp(parm, "LastStatus") &&
				(status_rtn != NULL)
		)
		{
			gint vi[1];
			FGetValuesI(fp, vi, 1);
			*status_rtn = vi[0];
		}
		else
		{
			FSeekNextLine(fp);
		}
	}

	/* Close download last configuration file */
	(void)fclose(fp);
	g_free(cfg_file);

	return(
		((*source_url_rtn != NULL) &&
		 (*destination_path_rtn != NULL)
		) ? 0 : -1
	);
}

/*
 *	Saves the last download values to the last download
 *	configuration file.
 */
void dlf_cfg_set_last(
	EDVContext *ctx,
	const gchar *source_url,
	const gchar *destination_path,
	const gint status
)
{
	FILE *fp;
	gboolean file_existed;
	gchar *cfg_file;

	if((ctx == NULL) || (source_url == NULL) || (destination_path == NULL))
		return;

	/* Get download last configuration file path */
	cfg_file = dlf_cfg_get_path(ctx);
	if(cfg_file == NULL)
		return;

	file_existed = edv_path_lexists(cfg_file);

	/* Open download last configuration file for writing */
	fp = fopen((const char *)cfg_file, "wb");
	if(fp == NULL)
	{
		g_free(cfg_file);
		return;
	}

	/* Begin writing download last configuration file */
	(void)fprintf(
		fp,
"# Download Front End - Last Downloaded Object\n\
#\n\
# This file records the last downloaded object, it is used for\n\
# storing information in order to (re)download the last object.\n\
#\n"
	);
	if(source_url != NULL)
		(void)fprintf(
			fp,
			"LastSourceURL = %s\n",
			source_url
		);
	if(destination_path != NULL)
		(void)fprintf(
			fp,
			"LastDestinationPath = %s\n",
			destination_path
		);
	(void)fprintf(
		fp,
		"LastStatus = %i\n",
		status
	);

	/* Close download last configuration file */
	(void)fclose(fp);
	if(file_existed)
		edv_notify_queue_vfs_object_modified(
			ctx,
			cfg_file,
			NULL
		);
	else
		edv_notify_queue_vfs_object_added(
			ctx,
			cfg_file
		);
	edv_context_flush(ctx);
	g_free(cfg_file);
}
