#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../include/string.h"

#include "guiutils.h"
#include "pie_chart.h"
#include "cdialog.h"
#include "progressdialog.h"

#include "cfg.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_fs_type.h"
#include "libendeavour2-base/edv_property.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "edv_pixmap.h"
#include "edv_device.h"
#include "edv_devices_list.h"
#include "edv_utils_gtk.h"
#include "edv_list_cb.h"
#include "device_edit_dlg.h"
#include "devices_list_win.h"
#include "edv_device_mount.h"
#include "edv_emit.h"
#include "edv_op.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_search_20x20.xpm"
#include "images/icon_add_20x20.xpm"
#include "images/icon_edit_20x20.xpm"
#include "images/icon_remove_20x20.xpm"
#include "images/icon_close_20x20.xpm"
#include "images/icon_mount_20x20.xpm"
#include "images/icon_unmount_20x20.xpm"
#include "images/icon_eject_20x20.xpm"
#include "images/icon_properties_20x20.xpm"
#include "images/icon_fsck_20x20.xpm"
#include "images/icon_tools_20x20.xpm"
#include "images/icon_floppy_20x20.xpm"

#include "images/icon_devices_list_32x32.xpm"
#include "images/icon_devices_list_48x48.xpm"


/* Callbacks */
static gint EDVDevicesListProgressCB(
	gpointer data, const gulong i, const gulong m
);

static gint EDVDevicesListWinDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static gint EDVDevicesListWinKeyEventCB(
	GtkWidget *widget, GdkEventKey *key, gpointer data
);
static gint EDVDevicesListWinButtonPressEventCB(
	GtkWidget *widget, GdkEventButton *button, gpointer data
);
static void EDVDevicesListWinSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
static void EDVDevicesListWinUnselectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
static void EDVDevicesListWinMountCB(GtkWidget *widget, gpointer data);
static void EDVDevicesListWinEjectCB(GtkWidget *widget, gpointer data);
static void EDVDevicesListWinPropertiesCB(GtkWidget *widget, gpointer data);
static void EDVDevicesListWinFSCKCB(GtkWidget *widget, gpointer data);
static void EDVDevicesListWinToolsCB(GtkWidget *widget, gpointer data);
static void EDVDevicesListWinFormatCB(GtkWidget *widget, gpointer data);

static void EDVDevicesListWinFindCB(GtkWidget *widget, gpointer data);

static void EDVDevicesListWinAddCB(GtkWidget *widget, gpointer data);
static void EDVDevicesListWinEditCB(GtkWidget *widget, gpointer data);
static void EDVDevicesListWinRemoveCB(GtkWidget *widget, gpointer data);
static void EDVDevicesListWinUpCB(GtkWidget *widget, gpointer data);
static void EDVDevicesListWinDownCB(GtkWidget *widget, gpointer data);
static void EDVDevicesListWinCloseCB(GtkWidget *widget, gpointer data);

/* Update Display */
static void EDVDevicesListWinUpdateDisplay(
	edv_devices_list_win_struct *lw,
	EDVDevice *d
);

/* Set Row */
void EDVDevicesListWinSetRow(
	edv_devices_list_win_struct *lw,
	const gint row,
	EDVDevice *d
);

/* Select */
void EDVDevicesListWinSelect(
	edv_devices_list_win_struct *lw,
	const gint i
);
gint EDVDevicesListSelectByPath(
	edv_devices_list_win_struct *lw,
	const gchar *path
);

/* Get Devices List */
void EDVDevicesListWinGetList(
	edv_devices_list_win_struct *lw,
	const gboolean verbose
);

void EDVDevicesListWinUsageChangedCB(
	edv_devices_list_win_struct *lw
);
void EDVDevicesListWinReconfiguredNotifyCB(
	edv_devices_list_win_struct *lw
);
void EDVDevicesListWinMountNotifyCB(
	edv_devices_list_win_struct *lw,
	const gint dev_num, EDVDevice *d,
	const gboolean is_mounted
);
void EDVDevicesListWinDeviceAddedCB(
	edv_devices_list_win_struct *lw,
	const gint dev_num, EDVDevice *d
);
void EDVDevicesListWinDeviceModifiedCB(
	edv_devices_list_win_struct *lw,
	const gint dev_num, EDVDevice *d
);
void EDVDevicesListWinDeviceRemovedCB(
	edv_devices_list_win_struct *lw,
	const gint dev_num
);

edv_devices_list_win_struct *EDVDevicesListWinNew(
	EDVCore *core
);
void EDVDevicesListWinUpdateMenus(
	edv_devices_list_win_struct *lw
);
void EDVDevicesListSetBusy(
	edv_devices_list_win_struct *lw, const gboolean busy
);
gboolean EDVDevicesListWinIsMapped(
	edv_devices_list_win_struct *lw
);
void EDVDevicesListWinMap(
	edv_devices_list_win_struct *lw
);
void EDVDevicesListWinUnmap(
	edv_devices_list_win_struct *lw
);
void EDVDevicesListWinDelete(
	edv_devices_list_win_struct *lw
);


#if defined(PROG_LANGUAGE_SPANISH)
# define EDV_DEVICES_LIST_WIN_TITLE	"Artefactos"
#elif defined(PROG_LANGUAGE_FRENCH)
# define EDV_DEVICES_LIST_WIN_TITLE	"D'appareils"
#elif defined(PROG_LANGUAGE_GERMAN)
# define EDV_DEVICES_LIST_WIN_TITLE	"Vorrichtungen"
#elif defined(PROG_LANGUAGE_ITALIAN)
# define EDV_DEVICES_LIST_WIN_TITLE	"I Congegni"
#elif defined(PROG_LANGUAGE_DUTCH)
# define EDV_DEVICES_LIST_WIN_TITLE	"Apparaten"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
# define EDV_DEVICES_LIST_WIN_TITLE	"Os Artifcios"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
# define EDV_DEVICES_LIST_WIN_TITLE	"Innretninger"
#else
# define EDV_DEVICES_LIST_WIN_TITLE	"Devices"
#endif

#define EDV_DEVICES_LIST_WIN_WIDTH	550
#define EDV_DEVICES_LIST_WIN_HEIGHT	460


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Progress callback.
 */
static gint EDVDevicesListProgressCB(
	gpointer data, const gulong i, const gulong m
)
{
	edv_devices_list_win_struct *lw = EDV_DEVICES_LIST_WIN(data);
	if(lw == NULL)
		return(1);

	if(ProgressDialogIsQuery())
	{
		if(ProgressDialogStopCount() > 0)
			return(1);

		if(m > 0l)
			ProgressDialogUpdate(
				NULL, NULL, NULL, NULL,
				(gfloat)i / (gfloat)m,
				EDV_PROGRESS_BAR_NTICKS, TRUE
			);
		else
			ProgressDialogUpdateUnknown(
				NULL, NULL, NULL, NULL,
				TRUE
			);
	}

	return(0);
}


/*
 *	Toplevel GtkWindow "delete_event" signal callback.
 */
static gint EDVDevicesListWinDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	edv_devices_list_win_struct *lw = EDV_DEVICES_LIST_WIN(data);
	if(lw == NULL)
		return(TRUE);

	if(lw->freeze_count > 0)
		return(TRUE);

	EDVDevicesListWinCloseCB(widget, lw);

	return(TRUE);
}

/*
 *	Any GtkWidget "key_press_event" or "key_release_event" signal
 *	callback.
 */
static gint EDVDevicesListWinKeyEventCB(
	GtkWidget *widget, GdkEventKey *key, gpointer data
)
{
	gboolean press;
	gint status = FALSE, etype;
	guint keyval, state;
	GtkCList *clist;
	EDVCore *core;
	edv_devices_list_win_struct *lw = EDV_DEVICES_LIST_WIN(data);
	if((widget == NULL) || (key == NULL) || (lw == NULL))
		return(status);

	if(lw->freeze_count > 0)
		return(status);

	clist = GTK_CLIST(lw->devices_clist);
	core = lw->core;

	etype = key->type;
	press = (etype == GDK_KEY_PRESS) ? TRUE : FALSE;
	keyval = key->keyval;
	state = key->state;

/* Stop emit of signal */
#define STOP_SIGNAL_EMIT(_w_)	{		\
 gtk_signal_emit_stop_by_name(			\
  GTK_OBJECT(_w_),				\
  press ?					\
   "key_press_event" : "key_release_event"	\
 );						\
}

	/* Devices GtkCList */
	if(widget == lw->devices_clist)
	{
		gint row;
		GList *glist;
		GtkCList *clist = GTK_CLIST(widget);

		/* Get last selected row */
		row = edv_clist_get_selected_last(clist, NULL);

		/* Handle by key value */
		switch(keyval)
		{
		  case GDK_Return:
		  case GDK_KP_Enter:
		  case GDK_ISO_Enter:
		  case GDK_3270_Enter:
			if(press)
			{
				EDVDevicesListWinEditCB(NULL, lw);
			}
			STOP_SIGNAL_EMIT(widget);
			status = TRUE;
			break;

		  case GDK_Delete:
			if(press)
			{
				EDVDevicesListWinRemoveCB(NULL, lw);
			}
			STOP_SIGNAL_EMIT(widget);
			status = TRUE;
			break;

		  case GDK_space:
		  case GDK_KP_Space:
			row = clist->focus_row;
			if((row >= 0) && (row < clist->rows) && press)
			{
				gboolean already_selected = FALSE;

				/* Check if this row is already selected */
				glist = clist->selection;
				while(glist != NULL)
				{
					if(row == (gint)glist->data)
					{
						already_selected = TRUE;
						break;
					}
					glist = g_list_next(glist);
				}

				gtk_clist_freeze(clist);
				if(already_selected)
					gtk_clist_unselect_row(clist, row, 0);
				else
					gtk_clist_select_row(clist, row, 0);
				gtk_clist_thaw(clist);
			}
			STOP_SIGNAL_EMIT(widget);
			status = TRUE;
			break;
		}
	}

	return(status);
#undef STOP_SIGNAL_EMIT
}

/*
 *	Any GtkWidget "button_press_event" or "button_release_event"
 *	signal callback.
 */
static gint EDVDevicesListWinButtonPressEventCB(
	GtkWidget *widget, GdkEventButton *button, gpointer data
)
{
	gint status = FALSE, etype;
	GtkCList *clist;
	const CfgItem *cfg_list;
	EDVCore *core;
	edv_devices_list_win_struct *lw = EDV_DEVICES_LIST_WIN(data);
	if((widget == NULL) || (button == NULL) || (lw == NULL))
		return(status);

	if(lw->freeze_count > 0)
		return(status);

	clist = GTK_CLIST(lw->devices_clist);
	core = lw->core;
	cfg_list = core->cfg_list;
	etype = (gint)button->type;

	/* Begin checking which widget this event is for
	 *
	 * Devices GtkCList
	 */
	if(widget == lw->devices_clist)
	{
		gint row, column;

		/* Handle by the event type */
		switch(etype)
		{
		  case GDK_BUTTON_PRESS:
			switch(button->button)
			{
			  case GDK_BUTTON3:
				/* Find the row and column that this button press
				 * occured over
				 */
				if(!gtk_clist_get_selection_info(
					clist,
					button->x, button->y,
					&row, &column
				))
				{
					row = -1;
					column = 0;
				}

				/* Select before mapping the menu? */
				if(EDV_GET_B(EDV_CFG_PARM_RIGHT_CLICK_MENU_SELECTS) &&
				   (row >= 0) && (row < clist->rows)
				)
				{
					gtk_clist_freeze(clist);
					gtk_clist_select_row(clist, row, column);
					gtk_clist_thaw(clist);
				}

				/* Update the menus and then map the right-click
				 * menu
				 */
				EDVDevicesListWinUpdateMenus(lw);
				gtk_menu_popup(
					GTK_MENU(lw->menu),
					NULL, NULL,
					NULL, NULL,
					button->button, button->time
				);
				status = TRUE;
				break;
			}
			break;
		}
	}

	return(status);
}

/*
 *	Devices GtkCList "select_row" signal callback.
 */
static void EDVDevicesListWinSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	EDVCore *core;
	edv_devices_list_win_struct *lw = EDV_DEVICES_LIST_WIN(data);
	if((clist == NULL) || (lw == NULL))
		return;

	if(lw->freeze_count > 0)
		return;

	core = lw->core;

	/* If the row is not fully visible then scroll to it */
	if(gtk_clist_row_is_visible(clist, row) !=
		GTK_VISIBILITY_FULL
	)
		gtk_clist_moveto(
			clist,
			row, -1,		/* Row, column */
			0.5f, 0.0f		/* Row, column */
		);

	/* Update the displayed Device */
	if(row > -1)
		EDVDevicesListWinUpdateDisplay(
			lw,
			EDV_DEVICE(g_list_nth_data(
				core->devices_list,
				(guint)row
			))
		);

	EDVDevicesListWinUpdateMenus(lw);

	/* Double click? */
	if(event != NULL)
	{
		if(event->type == GDK_2BUTTON_PRESS)
		{
			/* Edit */
			EDVDevicesListWinEditCB(
				GTK_WIDGET(clist),
				lw
			);
		}
	}
}

/*
 *	Devices GtkCList "unselect_row" signal callback.
 */
static void EDVDevicesListWinUnselectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	EDVCore *core;
	edv_devices_list_win_struct *lw = EDV_DEVICES_LIST_WIN(data);
	if((clist == NULL) || (lw == NULL))
		return;

	if(lw->freeze_count > 0)
		return;

	core = lw->core;

	/* Clear the displayed Device */
	EDVDevicesListWinUpdateDisplay(lw, NULL);

	EDVDevicesListWinUpdateMenus(lw);
}

/*
 *	Mount/unmunt callback.
 */
static void EDVDevicesListWinMountCB(GtkWidget *widget, gpointer data)
{
	gboolean original_mount_state;
	gint status, row, dev_num;
	GtkWidget *toplevel;
	GtkCList *clist;
	EDVDevice *dev;
	EDVCore *core;
	edv_devices_list_win_struct *lw = EDV_DEVICES_LIST_WIN(data);
	if(lw == NULL)
		return;

	if(lw->freeze_count > 0)
		return;

	toplevel = lw->toplevel;
	clist = GTK_CLIST(lw->devices_clist);
	core = lw->core;

	/* Get the selected Device */
	dev_num = row = edv_clist_get_selected_last(clist, NULL);
	dev = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		dev_num
	));
	if(dev == NULL)
		return;

	original_mount_state = EDV_DEVICE_IS_MOUNTED(dev);

	EDVDevicesListSetBusy(lw, TRUE);

	/* Unmount or mount? */
	if(EDV_DEVICE_IS_MOUNTED(dev))
		status = edv_device_unmount(
			core,
			dev,
			TRUE,				/* Show progress */
			TRUE,				/* Verbose */
			toplevel
		);
	else
		status = edv_device_mount(
			core,
			dev,
			TRUE,				/* Show progresss */
			TRUE,				/* Verbose */
			toplevel
		);

	/* Update all the devices' statisticss */
	edv_devices_list_update_statistics(core->devices_list);

	EDVDevicesListSetBusy(lw, FALSE);

	/* Mount error? */
	if(status != 0)
	{
		const gchar *last_error = edv_device_mount_get_error(core);
		if(!STRISEMPTY(last_error))
		{
			edv_play_sound_error(core);
			edv_message_error(
				original_mount_state ?
					"Unmount Failed" : "Mount Failed",
				last_error,
				NULL,
				toplevel
			);
		}
	}
	else
	{
		/* Notify about this device being mounted */
		edv_emit_device_mount(
			core,
			dev_num, dev,
			EDV_DEVICE_IS_MOUNTED(dev)
		);
	}
}

/*
 *	Eject callback.
 */
static void EDVDevicesListWinEjectCB(GtkWidget *widget, gpointer data)
{
	gboolean original_mount_state;
	gint status, row, dev_num;
	GtkWidget *toplevel;
	GtkCList *clist;
	EDVDevice *dev;
	EDVCore *core;
	edv_devices_list_win_struct *lw = EDV_DEVICES_LIST_WIN(data);
	if(lw == NULL)
		return;

	if(lw->freeze_count > 0)
		return;

	toplevel = lw->toplevel;
	clist = GTK_CLIST(lw->devices_clist);
	core = lw->core;

	/* Get the selected Device */
	dev_num = row = edv_clist_get_selected_last(clist, NULL);
	dev = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		(guint)dev_num
	));
	if(dev == NULL)
		return;

	original_mount_state = EDV_DEVICE_IS_MOUNTED(dev);

	EDVDevicesListSetBusy(lw, TRUE);

	/* Need to unmount the Device first? */
	if(EDV_DEVICE_IS_MOUNTED(dev))
	{
		/* Unmount */
		gint status = edv_device_unmount(
			core,
			dev,
			TRUE,				/* Show progress */
			TRUE,				/* Verbose */
			toplevel
		);
		if(status != 0)
		{
			/* Unmount failed */
			gint response;
			const gchar *last_error = edv_device_mount_get_error(core);
			gchar *msg;

			if(!STRISEMPTY(last_error))
				msg = g_strdup_printf(
"An error occured while unmounting the device\n\
\"%s\" before ejecting.\n\
\n\
%s\n\
\n\
Continue ejecting?",
					STRISEMPTY(dev->name) ?
						dev->device_path : dev->name,
					last_error
				);
			else
				msg = g_strdup_printf(
"An unknown error occured while unmounting the device\n\
\"%s\" before ejecting.\n\
\n\
Continue ejecting?",
					STRISEMPTY(dev->name) ?
						dev->device_path : dev->name
				);
			edv_play_sound_error(core);
			CDialogSetTransientFor(toplevel);
			response = CDialogGetResponse(
				"Unmount Failed",
				msg,
				NULL,
				CDIALOG_ICON_ERROR,
				CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
				CDIALOG_BTNFLAG_NO
			);
			g_free(msg);
			CDialogSetTransientFor(NULL);
			switch(response)
			{
			  case CDIALOG_RESPONSE_YES_TO_ALL:
			  case CDIALOG_RESPONSE_YES:
			  case CDIALOG_RESPONSE_OK:
				break;

			  default:
				EDVDevicesListSetBusy(lw, FALSE);
				return;
				break;
			}
		}
	}

	/* Eject */
	status = edv_device_eject(
		core,
		dev,
		TRUE,				/* Show progress */
		TRUE,				/* Verbose */
		toplevel
	);

	/* Update all device mount states and stats */
	edv_devices_list_update_mount_states(core->devices_list);
	edv_devices_list_update_statistics(core->devices_list);

	EDVDevicesListSetBusy(lw, FALSE);

	/* Eject error? */
	if(status != 0)
	{
		const gchar *last_error = edv_device_mount_get_error(core);
		if(!STRISEMPTY(last_error))
		{
			edv_play_sound_error(core);
			edv_message_error(
				"Eject Failed",
				last_error,
				NULL,
				toplevel
			);
		}
	}
	else
	{
		/* Notify about this Device's media being ejected */
		edv_emit_device_mount(
			core,
			dev_num, dev,
			EDV_DEVICE_IS_MOUNTED(dev)
		);
	}
}

/*
 *	Properties callback.
 */
static void EDVDevicesListWinPropertiesCB(GtkWidget *widget, gpointer data)
{
	gint row, dev_num;
	const gchar *mount_path;
	GtkWidget *toplevel;
	GtkCList *clist;
	EDVDevice *dev;
	EDVCore *core;
	edv_devices_list_win_struct *lw = EDV_DEVICES_LIST_WIN(data);
	if(lw == NULL)
		return;

	if(lw->freeze_count > 0)
		return;

	toplevel = lw->toplevel;
	clist = GTK_CLIST(lw->devices_clist);
	core = lw->core;

	/* Get the selected Device */
	dev_num = row = edv_clist_get_selected_last(clist, NULL);
	dev = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		(guint)dev_num
	));
	if(dev == NULL)
		return;

	EDVDevicesListSetBusy(lw, TRUE);

	/* Get the mount path */
	mount_path = dev->mount_path;
	if(!STRISEMPTY(mount_path))
	{
		/* Create a new Properties Dialog to display the object */
		edv_new_properties_dialog_vfs(
			core,
			mount_path,
			"Device",			/* Page Name */
			NULL,				/* No extended properties list */
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL,
			toplevel
		);
	}

	EDVDevicesListSetBusy(lw, FALSE);
}

/*
 *	FSCK callback.
 */
static void EDVDevicesListWinFSCKCB(GtkWidget *widget, gpointer data)
{
	gint row, dev_num;
	GtkWidget *toplevel;
	GtkCList *clist;
	EDVDevice *dev;
	EDVCore *core;
	edv_devices_list_win_struct *lw = EDV_DEVICES_LIST_WIN(data);
	if(lw == NULL)
		return;

	if(lw->freeze_count > 0)
		return;

	toplevel = lw->toplevel;
	clist = GTK_CLIST(lw->devices_clist);
	core = lw->core;

	/* Get the selected Device */
	dev_num = row = edv_clist_get_selected_last(clist, NULL);
	dev = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		(guint)dev_num
	));
	if(dev == NULL)
		return;

	EDVDevicesListSetBusy(lw, TRUE);

	/* Run the device check */
	edv_run_device_check(core, dev, toplevel);

	EDVDevicesListSetBusy(lw, FALSE);
}

/*
 *	Device Tools callback.
 */
static void EDVDevicesListWinToolsCB(GtkWidget *widget, gpointer data)
{
	gint row, dev_num;
	GtkWidget *toplevel;
	GtkCList *clist;
	EDVDevice *dev;
	EDVCore *core;
	edv_devices_list_win_struct *lw = EDV_DEVICES_LIST_WIN(data);
	if(lw == NULL)
		return;

	if(lw->freeze_count > 0)
		return;

	toplevel = lw->toplevel;
	clist = GTK_CLIST(lw->devices_clist);
	core = lw->core;

	/* Get the selected Device */
	dev_num = row = edv_clist_get_selected_last(clist, NULL);
	dev = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		(guint)dev_num
	));
	if(dev == NULL)
		return;

	EDVDevicesListSetBusy(lw, TRUE);

	/* Run the device tools */
	edv_run_device_tools(core, dev, toplevel);

	EDVDevicesListSetBusy(lw, FALSE);
}

/*
 *	Format Media callback.
 */
static void EDVDevicesListWinFormatCB(GtkWidget *widget, gpointer data)
{
	gint row, dev_num;
	GtkWidget *toplevel;
	GtkCList *clist;
	EDVDevice *dev;
	EDVCore *core;
	edv_devices_list_win_struct *lw = EDV_DEVICES_LIST_WIN(data);
	if(lw == NULL)
		return;

	if(lw->freeze_count > 0)
		return;

	toplevel = lw->toplevel;
	clist = GTK_CLIST(lw->devices_clist);
	core = lw->core;

	/* Get the selected Device */
	dev_num = row = edv_clist_get_selected_last(clist, NULL);
	dev = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		(guint)dev_num
	));
	if(dev == NULL)
		return;

	EDVDevicesListSetBusy(lw, TRUE);

	/* Run the device format */
	edv_run_device_format(core, dev, toplevel);

	EDVDevicesListSetBusy(lw, FALSE);
}

/*
 *	Find callback.
 */
static void EDVDevicesListWinFindCB(GtkWidget *widget, gpointer data)
{
	guint8 spacing;
	gint		row, column,
					sel_row,
					nfind_iterations = 0;
	const gchar *find_str;
	gchar *cell_text;
	GdkBitmap *mask;
	GdkPixmap *pixmap;
	GtkEntry *entry;
	GtkCList *clist;
	edv_devices_list_win_struct *lw = EDV_DEVICES_LIST_WIN(data);
	if(lw == NULL)
		return;

	if(lw->freeze_count > 0)
		return;

	entry = GTK_ENTRY(lw->find_entry);
	clist = GTK_CLIST(lw->devices_clist);

	/* Get the search string */
	find_str = gtk_entry_get_text(entry);
	if(STRISEMPTY(find_str))
		return;

	/* Get the selected row + 1 or 0 if none */
	sel_row = edv_clist_get_selected_last(clist, NULL);
	if(sel_row < 0)
		sel_row = 0;
	else
		sel_row++;

	/* Iterate through rows, checking each Type column to see if
	 * there are any partial patches
	 */
	do
	{
		/* Iterate from selected row to end */
		for(row = sel_row; row < clist->rows; row++)
		{
			/* Iterate through all cells on this row */
			for(column = 0; column < clist->columns; column++)
			{
				cell_text = NULL;
				switch(gtk_clist_get_cell_type(clist, row, column))
				{
				  case GTK_CELL_TEXT:
					gtk_clist_get_text(clist, row, column, &cell_text);
					break;
				  case GTK_CELL_PIXTEXT:
					gtk_clist_get_pixtext(
						clist, row, column, &cell_text,
						&spacing, &pixmap, &mask
					);
					break;
				  case GTK_CELL_PIXMAP:
				  case GTK_CELL_WIDGET:
				  case GTK_CELL_EMPTY:
					break;
				}
				/* Got cell text? */
				if(!STRISEMPTY(cell_text))
				{
					/* Find string found inside cell text string? */
					if(strcasestr(cell_text, find_str))
						break;
				}
			}
			/* If column itteration broke before all columns were
			 * iterated through then that implies a match was made.
			 */
			if(column < clist->columns)
				break;
		}
		/* Got match? */
		if(row < clist->rows)
			break;
		else
			nfind_iterations++;

		/* Reset sel_row to 0 so that find starts at beginning */
		sel_row = 0;

	} while(nfind_iterations < 2);

	/* Got match? */
	if(row < clist->rows)
	{
		/* Select new matched row */
		gtk_clist_select_row(clist, row, 0);
	}
}

/*
 *	Add callback.
 */
static void EDVDevicesListWinAddCB(GtkWidget *widget, gpointer data)
{
	gint row, dev_num;
	GtkWidget *toplevel;
	GtkCList *clist;
	EDVDevice *dev;
	EDVCore *core;
	edv_device_edit_dlg_struct *ed;
	edv_devices_list_win_struct *lw = EDV_DEVICES_LIST_WIN(data);
	if(lw == NULL)
		return;

	if(lw->freeze_count > 0)
		return;

	toplevel = lw->toplevel;
	clist = GTK_CLIST(lw->devices_clist);
	core = lw->core;

	EDVDevicesListSetBusy(lw, TRUE);

	/* Get the last selected row */
	row = edv_clist_get_selected_last(clist, NULL);

	dev_num = row;

	/* Create a new Device */
	dev = edv_device_new();
	if(dev == NULL)
	{
		EDVDevicesListSetBusy(lw, FALSE);
		return;
	}
	if(dev_num < 0)
	{
		/* Append */
		core->devices_list = g_list_append(
			core->devices_list,
			dev
		);
		dev_num = g_list_length(core->devices_list) - 1;
	}
	else
	{
		/* Insert */
		core->devices_list = g_list_insert(
			core->devices_list,
			dev,
			dev_num
		);
	}

	/* Set the new Device's default values */
	dev->name = g_strdup("New Device");
#ifdef PATH_DEV_NULL
	dev->device_path = g_strdup(PATH_DEV_NULL);
#endif
#ifdef PATH_MNT
	dev->mount_path = g_strdup(PATH_MNT);
#endif
	dev->fs_type_name = NULL;
	dev->fs_type_code = EDV_FS_TYPE_EMPTY;

	/* Update all device mount states and stats */
	edv_devices_list_update_mount_states(core->devices_list);
	edv_devices_list_update_statistics(core->devices_list);

	/* Notify about this Device being added */
	edv_emit_device_added(core, dev_num, dev);

	/* Select this Device */
	gtk_clist_select_row(clist, dev_num, 0);

	/* Create/map the Device Edit Dialog */
	ed = lw->device_edit_dlg;
	if(ed == NULL)
		lw->device_edit_dlg = ed = EDVDeviceEditDlgNew(core);
	if(ed != NULL)
	{
		edv_center_window_to_window(toplevel, ed->toplevel);
		EDVDeviceEditDlgMap(ed);
		EDVDeviceEditDlgGetValues(ed, dev_num);
		EDVDeviceEditDlgResetHasChanges(ed, FALSE);
	}

	EDVDevicesListSetBusy(lw, FALSE);
}

/*
 *	Edit callback.
 */
static void EDVDevicesListWinEditCB(GtkWidget *widget, gpointer data)
{
	gint row, dev_num;
	GtkWidget *toplevel;
	GtkCList *clist;
	EDVDevice *dev;
	edv_device_edit_dlg_struct *ed;
	EDVCore *core;
	edv_devices_list_win_struct *lw = EDV_DEVICES_LIST_WIN(data);
	if(lw == NULL)
		return;

	if(lw->freeze_count > 0)
		return;

	toplevel = lw->toplevel;
	clist = GTK_CLIST(lw->devices_clist);
	core = lw->core;

	/* Get the last selected row */
	row = edv_clist_get_selected_last(clist, NULL);

	/* Get the Device */
	dev_num = row;
	dev = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		(guint)dev_num
	));
	if(dev == NULL)
		return;

	/* Check if this device is internally created by this program
	 * which means it cannot be edited
	 */
	if(EDV_DEVICE_IS_INTERNAL(dev))
	{
		edv_play_sound_warning(core);
		edv_message_warning(
"Edit Failed",
"This device is created and managed internally by this\n\
program and may not be edited.",
			NULL,
			toplevel
		);
		return;
	}


	EDVDevicesListSetBusy(lw, TRUE);

	/* Create/map the Device Edit Dialog */
	ed = lw->device_edit_dlg;
	if(ed == NULL)
		lw->device_edit_dlg = ed = EDVDeviceEditDlgNew(core);
	if(ed != NULL)
	{
		edv_center_window_to_window(toplevel, ed->toplevel);
		EDVDeviceEditDlgMap(ed);
		EDVDeviceEditDlgGetValues(ed, dev_num);
		EDVDeviceEditDlgResetHasChanges(ed, FALSE);
	}

	EDVDevicesListSetBusy(lw, FALSE);
}

/*
 *	Remove callback.
 */
static void EDVDevicesListWinRemoveCB(GtkWidget *widget, gpointer data)
{
	gint row, dev_num;
	GtkWidget *toplevel;
	GtkCList *clist;
	EDVDevice *dev;
	EDVCore *core;
	edv_devices_list_win_struct *lw = EDV_DEVICES_LIST_WIN(data);
	if(lw == NULL)
		return;

	if(lw->freeze_count > 0)
		return;

	toplevel = lw->toplevel;
	clist = GTK_CLIST(lw->devices_clist);
	core = lw->core;

	/* Get the last selected row */
	row = edv_clist_get_selected_last(clist, NULL);

	/* Get the Device */
	dev_num = row;
	dev = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		(guint)dev_num
	));
	if(dev == NULL)
		return;

	EDVDevicesListSetBusy(lw, TRUE);

	/* Check if this device is internally allocated, in which case
	 * it cannot be removed
	 */
	if(EDV_DEVICE_IS_INTERNAL(dev))
	{
		gchar *msg = g_strdup_printf(
"Device \"%s\"\n\
may not be removed, because it was either created\n\
internally by %s or opened from a global\n\
configuration.",
			STRISEMPTY(dev->name) ?
				dev->device_path : dev->name,
			PROG_NAME
		);
		edv_play_sound_warning(core);
		edv_message_warning(
"Remove Failed",
			msg,
			NULL,
			toplevel
		);
		g_free(msg);
		EDVDevicesListSetBusy(lw, FALSE);
		return;
	}

	/* Confirm remove? */
	if(TRUE)
	{
		gint response;
		gchar *msg = g_strdup_printf(
"Remove device \"%s\"?\n\
\n\
Removing this device will only remove its reference from\n\
%s's configuration and not from the system's\n\
configuration.",
			STRISEMPTY(dev->name) ?
				dev->device_path : dev->name,
			PROG_NAME
		);
		edv_play_sound_question(core);
		CDialogSetTransientFor(toplevel);
		response = CDialogGetResponse(
			"Confirm Remove",
			msg,
			NULL,
			CDIALOG_ICON_QUESTION,
			CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
			CDIALOG_BTNFLAG_NO
		);
		g_free(msg);
		CDialogSetTransientFor(NULL);
		switch(response)
		{
		  case CDIALOG_RESPONSE_YES_TO_ALL:
		  case CDIALOG_RESPONSE_YES:
		  case CDIALOG_RESPONSE_OK:
			break;

		  default:
			EDVDevicesListSetBusy(lw, FALSE);
			return;
			break;
		}
	}

	/* Delete this Device */
	edv_device_delete(dev);

	/* Remove this Device's pointer from the Devices List */
	core->devices_list = g_list_remove(
		core->devices_list,
		dev
	);

	/* Notify about this Device being removed */
	edv_emit_device_removed(core, dev_num);

	EDVDevicesListSetBusy(lw, FALSE);
	edv_play_sound_completed(core);
}

/*
 *	Shift Up callback.
 */
static void EDVDevicesListWinUpCB(GtkWidget *widget, gpointer data)
{
	gint row;
	GtkCList *clist;
	EDVCore *core;
	edv_devices_list_win_struct *lw = EDV_DEVICES_LIST_WIN(data);
	if(lw == NULL)
		return;

	if(lw->freeze_count > 0)
		return;

	clist = GTK_CLIST(lw->devices_clist);
	core = lw->core;

	EDVDevicesListSetBusy(lw, TRUE);

	/* Get the last selected row */
	row = edv_clist_get_selected_last(clist, NULL);

	/* Can shift the selected row up? */
	if(row > 0)
	{
		/* Get the GList pointers to the Devices being shifted */
		GList	*glist1 = g_list_nth(
			core->devices_list,
			(guint)(row - 1)
		),
					*glist2 = g_list_nth(
			core->devices_list,
			(guint)row
		);
		if((glist1 != NULL) && (glist2 != NULL))
		{
			/* Shift the Devices up */
			gpointer data = glist1->data;
			glist1->data = glist2->data;
			glist2->data = data;

			/* Shift the GtkCList rows up */
			gtk_clist_freeze(clist);
			gtk_clist_swap_rows(
				clist,
				row - 1,
				row
			);
			gtk_clist_thaw(clist);

			/* Notify about the Devices being shifted */
			edv_emit_device_modified(
				core,
				row - 1,
				EDV_DEVICE(glist1->data)
			);
			edv_emit_device_modified(
				core,
				row,
				EDV_DEVICE(glist2->data)
			);

			EDVDevicesListWinUpdateMenus(lw);
		}
	}

	EDVDevicesListSetBusy(lw, FALSE);
}

/*
 *	Shift Down callback.
 */
static void EDVDevicesListWinDownCB(GtkWidget *widget, gpointer data)
{
	gint row;
	GtkCList *clist;
	EDVCore *core;
	edv_devices_list_win_struct *lw = EDV_DEVICES_LIST_WIN(data);
	if(lw == NULL)
		return;

	if(lw->freeze_count > 0)
		return;

	clist = GTK_CLIST(lw->devices_clist);
	core = lw->core;

	EDVDevicesListSetBusy(lw, TRUE);

	/* Get the last selected row */
	row = edv_clist_get_selected_last(clist, NULL);

	/* Can shift the selected row down? */
	if((row > -1) && (row < (clist->rows - 1)))
	{
		/* Get the GList pointers to the Devices being shifted */
		GList	*glist1 = g_list_nth(
			core->devices_list,
			(guint)row
		);
		GList	*glist2 = g_list_nth(
			core->devices_list,
			(guint)(row + 1)
		);
		if((glist1 != NULL) && (glist2 != NULL))
		{
			/* Shift the Devices down */
			gpointer data = glist1->data;
			glist1->data = glist2->data;
			glist2->data = data;

			/* Shift the GtkCList rows down */
			gtk_clist_freeze(clist);
			gtk_clist_swap_rows(
				clist,
				row,
				row + 1
			);
			gtk_clist_thaw(clist);

			/* Notify about the Devices being shifted */
			edv_emit_device_modified(
				core,
				row,
				EDV_DEVICE(glist1->data)
			);
			edv_emit_device_modified(
				core,
				row + 1,
				EDV_DEVICE(glist2->data)
			);

			EDVDevicesListWinUpdateMenus(lw);
		}
	}

	EDVDevicesListSetBusy(lw, FALSE);
}

/*
 *	Close callback.
 */
static void EDVDevicesListWinCloseCB(GtkWidget *widget, gpointer data)
{
	edv_devices_list_win_struct *lw = EDV_DEVICES_LIST_WIN(data);
	if(lw == NULL)
		return;

	if(lw->freeze_count > 0)
		return;

	EDVDevicesListWinUnmap(lw);
}


/*
 *	Updates the displayed Device.
 *
 *	This will destroy the display_client widget and create a
 *	new one if dev is not NULL.
 */
static void EDVDevicesListWinUpdateDisplay(
	edv_devices_list_win_struct *lw,
	EDVDevice *d
)
{
	const gint	border_major = 5,
					border_minor = 2;
	gint font_size;
	const gchar *font_encoding;
	gchar	*font_name_h1_bold,
			*font_name_h1,
			*font_name_h2_bold,
			*font_name_h2;
	GdkFont *font;
	GtkRcStyle *rcstyle, *standard_rcstyle;
	GtkStyle *style;
	GtkWidget *w, *parent, *parent2, *parent3, *parent4;
	const CfgItem *cfg_list;
	EDVCore *core;

	if(lw == NULL)
		return;

	core = lw->core;
	cfg_list = core->cfg_list;
	standard_rcstyle = core->standard_rcstyle;

	/* Destroy display widgets */
	if(lw->display_client != NULL)
	{
		GTK_WIDGET_DESTROY(lw->dev_icon_pm);
		lw->dev_icon_pm = NULL;
		GTK_WIDGET_DESTROY(lw->dev_name_label);
		lw->dev_name_label = NULL;

		GTK_WIDGET_DESTROY(lw->dev_device_path_label);
		lw->dev_device_path_label = NULL;
		GTK_WIDGET_DESTROY(lw->dev_mount_path_label);
		lw->dev_mount_path_label = NULL;
		GTK_WIDGET_DESTROY(lw->dev_fs_type_label);
		lw->dev_fs_type_label = NULL;

		GTK_WIDGET_DESTROY(lw->dev_mount_btn);
		lw->dev_mount_btn = NULL;
		GTK_WIDGET_DESTROY(lw->dev_eject_btn);
		lw->dev_eject_btn = NULL;
		GTK_WIDGET_DESTROY(lw->dev_properties_btn);
		lw->dev_properties_btn = NULL;

		GTK_WIDGET_DESTROY(lw->dev_fsck_btn);
		lw->dev_fsck_btn = NULL;
		GTK_WIDGET_DESTROY(lw->dev_tools_btn);
		lw->dev_tools_btn = NULL;
		GTK_WIDGET_DESTROY(lw->dev_format_btn);
		lw->dev_format_btn = NULL;

		GTK_WIDGET_DESTROY(lw->dev_pie_chart);
		lw->dev_pie_chart = NULL;

		GTK_WIDGET_DESTROY(lw->display_client);
		lw->display_client = NULL;
	}

	/* If device is not available then do not recreate the
	 * display widgets
	 */
	if(d == NULL)
		return;

	/* Realize the Device since we need to use its icons */
	edv_device_realize(d, FALSE);


	/* Get parent to create display client widget on */
	parent = lw->display_parent;
	if(parent == NULL)
		return;

	style = gtk_widget_get_style(parent);
	if(style == NULL)
		return;

	/* Get the base font size */
	font = style->font;
	font_size = GDK_FONT_GET_FONT_NAME_SIZE(font);
	if(font_size < 3)
		font_size = 3;

	/* Format the font names */
#if defined(PROG_LANGUAGE_POLISH)
	font_encoding = "iso8859-2";
#else
	font_encoding = "iso8859-1";
#endif

	font_name_h1_bold = g_strdup_printf(
"-adobe-helvetica-bold-r-normal-*-%i-*-*-*-*-*-%s",
		font_size + 2,
		font_encoding
	);
	font_name_h1 = g_strdup_printf(
"-adobe-helvetica-medium-r-normal-*-%i-*-*-*-*-*-%s",
		font_size + 2,
		font_encoding
	);
	font_name_h2_bold = g_strdup_printf(
"-adobe-helvetica-bold-r-normal-*-%i-*-*-*-*-*-%s",
		font_size,
		font_encoding
	);
	font_name_h2 = g_strdup_printf(
"-adobe-helvetica-medium-r-normal-*-%i-*-*-*-*-*-%s",
		font_size,
		font_encoding
	);

#define CLEANUP_RETURN	{		\
 g_free(font_name_h1_bold);		\
 g_free(font_name_h1);			\
 g_free(font_name_h2_bold);		\
 g_free(font_name_h2);			\
									\
 return;				\
}

	/* Create the new client GtkHBox for multiple columns */
	lw->display_client = w = gtk_hbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;


	/* Left column GtkVBox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent3 = w;


	/* Icon & Name GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Create the icon? */
	if(TRUE)
	{
		const EDVDeviceIconState state = EDV_DEVICE_ICON_STATE_STANDARD;
		EDVPixmap *icon = edv_pixmap_ref(d->medium_icon[state]);
		if(icon == NULL)
			icon = edv_pixmap_ref(d->small_icon[state]);
		if(edv_pixmap_is_loaded(icon))
		{
			lw->dev_icon_pm = w = edv_pixmap_new_gtk_pixmap(icon);
			if(w != NULL)
			{
				gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
				gtk_widget_show(w);
			}
		}
		(void)edv_pixmap_unref(icon);
	}
	/* Name label (or device path if name is empty) */
	if(!STRISEMPTY(d->name))
	{
		lw->dev_name_label = w = gtk_label_new(d->name);
	}
	else
	{
		const gchar	*dev_path = d->device_path,
					*dev_name = g_basename(dev_path);
		if(dev_name == NULL)
			dev_name = "(null)";
		lw->dev_name_label = w = gtk_label_new(dev_name);
	}
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	rcstyle = gtk_rc_style_new();
	rcstyle->font_name = STRDUP(font_name_h1_bold);
	gtk_widget_modify_style(w, rcstyle);
	gtk_widget_show(w);
	GTK_RC_STYLE_UNREF(rcstyle);


	/* Device Path GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Device Path GtkLabel */
	rcstyle = gtk_rc_style_new();
	rcstyle->font_name = STRDUP(font_name_h2);
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"El Sendero Del Artefacto: "
#elif defined(PROG_LANGUAGE_FRENCH)
"Sentier D'appareil: "
#else
"Device Path: "
#endif
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_modify_style(w, rcstyle);
	gtk_widget_show(w);
	GTK_RC_STYLE_UNREF(rcstyle);

	/* Device Path value GtkLabel */
	if(!STRISEMPTY(d->device_path))
	{
		rcstyle = gtk_rc_style_new();
		rcstyle->font_name = STRDUP(font_name_h2);
		lw->dev_device_path_label = w = gtk_label_new(
			d->device_path
		);
		gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
		gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
		gtk_widget_modify_style(w, rcstyle);
		gtk_widget_show(w);
		GTK_RC_STYLE_UNREF(rcstyle);
	}


	/* Mount Path GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Mount Path GtkLabel */
	rcstyle = gtk_rc_style_new();
	rcstyle->font_name = STRDUP(font_name_h2);
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Monte Sendero: "
#elif defined(PROG_LANGUAGE_FRENCH)
"Monter Le Sentier: "
#else
"Mount Path: "
#endif
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_modify_style(w, rcstyle);
	gtk_widget_show(w);
	GTK_RC_STYLE_UNREF(rcstyle);

	/* Mount Path value GtkLabel */
	if(!STRISEMPTY(d->mount_path))
	{
		rcstyle = gtk_rc_style_new();
		rcstyle->font_name = STRDUP(font_name_h2);
		lw->dev_mount_path_label = w = gtk_label_new(
			d->mount_path
		);
		gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
		gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
		gtk_widget_modify_style(w, rcstyle);
		GTK_RC_STYLE_UNREF(rcstyle);
		gtk_widget_show(w);
	}


	/* Filesystem GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Filesystem GtkLabel */
	rcstyle = gtk_rc_style_new();
	rcstyle->font_name = STRDUP(font_name_h2);
	w = gtk_label_new("Filesystem:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_modify_style(w, rcstyle);
	gtk_widget_show(w);
	GTK_RC_STYLE_UNREF(rcstyle);

	/* Filesystem value GtkLabel */
	rcstyle = gtk_rc_style_new();
	rcstyle->font_name = STRDUP(font_name_h2);
	lw->dev_fs_type_label = w = gtk_label_new(
		(d->fs_type_name != NULL) ? d->fs_type_name : ""
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_modify_style(w, rcstyle);
	GTK_RC_STYLE_UNREF(rcstyle);
	gtk_widget_show(w);


	/* Mount, eject, and properties buttons GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Mount/unmount button */
	lw->dev_mount_btn = w = GUIButtonPixmap(
		(guint8 **)(EDV_DEVICE_IS_MOUNTED(d) ?
			icon_unmount_20x20_xpm : icon_mount_20x20_xpm
		)
	);
	gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(EDVDevicesListWinMountCB), lw
	);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"Monte o de dispositivo de monte"
#elif defined(PROG_LANGUAGE_FRENCH)
"Monter ou de l'appareil de mont"
#elif defined(PROG_LANGUAGE_GERMAN)
"Untersatz oder ab untersatz vorrichtung"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Montare o via dal congegno di monte"
#elif defined(PROG_LANGUAGE_DUTCH)
"Berg of van berg apparaat"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Monte ou fora monte artifcio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Fotstykke eller av fotstykkeinnretning"
#else
"Mount/unmount the device"
#endif
	);
	gtk_widget_set_sensitive(w, !EDV_DEVICE_IS_NO_UNMOUNT(d));
	if(standard_rcstyle != NULL)
		gtk_widget_modify_style_recursive(w, standard_rcstyle);
	gtk_widget_show(w);

	/* Eject button */
	lw->dev_eject_btn = w = GUIButtonPixmap(
		(guint8 **)icon_eject_20x20_xpm
	);
	gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(EDVDevicesListWinEjectCB), lw
	);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"Expulse medios del dispositivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Ejecter la presse de l'appareil"
#elif defined(PROG_LANGUAGE_GERMAN)
"Werfen sie medien von der vorrichtung aus"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Espellere la stampa dal congegno"
#elif defined(PROG_LANGUAGE_DUTCH)
"Stoot media van het apparaat uit"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Expulse imprensa do artifcio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kast ut medier fra innretningen"
#else
"Eject the media from the device"
#endif
	);
	gtk_widget_set_sensitive(
		w,
		!STRISEMPTY(d->command_eject) ? TRUE : FALSE
	);
	if(standard_rcstyle != NULL)
		gtk_widget_modify_style_recursive(w, standard_rcstyle);
	gtk_widget_show(w);

	/* Properties button */
	lw->dev_properties_btn = w = GUIButtonPixmap(
		(guint8 **)icon_properties_20x20_xpm
	);
	gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(EDVDevicesListWinPropertiesCB), lw
	);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"Modifique las propiedades de sendero de monte"
#elif defined(PROG_LANGUAGE_FRENCH)
"Modifier les proprits de sentier de mont"
#elif defined(PROG_LANGUAGE_GERMAN)
"Modifizieren sie untersatz pfad eigentmer"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Modificare le propriet di sentiero di monte"
#elif defined(PROG_LANGUAGE_DUTCH)
"Wijziig berg pad eigendommen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Modifique propriedades de caminho de monte"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Modifiser fotstykkestieiendomer"
#else
"Modify the properties of the mount path"
#endif
	);
	gtk_widget_set_sensitive(
		w,
		!STRISEMPTY(d->mount_path) ? TRUE : FALSE
	);
	if(standard_rcstyle != NULL)
		gtk_widget_modify_style_recursive(w, standard_rcstyle);
	gtk_widget_show(w);

	/* FSCK button */
	lw->dev_fsck_btn = w = GUIButtonPixmap(
		(guint8 **)icon_fsck_20x20_xpm
	);
	gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(EDVDevicesListWinFSCKCB), lw
	);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"Verifique el sistema del archivo de dispositivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Vrifier le systme de classement de l'appareil"
#elif defined(PROG_LANGUAGE_GERMAN)
"Prfen sie das dateisystem der vorrichtung"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Controllare il sistema di file del congegno"
#elif defined(PROG_LANGUAGE_DUTCH)
"Controleer het dossier van het apparaat systeem"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Verifique o sistema de arquivo do artifcio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Sjekk innretningens arkivsystem"
#else
"Check the device's filesystem"
#endif
	);
	gtk_widget_set_sensitive(
		w,
		!STRISEMPTY(d->command_check) ? TRUE : FALSE
	);
	if(standard_rcstyle != NULL)
		gtk_widget_modify_style_recursive(w, standard_rcstyle);
	gtk_widget_show(w);

	/* Tools button */
	lw->dev_tools_btn = w = GUIButtonPixmap(
		(guint8 **)icon_tools_20x20_xpm
	);
	gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(EDVDevicesListWinToolsCB), lw
	);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
"Corra el programa de instrumentos de dispositivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Courir le programme d'outils de l'appareil"
#elif defined(PROG_LANGUAGE_GERMAN)
"Laufen sie die werkzeuge der vorrichtung programmieren"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Ha correto il programma di attrezzi del congegno"
#elif defined(PROG_LANGUAGE_DUTCH)
"Loop de werktuigen van het apparaat programma"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Corra o programa de ferramentas do artifcio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kjr innretningens redskapprogram"
#else
"Run the device's tools program"
#endif
	);
	gtk_widget_set_sensitive(
		w,
		!STRISEMPTY(d->command_tools) ? TRUE : FALSE
	);
	if(standard_rcstyle != NULL)
		gtk_widget_modify_style_recursive(w, standard_rcstyle);
	gtk_widget_show(w);

	/* Format button */
	lw->dev_format_btn = w = GUIButtonPixmap(
		(guint8 **)icon_floppy_20x20_xpm
	);
	gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(EDVDevicesListWinFormatCB), lw
	);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"Formatear los medios en el dispositivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Met en format la presse dans l'appareil"
#elif defined(PROG_LANGUAGE_GERMAN)
"Formatieren sie die medien in der vorrichtung"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il formato la stampa nel congegno"
#elif defined(PROG_LANGUAGE_DUTCH)
"Formatteer de media in het apparaat"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O formato a imprensa no artifcio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Formater mediene i innretningen"
#else
"Format the media in the device"
#endif
	);
	gtk_widget_set_sensitive(
		w,
		!STRISEMPTY(d->command_format) ? TRUE : FALSE
	);
	if(standard_rcstyle != NULL)
		gtk_widget_modify_style_recursive(w, standard_rcstyle);
	gtk_widget_show(w);


	/* Right column vbox */
	w = gtk_vbox_new(TRUE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Create the Usage Pie Chart if mounted */
	if(EDV_DEVICE_IS_MOUNTED(d))
	{
		const gulong	adj_division = 10l,
							block_size = EDV_GET_UL(EDV_CFG_PARM_BLOCK_SIZE),
							total_kb = d->blocks_total,
							free_kb = d->blocks_free,
							available_kb = d->blocks_available,
							used_kb = MAX(total_kb - free_kb, 0l);
		gchar *ss_kb, *ss_mb, *text, *text2;
		GdkColor c;
		GtkAdjustment *adj;

		adj = (GtkAdjustment *)gtk_adjustment_new(
		    0.0f, 0.0f,
		    (gfloat)(total_kb / adj_division),
		    0.0f, 0.0f, 0.0f
		);
		GDK_COLOR_SET_COEFF(&c, 0.0f, 0.0f, 1.0f);
		ss_kb = STRDUP(edv_str_size_delim(total_kb));
		ss_mb = STRDUP(edv_str_size_delim(
			(block_size > 0l) ?
				(total_kb / block_size) : (total_kb / 1024l)
		));
		text = g_strdup_printf(
		    "%s: %s kb (%s mb)",
#if defined(PROG_LANGUAGE_SPANISH)
"La Capacidad Total"
#elif defined(PROG_LANGUAGE_FRENCH)
"Capacite' Totale"
#elif defined(PROG_LANGUAGE_GERMAN)
"Gesamte Kapazitt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Capacit Totale"
#elif defined(PROG_LANGUAGE_DUTCH)
"Totale Capaciteit"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Capacidade Total"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Total Kapasitet"
#else
"Total Capacity"
#endif
			, ss_kb, ss_mb
		);
		g_free(ss_kb);
		g_free(ss_mb);

		ss_kb = STRDUP(edv_str_size_delim(free_kb));
		ss_mb = STRDUP(edv_str_size_delim(
			(block_size > 0l) ?
				(free_kb / block_size) : (free_kb / 1024l)
		));
		text2 = g_strdup_printf(
		    "%s kb (%s mb)",
			ss_kb, ss_mb
		);
		g_free(ss_kb);
		g_free(ss_mb);

		lw->dev_pie_chart = w = PieChartNew(
			(EDV_GET_B(EDV_CFG_PARM_LISTS_DOUBLE_BUFFER) ?
				PIE_CHART_DOUBLE_BUFFER : 0) |
				PIE_CHART_SHOW_LABELS |
				PIE_CHART_SHOW_SHADOW |
				PIE_CHART_SHOW_OUTLINE,
			110, 70,
		    adj,
			&c,
		    NULL,
			text,
#if defined(PROG_LANGUAGE_SPANISH)
"Libre"
#elif defined(PROG_LANGUAGE_FRENCH)
"Libre"
#elif defined(PROG_LANGUAGE_GERMAN)
"Frei"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Libero"
#elif defined(PROG_LANGUAGE_DUTCH)
"Vrij"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Livre"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Fri"
#else
"Free"
#endif
			":",
			text2
		);
		adj = (GtkAdjustment *)GTK_OBJECT_UNREF(GTK_OBJECT(adj));
		g_free(text);
		g_free(text2);
		gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, FALSE, 0);
		gtk_widget_show(w);

		GDK_COLOR_SET_COEFF(&c, 0.0f, 1.0f, 1.0f);
		adj = (GtkAdjustment *)gtk_adjustment_new(
		    0.0f, 0.0f,
		    (gfloat)(available_kb / adj_division),
		    0.0f, 0.0f, 0.0f
		);
		ss_kb = STRDUP(edv_str_size_delim(available_kb));
		ss_mb = STRDUP(edv_str_size_delim(
			(block_size > 0l) ?
				(available_kb / block_size) : (available_kb / 1024l)
		));
		text = g_strdup_printf(
		    "%s kb (%s mb)",
			ss_kb, ss_mb
		);
		g_free(ss_kb);
		g_free(ss_mb);
		PieChartValueAdd(
			w,
			adj,
			&c,
#if defined(PROG_LANGUAGE_SPANISH)
"Libre & Disponible"
#elif defined(PROG_LANGUAGE_FRENCH)
"Libre & Disponible"
#elif defined(PROG_LANGUAGE_GERMAN)
"Frei & Verfgbar"
#elif defined(PROG_LANGUAGE_ITALIAN)
"ibero & Disponibile"
#elif defined(PROG_LANGUAGE_DUTCH)
"Vrij & Verkrijgbare"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Livre & Disponvel"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Fri & Tilgjengelig"
#else
"Free & Available"
#endif
			":",
			text
		);
		adj = (GtkAdjustment *)GTK_OBJECT_UNREF(GTK_OBJECT(adj));
		g_free(text);

		GDK_COLOR_SET_COEFF(&c, 1.0f, 0.0f, 1.0f);
		adj = (GtkAdjustment *)gtk_adjustment_new(
		    0.0f, 0.0f,
		    (gfloat)(used_kb / adj_division),
		    0.0f, 0.0f, 0.0f
		);
		ss_kb = STRDUP(edv_str_size_delim(used_kb));
		ss_mb = STRDUP(edv_str_size_delim(
			(block_size > 0l) ?
				(used_kb / block_size) : (used_kb / 1024l)
		));
		text = g_strdup_printf(
		    "%s kb (%s mb)",
			ss_kb, ss_mb
		);
		g_free(ss_kb);
		g_free(ss_mb);
 	    PieChartValueAdd(
		    w,
			adj,
			&c,
#if defined(PROG_LANGUAGE_SPANISH)
"Utilizado"
#elif defined(PROG_LANGUAGE_FRENCH)
"Utilise'"
#elif defined(PROG_LANGUAGE_GERMAN)
"Benutzt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Usato"
#elif defined(PROG_LANGUAGE_DUTCH)
"Gebruikt"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Usado"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Brukt"
#else
"Used"
#endif
			":", text
		);
		adj = (GtkAdjustment *)GTK_OBJECT_UNREF(GTK_OBJECT(adj));
		g_free(text);

		if(standard_rcstyle != NULL)
			gtk_widget_modify_style_recursive(
				w,
				standard_rcstyle
			);
	}
	else
	{
		w = gtk_label_new("(Device Not Mounted)");
		gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, FALSE, 0);
		rcstyle = gtk_rc_style_new();
		rcstyle->font_name = STRDUP(font_name_h1_bold);
		gtk_widget_modify_style(w, rcstyle);
		GTK_RC_STYLE_UNREF(rcstyle);
		gtk_widget_show(w);
	}

	CLEANUP_RETURN;
#undef CLEANUP_RETURN
}


/*
 *	Sets the row values to reflect the values of the device.
 *
 *	The row specifies the row who's values are to be updated.
 *
 *	The dev specifies the device who's values are to be used
 *	when updating the row's values.
 *
 *	The specified dev may or may not be the row's data and it
 *	will not be set as the row's data by this call.
 */
void EDVDevicesListWinSetRow(
	edv_devices_list_win_struct *lw,
	const gint row,
	EDVDevice *d
)
{
	gint		column,
					ncolumns;
	GtkCList *clist;
	EDVCore *core;

	if((lw == NULL) || (d == NULL))
		return;

	clist = GTK_CLIST(lw->devices_clist);
	ncolumns = clist->columns;
	core = lw->core;

	if((row < 0) || (row >= clist->rows))
		return;

	/* Realize the Device since we need to use its icons */
	edv_device_realize(d, FALSE);

	/* Column 0 - Name or Device Path */
	column = 0;
	if(column < ncolumns)
	{
		EDVPixmap *icon = edv_pixmap_ref(d->small_icon[
			EDV_DEVICE_ICON_STATE_STANDARD
		]);
		const gchar *text = d->name;
		if((text == NULL) && (d->device_path != NULL))
			text = g_basename(d->device_path);
		if(text == NULL)
			text = "";

		if(edv_pixmap_is_loaded(icon))
			gtk_clist_set_pixtext(
				clist,
				row, column,
				text,
				EDV_LIST_PIXMAP_TEXT_SPACING,
				icon->pixmap, icon->mask
			);
		else
			gtk_clist_set_text(
				clist,
				row, column,
				text
			);

		(void)edv_pixmap_unref(icon);
	}

	/* Column 1 - Device Path */
	column = 1;
	if(column < ncolumns)
	{
		const gchar *text = d->device_path;
		if(text == NULL)
			text = "";

		gtk_clist_set_text(
			clist,
			row, column,
			text
		);
	}

	/* Column 2 - Mount Path */
	column = 2;
	if(column < ncolumns)
	{
		const gchar *text = d->mount_path;
		if(text == NULL)
			text = "";

		gtk_clist_set_text(
			clist,
			row, column,
			text
		);
	}

	/* Column 3 - Usage */
	column = 3;
	if(column < ncolumns)
	{
		const gint column = 3;
		GtkCListColumn *column_ptr = &clist->column[column];
		const gint	width = (column_ptr->width_set) ? column_ptr->width : -1,
					height = GTK_CLIST_ROW_HEIGHT_SET(clist) ?
						(clist->row_height - 4) : -1;
		EDVPixmap *p;
		if(EDV_DEVICE_IS_MOUNTED(d))
		{
			GtkWidget *w = GTK_WIDGET(clist);
			gfloat usage_coeff = (d->blocks_total > 0l) ?
				(1.0f -
				((gfloat)d->blocks_available / (gfloat)d->blocks_total)) : 0.0f;
			if(usage_coeff > 1.0f)
				usage_coeff = 1.0f;
			else if(usage_coeff < 0.0f)
				usage_coeff = 0.0f;
			p = edv_new_progress_pixmap(
				w->window,
				gtk_widget_get_style(w),
				width, height,
				usage_coeff,
				TRUE,			/* Draw value */
				GTK_ORIENTATION_HORIZONTAL,
				FALSE			/* Not reverse */
			);
		}
		else
		{
			p = NULL;
		}
		if(edv_pixmap_is_loaded(p))
			gtk_clist_set_pixmap(
				clist,
				row, column,
				p->pixmap, p->mask
			);
		else
			gtk_clist_set_text(
				clist,
				row, column,
				"Not Mounted"
			);

		(void)edv_pixmap_unref(p);
	}

	/* Column 4 - Filesystem */
	column = 4;
	if(column < ncolumns)
	{
		const gchar *text = d->fs_type_name;
		if(text == NULL)
			text = "";
			
		gtk_clist_set_text(
			clist,
			row, column,
			text
		);
	}
}


/*
 *      Selects the Device at index i.
 */
void EDVDevicesListWinSelect(
	edv_devices_list_win_struct *lw,
	const gint i
)
{
	GtkCList *clist;

	if(lw == NULL)
		return;

	clist = GTK_CLIST(lw->devices_clist);
	if((i >= 0) && (i < clist->rows))
	{
		gtk_clist_freeze(clist);
		gtk_clist_unselect_all(clist);
		gtk_clist_select_row(clist, i, 0);
		gtk_clist_thaw(clist);
	}
}

/*
 *	Sets a device by matching it with a path.
 *
 *	The path specifies the object that resides on the device,
 *	the mount path itself, or the device path itself.
 *
 *      Returns the index of the matched MIME Type of -1 on failed
 *      match.
 */
gint EDVDevicesListSelectByPath(
	edv_devices_list_win_struct *lw,
	const gchar *path
)
{
	gint i;
	GList *glist;
	EDVDevice *d;
	EDVCore *core;

	if((lw == NULL) || STRISEMPTY(path))
		return(-2);

	core = lw->core;

	/* Iterate through the Devices list */
	for(glist = core->devices_list, i = 0;
	    glist != NULL;
	    glist = g_list_next(glist), i++
	)
	{
		d = EDV_DEVICE(glist->data);
		if(d == NULL)
			continue;

		if(d->device_path != NULL)
		{
			if(!strcmp(d->device_path, path))
			{
				EDVDevicesListWinSelect(lw, i);
				return(i);
			}
		}

		if(d->mount_path != NULL)
		{
			if(edv_path_is_parent(
				d->mount_path,
				path
			))
			{
				EDVDevicesListWinSelect(lw, i);
				return(i);
			}
		}
	}

	return(-1);
}


/*
 *	Gets the Devices List.
 */
void EDVDevicesListWinGetList(
	edv_devices_list_win_struct *lw,
	const gboolean verbose
)
{
	gint i, ndevices, row, ncolumns, progress_update_mod;
	gchar **strv;
	GList *glist;
	GtkWidget *toplevel;
	GtkCList *clist;
	EDVDevice *d;
	EDVCore *core;

	if(lw == NULL)
		return;

	if(verbose && ProgressDialogIsQuery())
		return;

	toplevel = lw->toplevel;
	clist = GTK_CLIST(lw->devices_clist);
	core = lw->core;

	ncolumns = clist->columns;
	if(ncolumns < 1)
		return;

	/* Allocate the row cell values */
	strv = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
	for(i = 0; i < ncolumns; i++)
		strv[i] = "";

	/* Map the progress dialog? */
	if(verbose)
	{
		ProgressDialogSetTransientFor(toplevel);
		ProgressDialogMap(
			"Loading Devices",
			"Loading Devices...",
			(const guint8 **)icon_devices_list_32x32_xpm,
			"Stop"
		);
		gdk_flush();
	}

	gtk_clist_freeze(clist);

	/* Remove all the existing rows and clear the display */
	gtk_clist_clear(clist);
	EDVDevicesListWinUpdateDisplay(lw, NULL);

	/* Get each Device from the Devices List and append them
	 * to the GtkCList
	 */
	ndevices = g_list_length(core->devices_list);
	progress_update_mod = MAX(ndevices / 100, 1);
	for(glist = core->devices_list, i = 0;
		glist != NULL;
		glist = g_list_next(glist), i++
	)
	{
		d = EDV_DEVICE(glist->data);
		if(d == NULL)
			continue;

		row = gtk_clist_append(clist, strv);
		if(row < 0)
			break;

		EDVDevicesListWinSetRow(lw, row, d);

		/* Report progress and check for user abort */
		if(verbose && ((i % progress_update_mod) == 0))
		{
			if(EDVDevicesListProgressCB(
				lw,
				(gulong)(i + 1), (gulong)ndevices
			))
				break;
		}
	}

	gtk_clist_thaw(clist);

	g_free(strv);

	EDVDevicesListWinUpdateMenus(lw);

	/* Unmap the progress dialog? */
	if(verbose)
	{
		ProgressDialogBreakQuery(TRUE);
		ProgressDialogSetTransientFor(NULL);
	}
}


/*
 *	Disk usage changed callback.
 */
void EDVDevicesListWinUsageChangedCB(
	edv_devices_list_win_struct *lw
)
{
	gint i, sel_dev_num;
	GList *glist;
	GtkCList *clist;
	EDVDevice *d;
	EDVCore *core;

	if(lw == NULL)
		return;

	/* Skip if the devices list is not mapped */
	if(!EDVDevicesListWinIsMapped(lw))
		return;

	clist = GTK_CLIST(lw->devices_clist);
	core = lw->core;

	/* Get the selected device */
	sel_dev_num = edv_clist_get_selected_last(clist, NULL);

	/* Update each device listed */
	for(glist = core->devices_list, i = 0;
		glist != NULL;
		glist = g_list_next(glist), i++
	)
	{
		d = EDV_DEVICE(glist->data);
		if(d == NULL)
			continue;

		EDVDevicesListWinSetRow(lw, i, d);

		if(sel_dev_num == i)
			EDVDevicesListWinUpdateDisplay(lw, d);
	}
}


/*
 *	Reconfigured notify callback.
 */
void EDVDevicesListWinReconfiguredNotifyCB(
	edv_devices_list_win_struct *lw
)
{
	GtkRcStyle *standard_rcstyle, *lists_rcstyle;
	GtkWidget *w;
	const CfgItem *cfg_list;
	EDVCore *core;

	if(lw == NULL)
		return;

	core = lw->core;
	if(core == NULL)
		return;

	cfg_list = core->cfg_list;
	standard_rcstyle = core->standard_rcstyle;
	lists_rcstyle = core->lists_rcstyle;

	/* Update RC styles */
	w = lw->toplevel;
	if((w != NULL) && (standard_rcstyle != NULL))
		gtk_widget_modify_style_recursive(w, standard_rcstyle);
	w = lw->devices_clist;
	if((w != NULL) && (lists_rcstyle != NULL))
		gtk_widget_modify_style_recursive(w, lists_rcstyle);

	EDVDevicesListWinUpdateMenus(lw);
}

/*
 *      Called whenever a device has been mounted or unmounted.
 */
void EDVDevicesListWinMountNotifyCB(
	edv_devices_list_win_struct *lw,
	const gint dev_num, EDVDevice *d,
	const gboolean is_mounted
)
{
	/* Forward this to the device modified callback */
	EDVDevicesListWinDeviceModifiedCB(lw, dev_num, d);
}

/*
 *	Device added notify.
 */
void EDVDevicesListWinDeviceAddedCB(
	edv_devices_list_win_struct *lw,
	const gint dev_num, EDVDevice *d
)
{
	gint row, new_row, ncolumns;
	gchar **strv;
	GtkCList *clist;

	if((lw == NULL) || (d == NULL))
		return;

	clist = GTK_CLIST(lw->devices_clist);
	ncolumns = MAX(clist->columns, 1);

	/* The added device index corresponds to the GtkCList row */
	row = dev_num;

	/* Add the new device to the GtkCList */
	strv = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
	if(strv != NULL)
	{
		gint i;
		for(i = 0; i < ncolumns; i++)
			strv[i] = "";
	}
	if(row >= clist->rows)
		new_row = gtk_clist_append(clist, strv);
	else
		new_row = gtk_clist_insert(clist, row, strv);

	g_free(strv);

	if(new_row < 0)
		return;

	EDVDevicesListWinSetRow(lw, new_row, d);

	EDVDevicesListWinUpdateMenus(lw);
}

/*
 *	Device modified notify.
 */
void EDVDevicesListWinDeviceModifiedCB(
	edv_devices_list_win_struct *lw,
	const gint dev_num, EDVDevice *d
)
{
	gint row;
	GtkCList *clist;

	if((lw == NULL) || (d == NULL))
		return;

	clist = GTK_CLIST(lw->devices_clist);

	/* The modified device index corresponds to the GtkCList row */
	row = dev_num;
	if((row >= 0) && (row < clist->rows))
	{
		const gint sel_row = edv_clist_get_selected_last(clist, NULL);

		/* Update the GtkCList row corresponding to this device */
		EDVDevicesListWinSetRow(lw, row, d);

		/* Is this device currently selected? */
		if(row == sel_row)
		{
			/* Update the displayed device info */
			EDVDevicesListWinUpdateDisplay(lw, d);
		}
	}

	EDVDevicesListWinUpdateMenus(lw);
}

/*
 *	Device removed notify.
 */
void EDVDevicesListWinDeviceRemovedCB(
	edv_devices_list_win_struct *lw,
	const gint dev_num
)
{
	gint row;
	GtkCList *clist;

	if(lw == NULL)
		return;

	clist = GTK_CLIST(lw->devices_clist);

	/* The removed device index corresponds to the GtkCList row */
	row = dev_num;

	/* Delete the GtkCList row corresponding to this device */
	if((row >= 0) && (row < clist->rows))
		gtk_clist_remove(clist, row);

	EDVDevicesListWinUpdateMenus(lw);

	/* Forward this signal to the Device Edit Dialog */
	EDVDeviceEditDlgDeviceRemovedCB(lw->device_edit_dlg, dev_num);
}


/*
 *	Creates a new Devices List Window.
 */
edv_devices_list_win_struct *EDVDevicesListWinNew(
	EDVCore *core
)
{
	const gint	border_major = 5,
					border_minor = 2;
	const gchar *wm_name = NULL, *wm_class;
	gchar *heading[5];
	gpointer mclient_data;
	gpointer entry_rtn;
	GdkWindow *window;
	GtkRcStyle *standard_rcstyle, *lists_rcstyle;
	GtkAccelGroup *accelgrp;
	GtkStyle *style;
	GtkWidget	*w, *menu,
					*parent, *parent2, *parent3, *parent4,
					*parent5,
					*toplevel;
	GtkCList *clist;
	edv_devices_list_win_struct *lw = EDV_DEVICES_LIST_WIN(
		g_malloc0(sizeof(edv_devices_list_win_struct))
	);
	if((lw == NULL) || (core == NULL))
	{
		g_free(lw);
		return(NULL);
	}

	standard_rcstyle = core->standard_rcstyle;
	lists_rcstyle = core->lists_rcstyle;

	wm_name = core->wm_name;
	wm_class = core->wm_class;

	lw->toplevel = toplevel = gtk_window_new(GTK_WINDOW_DIALOG);
	lw->accelgrp = accelgrp = gtk_accel_group_new();
	lw->busy_count = 0;
	lw->freeze_count = 0;
	lw->core = core;

	lw->freeze_count++;

	/* Toplevel GtkWindow */
	w = toplevel;
	if(!STRISEMPTY(wm_name) && !STRISEMPTY(wm_class))
		gtk_window_set_wmclass(GTK_WINDOW(w), wm_name, wm_class);
	else
		gtk_window_set_wmclass(
			GTK_WINDOW(w),
			EDV_DEVICES_LIST_WM_CLASS_WINDOW_NAME,
			PROG_NAME
		);
	gtk_window_set_policy(GTK_WINDOW(w), FALSE, FALSE, FALSE);
	gtk_widget_set_usize(w, EDV_DEVICES_LIST_WIN_WIDTH, EDV_DEVICES_LIST_WIN_HEIGHT);
	gtk_window_set_title(GTK_WINDOW(w), EDV_DEVICES_LIST_WIN_TITLE);
	gtk_widget_realize(w);
	style = gtk_widget_get_style(w);
	window = w->window;
	if(window != NULL)
	{
		gdk_window_set_decorations(
			window,
			GDK_DECOR_BORDER | GDK_DECOR_TITLE | GDK_DECOR_MENU |
			GDK_DECOR_MINIMIZE
		);
		gdk_window_set_functions(
			window,
			GDK_FUNC_MOVE | GDK_FUNC_MINIMIZE | GDK_FUNC_CLOSE
		);
		GUISetWMIcon(window, (guint8 **)icon_devices_list_48x48_xpm);
	}
	gtk_signal_connect(
		GTK_OBJECT(w), "delete_event",
		GTK_SIGNAL_FUNC(EDVDevicesListWinDeleteEventCB), lw
	);
	gtk_container_border_width(GTK_CONTAINER(w), 0);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	parent = w;


	/* Main GtkVBox */
	lw->main_vbox = w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;


	/* Find GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	w = GUIPromptBar(
		(guint8 **)icon_search_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"El Hallazgo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Dcouverte"
#elif defined(PROG_LANGUAGE_GERMAN)
"Fund"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Trovare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Vondst"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Ache"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Funn"
#else
"Find"
#endif
		":", NULL, &entry_rtn
	);
	lw->find_entry = (GtkWidget *)entry_rtn;
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);

	w = (GtkWidget *)entry_rtn;
	gtk_signal_connect(
		GTK_OBJECT(w), "activate",
		GTK_SIGNAL_FUNC(EDVDevicesListWinFindCB), lw
	);
	edv_entry_set_dnd(core, w);
	GUIEditableEndowPopupMenu(w, 0);

	/* Devices GtkCList & Buttons GtkVBox */
	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* GtkScrolledWindow for devices clist */
	w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
		GTK_SCROLLED_WINDOW(w),
		GTK_POLICY_AUTOMATIC,
		GTK_POLICY_AUTOMATIC
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Devices GtkCList */
#if defined(PROG_LANGUAGE_SPANISH)
	heading[0] = "El Nombre";
	heading[1] = "El Sendero Del Artefacto";
	heading[2] = "Monte Sendero";
	heading[3] = "Usage";
	heading[4] = "Archive Sistema";
#elif defined(PROG_LANGUAGE_FRENCH)
	heading[0] = "Nom";
	heading[1] = "Sentier D'appareil";
	heading[2] = "Monter Le Sentier";
	heading[3] = "Usage";
	heading[4] = "Classer Le Systme";
#elif defined(PROG_LANGUAGE_GERMAN)
	heading[0] = "Name";
	heading[1] = "Gertepfad";
	heading[2] = "Stellen Sie Pfad Auf";
	heading[3] = "Usage";
	heading[4] = "Dateisystem";
#elif defined(PROG_LANGUAGE_ITALIAN)
	heading[0] = "Il Nome";
	heading[1] = "Il Sentiero Di Congegno";
	heading[2] = "Montare Il Sentiero";
	heading[3] = "Usage";
	heading[4] = "Schedare Il Sistema";
#elif defined(PROG_LANGUAGE_DUTCH)
	heading[0] = "Naam";
	heading[1] = "Apparaat Pad";
	heading[2] = "Bestijg Pad";
	heading[3] = "Usage";
	heading[4] = "Archiveer Systeem";
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	heading[0] = "O Nome";
	heading[1] = "O Caminho De Artifcio";
	heading[2] = "Monte Caminho";
	heading[3] = "Usage";
	heading[4] = "Arquive Sistema";
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	heading[0] = "Navn";
	heading[1] = "Innretning Path";
	heading[2] = "Monter Path";
	heading[3] = "Usage";
	heading[4] = "Arkiver System";
#else
	heading[0] = "Name";
	heading[1] = "Device Path";
	heading[2] = "Mount Path";
	heading[3] = "Usage";
	heading[4] = "Filesystem";
#endif
	lw->devices_clist = w = gtk_clist_new_with_titles(
		sizeof(heading) / sizeof(gchar *),
		heading
	);
	clist = GTK_CLIST(w);
	gtk_widget_set_usize(w, -1, (gint)(EDV_DEVICES_LIST_WIN_HEIGHT * 0.38f));
	gtk_widget_add_events(
		w,
		GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
		GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
		GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(edv_clist_key_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(edv_clist_key_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(edv_clist_button_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(edv_clist_button_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "motion_notify_event",
		GTK_SIGNAL_FUNC(edv_clist_motion_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(EDVDevicesListWinKeyEventCB), lw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(EDVDevicesListWinKeyEventCB), lw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(EDVDevicesListWinButtonPressEventCB), lw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "select_row",
		GTK_SIGNAL_FUNC(EDVDevicesListWinSelectRowCB), lw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "unselect_row",
		GTK_SIGNAL_FUNC(EDVDevicesListWinUnselectRowCB), lw
	);
	gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_widget_realize(w);
	gtk_clist_set_column_width(clist, 0, 130);
	gtk_clist_set_column_justification(
		clist, 0, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_column_width(clist, 1, 100);
	gtk_clist_set_column_justification(
		clist, 1, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_column_width(clist, 2, 100);
	gtk_clist_set_column_justification(
		clist, 2, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_column_width(clist, 3, 80);
	gtk_clist_set_column_justification(
		clist, 3, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_column_width(clist, 4, 50);
	gtk_clist_set_column_justification(
		clist, 4, GTK_JUSTIFY_LEFT
	);
	gtk_clist_column_titles_passive(clist);
	gtk_clist_set_row_height(clist, EDV_LIST_ROW_SPACING);
	gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
	gtk_widget_show(w);


	/* Buttons GtkHBox */
	w = gtk_hbox_new(FALSE, 2 * border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Vbox for add, edit, and remove set of buttons */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Add button */
	lw->add_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_add_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Agregue"
#elif defined(PROG_LANGUAGE_FRENCH)
"Ajouter"
#elif defined(PROG_LANGUAGE_GERMAN)
"Fgen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Aggiungere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Toevoeg"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Adicione"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Tilfy"
#else
"Add"
#endif
		"...", NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(EDVDevicesListWinAddCB), lw
	);
	gtk_accel_group_add(
		accelgrp, GDK_a, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_a);
	gtk_widget_show(w);

	/* Edit button */
	lw->edit_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_edit_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Redacte"
#elif defined(PROG_LANGUAGE_FRENCH)
"Editer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Redigieren"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Redigere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bewerking"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Edite"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Rediger"
#else
"Edit"
#endif
		"...", NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(EDVDevicesListWinEditCB), lw
	);
	gtk_accel_group_add(
		accelgrp, GDK_e, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_e);
	gtk_widget_show(w);

	/* Remove button */
	lw->remove_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_remove_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Quite"
#elif defined(PROG_LANGUAGE_FRENCH)
"Enlever"
#elif defined(PROG_LANGUAGE_GERMAN)
"Nehmen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Togliere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Verwijdeer"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Retire"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Fjern"
#else
"Remove"
#endif
		, NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(EDVDevicesListWinRemoveCB), lw
	);
	gtk_accel_group_add(
		accelgrp, GDK_r, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_r);
	gtk_widget_show(w);


	/* GtkHBox for the up and down buttons */
	w = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* GtkVBox for the up button */
	w = gtk_vbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
	/* Up button */
	lw->up_btn = w = GUIButtonArrow(
		GTK_ARROW_UP,
		-1, -1
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, TRUE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(EDVDevicesListWinUpCB), lw
	);
	gtk_widget_show(w);

	/* GtkVBox for the down button */
	w = gtk_vbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
	/* Down button */
	lw->down_btn = w = GUIButtonArrow(
		GTK_ARROW_DOWN,
		-1, -1
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, TRUE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(EDVDevicesListWinDownCB), lw
	);
	gtk_widget_show(w);



	/* Horizontal separator */
	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Parent vbox for displaying selected device */
	lw->display_parent = w = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Client vbox that is parented to the display_parent, leave this
	 * NULL, it'll be created when needed.
	 */
	lw->display_client = NULL;


	/* Horizontal separator */
	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Hbox for buttons */
	w = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Close button */
	lw->close_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_close_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Cierre"
#elif defined(PROG_LANGUAGE_FRENCH)
"Fin"
#elif defined(PROG_LANGUAGE_GERMAN)
"Nah"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Vicino"
#elif defined(PROG_LANGUAGE_DUTCH)
"Einde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Prximo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Nr"
#else
"Close"
#endif
		, NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_end(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(EDVDevicesListWinCloseCB), lw
	);
	gtk_accel_group_add(
		accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
		accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);
	gtk_widget_show(w);



	/* Right click menu */
	lw->menu = menu = GUIMenuCreate();
	mclient_data = lw;
	if(menu != NULL)
	{
		guint accel_key, accel_mods;
		guint8 **icon;
		const gchar *label;
		void (*func_cb)(GtkWidget *, gpointer);

#define ADD_MENU_ITEM_LABEL	{			\
 w = GUIMenuItemCreate(					\
  menu,							\
  GUI_MENU_ITEM_TYPE_LABEL,				\
  accelgrp,						\
  icon,							\
  label,						\
  accel_key, accel_mods,				\
  func_cb, mclient_data					\
 );							\
}
#define ADD_MENU_ITEM_SEPARATOR		{		\
 w = GUIMenuItemCreate(					\
  menu,							\
  GUI_MENU_ITEM_TYPE_SEPARATOR,				\
  NULL,							\
  NULL, NULL,						\
  0, 0,							\
  NULL, NULL						\
 );							\
}
		icon = (guint8 **)icon_add_20x20_xpm;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"Agregue"
#elif defined(PROG_LANGUAGE_FRENCH)
"Ajouter"
#elif defined(PROG_LANGUAGE_GERMAN)
"Fgen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Aggiungere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Toevoeg"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Adicione"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Tilfy"
#else
"Add"
#endif
			"...";
		accel_key = 0;
		accel_mods = 0;
		func_cb = EDVDevicesListWinAddCB;
		ADD_MENU_ITEM_LABEL
		lw->add_mi = w;

		icon = (guint8 **)icon_edit_20x20_xpm;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"Redacte"
#elif defined(PROG_LANGUAGE_FRENCH)
"Editer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Redigieren"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Redigere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bewerking"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Edite"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Rediger"
#else
"Edit"
#endif
			"...";
		accel_key = 0;
		accel_mods = 0;
		func_cb = EDVDevicesListWinEditCB;
		ADD_MENU_ITEM_LABEL
		lw->edit_mi = w;

		icon = (guint8 **)icon_remove_20x20_xpm;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"Quite"
#elif defined(PROG_LANGUAGE_FRENCH)
"Enlever"
#elif defined(PROG_LANGUAGE_GERMAN)
"Nehmen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Togliere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Verwijdeer"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Retire"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Fjern"
#else
"Remove"
#endif
			;
		accel_key = 0;
		accel_mods = 0;
		func_cb = EDVDevicesListWinRemoveCB;
		ADD_MENU_ITEM_LABEL
		lw->remove_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		icon = NULL;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"Cambie Arriba"
#elif defined(PROG_LANGUAGE_FRENCH)
"Changer En Haut"
#elif defined(PROG_LANGUAGE_GERMAN)
"Spostare Su"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Spostare Su"
#elif defined(PROG_LANGUAGE_DUTCH)
"Verplaats Op"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Mude Para Cima"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Forskyv Opp"
#else
"Shift Up"
#endif
			;
		accel_key = 0;
		accel_mods = 0;
		func_cb = EDVDevicesListWinUpCB;
		ADD_MENU_ITEM_LABEL
		lw->up_mi = w;

		icon = NULL;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"Cambie Hacia Abajo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Changer En Bas"
#elif defined(PROG_LANGUAGE_GERMAN)
"Spostare Gi"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Spostare Gi"
#elif defined(PROG_LANGUAGE_DUTCH)
"Verplaats Beneden"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Mude Para Baixo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Forskyv Ned"
#else
"Shift Down"
#endif
			;
		accel_key = 0;
		accel_mods = 0;
		func_cb = EDVDevicesListWinDownCB;
		ADD_MENU_ITEM_LABEL
		lw->down_mi = w;

#undef ADD_MENU_ITEM_LABEL
#undef ADD_MENU_ITEM_SEPARATOR
	}


	/* Set initial RC styles */
	if(standard_rcstyle != NULL)
		gtk_widget_modify_style_recursive(
			lw->toplevel, standard_rcstyle
		);
	if(lists_rcstyle != NULL)
		gtk_widget_modify_style_recursive(
			lw->devices_clist, lists_rcstyle
		);



	EDVDevicesListWinUpdateMenus(lw);

	lw->freeze_count--;

	return(lw);
}

/*
 *	Updates menus and other widgets on the given devices list
 *	window to reflect its current data.
 */
void EDVDevicesListWinUpdateMenus(
	edv_devices_list_win_struct *lw
)
{
	gboolean sensitive;
	gint nrows, sel_row, nselected;
	GtkCList *clist;

	if(lw == NULL)
		return;

	lw->freeze_count++;

	/* Devices GtkCList */
	clist = GTK_CLIST(lw->devices_clist);
	nrows = clist->rows;
	nselected = g_list_length(clist->selection);
	sel_row = edv_clist_get_selected_last(clist, NULL);

	/* Add */
	sensitive = (nselected <= 1) ? TRUE : FALSE;
	gtk_widget_set_sensitive(lw->add_btn, sensitive);
	gtk_widget_set_sensitive(lw->add_mi, sensitive);

	/* Edit */
	sensitive = (nselected == 1) ? TRUE : FALSE;
	gtk_widget_set_sensitive(lw->edit_btn, sensitive);
	gtk_widget_set_sensitive(lw->edit_mi, sensitive);

	/* Remove */
	sensitive = (nselected >= 1) ? TRUE : FALSE;
	gtk_widget_set_sensitive(lw->remove_btn, sensitive);
	gtk_widget_set_sensitive(lw->remove_mi, sensitive);

	/* Shift Up */
	sensitive = ((g_list_find(clist->selection, (gpointer)0) == NULL) &&
		(nselected > 0)) ? TRUE : FALSE;
	gtk_widget_set_sensitive(lw->up_btn, sensitive);
	gtk_widget_set_sensitive(lw->up_mi, sensitive);

	/* Shift Down */
	sensitive = ((g_list_find(clist->selection, (gpointer)(nrows - 1)) == NULL) &&
		(nselected > 0)) ? TRUE : FALSE;
	gtk_widget_set_sensitive(lw->down_btn, sensitive);
	gtk_widget_set_sensitive(lw->down_mi, sensitive);

	lw->freeze_count--;
}

/*
 *	Sets the Devices List Window as busy or ready.
 */
void EDVDevicesListSetBusy(
	edv_devices_list_win_struct *lw, const gboolean busy
)
{
	GdkCursor *cursor;
	GtkWidget *w;
	EDVCore *core;

	if(lw == NULL)
		return;

	w = lw->toplevel;
	core = lw->core;
	if((w == NULL) || (core == NULL))
		return;

	if(busy)
	{
		/* Increase busy count */
		lw->busy_count++;

		/* If already busy then don't change anything */
		if(lw->busy_count > 1)
			return;

		cursor = edv_get_cursor(core, EDV_CURSOR_CODE_BUSY);
	}
	else
	{
		/* Reduce busy count */
		lw->busy_count--;
		if(lw->busy_count < 0)
			lw->busy_count = 0;

		/* If still busy do not change anything */
		if(lw->busy_count > 0)
			return;

		cursor = NULL;	/* Use default cursor */
	}

	/* Update toplevel window's cursor */
	if(w->window != NULL)
	{
		gdk_window_set_cursor(w->window, cursor);
		gdk_flush();
	}
}

/*
 *	Checks if the Devices List Window is mapped.
 */
gboolean EDVDevicesListWinIsMapped(
	edv_devices_list_win_struct *lw
)
{
	if(lw == NULL)
		return(FALSE);

	return(GTK_WIDGET_MAPPED(lw->toplevel));
}

/*
 *	Maps the Devices List Window.
 */
void EDVDevicesListWinMap(
	edv_devices_list_win_struct *lw
)
{
	if(lw == NULL)
		return;

	gtk_widget_show_raise(lw->toplevel);
	gtk_widget_grab_focus(lw->devices_clist);
}

/*
 *	Unmaps the Devices List Window.
 */
void EDVDevicesListWinUnmap(
	edv_devices_list_win_struct *lw
)
{
	if(lw == NULL)
		return;

	gtk_widget_hide(lw->toplevel);
}

/*
 *	Deletes the Devices List Window.
 */
void EDVDevicesListWinDelete(
	edv_devices_list_win_struct *lw
)
{
	GtkCList *clist;

	if(lw == NULL)
		return;

	/* Delete the Device Edit dialog */
	EDVDeviceEditDlgDelete(lw->device_edit_dlg);
	lw->device_edit_dlg = NULL;

	EDVDevicesListWinUnmap(lw);

	clist = GTK_CLIST(lw->devices_clist);
	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);
	gtk_clist_thaw(clist);

	lw->freeze_count++;

	gtk_widget_destroy(lw->menu);

	gtk_widget_destroy(lw->toplevel);
	gtk_accel_group_unref(lw->accelgrp);

	lw->freeze_count--;

	g_free(lw);
}
