#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "guiutils.h"
#include "pulist.h"
#include "fprompt.h"
#include "icon_picker.h"
#include "icon_sel_dlg.h"
#include "cdialog.h"
#include "fb.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_fs_type.h"
#include "edv_device.h"
#include "edv_devices_list.h"
#include "libendeavour2-base/edv_utils.h"
#include "device_edit_dlg.h"
#include "endeavour2.h"
#include "edv_emit.h"
#include "edv_utils_gtk.h"
#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_ok_20x20.xpm"
#include "images/icon_select_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"
#include "images/icon_close_20x20.xpm"
#include "images/icon_device_misc_48x48.xpm"


/* Callbacks */
static gint EDVDeviceEditDlgDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void EDVDeviceEditDlgAnyChangedCB(GtkWidget *widget, gpointer data);
static void EDVDeviceEditDlgBrowseDeviceCB(GtkWidget *widget, gpointer data);
static void EDVDeviceEditDlgBrowseMountPathCB(GtkWidget *widget, gpointer data);
static void EDVDeviceEditDlgBrowseProgramCB(GtkWidget *widget, gpointer data);
static void EDVDeviceEditDlgFSTypeChangedCB(
	GtkWidget *widget, const gint i, gpointer data
);

static void EDVDeviceEditDlgIconPickerIconSwitchedCB(
	GtkWidget *w,
	const gint icon_num, IconPickerIcon *icon,
	gpointer data
);

static void EDVDeviceEditDlgOKCB(GtkWidget *widget, gpointer data);
static void EDVDeviceEditDlgApplyCB(GtkWidget *widget, gpointer data);
static void EDVDeviceEditDlgCancelCB(GtkWidget *widget, gpointer data);

void EDVDeviceEditDlgDeviceRemovedCB(
	edv_device_edit_dlg_struct *d,
	const gint dev_num
);

/* Get & Set Values */
void EDVDeviceEditDlgGetValues(
	edv_device_edit_dlg_struct *d,
	const gint dev_num
);
void EDVDeviceEditDlgSetValues(
	edv_device_edit_dlg_struct *d,
	const gint dev_num,
	const gboolean verbose
);

/* Device Edit Dialog */
edv_device_edit_dlg_struct *EDVDeviceEditDlgNew(EDVCore *core);
void EDVDeviceEditDlgResetHasChanges(
	edv_device_edit_dlg_struct *d,
	const gboolean has_changes
);
void EDVDeviceEditDlgUpdateMenus(edv_device_edit_dlg_struct *d);
gboolean EDVDeviceEditDlgIsMapped(edv_device_edit_dlg_struct *d);
void EDVDeviceEditDlgMap(edv_device_edit_dlg_struct *d);
void EDVDeviceEditDlgUnmap(edv_device_edit_dlg_struct *d);
void EDVDeviceEditDlgDelete(edv_device_edit_dlg_struct *d);


#if defined(PROG_LANGUAGE_SPANISH)
# define EDV_DEVICE_EDIT_DLG_TITLE	"El Dispositivo"
#elif defined(PROG_LANGUAGE_FRENCH)
# define EDV_DEVICE_EDIT_DLG_TITLE	"Appareil"
#elif defined(PROG_LANGUAGE_GERMAN)
# define EDV_DEVICE_EDIT_DLG_TITLE	"Vorrichtung"
#elif defined(PROG_LANGUAGE_ITALIAN)
# define EDV_DEVICE_EDIT_DLG_TITLE	"Il Congegno"
#elif defined(PROG_LANGUAGE_DUTCH)
# define EDV_DEVICE_EDIT_DLG_TITLE	"Apparaat"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
# define EDV_DEVICE_EDIT_DLG_TITLE	"O Artifcio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
# define EDV_DEVICE_EDIT_DLG_TITLE	"Innretning"
#else
# define EDV_DEVICE_EDIT_DLG_TITLE	"Device"
#endif

#define EDV_DEVICE_EDIT_DLG_WIDTH	500
#define EDV_DEVICE_EDIT_DLG_HEIGHT	-1


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	GtkWindow "delete_event" signal callback.
 */
static gint EDVDeviceEditDlgDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	edv_device_edit_dlg_struct *d = EDV_DEVICE_EDIT_DLG(data);
	if(d == NULL)
	    return(TRUE);

	EDVDeviceEditDlgCancelCB(widget, d);

	return(TRUE);
}

/*
 *	GtkWidget any "changed" signal callback.
 */
static void EDVDeviceEditDlgAnyChangedCB(GtkWidget *widget, gpointer data)
{
	edv_device_edit_dlg_struct *d = EDV_DEVICE_EDIT_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;

	if(!d->has_changes)
	    EDVDeviceEditDlgResetHasChanges(d, TRUE);

	d->freeze_count--;
}


/*
 *	Devices Edit Window browse device callback.
 */
static void EDVDeviceEditDlgBrowseDeviceCB(GtkWidget *widget, gpointer data)
{
	gboolean response;
	GtkWidget *toplevel;
	gint nftypes = 0, npaths = 0;
	gchar **paths_list = NULL;
	fb_type_struct **ftypes_list = NULL, *ftype_rtn = NULL;
	GtkEntry *entry = (GtkEntry *)data;
	if((entry == NULL) || FileBrowserIsQuery())
	    return;

	toplevel = gtk_widget_get_toplevel(GTK_WIDGET(entry));

	/* Create the file types list */
	FileBrowserTypeListNew(
	    &ftypes_list, &nftypes,
	    "*.*", "All Files"
	);

	/* Query the user for the device */
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
	    "Select Device",
	    "Select", "Cancel",
	    gtk_entry_get_text(entry),		/* Startup path */
	    ftypes_list, nftypes,
	    &paths_list, &npaths,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(response)
	{
	    const gchar *path = (npaths > 0) ?
		paths_list[0] : NULL;
	    if(!STRISEMPTY(path))
	    {
		gtk_entry_set_text(entry, path);
		gtk_entry_set_position(entry, -1);
	    }
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);
}

/*
 *	Devices Edit Window browse mount path callback.
 */
static void EDVDeviceEditDlgBrowseMountPathCB(GtkWidget *widget, gpointer data)
{
	gboolean response;
	GtkWidget *toplevel;
	gint nftypes = 0, npaths = 0;
	gchar **paths_list = NULL;
	fb_type_struct **ftypes_list = NULL, *ftype_rtn = NULL;
	GtkEntry *entry = (GtkEntry *)data;
	if((entry == NULL) || FileBrowserIsQuery())
	    return;

	toplevel = gtk_widget_get_toplevel(GTK_WIDGET(entry));

	/* Create the file types list */
	FileBrowserTypeListNew(
	    &ftypes_list, &nftypes,
	    "*.*", "All Files"
	);

	/* Query the user for the mount path */
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
	    "Select Mount Path",
	    "Select", "Cancel",
	    gtk_entry_get_text(entry),          /* Startup path */
	    ftypes_list, nftypes,
	    &paths_list, &npaths,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(response)
	{
	    const gchar *path = (npaths > 0) ?
		paths_list[0] : NULL;
	    if(!STRISEMPTY(path))
	    {
		gtk_entry_set_text(entry, path);
		gtk_entry_set_position(entry, -1);
	    }
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);
}

/*
 *	Devices Edit Window browse program callback.
 */
static void EDVDeviceEditDlgBrowseProgramCB(GtkWidget *widget, gpointer data)
{
	gboolean response;
	GtkWidget *toplevel;
	gint nftypes = 0, npaths = 0;
	gchar **paths_list = NULL;
	fb_type_struct **ftypes_list = NULL, *ftype_rtn = NULL;
	GtkEntry *entry = (GtkEntry *)data;
	if((entry == NULL) || FileBrowserIsQuery())
	    return;

	toplevel = gtk_widget_get_toplevel(GTK_WIDGET(entry));

	/* Create the file types list */
	FileBrowserTypeListNew(
	    &ftypes_list, &nftypes,
	    "*.*", "All Files"
	);

	/* Query the user for the program */
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
	    "Select Program",
	    "Select", "Cancel",
	    gtk_entry_get_text(entry),          /* Startup path */
	    ftypes_list, nftypes,
	    &paths_list, &npaths,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(response)
	{
	    const gchar *path = (npaths > 0) ?
		paths_list[0] : NULL;
	    if(!STRISEMPTY(path))
	    {
		gtk_entry_set_text(entry, path);
		gtk_entry_set_position(entry, -1);
	    }
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);
}

/*
 *	Filesystem type popup list box changed signal callback.
 */
static void EDVDeviceEditDlgFSTypeChangedCB(
	GtkWidget *widget, const gint i, gpointer data
)
{
	EDVDeviceEditDlgAnyChangedCB(widget, data);
}


/*
 *      Icon Picker icon switched signal callback.
 */
static void EDVDeviceEditDlgIconPickerIconSwitchedCB(
	GtkWidget *w,
	const gint icon_num, IconPickerIcon *icon,
	gpointer data
)
{
	edv_device_edit_dlg_struct *d = EDV_DEVICE_EDIT_DLG(data);
	if((w == NULL) || (d == NULL))
	    return;

}


/*
 *	OK callback.
 */
static void EDVDeviceEditDlgOKCB(GtkWidget *widget, gpointer data)
{
	edv_device_edit_dlg_struct *d = EDV_DEVICE_EDIT_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	EDVDeviceEditDlgSetValues(
	    d,
	    d->dev_num,
	    TRUE					/* Verbose */
	);

	EDVDeviceEditDlgUnmap(d);
}

/*
 *	Apply callback.
 */
static void EDVDeviceEditDlgApplyCB(GtkWidget *widget, gpointer data)
{
	edv_device_edit_dlg_struct *d = EDV_DEVICE_EDIT_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	EDVDeviceEditDlgSetValues(
	    d,
	    d->dev_num,
	    TRUE					/* Verbose */
	);
}

/*
 *	Devices Edit Window cancel or close callback.
 */
static void EDVDeviceEditDlgCancelCB(GtkWidget *widget, gpointer data)
{
	edv_device_edit_dlg_struct *d = EDV_DEVICE_EDIT_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	EDVDeviceEditDlgUnmap(d);
}


/*
 *	Device removed signal callback.
 */
void EDVDeviceEditDlgDeviceRemovedCB(
	edv_device_edit_dlg_struct *d,
	const gint dev_num
)
{
	if(d == NULL)
	    return;

	if(d->dev_num != dev_num)
	    return;

	d->dev_num = -1;
	EDVDeviceEditDlgUnmap(d);
}


/*
 *	Gets the values from the Device and sets them to the dialog's
 *	widgets.
 *
 *      The dev_num specifies the Device.
 */
void EDVDeviceEditDlgGetValues(
	edv_device_edit_dlg_struct *d,
	const gint dev_num
)
{
	const gint nicon_states = EDV_DEVICE_TOTAL_ICON_STATES;
	gint i;
	const gchar *icon_state_names[] = EDV_DEVICE_ICON_STATE_NAMES;
	GList *icons_list;
	GtkWidget *w;
	IconPickerIcon *icon;
	EDVPixmap *p;
	EDVDevice *dev;
	EDVCore *core;

	if(d == NULL)
	    return;

	core = d->core;

	/* Get the Device */
	dev = EDV_DEVICE(g_list_nth_data(
	    core->devices_list,
	    (guint)dev_num
	));
	if(dev == NULL)
	    return;

	d->dev_num = dev_num;

	/* Name */
	gtk_entry_set_text(
	    GTK_ENTRY(d->name_entry),
	    (dev->name != NULL) ? dev->name : ""
	);

	/* Device Path */
	gtk_entry_set_text(
	    GTK_ENTRY(d->device_path_entry),
	    (dev->device_path != NULL) ? dev->device_path : ""
	);

	/* Mount Path */
	gtk_entry_set_text(
	    GTK_ENTRY(d->mount_path_entry),
	    (dev->mount_path != NULL) ? dev->mount_path : ""
	);

	/* File System Type */
	if(STRISEMPTY(dev->fs_type_name))
	{
	    PUListBoxSelect(d->fs_type_pulistbox, 0);
	}
	else
	{
	    GtkWidget *pulist = PUListBoxGetPUList(d->fs_type_pulistbox);
	    const gint m = PUListGetTotalItems(pulist);
	    gint i;
	    const gchar *name = dev->fs_type_name;
	    EDVFSType *fs_type;

	    for(i = 0; i < m; i++)
	    {
		fs_type = EDV_FS_TYPE(PUListGetItemData(pulist, i));
		if(fs_type == NULL)
		    continue;

		if(fs_type->name == NULL)
		    continue;

		if(!g_strcasecmp(name, fs_type->name))
		{
		    PUListBoxSelect(d->fs_type_pulistbox, i);
		    break;
		}
	    }
	}


	/* No Unmount */
	GTK_TOGGLE_BUTTON_SET_ACTIVE(
	    d->no_unmount_check,
	    EDV_DEVICE_IS_NO_UNMOUNT(dev)
	);
	/* Read-Only */
	GTK_TOGGLE_BUTTON_SET_ACTIVE(
	    d->read_only_check,
	    EDV_DEVICE_IS_READ_ONLY(dev)
	);
	/* Unlisted */
	GTK_TOGGLE_BUTTON_SET_ACTIVE(
	    d->unlisted_check,
	    EDV_DEVICE_IS_UNLISTED(dev)
	);
	/* No Scan */
	GTK_TOGGLE_BUTTON_SET_ACTIVE(
	    d->no_scan_check,
	    EDV_DEVICE_IS_NO_SCAN(dev)
	);


	/* Mount Command */
	gtk_entry_set_text(
	    GTK_ENTRY(d->command_mount_entry),
	    (dev->command_mount != NULL) ? dev->command_mount : ""
	);

	/* Unmount Command */
	gtk_entry_set_text(
	    GTK_ENTRY(d->command_unmount_entry),
	    (dev->command_unmount != NULL) ? dev->command_unmount : ""
	);

	/* Eject Command */
	gtk_entry_set_text(
	    GTK_ENTRY(d->command_eject_entry),
	    (dev->command_eject != NULL) ? dev->command_eject : ""
	);

	/* Check Command */
	gtk_entry_set_text(
	    GTK_ENTRY(d->command_check_entry),
	    (dev->command_check != NULL) ? dev->command_check : ""
	);

	/* Tools Command */
	gtk_entry_set_text(
	    GTK_ENTRY(d->command_tools_entry),
	    (dev->command_tools != NULL) ? dev->command_tools : ""
	);

	/* Format Command */
	gtk_entry_set_text(
	    GTK_ENTRY(d->command_format_entry),
	    (dev->command_format != NULL) ? dev->command_format : ""
	);


	/* Small icons */
	w = d->icon_small_ip;
	icons_list = NULL;
	for(i = 0; i < nicon_states; i++)
	{
	    icon = icon_picker_icon_new();
	    if(icon == NULL)
		break;

	    icon->label = STRDUP(icon_state_names[i]);
	    icon->path = STRDUP(dev->small_icon_path[i]);
	    p = edv_pixmap_ref(dev->small_icon[i]);
	    if(edv_pixmap_is_loaded(p))
	    {
		icon->pixmap = GDK_PIXMAP_REF(p->pixmap);
		icon->mask = GDK_BITMAP_REF(p->mask);
		icon->width = p->width;
		icon->height = p->height;
	    }
	    (void)edv_pixmap_unref(p);
	    icons_list = g_list_append(
		icons_list,
		icon
	    );
	}
	icon_picker_set_icons_list(w, icons_list);
	icon_picker_set_has_changes(w, FALSE);

	/* Medium icons */
	w = d->icon_medium_ip;
	icons_list = NULL;
	for(i = 0; i < nicon_states; i++)
	{
	    icon = icon_picker_icon_new();
	    if(icon == NULL)
		break;

	    icon->label = STRDUP(icon_state_names[i]);
	    icon->path = STRDUP(dev->medium_icon_path[i]);
	    p = edv_pixmap_ref(dev->medium_icon[i]);
	    if(edv_pixmap_is_loaded(p))
	    {
		icon->pixmap = GDK_PIXMAP_REF(p->pixmap);
		icon->mask = GDK_BITMAP_REF(p->mask);
		icon->width = p->width;
		icon->height = p->height;
	    }
	    (void)edv_pixmap_unref(p);
	    icons_list = g_list_append(
		icons_list,
		icon
	    );
	}
	icon_picker_set_icons_list(w, icons_list);
	icon_picker_set_has_changes(w, FALSE);

	/* Large icons */
	w = d->icon_large_ip;
	icons_list = NULL;
	for(i = 0; i < nicon_states; i++)
	{
	    icon = icon_picker_icon_new();
	    if(icon == NULL)
		break;

	    icon->label = STRDUP(icon_state_names[i]);
	    icon->path = STRDUP(dev->large_icon_path[i]);
	    p = edv_pixmap_ref(dev->large_icon[i]);
	    if(edv_pixmap_is_loaded(p))
	    {
		icon->pixmap = GDK_PIXMAP_REF(p->pixmap);
		icon->mask = GDK_BITMAP_REF(p->mask);
		icon->width = p->width;
		icon->height = p->height;
	    }
	    (void)edv_pixmap_unref(p);
	    icons_list = g_list_append(
		icons_list,
		icon
	    );
	}
	icon_picker_set_icons_list(w, icons_list);
	icon_picker_set_has_changes(w, FALSE);


	EDVDeviceEditDlgUpdateMenus(d);
}

/*
 *	Sets the values from the dialog's widgets to the Device.
 *
 *	The dev_num specifies the Device.
 */
void EDVDeviceEditDlgSetValues(
	edv_device_edit_dlg_struct *d,
	const gint dev_num,
	const gboolean verbose
)
{
	const gint nicon_states = EDV_DEVICE_TOTAL_ICON_STATES;
	GList *icons_list;
	GtkWidget *toplevel;
	EDVFSType *fs_type;
	EDVDevice *dev;
	EDVCore *core;

	if(d == NULL)
	    return;

	toplevel = d->toplevel;
	core = d->core;

	/* Get the Device */
	dev = EDV_DEVICE(g_list_nth_data(
	    core->devices_list,
	    (guint)dev_num
	));
	if(dev == NULL)
	    return;

	/* Name */
	g_free(dev->name);
	dev->name = STRDUP(gtk_entry_get_text(
	    GTK_ENTRY(d->name_entry)
	));

	/* Device Path */
	g_free(dev->device_path);
	dev->device_path = STRDUP(gtk_entry_get_text(
	    GTK_ENTRY(d->device_path_entry)
	));

	/* Mount Path */
	g_free(dev->mount_path);
	dev->mount_path = STRDUP(gtk_entry_get_text(
	    GTK_ENTRY(d->mount_path_entry)
	));

	/* File system type */
	fs_type = EDV_FS_TYPE(PUListBoxGetSelectedData(d->fs_type_pulistbox));
	if(fs_type != NULL)
	{
	    g_free(dev->fs_type_name);
	    dev->fs_type_name = STRDUP(fs_type->name);
	    dev->fs_type_code = fs_type->code;
	}
	else
	{
	    g_free(dev->fs_type_name);
	    dev->fs_type_name = NULL;
	    dev->fs_type_code = 0;
	}


	/* No Unmount */
	if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->no_unmount_check))
	    dev->flags |= EDV_DEVICE_NO_UNMOUNT;
	else
	    dev->flags &= ~EDV_DEVICE_NO_UNMOUNT;
	/* Read-Only */
	if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->read_only_check))
	    dev->flags |= EDV_DEVICE_READ_ONLY;
	else
	    dev->flags &= ~EDV_DEVICE_READ_ONLY;
	/* Unlisted */
	if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->unlisted_check))
	    dev->flags |= EDV_DEVICE_UNLISTED;
	else
	    dev->flags &= ~EDV_DEVICE_UNLISTED;
	/* No Scan */
	if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->no_scan_check))
	    dev->flags |= EDV_DEVICE_NO_SCAN;
	else
	    dev->flags &= ~EDV_DEVICE_NO_SCAN;

	/* Mount command */
	g_free(dev->command_mount);
	dev->command_mount = STRDUP(gtk_entry_get_text(
	    GTK_ENTRY(d->command_mount_entry)
	));

	/* Unmount command */
	g_free(dev->command_unmount);
	dev->command_unmount = STRDUP(gtk_entry_get_text(
	    GTK_ENTRY(d->command_unmount_entry)
	));

	/* Eject command */
	g_free(dev->command_eject);
	dev->command_eject = STRDUP(gtk_entry_get_text(
	    GTK_ENTRY(d->command_eject_entry)
	));

	/* Check command */
	g_free(dev->command_check);
	dev->command_check = STRDUP(gtk_entry_get_text(
	    GTK_ENTRY(d->command_check_entry)
	));

	/* Tools command */
	g_free(dev->command_tools);
	dev->command_tools = STRDUP(gtk_entry_get_text(
	    GTK_ENTRY(d->command_tools_entry)
	));

	/* Format command */
	g_free(dev->command_format);
	dev->command_format = STRDUP(gtk_entry_get_text(
	    GTK_ENTRY(d->command_format_entry)
	));


	/* Small icon */
	icons_list = icon_picker_get_icons_list(d->icon_small_ip);
	if(icons_list != NULL)
	{
	    gint i;
	    GList *glist;
	    IconPickerIcon *icon;

	    for(glist = icons_list, i = 0;
		(glist != NULL) && (i < nicon_states);
		glist = g_list_next(glist), i++
	    )
	    {
		icon = ICON_PICKER_ICON(glist->data);
		if(icon == NULL)
		    continue;

		g_free(dev->small_icon_path[i]);
		dev->small_icon_path[i] = STRDUP(icon->path);
	    }
	}

	/* Medium icon */
	icons_list = icon_picker_get_icons_list(d->icon_medium_ip);
	if(icons_list != NULL)
	{
	    gint i;
	    GList *glist;
	    IconPickerIcon *icon;

	    for(glist = icons_list, i = 0;
		(glist != NULL) && (i < nicon_states);
		glist = g_list_next(glist), i++
	    )
	    {
		icon = ICON_PICKER_ICON(glist->data);
		if(icon == NULL)
		    continue;

		g_free(dev->medium_icon_path[i]);
		dev->medium_icon_path[i] = STRDUP(icon->path);
	    }
	}

	/* Large icon */
	icons_list = icon_picker_get_icons_list(d->icon_large_ip);
	if(icons_list != NULL)
	{
	    gint i;
	    GList *glist;
	    IconPickerIcon *icon;

	    for(glist = icons_list, i = 0;
		(glist != NULL) && (i < nicon_states);
		glist = g_list_next(glist), i++
	    )
	    {
		icon = ICON_PICKER_ICON(glist->data);
		if(icon == NULL)
		    continue;

		g_free(dev->large_icon_path[i]);
		dev->large_icon_path[i] = STRDUP(icon->path);
	    }
	}


	/* Clear the has changes flag */
	EDVDeviceEditDlgResetHasChanges(d, FALSE);


	/* Re-realize this Device */
	edv_device_realize(dev, TRUE);

	/* Update all device mount states and stats */
	edv_devices_list_update_mount_states(core->devices_list);
	edv_devices_list_update_statistics(core->devices_list);

	/* Notify about this Device being modified */
	edv_emit_device_modified(core, dev_num, dev);
}


/*
 *	Creates a new Devices Edit Window.
 */
edv_device_edit_dlg_struct *EDVDeviceEditDlgNew(EDVCore *core)
{
	const gint	border_major = 5,
			border_minor = 2;
	gulong load_int;
	gchar *default_icons_path;
	gpointer entry_rtn, browse_rtn;
	GdkWindow *window;
	GtkAccelGroup *accelgrp;
	GtkWidget	*w,
			*parent, *parent2, *parent3, *parent4,
			*parent5, *parent6,
			*toplevel, *icon_sel_dlg;
	CfgList *cfg_list;
	GtkWidget *pulist;
	IconPickerFlags ip_flags;
	EDVListsPointerOpButton2 lists_pointer_op_button2;
	edv_device_edit_dlg_struct *d;

	if(core == NULL)
	    return(NULL);

	cfg_list = core->cfg_list;

	/* Create a new Device Edit Dialog */
	d = EDV_DEVICE_EDIT_DLG(g_malloc0(sizeof(edv_device_edit_dlg_struct)));
	if(d == NULL)
	    return(NULL);

	d->toplevel = toplevel = gtk_window_new(GTK_WINDOW_DIALOG);
	d->accelgrp = accelgrp = gtk_accel_group_new();
	d->busy_count = 0;
	d->freeze_count = 0;
	d->has_changes = 0;
	d->core = core;
	d->dev_num = -1;

	d->freeze_count++;

	/* Toplevel GtkWindow */
	w = toplevel;
	gtk_window_set_wmclass(
	    GTK_WINDOW(w), "dialog", PROG_NAME
	);
	gtk_window_set_policy(GTK_WINDOW(w), FALSE, FALSE, FALSE);
	gtk_widget_set_usize(w, EDV_DEVICE_EDIT_DLG_WIDTH, EDV_DEVICE_EDIT_DLG_HEIGHT);
	gtk_window_set_title(GTK_WINDOW(w), EDV_DEVICE_EDIT_DLG_TITLE);
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
	    gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_TITLE | GDK_DECOR_MENU |
		GDK_DECOR_MINIMIZE
	    );
	    gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_MINIMIZE | GDK_FUNC_CLOSE
	    );
	    GUISetWMIcon(window, (guint8 **)icon_device_misc_48x48_xpm);
	}
	gtk_widget_add_events(
	    w,
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(EDVDeviceEditDlgDeleteEventCB), d
	);
	gtk_container_border_width(GTK_CONTAINER(w), 0);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	parent = w;

	/* Main GtkVBox */
	d->main_vbox = w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;


	/* GtkHBox to make two columns */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;


	/* Left side GtkVBox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent3 = w;


	/* Identification GtkFrame */
	w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Identificacin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Identification"
#elif defined(PROG_LANGUAGE_GERMAN)
"Identifikation"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Identificazione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Identificatie"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"A Identificao"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Identification"
#else
"Identification"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	/* Name GtkEntry */
	w = GUIPromptBar(
	    NULL,
#if defined(PROG_LANGUAGE_SPANISH)
"El Nombre"
#elif defined(PROG_LANGUAGE_FRENCH)
"Nom"
#elif defined(PROG_LANGUAGE_GERMAN)
"Name"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il Nome"
#elif defined(PROG_LANGUAGE_DUTCH)
"Naam"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Nome"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Navn"
#else
"Name"
#endif
	    ":", NULL, &entry_rtn
	);
	d->name_entry = (GtkWidget *)entry_rtn;
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	w = (GtkWidget *)entry_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(EDVDeviceEditDlgAnyChangedCB), d
	);
	edv_entry_set_dnd(core, w);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(
	    w,
#if defined(PROG_LANGUAGE_SPANISH)
"Entre el nombre de este artefacto (este valor hace slo sea usado\
 para propsitos de despliegue)"
#elif defined(PROG_LANGUAGE_FRENCH)
"Entrer le nom de cet appareil (cette valeur sera seulement utilise\
 pour les buts d'exposition)"
#elif defined(PROG_LANGUAGE_GERMAN)
"Tragen sie den namen dieser vorrichtung ein (dieser wert wird nur\
 fr ausstellung zwecke benutzt werden)"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Entrare il nome di questo congegno (questo valore sar soltanto usato\
 per gli scopi di mostra)"
#elif defined(PROG_LANGUAGE_DUTCH)
"Ga de (deze enige waarde zal voor tentoonstelling doelen gebruikt\
 worden) naam van deze apparaat binnen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Entre o nome deste artifcio (este valor s ser usado para propsitos\
 de exposio)"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"G inn i navnet av denne innretningen (denne verdien brukt bare for\
 utstillingsforml)"
#else
"Enter the name of this device (this value will only be used for\
 display purposes)"
#endif
	);

	/* Device Path GtkEntry */
	w = GUIPromptBarWithBrowse(
	    NULL,
#if defined(PROG_LANGUAGE_SPANISH)
"Sendero Dispositivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Sentier D'appareil"
#elif defined(PROG_LANGUAGE_GERMAN)
"Gertepfad"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Sentiero Congegno"
#elif defined(PROG_LANGUAGE_DUTCH)
"Apparaat Pad"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Caminho Artifcio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Innretnings Sti"
#else
"Device Path"
#endif
	    ":", NULL, &entry_rtn, &browse_rtn,
	    NULL, NULL
	);
	d->device_path_entry = (GtkWidget *)entry_rtn;
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	w = (GtkWidget *)entry_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(EDVDeviceEditDlgAnyChangedCB), d
	);
	edv_entry_set_dnd(core, w);
	edv_entry_set_complete_path(core, w);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(
	    w,
#if defined(PROG_LANGUAGE_SPANISH)
"Entre el sendero al objeto del artefacto"
#elif defined(PROG_LANGUAGE_FRENCH)
"Le objeto d'al de sendero d'el de Entre del artefacto"
#elif defined(PROG_LANGUAGE_GERMAN)
"Tragen sie den pfad zum gerteobjekt ein"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Entrare il sentiero all'oggetto di congegno"
#elif defined(PROG_LANGUAGE_DUTCH)
"Ga het pad aan het apparaat voorwerp binnen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Entre o caminho ao objeto de artifcio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"G inn i stien til innretnings objektet"
#else
"Enter the full path to this device's device node."
#endif
	);

	w = (GtkWidget *)browse_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVDeviceEditDlgBrowseDeviceCB), entry_rtn
	);

	/* Mount Path GtkEntry */
	w = GUIPromptBarWithBrowse(
	    NULL,
#if defined(PROG_LANGUAGE_SPANISH)
"Monte Sendero"
#elif defined(PROG_LANGUAGE_FRENCH)
"Monter Sentier"
#elif defined(PROG_LANGUAGE_GERMAN)
"Stellen Pfad"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Montare Sentiero"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bestijg Pad"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Monte Caminho"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Monter Sti"
#else
"Mount Path"
#endif
	    ":", NULL, &entry_rtn, &browse_rtn,
	    NULL, NULL
	);
	d->mount_path_entry = (GtkWidget *)entry_rtn;
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	w = (GtkWidget *)entry_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(EDVDeviceEditDlgAnyChangedCB), d
	);
	edv_entry_set_dnd(core, w);
	edv_entry_set_complete_path(core, w);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(
	    w,
#if defined(PROG_LANGUAGE_SPANISH)
"Entre el sendero al objeto de la gua que este artefacto\
 deber ser montado en"
#elif defined(PROG_LANGUAGE_FRENCH)
"Entrer le sentier  l'objet d'annuaire que cet appareil va\
 tre mont sur"
#elif defined(PROG_LANGUAGE_GERMAN)
"Tragen sie den pfad zum verzeichnis objekt, das diese vorrichtung auf\
 aufgestellt werden soll ein"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Entrare il sentiero all'oggetto di elenco che questo congegno \
 essere montato su"
#elif defined(PROG_LANGUAGE_DUTCH)
"Ga het pad aan het gids voorwerp binnen dat deze apparaat op bestegen\
 worden zal"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Entre o caminho ao objeto de guia que este artifcio  ser montado\
 em"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"G inn i stien til katalogobjektet som denne innretningen er montert\
 p"
#else
"Enter the full path to the directory that this device is to\
 be mounted on."
#endif
	);

	w = (GtkWidget *)browse_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVDeviceEditDlgBrowseMountPathCB), entry_rtn
	);


	/* Filesystem GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent6 = w;

	/* Label */
	w = gtk_label_new("Filesystem Type:");
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Filesystem Popup List Box */
	d->fs_type_pulistbox = w = PUListBoxNew(
	    -1, -1
	);
        gtk_box_pack_start(GTK_BOX(parent6), w, TRUE, TRUE, 0);
	pulist = PUListBoxGetPUList(w);
	if(core->run_flags & EDV_RUN_SAFE_MODE)
	    PUListSetShadowStyle(pulist, PULIST_SHADOW_NONE);
	PUListBoxSetChangedCB(
	    w,
	    EDVDeviceEditDlgFSTypeChangedCB, d
	);
	if(pulist != NULL)
	{
	    gint i = 0;
	    GList	*glist,
			*fs_types_list = edv_fs_types_list_get_from_system();
	    EDVFSType *fs_type;

	    for(glist = fs_types_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		fs_type = EDV_FS_TYPE(glist->data);
		if(fs_type == NULL)
		    continue;

		if(STRISEMPTY(fs_type->name))
		    continue;

		i = PUListAddItem(
		    pulist,
		    fs_type->name
		);
		if(i > -1)
		{
		    PUListSetItemDataFull(
			pulist,
			i,
			edv_fs_type_copy(fs_type),
			(GtkDestroyNotify)edv_fs_type_delete
		    );
		}
	    }
	    PUListBoxSetLinesVisible(
		w,
		CLIP((i + 1), 1, 15)
	    );

	    edv_fs_types_list_delete(fs_types_list);
	}
	PUListBoxSetTip(
	    w,
#if defined(PROG_LANGUAGE_SPANISH)
"Escoja el de tipo filesystem de los medios usados con este\
 artefacto"
#elif defined(PROG_LANGUAGE_FRENCH)
"Choisir le type de filesystem du presse utilis avec cet\
 appareil"
#elif defined(PROG_LANGUAGE_GERMAN)
"Whlen sie den filesystem typ der medien hat benutzt mit dieser\
 vorrichtung aus"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scegliere il tipo di filesystem della stampa usata con questo\
 congegno"
#elif defined(PROG_LANGUAGE_DUTCH)
"Selecteer het filesystem type van de media gebruikte met deze\
 apparaat"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Selecione o filesystem tipo da imprensa usada com este\
 artifcio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Velg ut filesystem type av mediene som brukt med denne\
 innretningen"
#else
"Select the filesystem type of the media used with this device."
#endif
	);
	gtk_widget_show(w);


	/* Options GtkFrame */
	w = gtk_frame_new(
#ifdef PROG_LANGUAGE_ENGLISH
	    "Options"
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Las Opciones"
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Options"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	/* No Unmount GtkCheckButton */
	d->no_unmount_check = w = gtk_check_button_new_with_label(
#if defined(PROG_LANGUAGE_SPANISH)
	    "Ningn Unmount"
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Aucun Unmount"
#else
	    "No Unmount"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(EDVDeviceEditDlgAnyChangedCB), d
	);
	GUISetWidgetTip(
	    w,
"Check this to mark this device as not unmountable, so that\
 " PROG_NAME " will not make mount or unmount functions available\
 for this device."
	);
	gtk_widget_show(w);

	/* Read-Only GtkCheckButton */
	d->read_only_check = w = gtk_check_button_new_with_label(
#if defined(PROG_LANGUAGE_SPANISH)
	    "Lea Slo"
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Lire Seulement"
#else
	    "Read-Only"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(EDVDeviceEditDlgAnyChangedCB), d
	);
	GUISetWidgetTip(
	    w,
"Check this to always mount this device as read-only."
	);
	gtk_widget_show(w);


	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	/* Unlisted GtkCheckButton */
	d->unlisted_check = w = gtk_check_button_new_with_label(
#if defined(PROG_LANGUAGE_SPANISH)
	    "Unlisted"
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Unlisted"
#else
	    "Unlisted"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(EDVDeviceEditDlgAnyChangedCB), d
	);
	GUISetWidgetTip(
	    w,
"Check this to omit the listing of this device in \"brief\" listings\
 (such as on the mount bars)"
	);
	gtk_widget_show(w);

	/* No Scan GtkCheckButton */
	d->no_scan_check = w = gtk_check_button_new_with_label(
	    "No Scan"
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(EDVDeviceEditDlgAnyChangedCB), d
	);
	GUISetWidgetTip(
	    w,
"Check this to avoid scanning this device's mount path for objects\
 unless you explicitly select it (useful for\
 conserving bandwidth on NFS filesystems)"
	);
	gtk_widget_show(w);


	/* Commands GtkFrame */
	w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Las rdenes"
#elif defined(PROG_LANGUAGE_FRENCH)
"Ordres"
#elif defined(PROG_LANGUAGE_GERMAN)
"Befehle"
#elif defined(PROG_LANGUAGE_ITALIAN)
"I Comandi"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bevelen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Os Comandos"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kommandoer"
#else
"Commands"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	/* Mount Command GtkEntry */
	w = GUIPromptBarWithBrowse(
	    NULL,
#if defined(PROG_LANGUAGE_SPANISH)
"El Monte"
#elif defined(PROG_LANGUAGE_FRENCH)
"Mont"
#elif defined(PROG_LANGUAGE_GERMAN)
"Untersatz"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il Monte"
#elif defined(PROG_LANGUAGE_DUTCH)
"Berg"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Monte"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Fotstykke"
#else
"Mount"
#endif
	    ":", NULL, &entry_rtn, &browse_rtn,
	    NULL, NULL
	);
	d->command_mount_entry = (GtkWidget *)entry_rtn;
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	w = (GtkWidget *)entry_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(EDVDeviceEditDlgAnyChangedCB), d
	);
	edv_entry_set_dnd(core, w);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(
	    w,
#ifdef PROG_LANGUAGE_ENGLISH
"Enter the command that will be executed to mount\
 this device (or leave it blank if it is not mountable)"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Entre la orden que ser ejecutada al mount este\
 artefacto (o le sale blanco si no es mountable)"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Entrer l'ordre qui sera excut  moun cet\
 appareil (ou le part le vide si ne ce pas est mountable)"
#endif
	);

	w = (GtkWidget *)browse_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVDeviceEditDlgBrowseProgramCB), entry_rtn
	);

	/* Unmount Command GtkEntry */
	w = GUIPromptBarWithBrowse(
	    NULL,
#ifdef PROG_LANGUAGE_ENGLISH
	    "Unmount:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "El Unmoute:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Unmont:",
#endif
	    NULL, &entry_rtn, &browse_rtn,
	    NULL, NULL
	);
	d->command_unmount_entry = (GtkWidget *)entry_rtn;
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	w = (GtkWidget *)entry_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(EDVDeviceEditDlgAnyChangedCB), d
	);
	edv_entry_set_dnd(core, w);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(
	    w,
#ifdef PROG_LANGUAGE_ENGLISH
	    "Enter the command that will be executed to unmount\
 this device (or leave it blank if it is not unmountable)"
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Entre la orden que ser ejecutada al unmount este\
 artefacto (o le sale blanco si no es unmountable)"
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Entrer l'ordre qui sera excut  unmount cet\
 appareil (ou le part le vide si ne ce pas est unmountable)"
#endif
	);

	w = (GtkWidget *)browse_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVDeviceEditDlgBrowseProgramCB), entry_rtn
	);

	/* Eject Command GtkEntry */
	w = GUIPromptBarWithBrowse(
	    NULL,
#ifdef PROG_LANGUAGE_ENGLISH
	    "Eject:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Expulse:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Ejecter:",
#endif
	    NULL, &entry_rtn, &browse_rtn,
	    NULL, NULL
	);
	d->command_eject_entry = (GtkWidget *)entry_rtn;
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	w = (GtkWidget *)entry_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(EDVDeviceEditDlgAnyChangedCB), d
	);
	edv_entry_set_dnd(core, w);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(
	    w,
#ifdef PROG_LANGUAGE_ENGLISH
	    "Enter the command that will be executed to eject\
 the media from the device (or leave it blank if the device does not\
 support ejecting)"
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Entre la orden que ser ejecutada a eject los\
 medios del artefacto (o le sale blanco si el artefacto hace\
 expulsar de apoyo de not)"
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Entrer l'ordre qui sera excut  eject le presse\
 de l'appareil (ou le part le vide si l'appareil fait\
 l'jecter de soutien de not)"
#endif
	);

	w = (GtkWidget *)browse_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVDeviceEditDlgBrowseProgramCB), entry_rtn
	);


	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	/* Check Command GtkEntry */
	w = GUIPromptBarWithBrowse(
	    NULL,
#ifdef PROG_LANGUAGE_ENGLISH
	    "Check:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Cheque:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Chque:",
#endif
	    NULL, &entry_rtn, &browse_rtn,
	    NULL, NULL
	);
	d->command_check_entry = (GtkWidget *)entry_rtn;
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	w = (GtkWidget *)entry_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(EDVDeviceEditDlgAnyChangedCB), d
	);
	edv_entry_set_dnd(core, w);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(
	    w,
#ifdef PROG_LANGUAGE_ENGLISH
	    "Enter the command that will be executed to run the\
 file system check program for this device (or leave it blank if\
 you do not have one)"
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Entre la orden que se ejecutar para correr el\
 programa de cheque de sistema de archivo para este artefacto\
 (o le sale blanco si usted no tiene uno)"
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Entrez l'ordre qui sera excut courir le\
 programme de chque de systme de fichier pour cet appareil\
 (ou le part le vide si vous ne celui avez pas)"
#endif
	);

	w = (GtkWidget *)browse_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVDeviceEditDlgBrowseProgramCB), entry_rtn
	);

	/* Tools Command GtkEntry */
	w = GUIPromptBarWithBrowse(
	    NULL,
#ifdef PROG_LANGUAGE_ENGLISH
	    "Tools:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Las Herramientas:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Outils:",
#endif
	    NULL, &entry_rtn, &browse_rtn,
	    NULL, NULL
	);
	d->command_tools_entry = (GtkWidget *)entry_rtn;
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	w = (GtkWidget *)entry_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(EDVDeviceEditDlgAnyChangedCB), d
	);
	edv_entry_set_dnd(core, w);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(
	    w,
#ifdef PROG_LANGUAGE_ENGLISH
	    "Enter the command that will be executed to run the\
 tools program for this device (or leave it blank if you do not\
 have one)"
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Entre la orden que se ejecutar para correr el\
 programa de herramientas para este artefacto (o le sale\
 blanco si usted no tiene uno)"
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Entrer l'ordre qui sera excut courir le\
 programme d'outils pour cet appareil (ou le part le vide\
 si vous ne celui avez pas)"
#endif
	);

	w = (GtkWidget *)browse_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVDeviceEditDlgBrowseProgramCB), entry_rtn
	);

	/* Format Command GtkEntry */
	w = GUIPromptBarWithBrowse(
	    NULL,
#ifdef PROG_LANGUAGE_ENGLISH
	    "Format:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Formato:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Format:",
#endif
	    NULL, &entry_rtn, &browse_rtn,
	    NULL, NULL
	);
	d->command_format_entry = (GtkWidget *)entry_rtn;
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	w = (GtkWidget *)entry_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(EDVDeviceEditDlgAnyChangedCB), d
	);
	edv_entry_set_dnd(core, w);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(
	    w,
#ifdef PROG_LANGUAGE_ENGLISH
"Enter the command that will be executed to run the\
 format program for this device (or leave it blank if you do not\
 have one)"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Entre la orden que se ejecutar para correr the el\
 programa de formato para este artefacto (o le sale\
 blanco si usted hace not tiene uno)"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Entrer l'ordre qui sera excut courir le programme\
 de format pour cet appareil (ou le part le vide si\
 vous ne celui avez pas)"
#endif
	);

	w = (GtkWidget *)browse_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVDeviceEditDlgBrowseProgramCB), entry_rtn
	);


	/* Right Side GtkVBox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Icons GtkFrame */
	w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Iconos"
#elif defined(PROG_LANGUAGE_FRENCH)
"Icnes"
#elif defined(PROG_LANGUAGE_GERMAN)
"Abbilder"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Le Icone"
#elif defined(PROG_LANGUAGE_DUTCH)
"Beelden"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Os cones"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Ikoner"
#else
"Icons"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Icon Pickers GtkVBox */
	w = gtk_vbox_new(FALSE, 2 * border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;

	/* Set the Icon Picker flags */
	ip_flags = ICON_PICKER_CAN_CLEAR_ICON |
	    ICON_PICKER_ACCEPT_SMALLER_SIZE |
	    ICON_PICKER_ACCEPT_LARGER_SIZE |
	    ICON_PICKER_WARN_DIFFERENT_SIZE |
	    ICON_PICKER_ACCEPT_DND_DROP |
	    ICON_PICKER_PROVIDE_DND_DRAG;
	if(EDV_GET_B(EDV_CFG_PARM_LISTS_DOUBLE_BUFFER))
	    ip_flags |= ICON_PICKER_DOUBLE_BUFFER;
	if(EDV_GET_B(EDV_CFG_PARM_SHOW_TEXTTIPS))
	    ip_flags |= ICON_PICKER_SHOW_TEXTTIPS;

	lists_pointer_op_button2 = (EDVListsPointerOpButton2)EDV_GET_I(
	    EDV_CFG_PARM_LISTS_POINTER_OP_BUTTON2
	);

	/* Load images interval (in milliseconds) */
	load_int = edv_get_interval_from_load_images_priority(
	    (EDVListsLoadImagesPriority)EDV_GET_I(
		EDV_CFG_PARM_LISTS_LOAD_IMAGES_PRIORITY
	    )
	);

	/* Default icons location */
	default_icons_path = g_strconcat(
	    EDV_GET_S(EDV_CFG_PARM_DIR_GLOBAL),
	    G_DIR_SEPARATOR_S,
	    EDV_NAME_ICONS_SUBDIR,
	    NULL
	);

	/* Icon Selector Dialog */
	d->icon_sel_dlg = icon_sel_dlg = IconSelDlgNew(
	    ((ip_flags & ICON_PICKER_DOUBLE_BUFFER) ?
		ICON_SEL_DLG_DOUBLE_BUFFER : 0) |
	    ((ip_flags & ICON_PICKER_SHOW_TEXTTIPS) ?
		ICON_SEL_DLG_SHOW_TEXTTIPS : 0) |
	    ((lists_pointer_op_button2 == EDV_LISTS_POINTER_OP_BUTTON2_SCROLL_XY) ?
	        ICON_SEL_DLG_ENABLE_DRAG_SCROLL : 0),
	    GTK_ORIENTATION_HORIZONTAL,
	    -1,					/* Default list height */
	    50, 50,				/* Thumb size */
	    0,					/* Thumb border */
	    default_icons_path,
	    load_int
	);

	g_free(default_icons_path);

	/* Large Icon Picker */
	d->icon_large_ip = w = icon_picker_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Grande"
#elif defined(PROG_LANGUAGE_FRENCH)
"Grand"
#elif defined(PROG_LANGUAGE_GERMAN)
"Gro"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Grande"
#elif defined(PROG_LANGUAGE_DUTCH)
"Groot"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Grande"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stor"
#else
"Large"
#endif
	    " 48x48:",
	    ip_flags,
	    ICON_PICKER_ICON_POSITION_CENTER,
	    48, 48,				/* Icon size */
	    54, 54,				/* Display size */
	    80,					/* Label width */
	    icon_sel_dlg
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	icon_picker_set_switched_icon_cb(
	    w,
	    EDVDeviceEditDlgIconPickerIconSwitchedCB, d
	);
	icon_picker_set_changed_cb(
	    w,
	    EDVDeviceEditDlgAnyChangedCB, d
	);
	gtk_widget_show(w);

	/* Medium Icon Picker */
	d->icon_medium_ip = w = icon_picker_new(
#if defined(PROG_LANGUAGE_SPANISH)
"El Medio"
#elif defined(PROG_LANGUAGE_FRENCH)
"Milieu"
#elif defined(PROG_LANGUAGE_GERMAN)
"Medium"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il Mezzo"
#elif defined(PROG_LANGUAGE_DUTCH)
"Middel"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Mdio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Middel"
#else
"Medium"
#endif
	    " 32x32:",
	    ip_flags,
	    ICON_PICKER_ICON_POSITION_CENTER,
	    32, 32,				/* Icon size */
	    38, 38,				/* Display size */
	    80,					/* Label width */
	    icon_sel_dlg
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	icon_picker_set_switched_icon_cb(
	    w,
	    EDVDeviceEditDlgIconPickerIconSwitchedCB, d
	);
	icon_picker_set_changed_cb(
	    w,
	    EDVDeviceEditDlgAnyChangedCB, d
	);
	gtk_widget_show(w);

	/* Small Icon Picker */
	d->icon_small_ip = w = icon_picker_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Pequeo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Petit"
#elif defined(PROG_LANGUAGE_GERMAN)
"Klein"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Piccolo"
#elif defined(PROG_LANGUAGE_DUTCH)
"Klein"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Pequeno"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Liten"
#else
"Small"
#endif
	    " 20x20:",
	    ip_flags,
	    ICON_PICKER_ICON_POSITION_CENTER,
	    20, 20,				/* Icon size */
	    26, 26,				/* Display size */
	    80,					/* Label width */
	    icon_sel_dlg
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	icon_picker_set_switched_icon_cb(
	    w,
	    EDVDeviceEditDlgIconPickerIconSwitchedCB, d
	);
	icon_picker_set_changed_cb(
	    w,
	    EDVDeviceEditDlgAnyChangedCB, d
	);
	gtk_widget_show(w);


	/* Horizontal GtkSeparator */
	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Buttons GtkHBox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* OK button */
	d->ok_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_ok_20x20_xpm, "OK", NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVDeviceEditDlgOKCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_o, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_o);
	gtk_widget_show(w);

	/* Apply button */
	d->apply_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_select_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Aplique"
#elif defined(PROG_LANGUAGE_FRENCH)
"S'Appliquer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Verwenden"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Applicare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Toepas"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Aplique"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Gjeld"
#else
"Apply"
#endif
	    , NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVDeviceEditDlgApplyCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_a, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_a);
	gtk_widget_show(w);

	/* Cancel button */
	d->cancel_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_cancel_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Cancele"
#elif defined(PROG_LANGUAGE_FRENCH)
"Annuler"
#elif defined(PROG_LANGUAGE_GERMAN)
"Heben"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Annullare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Annuleer"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Cancelamento"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kanseller"
#else
"Cancel"
#endif
	    , NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVDeviceEditDlgCancelCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);
	gtk_widget_show(w);

	/* Close button */
	d->close_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_close_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Cierre"
#elif defined(PROG_LANGUAGE_FRENCH)
"Fin"
#elif defined(PROG_LANGUAGE_GERMAN)
"Nah"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Vicino"
#elif defined(PROG_LANGUAGE_DUTCH)
"Einde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Prximo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Nr"
#else
"Close"
#endif
	    , NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVDeviceEditDlgCancelCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);
	gtk_widget_show(w);


	EDVDeviceEditDlgUpdateMenus(d);

	d->freeze_count--;

	return(d);
}

/*
 *	Resets the Devices Edit Window's has changes marker.
 */
void EDVDeviceEditDlgResetHasChanges(
	edv_device_edit_dlg_struct *d,
	const gboolean has_changes
)
{
	if(d == NULL)
	    return;

	if(d->has_changes == has_changes)
	    return;

	d->has_changes = has_changes;
	EDVDeviceEditDlgUpdateMenus(d);
}

/*
 *	Updates the Devices Edit Window's widgets to reflect current
 *	values.
 */
void EDVDeviceEditDlgUpdateMenus(edv_device_edit_dlg_struct *d)
{
	gboolean        sensitive,
			has_changes;
	gint dev_num;
	gchar *s;
	const EDVDevice *dev;
	EDVCore *core;

	if(d == NULL)
	    return;

	d->freeze_count++;

	core = d->core;

	has_changes = d->has_changes;
	dev_num = d->dev_num;
	dev = EDV_DEVICE(g_list_nth_data(
	    core->devices_list,
	    (guint)dev_num
	));

	/* Title */
	s = g_strconcat(
	    EDV_DEVICE_EDIT_DLG_TITLE,
	    (has_changes) ? " (*)" : "",
	    NULL
	);
	gtk_window_set_title(GTK_WINDOW(d->toplevel), s);
	g_free(s);

	/* OK, Apply, Cancel, Close Buttons */
	sensitive = has_changes ? TRUE : FALSE;
	gtk_widget_set_sensitive(d->ok_btn, sensitive);
	gtk_widget_set_sensitive(d->apply_btn, sensitive);
	sensitive = has_changes;
	gtk_widget_set_sensitive(d->cancel_btn, sensitive);
	sensitive = has_changes ? FALSE : TRUE;      
	gtk_widget_set_sensitive(d->close_btn, sensitive);
	if(has_changes)                                  
	{
	    gtk_widget_show(d->ok_btn);
	    gtk_widget_show(d->apply_btn);
	    gtk_widget_show(d->cancel_btn);
	    gtk_widget_hide(d->close_btn);
	}
	else
	{
	    gtk_widget_show(d->ok_btn);
	    gtk_widget_show(d->apply_btn);
	    gtk_widget_hide(d->cancel_btn);
	    gtk_widget_show(d->close_btn);
	}

	d->freeze_count--;
}

/*
 *	Checks if the Devices Edit Window.
 */
gboolean EDVDeviceEditDlgIsMapped(
	edv_device_edit_dlg_struct *d
)
{
	if(d == NULL)
	    return(FALSE);

	return(GTK_WIDGET_MAPPED(d->toplevel));
}

/*
 *	Maps the Devices Edit Window.
 */
void EDVDeviceEditDlgMap(
	edv_device_edit_dlg_struct *d
)
{
	if(d == NULL)
	    return;

	gtk_widget_show_raise(d->toplevel);

	gtk_widget_grab_focus(d->name_entry);
}

/*
 *	Unmaps the Devices Edit Window.
 */
void EDVDeviceEditDlgUnmap(
	edv_device_edit_dlg_struct *d
)
{
	if(d == NULL)
	    return;

	gtk_widget_hide(d->toplevel);
}

/*
 *	Deletes the Devices List Window.
 */
void EDVDeviceEditDlgDelete(
	edv_device_edit_dlg_struct *d
)
{
	if(d == NULL)
	    return;

	EDVDeviceEditDlgUnmap(d);

	d->freeze_count++;

	d->icon_sel_dlg = IconSelDlgUnref(d->icon_sel_dlg);

	gtk_widget_destroy(d->toplevel);
	gtk_accel_group_unref(d->accelgrp);

	d->freeze_count--;

	g_free(d);
}
