#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>

#include "guiutils.h"
#include "fprompt.h"
#include "desktop_icon.h"


typedef struct _DesktopIcon		DesktopIcon;
#define DESKTOP_ICON(p)			((DesktopIcon *)(p))
#define DESKTOP_ICON_KEY		"DesktopIcon"


/* Callbacks */
static void DesktopIconDestroyCB(gpointer data);
static void DesktopIconRealizeCB(GtkWidget *widget, gpointer data);
static void DesktopIconLabelRealizeCB(GtkWidget *widget, gpointer data);
static gint DesktopIconEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void DesktopIconFPromptApplyCB(gpointer data, const gchar *s);

/* Rename */
static void DesktopIconRenameQuery(DesktopIcon *icon);

/* Draw */
static void DesktopIconDraw(DesktopIcon *icon);
static void DesktopIconQueueDraw(DesktopIcon *icon);

/* Resize */
static void DesktopIconResizeIcon(
	DesktopIcon *icon,
	GdkPixmap *pixmap, GdkBitmap *mask
);
static void DesktopIconMoveLabelToIcon(
	DesktopIcon *icon,
	const gint x, const gint y
);
static void DesktopIconResizeLabel(
	DesktopIcon *icon,
	const gchar *label
);
#if 0
static void DesktopIconResize(DesktopIcon *icon);
#endif

/* Set Icon */
static gboolean DesktopIconSetIcon(
	DesktopIcon *icon,
	guint8 **icon_data,
	const gchar *icon_path,
	const gboolean force
);
/* Set Label */
static gboolean DesktopIconSetLabel(
	DesktopIcon *icon,
	const gchar *label,
	const gboolean force
);


/* Desktop Icon */
static DesktopIcon *DesktopIconGetWidgetData(
	GtkWidget *w,
	const gchar *func_name
);

GtkWidget *DesktopIconNew(
	const desktop_icon_flags flags,
	const gint x, const gint y,
	guint8 **icon_data,
	const gchar *icon_path,
	const gchar *label
);

GtkWidget *DesktopIconGetIconWidget(GtkWidget *w);

void DesktopIconSetPosition(
	GtkWidget *w,
	const gint x, const gint y
);
gboolean DesktopIconGetPosition(
	GtkWidget *w,
	gint *x_rtn, gint *y_rtn
);

void DesktopIconSet(
	GtkWidget *w,
	guint8 **icon_data,
	const gchar *label
);
void DesktopIconSetFile(
	GtkWidget *w,
	const gchar *icon_path,
	const gchar *label
);
gboolean DesktopIconGet(
	GtkWidget *w,
	guint8 ***icon_data_rtn,
        const gchar **icon_path_rtn,
	const gchar **label_rtn
);

void DesktopIconSetSensitive(
	GtkWidget *w,
	const gboolean sensitive
);
gboolean DesktopIconGetSensitive(GtkWidget *w);

void DesktopIconSetAlwaysOnTop(
	GtkWidget *w,
	const gboolean always_on_top
);
gboolean DesktopIconGetAlwaysOnTop(GtkWidget *w);

gboolean DesktopIconGetSelected(GtkWidget *w);
void DesktopIconSelect(GtkWidget *w);
void DesktopIconUnselect(GtkWidget *w);

void DesktopIconSetMenu(
	GtkWidget *w,
	GtkWidget *menu
);
GtkWidget *DesktopIconGetMenu(GtkWidget *w);

void DesktopIconSetMapRefCB(
	GtkWidget *w,
	void (*activate_cb)(GtkWidget *, gpointer),
	gpointer data
);
void DesktopIconSetSelectCB(
	GtkWidget *w,
	void (*select_cb)(GtkWidget *, gpointer),
	gpointer data
);
void DesktopIconSetUnselectCB(
	GtkWidget *w,
	void (*unselect_cb)(GtkWidget *, gpointer),
	gpointer data
);
void DesktopIconSetRenameCB(
	GtkWidget *w,
	void (*rename_cb)(GtkWidget *, const gchar *, const gchar *, gpointer),
	gpointer data
);
void DesktopIconSetMoveBeginCB(
	GtkWidget *w,
	void (*move_begin_cb)(GtkWidget *, const gint, const gint, gpointer),
	gpointer data
);
void DesktopIconSetMovingCB(
	GtkWidget *w,
	void (*moving_cb)(GtkWidget *, const gint, const gint, gpointer),
	gpointer data
);
void DesktopIconSetMovedCB(
	GtkWidget *w,
	void (*moved_cb)(GtkWidget *, const gint, const gint, gpointer),
	gpointer data
);

static void DesktopIconRaise(DesktopIcon *icon);
static void DesktopIconLower(DesktopIcon *icon);
void DesktopIconMap(GtkWidget *w);
void DesktopIconUnmap(GtkWidget *w);
GtkWidget *DesktopIconRef(GtkWidget *w);
GtkWidget *DesktopIconUnref(GtkWidget *w);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Desktop Icon:
 */
struct _DesktopIcon {

	gint		ref_count,
			freeze_count;
	GdkGC		*gc;
	desktop_icon_flags	flags;

	GdkWindow	*root;			/* Reference to the desktop */
	gint		root_width,
			root_height;

	/* Icon GtkWidgets and values */
	GtkWidget	*icon_toplevel,		/* GtkWindow */
			*icon_da;		/* GtkDrawingArea */
	guint8		**last_icon_data;
	gchar		*last_icon_path;
	GdkPixmap	*icon_pixmap;
	GdkBitmap	*icon_mask;
	gint		icon_x,			/* Icon GtkWindow position
						 * relative to the desktop,
						 * which is also considered
						 * the source position of the
						 * entire Desktop Icon */
			icon_y,
			icon_width,
			icon_height;

	/* Label GtkWidgets and values */
	GtkWidget	*label_toplevel,	/* GtkWindow */
			*label_da;		/* GtkDrawingArea */
	gchar		*label_str;
	gint		label_x,		/* Label GtkWindow position
						 * relative to the desktop */
			label_y,
			label_width,
			label_height;

	/* Popup GtkMenu */
	GtkWidget	*menu;

	/* Last button that was pressed and pointer drag position */
	gint		button,
			drag_last_x,		/* Relative to the desktop */
			drag_last_y;

	/* Callbacks
	 *
	 * Map referenced window, called on double-clicks
	 */
	void		(*activate_cb)(
		GtkWidget *,			/* Desktop Icon */
		gpointer			/* Data */
	);
	gpointer	activate_data;

	/* Select callback, called when deskicon is selected */
	void		(*select_cb)(
		GtkWidget *,			/* Desktop Icon */
		gpointer			/* Data */
	);
	gpointer	select_data;

	/* Unselect callback, called when deskicon is unselected */
	void		(*unselect_cb)(
		GtkWidget *,			/* Desktop Icon */
		gpointer			/* Data */
	);
	gpointer	unselect_data;

	/* Rename callback, called when deskicon label has changed */
	void		(*rename_cb)(
		GtkWidget *,			/* Desktop Icon */
		const gchar *,			/* Old Name */
		const gchar *,			/* New Name */
		gpointer			/* Data */
	);
	gpointer	rename_data;

	/* Move begin callback, called at the beginning of any move */
	void		(*move_begin_cb)(
		GtkWidget *,			/* Desktop Icon */
		const gint, const gint,		/* Position */
		gpointer			/* Data */
	);
	gpointer	move_begin_data;

	/* Moving callback, called during any movement */
	void		(*moving_cb)(
		GtkWidget *,			/* Desktop Icon */
		const gint, const gint,		/* Position */
		gpointer			/* Data */
	);
	gpointer	moving_data;

	/* Moved callback, called after each move */
	void		(*moved_cb)(
		GtkWidget *,			/* Desktop Icon */
		const gint, const gint,		/* Position */
		gpointer			/* Data */
	);
	gpointer	moved_data;

};


/*
 *	Toplevel GtkWindow "destroy" signal callback.
 */
static void DesktopIconDestroyCB(gpointer data)
{
	DesktopIcon *icon = DESKTOP_ICON(data);
	if(icon == NULL)
	    return;

	if(icon->ref_count > 0)
	{
	    g_printerr(
"DesktopIconDestroyCB():\
 Warning: Destroying DesktopIcon %p with %i reference counts remaining.\n\
Was DesktopIconUnref() not used properly to unref this DesktopIcon?\n",
		icon->icon_toplevel,
		icon->ref_count
	    );
	}

	icon->freeze_count++;

	GTK_WIDGET_DESTROY(icon->menu);
	GTK_WIDGET_DESTROY(icon->label_toplevel);

	(void)GDK_PIXMAP_UNREF(icon->icon_pixmap);
	(void)GDK_BITMAP_UNREF(icon->icon_mask);

	(void)GDK_GC_UNREF(icon->gc);

	g_free(icon->last_icon_path);
	g_free(icon->label_str);

	icon->freeze_count--;

	g_free(icon);
}

/*
 *	Icon GtkDrawingArea "realize" signal callback.
 */
static void DesktopIconRealizeCB(GtkWidget *widget, gpointer data)
{
	GdkWindow	*window,
			*root;
	DesktopIcon *icon = DESKTOP_ICON(data);
	if((widget == NULL) || (icon == NULL))
	    return;

	window = widget->window;

	/* Graphics context */
	if(icon->gc == NULL)
	    icon->gc = gdk_gc_new(window);

	/* Get the desktop GdkWindow and its size */
	icon->root = root = gdk_window_get_parent(window);
	if(root != NULL)
	    gdk_window_get_size(
		root,
		&icon->root_width, &icon->root_height
	    );		    

	/* Set the initial icon and resize the icon GtkWindow */
	DesktopIconSetIcon(
	    icon,
	    icon->last_icon_data,
	    icon->last_icon_path,
	    TRUE				/* Force change */
	);

	/* Mark the Desktop Icon as realized */
	icon->flags |= DESKTOP_ICON_REALIZED;
}

/*
 *	Label GtkDrawingArea "realize" signal callback.
 */
static void DesktopIconLabelRealizeCB(GtkWidget *widget, gpointer data)
{
	DesktopIcon *icon = DESKTOP_ICON(data);
	if((widget == NULL) || (icon == NULL))
	    return;

	/* Set the initial label string, resize the label GtkWindow
	 * and align its position with the icon GtkWindow
	 */
	DesktopIconSetLabel(
	    icon,
	    icon->label_str,
	    TRUE				/* Force change */
	);
}

/*
 *	Any GtkWidget event signal callback.
 */
static gint DesktopIconEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	GdkEventConfigure *configure;
	GdkEventFocus *focus;
	GdkEventKey *key;
	GdkEventButton *button;
	GdkEventMotion *motion;
	DesktopIcon *icon = DESKTOP_ICON(data);
	if((widget == NULL) || (event == NULL) || (icon == NULL))
	    return(status);

	switch((gint)event->type)
	{
	  case GDK_CONFIGURE:
	    configure = (GdkEventConfigure *)event;
	    if(widget == icon->icon_toplevel)
	    {
		/* Update the current position */
		icon->icon_x = configure->x;
		icon->icon_y = configure->y;

		/* Move the label GtkWindow to the position aligned
		 * with the current position of the icon GtkWindow
		 */
		DesktopIconMoveLabelToIcon(
		    icon,
		    icon->icon_x, icon->icon_y
		);

		/* Notify about the icon moving */
		if(icon->moving_cb != NULL)
		    icon->moving_cb(
			icon->icon_toplevel,	/* Desktop Icon */
			icon->icon_x,		/* Position */
			icon->icon_y,
			icon->moving_data	/* Data */
		    );
	    }
	    status = TRUE;
	    break;

	  case GDK_EXPOSE:
	    DesktopIconDraw(icon);
	    status = TRUE;
	    break;

	  case GDK_FOCUS_CHANGE:
	    focus = (GdkEventFocus *)event;
	    if(focus->in)
	    {
		GtkWidget *w = icon->icon_toplevel;
		if(!GTK_WIDGET_HAS_FOCUS(w))
		{
		    GTK_WIDGET_SET_FLAGS(w, GTK_HAS_FOCUS);
		    DesktopIconQueueDraw(icon);
		}
	    }
	    else
	    {
		GtkWidget *w = icon->icon_toplevel;
		if(GTK_WIDGET_HAS_FOCUS(w))
		{
		    GTK_WIDGET_UNSET_FLAGS(w, GTK_HAS_FOCUS);
		    DesktopIconQueueDraw(icon);
		}
	    }
	    status = TRUE;
	    break;

	  case GDK_KEY_PRESS:
	    if(icon->freeze_count > 0)
		return(status);
	    key = (GdkEventKey *)event;
	    status = TRUE;
	    break;

	  case GDK_KEY_RELEASE:
	    if(icon->freeze_count > 0)
		return(status);
	    key = (GdkEventKey *)event;
	    status = TRUE;
	    break;

	  case GDK_BUTTON_PRESS:
	    if(icon->freeze_count > 0)
		return(status);

	    /* If any of the Desktop Icon GtkWidgets was clicked on
	     * then the icon GtkWindow should grab focus
	     */
	    if(!GTK_WIDGET_HAS_FOCUS(icon->icon_toplevel))
	    {
		GtkWidget *w = icon->icon_toplevel;
		gtk_widget_grab_focus(w);
		GTK_WIDGET_SET_FLAGS(w, GTK_HAS_FOCUS);
		DesktopIconQueueDraw(icon);
	    }
	    button = (GdkEventButton *)event;
	    icon->button = button->button;
	    icon->drag_last_x = (gint)button->x_root;
	    icon->drag_last_y = (gint)button->y_root;
	    switch(button->button)
	    {
	      case GDK_BUTTON1:			/* Select/unselect */
		if(DesktopIconIsSelected(icon->icon_toplevel))
		    DesktopIconUnselect(icon->icon_toplevel);
		else
		    DesktopIconSelect(icon->icon_toplevel);
		break;

	      case GDK_BUTTON2:			/* Rename */
		DesktopIconRenameQuery(icon);
		break;

	      case GDK_BUTTON3:			/* Map the menu */
		if(icon->menu != NULL)
		    gtk_menu_popup(
			GTK_MENU(icon->menu),
			NULL, NULL,
			NULL, NULL,
			button->button, button->time
		    );
		break;
	    }
	    status = TRUE;
	    break;

	  case GDK_2BUTTON_PRESS:
	    if(icon->freeze_count > 0)
		return(status);
	    button = (GdkEventButton *)event;
	    /* Must reset the button and drag values on double clicks */
	    icon->button = 0;
	    icon->drag_last_x = 0;
	    icon->drag_last_y = 0;
	    switch(button->button)
	    {
	      case GDK_BUTTON1:			/* Activate */
		if(icon->activate_cb != NULL)
		    icon->activate_cb(
			icon->icon_toplevel,	/* Desktop Icon */
			icon->activate_data	/* Data */
		    );
		break;
	    }
	    status = TRUE;
	    break;

	  case GDK_BUTTON_RELEASE:
	    if(icon->freeze_count > 0)
		return(status);

	    button = (GdkEventButton *)event;
	    icon->button = 0;
	    icon->drag_last_x = 0;
	    icon->drag_last_y = 0;

	    /* Ungrab pointer as needed */
	    if(gdk_pointer_is_grabbed())
		gdk_pointer_ungrab(button->time);

	    /* Was the icon moving? */
	    if(icon->flags & DESKTOP_ICON_IS_MOVING)
	    {
		/* Remove the moving flag */
		icon->flags &= ~DESKTOP_ICON_IS_MOVING;

		/* Lower the Desktop Icon below all the other windows? */
		if(!(icon->flags & DESKTOP_ICON_ALWAYS_ON_TOP))
		    DesktopIconLower(icon);

		/* Notify about the final move */
		if(icon->moved_cb != NULL)
		    icon->moved_cb(
			icon->icon_toplevel,	/* Desktop Icon */
			icon->icon_x,		/* Position */
			icon->icon_y,
			icon->moved_data	/* Data */
		    );
	    }

	    status = TRUE;
	    break;

	  case GDK_MOTION_NOTIFY:
	    if(icon->freeze_count > 0)
		return(status);

	    motion = (GdkEventMotion *)event;

	    /* Button 1 pressed? */
	    if((icon->button == GDK_BUTTON1) &&
	       (icon->flags & DESKTOP_ICON_CAN_MOVE)
	    )
	    {
		gint	x, y,
			dx, dy,
			drag_tolor = 5;
		GdkModifierType mask;
		GdkWindow *window = motion->window;

		/* If this event is a motion hint then request additional
		 * motion events to be sent
		 */
		if(motion->is_hint)
		    gdk_window_get_pointer(
			window,
			&x, &y,
			&mask
		    );

		/* Get the current pointer coordinates */
		x = (gint)motion->x_root;
		y = (gint)motion->y_root;

		/* Calculate the motion movement */
		dx = x - icon->drag_last_x;
		dy = y - icon->drag_last_y;

		/* Moving yet? */
		if(icon->flags & DESKTOP_ICON_IS_MOVING)
		{
		    /* Is moving, so move the icon and sync label to
		     * the icon's new position
		     */
		    const gint	icon_x = icon->icon_x + dx,
				icon_y = icon->icon_y + dy;

		    gtk_widget_set_uposition(
			icon->icon_toplevel,
			icon_x, icon_y
		    );

		    /* Do not update the icon's current position or
		     * move the label, wait for the "configure_event"
		     * signal to do that
		     */

		    /* Record the new drag position */
		    icon->drag_last_x = x;
		    icon->drag_last_y = y;
		}
		else
		{
		    /* Not yet moving, check if the pointer moved
		     * beyond the drag tolorance
		     */
		    if((dx >= drag_tolor) || (dx <= -drag_tolor) ||
		       (dy >= drag_tolor) || (dy <= -drag_tolor)
		    )
		    {
			/* Begin moving */
			GtkWidget *w = icon->icon_toplevel;

			/* Select as needed */
			if(!DesktopIconIsSelected(w))
			    DesktopIconSelect(w);

			/* Raise the Desktop Icon above all the other
			 * windows
			 */
			DesktopIconRaise(icon);

			/* Mark that we are now moving */
			icon->flags |= DESKTOP_ICON_IS_MOVING;

			/* Grab the pointer relative to the desktop */
			gdk_pointer_grab(
			    motion->window,
			    FALSE,
			    GDK_BUTTON_PRESS_MASK |
				GDK_BUTTON_RELEASE_MASK |
				GDK_POINTER_MOTION_MASK,
			    NULL,
			    NULL,
			    motion->time
			);

			/* Record the new drag position */
			icon->drag_last_x = x;
			icon->drag_last_y = y;

			/* Notify about the start of the move */
			if(icon->move_begin_cb != NULL)
			    icon->move_begin_cb(
				icon->icon_toplevel,	/* Desktop Icon */
				icon->icon_x,		/* Position */
				icon->icon_y,
				icon->move_begin_data	/* Data */
			    );
		    }
		}
	    }
	    status = TRUE;
	    break;
	}

	return(status);
}

/*
 *	Float prompt rename apply callback.
 */
static void DesktopIconFPromptApplyCB(gpointer data, const gchar *s)
{
	gchar *new_label;
	DesktopIcon *icon = DESKTOP_ICON(data);
	if((icon == NULL) || (s == NULL))
	    return;

	/* Get new label */
	new_label = STRDUP(s);

	/* Set the new label */
	DesktopIconSet(
	    icon->icon_toplevel,
	    NULL,
	    new_label
	);

	g_free(new_label);
}


/*
 *	Puts the Desktop Icon into rename query.
 */
static void DesktopIconRenameQuery(DesktopIcon *icon)
{
	gint x, y, width;

	if((icon == NULL) || FPromptIsQuery())
	    return;

	/* Renaming not allowed? */
	if(!(icon->flags & DESKTOP_ICON_CAN_RENAME))
	    return;

	/* Get the label's position */
	x = icon->label_x;
	y = icon->label_y;

	/* Calculate position of prompt to fit over the current position 
	 * and size of the deskicon's label
	 */
	width = MAX(icon->label_width, 100);
	x = x - 2 - ((width - icon->label_width) / 2);
	y = y - 4;

	/* Set position and map prompt */
	FPromptSetPosition(x, y);
	FPromptMapQuery(
	    NULL,
	    icon->label_str,
	    NULL,
	    FPROMPT_MAP_TO_POSITION,
	    width, -1,
	    0,		/* Flags */
	    icon,
	    NULL,
	    DesktopIconFPromptApplyCB,
	    NULL
	);
}

/*
 *	Draws the Desktop Icon's icon and label.
 */
static void DesktopIconDraw(DesktopIcon *icon)
{
	GdkGC *gc;
	GdkPixmap *pixmap;
	GdkBitmap *mask;
	GtkWidget *w;

	/* Do not draw if the Desktop Icon is not mapped */
	if(!(icon->flags & DESKTOP_ICON_MAPPED))
	    return;

	gc = icon->gc;

	/* Draw the icon */
	w = icon->icon_da;
	pixmap = icon->icon_pixmap;
	mask = icon->icon_mask;
	if((w != NULL) && (pixmap != NULL))
	{
	    GdkWindow *window = w->window;
	    const GtkStateType state = GTK_WIDGET_STATE(w);
	    GtkStyle *style = gtk_widget_get_style(w);
	    gint	x = 0,
			y = 0,
			width = icon->icon_width,
			height = icon->icon_height;

	    if(GTK_WIDGET_VISIBLE(w) && (window != NULL) && (style != NULL))
	    {
		gdk_gc_set_function(gc, GDK_COPY);
		gdk_gc_set_fill(gc, GDK_SOLID);
		gdk_gc_set_clip_mask(gc, mask);
		gdk_gc_set_clip_origin(gc, x, y);
		gdk_gc_set_foreground(gc, &style->bg[state]);
		gdk_draw_pixmap(
		    (GdkDrawable *)window,
		    gc,
		    (GdkDrawable *)pixmap,
		    0, 0,
		    x, y,
		    width, height
		);
		gdk_gc_set_clip_mask(gc, NULL);

		/* Draw the selection if selected */
		if(state == GTK_STATE_SELECTED)
		{
		    gdk_gc_set_function(gc, GDK_INVERT);
		    gdk_gc_set_foreground(gc, &style->white);
		    gdk_draw_rectangle(
			(GdkDrawable *)window,
			gc,
			TRUE,			/* Fill */
			x, y,
			width, height
		    );
		    gdk_gc_set_function(gc, GDK_COPY);
		}
	    }
	}

	/* Drawing the label */
	w = icon->label_da;
	if((w != NULL) && (icon->label_str != NULL))
	{
	    const gint	border_minor = 2;
	    GdkWindow *window = w->window;
	    const GtkStateType state = GTK_WIDGET_STATE(w);
	    GtkStyle *style = gtk_widget_get_style(w);
	    gint	x = 0,
			y = 0,
			width = icon->label_width,
			height = icon->label_height;

	    if(GTK_WIDGET_VISIBLE(w) && (window != NULL) && (style != NULL))
	    {
		/* Draw the background */
		gdk_gc_set_function(gc, GDK_COPY);
		gdk_gc_set_fill(gc, GDK_SOLID);
		gdk_gc_set_foreground(
		    gc,
		    (state == GTK_STATE_SELECTED) ?
			&style->bg[state] : &style->base[state]
		);
		gdk_draw_rectangle(
		    (GdkDrawable *)window,
		    gc,
		    TRUE,			/* Fill */
		    x, y,
		    width, height
		);

		/* Draw the label */
		if(style->font != NULL)
		{
		    gint x, y, line_len;
		    const gchar		*line = icon->label_str,
					*line_end;
		    GdkTextBounds b;
		    GdkFont *font = style->font;
		    const gint line_height = font->ascent + font->descent;

		    gdk_gc_set_foreground(gc, &style->text[state]);

		    /* Calculate the starting position */
		    x = border_minor;
		    y = border_minor;

		    /* Draw each line */
		    while(*line != '\0')
		    {
			line_end = (const gchar *)strchr(
			    (const char *)line,
			    '\n'
			);
			if(line_end != NULL)
			    line_len = (gint)(line_end - line);
			else
			    line_len = STRLEN(line);

			if(line_len > 0)
			{
			    /* Get the geometry of this line */
			    gdk_text_bounds(
				font,
				line, line_len,
				&b
			    );

			    /* Calculate the x position */
			    x = (width - b.width) / 2;

			    /* Draw this line */
			    gdk_draw_text(
				(GdkDrawable *)window,
				font,
				gc,
				x - b.lbearing,
				y + font->ascent,
				line, line_len
			    );
			}

			/* Increment the y position to the next line */
			y += line_height;

			/* Seek to the next line */
			line += line_len;

			/* Increment once more if the line ended with a
			 * line deliminator character
			 */
			if(*line != '\0')
			    line++;
		    }
		}

		/* Draw the focus rectangle for the label if the
		 * icon GtkWindow is in focus
		 */
		if(GTK_WIDGET_HAS_FOCUS(icon->icon_toplevel))
		{
		    gdk_gc_set_function(gc, GDK_INVERT);
		    gdk_gc_set_foreground(gc, &style->white);
		    gdk_draw_rectangle(
			(GdkWindow *)window,
			gc,
			FALSE,			/* Outline */
			0, 0,
			width - 1, height - 1
		    );
		    gdk_gc_set_function(gc, GDK_COPY);
		}
	    }
	}
}

/*
 *	Queue draw for the Desktop Icon.
 */
static void DesktopIconQueueDraw(DesktopIcon *icon)
{
	if(icon == NULL)
	    return;

	/* Only the icon toplevel GtkWindow needs to be queued
	 * to draw because DesktopIconDraw() will be called to
	 * draw, and DesktopIconDraw() draws both the icon and
	 * the label
	 */
	gtk_widget_queue_draw(icon->icon_toplevel);
}


/*
 *	Recalculate the icon's size based on the current icon and
 *	resizes the label GtkWindow and GtkDrawingArea.
 *
 *	The pixmap and mask specifies the icon. If mask is not NULL
 *	then the shape of the icon GtkWindow will be modified.
 */
static void DesktopIconResizeIcon(
	DesktopIcon *icon,
	GdkPixmap *pixmap, GdkBitmap *mask
)
{
	gint width, height;
	GtkWidget *w;

	if(pixmap == NULL)
	    return;

	/* Get the size of the current icon */
	gdk_window_get_size(
	    (GdkWindow *)pixmap,
	    &width, &height
	);

	/* No change in size? */
	if((icon->icon_width == width) && (icon->icon_height == height))
	    return;

	/* Record the new icon size */
	icon->icon_width = width;
	icon->icon_height = height;

	/* Update the size and the shape of the toplevel GtkWindow
	 * and GtkDrawingArea
	 */
	w = icon->icon_toplevel;
	gtk_widget_set_usize(w, width, height);
	gtk_widget_shape_combine_mask(w, mask, 0, 0);

	w = icon->icon_da;
	gtk_widget_set_usize(w, width, height);
/*	gtk_widget_shape_combine_mask(w, mask, 0, 0); */
}

/*
 *	Recalculates the label's position aligned to the current icon
 *	position and moves the label GtkWindow.
 *
 *	The x and y specifies the position of the icon GtkWindow.
 */
static void DesktopIconMoveLabelToIcon(
	DesktopIcon *icon,
	const gint x, const gint y
)
{
	gint label_x, label_y;

	/* Calculate the position of the label GtkWindow aligned with
	 * the icon GtkWindow
	 */
	label_x = x + ((icon->icon_width - icon->label_width) / 2);
	label_y = y + icon->icon_height;

	/* No change in position? */
	if((icon->label_x == label_x) && (icon->label_y == label_y))
	    return;

	/* Record the new label position */
	icon->label_x = label_x;
	icon->label_y = label_y;

	/* Set the new position of the label GtkWindow */
	gtk_widget_set_uposition(
	    icon->label_toplevel,
	    label_x, label_y
	);
}

/*
 *	Recalculate the label's size based on the current label
 *	string and resizes the label GtkWindow and GtkDrawingArea.
 *
 *	The label specifies the string describing the label.
 */
static void DesktopIconResizeLabel(
	DesktopIcon *icon,
	const gchar *label
)
{
	const gint border_minor = 2;
	gint width, height;
	GtkWidget	*toplevel = icon->label_toplevel,
			*w = icon->label_da;
	GtkStyle *style = gtk_widget_get_style(w);
	if(style == NULL)
	    return;

	/* Calculate the new label's geometry */
	width = 0;
	height = 0;
	if((style->font != NULL) && (label != NULL))
	{
	    gint len;
	    const gchar		*s = label,
				*s2;
	    GdkFont *font = style->font;
	    const gint line_height = font->ascent + font->descent;

	    /* Calculate the longest line's width and the height
	     * of all the lines
	     */
	    while(*s != '\0')
	    {
		/* Does this line end with a newline character? */
		s2 = (const gchar *)strchr(
		    (const char *)s,
		    '\n'
		);

		/* Calculate the length of this line */
		if(s2 != NULL)
		    len = (gint)(s2 - s);
		else
		    len = STRLEN(s);
		if(len > 0)
		{
		    /* Calculate the length of this line in pixels */
		    const gint line_width = gdk_text_measure(
			font,
			s,
			len
		    );

		    /* Increase the width if this line's length is
		     * longer than any of the previous lines
		     */
		    if(line_width > width)
			width = line_width;
		}

		/* Increment the height for this line */
		height += line_height;

		/* Seek to next line */
		s += len;

		/* Increment once more if line ended with a line
		 * deliminator character
		 */
		if(*s != '\0')
		    s++;
	    }

	    /* Add the border to the size */
	    width += (2 * border_minor);
	    height += (2 * border_minor);
	}
	else
	{
	    width = 10;
	    height = 10;
	}

	/* No change in size? */
	if((icon->label_width == width) && (icon->label_height == height))
	    return;

	/* Record the new label size */
	icon->label_width = width;
	icon->label_height = height;

	if((width > 0) && (height > 0))
	{
	    /* Resize the label GtkWindow and GtkDrawingArea */
	    gtk_widget_set_usize(
		toplevel,
		width, height
	    );
	    gtk_widget_set_usize(
		w,
		width, height
	    );
	}
}

#if 0
/*
 *	Recalculates the icon's size and resizes the icon and
 *	recalculates the label's position and size and moves and
 *	resizes the label.
 */
static void DesktopIconResize(DesktopIcon *icon)
{
	/* Recalculate the icon's size based on the current icon
	 * and resize the icon GtkWindow and GtkDrawingArea.
	 */
	DesktopIconResizeIcon(
	    icon,
	    icon->icon_pixmap, icon->icon_mask	/* Current icon */
	);

	/* Recalculate the label's size based on the current label
	 * string and resize the label GtkWindow and GtkDrawingArea
	 */
	DesktopIconResizeLabel(
	    icon,
	    icon->label_str			/* Current label string */
	);

	/* Recalculate the label's position aligned to the current
	 * icon position and move the label GtkWindow
	 */
	DesktopIconMoveLabelToIcon(
	    icon,
	    icon->icon_x, icon->icon_y		/* Current icon position */
	);
}
#endif


/*
 *	Sets the Desktop Icon's icon from the XPM data and resizes the
 *	icon GtkWindow.
 *
 *	The icon_data specifies the XPM data to use for the icon. If
 *	icon_data is NULL or is the same as the previous icon data
 *	then nothing will be done.
 *
 *	Returns TRUE if there is a change.
 */
static gboolean DesktopIconSetIcon(
	DesktopIcon *icon,
	guint8 **icon_data,
	const gchar *icon_path,
	const gboolean force
)
{
	GdkPixmap *pixmap;
	GdkBitmap *mask;
	GtkWidget *w;

	if(icon == NULL)
	    return(FALSE);

	w = icon->icon_da;

	/* Load the icon from XPM data? */
	if(icon_data != NULL)
	{
	    /* No change in the icon data? */
	    if(icon->last_icon_data == icon_data)
	    {
		if(!force)
		    return(FALSE);
	    }

	    /* Load the icon */
	    pixmap = gdk_pixmap_create_from_xpm_d(
		w->window,
		&mask,
		NULL,				/* No background color */
		(gchar **)icon_data
	    );
	    if(pixmap == NULL)
		return(FALSE);

	    /* Record the icon data pointer */
	    icon->last_icon_data = icon_data;
	}
	/* Open the icon from an XPM file? */
	else if(icon_path != NULL)
	{
	    /* No change in the icon file? */
	    if(icon->last_icon_path == icon_path)
	    {
		if(!force)
		    return(FALSE);
	    }
	    else if(icon->last_icon_path != NULL)
	    {
		if(!strcmp(
		    (const char *)icon->last_icon_path,
		    (const char *)icon_path
		))
		{
		    if(!force)
			return(FALSE);
		}
	    }

	    /* Open the icon */
	    pixmap = gdk_pixmap_create_from_xpm(
		w->window,
		&mask,
		NULL,				/* No background color */
		icon_path
	    );
	    if(pixmap == NULL)
		return(FALSE);

	    /* Record the icon path */
	    if(icon->last_icon_path != icon_path)
	    {
		g_free(icon->last_icon_path);
		icon->last_icon_path = g_strdup(icon_path);
	    }
	}
	else
	{
	    return(FALSE);
	}

	/* Unref the old icon */
	(void)GDK_PIXMAP_UNREF(icon->icon_pixmap);
	(void)GDK_BITMAP_UNREF(icon->icon_mask);

	/* Set the new icon */
	icon->icon_pixmap = pixmap;
	icon->icon_mask = mask;

	/* Update the size of the icon and resize and shape the icon
	 * GtkWindow
	 */
	DesktopIconResizeIcon(
	    icon,
	    pixmap, mask
	);

	DesktopIconQueueDraw(icon);

	return(TRUE);
}

/*
 *	Sets the Desktop Icon's label and moves and resizes the label
 *	GtkWindow.
 *
 *	The label specifies the string describing the label. If label
 *	is NULL or is the same as the previous label then nothing will
 *	be done. If label is an empty string then the Desktop Icon's
 *	label GtkWindow will be unmapped. Otherwise the new label will
 *	be set and the label GtkWindow will be mapped (only if the
 *	Desktop Icon is mapped) and resized.
 *
 *      Returns TRUE if there is a change.
 */
static gboolean DesktopIconSetLabel(
	DesktopIcon *icon,
	const gchar *label,
	const gboolean force
)
{
	gchar *dlabel;

	if(label == NULL)
	    return(FALSE);			/* No change */

	dlabel = g_strdup(label);

	/* If the specified label is an empty string then do not show
	 * a label
	 */
	if(*dlabel == '\0')
	{
	    /* If a label was set then clear the label and hide the
	     * label GtkWindow
	     */
	    if(icon->label_str != NULL)
	    {
		/* Clear the label */
		g_free(icon->label_str);
		icon->label_str = NULL;
		icon->label_width = 0;
		icon->label_height = 0;

		/* Update the position and size of the label and move
		 * and resize the label GtkWindow
		 */
		DesktopIconResizeLabel(
		    icon,
		    icon->label_str
		);
		DesktopIconMoveLabelToIcon(
		    icon,
		    icon->icon_x, icon->icon_y
		);

		/* Hide the label GtkWindow */
		gtk_widget_hide(icon->label_toplevel);

		g_free(dlabel);
		return(TRUE);			/* Change */
	    }
	    else
	    {
		g_free(dlabel);
		return(FALSE);			/* No change */
	    }
	}

	/* Check if there is no change in the label */
	if(icon->label_str != NULL)
	{
	    /* Are the current and the new label strings the same? */
	    if(!strcmp(
		(const char *)icon->label_str,
		(const char *)dlabel
	    ))
	    {
		if(!force)
		{
		    g_free(dlabel);
		    return(FALSE);		/* No change */
		}
	    }
	}

	/* Set the new label */
	g_free(icon->label_str);
	icon->label_str = g_strdup(dlabel);

	/* Update the position and size of the label and move and
	 * resize the label GtkWindow
	 */
	DesktopIconResizeLabel(
	    icon,
	    icon->label_str
	);
	DesktopIconMoveLabelToIcon(
	    icon,
	    icon->icon_x, icon->icon_y
	);

	DesktopIconQueueDraw(icon);

	/* If the Desktop Icon should be mapped then map the
	 * label GtkWindow
	 */
	if(icon->flags & DESKTOP_ICON_MAPPED)
	{
	    if(icon->flags & DESKTOP_ICON_ALWAYS_ON_TOP)
	    {
		/* Map the GtkWindow and keep it above all the other
		 * windows
		 */
		gtk_widget_show_raise(icon->label_toplevel);
	    }
	    else
	    {
		/* Map the GtkWindow but keep it below all the other
		 * windows
		 */
		GtkWidget *w = icon->label_toplevel;
		gtk_widget_show(w);
		if(w->window != NULL)
		    gdk_window_lower(w->window);
	    }
	}

	g_free(dlabel);

	return(TRUE);				/* Change */
}


/*
 *	Gets the Desktop Icon data from the GtkWidget.
 */
static DesktopIcon *DesktopIconGetWidgetData(
	GtkWidget *w,
	const gchar *func_name
)
{
	const gchar *key = DESKTOP_ICON_KEY;
	DesktopIcon *icon;

	if(w == NULL)
	    return(NULL);

	icon = DESKTOP_ICON(gtk_object_get_data(
	    GTK_OBJECT(w),
	    key
	));
	if(icon == NULL)
	{
	    g_printerr(
"%s(): Warning: GtkWidget %p:\
 Unable to find the data that matches the key \"%s\".\n",
		func_name,
		w,
		key
	    );
	    return(NULL);
	}

	return(icon);
}


/*
 *	Creates a new Desktop Icon.
 */
GtkWidget *DesktopIconNew(
	const desktop_icon_flags flags,
	const gint x, const gint y,
	guint8 **icon_data,
	const gchar *icon_path,
	const gchar *label
)
{
	GtkWidget	*w,
			*parent;
	DesktopIcon *icon = DESKTOP_ICON(g_malloc0(
	    sizeof(DesktopIcon)
	));
	if(icon == NULL)
	    return(NULL);

/*	icon->freeze_count = 0; */
	icon->ref_count = 1;
/*	icon->gc = NULL; */
	icon->flags = flags;
	icon->flags &= ~DESKTOP_ICON_MAPPED;
	icon->flags &= ~DESKTOP_ICON_REALIZED;
	icon->flags &= ~DESKTOP_ICON_IS_MOVING;

	icon->last_icon_data = icon_data;
	icon->last_icon_path = STRDUP(icon_path);
/*
	icon->icon_pixmap = NULL;
	icon->icon_mask = NULL;
	icon->icon_x = 0;
	icon->icon_y = 0;
	icon->icon_width = 0;
	icon->icon_height = 0;
 */
	icon->label_str = STRDUP(label);
/*
	icon->menu = NULL;
	icon->button = 0;
	icon->drag_last_x = 0;
	icon->drag_last_y = 0;
	icon->activate_cb = NULL;
	icon->activate_data = NULL;
	icon->select_cb = NULL;
	icon->select_data = NULL;
	icon->unselect_cb = NULL;
	icon->unselect_data = NULL;
	icon->rename_cb = NULL;
	icon->rename_data = NULL;
	icon->move_begin_cb = NULL;
	icon->move_begin_data = NULL;
	icon->moving_cb = NULL;
	icon->moving_data = NULL;
	icon->moved_cb = NULL;
	icon->moved_data = NULL;
 */

	icon->freeze_count++;

	/* Toplevel GtkWindow */
	icon->icon_toplevel = w = gtk_window_new(GTK_WINDOW_POPUP);
	gtk_object_set_data_full(
	    GTK_OBJECT(w), DESKTOP_ICON_KEY,
	    icon, DesktopIconDestroyCB
	);
	gtk_window_set_policy(GTK_WINDOW(w), FALSE, FALSE, FALSE);
	gtk_widget_set_uposition(w, x, y);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_FOCUS | GTK_CAN_DEFAULT);
	gtk_widget_add_events(
	    w,
	    GDK_STRUCTURE_MASK | GDK_FOCUS_CHANGE_MASK |
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
	    GDK_POINTER_MOTION_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "configure_event",
	    GTK_SIGNAL_FUNC(DesktopIconEventCB), icon
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "focus_in_event",
	    GTK_SIGNAL_FUNC(DesktopIconEventCB), icon
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "focus_out_event",
	    GTK_SIGNAL_FUNC(DesktopIconEventCB), icon
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(DesktopIconEventCB), icon
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(DesktopIconEventCB), icon
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(DesktopIconEventCB), icon
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_release_event",
	    GTK_SIGNAL_FUNC(DesktopIconEventCB), icon
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "motion_notify_event",
	    GTK_SIGNAL_FUNC(DesktopIconEventCB), icon
	);
	parent = w;

	/* Icon GtkDrawingArea */
	icon->icon_da = w = gtk_drawing_area_new();
	gtk_widget_add_events(
	    w,
	    GDK_EXPOSURE_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "realize",
	    GTK_SIGNAL_FUNC(DesktopIconRealizeCB), icon
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "expose_event",
	    GTK_SIGNAL_FUNC(DesktopIconEventCB), icon
	);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);


	/* Label Toplevel GtkWindow */
	icon->label_toplevel = w = gtk_window_new(GTK_WINDOW_POPUP);
	gtk_window_set_policy(GTK_WINDOW(w), FALSE, FALSE, FALSE);
	gtk_widget_add_events(
	    w,
	    GDK_STRUCTURE_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
	    GDK_POINTER_MOTION_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "configure_event",
	    GTK_SIGNAL_FUNC(DesktopIconEventCB), icon
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(DesktopIconEventCB), icon
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_release_event",
	    GTK_SIGNAL_FUNC(DesktopIconEventCB), icon
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "motion_notify_event",
	    GTK_SIGNAL_FUNC(DesktopIconEventCB), icon
	);
	parent = w;

	/* Label GtkDrawingArea */
	icon->label_da = w = gtk_drawing_area_new();
	gtk_widget_add_events(
	    w,
	    GDK_EXPOSURE_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "realize",
	    GTK_SIGNAL_FUNC(DesktopIconLabelRealizeCB), icon
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "expose_event",
	    GTK_SIGNAL_FUNC(DesktopIconEventCB), icon
	);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);

	icon->freeze_count--;

	return(icon->icon_toplevel);
}


/*
 *	Gets the Desktop Icon's GtkDrawingArea.
 */
GtkWidget *DesktopIconGetIconWidget(GtkWidget *w)
{
	DesktopIcon *icon = DesktopIconGetWidgetData(
	    w,
	    "DesktopIconGetIconWidget"
	);
	if(icon == NULL)
	    return(NULL);

	return(icon->icon_da);
}


/*
 *	Moves the Desktop Icon.
 */
void DesktopIconSetPosition(
	GtkWidget *w,
	const gint x, const gint y
)
{
	gint	_x = x,
		_y = y,
		width, height,
		root_width, root_height;
	DesktopIcon *icon = DesktopIconGetWidgetData(
	    w,
	    "DesktopIconSetPosition"
	);
	if(icon == NULL)
	    return;

	/* Calculate the entire size of the Desktop Icon */
	width = icon->icon_width;
	height = icon->icon_height;
	if(icon->label_str != NULL)
	{
	    if(icon->label_width > width)
		width = icon->label_width;
	    height += icon->label_height;
	}

	/* Get the size of the desktop */
	root_width = icon->root_width;
	root_height = icon->root_height;

	/* Clip the position to stay within the desktop */
	if(_x >= (root_width - width))
	    _x = root_width - width - 1;
	if(_x < 0)
	    _x = 0;
	if(_y >= (root_height - height))
	    _y = root_height - height - 1;
	if(_y < 0)
	    _y = 0;

	/* No change in position? */
	if((icon->icon_x == _x) && (icon->icon_y == _y))
	    return;

	/* Set the icon GtkWindow's position as the specified position */
	gtk_widget_set_uposition(
	    w,
	    _x, _y
	);

	/* Record the new position of the icon GtkWindow
	 *
	 * Even though a "configure_event" will report it later on,
	 * the new position may need to be known right after this
	 * call
	 */
	icon->icon_x = _x;
	icon->icon_y = _y;

	/* Move the label GtkWindow to be aligned with the icon
	 * GtkWindow's new position
	 */
	DesktopIconMoveLabelToIcon(
	    icon,
	    _x, _y
	);

	/* Call the move callback */
	if(icon->moved_cb != NULL)
	    icon->moved_cb(
		w,				/* Desktop Icon */
		_x, _y,				/* Position */
		icon->moved_data		/* Data */
	    );
}

/*
 *	Gets the Desktop Icon's position.
 */
gboolean DesktopIconGetPosition(
	GtkWidget *w,
	gint *x_rtn, gint *y_rtn
)
{
	DesktopIcon *icon;

	if(x_rtn != NULL)
	    *x_rtn = 0;
	if(y_rtn != NULL)
	    *y_rtn = 0;

	icon = DesktopIconGetWidgetData(
	    w,
	    "DesktopIconGetPosition"
	);
	if(icon == NULL)
	    return(FALSE);

	if(x_rtn != NULL)
	    *x_rtn = icon->icon_x;
	if(y_rtn != NULL)
	    *y_rtn = icon->icon_y;

	return(TRUE);
}


/*
 *	Sets the Desktop Icon's icon and/or label.
 *
 *	The icon_data specifies the icon's XPM data. If icon_data is
 *	NULL then the icon will not be changed.
 *
 *	The label specifies the string describing the label. If label
 *	is NULL then the label will not be changed. If label is an
 *	empty string then the label will be cleared and not shown.
 */
void DesktopIconSet(
	GtkWidget *w,
	guint8 **icon_data,
	const gchar *label
)
{
	gboolean	icon_changed,
			label_changed;
	gchar *old_label_str;
	DesktopIcon *icon = DesktopIconGetWidgetData(
	    w,
	    "DesktopIconSet"
	);
	if(icon == NULL)
	    return;

	/* Record the previous label string */
	old_label_str = STRDUP(icon->label_str);

	/* Set the new icon and label and check for change */
	icon_changed = DesktopIconSetIcon(
	    icon,
	    icon_data,
	    NULL,
	    FALSE				/* Do not force change */
	);
	label_changed = DesktopIconSetLabel(
	    icon,
	    label,
	    FALSE				/* Do not force change */
	);

	/* Notify about the label change */
	if(label_changed && (icon->rename_cb != NULL))
	    icon->rename_cb(
		w,				/* Desktop Icon */
		old_label_str,			/* Old name */
		icon->label_str,		/* New name */
		icon->rename_data		/* Data */
	    );

	g_free(old_label_str);
}

/*
 *	Sets the Desktop Icon's icon and/or label.
 *
 *	The icon_path specifies the icon's XPM file. If icon_path is
 *	NULL then the icon will not be changed.
 *
 *	The label specifies the string describing the label. If label
 *	is NULL then the label will not be changed. If label is an
 *	empty string then the label will be cleared and not shown.
 */
void DesktopIconSetFile(
	GtkWidget *w,
	const gchar *icon_path,
	const gchar *label
)
{
	gboolean	icon_changed,
			label_changed;
	gchar *old_label_str;
	DesktopIcon *icon = DesktopIconGetWidgetData(
	    w,
	    "DesktopIconSet"
	);
	if(icon == NULL)
	    return;

	/* Record the previous label string */
	old_label_str = STRDUP(icon->label_str);

	/* Set the new icon and label and check for change */
	icon_changed = DesktopIconSetIcon(
	    icon,
	    NULL,
	    icon_path,
	    FALSE				/* Do not force change */
	);
	label_changed = DesktopIconSetLabel(
	    icon,
	    label,
	    FALSE				/* Do not force change */
	);

	/* Notify about the label change */
	if(label_changed && (icon->rename_cb != NULL))
	    icon->rename_cb(
		w,				/* Desktop Icon */
		old_label_str,			/* Old name */
		icon->label_str,		/* New name */
		icon->rename_data		/* Data */
	    );

	g_free(old_label_str);
}

/*
 *	Gets the Desktop Icon's current icon data and label string.
 */
gboolean DesktopIconGet(
	GtkWidget *w,
	guint8 ***icon_data_rtn,
        const gchar **icon_path_rtn,
	const gchar **label_rtn
)
{
	DesktopIcon *icon;

	if(icon_data_rtn != NULL)
	    *icon_data_rtn = NULL;
	if(label_rtn != NULL)
	    *label_rtn = NULL;

	icon = DesktopIconGetWidgetData(
	    w,
	    "DesktopIconGet"
	);
	if(icon == NULL)
	    return(FALSE);

	if(icon_data_rtn != NULL)
	    *icon_data_rtn = icon->last_icon_data;
	if(icon_path_rtn != NULL)
	    *icon_path_rtn = icon->last_icon_path;
	if(label_rtn != NULL)
	    *label_rtn = icon->label_str;

	return(TRUE);
}


/*
 *	Sets the Desktop Icon's sensitivity.
 */
void DesktopIconSetSensitive(
	GtkWidget *w,
	const gboolean sensitive
)
{
	DesktopIcon *icon = DesktopIconGetWidgetData(
	    w,
	    "DesktopIconSetSensitive"
	);
	if(icon == NULL)
	    return;

	if(sensitive && !GTK_WIDGET_SENSITIVE(w))
	{
	    gtk_widget_set_sensitive(icon->icon_toplevel, TRUE);
	    gtk_widget_set_sensitive(icon->label_toplevel, TRUE);
	    DesktopIconQueueDraw(icon);
	}
	else if(!sensitive && GTK_WIDGET_SENSITIVE(w))
	{
	    gtk_widget_set_sensitive(icon->icon_toplevel, FALSE);
	    gtk_widget_set_sensitive(icon->label_toplevel, FALSE);
	    DesktopIconQueueDraw(icon);
	}
}

/*
 *	Gets the Desktop Icon's sensitivity.
 */
gboolean DesktopIconGetSensitive(GtkWidget *w)
{
	DesktopIcon *icon = DesktopIconGetWidgetData(
	    w,
	    "DesktopIconGetSensitive"
	);
	if(icon == NULL)
	    return(FALSE);

	return(GTK_WIDGET_SENSITIVE(icon->icon_toplevel));
}


/*
 *	Sets the Desktop Icon as always on top or always below.
 *
 *	If always_on_top is TRUE then the Desktop Icon will always be
 *	above the other windows, otherwise it will always be below the
 *	other windows.
 */
void DesktopIconSetAlwaysOnTop(
	GtkWidget *w,
	const gboolean always_on_top
)
{
	DesktopIcon *icon = DesktopIconGetWidgetData(
	    w,
	    "DesktopIconSetAlwaysOnTop"
	);
	if(icon == NULL)
	    return;

	if(always_on_top && !(icon->flags & DESKTOP_ICON_ALWAYS_ON_TOP))
	{
	    icon->flags |= DESKTOP_ICON_ALWAYS_ON_TOP;
	    if(icon->flags & DESKTOP_ICON_MAPPED)
		DesktopIconRaise(icon);
	}
	else if(!always_on_top && (icon->flags & DESKTOP_ICON_ALWAYS_ON_TOP))
	{
	    icon->flags &= ~DESKTOP_ICON_ALWAYS_ON_TOP;
	    if(icon->flags & DESKTOP_ICON_MAPPED)
		DesktopIconLower(icon);
	}
}

/*
 *	Gets the Desktop Icon's always on top setting.
 */
gboolean DesktopIconGetAlwaysOnTop(GtkWidget *w)
{
	DesktopIcon *icon = DesktopIconGetWidgetData(
	    w,
	    "DesktopIconGetAlwaysOnTop"
	);
	if(icon == NULL)
	    return(FALSE);

	return((icon->flags & DESKTOP_ICON_ALWAYS_ON_TOP) ? TRUE : FALSE);
}


/*
 *	Gets the Desktop Icon's select state.
 */
gboolean DesktopIconGetSelected(GtkWidget *w)
{
	DesktopIcon *icon = DesktopIconGetWidgetData(
	    w,
	    "DesktopIconGetSelected"
	);
	if(icon == NULL)
	    return(FALSE);

	return(GTK_WIDGET_STATE(icon->icon_toplevel) == GTK_STATE_SELECTED);
}

/*
 *	Selects the Desktop Icon.
 */
void DesktopIconSelect(GtkWidget *w)
{
	DesktopIcon *icon = DesktopIconGetWidgetData(
	    w,
	    "DesktopIconSelect"
	);
	if(icon == NULL)
	    return;

	if(!(icon->flags & DESKTOP_ICON_CAN_SELECT))
	    return;

	if(DesktopIconGetSelected(icon->icon_toplevel))
	    return;

	gtk_widget_set_state(icon->icon_toplevel, GTK_STATE_SELECTED);
	gtk_widget_set_state(icon->label_toplevel, GTK_STATE_SELECTED);

	DesktopIconQueueDraw(icon);

	/* Call the select callback */
	if(icon->select_cb != NULL)
	    icon->select_cb(
		w,			/* Desktop Icon */
		icon->select_data	/* Data */
	    );
}

/*
 *	Unselects the Desktop Icon.
 */
void DesktopIconUnselect(GtkWidget *w)
{
	DesktopIcon *icon = DesktopIconGetWidgetData(
	    w,
	    "DesktopIconUnselect"
	);
	if(icon == NULL)
	    return;

	if(!(icon->flags & DESKTOP_ICON_CAN_SELECT))
	    return;

	if(!DesktopIconIsSelected(w))
	    return;

	gtk_widget_set_state(icon->icon_toplevel, GTK_STATE_NORMAL);
	gtk_widget_set_state(icon->label_toplevel, GTK_STATE_NORMAL);

	DesktopIconQueueDraw(icon);

	/* Notify about the unselect */
	if(icon->unselect_cb != NULL)
	    icon->unselect_cb(
		w,				/* Desktop Icon */
		icon->unselect_data		/* Data */
	    );
}


/*
 *	Sets the Desktop Icon's GtkMenu.
 *
 *	If there was a previous GtkMenu set and it is not the same as
 *	the new GtkMenu then the previous GtkMenu will be destroyed.
 *
 *	If the new GtkMenu is NULL then the previous GtkMenu will be
 *	destroyed.
 */
void DesktopIconSetMenu(
	GtkWidget *w,
	GtkWidget *menu
)
{
	DesktopIcon *icon = DesktopIconGetWidgetData(
	    w,
	    "DesktopIconSetMenu"
	);
	if(icon == NULL)
	    return;

	/* is the new menu not valid? */
	if(menu != NULL)
	{
	    if(!GTK_IS_MENU(menu))
		return;
	}

	/* No change in menu? */
	if(icon->menu == menu)
	    return;

	/* Destroy the previous menu */
	if(icon->menu != NULL)
	    gtk_widget_destroy(icon->menu);

	/* Set the new menu (which may be NULL) */
	icon->menu = menu;
}

/*
 *	Gets the Desktop Icon's current GtkMenu.
 */
GtkWidget *DesktopIconGetMenu(GtkWidget *w)
{
	DesktopIcon *icon = DesktopIconGetWidgetData(
	    w,
	    "DesktopIconGetMenu"
	);
	if(icon == NULL)
	    return(NULL);

	return(icon->menu);
}


/*
 *	Sets the Desktop Icon's map reference callback function.
 */
void DesktopIconSetMapRefCB(
	GtkWidget *w,
	void (*activate_cb)(GtkWidget *, gpointer),
	gpointer data
)
{
	DesktopIcon *icon = DesktopIconGetWidgetData(
	    w,
	    "DesktopIconSetMapRefCB"
	);
	if(icon == NULL)
	    return;

	icon->activate_cb = activate_cb;
	icon->activate_data = data;
}

/*
 *      Sets the Desktop Icon's select callback function.
 */
void DesktopIconSetSelectCB(
	GtkWidget *w,
	void (*select_cb)(GtkWidget *, gpointer),
	gpointer data
)
{
	DesktopIcon *icon = DesktopIconGetWidgetData(
	    w,
	    "DesktopIconSetSelectCB"
	);
	if(icon == NULL)
	    return;

	icon->select_cb = select_cb;
	icon->select_data = data;
}

/*
 *	Sets the Desktop Icon's unselect callback function.
 */
void DesktopIconSetUnselectCB(
	GtkWidget *w,
	void (*unselect_cb)(GtkWidget *, gpointer),
	gpointer data
)
{
	DesktopIcon *icon = DesktopIconGetWidgetData(
	    w,
	    "DesktopIconSetUnselectCB"
	);
	if(icon == NULL)
	    return;

	icon->unselect_cb = unselect_cb;
	icon->unselect_data = data;
}

/*
 *	Sets the Desktop Icon's rename callback function.
 */
void DesktopIconSetRenameCB(
	GtkWidget *w,
	void (*rename_cb)(GtkWidget *, const gchar *, const gchar *, gpointer),
	gpointer data
)
{
	DesktopIcon *icon = DesktopIconGetWidgetData(
	    w,
	    "DesktopIconSetRenameCB"
	);
	if(icon == NULL)
	    return;

	icon->rename_cb = rename_cb;
	icon->rename_data = data;
}

/*
 *	Sets the Desktop Icon's move begin callback function.
 */
void DesktopIconSetMoveBeginCB(
	GtkWidget *w,
	void (*move_begin_cb)(GtkWidget *, const gint, const gint, gpointer),
	gpointer data
)
{
	DesktopIcon *icon = DesktopIconGetWidgetData(
	    w,
	    "DesktopIconSetMoveBeginCB"
	);
	if(icon == NULL)
	    return;

	icon->move_begin_cb = move_begin_cb;
	icon->move_begin_data = data;
}

/*
 *	Sets the Desktop Icon's moving callback function.
 */
void DesktopIconSetMovingCB(
	GtkWidget *w,
	void (*moving_cb)(GtkWidget *, const gint, const gint, gpointer),
	gpointer data
)
{
	DesktopIcon *icon = DesktopIconGetWidgetData(
	    w,
	    "DesktopIconSetMovingCB"
	);
	if(icon == NULL)
	    return;

	icon->moving_cb = moving_cb;
	icon->moving_data = data;
}

/*
 *	Sets the Desktop Icon's moved callback function.
 */
void DesktopIconSetMovedCB(
	GtkWidget *w,
	void (*moved_cb)(GtkWidget *, const gint, const gint, gpointer),
	gpointer data
)
{
	DesktopIcon *icon = DesktopIconGetWidgetData(
	    w,
	    "DesktopIconSetMovedCB"
	);
	if(icon == NULL)
	    return;

	icon->moved_cb = moved_cb;
	icon->moved_data = data;
}


/*
 *	Checks if the Desktop Icon is mapped.
 */
gboolean DesktopIconIsMapped(GtkWidget *w)
{
	DesktopIcon *icon = DesktopIconGetWidgetData(
	    w,
	    "DesktopIconIsMapped"
	);
	if(icon == NULL)
	    return(FALSE);

	return((icon->flags & DESKTOP_ICON_MAPPED) ? TRUE : FALSE);
}

/*
 *	Shows and raises the Desktop Icon, keeping it above all
 *	theo ther windows.
 */
static void DesktopIconRaise(DesktopIcon *icon)
{
	gtk_widget_show_raise(icon->icon_toplevel);
	if(icon->label_str != NULL)
	    gtk_widget_show_raise(icon->label_toplevel);
}

/*
 *	Shows and lowers the Desktop Icon, keeping it below all the
 *	other windows.
 */
static void DesktopIconLower(DesktopIcon *icon)
{
	GtkWidget *w = icon->icon_toplevel;
	gtk_widget_show(w);
	if(w->window != NULL)
	    gdk_window_lower(w->window);

	if(icon->label_str != NULL)
	{
	    GtkWidget *w = icon->label_toplevel;
	    gtk_widget_show(w);
	    if(w->window != NULL)
		gdk_window_lower(w->window);
	}
}

/*
 *	Maps the Desktop Icon.
 */
void DesktopIconMap(GtkWidget *w)
{
	DesktopIcon *icon = DesktopIconGetWidgetData(
	    w,
	    "DesktopIconMap"
	);
	if(icon == NULL)
	    return;

	if(icon->flags & DESKTOP_ICON_MAPPED)
	    return;

	if(icon->flags & DESKTOP_ICON_ALWAYS_ON_TOP)
	    DesktopIconRaise(icon);
	else
	    DesktopIconLower(icon);

	icon->flags |= DESKTOP_ICON_MAPPED;
}

/*
 *	Unmaps the Desktop Icon.
 */
void DesktopIconUnmap(GtkWidget *w)
{
	DesktopIcon *icon = DesktopIconGetWidgetData(
	    w,
	    "DesktopIconUnmap"
	);
	if(icon == NULL)
	    return;

	if(!(icon->flags & DESKTOP_ICON_MAPPED))
	    return;

	gtk_widget_hide(icon->icon_toplevel);
	gtk_widget_hide(icon->label_toplevel);

	icon->flags &= ~DESKTOP_ICON_MAPPED;
}

/*
 *	Adds a reference count to the Desktop Icon.
 *
 *	Returns the Desktop Icon.
 */
GtkWidget *DesktopIconRef(GtkWidget *w)
{
	DesktopIcon *icon = DesktopIconGetWidgetData(
	    w,
	    "DesktopIconRef"
	);
	if(icon == NULL)
	    return(NULL);

	icon->ref_count++;

	return(w);
}

/*
 *	Removes a reference count from the Desktop Icon.
 *
 *	If the reference counts reach 0 then the Desktop Icon is
 *	destroyed.
 *
 *	This function always returns NULL.
 */
GtkWidget *DesktopIconUnref(GtkWidget *w)
{
	DesktopIcon *icon = DesktopIconGetWidgetData(
	    w,
	    "DesktopIconUnref"
	);
	if(icon == NULL)
	    return(NULL);

	icon->ref_count--;

	/* All reference counts removed? */
	if(icon->ref_count <= 0)
	{
	    DesktopIconUnmap(w);

	    /* Destroy the icon toplevel GtkWindow and call the
	     * "destroy" callback
	     */
	    gtk_widget_destroy(icon->icon_toplevel);
	}

	return(NULL);
}
