#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include "guiutils.h"
#include "csd.h"


typedef struct _ColorSelectionDialog	ColorSelectionDialog;
#define COLOR_SELECTION_DIALOG(p)	((ColorSelectionDialog *)(p))

typedef struct _ColorButton		ColorButton;
#define COLOR_BUTTON(p)			((ColorButton *)(p))
#define COLOR_BUTTON_DATA_KEY		"/CSD/ColorButton"


/*
 *	Flags:
 */
typedef enum {
	CSD_MAPPED			= (1 << 0),
	CSD_REALIZED			= (1 << 1),
	CSD_RESPONSE_OK			= (1 << 8)
} CSDFlags;


/* Callbacks */
static gint CSDDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void CSDRealizeCB(GtkWidget *widget, gpointer data);
static void CSDOKCB(GtkWidget *widget, gpointer data);
static void CSDCancelCB(GtkWidget *widget, gpointer data);
static void CSDColorChangedCB(GtkWidget *widget, gpointer data);


/* Color Selection Dialog */
gint CSDInit(void);
void CSDSetStyle(GtkRcStyle *rc_style);
void CSDSetTransientFor(GtkWidget *w);
gboolean CSDIsQuery(void);
void CSDBreakQuery(void);
gboolean CSDGetResponse(
	const gchar *title,
	const gchar *ok_label, const gchar *cancel_label,
	csd_color_struct *start_color,
	csd_color_struct **color_rtn,
	void (*color_changed_cb)(csd_color_struct *, gpointer),
	gpointer color_changed_data
);
void CSDMap(void);
void CSDUnmap(void);
void CSDShutdown(void);


/* Color Button */
static void CSDColorButtonDestroyCB(gpointer data);
static void CSDColorButtonRealizeCB(GtkWidget *widget, gpointer data);
static void CSDColorButtonSelectCB(GtkWidget *widget, gpointer data);
static void CSDColorButtonColorChangedCB(csd_color_struct *c, gpointer data);
static void CSDColorButtonRealizeColor(ColorButton *b);
GtkWidget *CSDColorButtonNew(
	const gchar *label,
	const gint label_width,
	void (*cb)(GtkWidget *, gpointer),
	gpointer data
);
GtkWidget *CSDColorButtonNewSimple(
	const gchar *label,
	const gint label_width
);
GtkWidget *CSDColorButtonGetButton(GtkWidget *w);
void CSDColorButtonGetColor(
	GtkWidget *w,
	csd_color_struct *c
);
void CSDColorButtonSetColor(
	GtkWidget *w,
	const csd_color_struct *c
);


#define COLOR_BUTTON_WIDTH		20
#define COLOR_BUTTON_HEIGHT		20


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Color Selection Dialog:
 */
struct _ColorSelectionDialog {

	GtkWidget	*csd;		/* GtkColorSelectionDialog */
	gint		freeze_count,
			main_level;
	CSDFlags	flags;

	csd_color_struct	current_color,
				response_color;

	void		(*color_changed_cb)(
		csd_color_struct *,		/* Color */
		gpointer			/* Data */
	);
	gpointer	color_changed_data;

};

static ColorSelectionDialog	*color_selection_dialog = NULL;


/*
 *	Color Button:
 */
struct _ColorButton {

	GtkWidget	*toplevel,
			*label,
			*button,
			*frame,
			*drawing_area;

	gchar		*label_text;

	csd_color_struct	color;

};


/*
 *	GtkColorSelectionDialog "delete_event" signal callback.
 */
static gint CSDDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	CSDCancelCB(widget, data);
	return(TRUE);
}

/*
 *	ColorSelectionDialog "realize" signal callback.
 */
static void CSDRealizeCB(GtkWidget *widget, gpointer data)
{
	GdkWindow *window;
	ColorSelectionDialog *csdd = COLOR_SELECTION_DIALOG(data);
	if((widget == NULL) || (csdd == NULL))
	    return;

	window = widget->window;
	if(window != NULL)
	{
	    gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_TITLE
	    );
	    gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_CLOSE
	    );
	}

	/* Mark the Color Selection Dialog as realized */
	csdd->flags |= CSD_REALIZED;
}

/*
 *	Ok callback.
 */
static void CSDOKCB(GtkWidget *widget, gpointer data)
{
	ColorSelectionDialog *csdd = COLOR_SELECTION_DIALOG(data);
	if(csdd == NULL)
	    return;

	/* Set the response OK flag */
	csdd->flags |= CSD_RESPONSE_OK;

	/* Copy the response color */
	(void)memcpy(
	    &csdd->response_color,
	    &csdd->current_color,
	    sizeof(csd_color_struct)
	);

	/* Unmap */
	CSDUnmap();

	/* Break out of one main level */
	if(csdd->main_level > 0)
	{
	    gtk_main_quit();
	    csdd->main_level--;
	}
}

/*
 *	Cancel callback.
 */
static void CSDCancelCB(GtkWidget *widget, gpointer data)
{
	ColorSelectionDialog *csdd = COLOR_SELECTION_DIALOG(data);
	if(csdd == NULL)
	    return;

	/* Clear the response OK flag */
	csdd->flags &= ~CSD_RESPONSE_OK;

	/* Unmap */
	CSDUnmap();

	/* Break out of one main level */
	if(csdd->main_level > 0)
	{
	    gtk_main_quit();
	    csdd->main_level--;
	}
}           

/*
 *	GtkColorSelectionDialog "color_changed" signal callback.
 */
static void CSDColorChangedCB(GtkWidget *widget, gpointer data)
{
	gdouble vd[3];
	GtkColorSelectionDialog *csd;
	csd_color_struct *tc;
	ColorSelectionDialog *csdd = COLOR_SELECTION_DIALOG(data);
	if(csdd == NULL)
	    return;

	csd = GTK_COLOR_SELECTION_DIALOG(csdd->csd);
	if(csd == NULL)
	    return;

	/* Get the current color */
	gtk_color_selection_get_color(
	    GTK_COLOR_SELECTION(csd->colorsel), vd
	);

	/* Copy current color values to csdd structure's
	 * csd_color_struct.
	 */
	tc = &csdd->current_color;
	tc->r = (gfloat)vd[0];
	tc->g = (gfloat)vd[1];
	tc->b = (gfloat)vd[2];
	tc->a = 1.0f;

	/* Notify about the color change */
	if(csdd->color_changed_cb != NULL)
	    csdd->color_changed_cb(
		&csdd->current_color,
		csdd->color_changed_data
	    );
}


/*
 *	Initializes color selection dialog..
 */
gint CSDInit(void)
{
	GtkWidget *w;
	GtkColorSelectionDialog *csd;
	ColorSelectionDialog *csdd;

        /* Already initialized? */
        if(color_selection_dialog != NULL)
            return(0);

        /* Create a new Color Selection Dialog */
	color_selection_dialog = csdd = COLOR_SELECTION_DIALOG(g_malloc0(
            sizeof(ColorSelectionDialog)
        ));
        if(csdd == NULL)
            return(-3);

/*
	csdd->freeze_count = 0;
	csdd->main_level = 0;
	csdd->flags = 0;
	memset(
	    &csdd->current_color,
	    0x00,
	    sizeof(csd_color_struct)
	);
	memset(
	    &csdd->response_color,
	    0x00,
	    sizeof(csd_color_struct)
	);
 */

	csdd->freeze_count++;

	/* Create the GtkColorSelectionDialog */
	csdd->csd = w = gtk_color_selection_dialog_new("Select Color");
	csd = GTK_COLOR_SELECTION_DIALOG(w);
	gtk_window_set_policy(GTK_WINDOW(w), FALSE, FALSE, FALSE);
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(CSDDeleteEventCB), csdd
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "realize",
	    GTK_SIGNAL_FUNC(CSDRealizeCB), csdd
	);

	/* Get the GtkColorSelection */
	w = csd->colorsel;

	/* Connect "color_changed" signal to the GtkColorSelection
	 * widget.
	 */
	gtk_signal_connect(
	    GTK_OBJECT(w), "color_changed",
	    GTK_SIGNAL_FUNC(CSDColorChangedCB), (gpointer)csdd
	);

	/* OK GtkButton */
	gtk_signal_connect(
	    GTK_OBJECT(csd->ok_button), "clicked",
	    GTK_SIGNAL_FUNC(CSDOKCB), (gpointer)csdd
	);
	gtk_widget_show(csd->ok_button);

	/* Hide the Reset GtkButton */
	if(csd->reset_button != NULL)
	    gtk_widget_hide(csd->reset_button);

	/* Cancel GtkButton */
	gtk_signal_connect(
	    GTK_OBJECT(csd->cancel_button), "clicked",
	    GTK_SIGNAL_FUNC(CSDCancelCB), (gpointer)csdd
	);
	gtk_widget_show(csd->cancel_button);

	/* Hide the Help GtkButton */
	if(csd->help_button != NULL)
	    gtk_widget_hide(csd->help_button);

	csdd->freeze_count--;

	return(0);
}

/*
 *      Sets the Dialog's style.
 */
void CSDSetStyle(GtkRcStyle *rc_style)
{
	GtkWidget *w;
	ColorSelectionDialog *csdd = color_selection_dialog;
	if(csdd == NULL)
	    return;

	w = csdd->csd;
	if(w != NULL)
	{
	    if(rc_style != NULL)
	    {
		gtk_widget_modify_style(w, rc_style);
	    }
	    else
	    {
		rc_style = gtk_rc_style_new();
		gtk_widget_modify_style_recursive(w, rc_style);
		GTK_RC_STYLE_UNREF(rc_style)
	    }
	}
}

/*
 *      Sets the Dialog to be a transient for the given GtkWindow w.
 *
 *      If w is NULL then transient for will be unset.
 */
void CSDSetTransientFor(GtkWidget *w)
{
	ColorSelectionDialog *csdd = color_selection_dialog;
	if(csdd == NULL)
	    return;

	if(csdd->csd != NULL)
	{
	    if(w != NULL)
	    {
		/* Given widget if not NULL, must be a window */
		if(!GTK_IS_WINDOW(GTK_OBJECT(w)))
		    return;

		gtk_window_set_modal(
		    GTK_WINDOW(csdd->csd),
		    TRUE
		);
		gtk_window_set_transient_for(
		    GTK_WINDOW(csdd->csd),
		    GTK_WINDOW(w)
		);
	    }
	    else
	    {
		gtk_window_set_modal(
		    GTK_WINDOW(csdd->csd),
		    FALSE
		);
		gtk_window_set_transient_for(
		    GTK_WINDOW(csdd->csd),
		    NULL
		);
	    }
	}
}


/*
 *      Returns TRUE if currently blocking for query.
 */
gboolean CSDIsQuery(void)
{
	ColorSelectionDialog *csdd = color_selection_dialog;
	if(csdd == NULL)
	    return(FALSE);

	return((csdd->main_level > 0) ? TRUE : FALSE);
}

/*
 *	Ends query if any and returns a not available response.
 */
void CSDBreakQuery(void)
{
	ColorSelectionDialog *csdd = color_selection_dialog;
	if(csdd == NULL)
	    return;

	csdd->flags &= ~CSD_RESPONSE_OK;

	/* Break out of all the main levels */
	while(csdd->main_level > 0)
	{
	    gtk_main_quit();
	    csdd->main_level--;
	}
	csdd->main_level = 0;
}

/*
 *	Maps the color selection dialog and sets up the inital values.
 *
 *	Returns TRUE if a color was sselected or FALSE if user canceled.
 *
 *	For most values that are set NULL, the value is left unchanged.
 *	All given values are coppied.
 *
 *	All returned pointer values should be considered statically
 *	allocated, do not deallocate them.
 */
gboolean CSDGetResponse(
	const gchar *title,
	const gchar *ok_label, const gchar *cancel_label,
	csd_color_struct *start_color,
	csd_color_struct **color_rtn,
	void (*color_changed_cb)(csd_color_struct *, gpointer),
	gpointer color_changed_data
)
{
	GtkWidget *w;
	GtkColorSelectionDialog *csd;
	ColorSelectionDialog *csdd = color_selection_dialog;

	if(color_rtn != NULL)
	    *color_rtn = NULL;

	if(csdd == NULL)
	    return(FALSE);

	/* Already waiting for a response? */
	if(csdd->main_level > 0)
	    return(FALSE);

	/* Reset the response values */
	csdd->flags &= ~CSD_RESPONSE_OK;
	memset(
	    &csdd->response_color,
	    0x00,
	    sizeof(csd_color_struct)
	);

	/* Get the GtkColorSelectionDialog */
	w = csdd->csd;
	if(w == NULL)
	    return(FALSE);

	csd = GTK_COLOR_SELECTION_DIALOG(w);

	/* Update title if specified */
	if(title != NULL)
	    gtk_window_set_title(GTK_WINDOW(w), title);

	/* Update button labels */
	if((ok_label != NULL) && (csd->ok_button != NULL))
	{
	    GtkButton *button = (GtkButton *)csd->ok_button;
	    GtkWidget *w2;

	    w2 = GTK_BIN(button)->child;
	    if((w2 != NULL) ? GTK_IS_LABEL(w2) : 0)
		gtk_label_set_text(GTK_LABEL(w2), ok_label);
	}
	if((cancel_label != NULL) && (csd->cancel_button != NULL))
	{
	    GtkButton *button = (GtkButton *)csd->cancel_button;
	    GtkWidget *w2;

	    w2 = GTK_BIN(button)->child;
	    if((w2 != NULL) ? GTK_IS_LABEL(w2) : 0)
		gtk_label_set_text(GTK_LABEL(w2), cancel_label);
	}


	/* Update initial start color if specified */
	if(start_color != NULL)
	{
	    gdouble vd[3];
	    GtkColorSelection *colorsel;

	    colorsel = GTK_COLOR_SELECTION(
		GTK_COLOR_SELECTION_DIALOG(w)->colorsel
	    );

	    vd[0] = start_color->r;
	    vd[1] = start_color->g;
	    vd[2] = start_color->b;
/*		  = start_color->a; */

	    gtk_color_selection_set_color(colorsel, vd);

	    csdd->current_color.r = start_color->r;
	    csdd->current_color.g = start_color->g;
	    csdd->current_color.b = start_color->b;
	    csdd->current_color.a = start_color->a;
	}

	/* Set up the callbacks */
	csdd->color_changed_cb = color_changed_cb;
	csdd->color_changed_data = color_changed_data;

	/* Map the dialog */
	CSDMap();

	/* Wait for user response */
	csdd->main_level++;
	gtk_main();

	/* Unmap the dialog */
	CSDUnmap();

	/* Break out of any remaining main levels */
	while(csdd->main_level > 0)
	{
	    gtk_main_quit();
	    csdd->main_level--;
	}
	csdd->main_level = 0;


	/* Begin setting returns */

	/* Color return */
	if(color_rtn != NULL)
	    *color_rtn = &csdd->response_color;

	return((csdd->flags & CSD_RESPONSE_OK) ? TRUE : FALSE);
}


/*
 *	Maps the Color Selection Dialog.
 */
void CSDMap(void)
{
	ColorSelectionDialog *csdd = color_selection_dialog;
	if(csdd == NULL)
	    return;

	gtk_widget_show_raise(csdd->csd);
	csdd->flags |= CSD_MAPPED;
}

/*
 *	Unmaps the Color Selection Dialog.
 */
void CSDUnmap(void)
{
	ColorSelectionDialog *csdd = color_selection_dialog;
	if(csdd == NULL)
	    return;

	gtk_widget_hide(csdd->csd);
	csdd->flags &= ~CSD_MAPPED;
}

/*
 *	Shuts down the Color Selection Dialog.
 */
void CSDShutdown(void)
{
	ColorSelectionDialog *csdd = color_selection_dialog;
	if(csdd == NULL)
	    return;

	csdd->flags &= ~CSD_RESPONSE_OK;

	/* Break out of any remaining main levels */
	while(csdd->main_level > 0)
	{
	    gtk_main_quit();
	    csdd->main_level--;
	}
	csdd->main_level = 0;

	CSDUnmap();

	/* Mark the Color Selection Dialog as shut down */
	color_selection_dialog = NULL;

	csdd->freeze_count++;

	GTK_WIDGET_DESTROY(csdd->csd);

	csdd->freeze_count--;

	g_free(csdd);
}


/*
 *	Color Button "destroy" signal callback.
 */
static void CSDColorButtonDestroyCB(gpointer data)
{
	ColorButton *b = COLOR_BUTTON(data);
	if(b == NULL)
	    return;

	g_free(b->label_text);
	g_free(b);
}

/*
 *	Color Button GtkDrawingArea "realize" signal callback.
 */
static void CSDColorButtonRealizeCB(GtkWidget *widget, gpointer data)
{
	ColorButton *b = COLOR_BUTTON(data);
	if((widget == NULL) || (b == NULL))
	    return;

	/* Realize the GtkDrawingArea's color */
	CSDColorButtonRealizeColor(b);
}

/*
 *	Color Button select signal callback.
 */
static void CSDColorButtonSelectCB(GtkWidget *widget, gpointer data)
{
	gboolean response;
	GtkWidget *w, *toplevel;
	csd_color_struct c, *c_rtn = NULL;
	ColorButton *b = COLOR_BUTTON(data);
	if((b == NULL) || CSDIsQuery())
	    return;

	w = b->toplevel;
	toplevel = gtk_widget_get_toplevel(w);

	/* Record the current color */
	memcpy(
	    &c,
	    &b->color,
	    sizeof(csd_color_struct)
	);

	/* Need to restore the color button's GtkStateType to
	 * GTK_STATE_NORMAL because the "released" signal will
	 * not get to the color button while the user is being
	 * queried to select a color
	 */
	gtk_widget_set_state(w, GTK_STATE_NORMAL);

	/* Query the user to select a color */
	CSDSetTransientFor(toplevel);
	response = CSDGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
	    "Escoja El Color",
	    "Escoja", "Cancele",
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Couleur Privilgie",
	    "Privilgi", "Annuler",
#elif defined(PROG_LANGUAGE_GERMAN)
	    "Wahlen Sie Farbe Aus",
	    "Wahlen", "Heben",
#elif defined(PROG_LANGUAGE_ITALIAN)
	    "Scegliere Il Colore",
	    "Sceglier", "Annullar",
#elif defined(PROG_LANGUAGE_DUTCH)
	    "Selecteer De Kleur",
	    "Selecter", "Annuler",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	    "Selecione Cor",
	    "Selecion", "Cancela",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	    "Velg Ut Farge",
	    "Velg", "Kanseler",
#else
	    "Select Color",
	    "Select", "Cancel",
#endif
	    &c,					/* Current color */
	    &c_rtn,				/* Return color */
	    CSDColorButtonColorChangedCB, w
	);
	CSDSetTransientFor(NULL);

	/* Got response? */
	if(response && (c_rtn != NULL))
	{
	    CSDColorButtonSetColor(w, c_rtn);
	}
	else
	{
	    CSDColorButtonSetColor(w, &c);
	}
}

/*
 *	Color button color changed callback.
 */
static void CSDColorButtonColorChangedCB(csd_color_struct *c, gpointer data)
{
	GtkWidget *w = (GtkWidget *)data;
	if((c == NULL) || (w == NULL))
	    return;

	/* Update the color button's color */
	CSDColorButtonSetColor(w, c);
}

/*
 *	Sets the Color Button's current color to its GtkDrawingArea's
 *	GdkWindow.
 */
static void CSDColorButtonRealizeColor(ColorButton *b)
{
	GdkColor gdk_c;
	csd_color_struct *c;
	GtkWidget *w = b->drawing_area;
	GdkColormap *colormap = gtk_widget_get_colormap(w);
	if(colormap == NULL)
	    return;

	/* Convert/allocate the GdkColor from the Color Button's
	 * current color
	 */
	c = &b->color;
	gdk_c.red	= (gushort)(c->r * (gushort)-1);
	gdk_c.green	= (gushort)(c->g * (gushort)-1);
	gdk_c.blue	= (gushort)(c->b * (gushort)-1);
	GDK_COLORMAP_ALLOC_COLOR(colormap, &gdk_c);

	/* Set the GdkColor to the GtkDrawingArea's GdkWindow */
	gdk_window_set_background(w->window, &gdk_c);

	/* Delete the GdkColor */
	GDK_COLORMAP_FREE_COLOR(colormap, &gdk_c);

	/* Get the GdkWindow to redraw */
	gdk_window_clear(w->window);
}


/*
 *	Creates a new color button.
 */
GtkWidget *CSDColorButtonNew(
	const gchar *label,
	const gint label_width,
	void (*cb)(GtkWidget *, gpointer),
	gpointer data
)
{
	const gint border_minor = 2;
	GtkWidget *w, *parent, *parent2;
	ColorButton *b = COLOR_BUTTON(g_malloc0(sizeof(ColorButton)));
	if(b == NULL)
	    return(NULL);

	/* Toplevel GtkHBox */
	b->toplevel = w = gtk_hbox_new(FALSE, border_minor);
	gtk_object_set_data_full(
	    GTK_OBJECT(w), COLOR_BUTTON_DATA_KEY,
	    b, CSDColorButtonDestroyCB
	);
	parent = w;

	/* Create a label? */
	if(!STRISEMPTY(label))
	{
	    w = gtk_alignment_new(
		1.0f, 0.5f,
		0.0f, 0.0f
	    );
	    gtk_widget_set_usize(w, label_width, -1);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent2 = w;

	    /* GtkLabel */
	    b->label = w = gtk_label_new(label);
	    b->label_text = STRDUP(label);
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	    gtk_container_add(GTK_CONTAINER(parent2), w);
	    gtk_widget_show(w);
	}

	/* GtkButton */
	b->button = w = gtk_button_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* GtkFrame */
	b->frame = w = gtk_frame_new(NULL);
	gtk_widget_set_usize(
	    w,
	    COLOR_BUTTON_WIDTH, COLOR_BUTTON_HEIGHT
	);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);
	parent2 = w;

	b->drawing_area = w = gtk_drawing_area_new();
	gtk_widget_add_events(
	    w,
	    GDK_EXPOSURE_MASK | GDK_STRUCTURE_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
	    GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "realize",
	    GTK_SIGNAL_FUNC(CSDColorButtonRealizeCB), b
	);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	GUISetWidgetTip(
	    w,
#if defined(PROG_LANGUAGE_SPANISH)
"El clic para escoger el color"
#elif defined(PROG_LANGUAGE_FRENCH)
"Le dclic pour choisir la couleur"
#elif defined(PROG_LANGUAGE_GERMAN)
"Klicken, Farbe auszuwhlen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Lo scatto di scegliere il colore"
#elif defined(PROG_LANGUAGE_DUTCH)
"Klik kleur te selecteren"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O estalido selecionar cor"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Klikk velge ut farge"
#else
"Click to select color"
#endif
	);
	gtk_widget_show(w);

	if(cb != NULL)
	    gtk_signal_connect(
		GTK_OBJECT(b->button), "clicked",
		GTK_SIGNAL_FUNC(cb), data
	    );

	return(b->toplevel);
}

/*
 *	Creates a new color button with the "clicked" signal already
 *	connected to a default callback which prompts for a new color
 *	using the color selection dialog and sets the new color for
 *	the color button as appropriate.
 */
GtkWidget *CSDColorButtonNewSimple(
	const gchar *label,
	const gint label_width)
{
	GtkWidget *w = CSDColorButtonNew(
	    label,
	    label_width,
	    NULL, NULL
	);
	ColorButton *b;

	if(w == NULL)
	    return(NULL);

	b = COLOR_BUTTON(GTK_OBJECT_GET_DATA(w, COLOR_BUTTON_DATA_KEY));
	if(b == NULL)
	    return(w);

	if(b->button != NULL)
	    gtk_signal_connect(
		GTK_OBJECT(b->button), "clicked",
		GTK_SIGNAL_FUNC(CSDColorButtonSelectCB), b
	    );

	return(w);
}

/*
 *	Returns the GtkButton of the given color button.
 */
GtkWidget *CSDColorButtonGetButton(GtkWidget *w)
{
	ColorButton *b = COLOR_BUTTON(GTK_OBJECT_GET_DATA(
	    w,
	    COLOR_BUTTON_DATA_KEY
	));
	if(b == NULL)
	    return(NULL);

	return(b->button);
}

/*
 *	Gets the color button's current color.
 */
void CSDColorButtonGetColor(
	GtkWidget *w,
	csd_color_struct *c
)
{
	ColorButton *b = COLOR_BUTTON(GTK_OBJECT_GET_DATA(
	    w,
	    COLOR_BUTTON_DATA_KEY
	));
	if((b == NULL) || (c == NULL))
	    return;

	if(c != &b->color)
	    memcpy(
		c,
		&b->color,
		sizeof(csd_color_struct)
	    );
}

/*
 *	Sets the color button's color.
 */
void CSDColorButtonSetColor(
	GtkWidget *w,
	const csd_color_struct *c
)
{
	ColorButton *b = COLOR_BUTTON(GTK_OBJECT_GET_DATA(
	    w,
	    COLOR_BUTTON_DATA_KEY
	));
	if((b == NULL) || (c == NULL))
	    return;

	/* Record the specified color */
	if(c != &b->color)
	    memcpy(
		&b->color,
		c,
		sizeof(csd_color_struct)
	    );

	/* Has the color button been parented? */
	w = b->toplevel;
	if(w->parent != NULL)
	{
	    /* Get the GtkDrawingArea and realize it as needed */
	    w = b->drawing_area;
	    if(GTK_WIDGET_REALIZED(w))
	    {
		/* The GtkDrawingArea is already realized, so
		 * just realize the color
		 */
		CSDColorButtonRealizeColor(b);
	    }
	    else
	    {
		/* Realize the GtkDrawingArea and its color */
		gtk_widget_realize(w);
	    }
	}
}
