#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>

#include "cfg.h"

#include "guiutils.h"
#include "stacklist.h"
#include "style_edit.h"
#include "keymap_list.h"
#include "menucfg_list.h"
#include "csd.h"

#include "cfg_win.h"
#include "cfg_win_op.h"


void CfgWinSetCfgWinLinkValueFromCfgItemValue(
	CfgWin *cfg_win,
	CfgWinLink *cfg_link
);

void CfgWinSetCfgItemValueFromCfgWinLinkValue(
	CfgWin *cfg_win,
	CfgWinLink *cfg_link
);


/*
 *      Get Macros:
 */
#define CFG_GET_B(_parm_)       ( \
 (CFGItemListGetValueI(cfg_list, (_parm_))) ? (TRUE) : (FALSE) \
)
#define CFG_GET_I(_parm_)       (CFGItemListGetValueI(cfg_list, (_parm_)))
#define CFG_GET_L(_parm_)       (CFGItemListGetValueL(cfg_list, (_parm_)))
#define CFG_GET_UL(_parm_)      (CFGItemListGetValueUL(cfg_list, (_parm_)))
#define CFG_GET_F(_parm_)       (CFGItemListGetValueF(cfg_list, (_parm_)))
#define CFG_GET_D(_parm_)       (CFGItemListGetValueD(cfg_list, (_parm_)))
#define CFG_GET_S(_parm_)       (CFGItemListGetValueS(cfg_list, (_parm_)))
#define CFG_GET_COLOR(_parm_)   (CFGItemListGetValueColor(cfg_list, (_parm_)))
#define CFG_GET_INT_LIST(_parm_) (CFGItemListGetValueIntList(cfg_list, (_parm_)))
#define CFG_GET_STRING_LIST(_parm_) (CFGItemListGetValueStringList(cfg_list, (_parm_)))
#define CFG_GET_ACCELKEY_LIST(_parm_) (CFGItemListGetValueAccelkeyList(cfg_list$
#define CFG_GET_STYLE(_parm_)   (CFGItemListGetValueStyle(cfg_list, (_parm_)))
#define CFG_GET_MENU(_parm_)    (CFGItemListGetValueMenu(cfg_list, (_parm_)))

/*
 *      Set Macros:
 */
#define CFG_SET_B(_parm_,_v_)   (CFGItemListSetValueI( \
 cfg_list, (_parm_), (_v_) ? (TRUE) : (FALSE), (FALSE) \
))
#define CFG_SET_I(_parm_,_v_)   (CFGItemListSetValueI( \
 cfg_list, (_parm_), (_v_), (FALSE) \
))
#define CFG_SET_L(_parm_,_v_)   (CFGItemListSetValueL( \
 cfg_list, (_parm_), (_v_), (FALSE) \
))
#define CFG_SET_UL(_parm_,_v_)  (CFGItemListSetValueUL( \
 cfg_list, (_parm_), (_v_), (FALSE) \
))
#define CFG_SET_F(_parm_,_v_)   (CFGItemListSetValueF( \
 cfg_list, (_parm_), (_v_), (FALSE) \
))
#define CFG_SET_D(_parm_,_v_)   (CFGItemListSetValueD( \
 cfg_list, (_parm_), (_v_), (FALSE) \
))                                                                              
#define CFG_SET_S(_parm_,_v_)   (CFGItemListSetValueS( \
 cfg_list, (_parm_), (_v_), (FALSE) \
))
#define CFG_SET_COLOR(_parm_,_v_)	(CFGItemListSetValueColor( \
 cfg_list, (_parm_), (_v_), (FALSE) \
))
#define CFG_SET_INT_LIST(_parm_,_v_)	(CFGItemListSetValueIntList( \
 cfg_list, (_parm_), (_v_), (FALSE) \
))
#define CFG_SET_STRING_LIST(_parm_,_v_)	(CFGItemListSetValueStringList( \
 cfg_list, (_parm_), (_v_), (FALSE) \
))
#define CFG_SET_ACCELKEY_LIST(_parm_,_v_)	(CFGItemListSetValueAccelkeyList( \
 cfg_list, (_parm_), (_v_), (FALSE) \
))
#define CFG_SET_STYLE(_parm_,_v_)	(CFGItemListSetValueStyle( \
 cfg_list, (_parm_), (_v_), (FALSE) \
))
#define CFG_SET_MENU(_parm_,_v_)        (CFGItemListSetValueMenu( \
 cfg_list, (_parm_), (_v_), (FALSE) \
))


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Sets the widget's value from its corresponding CfgItem's
 *	value.
 *
 *	The cfg_win specifies the CfgWin that contains the CfgList
 *	with the CfgItems and the CfgWinLink.
 *
 *	The cfg_link specifies the CfgWinLink that references the
 *	widget who's value is to be set with its corresponding
 *	CfgItem's value.
 *
 *	The "changed" signal will be emitted to the CfgWinLink's
 *	signal callbacks.
 */
void CfgWinSetCfgWinLinkValueFromCfgItemValue(
	CfgWin *cfg_win,
	CfgWinLink *cfg_link
)
{
	gint i;
	gpointer w;
	const gchar *cfg_parm = cfg_link->cfg_parm;
	CfgItemType cfg_type;
	CfgItem *cfg_item;
	CfgList *cfg_list = cfg_win->cfg_list;
	if(STRISEMPTY(cfg_parm))
	    return;

#define FREEZE	cfg_win->freeze_count++;
#define THAW	cfg_win->freeze_count--;

	/* Match cfg item from the cfg list by the given parameter */
	i = CFGItemListMatchParameter(cfg_list, cfg_parm);
	if(i < 0)
	    return;

	cfg_item = &cfg_list[i];
	cfg_type = cfg_item->type;
	w = cfg_link->w;

	/* Set the configuration value to the widget based on the
	 * widget's type
	 */
	switch(cfg_link->type)
	{
	  case CFG_WIN_LINK_WIDGET_UNKNOWN:
	    break;

	  case CFG_WIN_LINK_WIDGET_DRAWING_AREA:
	    break;
	  case CFG_WIN_LINK_WIDGET_BUTTON:
	    break;
	  case CFG_WIN_LINK_WIDGET_TOGGLE_BUTTON:
	    if(w != NULL)
	    {
		gboolean b;
		GtkToggleButton *tb = (GtkToggleButton *)w;
		FREEZE
		b = CFG_GET_B(cfg_parm);
		tb->active = b ? TRUE : FALSE;
		THAW
	    }
	    break;
	  case CFG_WIN_LINK_WIDGET_RADIO_BUTTON:
	    if(w != NULL)
	    {
		const gint radio_value = (gint)cfg_link->data;
		gint i;
		GtkToggleButton *tb = (GtkToggleButton *)w;
		FREEZE
		i = CFG_GET_I(cfg_parm);
		/* Set this GtkRadioButton active if the Widget
		 * Reference's radio_value matches the configuration
		 * value, otherwise set it inactive
		 */
		tb->active = (i == radio_value) ? TRUE : FALSE;
		THAW
	    }
	    break;
	  case CFG_WIN_LINK_WIDGET_RANGE:
	    if(w != NULL)
	    {
		gfloat f;
		GtkRange *range = (GtkRange *)w;
		GtkAdjustment *adj = range->adjustment;
		FREEZE
		f = CFG_GET_F(cfg_parm);
		if(adj != NULL)
		{
		    if(f > (adj->upper - adj->page_size))
			f = adj->upper - adj->page_size;
		    if(f < adj->lower)
			f = adj->lower;
		    gtk_adjustment_set_value(adj, f);
		}
		THAW
	    }
	    break;
	  case CFG_WIN_LINK_WIDGET_EDITABLE:
	    if(w != NULL)
	    {
	        GtkEntry *entry = (GtkEntry *)w;
	        FREEZE
	        if(cfg_type == CFG_ITEM_TYPE_STRING)
	        {
		    const gchar *values = (const gchar *)cfg_item->value;
		    if(values != NULL)
		        gtk_entry_set_text(entry, values);
	        }
	        else
	        {
		    gchar num_str[80];

		    *num_str = '\0';
		    switch((gint)cfg_type)
		    {
		      case CFG_ITEM_TYPE_INT8:
		      case CFG_ITEM_TYPE_UINT8:
		      case CFG_ITEM_TYPE_INT16:
		      case CFG_ITEM_TYPE_UINT16:
		      case CFG_ITEM_TYPE_INT32:
		      case CFG_ITEM_TYPE_UINT32:
		        g_snprintf(
			    num_str, sizeof(num_str),
			    "%i",
			    CFG_GET_I(cfg_parm)
		        );
		        break;

		      case CFG_ITEM_TYPE_INT64:
		      case CFG_ITEM_TYPE_UINT64:
		        g_snprintf(
			    num_str, sizeof(num_str),
			    "%ld",
			    CFG_GET_L(cfg_parm)
		        );
		        break;

		      case CFG_ITEM_TYPE_FLOAT:
		        g_snprintf(
			    num_str, sizeof(num_str),
			    "%f",
			    CFG_GET_F(cfg_parm)
		        );
		        break;

		      case CFG_ITEM_TYPE_DOUBLE:
		        g_snprintf(
			    num_str, sizeof(num_str),
			    "%f",
			    CFG_GET_D(cfg_parm)
		        );
		        break;
		    }
		    gtk_entry_set_text(entry, num_str);
	        }
	        THAW
	    }
	    break;
	  case CFG_WIN_LINK_WIDGET_SPIN_BUTTON:
	    if(w != NULL)
	    {
		GtkSpinButton *spin_button = (GtkSpinButton *)w;
	        FREEZE
	        if(cfg_type == CFG_ITEM_TYPE_STRING)
	        {
		    const gchar *values = (const gchar *)cfg_item->value;
		    if(values != NULL)
		        gtk_entry_set_text(GTK_ENTRY(w), values);
	        }
	        else
	        {
		    gtk_spin_button_set_value(spin_button, CFG_GET_F(cfg_parm));
	        }
	        THAW
	    }
	    break;
	  case CFG_WIN_LINK_WIDGET_COMBO:
	    if(w != NULL)
	    {
	        GtkCombo *combo = (GtkCombo *)w;
	        GtkEntry *entry = GTK_ENTRY(combo->entry);
	        FREEZE
	        if(cfg_type == CFG_ITEM_TYPE_STRING)
	        {
		    /* Set the GtkCombo's GtkEntry value to the
		     * configuration value string
		     */
		    const gchar *values = (const gchar *)cfg_item->value;
		    if(values != NULL)
		        gtk_entry_set_text(entry, values);
	        }
	        else
	        {
		    /* Set the GtkCombo's GtkEntry value to the
		     * string value of the GtkCombo's list item index
		     * specified by the configuration value
		     */
		    const gint i = CFG_GET_I(cfg_parm);
		    GList *glist = GUIComboGetList(w);
		    if(glist != NULL)
		    {
		        const gchar *s;
		        glist = g_list_nth(glist, i);
		        s = (const gchar *)((glist != NULL) ?
			    glist->data : NULL
		        );
		        if(!STRISEMPTY(s))
			    gtk_entry_set_text(entry, s);
		    }
	        }
	        THAW
	    }
	    break;
	  case CFG_WIN_LINK_WIDGET_CLIST:
	    if(w != NULL)
	    {
		GtkCList *clist = (GtkCList *)w;
		FREEZE
	        if((cfg_type == CFG_ITEM_TYPE_INT8) ||
		   (cfg_type == CFG_ITEM_TYPE_UINT8) ||
		   (cfg_type == CFG_ITEM_TYPE_INT16) ||
		   (cfg_type == CFG_ITEM_TYPE_UINT16) ||
		   (cfg_type == CFG_ITEM_TYPE_INT32) ||
		   (cfg_type == CFG_ITEM_TYPE_UINT32)
	        )
	        {
		    /* Select the GtkCList row who's data matches the
		     * CfgItem's value
		     */
		    gint valuei = CFG_GET_I(cfg_parm);
		    const gint nrows = clist->rows;
		    gint i;
		    for(i = 0; i < nrows; i++)
		    {
			if(valuei == (gint)gtk_clist_get_row_data(clist, i))
			{
			    gtk_clist_unselect_all(clist);
			    gtk_clist_select_row(clist, i, -1);
			}
		    }
	        }
	        else if(cfg_type == CFG_ITEM_TYPE_STRING)
	        {
		    /* Select the GtkCList row who's data matches the
		     * CfgItem's value
		     */
		    const gchar *s = CFG_GET_S(cfg_parm);
		    const gint nrows = clist->rows;
		    gint i;
		    const gchar *s2;

		    if(s == NULL)
			s = "";

		    for(i = 0; i < nrows; i++)
		    {
			s2 = (const gchar *)gtk_clist_get_row_data(clist, i);
			if(s2 == NULL)
			    continue;

			if(s == s2)
			{
			    gtk_clist_unselect_all(clist);
			    gtk_clist_select_row(clist, i, -1);
			}
			else if(!strcmp((const char *)s, (const char *)s2))
			{
			    gtk_clist_unselect_all(clist);
			    gtk_clist_select_row(clist, i, -1);
			}
		    }
	        }
		else if(cfg_type == CFG_ITEM_TYPE_INT_LIST)
	        {
		    /* Set the GtkCList row cell values and datas to
		     * the values in the CfgStringList
		     */
		    CfgIntList *intlist = CFG_GET_INT_LIST(cfg_parm);
		    const gint ncolumns = MAX(clist->columns, 1);
		    gint	i,
				row,
				valuei;
		    gchar **strv = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
		    GList *glist;

		    for(i = 0; i < ncolumns; i++)
			strv[i] = "";

		    gtk_clist_clear(clist);
		    for(glist = intlist->list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			valuei = (gint)glist->data;
			strv[0] = g_strdup_printf("%i", valuei);
			row = gtk_clist_append(clist, strv);
			g_free(strv[0]);
			if(row < 0)
			    break;

			gtk_clist_set_row_data(
			    clist,
			    row,
			    (gpointer)valuei
			);
		    }

		    g_free(strv);
	        }
		else if(cfg_type == CFG_ITEM_TYPE_STRING_LIST)
	        {
		    /* Set the GtkCList row cell values and datas to
		     * the values in the CfgStringList
		     */
		    CfgStringList *string_list = CFG_GET_STRING_LIST(cfg_parm);
		    const gint ncolumns = MAX(clist->columns, 1);
		    gint i, row;
		    gchar	*s,
				**strv = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
		    GList *glist;

		    for(i = 0; i < ncolumns; i++)
			strv[i] = "";

		    gtk_clist_clear(clist);
		    for(glist = string_list->list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			s = (gchar *)glist->data;
			strv[0] = (s != NULL) ? s : "";
			row = gtk_clist_append(clist, strv);
			if(row < 0)
			    break;

			if(s != NULL)
			    gtk_clist_set_row_data_full(
				clist,
				row,
				g_strdup(s), (GtkDestroyNotify)g_free
			    );
		    }

		    g_free(strv);
	        }
	        THAW
	    }
	    break;

	  case CFG_WIN_LINK_WIDGET_COLOR_BUTTON:
	    if(w != NULL)
	    {
		FREEZE
		if(cfg_type == CFG_ITEM_TYPE_COLOR)
		{
		    CfgColor *cfg_color = CFG_COLOR(cfg_item->value);
		    if(cfg_color != NULL)
		    {
			csd_color_struct csd_color;
			csd_color.a = cfg_color->a;
			csd_color.r = cfg_color->r;
			csd_color.g = cfg_color->g;
			csd_color.b = cfg_color->b;
			CSDColorButtonSetColor(w, &csd_color);
		    }
		}
		THAW
	    }
	    break;
	  case CFG_WIN_LINK_WIDGET_POPUP_LIST_BOX:
	    if(w != NULL)
	    {
		FREEZE
		if(cfg_type == CFG_ITEM_TYPE_STRING)
		{
		    const gchar *vs = CFG_GET_S(cfg_parm);
		    GtkWidget *pulist = PUListBoxGetPUList(w);
		    const gint m = PUListGetTotalItems(pulist);
		    if((vs != NULL) && (m > 0))
		    {
			gint i;
			const gchar *text;
			for(i = 0; i < m; i++)
			{
			    PUListGetItemText(
				pulist,
				i,
				&text
			    );
			    if(text == NULL)
				continue;

			    if(!g_strcasecmp(vs, text))
			    {
				PUListBoxSelect(w, i);
				break;
			    }
			}
		    }
		}
	        else if((cfg_type == CFG_ITEM_TYPE_INT8) ||
		        (cfg_type == CFG_ITEM_TYPE_UINT8) ||
		        (cfg_type == CFG_ITEM_TYPE_INT16) ||
		        (cfg_type == CFG_ITEM_TYPE_UINT16) ||
		        (cfg_type == CFG_ITEM_TYPE_INT32) ||
		        (cfg_type == CFG_ITEM_TYPE_UINT32)
	        )
		{
		    const gint vi = CFG_GET_I(cfg_parm);
		    GtkWidget *pulist = PUListBoxGetPUList(w);
		    const gint m = PUListGetTotalItems(pulist);
		    gint i;
		    for(i = 0; i < m; i++)
		    {
			if((gint)PUListGetItemData(pulist, i) == vi)
			{
			    PUListBoxSelect(w, i);
			    break;
			}
		    }
		}
		else if((cfg_type == CFG_ITEM_TYPE_INT64) ||
		        (cfg_type == CFG_ITEM_TYPE_UINT64)
		)
		{
		    const glong vl = CFG_GET_L(cfg_parm);
		    GtkWidget *pulist = PUListBoxGetPUList(w);
		    const gint m = PUListGetTotalItems(pulist);
		    gint i;
		    for(i = 0; i < m; i++)
		    {
			if((glong)PUListGetItemData(pulist, i) == vl)
			{
			    PUListBoxSelect(w, i);
			    break;
			}
		    }
		}
		THAW
	    }
	    break;
	  case CFG_WIN_LINK_WIDGET_STYLE_EDIT:
	    if(w != NULL)
	    {
		style_edit_struct *se = STYLE_EDIT(w);
		FREEZE
		if(cfg_type == CFG_ITEM_TYPE_STYLE)
		{
		    CfgStyle *src_cfg_style = CFG_STYLE(
			cfg_item->value
		    );
		    GtkRcStyle *tar_gtkrc_style = gtk_rc_style_new();

#define STRDUP2(s)	((STRISEMPTY(s)) ? NULL : g_strdup(s))
		    if(src_cfg_style != NULL)
		    {
			/* Copy values from the Cfg Item source style
			 * to the Style Edit target style
		         */
			gint i;
			GdkColor *tar_gdk_color;
			const CfgColor *src_cfg_color;
			CfgStyleFlags src_cfg_color_flags;

			g_free(tar_gtkrc_style->font_name);
			tar_gtkrc_style->font_name = STRDUP2(src_cfg_style->font_name);

			for(i = 0; i < STYLE_EDIT_STATES; i++)
			{
			    src_cfg_color_flags = src_cfg_style->color_flags[i];
			    if(src_cfg_color_flags & CFG_STYLE_FG)
			    {
				tar_gtkrc_style->color_flags[i] |= GTK_RC_FG;
				src_cfg_color = &src_cfg_style->fg[i];
				tar_gdk_color = &tar_gtkrc_style->fg[i];
				GDK_COLOR_SET_COEFF(
				    tar_gdk_color,
				    src_cfg_color->r,
				    src_cfg_color->g,
				    src_cfg_color->b
				);
			    }
			    if(src_cfg_color_flags & CFG_STYLE_BG)
			    {
				tar_gtkrc_style->color_flags[i] |= GTK_RC_BG;
				src_cfg_color = &src_cfg_style->bg[i];
				tar_gdk_color = &tar_gtkrc_style->bg[i];
				GDK_COLOR_SET_COEFF(
				    tar_gdk_color,
				    src_cfg_color->r,
				    src_cfg_color->g,
				    src_cfg_color->b
				);
			    }
			    if(src_cfg_color_flags & CFG_STYLE_TEXT)
			    {
				tar_gtkrc_style->color_flags[i] |= GTK_RC_TEXT;
				src_cfg_color = &src_cfg_style->text[i];
				tar_gdk_color = &tar_gtkrc_style->text[i];
				GDK_COLOR_SET_COEFF(
				    tar_gdk_color,
				    src_cfg_color->r,
				    src_cfg_color->g,
				    src_cfg_color->b
				);
			    }
			    if(src_cfg_color_flags & CFG_STYLE_BASE)
			    {
				tar_gtkrc_style->color_flags[i] |= GTK_RC_BASE;
				src_cfg_color = &src_cfg_style->base[i];
				tar_gdk_color = &tar_gtkrc_style->base[i];
				GDK_COLOR_SET_COEFF(
				    tar_gdk_color,
				    src_cfg_color->r,
				    src_cfg_color->g,
				    src_cfg_color->b
				);
			    }

			    g_free(tar_gtkrc_style->bg_pixmap_name[i]);
			    tar_gtkrc_style->bg_pixmap_name[i] = STRDUP2(
				src_cfg_style->bg_pixmap_name[i]
			    );
			}

			/* Set style to the Style Edit */
			StyleEditSetStyle(se, tar_gtkrc_style);
		    }
		    GTK_RC_STYLE_UNREF(tar_gtkrc_style);
#undef STRDUP2
		}
		THAW
	    }
	    break;
	  case CFG_WIN_LINK_WIDGET_STACK_LIST:
	    if(w != NULL)
	    {
		stack_list_struct *slist = STACK_LIST(w);
	        FREEZE
	        if(cfg_type == CFG_ITEM_TYPE_INT_LIST)
	        {
		    CfgIntList *intlist = CFG_INT_LIST(
		        cfg_item->value
		    );
		    if(intlist != NULL)
		    {
		        GList *glist;

		        /* Update stack list's source list by deleting all
		         * items in its source list and then adding all
		         * items from its cache to its source list
		         */
		        StackListItemClearSrc(slist);
		        StackListItemSetAllFromCacheSrc(slist);

		        /* Remove items from the source list that are
		         * specified in the intlist (except items that
		         * allow multiple occurances)
		         */
		        for(glist = intlist->list;
		            glist != NULL;
		            glist = g_list_next(glist)
		        )
			    StackListItemRemoveByIDSrc(
			        slist,
				(gint)glist->data,
			        TRUE		/* Exclude allow_multiples */
			    );

		        /* Delete all items in the target list and then add
		         * items to the target list that are specified in
		         * the intlist
		         */
		        StackListItemClearTar(slist);
		        for(glist = intlist->list;
			    glist != NULL;
			    glist = g_list_next(glist)
		        )
			    StackListItemAppendTar(
			        slist,
				(gint)glist->data
			    );
		    }
	        }
	        THAW
	    }
	    break;
	  case CFG_WIN_LINK_WIDGET_KEYMAP_LIST:
	    if(w != NULL)
	    {
		keymap_list_struct *kmlist = KEYMAP_LIST(w);
	        FREEZE
	        if(cfg_type == CFG_ITEM_TYPE_ACCELKEY_LIST)
	        {
		    CfgAccelkey *ak;
		    CfgAccelkeyList *ak_list = CFG_ACCELKEY_LIST(
		        cfg_item->value
		    );
		    if(ak_list != NULL)
		    {
		        gint row;
		        GList *glist;

		        /* Iterate through Accelkey list and set them to
		         * the Keymap List
		         */
		        for(glist = ak_list->list, row = 0;
			    glist != NULL;
			    glist = g_list_next(glist), row++
		        )
		        {
			    ak = CFG_ACCELKEY(glist->data);
			    if(ak == NULL)
			        continue;

			    KeymapListSetKey(
			        kmlist,
				row,
				KeymapKeyNew(
				    ak->key,		/* Keyval */
				    ak->modifiers,	/* State */
				    (gpointer)ak->op_id	/* Data */
				)
			    );
		        }
		    }
	        }
		THAW
	    }
	    break;
	  case CFG_WIN_LINK_WIDGET_MENUCFG_LIST:
	    if(w != NULL)
	    {
		menucfg_list_struct *list = MENUCFG_LIST(w);
	        FREEZE
	        if(cfg_type == CFG_ITEM_TYPE_MENU)
	        {
		    CfgMenuItem *item;
		    CfgMenu *menu = CFG_MENU(
		        cfg_item->value
		    );
		    if(menu != NULL)
		    {
			const gint ncustom_datas = MeuCfgListGetTotalCustomDatas(list);
			gint row;
		        GList *glist;
			CfgMenuItemFlags flags;
			menuCfgItem *item2;

			MenuCfgListClear(list);

		        for(glist = menu->list;
			    glist != NULL;
			    glist = g_list_next(glist)
		        )
		        {
			    item = CFG_MENU_ITEM(glist->data);
			    if(item == NULL)
			        continue;

			    flags = item->flags;

			    item2 = MenuCfgItemNew(
				((flags & CFG_MENU_ITEM_HAS_FUNCTION) ?
				    MENUCFG_ITEM_HAS_FUNCTION : 0) |
				((flags & CFG_MENU_ITEM_IS_SEPARATOR) ?
				    MENUCFG_ITEM_IS_SEPARATOR : 0) |
				((flags & CFG_MENU_ITEM_RUN_IN_TERMINAL) ?
				    MENUCFG_ITEM_RUN_IN_TERMINAL : 0),
				item->label,
				item->command,
				item->icon_file,
				item->description
			    );
			    if(item2 == NULL)
				break;

			    if(ncustom_datas > 0)
			    {
				item2->custom_datas_list = g_list_append(
				    item2->custom_datas_list,
				    STRDUP(item->ext_data)
				);
			    }

			    row = MenuCfgListAppend(list, item2);
		        }
		    }
	        }
		THAW
	    }
	    break;
	}

#undef THAW
#undef FREEZE

	/* Emit the "changed" signal to this Widget Reference's
	 * signal callbacks
	 */
	CfgWinLinkEmitSignalByName(
	    cfg_link,
	    "changed"
	);
}


/*
 *	Sets the CfgItem's value from its corresponding widget's
 *	value.
 *
 *	The cfg_win specifies the CfgWin that contains the CfgList
 *	with the CfgItems and the CfgWinLink.
 *
 *	The cfg_link specifies the CfgWinLink that references the
 *	CfgItem who's value is to be set with its corresponding
 *	widget's value.
 */
void CfgWinSetCfgItemValueFromCfgWinLinkValue(
	CfgWin *cfg_win,
	CfgWinLink *cfg_link
)
{
	gint i;
	gpointer w;
	const gchar *cfg_parm = cfg_link->cfg_parm;
	CfgItemType cfg_type;
	CfgItem *cfg_item;
	CfgList *cfg_list = cfg_win->cfg_list;
	if(STRISEMPTY(cfg_parm))
	    return;

	/* Get the CfgItem from the CfgList that corresponds with
	 * this CfgWinLink's parameter
	 */
	i = CFGItemListMatchParameter(
	    cfg_list,
	    cfg_parm
	);
	if(i < 0)
	    return;

	cfg_item = &cfg_list[i];
	cfg_type = cfg_item->type;
	w = cfg_link->w;

	/* Set the CfgItem's value from the widget based on the type
	 * of widget specified by the CfgWinLink
	 */
	switch(cfg_link->type)
	{
	  case CFG_WIN_LINK_WIDGET_UNKNOWN:
	    break;

	  case CFG_WIN_LINK_WIDGET_DRAWING_AREA:
	    break;
	  case CFG_WIN_LINK_WIDGET_BUTTON:
	    break;
	  case CFG_WIN_LINK_WIDGET_TOGGLE_BUTTON:
	    if(w != NULL)
	    {
		gint i = GTK_TOGGLE_BUTTON_GET_ACTIVE((GtkWidget *)w);
		CFG_SET_I(cfg_parm, i);
	    }
	    break;
	  case CFG_WIN_LINK_WIDGET_RADIO_BUTTON:
	    if(w != NULL)
	    {
		/* For GtkRadioButtons, set the CfgItem's value to
		 * the CfgWinLink's data value if the GtkRadioButton
		 * is active
		 */
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE((GtkWidget *)w))
		    CFG_SET_I(
			cfg_parm,
			(gint)cfg_link->data	/* Radio value */
		    );
	    }
	    break;
	  case CFG_WIN_LINK_WIDGET_RANGE:
	    if(w != NULL)
	    {
		GtkRange *range = (GtkRange *)w;
		GtkAdjustment *adj = range->adjustment;
		switch(cfg_type)
	        {
	          case CFG_ITEM_TYPE_FLOAT:
	          case CFG_ITEM_TYPE_DOUBLE:
		    CFG_SET_F(cfg_parm, adj->value);
		    break;
	          default:
		    CFG_SET_I(cfg_parm, (gint)adj->value);
		    break;
	        }
	    }
	    break;
	  case CFG_WIN_LINK_WIDGET_EDITABLE:
	    if(w != NULL)
	    {
		GtkEntry *entry = (GtkEntry *)w;
	        if(cfg_type == CFG_ITEM_TYPE_STRING)
	        {
		    const gchar *values = gtk_entry_get_text(entry);
		    if(values != NULL)
		        CFG_SET_S(cfg_parm, values);
	        }
	        else
	        {
		    const gchar *values = gtk_entry_get_text(entry);
		    if(values != NULL)
		    {
		        /* Decimal in the string? */
		        if(strchr(values, '.'))
			    CFG_SET_F(cfg_parm, ATOF(values));
		        else
			    CFG_SET_I(cfg_parm, ATOI(values));
		    }
	        }
	    }
	    break;
	  case CFG_WIN_LINK_WIDGET_SPIN_BUTTON:
	    if(w != NULL)
	    {
		GtkSpinButton *spin_button = (GtkSpinButton *)w;

	        if(cfg_type == CFG_ITEM_TYPE_STRING)
	        {
		    const gchar *values = gtk_entry_get_text(GTK_ENTRY(w));
		    if(values != NULL)
		        CFG_SET_S(cfg_parm, values);
	        }
	        else
	        {
		    if((spin_button->digits > 0) ||
		       (cfg_type == CFG_ITEM_TYPE_FLOAT) ||
		       (cfg_type == CFG_ITEM_TYPE_DOUBLE)
		    )
		    {
			const gfloat v = gtk_spin_button_get_value_as_float(
			    spin_button
		        );
		        CFG_SET_F(cfg_parm, v);
		    }
		    else if((cfg_type == CFG_ITEM_TYPE_UINT64) ||
			    (cfg_type == CFG_ITEM_TYPE_INT64)
		    )
		    {
		        const gint v = gtk_spin_button_get_value_as_int(
			    spin_button
		        );
		        CFG_SET_L(cfg_parm, v);
		    }
		    else
		    {
		        const gint v = gtk_spin_button_get_value_as_int(
			    spin_button
		        );
		        CFG_SET_I(cfg_parm, v);
		    }
	        }
	    }
	    break;
	  case CFG_WIN_LINK_WIDGET_COMBO:
	    if(w != NULL)
	    {
		GtkCombo *combo = (GtkCombo *)w;
	        GtkEntry *entry = GTK_ENTRY(combo->entry);

	        if(cfg_type == CFG_ITEM_TYPE_STRING)
	        {
		    /* Set the CfgItem's value as the GtkCombo's
		     * GtkEntry's string value
		     */
		    const gchar *values = gtk_entry_get_text(entry);
		    if(values != NULL)
		        CFG_SET_S(cfg_parm, values);
	        }
	        else
	        {
		    /* Set the CfgItem's value to the index of the
		     * GtkCombo's list item who's string value matches
		     * the GtkCombo's GtkEntry value
		     */
		    const gchar *s = gtk_entry_get_text(entry);
		    if(!STRISEMPTY(s))
		    {
		        const gchar *s2;
		        gint valuei = 0;
		        GList *glist = GUIComboGetList(w);
		        while(glist != NULL)
		        {
			    s2 = (const gchar *)glist->data;
			    if(!STRISEMPTY(s2))
			    {
			        if(!g_strcasecmp(s, s2))
				    break;
			    }
			    valuei++;
			    glist = g_list_next(glist);
		        }
		        if(glist != NULL)
			    CFG_SET_I(cfg_parm, valuei);
		    }
	        }
	    }
	    break;
	  case CFG_WIN_LINK_WIDGET_CLIST:
	    if(w != NULL)
	    {
		GtkCList *clist = (GtkCList *)w;
	        if((cfg_type == CFG_ITEM_TYPE_INT8) ||
		   (cfg_type == CFG_ITEM_TYPE_UINT8) ||
		   (cfg_type == CFG_ITEM_TYPE_INT16) ||
		   (cfg_type == CFG_ITEM_TYPE_UINT16) ||
		   (cfg_type == CFG_ITEM_TYPE_INT32) ||
		   (cfg_type == CFG_ITEM_TYPE_UINT32)
	        )
	        {
		    /* Set the CfgItem's value to the row data value
		     * of the selected row
		     */
		    const GList *glist = clist->selection_end;
		    if(glist != NULL)
		    {
		        const gint row = (gint)glist->data;
			const gint valuei = (gint)gtk_clist_get_row_data(clist, row);
		        CFG_SET_I(cfg_parm, valuei);
		    }
	        }
		else if(cfg_type == CFG_ITEM_TYPE_STRING)
	        {
		    /* Set the CfgItem's value to the row data value
		     * of the selected row
		     */
		    const GList *glist = clist->selection_end;
		    if(glist != NULL)
		    {
		        const gint row = (gint)glist->data;
			const gchar *s = (gchar *)gtk_clist_get_row_data(clist, row);
			if(s == NULL)
			    s = "";
			CFG_SET_S(cfg_parm, s);
		    }
	        }
		else if(cfg_type == CFG_ITEM_TYPE_INT_LIST)
	        {
		    /* Set the CfgStringList values to the row datas */
		    CfgIntList *intlist = CFGIntListNew(NULL);
		    if(intlist != NULL)
		    {
			const gint nrows = clist->rows;
			gint	i,
				valuei;
			for(i = 0; i < nrows; i++)
			{
			    valuei = (gint)gtk_clist_get_row_data(clist, i);
			    intlist->list = g_list_append(
				intlist->list,
				(gpointer)valuei
			    );
			}
			CFG_SET_INT_LIST(cfg_parm, intlist);
			CFGIntListDelete(intlist);
		    }
	        }
		else if(cfg_type == CFG_ITEM_TYPE_STRING_LIST)
	        {
		    /* Set the CfgStringList values to the row datas */
		    CfgStringList *string_list = CFGStringListNew(NULL);
		    if(string_list != NULL)
		    {
			const gint nrows = clist->rows;
			gint i;
			const gchar *s;
			for(i = 0; i < nrows; i++)
			{
			    s = (const gchar *)gtk_clist_get_row_data(clist, i);
			    if(s == NULL)
				s = "";
			    string_list->list = g_list_append(
				string_list->list,
				g_strdup(s)
			    );
			}
			CFG_SET_STRING_LIST(cfg_parm, string_list);
			CFGStringListDelete(string_list);
		    }
	        }
	    }
	    break;

	  case CFG_WIN_LINK_WIDGET_COLOR_BUTTON:
	    if((w != NULL) && (cfg_type == CFG_ITEM_TYPE_COLOR))
	    {
		CfgColor *c_tar;
		csd_color_struct csd_color, *c_src = &csd_color;

		/* Get color value from the Color Button */
		CSDColorButtonGetColor(w, c_src);

		/* Create a new Cfg Color and set the new value */
		c_tar = CFGColorNew(
		    c_src->r,
		    c_src->g,
		    c_src->b,
		    c_src->a
		);
		CFG_SET_COLOR(cfg_parm, c_tar);
		CFGColorDelete(c_tar);
	    }
	    break;
	  case CFG_WIN_LINK_WIDGET_POPUP_LIST_BOX:
	    if(w != NULL)
	    {
		if(cfg_type == CFG_ITEM_TYPE_STRING)
		{
		    const gchar *text;
		    const gint i = PUListBoxGetSelected(w);
		    GtkWidget *pulist = PUListBoxGetPUList(w);
		    PUListGetItemText(
			pulist,
			i,
			&text
		    );
		    if(text == NULL)
			text = "";
		    CFG_SET_S(
			cfg_parm,
			text
		    );
		}
		else if((cfg_type == CFG_ITEM_TYPE_INT8) ||
			(cfg_type == CFG_ITEM_TYPE_UINT8) ||
			(cfg_type == CFG_ITEM_TYPE_INT16) ||
			(cfg_type == CFG_ITEM_TYPE_UINT16) ||
			(cfg_type == CFG_ITEM_TYPE_INT32) ||
			(cfg_type == CFG_ITEM_TYPE_UINT32)
		)
		{
		    CFG_SET_I(
			cfg_parm,
			(gint)PUListBoxGetSelectedData(w)
		    );
		}
		else if((cfg_type == CFG_ITEM_TYPE_INT64) ||
			(cfg_type == CFG_ITEM_TYPE_UINT64)
		)
		{
		    CFG_SET_L(
			cfg_parm,
			(glong)PUListBoxGetSelectedData(w)
		    );
		}
	    }
	    break;
	  case CFG_WIN_LINK_WIDGET_STYLE_EDIT:
	    if((w != NULL) && (cfg_type == CFG_ITEM_TYPE_STYLE))
	    {
		gint i;
		CfgStyle *tar_style = CFGStyleNew();
		style_edit_struct *se = STYLE_EDIT(w);
		const GtkRcStyle *src_style = StyleEditGetStyle(se);

		if(src_style != NULL)
		{
		    /* Copy values from the Style Edit source style to
		     * the Cfg Item target style
		     */
		    GtkRcFlags src_color_flags;
		    CfgColor *tar_c;
		    const GdkColor *src_c;

#define STRDUP2(s)	((STRISEMPTY(s)) ? NULL : g_strdup(s))

		    tar_style->font_name = STRDUP2(src_style->font_name);

		    for(i = 0; i < CFG_STYLE_STATES; i++)
		    {
			src_color_flags = src_style->color_flags[i];
			if(src_color_flags & GTK_RC_FG)
			{
			    tar_style->color_flags[i] |= CFG_STYLE_FG;
			    src_c = &src_style->fg[i];
			    tar_c = &tar_style->fg[i];
			    tar_c->a = 1.0f;
			    tar_c->r = (gfloat)src_c->red   / (gfloat)((gushort)-1);
			    tar_c->g = (gfloat)src_c->green / (gfloat)((gushort)-1);
			    tar_c->b = (gfloat)src_c->blue  / (gfloat)((gushort)-1);
			}
			if(src_color_flags & GTK_RC_BG)
			{
			    tar_style->color_flags[i] |= CFG_STYLE_BG;
			    src_c = &src_style->bg[i];
			    tar_c = &tar_style->bg[i];
			    tar_c->a = 1.0f;
			    tar_c->r = (gfloat)src_c->red   / (gfloat)((gushort)-1);
			    tar_c->g = (gfloat)src_c->green / (gfloat)((gushort)-1);
			    tar_c->b = (gfloat)src_c->blue  / (gfloat)((gushort)-1);
			}
			if(src_color_flags & GTK_RC_TEXT)
			{
			    tar_style->color_flags[i] |= CFG_STYLE_TEXT;
			    src_c = &src_style->text[i];
			    tar_c = &tar_style->text[i];
			    tar_c->a = 1.0f;
			    tar_c->r = (gfloat)src_c->red   / (gfloat)((gushort)-1);
			    tar_c->g = (gfloat)src_c->green / (gfloat)((gushort)-1);
			    tar_c->b = (gfloat)src_c->blue  / (gfloat)((gushort)-1);
			}
			if(src_color_flags & GTK_RC_BASE)
			{
			    tar_style->color_flags[i] |= CFG_STYLE_BASE;
			    src_c = &src_style->base[i];
			    tar_c = &tar_style->base[i];
			    tar_c->a = 1.0f;
			    tar_c->r = (gfloat)src_c->red   / (gfloat)((gushort)-1);
			    tar_c->g = (gfloat)src_c->green / (gfloat)((gushort)-1);
			    tar_c->b = (gfloat)src_c->blue  / (gfloat)((gushort)-1);
			}

			tar_style->bg_pixmap_name[i] = STRDUP2(
			    src_style->bg_pixmap_name[i]
			);
		    }

		    /* Set target style to cfg item */
		    CFG_SET_STYLE(cfg_parm, tar_style);

#undef STRDUP2
		}

		/* Delete the Cfg Style */
		CFGStyleDelete(tar_style);
	    }
	    break;
	  case CFG_WIN_LINK_WIDGET_STACK_LIST:
	    if((w != NULL) && (cfg_type == CFG_ITEM_TYPE_INT_LIST))
	    {
		CfgIntList *intlist = CFGIntListNew(NULL);
		stack_list_struct *slist = STACK_LIST(w);
		if(intlist != NULL)
		{
		    /* Get the target list's list of IDs and add each
		     * ID to the intlist
		     */
		    gint nids;
		    gint *id = StackListItemGetTar(slist, &nids);
		    if(id != NULL)
		    {
			gint i;

			/* Add ids to the new intlist */
			for(i = 0; i < nids; i++)
			    intlist->list = g_list_append(
				intlist->list,
				(gpointer)id[i]
			    );

			/* Delete target list ids */
			g_free(id);
		    }

		    /* Set/copy intlist to cfg item */
		    CFG_SET_INT_LIST(cfg_parm, intlist);
		}

		CFGIntListDelete(intlist);
	    }
	    break;
	  case CFG_WIN_LINK_WIDGET_KEYMAP_LIST:
	    if((w != NULL) && (cfg_type == CFG_ITEM_TYPE_ACCELKEY_LIST))
	    {
		gint row;
		GList *glist = NULL;
		CfgAccelkeyList *ak_list;
		keymap_key_struct *km;
		keymap_list_struct *kmlist = KEYMAP_LIST(w);

		/* Iterate through each row on the Keymap List and
		 * append/copy each key to the glist
		 */
		for(row = 0; TRUE; row++)
		{
		    km = KeymapListGet(kmlist, row);
		    if(km == NULL)
			break;

		    glist = g_list_append(
			glist,
			CFGAccelkeyNew(
			    (gint)km->data,	/* OPID */
			    km->keyval,		/* Key */
			    km->state		/* Modifiers */
			)
		    );
		}

		/* Create a new Accelkey List and then delete the glist
		 * containing the Accelkeys
		 */
		ak_list = CFGAccelkeyListNew(glist);
		if(glist != NULL)
		{
		    g_list_foreach(glist, (GFunc)CFGAccelkeyDelete, NULL);
		    g_list_free(glist);
		}

		/* Set/copy the Accelkey List to cfg item */
		CFG_SET_ACCELKEY_LIST(cfg_parm, ak_list);

		CFGAccelkeyListDelete(ak_list);
	    }
	    break;
	  case CFG_WIN_LINK_WIDGET_MENUCFG_LIST:
	    if((w != NULL) && (cfg_type == CFG_ITEM_TYPE_MENU))
	    {
		gint row;
		GList *menu_items_list;
		CfgMenu *menu;
		menucfg_item_flags flags;
		menuCfgItem *item;
		menucfg_list_struct *list = MENUCFG_LIST(w);

		/* Iterate through each row on the Keymap List and
		 * append/copy each key to the glist
		 */
		menu_items_list = NULL;
		for(row = 0; TRUE; row++)
		{
		    item = MenuCfgListGet(list, row);
		    if(item == NULL)
			break;

		    flags = item->flags;

		    menu_items_list = g_list_append(
			menu_items_list,
			CFGMenuItemNew(
			    CFG_MENU_ITEM_SENSITIVE |
			    ((flags & MENUCFG_ITEM_HAS_FUNCTION) ?
				CFG_MENU_ITEM_HAS_FUNCTION : 0) |
			    ((flags & MENUCFG_ITEM_IS_SEPARATOR) ?
				CFG_MENU_ITEM_IS_SEPARATOR : 0) |
			    ((flags & MENUCFG_ITEM_RUN_IN_TERMINAL) ?
				CFG_MENU_ITEM_RUN_IN_TERMINAL : 0),
			    item->label,
			    item->command,
			    item->icon_file,
			    item->description,
			    MenuCfgItemGetCustomDataValue(item, 0)
			)
		    );
		}

		/* Create a new Accelkey List and then delete the glist
		 * containing the Accelkeys
		 */
		menu = CFGMenuNew(menu_items_list);
		if(menu_items_list != NULL)
		{
		    g_list_foreach(
			menu_items_list, (GFunc)CFGMenuItemDelete, NULL
		    );
		    g_list_free(menu_items_list);
		}

		/* Set/copy the Menu List to cfg item */
		CFG_SET_MENU(cfg_parm, menu);

		CFGMenuDelete(menu);
	    }
	    break;
	}
}
