#include <stdlib.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "cfg.h"
#include "cfg_fio.h"

#include "guiutils.h"
#include "pulist.h"
#include "stacklist.h"
#include "style_edit.h"
#include "keymap_list.h"
#include "menucfg_list.h"
#include "progressdialog.h"
#include "csd.h"

#include "cfg_win.h"
#include "cfg_win_op.h"

#include "config.h"

#include "images/icon_ok_20x20.xpm"
#include "images/icon_select_20x20.xpm"
#include "images/icon_save_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"
#include "images/icon_close_20x20.xpm"

#include "images/icon_select_32x32.xpm"


typedef struct _CfgCallback	CfgCallback;
#define CFG_CALLBACK(p)		((CfgCallback *)(p))


/* Callbacks */
static void CfgWinDestroyCB(gpointer data);
static gint CfgWinDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void CfgWinRealizeCB(GtkWidget *widget, gpointer data);
static CfgWinPage *CfgWinSwitchPageCBIterate(
	GList *pages_list,
	GtkWidget *notebook,
	const gint page_num
);
static void CfgWinSwitchPageCB(
	GtkNotebook *notebook, GtkNotebookPage *page, guint page_num,
	gpointer data
);
static void CfgWinTreeSelectRowCB(
	GtkCTree *ctree, GtkCTreeNode *node, gint column,
	gpointer data
);
static void CfgWinTreeUnselectRowCB(
	GtkCTree *ctree, GtkCTreeNode *node, gint column,
	gpointer data
);
static void CfgWinOKCB(GtkWidget *widget, gpointer data);
static void CfgWinApplyCB(GtkWidget *widget, gpointer data);
static void CfgWinSaveCB(GtkWidget *widget, gpointer data);
static void CfgWinCancelCB(GtkWidget *widget, gpointer data);
static void CfgWinAnyChangedCB(GtkWidget *widget, gpointer data);
#if 0
static void CfgWinCListChangedCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
#endif
static void CfgWinPopupListBoxChangedCB(
	GtkWidget *widget, const gint i, gpointer data
);
static void CfgWinStyleEditChangedCB(
	style_edit_struct *se, gpointer data
);
static void CfgWinStackListChangedCB(
	stack_list_struct *slist, gpointer data
);
static void CfgWinKeymapListChangedCB(
	keymap_list_struct *kmlist, gpointer data
);
static void CfgWinMenuCfgListChangedCB(
	menucfg_list_struct *list, gpointer data
);

static void CfgWinPageDestroyCB(gpointer data);

static gint CfgWinSetValuesSaveCB(
	const gulong i, const gulong m,
	gpointer data
);


/* CfgWinLink Signal Callback */
static CfgCallback *CfgCallbackNew(void);
static void CfgCallbackDelete(CfgCallback *cfg_cb);


/* CfgWinLink Get/Set */
static CfgWinLink *CfgWinLinkNew(void);
static void CfgWinLinkDelete(CfgWinLink *cfg_link);

CfgItemType CfgWinLinkGetType(CfgWinLink *cfg_link);
const gchar *CfgWinLinkGetParameter(CfgWinLink *cfg_link);
gpointer CfgWinLinkGetWidget(CfgWinLink *cfg_link);
GtkWidget *CfgWinLinkGetParentPage(CfgWinLink *cfg_link);
GtkWidget *CfgWinLinkGetCfgWin(CfgWinLink *cfg_link);

void CfgWinLinkSetIgnoreChanges(
	CfgWinLink *cfg_link,
	const gboolean ignore_changes
);

void CfgWinLinkSetData(
	CfgWinLink *cfg_link,
	gpointer data           
);           
gpointer CfgWinLinkGetData(CfgWinLink *cfg_link); 
	
/* CfgWinLink Signal Callbacks */
static guint CfgWinLinkSignalConnectNexus(
	CfgWinLink *cfg_link,
	const gchar *signal_name,
	void (*cb)(
			CfgWinLink *,
			gpointer
	),
	gpointer data,
	const gboolean after
);
guint CfgWinLinkSignalConnect(
	CfgWinLink *cfg_link,
	const gchar *signal_name,
	void (*cb)(
			CfgWinLink *,
			gpointer
	),
	gpointer data
);
guint CfgWinLinkSignalConnectAfter(
	CfgWinLink *cfg_link,
	const gchar *signal_name,
	void (*cb)(
			CfgWinLink *,
			gpointer
	),
	gpointer data
);
void CfgWinLinkSignalDisconnect(
	CfgWinLink *cfg_link,
	const guint id
);
void CfgWinLinkEmitSignalByName(
	CfgWinLink *cfg_link,
	const gchar *signal_name
);

/* CfgWinLink Add/Set/Get/Remove */
CfgWinLink *CfgWinAddLink(
	GtkWidget *w,
	GtkWidget *page,
	const CfgWinLinkWidgetType type,
	gpointer widget,
	const gchar *cfg_parm
);
static gint CfgWinGetTotalCfgWinLinksIterate(
	CfgWin *cfg_win,
	CfgWinPage *page
);
gint CfgWinGetTotalCfgWinLinks(GtkWidget *w);
static CfgWinLink *CfgWinGetCfgWinLinkByParameterIterate(
	CfgWin *cfg_win,
	CfgWinPage *page,
	const gchar *cfg_parm
);
CfgWinLink *CfgWinGetCfgWinLinkByParameter(
	GtkWidget *w,
	const gchar *cfg_parm
);
void CfgWinRemoveLink(
	GtkWidget *w,
	CfgWinLink *cfg_link 
);


/* CfgWinPage Get/Set */
static CfgWinPage *CfgWinPageGetWidgetData(
	GtkWidget *w,
	const gchar *func_name
);
GtkWidget *CfgWinPageGetParentPage(GtkWidget *page);
GtkWidget *CfgWinPageGetChild(GtkWidget *page);
GtkWidget *CfgWinPageGetSibling(GtkWidget *page);
const gchar *CfgWinPageGetLabel(GtkWidget *page);
gint CfgWinPageGetIndex(GtkWidget *page);

/* CfgWinPage Add/Remove */
static void CfgWinUpdatePageFamilyNumbers(GList *pages_list);
GtkWidget *CfgWinAddPageNotebook(
	GtkWidget *w,
	GtkWidget *parent_page,
	const gint insert_pos,
	const gchar *label,
	GtkWidget *tab_label_widget,
	const gboolean is_parent
);
GtkWidget *CfgWinAddPageTree(
	GtkWidget *w,
	GtkWidget *parent_page,
	GtkWidget *sibling_page,
	const gchar *label,
	GdkPixmap *pixmap_closed,
	GdkBitmap *mask_closed,
	GdkPixmap *pixmap_opened,
	GdkBitmap *mask_opened,
	const gboolean create_heading,
	const gboolean is_parent
);
static void CfgWinRemovePageIterate(
	CfgWin *cfg_win,
	CfgWinPage *cfg_page
);
void CfgWinRemovePage(
	GtkWidget *w,
	GtkWidget *page 
);

/* CfgWin Set CfgWinPage */
void CfgWinSetPage(
	GtkWidget *w,
	GtkWidget *page
);

/* CfgWin Get CfgWinPage */
GtkWidget *CfgWinGetDefaultPage(GtkWidget *w);
GtkWidget *CfgWinGetCurrentPage(GtkWidget *w);
static GtkWidget *CfgWinGetPageByLabelIterate(
	CfgWin *cfg_win,
	GList *child_pages_list,
	const gchar *label,
	const gboolean recursive
);
GtkWidget *CfgWinGetPageByLabel(
	GtkWidget *w,
	GtkWidget *parent_page,
	const gchar *label,
	const gboolean recursive
);


/* CfgWin Set GtkWidget Values From CfgItem Values */
static void CfgWinGetValuesIterate(
	CfgWin *cfg_win,
	CfgWinPage *page
);
void CfgWinGetValues(
	GtkWidget *w,
	const gboolean show_progress
);

/* CfgWin Set CfgItem Values From GtkWidget Values */
static void CfgWinSetValuesIterate(
	CfgWin *cfg_win,
	CfgWinPage *page,
	const gboolean show_progress,
	const gboolean save_to_disk,
	gboolean *user_aborted,
	gint *ncfg_widgets_set,
	const gint ncfg_widgets
);
void CfgWinSetValues(
	GtkWidget *w,
	const gboolean show_progress,
	const gboolean call_apply_cb,
	const gboolean save_to_disk
);

/* CfgWin */
static CfgWin *CfgWinGetWidgetData(
	GtkWidget *w,
	const gchar *func_name
);
GtkWidget *CfgWinNew(
	const CfgWinStyle cfg_win_style,
	const gchar *title,
	guint8 **icon_data,
	const CfgWinButtons cfg_win_buttons,
	const gint width, const gint height,
	CfgList *cfg_list,
	const gchar *cfg_path,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry
);
GtkWidget *CfgWinGetOKButton(GtkWidget *w);
GtkWidget *CfgWinGetApplyButton(GtkWidget *w);
GtkWidget *CfgWinGetSaveButton(GtkWidget *w);
GtkWidget *CfgWinGetCancelButton(GtkWidget *w);
GtkWidget *CfgWinGetCloseButton(GtkWidget *w);
void CfgWinSetPageChangedCB(
	GtkWidget *w,
	void (*cb)(
			GtkWidget *,
			GtkWidget *,
			gpointer
	),
	gpointer data
);
void CfgWinSetApplyCB(
	GtkWidget *w,
	void (*cb)(
			GtkWidget *,
			CfgList *,
			gpointer
	),
	gpointer data
);
gboolean CfgWinGetHasChanges(GtkWidget *w);
static void CfgWinSetHasChangesIterate(
	CfgWin *cfg_win,
	CfgWinPage *page,
	const gboolean has_changes
);
void CfgWinSetHasChanges(
	GtkWidget *w,
	const gboolean has_changes
);
void CfgWinUpdateMenus(GtkWidget *w);
void CfgWinSetBusy(
	GtkWidget *w,
	const gboolean busy
);
GtkWidget *CfgWinRef(GtkWidget *w);
GtkWidget *CfgWinUnref(GtkWidget *w);


#define CFG_WIN_WIDTH			640
#define CFG_WIN_HEIGHT			480

#define CFG_WIN_PROGRESS_BAR_TICKS	25


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Signal Callback:
 */
struct _CfgCallback {

	guint		id;			/* This signal callback's ID */

	gchar		*signal_name;

	void		(*cb)(
			CfgWinLink *,
			gpointer
	);
	gpointer	data;

};


/*
 *	Configuration Window "destroy" signal callback.
 */
static void CfgWinDestroyCB(gpointer data)
{
	CfgWin *cfg_win = CFG_WIN(data);
	if(cfg_win == NULL)
		return;

	if(cfg_win->ref_count > 0)
	{
		g_printerr(
"CfgWinDestroyCB():\
 Warning: Destroying CfgWin %p with %i reference counts remaining.\n\
Was CfgWinUnref() not used properly to unref this CfgWin?\n",
			cfg_win->toplevel,
			cfg_win->ref_count
		);
	}

	cfg_win->freeze_count++;

	/* The CfgWinLinks list and Pages list should already have
	 * been destroyed
	 */
	if(cfg_win->toplevel_pages_list != NULL)
	{
		g_printerr(
"CfgWinDestroyCB():\
 Warning: cfg_win=%p cfg_win->title=\"%s\" toplevel CfgWinPages list\
 cfg_win->toplevel_pages_list=%p (length=%i)\
 was not cleared properly.\n",
 cfg_win,
 cfg_win->title,
 cfg_win->toplevel_pages_list,
 g_list_length(cfg_win->toplevel_pages_list)
		);
		g_list_free(cfg_win->toplevel_pages_list);
		cfg_win->toplevel_pages_list = NULL;
	}	

	gtk_accel_group_unref(cfg_win->accelgrp);

	GDK_CURSOR_DESTROY(cfg_win->busy_cur);

	g_free(cfg_win->cfg_path);
	g_free(cfg_win->title);

	cfg_win->freeze_count--;

	g_free(cfg_win);
}

/*
 *	GtkWindow "delete_event" signal callback.
 */
static gint CfgWinDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	CfgWin *cfg_win = CFG_WIN(data);
	if(cfg_win == NULL)
		return(FALSE);

	CfgWinCancelCB(NULL, cfg_win);

	return(TRUE);
}

/*
 *	GtkWindow "realize" signal callback.
 */
static void CfgWinRealizeCB(GtkWidget *widget, gpointer data)
{
	GdkWindow *window;
	CfgWin *cfg_win = CFG_WIN(data);
	if((widget == NULL) || (cfg_win == NULL))
		return;

	window = widget->window;
	if(window != NULL)
	{
		gdk_window_set_decorations(
			window,
			GDK_DECOR_BORDER | GDK_DECOR_TITLE | GDK_DECOR_MENU |
				GDK_DECOR_MINIMIZE
		);
		gdk_window_set_functions(
			window,
			GDK_FUNC_MOVE | GDK_FUNC_MINIMIZE | GDK_FUNC_CLOSE
		);
		GUISetWMIcon(
			window,
			(guint8 **)cfg_win->icon_data
		);
	}

	/* Mark the CfgWin as realized */
	cfg_win->flags |= CFG_WIN_REALIZED;
}

/*
 *	Used by CfgWinSwitchPageCB() to get the page specified by
 *	the notebook and page number.
 */
static CfgWinPage *CfgWinSwitchPageCBIterate(
	GList *pages_list,
	GtkWidget *notebook,
	const gint page_num
)
{
	GList *glist;
	CfgWinPage *page, *child_page;

	for(glist = pages_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		page = CFG_WIN_PAGE(glist->data);
		if(page == NULL)
			continue;

		if(!(page->flags & CFG_WIN_PAGE_IS_PARENT))
			continue;

		/* Does this page have the specified GtkNotebook? */
		if(page->notebook == notebook)
			return(CFG_WIN_PAGE(g_list_nth_data(
				page->child_pages_list,
				(guint)page_num
			)));

		/* Search through this page's children */
		child_page = CfgWinSwitchPageCBIterate(
			page->child_pages_list,
			notebook,
			page_num
		);
		if(child_page != NULL)
			return(child_page);
	}

	return(NULL);
}

/*
 *	Toplevel or CfgWinPages GtkNotebook "switch_page" signal
 *	callback.
 */
static void CfgWinSwitchPageCB(
	GtkNotebook *notebook, GtkNotebookPage *page, guint page_num,
	gpointer data
)
{
	CfgWinPage *cfg_page;
	CfgWin *cfg_win = CFG_WIN(data);
	if((notebook == NULL) || (cfg_win == NULL))
		return;

	if(cfg_win->freeze_count > 0)
		return;

	cfg_win->freeze_count++;

	/* Get the switched to CfgWinPage */
	if(cfg_win->toplevel_notebook == GTK_WIDGET(notebook))
		cfg_page = CFG_WIN_PAGE(g_list_nth_data(
			cfg_win->toplevel_pages_list,
			page_num
		));
	else
		cfg_page = CfgWinSwitchPageCBIterate(
			cfg_win->toplevel_pages_list,
			GTK_WIDGET(notebook),
			(gint)page_num
		);

	/* Change in pages? */
	if(cfg_page != cfg_win->current_page)
	{
		cfg_win->current_page = cfg_page;

		/* Notify about the page change */
		if(cfg_win->page_changed_cb != NULL)
			cfg_win->page_changed_cb(
				cfg_win->toplevel,
				cfg_page->toplevel,
				cfg_win->page_changed_data
			);
	}

	CfgWinUpdateMenus(cfg_win->toplevel);

	cfg_win->freeze_count--;
}

/*
 *	Categories GtkCTree "tree_select_row" signal callback.
 */
static void CfgWinTreeSelectRowCB(
	GtkCTree *ctree, GtkCTreeNode *node, gint column,
	gpointer data
)
{
	CfgWin *cfg_win = CFG_WIN(data);
	if((ctree == NULL) || (cfg_win == NULL))
		return;

	if(cfg_win->freeze_count > 0)
		return;

	CfgWinSetBusy(cfg_win->toplevel, TRUE);
	cfg_win->freeze_count++;

	/* Show the CfgWinPage that this node represents */
	if(node != NULL)
	{
		CfgWinPage *cfg_page = CFG_WIN_PAGE(gtk_ctree_node_get_row_data(
			ctree,
			node
		));
		if(cfg_page != NULL)
		{
			gtk_widget_show(cfg_page->toplevel);
			if(cfg_page != cfg_win->current_page)
			{
				cfg_win->current_page = cfg_page;

				/* Notify about the page change */
				if(cfg_win->page_changed_cb != NULL)
					cfg_win->page_changed_cb(
						cfg_win->toplevel,
						cfg_page->toplevel,
						cfg_win->page_changed_data
					);
			}
		}
	}

	CfgWinUpdateMenus(cfg_win->toplevel);

	cfg_win->freeze_count--;
	CfgWinSetBusy(cfg_win->toplevel, FALSE);
}

/*
 *	Categories GtkCTree "tree_select_row" signal callback.
 */
static void CfgWinTreeUnselectRowCB(
	GtkCTree *ctree, GtkCTreeNode *node, gint column,
	gpointer data
)
{
	CfgWin *cfg_win = CFG_WIN(data);
	if((ctree == NULL) || (cfg_win == NULL))
		return;

	if(cfg_win->freeze_count > 0)
		return;

	cfg_win->freeze_count++;

	/* Hide the CfgWinPage that this node represents */
	if(node != NULL)
	{
		CfgWinPage *cfg_page = CFG_WIN_PAGE(gtk_ctree_node_get_row_data(
			ctree,
			node
		));
		if(cfg_page != NULL)
		{
			gtk_widget_hide(cfg_page->toplevel);
			if(cfg_page == cfg_win->current_page)
			{
				cfg_win->current_page = NULL;
			}
		}
	}

	CfgWinUpdateMenus(cfg_win->toplevel);

	cfg_win->freeze_count--;
}

/*
 *	OK callback.
 */
static void CfgWinOKCB(GtkWidget *widget, gpointer data)
{
	CfgWin *cfg_win = CFG_WIN(data);
	if(cfg_win == NULL)
		return;

	if(cfg_win->freeze_count > 0)
		return;

	CfgWinSetBusy(cfg_win->toplevel, TRUE);
	cfg_win->freeze_count++;

	CfgWinSetValues(
		cfg_win->toplevel,
		TRUE,				/* Show progress */
		TRUE,				/* Call the apply callback */
		TRUE				/* Save to disk */
	);
	CfgWinSetHasChanges(cfg_win->toplevel, FALSE);
	gtk_widget_hide(cfg_win->toplevel);

	cfg_win->freeze_count--;
	CfgWinSetBusy(cfg_win->toplevel, FALSE);
}

/*
 *	Apply callback.
 */
static void CfgWinApplyCB(GtkWidget *widget, gpointer data)
{
	CfgWin *cfg_win = CFG_WIN(data);
	if(cfg_win == NULL)
		return;

	if(cfg_win->freeze_count > 0)
		return;

	CfgWinSetBusy(cfg_win->toplevel, TRUE);
	cfg_win->freeze_count++;

	CfgWinSetValues(
		cfg_win->toplevel,
		TRUE,				/* Show progress */
		TRUE,				/* Call the apply callback */
		TRUE				/* Save to disk */
	);
	CfgWinSetHasChanges(cfg_win->toplevel, FALSE);
	CfgWinUpdateMenus(cfg_win->toplevel);

	cfg_win->freeze_count--;
	CfgWinSetBusy(cfg_win->toplevel, FALSE);
}

/*
 *	Save callback.
 */
static void CfgWinSaveCB(GtkWidget *widget, gpointer data)
{
	CfgWin *cfg_win = CFG_WIN(data);
	if(cfg_win == NULL)
		return;

	if(cfg_win->freeze_count > 0)
		return;

	CfgWinSetBusy(cfg_win->toplevel, TRUE);
	cfg_win->freeze_count++;

/* TODO */


	CfgWinUpdateMenus(cfg_win->toplevel);

	cfg_win->freeze_count--;
	CfgWinSetBusy(cfg_win->toplevel, FALSE);
}

/*
 *	Cancel callback.
 */
static void CfgWinCancelCB(GtkWidget *widget, gpointer data)
{
	CfgWin *cfg_win = CFG_WIN(data);
	if(cfg_win == NULL)
		return;

	if(cfg_win->freeze_count > 0)
		return;

	cfg_win->freeze_count++;

	gtk_widget_hide(cfg_win->toplevel);

	cfg_win->freeze_count--;
}

/*
 *	Any widget changed signal callback.
 *
 *	The specified widget must be ignored, since it may not be
 *	the actual referenced widget, cfg_link->w will be used instead.
 */
static void CfgWinAnyChangedCB(GtkWidget *widget, gpointer data)
{
	CfgWin *cfg_win;
	CfgWinLink *cfg_link = CFG_WIN_LINK(data);
	if(cfg_link == NULL)
		return;

	widget = cfg_link->w;

	/* Ignore changes? */
	if(cfg_link->flags & CFG_WIN_LINK_IGNORE_CHANGES)
		return;

	cfg_win = cfg_link->cfg_win;
	if(cfg_win == NULL)
		return;

	if(cfg_win->freeze_count > 0)
		return;

	cfg_win->freeze_count++;

	/* CfgWinLink does not have the has changes flag set? */
	if(!(cfg_link->flags & CFG_WIN_LINK_WIDGET_HAS_CHANGES))
	{
		/* Set the has changes flag on this CfgWinLink */
		cfg_link->flags |= CFG_WIN_LINK_WIDGET_HAS_CHANGES;
		CfgWinUpdateMenus(cfg_win->toplevel);
	}

	/* Configuration Window does not have the has changes flag set? */
	if(!(cfg_win->flags & CFG_WIN_HAS_CHANGES))
	{
		/* Set the has changes flag on the Configuration Window
		 * and update the menus
		 */
		cfg_win->flags |= CFG_WIN_HAS_CHANGES;
		CfgWinUpdateMenus(cfg_win->toplevel);
	}

	cfg_win->freeze_count--;

	/* Emit the "changed" signal to this CfgWinLink's signal
	 * callbacks
	 */
	CfgWinLinkEmitSignalByName(
		cfg_link,
		"changed"
	);
}
#if 0
static void CfgWinCListChangedCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	CfgWinAnyChangedCB(NULL, data);
}
#endif
static void CfgWinPopupListBoxChangedCB(
	GtkWidget *widget, const gint i, gpointer data
)
{
	CfgWinAnyChangedCB(NULL, data);
}
static void CfgWinStyleEditChangedCB(
	style_edit_struct *se, gpointer data
)
{
	CfgWinAnyChangedCB(NULL, data);
}
static void CfgWinStackListChangedCB(
	stack_list_struct *slist, gpointer data
)
{
	CfgWinAnyChangedCB(NULL, data);
}
static void CfgWinKeymapListChangedCB(
	keymap_list_struct *kmlist, gpointer data
)
{
	CfgWinAnyChangedCB(NULL, data);
}
static void CfgWinMenuCfgListChangedCB(
	menucfg_list_struct *list, gpointer data
)
{
	CfgWinAnyChangedCB(NULL, data);
}


/*
 *	CfgWinPage "destroy" signal callback.
 */
static void CfgWinPageDestroyCB(gpointer data)
{
	CfgWinPage *page = CFG_WIN_PAGE(data);
	if(page == NULL)
		return;
#if 0
g_print(
 "CfgWinPageDestroyCB(): Destroying page %p \"%s\"\n",
 page, page->label
);
#endif

	/* Was this page's CfgWinLinks not deleted? */
	if(page->links_list != NULL)
	{
		g_printerr(
"CfgWinPageDestroyCB(): Warning:\n\
 page=%p page->label=\"%s\" %i CfgWinLinks on this page were not deleted.\n\
Did you forget to use CfgWinPageRemove() to properly delete this page?\n",
			page,
			page->label,
			g_list_length(page->links_list)
		);
		g_list_free(page->links_list);
		page->links_list = NULL;
	}

	/* Was this page's child pages not deleted? */
	if(page->child_pages_list != NULL)
	{
		g_printerr(
"CfgWinPageDestroyCB(): Warning:\n\
 page=%p page->label=\"%s\" %i child CfgWinPages on this page were not deleted.\n\
Did you forget to use CfgWinPageRemove() to properly delete this page?\n",
			page,
			page->label,
			g_list_length(page->child_pages_list)
		);
		g_list_free(page->child_pages_list);
		page->child_pages_list = NULL;
	}

	/* Remove any references to this page on the CfgWin (except
	 * for the toplevel pages list which will be done further
	 * below)
	 */
	if(page->cfg_win != NULL)
	{
		/* If the CfgWin's style is CFG_WIN_STYLE_TREE then
		 * remove the Categories GtkCTree GtkCTreeNode that
		 * represents this page
		 */
		CfgWin *cfg_win = page->cfg_win;
		if(cfg_win->style == CFG_WIN_STYLE_TREE)
		{
			GtkCTree *ctree = GTK_CTREE(cfg_win->categories_ctree);
			if(ctree != NULL)
			{
				/* Get the GtkCTreeNode that represents this page
				 * and remove it
				 */
				GtkCTreeNode *node = page->ctree_node;
				if(node != NULL)
				{
					/* Remove this GtkCTreeNode, no "destroy"
					 * signal callback should be set on this node
					 */
					gtk_ctree_remove_node(ctree, node);
					page->ctree_node = NULL;
				}
			}
		}

		/* If this page is marked as the current page on the
		 * CfgWin then clear the CfgWin's current page marker
		 */
		if(cfg_win->current_page == page)
			cfg_win->current_page = NULL;
	}

	/* Remove this page from its parent's list of child pages */
	if(page->parent_page != NULL)
	{
		/* This page is a child of another page, remove this page
		 * from its parent's list of child pages
		 */
		CfgWinPage *parent_page = page->parent_page;
		parent_page->child_pages_list = g_list_remove(
			parent_page->child_pages_list,
			page
		);

		/* Update each sibling's page's index */
		CfgWinUpdatePageFamilyNumbers(parent_page->child_pages_list);
	}
	else if(page->cfg_win != NULL)
	{
		/* This page is a toplevel page, remove this page from the
		 * CfgWin's list of toplevel pages
		 */
		CfgWin *cfg_win = page->cfg_win;
		cfg_win->toplevel_pages_list = g_list_remove(
			cfg_win->toplevel_pages_list,
			page
		);

		/* Update each sibling's page's index */
		CfgWinUpdatePageFamilyNumbers(cfg_win->toplevel_pages_list);
	}

	g_free(page->label);
	g_free(page);
}


/*
 *      Save configuration to file callback.
 */
static gint CfgWinSetValuesSaveCB(
	const gulong i, const gulong m,
	gpointer data
) 
{
	const gint	stage = 1,
					nstages = 2;
	CfgWin *cfg_win = CFG_WIN(data);
	if(cfg_win == NULL)
		return(0);

	if(m != 0l)
		ProgressDialogUpdate(
			NULL, NULL, NULL, NULL,
			((gfloat)stage / (gfloat)nstages) +
				(((gfloat)i / (gfloat)m) / (gfloat)nstages),
			CFG_WIN_PROGRESS_BAR_TICKS,
			TRUE
		);
	else
		ProgressDialogUpdate(
			NULL, NULL, NULL, NULL,
			((gfloat)stage / (gfloat)nstages) +
				(0.0f / (gfloat)nstages),
			CFG_WIN_PROGRESS_BAR_TICKS,
			TRUE
		);

	if(ProgressDialogStopCount() > 0)
		return(-1);
	else
		return(0);
}


/*
 *	Creates a new CfgCallback.
 */
static CfgCallback *CfgCallbackNew(void)
{
	return(CFG_CALLBACK(g_malloc0(sizeof(CfgCallback))));
}

/*
 *	Deletes the CfgCallback.
 */
static void CfgCallbackDelete(CfgCallback *cfg_cb)
{
	if(cfg_cb == NULL)
		return;

	g_free(cfg_cb->signal_name);
	g_free(cfg_cb);
}


/*
 *	Creates a new CfgWinLink.
 */
static CfgWinLink *CfgWinLinkNew(void)
{
	return(CFG_WIN_LINK(g_malloc0(sizeof(CfgWinLink))));
}

/*
 *	Deletes the CfgWinLink.
 *
 *	The "destroy" signal will be emitted to the CfgWinLink's
 *	signal callbacks.
 *
 *	Any references to this CfgWinLink on its CfgWinPage will be
 *	removed.
 *
 *	The actual widget referenced by the CfgWinLink will also be
 *	destroyed.
 */
static void CfgWinLinkDelete(CfgWinLink *cfg_link)
{
	CfgWin *cfg_win;

	if(cfg_link == NULL)
		return;

#if 0
g_print(
 "CfgWinLinkDelete(): CfgWinLink=%p \"%s\" w=%p CfgWinPage=%p\n",
 cfg_link,
 cfg_link->cfg_parm,
 cfg_link->w,
 cfg_link->page
);
#endif

	cfg_win = cfg_link->cfg_win;

	/* Delete all the CfgCallbacks, calling all "destroy" signal
	 * callbacks during the process
	 */
	if(cfg_link->cbs_list != NULL)
	{
		GList *glist;
		CfgCallback *cfg_cb;

		for(glist = cfg_link->cbs_list;
		    glist != NULL;
			glist = g_list_next(glist)
		)
		{
			cfg_cb = CFG_CALLBACK(glist->data);
			if(cfg_cb == NULL)
				continue;

			if(cfg_cb->signal_name != NULL)
			{
				/* Is this signal named "destroy"? */
				if(!g_strcasecmp(cfg_cb->signal_name, "destroy"))
				{
					/* Call this signal callback */
					if(cfg_cb->cb != NULL)
						cfg_cb->cb(
							cfg_link,
							cfg_cb->data
						);
				}
			}

			/* Delete this CfgCallback */
			CfgCallbackDelete(cfg_cb);
			glist->data = NULL;
		}
		g_list_free(cfg_link->cbs_list);
		cfg_link->cbs_list = NULL;
	}

	/* Remove any references to this CfgWinLink on its page */
	if(cfg_link->page != NULL)
	{
		CfgWinPage *page = cfg_link->page;

		/* Remove this CfgWinLink from the CfgWin's CfgWinLinks list */
		page->links_list = g_list_remove(
			page->links_list,
			cfg_link
		);
	}

	/* Destroy this link's widget */
	if(cfg_link->w != NULL)
	{
		switch(cfg_link->type)
		{
		  case CFG_WIN_LINK_WIDGET_UNKNOWN:
			g_printerr(
"CfgWinLinkDelete():\n\
cfg_win=%p\n\
cfg_link->cfg_parm=\"%s\"\n\
cfg_link->w=%p\n\
\n\
Unable to destroy of type CFG_WIN_LINK_WIDGET_UNKNOWN.\n",
				cfg_win,
				cfg_link->cfg_parm,
				cfg_link->w
			);
			break;

		  case CFG_WIN_LINK_WIDGET_DRAWING_AREA:
		  case CFG_WIN_LINK_WIDGET_BUTTON:
		  case CFG_WIN_LINK_WIDGET_TOGGLE_BUTTON:
		  case CFG_WIN_LINK_WIDGET_RADIO_BUTTON:
		  case CFG_WIN_LINK_WIDGET_RANGE:
		  case CFG_WIN_LINK_WIDGET_EDITABLE:
		  case CFG_WIN_LINK_WIDGET_SPIN_BUTTON:
		  case CFG_WIN_LINK_WIDGET_COMBO:
		  case CFG_WIN_LINK_WIDGET_CLIST:
			gtk_widget_destroy((GtkWidget *)cfg_link->w);
			break;

		  case CFG_WIN_LINK_WIDGET_COLOR_BUTTON:
			gtk_widget_destroy((GtkWidget *)cfg_link->w);
			break;

		  case CFG_WIN_LINK_WIDGET_POPUP_LIST_BOX:
			gtk_widget_destroy((GtkWidget *)cfg_link->w);
			break;

		  case CFG_WIN_LINK_WIDGET_STYLE_EDIT:
			StyleEditDelete(STYLE_EDIT(cfg_link->w));
			break;
		  case CFG_WIN_LINK_WIDGET_STACK_LIST:
			StackListDelete(STACK_LIST(cfg_link->w));
			break;
		  case CFG_WIN_LINK_WIDGET_KEYMAP_LIST:
			KeymapListDelete(KEYMAP_LIST(cfg_link->w));
			break;
		  case CFG_WIN_LINK_WIDGET_MENUCFG_LIST:
			MenuCfgListDelete(MENUCFG_LIST(cfg_link->w));
			break;
		}
	}

	/* Delete this link's CfgItem parameter */
	g_free(cfg_link->cfg_parm);

	g_free(cfg_link);
}


/*
 *	Gets the CfgWinLink's type.
 */
CfgItemType CfgWinLinkGetType(CfgWinLink *cfg_link)
{
	if(cfg_link == NULL)
		return(CFG_ITEM_TYPE_NONE);

	return(cfg_link->type);
}

/*
 *	Gets the CfgWinLink's parameter.
 */
const gchar *CfgWinLinkGetParameter(CfgWinLink *cfg_link)
{
	if(cfg_link == NULL)
		return(NULL);

	return(cfg_link->cfg_parm);
}

/*
 *	Gets the CfgWinLink's widget.
 */
gpointer CfgWinLinkGetWidget(CfgWinLink *cfg_link)
{
	if(cfg_link == NULL)
		return(NULL);

	return(cfg_link->w);
}

/*
 *	Gets the CfgWinLink's parent CfgWinPage.
 */
GtkWidget *CfgWinLinkGetParentPage(CfgWinLink *cfg_link)
{
	CfgWinPage *cfg_page;

	if(cfg_link == NULL)
		return(NULL);

	cfg_page = cfg_link->page;
	if(cfg_page == NULL)
		return(NULL);

	return(cfg_page->toplevel);
}

/*
 *	Gets the CfgWinLink's CfgWin.
 */
GtkWidget *CfgWinLinkGetCfgWin(CfgWinLink *cfg_link)
{
	CfgWin *cfg_win;

	if(cfg_link == NULL)
		return(NULL);

	cfg_win = cfg_link->cfg_win;
	if(cfg_win == NULL)
		return(NULL);

	return(cfg_win->toplevel);
}


/*
 *	Sets the CfgWinLink to ignore changes.
 *
 *	This will cause the CfgWinLink to ignore any change signals
 *	received by its widget, furthermore, no "changed" signal
 *	will ever be emitted to this CfgWinLink's signal callbacks.
 */
void CfgWinLinkSetIgnoreChanges(
	CfgWinLink *cfg_link,
	const gboolean ignore_changes
)
{
	if(cfg_link == NULL)
		return;

	if(ignore_changes)
		cfg_link->flags |= CFG_WIN_LINK_IGNORE_CHANGES;
	else
		cfg_link->flags &= ~CFG_WIN_LINK_IGNORE_CHANGES;
}


/*
 *	Sets the CfgWinLink's data.
 *
 *	This data is not user data, it is data used for certain
 *	types of widgets.
 */
void CfgWinLinkSetData(
	CfgWinLink *cfg_link,
	gpointer data
)
{
	if(cfg_link == NULL)
		return;

	cfg_link->data = data;
}

/*
 *	Gets the CfgWinLink's data.
 *
 *	This data is not user data, it is data used for certain
 *	types of widgets.
 */
gpointer CfgWinLinkGetData(CfgWinLink *cfg_link)
{
	if(cfg_link == NULL)
		return(NULL);

	return(cfg_link->data);
}


/*
 *	CfgWinLink signal callback connect nexus.
 */
static guint CfgWinLinkSignalConnectNexus(
	CfgWinLink *cfg_link,
	const gchar *signal_name,
	void (*cb)(
			CfgWinLink *,
			gpointer
	),
	gpointer data,
	const gboolean after
)
{
	CfgCallback *cfg_cb;

	if((cfg_link == NULL) || STRISEMPTY(signal_name) || (cb == NULL))
		return(0);

	/* Create a new signal callback */
	cfg_cb = CfgCallbackNew();
	if(cfg_cb == NULL)
		return(0);

	/* Set the new signal callback */
	cfg_cb->id = (guint)cfg_cb;		/* Use its pointer value as
											 * its ID */
	cfg_cb->signal_name = g_strdup(signal_name);
	cfg_cb->cb = cb;
	cfg_cb->data = data;

	/* Add this signal callback to the CfgWinLink's signal callbacks
	 * list
	 */
	if(after)
		cfg_link->cbs_list = g_list_append(
			cfg_link->cbs_list,
			cfg_cb
		);
	else
		cfg_link->cbs_list = g_list_insert(
			cfg_link->cbs_list,
			cfg_cb,
			0
		);

	return(cfg_cb->id);
}

/*
 *	Connects a new CfgWinLink signal callback.
 */
guint CfgWinLinkSignalConnect(
	CfgWinLink *cfg_link,
	const gchar *signal_name,
	void (*cb)(
			CfgWinLink *,
			gpointer
	),
	gpointer data
)
{
	return(CfgWinLinkSignalConnectNexus(
		cfg_link,
		signal_name,
		cb,
		data,
		FALSE				/* Connect before */
	));
}

/*
 *	Connects a new CfgWinLink signal callback after all the
 *	other signal callbacks.
 */
guint CfgWinLinkSignalConnectAfter(
	CfgWinLink *cfg_link,
	const gchar *signal_name,
	void (*cb)(
			CfgWinLink *,
			gpointer
	),
	gpointer data
)
{
	return(CfgWinLinkSignalConnectNexus(
		cfg_link,
		signal_name,
		cb,
		data,
		TRUE				/* Connect after */
	));
}

/*
 *	Disconncets the CfgWinLink's signal callback.
 */
void CfgWinLinkSignalDisconnect(
	CfgWinLink *cfg_link,
	const guint id
)
{
	GList *glist;
	CfgCallback *cfg_cb;

	if((cfg_link == NULL) || (id == 0))
		return;

	for(glist = cfg_link->cbs_list; 
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		cfg_cb = CFG_CALLBACK(glist->data);
		if(cfg_cb == NULL)
			continue;

		/* Is this the signal to disconnect? */
		if(cfg_cb->id == id)
		{
			/* Delete this CfgCallback and remove it from the
			 * CfgWinLink's list of signal callbacks
			 */
			CfgCallbackDelete(cfg_cb);
			cfg_link->cbs_list = g_list_remove(
				cfg_link->cbs_list,
				cfg_cb
			);
			break;
		}
	}
}

/*
 *	Emits a signal by name.
 */
void CfgWinLinkEmitSignalByName(
	CfgWinLink *cfg_link,
	const gchar *signal_name 
)
{
	if((cfg_link == NULL) || STRISEMPTY(signal_name))
		return;

	/* Call this CfgWinLink's "changed" signal callbacks */
	if(cfg_link->cbs_list != NULL)
	{
		GList *glist;
		CfgCallback *cfg_cb;

		for(glist = cfg_link->cbs_list; 
			glist != NULL;
			glist = g_list_next(glist)
		)
		{
			cfg_cb = CFG_CALLBACK(glist->data);
			if(cfg_cb == NULL)
				continue;

			if(cfg_cb->signal_name == NULL)
				continue;

			/* Is this signal named "destroy"? */
			if(!g_strcasecmp(cfg_cb->signal_name, signal_name))
			{
				/* Call this signal callback */
				if(cfg_cb->cb != NULL)
					cfg_cb->cb(
						cfg_link,
						cfg_cb->data
					);
			}
		}
	}
}


/*
 *	Adds a new CfgWinLink to the Configuration Window.
 *
 *	The cfg_win specifies the Configuration Window.
 *
 *	The type specifies the what type of "widget" w is.
 *
 *	The w_ptr specifies the widget.
 *
 *	The cfg_parm specifies string describing the CfgItem's
 *	parameter which will be used to reference the CfgItem.
 *
 *	Returns the new CfgWinLink or NULL on error.
 */
CfgWinLink *CfgWinAddLink(
	GtkWidget *w,
	GtkWidget *page,
	const CfgWinLinkWidgetType type,
	gpointer widget,
	const gchar *cfg_parm
)
{
	gpointer w_ptr = widget;
	GtkWidget *toplevel;
	CfgWinLink *cfg_link;
	CfgWinPage *cfg_page;
	CfgWin *cfg_win = CfgWinGetWidgetData(
		w,
		"CfgWinAddLink"
	);
	if(cfg_win == NULL)
		return(NULL);

	cfg_page = CfgWinPageGetWidgetData(
		page,
		"CfgWinAddLink"
	);
	if(cfg_page == NULL)
		return(NULL);

	toplevel = cfg_win->toplevel;

	/* Create a new CfgWinLink */
	cfg_link = CfgWinLinkNew();
	if(cfg_link != NULL)
	{
		guint sig_id = 0;
		void (*change_cb)(GtkWidget *, gpointer) = CfgWinAnyChangedCB;
		gpointer change_data = cfg_link;

		/* Set the new CfgWinLink's values */
		cfg_link->type = type;
#if 0
		cfg_link->flags = 0;
#endif
		cfg_link->cfg_win = cfg_win;
		cfg_link->page = cfg_page;
		cfg_link->w = w_ptr;
		cfg_link->cfg_parm = STRDUP(cfg_parm);
#if 0
		cfg_link->data = NULL;
#endif

		/* Attach the "changed" signal callback based on the
		 * widget's type
		 */
		switch(type)
		{
		  case CFG_WIN_LINK_WIDGET_UNKNOWN:
			if(TRUE)
			{
				g_printerr(
"CfgWinAddLink():\n\
cfg_win=%p\n\
page=%p\n\
cfg_parm=\"%s\"\n\
widget=%p\n\
\n\
Unable to append a widget of type CFG_WIN_LINK_WIDGET_UNKNOWN.\n",
					cfg_win->toplevel,
					page,
					cfg_parm,
					w_ptr
				);
			}
			break;

		  case CFG_WIN_LINK_WIDGET_DRAWING_AREA:
			if((w_ptr != NULL) ? GTK_IS_DRAWING_AREA(w_ptr) : FALSE)
			{
				/* Does not generate a signal */
			}
			else
			{
				/* Not a GtkDrawingArea */
				g_printerr(
"CfgWinAddLink():\n\
cfg_win=%p\n\
page=%p\n\
type=CFG_WIN_LINK_WIDGET_DRAWING_AREA\n\
cfg_parm=\"%s\"\n\
widget=%p\n\
\n\
The specified widget was not, in fact, a GtkDrawingArea.\n",
					cfg_win->toplevel,
					page,
					cfg_parm,
					w_ptr
				);
			}
			break;
		  case CFG_WIN_LINK_WIDGET_BUTTON:
			if((w_ptr != NULL) ? GTK_IS_BUTTON(w_ptr) : FALSE)
			{
				GtkButton *btn = (GtkButton *)w_ptr;
				sig_id = gtk_signal_connect(
					GTK_OBJECT(btn), "clicked",
					GTK_SIGNAL_FUNC(change_cb), change_data
				);
				if(GTK_IS_TOGGLE_BUTTON(w_ptr))
				{
					g_printerr(
"CfgWinAddLink():\n\
cfg_win=%p\n\
page=%p\n\
type=CFG_WIN_LINK_WIDGET_BUTTON\n\
cfg_parm=\"%s\"\n\
widget=%p\n\
\n\
The specified widget's type was specified as\n\
CFG_WIN_LINK_WIDGET_BUTTON but the widget was detected\n\
to be a GtkToggleButton, you should use specify the\n\
type as CFG_WIN_LINK_WIDGET_TOGGLE_BUTTON instead.\n",
						cfg_win,
						page,
						cfg_parm,
						w_ptr
					);
				}
			}
			else
			{
				/* Not a GtkButton */
				g_printerr(
"CfgWinAddLink():\n\
cfg_win=%p\n\
page=%p\n\
type=CFG_WIN_LINK_WIDGET_BUTTON\n\
cfg_parm=\"%s\"\n\
widget=%p\n\
\n\
The specified widget was not, in fact, a GtkButton.\n",
					cfg_win,
					page,
					cfg_parm,
					w_ptr
				);
			}
			break;
		  case CFG_WIN_LINK_WIDGET_TOGGLE_BUTTON:
			if((w_ptr != NULL) ? GTK_IS_TOGGLE_BUTTON(w_ptr) : FALSE)
			{
				GtkToggleButton *tb = (GtkToggleButton *)w_ptr;
				sig_id = gtk_signal_connect(
					GTK_OBJECT(tb), "toggled",
					GTK_SIGNAL_FUNC(change_cb), change_data
				);
				if(GTK_IS_RADIO_BUTTON(w_ptr))
				{
					g_printerr(
"CfgWinAddLink():\n\
cfg_win=%p\n\
page=%p\n\
type=CFG_WIN_LINK_WIDGET_TOGGLE_BUTTON\n\
cfg_parm=\"%s\"\n\
widget=%p\n\
\n\
The specified widget's type was specified as\n\
CFG_WIN_LINK_WIDGET_TOGGLE_BUTTON but the widget was detected\n\
to be a GtkRadioButton, you should use specify the\n\
type as CFG_WIN_LINK_WIDGET_RADIO_BUTTON instead.\n",
						cfg_win,
						page,
						cfg_parm,
						w_ptr
					);
				}
			}
			else
			{
				/* Not a GtkToggleButton */
				g_printerr(
"CfgWinAddLink():\n\
cfg_win=%p\n\
page=%p\n\
type=CFG_WIN_LINK_WIDGET_TOGGLE_BUTTON\n\
cfg_parm=\"%s\"\n\
widget=%p\n\
\n\
The specified widget was not, in fact, a GtkToggleButton.\n",
					cfg_win,
					page,
					cfg_parm,
					w_ptr
				);
			}
			break;
		  case CFG_WIN_LINK_WIDGET_RADIO_BUTTON:
			if((w_ptr != NULL) ? GTK_IS_RADIO_BUTTON(w_ptr) : FALSE)
			{
				GtkToggleButton *tb = (GtkToggleButton *)w_ptr;
				sig_id = gtk_signal_connect(
					GTK_OBJECT(tb), "toggled",
					GTK_SIGNAL_FUNC(change_cb), change_data
				);
			}
			else
			{
				/* Not a GtkRadioButton */
				g_printerr(
"CfgWinAddLink():\n\
cfg_win=%p\n\
page=%p\n\
type=CFG_WIN_LINK_WIDGET_RADIO_BUTTON\n\
cfg_parm=\"%s\"\n\
widget=%p\n\
\n\
The specified widget was not, in fact, a GtkRadioButton.\n",
					cfg_win,
					page,
					cfg_parm,
					w_ptr
				);
			}
			break;
		  case CFG_WIN_LINK_WIDGET_RANGE:
			if((w_ptr != NULL) ? GTK_IS_RANGE(w_ptr) : FALSE)
			{
				GtkRange *range = (GtkRange *)w_ptr;
				GtkAdjustment *adj = range->adjustment;
				sig_id = gtk_signal_connect(
					GTK_OBJECT(adj), "value_changed",
					GTK_SIGNAL_FUNC(change_cb), change_data
				);
			}
			else
			{
				/* Not a GtkRange */
				g_printerr(
"CfgWinAddLink():\n\
cfg_win=%p\n\
page=%p\n\
type=CFG_WIN_LINK_WIDGET_RANGE\n\
cfg_parm=\"%s\"\n\
widget=%p\n\
\n\
The specified widget was not, in fact, a GtkRange.\n",
					cfg_win,
					page,
					cfg_parm,
					w_ptr
				);
			}
			break;
		  case CFG_WIN_LINK_WIDGET_EDITABLE:
			if((w_ptr != NULL) ? GTK_IS_EDITABLE(w_ptr) : FALSE)
			{
				GtkEditable *editable = (GtkEditable *)w_ptr;
				sig_id = gtk_signal_connect(
					GTK_OBJECT(editable), "changed",
					GTK_SIGNAL_FUNC(change_cb), change_data
				);
				if(GTK_IS_SPIN_BUTTON(w_ptr))
				{
					g_printerr(
"CfgWinAddLink():\n\
cfg_win=%p\n\
page=%p\n\
type=CFG_WIN_LINK_WIDGET_EDITABLE\n\
cfg_parm=\"%s\"\n\
widget=%p\n\
\n\
The specified widget's type was specified as\n\
CFG_WIN_LINK_WIDGET_EDITABLE but the widget was detected\n\
to be a GtkSpinButton, you should use specify the\n\
type as CFG_WIN_LINK_WIDGET_SPIN_BUTTON instead.\n",
						cfg_win,
						page,
						cfg_parm,
						w_ptr
					);
				}
			}
			else
			{
				/* Not a GtkEditable */
				g_printerr(
"CfgWinAddLink():\n\
cfg_win=%p\n\
page=%p\n\
type=CFG_WIN_LINK_WIDGET_EDITABLE\n\
cfg_parm=\"%s\"\n\
widget=%p\n\
\n\
The specified widget was not, in fact, a GtkEditable.\n",
					cfg_win,
					page,
					cfg_parm,
					w_ptr
				);
			}
			break;
		  case CFG_WIN_LINK_WIDGET_SPIN_BUTTON:
			if((w_ptr != NULL) ? GTK_IS_SPIN_BUTTON(w_ptr) : FALSE)
			{
				GtkEditable *editable = (GtkEditable *)w_ptr;
				sig_id = gtk_signal_connect(
					GTK_OBJECT(editable), "changed",
					GTK_SIGNAL_FUNC(change_cb), change_data
				);
			}
			else
			{
				/* Not a GtkSpinButton */
				g_printerr(
"CfgWinAddLink():\n\
cfg_win=%p\n\
page=%p\n\
type=CFG_WIN_LINK_WIDGET_SPIN_BUTTON\n\
cfg_parm=\"%s\"\n\
widget=%p\n\
\n\
The specified widget was not, in fact, a GtkSpinButton.\n",
					cfg_win,
					page,
					cfg_parm,
					w_ptr
				);
			}
			break;
		  case CFG_WIN_LINK_WIDGET_COMBO:
			if((w_ptr != NULL) ? GTK_IS_COMBO(w_ptr) : FALSE)
			{
				GtkCombo *combo = (GtkCombo *)w_ptr;
				GtkEntry *entry = GTK_ENTRY(combo->entry);
				sig_id = gtk_signal_connect(
					GTK_OBJECT(entry), "changed",
					GTK_SIGNAL_FUNC(change_cb), change_data
				);
			}
			else
			{
				/* Not a GtkCombo */
				g_printerr(
"CfgWinAddLink():\n\
cfg_win=%p\n\
page=%p\n\
type=CFG_WIN_LINK_WIDGET_COMBO\n\
cfg_parm=\"%s\"\n\
widget=%p\n\
\n\
The specified widget was not, in fact, a GtkCombo.\n",
					cfg_win,
					page,
					cfg_parm,
					w_ptr
				);
			}
			break;
		  case CFG_WIN_LINK_WIDGET_CLIST:
			if((w_ptr != NULL) ? GTK_IS_CLIST(w_ptr) : FALSE)
			{
#if 0
/* selecting the row should not trigger a change, only changing a row's
 * value/data
 */
				GtkCList *clist = (GtkCList *)w_ptr;
				sig_id = gtk_signal_connect(
					GTK_OBJECT(clist), "select_row",
					GTK_SIGNAL_FUNC(CfgWinCListChangedCB), change_data
				);
#endif
			}
			else
			{
				/* Not a GtkCList */
				g_printerr(
"CfgWinAddLink():\n\
cfg_win=%p\n\
page=%p\n\
type=CFG_WIN_LINK_WIDGET_CLIST\n\
cfg_parm=\"%s\"\n\
widget=%p\n\
\n\
The specified widget was not, in fact, a GtkCList.\n",
					cfg_win,
					page,
					cfg_parm,
					w_ptr
				);
			}
			break;

		  case CFG_WIN_LINK_WIDGET_COLOR_BUTTON:
			if(w_ptr != NULL)
			{
				GtkButton *btn = (GtkButton *)CSDColorButtonGetButton((GtkWidget *)w_ptr);
				gtk_signal_connect(
					GTK_OBJECT(btn), "clicked",
					GTK_SIGNAL_FUNC(CfgWinAnyChangedCB), cfg_link
				);
			}
			break;
		  case CFG_WIN_LINK_WIDGET_POPUP_LIST_BOX:
			if(w_ptr != NULL)
			{
				PUListBoxSetChangedCB(
					(GtkWidget *)w_ptr,
					CfgWinPopupListBoxChangedCB, cfg_link
				);
			}
			break;
		  case CFG_WIN_LINK_WIDGET_STYLE_EDIT:
			if(w_ptr != NULL)
			{
				style_edit_struct *style_edit = STYLE_EDIT(w_ptr);
				StyleEditSetChangedCB(
					style_edit,
					CfgWinStyleEditChangedCB, cfg_link
				);
			}
			break;
		  case CFG_WIN_LINK_WIDGET_STACK_LIST:
			if(w_ptr != NULL)
			{
				stack_list_struct *slist = STACK_LIST(w_ptr);
				StackListSetChangedCB(
					slist,
					CfgWinStackListChangedCB, cfg_link
				);
			}
			break;
		  case CFG_WIN_LINK_WIDGET_KEYMAP_LIST:
			if(w_ptr != NULL)
			{
				keymap_list_struct *kmlist = KEYMAP_LIST(w_ptr);
				KeymapListSetChangedCB(
					kmlist,
					CfgWinKeymapListChangedCB, cfg_link
				);
			}
			break;
		  case CFG_WIN_LINK_WIDGET_MENUCFG_LIST:
			if(w_ptr != NULL)
			{
				menucfg_list_struct *menucfg_list = MENUCFG_LIST(w_ptr);
				MenuCfgListSetChangedCB(
					menucfg_list,
					CfgWinMenuCfgListChangedCB, cfg_link
				);
			}
			break;
		}

		/* Append this CfgWinLink to the page's CfgWinLinks list */
		cfg_page->links_list = g_list_append(
			cfg_page->links_list,
			cfg_link
		);
	}

	return(cfg_link);
}

/*
 *	Called by CfgWinGetTotalCfgWinLinks().
 */
static gint CfgWinGetTotalCfgWinLinksIterate(
	CfgWin *cfg_win,
	CfgWinPage *page
)
{
	gint ncfg_widgets = g_list_length(page->links_list);
	GList *glist;
	CfgWinPage *child_page;

	for(glist = page->child_pages_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		child_page = CFG_WIN_PAGE(glist->data);
		if(child_page == NULL)
			continue;

		ncfg_widgets += CfgWinGetTotalCfgWinLinksIterate(
			cfg_win,
			child_page
		);
	}

	return(ncfg_widgets);
}

/*
 *	Gets the total number of all the CfgWinLinks on the
 *	Configuration Window
 */
gint CfgWinGetTotalCfgWinLinks(GtkWidget *w)
{
	gint ncfg_widgets;
	GList *glist;
	CfgWinPage *page;
	CfgWin *cfg_win = CfgWinGetWidgetData(
		w,
		"CfgWinGetTotalCfgWinLinks"
	);
	if(cfg_win == NULL)
		return(0);

	ncfg_widgets = 0;
	for(glist = cfg_win->toplevel_pages_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		page = CFG_WIN_PAGE(glist->data);
		if(page == NULL)
			continue;

		ncfg_widgets += CfgWinGetTotalCfgWinLinksIterate(
			cfg_win,
			page
		);
	}

	return(ncfg_widgets);
}

/*
 *	Called by CfgWinGetCfgWinLinkByParameter().
 */
static CfgWinLink *CfgWinGetCfgWinLinkByParameterIterate(
	CfgWin *cfg_win,
	CfgWinPage *page,
	const gchar *cfg_parm
)
{
	GList *glist;
	CfgWinPage *child_page;
	CfgWinLink *cfg_link;

	/* Check if the CfgWinLink is on this page */
	for(glist = page->links_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		cfg_link = CFG_WIN_LINK(glist->data);
		if(cfg_link == NULL)
			continue;

		if(cfg_link->cfg_parm == NULL)
			continue;

		if(!g_strcasecmp(cfg_link->cfg_parm, cfg_parm))
			return(cfg_link);
	}

	/* Not on this page, recurse the search into into this page's
	 * child pages
	 */
	for(glist = page->child_pages_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		child_page = CFG_WIN_PAGE(glist->data);
		if(child_page == NULL)
			continue;

		cfg_link = CfgWinGetCfgWinLinkByParameterIterate(
			cfg_win,
			child_page,
			cfg_parm
		);
		if(cfg_link != NULL)
			return(cfg_link);
	}

	return(NULL);
}

/*
 *	Gets the CfgWinLink by configuration parameter.
 */
CfgWinLink *CfgWinGetCfgWinLinkByParameter(
	GtkWidget *w,
	const gchar *cfg_parm
)
{
	GList *glist;
	CfgWinLink *cfg_link;
	CfgWinPage *page;
	CfgWin *cfg_win = CfgWinGetWidgetData(
		w,
		"CfgWinGetCfgWinLinkByParameter"
	);
	if((cfg_win == NULL) || STRISEMPTY(cfg_parm))
		return(NULL);

	/* Check if the CfgWinLink is on one of the toplevel pages
	 * and recurse the search into any child pages
	 */
	for(glist = cfg_win->toplevel_pages_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		page = CFG_WIN_PAGE(glist->data);
		if(page == NULL)
			continue;

		cfg_link = CfgWinGetCfgWinLinkByParameterIterate(
			cfg_win,
			page,
			cfg_parm
		);
		if(cfg_link != NULL)
			return(cfg_link);
	}

	return(NULL);
}

/*
 *	Removes the CfgWinLink from the Configuration Window.
 */
void CfgWinRemoveLink(
	GtkWidget *w,
	CfgWinLink *cfg_link 
)
{
	CfgWin *cfg_win = CfgWinGetWidgetData(
		w,
		"CfgWinRemoveLink"
	);
	if((cfg_win == NULL) || (cfg_link == NULL))
		return;

	/* Emit the "destroy" signal to all of this CfgWinLink's
	 * signal callbacks, remove any references to this CfgWinLink
	 * on its CfgWinPage, and delete this CfgWinLink's widget
	 */
	CfgWinLinkDelete(cfg_link);
}


/*
 *	Gets the CfgWin data from the GtkWidget.
 */
static CfgWinPage *CfgWinPageGetWidgetData(
	GtkWidget *w,
	const gchar *func_name
)
{
	const gchar *key = CFG_WIN_PAGE_KEY;
	CfgWinPage *cfg_page;

	if(w == NULL)
		return(NULL);

	cfg_page = CFG_WIN_PAGE(gtk_object_get_data(
		GTK_OBJECT(w),
		key
	));
	if(cfg_page == NULL)
	{
		g_printerr(
"%s(): Warning: GtkWidget %p:\
 Unable to find the data that matches the key \"%s\".\n",
			func_name,
			w,
			key
		);
		return(NULL);
	}

	return(cfg_page);
}

/*
 *	Gets the page's parent page.
 */
GtkWidget *CfgWinPageGetParentPage(GtkWidget *page)
{
	CfgWinPage	*cfg_parent_page,
					*cfg_page = CfgWinPageGetWidgetData(
		page,
		"CfgWinPageGetParentPage"
	);
	if(cfg_page == NULL)
		return(NULL);

	cfg_parent_page = cfg_page->parent_page;
	if(cfg_parent_page == NULL)
		return(NULL);

	return(cfg_parent_page->toplevel);
}

/*
 *	Gets the page's first child page.
 */
GtkWidget *CfgWinPageGetChild(GtkWidget *page)
{
	GList *glist;
	CfgWinPage	*cfg_child_page,
					*cfg_page = CfgWinPageGetWidgetData(
		page,
		"CfgWinPageGetChild"
	);
	if(cfg_page == NULL)
		return(NULL);

	glist = cfg_page->child_pages_list;
	if(glist == NULL)
		return(NULL);

	cfg_child_page = CFG_WIN_PAGE(glist->data);
	if(cfg_child_page == NULL)
		return(NULL);

	return(cfg_child_page->toplevel);
}

/*
 *	Gets the page's sibling page (next page).
 */
GtkWidget *CfgWinPageGetSibling(GtkWidget *page)
{
	GList *glist;
	CfgWinPage	*cfg_parent_page,
					*cfg_child_page,
					*cfg_page = CfgWinPageGetWidgetData(
		page,
		"CfgWinPageGetSibling"
	);
	if(cfg_page == NULL)
		return(NULL);

	/* Get the specified page's parent page */
	cfg_parent_page = cfg_page->parent_page;
	if(cfg_parent_page == NULL)
		return(NULL);

	/* Iterate through the parent page's list of child pages */
	for(glist = cfg_parent_page->child_pages_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		cfg_child_page = CFG_WIN_PAGE(glist->data);
		if(cfg_child_page == NULL)
			continue;

		/* Is this child page the specified page? */
		if(cfg_child_page == cfg_page)
		{
			/* Get the next page, which should be the specified
			 * page's sibling
			 */
			glist = g_list_next(glist);
			if(glist == NULL)
				break;

			cfg_child_page = CFG_WIN_PAGE(glist->data);
			if(cfg_child_page == NULL)
				break;

			return(cfg_child_page->toplevel);
		}
	}

	return(NULL);
}

/*
 *	Gets the page's label.
 */
const gchar *CfgWinPageGetLabel(GtkWidget *page)
{
	CfgWinPage *cfg_page = CfgWinPageGetWidgetData(
		page,
		"CfgWinPageGetLabel"
	);
	if(cfg_page == NULL)
		return(NULL);

	return(cfg_page->label);
}

/*
 *	Gets the page's index relative to its parent page (birth
 *	order).
 */
gint CfgWinPageGetIndex(GtkWidget *page)
{
	CfgWinPage *cfg_page = CfgWinPageGetWidgetData(
		page,
		"CfgWinPageGetLabel"
	);
	if(cfg_page == NULL)
		return(0);

	return(cfg_page->page_num);
}


/*
 *	Updates each page's index.
 */
static void CfgWinUpdatePageFamilyNumbers(GList *pages_list)
{
	gint i;
	GList *glist;
	CfgWinPage *cfg_page;

	for(glist = pages_list, i = 0;
		glist != NULL;
		glist = g_list_next(glist), i++
	)
	{
		cfg_page = CFG_WIN_PAGE(glist->data);
		if(cfg_page == NULL)
			continue;

		cfg_page->page_num = i;
	}
}

/*
 *	Adds a page to the CfgWin.
 *
 *	The w specifies the Configuration Window, the Configuration
 *	Window's style must be CFG_WIN_STYLE_NOTEBOOK.
 *
 *	The parent_page specifies the parent page who's GtkNotebook
 *	the new page will be added to. If parent_page is NULL
 *	then the new page will be added to the toplevel GtkNotebook.
 *
 *	The insert_pos specifies the position to insert the new page
 *	at, if insert_pos is -1 then the new page is appended.
 *
 *	The label specifies a string describing the page's label.
 *
 *	The tab_label_widget specifies the GtkWidget to be used as the
 *	tab widget. If tab_label_widget is NULL then a generic
 *	GtkWidget will be used for the tab.
 *	
 *	If is_parent is TRUE then the new page will have a GtkNotebook
 *	of its own and can have child pages.
 *
 *	Returns the new page or NULL on error.
 */
GtkWidget *CfgWinAddPageNotebook(
	GtkWidget *w,
	GtkWidget *parent_page,
	const gint insert_pos,
	const gchar *label,
	GtkWidget *tab_label_widget,
	const gboolean is_parent
)
{
	const gint border_major = 5;
	gboolean is_first_page;
	GtkNotebook *notebook;
	CfgWinPage	*cfg_parent_page,
					*cfg_page;
	CfgWin *cfg_win = CfgWinGetWidgetData(
		w,
		"CfgWinAddPageNotebook"
	);
	if(cfg_win == NULL)
		return(NULL);

	if(cfg_win->style != CFG_WIN_STYLE_NOTEBOOK)
		return(NULL);

	if(parent_page != NULL)
	{
		cfg_parent_page = CfgWinPageGetWidgetData(
			parent_page,
			"CfgWinAddPageNotebook"
		);
		if(cfg_parent_page == NULL)
			return(NULL);
	}
	else
	{
		cfg_parent_page = NULL;
	}

	/* Get the parent GtkNotebook to add this page to */
	if(cfg_parent_page != NULL)
	{
		/* Is it allowed to be a parent? */
		if(!(cfg_parent_page->flags & CFG_WIN_PAGE_IS_PARENT))
			g_printerr(
"CfgWinAddPageNotebook(): Warning: Parent page \"%s\"\
 is not set to be a parent.\n",
				cfg_parent_page->label
			);
		notebook = GTK_NOTEBOOK(cfg_parent_page->notebook);
		if((notebook == NULL) && (cfg_parent_page->flags & CFG_WIN_PAGE_IS_PARENT))
			g_printerr(
"CfgWinAddPageNotebook(): Warning: Parent page \"%s\"\
 is set to be a parent but does not have a GtkNotebook to hold child pages.\n",
				cfg_parent_page->label
			);
	}
	else
	{
		notebook = GTK_NOTEBOOK(cfg_win->toplevel_notebook);
	}
	if(notebook == NULL)
		return(NULL);

	/* Create the new page */
	cfg_page = CFG_WIN_PAGE(g_malloc0(sizeof(CfgWinPage)));
	if(cfg_page == NULL)
		return(NULL);

	cfg_page->cfg_win = cfg_win;
	cfg_page->flags = ((is_parent) ? CFG_WIN_PAGE_IS_PARENT : 0);
	cfg_page->parent_page = cfg_parent_page;
	cfg_page->label = STRDUP(label);

	is_first_page = FALSE;

	/* Is the new page going to be a toplevel page? */
	if(cfg_parent_page != NULL)
	{
		/* Add this page to the parent page's list of child pages */
		if(insert_pos < 0)
			cfg_parent_page->child_pages_list = g_list_append(
				cfg_parent_page->child_pages_list,
				cfg_page
			);
		else
			cfg_parent_page->child_pages_list = g_list_insert(
				cfg_parent_page->child_pages_list,
				cfg_page,
				insert_pos
			);

		/* Update each sibling's page's index */
		CfgWinUpdatePageFamilyNumbers(cfg_parent_page->child_pages_list);
	}
	else
	{
		if(cfg_win->toplevel_pages_list == NULL)
			is_first_page = TRUE;

		/* Add this page to the CfgWin's list of toplevel pages */
		if(insert_pos < 0)
			cfg_win->toplevel_pages_list = g_list_append(
				cfg_win->toplevel_pages_list,
				cfg_page
			);
		else
			cfg_win->toplevel_pages_list = g_list_insert(
				cfg_win->toplevel_pages_list,
				cfg_page,
				insert_pos
			);

		/* Update each sibling's page's index */
		CfgWinUpdatePageFamilyNumbers(cfg_win->toplevel_pages_list);
	}

	/* If no label widget is given then create one */
	if(tab_label_widget == NULL)
	{
		if(label != NULL)
		{
			tab_label_widget = gtk_label_new(label);
		}
		else
		{
			gchar *s = g_strdup_printf(
				"Page %i",
				cfg_page->page_num + 1
			);
			g_free(s);
		}
		gtk_label_set_justify(
			GTK_LABEL(tab_label_widget),
			GTK_JUSTIFY_CENTER
		);
	}

	/* Freeze the CfgWin so that the GtkNotebook's "switched_page"
	 * signal callback does nothing, this is needed because
	 * gtk_notebook_*_page makes redunant calls
	 */
	cfg_win->freeze_count++;

	/* Create the new page's toplevel GtkVBox */
	cfg_page->toplevel = w = gtk_vbox_new(FALSE, 0);
	gtk_object_set_data_full(
		GTK_OBJECT(w), CFG_WIN_PAGE_KEY,
		cfg_page, CfgWinPageDestroyCB
	);
	if(insert_pos < 0)
		gtk_notebook_append_page(
			notebook,
			w,
			tab_label_widget
		);
	else
		gtk_notebook_insert_page(
			notebook,
			w,
			tab_label_widget,
			insert_pos
		);
	gtk_widget_show(w);

	/* Is this page going to be a parent? */
	if(is_parent)
	{
		/* Add a border to the new page's toplevel GtkBox */
		gtk_container_border_width(
			GTK_CONTAINER(cfg_page->toplevel),
			border_major
		);

		/* Create this page's GtkNotebook */
		cfg_page->notebook = w = gtk_notebook_new();
		gtk_notebook_set_tab_pos(GTK_NOTEBOOK(w), GTK_POS_TOP);
		gtk_box_pack_start(GTK_BOX(cfg_page->toplevel), w, TRUE, TRUE, 0);
		gtk_notebook_set_scrollable(GTK_NOTEBOOK(w), TRUE);
		gtk_notebook_set_show_tabs(GTK_NOTEBOOK(w), TRUE);
		gtk_notebook_set_show_border(GTK_NOTEBOOK(w), TRUE);
		gtk_signal_connect(
			GTK_OBJECT(w), "switch_page",
			GTK_SIGNAL_FUNC(CfgWinSwitchPageCB), cfg_win
		);
		gtk_widget_show(w);
	}

	cfg_win->freeze_count--;

	/* If this is the first page then we need to manually set the
	 * CfgWin's current page and call the CfgWin's page changed
	 * callback because the CfgWin was frozen and the
	 * GtkNotebook's "switched_page" signal callback has no affect
	 */
	if(is_first_page)
	{
		cfg_win->current_page = cfg_page;

		if(cfg_win->page_changed_cb != NULL)
			cfg_win->page_changed_cb(
				cfg_win->toplevel,
				cfg_page->toplevel,
				cfg_win->page_changed_data
			);
	}

	return(cfg_page->toplevel);
}

/*
 *	Adds a page to the CfgWin.
 *
 *	The w specifies the Configuration Window, the Configuration
 *	Window's style must be CFG_WIN_STYLE_TREE.
 *
 *	The parent_page specifies the parent page who's tree node the
 *	new page will be added to. If parent_page is NULL then the
 *	new page will be added as a toplevel node.
 *
 *	The sibling_page specifies the sibling page to insert the
 *	new page at. If sibling_page is NULL then the new page is
 *	appended.
 *
 *	The label specifies a string describing the page's label.
 *
 *	The pixmap_closed, mask_closed, pixmap_opened, and mask_opened
 *	specifies the icon that is to be displayed on the categories
 *	list for the new page.
 *
 *	If create_heading is TRUE then a heading describing the
 *	specified label will be added to the top of the page.
 *
 *	If is_parent is TRUE then the new page will have a GtkNotebook
 *	of its own and can have child pages.
 *
 *	Returns the new page or NULL on error.
 */
GtkWidget *CfgWinAddPageTree(
	GtkWidget *w,
	GtkWidget *parent_page,
	GtkWidget *sibling_page,
	const gchar *label,
	GdkPixmap *pixmap_closed,
	GdkBitmap *mask_closed,
	GdkPixmap *pixmap_opened,
	GdkBitmap *mask_opened,
	const gboolean create_heading,
	const gboolean is_parent
)
{
	const gint border_minor = 2;
	gboolean is_first_page;
	GtkWidget *pages_toplevel;
	GtkCTree *ctree;
	GtkCTreeNode	*parent_node,
					*sibling_node;
	CfgWinPage	*cfg_parent_page,
					*cfg_sibling_page,
					*cfg_page;
	CfgWin *cfg_win = CfgWinGetWidgetData(
		w,
		"CfgWinAddPageTree"
	);
	if(cfg_win == NULL)
		return(NULL);

	if(cfg_win->style != CFG_WIN_STYLE_TREE)
		return(NULL);

	pages_toplevel = cfg_win->pages_toplevel;
	ctree = GTK_CTREE(cfg_win->categories_ctree);
	if((pages_toplevel == NULL) || (ctree == NULL))
		return(NULL);

	/* Get the parent page */
	if(parent_page != NULL)
	{
		cfg_parent_page = CfgWinPageGetWidgetData(
			parent_page,
			"CfgWinAddPageTree"
		);
		if(cfg_parent_page == NULL)
			return(NULL);

		/* Is it allowed to be a parent? */
		if(!(cfg_parent_page->flags & CFG_WIN_PAGE_IS_PARENT))
			g_printerr(
"CfgWinAddPageTree(): Warning: Parent page \"%s\"\
 is not set to be a parent.\n",
				cfg_parent_page->label
			);

		parent_node = cfg_parent_page->ctree_node;
		if(parent_node == NULL)
		{
			g_printerr(
"CfgWinAddPageTree(): Warning: Unable to get the GtkCTreeNode for\
 the parent page \"%s\".\n",
				cfg_parent_page->label
			);
			return(NULL);
		}
	}
	else
	{
		cfg_parent_page = NULL;
		parent_node = NULL;
	}

	/* Get the sibling page */
	if(sibling_page != NULL)
	{
		cfg_sibling_page = CfgWinPageGetWidgetData(
			sibling_page,
			"CfgWinAddPageTree"
		);
		if(cfg_sibling_page == NULL)
			return(NULL);

		sibling_node = cfg_sibling_page->ctree_node;
		if(sibling_node == NULL)
		{
			g_printerr(
"CfgWinAddPageTree(): Warning: Unable to get the GtkCTreeNode for\
 the sibling page \"%s\".\n",
				cfg_sibling_page->label
			);
			return(NULL);
		}
	}
	else
	{
		cfg_sibling_page = NULL;
		sibling_node = NULL;
	}

	/* Create the new page */
	cfg_page = CFG_WIN_PAGE(g_malloc0(sizeof(CfgWinPage)));
	if(cfg_page == NULL)
		return(NULL);

	cfg_page->cfg_win = cfg_win;
	cfg_page->flags = ((is_parent) ? CFG_WIN_PAGE_IS_PARENT : 0);
	cfg_page->parent_page = cfg_parent_page;
	cfg_page->label = STRDUP(label);

	is_first_page = FALSE;

	/* Is the new page going to be a toplevel page? */
	if(cfg_parent_page != NULL)
	{
		/* Add this page to the parent page's list of child pages */
		if(cfg_sibling_page != NULL)
			cfg_parent_page->child_pages_list = g_list_insert(
				cfg_parent_page->child_pages_list,
				cfg_page,
				cfg_sibling_page->page_num
			);
		else
			cfg_parent_page->child_pages_list = g_list_append(
				cfg_parent_page->child_pages_list,
				cfg_page
			);

		/* Update each sibling's page's index */
		CfgWinUpdatePageFamilyNumbers(cfg_parent_page->child_pages_list);
	}
	else
	{
		if(cfg_win->toplevel_pages_list == NULL)
			is_first_page = TRUE;

		/* Add this page to the CfgWin's list of toplevel pages */
		if(cfg_sibling_page != NULL)
			cfg_win->toplevel_pages_list = g_list_insert(
				cfg_win->toplevel_pages_list,
				cfg_page,
				cfg_sibling_page->page_num
			);
		else
			cfg_win->toplevel_pages_list = g_list_append(
				cfg_win->toplevel_pages_list,
				cfg_page
			);

		/* Update each sibling's page's index */
		CfgWinUpdatePageFamilyNumbers(cfg_win->toplevel_pages_list);
	}

	/* Create the new page's toplevel GtkVBox */
	cfg_page->toplevel = w = gtk_vbox_new(FALSE, 0);
	gtk_object_set_data_full(
		GTK_OBJECT(w), CFG_WIN_PAGE_KEY,
		cfg_page, CfgWinPageDestroyCB
	);
	gtk_box_pack_start(GTK_BOX(pages_toplevel), w, TRUE, TRUE, 0);
	if(is_first_page)
		gtk_widget_show(w);

	if(create_heading)
	{
		GtkStyle *style;
		GtkWidget	*w,
					*parent = cfg_page->toplevel;

		w = cfg_win->toplevel;
		if(!GTK_WIDGET_REALIZED(w))
			gtk_widget_realize(w);

		style = gtk_widget_get_style(w);
		if(style != NULL)
		{
			gint font_size;
			gchar *font_name;
#if defined(PROG_LANGUAGE_POLISH)
			const gchar *font_encoding = "iso8859-2";
#else
			const gchar *font_encoding = "iso8859-1";
#endif
			GdkFont *font = NULL;

			font_size = GDK_FONT_GET_FONT_NAME_SIZE(style->font) * 1.5f;
			while(font_size > 0)
			{
				font_name = g_strdup_printf(
"-adobe-helvetica-bold-r-normal-*-%i-*-*-*-*-*-%s",
					font_size,
					font_encoding
				);
				font = gdk_font_load(font_name);
				g_free(font_name);
				if(font != NULL)
					break;

				font_size--;
			}
			w = GUIBannerCreate(
				label,
				font,
				&style->bg[GTK_STATE_NORMAL],
				&style->text[GTK_STATE_NORMAL],
				&style->fg[GTK_STATE_NORMAL],
				GTK_JUSTIFY_LEFT,
				TRUE,			/* Expand */
				border_minor,		/* Inside border */
				border_minor		/* Outside border */
			);
			font = GDK_FONT_UNREF(font);
			gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
			gtk_widget_show(w);
		}
	}

	/* Create the Categories GtkCTree GtkCTreeNode that will
	 * represent this page
	 */
	if(ctree != NULL)
	{
		gint i;
		GtkCList *clist = GTK_CLIST(ctree);
		GtkCTreeNode *node;
		const gint ncolumns = MAX(clist->columns, 1);
		gchar **strv = (gchar **)g_malloc(ncolumns * sizeof(gchar *));

		for(i = 0; i < ncolumns; i++)
			strv[i] = "";

		strv[0] = cfg_page->label;

		cfg_win->freeze_count++;

		gtk_clist_freeze(clist);
		cfg_page->ctree_node = node = gtk_ctree_insert_node(
			ctree,
			parent_node,
			sibling_node,
			strv,
			2,
			pixmap_closed, mask_closed,
			pixmap_opened, mask_opened,
			!is_parent,			/* Is leaf? */
			FALSE				/* Not expanded */
		);
		gtk_ctree_node_set_row_data(
			ctree,
			node,
			cfg_page
		);
		if(is_first_page)
			gtk_ctree_select(ctree, node);
		gtk_clist_thaw(clist);

		cfg_win->freeze_count--;

		g_free(strv);
	}

	/* If this is the first page then we need to manually set the
	 * CfgWin's current page and call the CfgWin's page changed
	 * callback because the CfgWin was frozen and the
	 * GtkCTree's "tree_select_row" signal callback has no affect
	 */
	if(is_first_page)
	{
		cfg_win->current_page = cfg_page;

		if(cfg_win->page_changed_cb != NULL)
			cfg_win->page_changed_cb(
				cfg_win->toplevel,
				cfg_page->toplevel,
				cfg_win->page_changed_data
			);
	}

	return(cfg_page->toplevel);
}

/*
 *	Called by CfgWinRemovePage().
 */
static void CfgWinRemovePageIterate(
	CfgWin *cfg_win,
	CfgWinPage *cfg_page
)
{
	GtkNotebook *notebook;

	/* Delete all the child pages */
	if(cfg_page->child_pages_list != NULL)
	{
		GList *glist;
		CfgWinPage *child_page;

		while(cfg_page->child_pages_list != NULL)
		{
			glist = cfg_page->child_pages_list;
			child_page = CFG_WIN_PAGE(glist->data);
			if(child_page != NULL)
				CfgWinRemovePageIterate(
					cfg_win,
					child_page
				);
			else
				cfg_page->child_pages_list = g_list_remove(
					cfg_page->child_pages_list,
					NULL
				);
		}
	}

	/* Delete all the CfgWinLinks */
	if(cfg_page->links_list != NULL)
	{
		GList *glist;
		CfgWinLink *cfg_link;

		while(cfg_page->links_list != NULL)
		{
			glist = cfg_page->links_list;
			cfg_link = CFG_WIN_LINK(glist->data);
			if(cfg_link != NULL)
				CfgWinLinkDelete(cfg_link);
			else
				cfg_page->links_list = g_list_remove(
					cfg_page->links_list,
					NULL
				);
		}
	}

	/* If this page does not have a toplevel GtkWidget then it
	 * means that it was not properly initialized and must be
	 * deleted explicitly
	 */
	if(cfg_page->toplevel == NULL)
	{
		GTK_WIDGET_DESTROY(cfg_page->notebook);
		g_free(cfg_page->label);
		g_free(cfg_page);
		return;
	}

	/* Destroy the rest of this page based on the CfgWin's
	 * style
	 */
	switch(cfg_win->style)
	{
	  case CFG_WIN_STYLE_NOTEBOOK:
		if(cfg_page->parent_page != NULL)
		{
			CfgWinPage *cfg_parent_page = cfg_page->parent_page;
			notebook = GTK_NOTEBOOK(cfg_parent_page->notebook);
		}
		else
		{
			notebook = GTK_NOTEBOOK(cfg_win->toplevel_notebook);
		}
		if(notebook != NULL)
		{
			/* Remove this page from the GtkNotebook, which will
			 * destroy this page's and all its child pages'
			 * toplevel GtkWidgets and which will call their
			 * "destroy" signal callbacks CfgWinPageDestroyCB()
			 * that will further delete the rest of this page and
			 * remove any references to this page on its parent
			 * page
			 */
			gtk_notebook_remove_page(
				notebook,
				cfg_page->page_num
			);
		}
		else
		{
			g_printerr(
"CfgWinRemovePageIterate(): Warning:\n\
 Destroying page \"%s\" with no parent page GtkNotebook.\n",
				cfg_page->label
			);

			/* Destroy this page's toplevel GtkWidget which will
			 * call its "destroy" signal callback and deleting
			 * this page and remove all references to this page
			 * on its parent
			 */
			gtk_widget_destroy(cfg_page->toplevel);
		}
		break;

	  case CFG_WIN_STYLE_TREE:
		/* Destroy this page's and all its child pages' toplevel
		 * GtkWidgets which will call their "destroy" signal
		 * callbacks CfgWinPageDestroyCB() that will further
		 * delete the rest of this page and remove any references
		 * to this page on its parent page and the Categories
		 * GtkCTree
		 */
		gtk_widget_destroy(cfg_page->toplevel);
		break;
	}
}

/*
 *	Removes the page.
 *
 *	The w specifies the Configuration Window.
 *
 *	The page specifies the page to remove.
 */
void CfgWinRemovePage(
	GtkWidget *w,
	GtkWidget *page 
)
{
	CfgWinPage *cfg_page;
	CfgWin *cfg_win = CfgWinGetWidgetData(
		w,
		"CfgWinRemovePage"
	);
	if(cfg_win == NULL)
		return;

	if(page == NULL)
		return;

	cfg_page = CfgWinPageGetWidgetData(
		w,
		"CfgWinRemovePage"
	);
	if(cfg_page == NULL)
		return;

	CfgWinRemovePageIterate(
		cfg_win,
		cfg_page
	);
}


/*
 *	Selects/switches to the page.
 *
 *	The w specifies the Configuration Window.
 *
 *	The page specifies the page to select/switch to.
 */
void CfgWinSetPage(
	GtkWidget *w,
	GtkWidget *page
)
{
	GtkCTree *ctree;
	GtkNotebook *notebook;
	CfgWinPage *cfg_page, *cfg_parent_page;
	CfgWin *cfg_win = CfgWinGetWidgetData(
		w,
		"CfgWinSetPage"
	);
	if(cfg_win == NULL)
		return;

	/* Default page? */
	if(page == NULL)
		page = CfgWinGetDefaultPage(w);
	if(page == NULL)
		return;

	cfg_page = CfgWinPageGetWidgetData(
		page,
		"CfgWinSetPage"
	);
	if(cfg_page == NULL)
		return;

	/* Is this already the current page? */
	if(cfg_win->current_page == cfg_page)
		return;

	switch(cfg_win->style)
	{
	  case CFG_WIN_STYLE_NOTEBOOK:
		/* First we need to switch to all of the specified page's
		 * parent pages so that the specified page will show on
		 * the GtkNotebooks
		 */
		cfg_parent_page = cfg_page->parent_page;
		while(cfg_parent_page != NULL)
		{
			/* Get the parent page's GtkNotebook and set the
			 * GtkNotebook page that contains this page
			 */
			notebook = GTK_NOTEBOOK(cfg_parent_page->notebook);
			if(notebook != NULL)
				gtk_notebook_set_page(
					notebook,
					cfg_page->page_num
				);

			/* Go on to the grand parent by setting this page to
			 * it's parent page and its parent page to its parent
			 * page
			 */
			cfg_page = cfg_parent_page;
			cfg_parent_page = cfg_page->parent_page;
		}
		/* No more parent pages, so cfg_page is now a toplevel
		 * page, set the toplevel GtkNotebook page that contains
		 * this toplevel page
		 */
		notebook = GTK_NOTEBOOK(cfg_win->toplevel_notebook);
		if(notebook != NULL)
			gtk_notebook_set_page(
				notebook,
				cfg_page->page_num
			);
		break;

	  case CFG_WIN_STYLE_TREE:
		ctree = GTK_CTREE(cfg_win->categories_ctree);
		if(ctree != NULL)
		{
			GtkCList *clist = GTK_CLIST(ctree);
			GtkCTreeNode	*parent_node,
							*node = cfg_page->ctree_node;

			gtk_clist_freeze(clist);

			/* Start from the specified page's parent and expand
			 * all parent GtkCTreeNodes
			 */
			cfg_parent_page = cfg_page->parent_page;
			while(cfg_parent_page != NULL)
			{
				parent_node = cfg_parent_page->ctree_node;
				if(parent_node != NULL)
					gtk_ctree_expand(
						ctree,
						parent_node
					);

				/* Go on to the grand parent by setting this page
				 * to it's parent page and its parent page to its
				 * parent page
				 */
				cfg_page = cfg_parent_page;
				cfg_parent_page = cfg_parent_page->parent_page;
			}

			/* Select the GtkCTree node representing the specified
			 * page
			 */
			if(node != NULL)
			{
				gtk_clist_unselect_all(clist);
				gtk_ctree_select(ctree, node);
			}

			gtk_clist_thaw(clist);
		}
		break;
	}
}


/*
 *	Gets the current page on the Configuration Window.
 *
 *	The w specifies the Configuration Window.
 *
 *	Returns the current page or NULL on error or if there is
 *	no current page.
 */
GtkWidget *CfgWinGetCurrentPage(GtkWidget *w)
{
	CfgWinPage *cfg_page;
	CfgWin *cfg_win = CfgWinGetWidgetData(
		w,
		"CfgWinGetCurrentPage"
	);
	if(cfg_win == NULL)
		return(NULL);

	cfg_page = cfg_win->current_page;
	if(cfg_page == NULL)
		return(NULL);

	return(cfg_page->toplevel);
}

/*
 *	Gets the default page on the Configuration Window.
 *
 *	The w specifies the Configuration Window.
 *
 *	Returns the default page or NULL on error or if there is no
 *	default page.
 */
GtkWidget *CfgWinGetDefaultPage(GtkWidget *w)
{
	CfgWin *cfg_win = CfgWinGetWidgetData(
		w,
		"CfgWinGetDefaultPage"
	);
	if(cfg_win == NULL)
		return(NULL);

	/* Get the default page */
	if(cfg_win->toplevel_pages_list != NULL)
	{
		GList *glist = cfg_win->toplevel_pages_list;
		CfgWinPage *cfg_page = CFG_WIN_PAGE(glist->data);
		if(cfg_page == NULL)
			return(NULL);

		return(cfg_page->toplevel);
	}
	else
	{
		return(NULL);
	}
}

/*
 *	Called by CfgWinGetPageByLabel().
 */
static GtkWidget *CfgWinGetPageByLabelIterate(
	CfgWin *cfg_win,
	GList *child_pages_list,
	const gchar *label,
	const gboolean recursive
)
{
	GList *glist;
	CfgWinPage *cfg_page;

	/* Search through the child pages list */
	for(glist = child_pages_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		cfg_page = CFG_WIN_PAGE(glist->data);
		if(cfg_page == NULL)
			continue;

		/* Does this page's label match the specified label? */
		if(cfg_page->label != NULL)
		{
			if(cfg_page->label == label)
				return(cfg_page->toplevel);
			else if(!g_strcasecmp(cfg_page->label, label))
				return(cfg_page->toplevel);
		}

		/* Does this page have child pages and should we recurse
		 * and search through them?
		 */
		if((cfg_page->child_pages_list != NULL) && recursive)
		{
			/* Search through this page's child pages */
			GtkWidget *w = CfgWinGetPageByLabelIterate(
				cfg_win,
				cfg_page->child_pages_list,
				label,
				recursive
			);
			if(w != NULL)
				return(w);
		}
	}

	return(NULL);
}

/*
 *	Gets the page by label.
 *
 *	The w specifies the Configuration Window.
 *
 *	The parent_page specifies the parent page to start the search
 *	from (including the parent page itself), if parent_page is
 *	NULL then the toplevel pages will be searched.
 *
 *	The label specifies the label.
 *
 *	If recursive is TRUE then child pages of each of the parent
 *	page's child pages will be searched recursively, otherwise
 *	only the parent page's child pages will be searched.
 *
 *	Returns the page or NULL on error.
 */
GtkWidget *CfgWinGetPageByLabel(
	GtkWidget *w,
	GtkWidget *parent_page,
	const gchar *label,
	const gboolean recursive
)
{
	GList *child_pages_list;
	CfgWinPage *cfg_parent_page;
	CfgWin *cfg_win = CfgWinGetWidgetData(
		w,
		"CfgWinGetPageByLabel"
	);
	if((cfg_win == NULL) || STRISEMPTY(label))
		return(NULL);

	if(parent_page != NULL)
	{
		cfg_parent_page = CfgWinPageGetWidgetData(
			parent_page,
			"CfgWinGetPageByLabel"
		);
		if(cfg_parent_page == NULL)
			return(NULL);

		/* Check if the parent page itself matches */
		if(cfg_parent_page->label != NULL)
		{
			if(cfg_parent_page->label == label)
				return(cfg_parent_page->toplevel);
			else if(!g_strcasecmp(cfg_parent_page->label, label))
				return(cfg_parent_page->toplevel);
		}
	}
	else
	{
		cfg_parent_page = NULL;
	}

	/* Get the child pages list */
	if(cfg_parent_page != NULL)
		child_pages_list = cfg_parent_page->child_pages_list;
	else
		child_pages_list = cfg_win->toplevel_pages_list;

	return(CfgWinGetPageByLabelIterate(
		cfg_win,
		child_pages_list,
		label,
		recursive
	));
}


/*
 *	Called by CfgWinGetValues().
 */
static void CfgWinGetValuesIterate(
	CfgWin *cfg_win,
	CfgWinPage *page
)
{
	GList *glist;
	CfgWinLink *cfg_link;
	CfgWinPage *child_page;

	/* Set all the CfgWinLink on this page */
	for(glist = page->links_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		cfg_link = CFG_WIN_LINK(glist->data);
		if(cfg_link == NULL)
			continue;

		/* Get this CfgWinLink's value from its CfgItem */
		CfgWinSetCfgWinLinkValueFromCfgItemValue(
			cfg_win,
			cfg_link
		);
	}

	/* Recurse into all child pages */
	for(glist = page->child_pages_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		child_page = CFG_WIN_PAGE(glist->data);
		if(child_page == NULL)
			continue;

		CfgWinGetValuesIterate(
			cfg_win,
			child_page
		);
	}
}

/*
 *	Gets the values from the CfgItems in the CfgWin's CfgList
 *	to the widgets referenced by CfgWin's CfgWinLinks.
 *
 *	The w specifies the CfgWin.
 *
 *	If show_progress is TRUE then the progress dialog will be
 *	displayed during this call.
 */
void CfgWinGetValues(
	GtkWidget *w,
	const gboolean show_progress
)
{
	GList *glist;
	CfgWinPage *page;
	CfgWin *cfg_win = CfgWinGetWidgetData(
		w,
		"CfgWinGetValues"
	);
	if(cfg_win == NULL)
		return;

	/* Toplevel pages */
	for(glist = cfg_win->toplevel_pages_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		page = CFG_WIN_PAGE(glist->data);
		if(page == NULL)
			continue;

		CfgWinGetValuesIterate(
			cfg_win,
			page
		);
	}
}


/*
 *	Called by CfgWinSetValues().
 */
static void CfgWinSetValuesIterate(
	CfgWin *cfg_win,
	CfgWinPage *page,
	const gboolean show_progress,
	const gboolean save_to_disk,
	gboolean *user_aborted,
	gint *ncfg_widgets_set,
	const gint ncfg_widgets
)
{
	GList *glist;
	CfgWinLink *cfg_link;
	CfgWinPage *child_page;

	/* Set all the CfgWinLinks on this page */
	for(glist = page->links_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		cfg_link = CFG_WIN_LINK(glist->data);
		if(cfg_link == NULL)
			continue;

		/* Set this CfgWinLink's CfgItem value */
		CfgWinSetCfgItemValueFromCfgWinLinkValue(
			cfg_win,
			cfg_link
		);

		*ncfg_widgets_set = (*ncfg_widgets_set) + 1;

		if(show_progress)
		{
			gfloat v = (gfloat)(*ncfg_widgets_set) / (gfloat)ncfg_widgets;
			if(save_to_disk)
				v = (0.0f / 2.0f) + (v / 2.0f);

			ProgressDialogUpdate(
				NULL, NULL, NULL, NULL,
				v,
				CFG_WIN_PROGRESS_BAR_TICKS,
				TRUE
			);
			if(ProgressDialogStopCount() > 0)
			{
				*user_aborted = TRUE;
				break;
			}
		}
	}
	if(*user_aborted)
		return;

	/* Recurse into all the child pages */
	for(glist = page->child_pages_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		child_page = CFG_WIN_PAGE(glist->data);
		if(child_page == NULL)
			continue;

		CfgWinSetValuesIterate(
			cfg_win,
			child_page,
			show_progress,
			save_to_disk,
			user_aborted,
			ncfg_widgets_set,
			ncfg_widgets
		);
		if(*user_aborted)
			break;
	}
}

/*
 *	Sets the values from the widgets referenced by the CfgWin's
 *	CfgWinLinks to the CfgItems in the CfgWin's CfgList.
 *
 *	The w specifies the CfgWin.
 *
 *	If show_progress is TRUE then the progress dialog will be
 *	displayed during this call.
 *
 *	If call_apply_cb is TRUE then the apply callback will be
 *	called after the values have been set.
 *
 *	If save_to_disk is TRUE then the configuration will be
 *	saved to disk.
 */
void CfgWinSetValues(
	GtkWidget *w,
	const gboolean show_progress,
	const gboolean call_apply_cb,
	const gboolean save_to_disk
) 
{
	gboolean user_aborted;
	gint		ncfg_widgets_set,
			ncfg_widgets;
	GList *glist;
	GtkWidget *toplevel;
	CfgList *cfg_list;
	CfgWinPage *page;
	CfgWin *cfg_win = CfgWinGetWidgetData(
		w,
		"CfgWinSetValues"
	);
	if(cfg_win == NULL)
		return;

	toplevel = cfg_win->toplevel;
	cfg_list = cfg_win->cfg_list;

	/* Map the progress dialog? */
	if(show_progress)
	{
		ProgressDialogSetTransientFor(toplevel);
		ProgressDialogMap(
			"Applying Values",
			"Applying Values...",
			(const guint8 **)icon_select_32x32_xpm,
			"Stop"
		);
	}

	user_aborted = FALSE;
	ncfg_widgets_set = 0;
	ncfg_widgets = CfgWinGetTotalCfgWinLinks(toplevel);

	/* Iterate through all the toplevel pages and recurse into
	 * each child page
	 */
	for(glist = cfg_win->toplevel_pages_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		page = CFG_WIN_PAGE(glist->data);
		if(page == NULL)
			continue;

		CfgWinSetValuesIterate(
			cfg_win,
			page,
			show_progress,
			save_to_disk,
			&user_aborted,
			&ncfg_widgets_set,
			ncfg_widgets
		);
		if(user_aborted)
			break;

		if(show_progress)
		{
			gfloat v = (gfloat)(ncfg_widgets_set) / (gfloat)ncfg_widgets;
			if(save_to_disk)
				v = (0.0f / 2.0f) + (v / 2.0f);

			ProgressDialogUpdate(
				NULL, NULL, NULL, NULL,
				v,
				CFG_WIN_PROGRESS_BAR_TICKS,
				TRUE
			);
			if(ProgressDialogStopCount() > 0)
			{
				user_aborted = TRUE;
				break;
			}
		}
	}

	/* Save the configuration to disk? */
	if(save_to_disk && !user_aborted &&
	   !STRISEMPTY(cfg_win->cfg_path)
	)
	{
		const gchar *path = cfg_win->cfg_path;
		if(show_progress)
		{
			CFGFileSaveProgress(
				path,
				cfg_list,
				CfgWinSetValuesSaveCB, cfg_win
			);
		}
		else
		{
			CFGFileSave(path, cfg_list);
		}
	}

	/* Unmap the progress dialog */
	if(show_progress)
	{
		ProgressDialogBreakQuery(TRUE);
		ProgressDialogSetTransientFor(NULL);
	}

	/* Notify about the apply? */
	if(call_apply_cb && (cfg_win->apply_cb != NULL))
	{
		cfg_win->apply_cb(
			toplevel,
			cfg_list,
			cfg_win->apply_data
		);
	}
}


/*
 *	Gets the CfgWin data from the GtkWidget.
 */
static CfgWin *CfgWinGetWidgetData(
	GtkWidget *w,
	const gchar *func_name
)
{
	const gchar *key = CFG_WIN_KEY;
	CfgWin *cfg_win;

	if(w == NULL)
		return(NULL);

	cfg_win = CFG_WIN(gtk_object_get_data(
		GTK_OBJECT(w),
		key
	));
	if(cfg_win == NULL)
	{
		g_printerr(
"%s(): Warning: GtkWidget %p:\
 Unable to find the data that matches the key \"%s\".\n",
			func_name,
			w,
			key
		);
		return(NULL);
	}

	return(cfg_win);
}

/*
 *	Creates a new CfgWin.
 */
GtkWidget *CfgWinNew(
	const CfgWinStyle cfg_win_style,
	const gchar *title,
	guint8 **icon_data,
	const CfgWinButtons buttons,
	const gint width, const gint height,
	CfgList *cfg_list,
	const gchar *cfg_path,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry
)
{
	const gint border_major = 5;
	gchar **strv;
	GtkAccelGroup *accelgrp;
	GtkWidget	*w,
					*parent, *parent2, *parent3,
					*toplevel;
	GtkCList *clist;
	GtkCTree *ctree;
	CfgWin *cfg_win = CFG_WIN(g_malloc0(sizeof(CfgWin)));
	if(cfg_win == NULL)
		return(NULL);

	cfg_win->toplevel = toplevel = gtk_window_new(GTK_WINDOW_DIALOG);
	cfg_win->accelgrp = accelgrp = gtk_accel_group_new();
/*	cfg_win->freeze_count = 0; */
	cfg_win->ref_count = 1;
/*
	cfg_win->busy_count = 0;
	cfg_win->flags = 0;
 */
	cfg_win->style = cfg_win_style;
	cfg_win->title = STRDUP(title);
	cfg_win->icon_data = icon_data;
	cfg_win->buttons = buttons;
	cfg_win->busy_cur = gdk_cursor_new(GDK_WATCH);
	cfg_win->cfg_list = cfg_list;
	cfg_win->cfg_path = STRDUP(cfg_path);
/*
	cfg_win->toplevel_pages_list = NULL;
	cfg_win->current_page = NULL;
 */
	cfg_win->freeze_count++;

	/* Toplevel GtkWindow */
	w = toplevel;
	gtk_object_set_data_full(
		GTK_OBJECT(w), CFG_WIN_KEY,
		cfg_win, CfgWinDestroyCB
	);
#if defined(EDV_OPTIONS_WM_CLASS_WINDOW_NAME)
	gtk_window_set_wmclass(
		GTK_WINDOW(w),
		EDV_OPTIONS_WM_CLASS_WINDOW_NAME,
		PROG_NAME
	);
#elif defined(PROG_NAME)
	gtk_window_set_wmclass(
		GTK_WINDOW(w),
		"CfgWin",
		PROG_NAME
	);
#endif

	gtk_window_set_policy(GTK_WINDOW(w), FALSE, FALSE, FALSE);
	gtk_window_set_title(GTK_WINDOW(w), title);
       if((geometry_flags != 0) && (geometry != NULL))
	{
		if((geometry_flags & GDK_GEOMETRY_X) || (geometry_flags & GDK_GEOMETRY_Y))
			gtk_widget_set_uposition(w, geometry->x, geometry->y);
	}
	gtk_widget_set_usize(w, CFG_WIN_WIDTH, CFG_WIN_HEIGHT);
	gtk_signal_connect(
		GTK_OBJECT(w), "realize",
		GTK_SIGNAL_FUNC(CfgWinRealizeCB), cfg_win
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "delete_event",
		GTK_SIGNAL_FUNC(CfgWinDeleteEventCB), cfg_win
	);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	parent = w;

	/* Main GtkVBox */
	cfg_win->main_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;

	/* Create the client toplevel widgets based on the style */
	switch(cfg_win->style)
	{
	  case CFG_WIN_STYLE_NOTEBOOK:
		w = gtk_vbox_new(FALSE, 0);
		gtk_container_border_width(GTK_CONTAINER(w), border_major);
		gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
		gtk_widget_show(w);
		parent2 = w;

		cfg_win->toplevel_notebook = w = gtk_notebook_new();
		gtk_notebook_set_tab_pos(GTK_NOTEBOOK(w), GTK_POS_TOP);
		gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
		gtk_notebook_set_scrollable(GTK_NOTEBOOK(w), TRUE);
		gtk_notebook_set_show_tabs(GTK_NOTEBOOK(w), TRUE);
		gtk_notebook_set_show_border(GTK_NOTEBOOK(w), TRUE);
		gtk_signal_connect(
			GTK_OBJECT(w), "switch_page",
			GTK_SIGNAL_FUNC(CfgWinSwitchPageCB), cfg_win
		);
		gtk_widget_show(w);
		break;

	  case CFG_WIN_STYLE_TREE:
		w = gtk_hpaned_new();
		if(width > 0)
			gtk_paned_set_position(
				GTK_PANED(w),
				(gint)(width * 0.3f)
			);
		gtk_paned_set_handle_size(GTK_PANED(w), 10);
		gtk_paned_set_gutter_size(GTK_PANED(w), 10);
		gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
		gtk_widget_show(w);
		parent2 = w;

		/* GtkScrolledWindow for the GtkCTree */
		w = gtk_scrolled_window_new(NULL, NULL);
		gtk_scrolled_window_set_policy(
			GTK_SCROLLED_WINDOW(w),
			GTK_POLICY_AUTOMATIC,
			GTK_POLICY_AUTOMATIC
		);
		gtk_paned_add1(GTK_PANED(parent2), w);
		gtk_widget_show(w);
		parent3 = w;

		/* GtkCTree */
		strv = (gchar **)g_malloc(1 * sizeof(gchar));
		strv[0] = "Categories";
		cfg_win->categories_ctree = w = gtk_ctree_new_with_titles(
			1,				/* Columns */
			0,				/* Tree Column */
			strv				/* Titles */
		);
		g_free(strv);
		clist = GTK_CLIST(w);
		ctree = GTK_CTREE(w);
		gtk_signal_connect(
			GTK_OBJECT(w), "tree_select_row",
			GTK_SIGNAL_FUNC(CfgWinTreeSelectRowCB), cfg_win
		);
		gtk_signal_connect(
			GTK_OBJECT(w), "tree_unselect_row",
			GTK_SIGNAL_FUNC(CfgWinTreeUnselectRowCB), cfg_win
		);
		gtk_container_add(GTK_CONTAINER(parent3), w);
		gtk_clist_column_titles_passive(clist);
		gtk_clist_set_selection_mode(clist, GTK_SELECTION_BROWSE);
		gtk_ctree_set_line_style(ctree, GTK_CTREE_LINES_DOTTED);
		gtk_clist_set_column_auto_resize(
			clist,
			0,
			TRUE
		);
		gtk_clist_set_column_justification(
			clist,
			0,
			GTK_JUSTIFY_LEFT
		);
		gtk_clist_set_row_height(clist, 20);
		gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
		gtk_widget_show(w);

		/* Pages Toplevel GtkBox */
		cfg_win->pages_toplevel = w = gtk_vbox_new(FALSE, 0);
		gtk_container_border_width(GTK_CONTAINER(w), border_major);
		gtk_paned_add2(GTK_PANED(parent2), w);
		gtk_widget_show(w);
		break;
	}

	/* Horizontal GtkSeparator */
	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Buttons GtkHBox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* OK GtkButton */
	cfg_win->ok_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_ok_20x20_xpm,
		"OK",
		NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(CfgWinOKCB), cfg_win
	);
	gtk_accel_group_add(
		accelgrp,
		GDK_o, GDK_CONTROL_MASK,
		GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w),
		"clicked"
	);
	GUIButtonLabelUnderline(w, GDK_o);

	/* Apply GtkButton */
	cfg_win->apply_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_select_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Aplique"
#elif defined(PROG_LANGUAGE_FRENCH)
"Appliquer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Verwenden"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Applicare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Toepas"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Aplique"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Gjeld"
#else
"Apply"
#endif
		, NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(CfgWinApplyCB), cfg_win
	);
	gtk_accel_group_add(
		accelgrp,
		GDK_a, GDK_CONTROL_MASK,
		GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w),
		"clicked"
	);
	GUIButtonLabelUnderline(w, GDK_a);

	/* Save GtkButton */
	cfg_win->save_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_save_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Salve"
#elif defined(PROG_LANGUAGE_FRENCH)
"Enregistrer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Auer"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Risparmiare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Red"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Poupe"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Untatt"
#else
"Save"
#endif
		, NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(CfgWinSaveCB), cfg_win
	);
	gtk_accel_group_add(
		accelgrp,
		GDK_s, GDK_CONTROL_MASK,
		GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w),
		"clicked"
	);
	GUIButtonLabelUnderline(w, GDK_s);

	/* Cancel GtkButton */
	cfg_win->cancel_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_cancel_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Cancele"
#elif defined(PROG_LANGUAGE_FRENCH)
"Annuler"
#elif defined(PROG_LANGUAGE_GERMAN)
"Heben"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Annullare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Annuleer"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Cancelamento"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kanseller"
#else
"Cancel"
#endif
		, NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(CfgWinCancelCB), cfg_win
	);
	gtk_accel_group_add(
		accelgrp,
		GDK_Escape, 0,
		GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w),
		"clicked"
	);
	gtk_accel_group_add(
		accelgrp,
		GDK_c, GDK_CONTROL_MASK,
		GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w),
		"clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);

	/* Close GtkButton */
	cfg_win->close_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_close_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Cierre"
#elif defined(PROG_LANGUAGE_FRENCH)
"Quitter"
#elif defined(PROG_LANGUAGE_GERMAN)
"Nah"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Vicino"
#elif defined(PROG_LANGUAGE_DUTCH)
"Einde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Prximo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Nr"
#else
"Close"
#endif
		, NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(CfgWinCancelCB), cfg_win
	);
	gtk_accel_group_add(
		accelgrp,
		GDK_Escape, 0,
		GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w),
		"clicked"
	);
	gtk_accel_group_add(
		accelgrp,
		GDK_c, GDK_CONTROL_MASK,
		GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w),
		"clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);


	cfg_win->freeze_count--;

	return(cfg_win->toplevel);
}

/*
 *	Gets the OK GtkButton.
 */
GtkWidget *CfgWinGetOKButton(GtkWidget *w)
{
	CfgWin *cfg_win = CfgWinGetWidgetData(
		w,
		"CfgWinGetOKButton"
	);
	if(cfg_win == NULL)
		return(NULL);

	return(cfg_win->ok_btn);
}

/*
 *	Gets the Apply GtkButton.
 */
GtkWidget *CfgWinGetApplyButton(GtkWidget *w)
{
	CfgWin *cfg_win = CfgWinGetWidgetData(
		w,
		"CfgWinGetApplyButton"
	);
	if(cfg_win == NULL)
		return(NULL);

	return(cfg_win->apply_btn);
}

/*
 *	Gets the Save GtkButton.
 */
GtkWidget *CfgWinGetSaveButton(GtkWidget *w)
{
	CfgWin *cfg_win = CfgWinGetWidgetData(
		w,
		"CfgWinGetSaveButton"
	);
	if(cfg_win == NULL)
		return(NULL);

	return(cfg_win->save_btn);
}

/*
 *	Gets the Cencel GtkButton.
 */
GtkWidget *CfgWinGetCancelButton(GtkWidget *w)
{
	CfgWin *cfg_win = CfgWinGetWidgetData(
		w,
		"CfgWinGetCancelButton"
	);
	if(cfg_win == NULL)
		return(NULL);

	return(cfg_win->cancel_btn);
}

/*
 *	Gets the Close GtkButton.
 */
GtkWidget *CfgWinGetCloseButton(GtkWidget *w)
{
	CfgWin *cfg_win = CfgWinGetWidgetData(
		w,
		"CfgWinGetCloseButton"
	);
	if(cfg_win == NULL)
		return(NULL);

	return(cfg_win->close_btn);
}

/*
 *	Sets the page changed callback.
 */
void CfgWinSetPageChangedCB(
	GtkWidget *w,
	void (*cb)(
			GtkWidget *,
			GtkWidget *,
			gpointer
	),
	gpointer data
)
{
	CfgWin *cfg_win = CfgWinGetWidgetData(
		w,
		"CfgWinSetPageChangedCB"
	);
	if(cfg_win == NULL)
		return;

	cfg_win->page_changed_cb = cb;
	cfg_win->page_changed_data = data;
}

/*
 *	Sets the apply callback.
 */
void CfgWinSetApplyCB(
	GtkWidget *w,
	void (*cb)(
			GtkWidget *, 
			CfgList *, 
			gpointer
	),
	gpointer data
)
{
	CfgWin *cfg_win = CfgWinGetWidgetData(
		w,
		"CfgWinSetApplyCB"
	);
	if(cfg_win == NULL)
		return;

	cfg_win->apply_cb = cb;
	cfg_win->apply_data = data;
}

/*
 *	Gets the Configuration Window's has changes flag.
 */
gboolean CfgWinGetHasChanges(GtkWidget *w)
{
	CfgWin *cfg_win = CfgWinGetWidgetData(
		w,
		"CfgWinGetHasChanges"
	);
	if(cfg_win == NULL)
		return(FALSE);

	return((cfg_win->flags & CFG_WIN_HAS_CHANGES) ? TRUE : FALSE);
}

/*
 *	Called by CfgWinSetHasChanges().
 */
static void CfgWinSetHasChangesIterate(
	CfgWin *cfg_win,
	CfgWinPage *page,
	const gboolean has_changes
)
{
	GList *glist;
	CfgWinLink *cfg_link;
	CfgWinPage *child_page;

	/* Set each CfgWinLink's has changes flag */
	for(glist = page->links_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		cfg_link = CFG_WIN_LINK(glist->data);
		if(cfg_link == NULL)
			continue;

		if(has_changes)
			cfg_link->flags |= CFG_WIN_LINK_WIDGET_HAS_CHANGES;
		else
			cfg_link->flags &= ~CFG_WIN_LINK_WIDGET_HAS_CHANGES;
	}

	/* Recurse into each child page */
	for(glist = page->child_pages_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		child_page = CFG_WIN_PAGE(glist->data);
		if(child_page == NULL)
			continue;

		CfgWinSetHasChangesIterate(
			cfg_win,
			child_page,
			has_changes
		);
	}
}

/*
 *	Sets the Configuration Window's has changes flag.
 */
void CfgWinSetHasChanges(
	GtkWidget *w,
	const gboolean has_changes
)
{
	GList *glist;
	CfgWinPage *page;
	CfgWin *cfg_win = CfgWinGetWidgetData(
		w,
		"CfgWinSetHasChanges"
	);
	if(cfg_win == NULL)
		return;

	/* Set each CfgWinLink's has changes flag */
	for(glist = cfg_win->toplevel_pages_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		page = CFG_WIN_PAGE(glist->data);
		if(page == NULL)
			continue;

		CfgWinSetHasChangesIterate(
			cfg_win,
			page,
			has_changes
		);
	}

	/* Set the Configuration Window's has changes flag */
	if(has_changes && !(cfg_win->flags & CFG_WIN_HAS_CHANGES))
	{
		cfg_win->flags |= CFG_WIN_HAS_CHANGES;
		CfgWinUpdateMenus(w);
	}
	else if(!has_changes && (cfg_win->flags & CFG_WIN_HAS_CHANGES))
	{
		cfg_win->flags &= ~CFG_WIN_HAS_CHANGES;
		CfgWinUpdateMenus(w);
	}
}

/*
 *	Updates the Configuration Window's GtkWidgets to reflect
 *	current values.
 *
 *	This does not update or modify any of the CfgWinLinks, use
 *	CfgWinGetValues() instead.
 */
void CfgWinUpdateMenus(GtkWidget *w)
{
	gboolean	sensitive,
					has_changes;
	gchar *s;
	CfgWinButtons buttons;
	CfgWin *cfg_win = CfgWinGetWidgetData(
		w,
		"CfgWinUpdateMenus"
	);
	if(cfg_win == NULL)
		return;

	cfg_win->freeze_count++;

	has_changes = (cfg_win->flags & CFG_WIN_HAS_CHANGES) ? TRUE : FALSE;
	buttons = cfg_win->buttons;

	s = g_strconcat(
		(cfg_win->title != NULL) ? cfg_win->title : "",
		(has_changes) ? " (*)" : "",
		NULL
	);
	gtk_window_set_title(GTK_WINDOW(cfg_win->toplevel), s);
	g_free(s);

	/* OK, Apply, Cancel, Close Buttons */
	sensitive = has_changes ? TRUE : FALSE;
	gtk_widget_set_sensitive(cfg_win->ok_btn, sensitive);
	gtk_widget_set_sensitive(cfg_win->apply_btn, sensitive);
	gtk_widget_set_sensitive(cfg_win->cancel_btn, sensitive);
	sensitive = has_changes ? FALSE : TRUE;
	gtk_widget_set_sensitive(cfg_win->close_btn, sensitive);
	if(has_changes)
	{
		if(buttons & CFG_WIN_BUTTON_OK)
			gtk_widget_show(cfg_win->ok_btn);
		else
			gtk_widget_hide(cfg_win->ok_btn);

		if(buttons & CFG_WIN_BUTTON_APPLY)
			gtk_widget_show(cfg_win->apply_btn);
		else
			gtk_widget_hide(cfg_win->apply_btn);

		if(buttons & CFG_WIN_BUTTON_SAVE)
			gtk_widget_show(cfg_win->save_btn);
		else
			gtk_widget_hide(cfg_win->save_btn);

		if(buttons & CFG_WIN_BUTTON_CANCEL)
			gtk_widget_show(cfg_win->cancel_btn);
		else
			gtk_widget_hide(cfg_win->cancel_btn);

		gtk_widget_hide(cfg_win->close_btn);
	}
	else
	{
		if(buttons & CFG_WIN_BUTTON_OK)
			gtk_widget_show(cfg_win->ok_btn);
		else
			gtk_widget_hide(cfg_win->ok_btn);

		if(buttons & CFG_WIN_BUTTON_APPLY)
			gtk_widget_show(cfg_win->apply_btn);
		else
			gtk_widget_hide(cfg_win->apply_btn);

		if(buttons & CFG_WIN_BUTTON_SAVE)
			gtk_widget_show(cfg_win->save_btn);
		else
			gtk_widget_hide(cfg_win->save_btn);

		gtk_widget_hide(cfg_win->cancel_btn);

		if(buttons & CFG_WIN_BUTTON_CLOSE)
			gtk_widget_show(cfg_win->close_btn);
		else
			gtk_widget_hide(cfg_win->close_btn);
	}

	cfg_win->freeze_count--;
}

/*
 *	Sets the Configuration Window as busy or ready.
 */
void CfgWinSetBusy(
	GtkWidget *w,
	const gboolean busy
)
{
	GdkCursor *cur = NULL;
	CfgWin *cfg_win = CfgWinGetWidgetData(
		w,
		"CfgWinSetBusy"
	);
	if(cfg_win == NULL)
		return;

	w = cfg_win->toplevel;

	if(busy)
	{
		/* Increase busy count */
		cfg_win->busy_count++;

		/* If already busy then don't change anything */
		if(cfg_win->busy_count > 1)
			return;

		cur = cfg_win->busy_cur;
	}
	else
	{
		/* Reduce busy count */
		cfg_win->busy_count--;
		if(cfg_win->busy_count < 0)
			cfg_win->busy_count = 0;

		/* If still busy do not change anything */
		if(cfg_win->busy_count > 0)
			return;

		cur = NULL;
	}
	if(w->window != NULL)
	{
		gdk_window_set_cursor(w->window, cur);
		gdk_flush();
	}
}

/*
 *	Adds a reference count to the Configuration Window.
 *
 *	Returns the Configuration Window or NULL on error.
 */
GtkWidget *CfgWinRef(GtkWidget *w)
{
	CfgWin *cfg_win = CfgWinGetWidgetData(
		w,
		"CfgWinRef"
	);
	if(cfg_win == NULL)
		return(NULL);

	cfg_win->ref_count++;

	return(w);
}

/*
 *	Removes a reference count from the Configuration Window.
 *
 *	If the reference counts reach 0 then the Configuration Window
 *	will be destroyed.
 *
 *	Always returns NULL.
 */
GtkWidget *CfgWinUnref(GtkWidget *w)
{
	CfgWin *cfg_win = CfgWinGetWidgetData(
		w,
		"CfgWinUnref"
	);
	if(cfg_win == NULL)
		return(NULL);

	cfg_win->ref_count--;

	if(cfg_win->ref_count <= 0)
	{
		cfg_win->freeze_count++;

		/* Delete all the pages and CfgWinLinks */
		if(cfg_win->toplevel_pages_list != NULL)
		{
			GList *glist;
			CfgWinPage *cfg_page;

			while(cfg_win->toplevel_pages_list != NULL)
			{
				glist = cfg_win->toplevel_pages_list;
				cfg_page = CFG_WIN_PAGE(glist->data);
				if(cfg_page != NULL)
					CfgWinRemovePageIterate(
						cfg_win,
						cfg_page
					);
				else
					cfg_win->toplevel_pages_list = g_list_remove(
						cfg_win->toplevel_pages_list,
						NULL
					);
			}
		}

		cfg_win->freeze_count--;

		/* Delete the Configuration Window */
		gtk_widget_destroy(w);
	}

	return(NULL);
}
