#include <stdlib.h>
#include <gtk/gtk.h>

#include "cfg.h"
#include "cfg_gtk.h"
#include "guiutils.h"


/* GtkCList */
void CfgGtkSetGtkCListColumnWidths(
	const CfgList *list,
	const gchar *parm,
	GtkWidget *w
);
void CfgGtkGetGtkCListColumnWidths(
	CfgList *list,
	const gchar *parm,
	GtkWidget *w
);

/* GtkEditable */
void CfgGtkSetGtkEditable(
	const CfgList *list,
	const gchar *parm,
	GtkWidget *w
);
void CfgGtkGetGtkEditable(
	CfgList *list,
	const gchar *parm,
	GtkWidget *w
);

/* GtkPaned */
void CfgGtkSetGtkPaned(
	const CfgList *list,
	const gchar *parm,
	GtkWidget *w
);
void CfgGtkGetGtkPaned(
	CfgList *list,
	const gchar *parm,
	GtkWidget *w
);

/* GtkSpinButton */
void CfgGtkSetGtkSpinButton(
	const CfgList *list,
	const gchar *parm,
	GtkWidget *w
);
void CfgGtkGetGtkSpinButton(
	CfgList *list,
	const gchar *parm,
	GtkWidget *w
);

/* GtkToggleButton */
void CfgGtkSetGtkToggleButton(
	const CfgList *list,
	const gchar *parm,
	GtkWidget *w
);
void CfgGtkGetGtkToggleButton(
	CfgList *list,
	const gchar *parm,
	GtkWidget *w
);

/* GtkWindow */
void CfgGtkSetGtkWindow(
	const CfgList *list,
	const gchar *parm_x,
	const gchar *parm_y,
	const gchar *parm_width,
	const gchar *parm_height,
	GtkWidget *w
);
void CfgGtkGetGtkWindow(
	CfgList *list,
	const gchar *parm_x,
	const gchar *parm_y,
	const gchar *parm_width,
	const gchar *parm_height,
	GtkWidget *w
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Sets the GtkCList's column widths from the configuration
 *	value.
 *
 *	The list specifies the configuration list.
 *
 *	The parm specifies the column widths configuration parameter
 *	name. The parameter must refer to a configuration item of type
 *	CFG_ITEM_TYPE_INT_LIST.
 *
 *	The w specifies the GtkCList.
 */
void CfgGtkSetGtkCListColumnWidths(
	const CfgList *list,
	const gchar *parm,
	GtkWidget *w
)
{
	gint i, ncolumns;
	GList *glist;
	GtkCList *clist;
	CfgIntList *intlist;

	if(w == NULL)
	    return;

	clist = GTK_CLIST(w);
	if(clist == NULL)
	    return;

	ncolumns = clist->columns;

	intlist = CFGItemListGetValueIntList(list, parm);
	if(intlist == NULL)
	    return;

	/* Set the column widths */
	for(i = 0, glist = intlist->list;
	    (i < ncolumns) && (glist != NULL);
	    i++, glist = g_list_next(glist)
	)
	    gtk_clist_set_column_width(
		clist,
		i,
		(gint)glist->data
	    );
}

/*
 *	Sets the configuration value from the GtkCList's column widths.
 *
 *	The list specifies the configuration list.
 *
 *	The parm specifies the column widths configuration parameter
 *	name. The parameter must refer to a configuration item of type
 *	CFG_ITEM_TYPE_INT_LIST.
 *
 *	The w specifies the GtkPaned.
 */
void CfgGtkGetGtkCListColumnWidths(
	CfgList *list,
	const gchar *parm,
	GtkWidget *w
)
{
	gint i, ncolumns;
	GtkCList *clist;
	GtkCListColumn *column;
	CfgIntList *intlist;

	if(w == NULL)
	    return;

	clist = GTK_CLIST(w);
	if(clist == NULL)
	    return;

	ncolumns = clist->columns;

	intlist = CFGItemListGetValueIntList(list, parm);
	if(intlist == NULL)
	    return;

	/* Clear the existing intlist */
	if(intlist->list != NULL)
	{
	    g_list_free(intlist->list);
	    intlist->list = NULL;
	}

	/* Get the column widths */
	for(i = 0; i < ncolumns; i++)
	{
	    column = &clist->column[i];
	    intlist->list = g_list_append(
		intlist->list,
		(gpointer)MAX(column->width, 0)
	    );
	}
}


/*
 *	Sets the GtkEditable's value from the configuration value.
 *
 *	The list specifies the configuration list.
 *
 *	The parm specifies the configuration parameter name.
 *
 *	The w specifies the GtkEditable, which can be a GtkCombo,
 *	GtkEntry, or GtkText.
 */
void CfgGtkSetGtkEditable(
	const CfgList *list,
	const gchar *parm,
	GtkWidget *w
)
{
	gint i;
	const CfgItem *item;

	if(w == NULL)
	    return;

	i = CFGItemListMatchParameter(list, parm);
	if(i < 0)
	    return;

	item = &list[i];

	/* If the widget is a GtkCombo then get its GtkEntry */
	if(GTK_IS_COMBO(w))
	{
	    GtkCombo *combo = GTK_COMBO(w);
	    w = combo->entry;
	    if(w == NULL)
		return;
	}

	/* Set the GtkEditable's value based on what specific type of
	 * GtkEditable it is
	 */
	if(GTK_IS_ENTRY(w))
	{
	    const CfgItemType type = item->type;
	    if(type == CFG_ITEM_TYPE_STRING)
	    {
		gtk_entry_set_text(
		    GTK_ENTRY(w),
		    (const gchar *)item->value
		);
	    }
	    else if(type == CFG_ITEM_TYPE_INT8)
	    {
		gchar *s = g_strdup_printf(
		    "%i",
		    (gint)*(gint8 *)item->value
		);
		gtk_entry_set_text(GTK_ENTRY(w), s);
		g_free(s);
	    }
	    else if(type == CFG_ITEM_TYPE_UINT8)
	    {
		gchar *s = g_strdup_printf(
		    "%u",
		    (guint)*(guint8 *)item->value
		);
		gtk_entry_set_text(GTK_ENTRY(w), s);
		g_free(s);
	    }
	    else if(type == CFG_ITEM_TYPE_INT16)
	    {
		gchar *s = g_strdup_printf(
		    "%i",
		    (gint)*(gint16 *)item->value
		);
		gtk_entry_set_text(GTK_ENTRY(w), s);
		g_free(s);
	    }
	    else if(type == CFG_ITEM_TYPE_UINT16)
	    {
		gchar *s = g_strdup_printf(
		    "%u",
		    (guint)*(guint16 *)item->value
		);
		gtk_entry_set_text(GTK_ENTRY(w), s);
		g_free(s);
	    }
	    else if(type == CFG_ITEM_TYPE_INT32)
	    {
		gchar *s = g_strdup_printf(
		    "%i",
		    (gint)*(gint32 *)item->value
		);
		gtk_entry_set_text(GTK_ENTRY(w), s);
		g_free(s);
	    }
	    else if(type == CFG_ITEM_TYPE_UINT32)
	    {
		gchar *s = g_strdup_printf(
		    "%u",
		    (guint)*(guint32 *)item->value
		);
		gtk_entry_set_text(GTK_ENTRY(w), s);
		g_free(s);
	    }
	    else if(type == CFG_ITEM_TYPE_INT64)
	    {
		gchar *s = g_strdup_printf(
		    "%ld",
		    (glong)*(gint64 *)item->value
		);
		gtk_entry_set_text(GTK_ENTRY(w), s);
		g_free(s);
	    }
	    else if(type == CFG_ITEM_TYPE_UINT64)
	    {
		gchar *s = g_strdup_printf(
		    "%ld",
		    (gulong)*(guint64 *)item->value
		);
		gtk_entry_set_text(GTK_ENTRY(w), s);
		g_free(s);
	    }
	}
	else if(GTK_IS_TEXT(w))
	{
	    const CfgItemType type = item->type;
	    if(type == CFG_ITEM_TYPE_STRING)
	    {
		gtk_text_insert(
		    GTK_TEXT(w),
		    NULL, NULL, NULL,
		    (const gchar *)item->value,
		    -1
		);
	    }
	}
}

/*
 *	Sets the configuration value from the GtkEditable's value.
 *
 *	The list specifies the configuration list.
 *
 *	The parm specifies the configuration parameter name.
 *
 *	The w specifies the GtkEditable, which can be a GtkCombo,
 *	GtkEntry, or GtkText.
 */
void CfgGtkGetGtkEditable(
	CfgList *list,
	const gchar *parm,
	GtkWidget *w
)
{
	gint i;
	CfgItem *item;

	if(w == NULL)
	    return;

	i = CFGItemListMatchParameter(list, parm);
	if(i < 0)
	    return;

	item = &list[i];

	/* If the widget is a GtkCombo then get its GtkEntry */
	if(GTK_IS_COMBO(w))
	{
	    GtkCombo *combo = GTK_COMBO(w);
	    w = combo->entry;
	    if(w == NULL)
		return;
	}

	/* Set the item's value based on the type of GtkEditable */
	if(GTK_IS_ENTRY(w))
	{
	    const CfgItemType type = item->type;

	    /* Delete the existing value */
	    CFGItemResetValue(item);

	    /* Set the new value based on the item's type */
	    if(type == CFG_ITEM_TYPE_STRING)
	    {
		item->value = STRDUP(gtk_entry_get_text(GTK_ENTRY(w)));
	    }
	    else if(type == CFG_ITEM_TYPE_INT8)
	    {
		const gint8 i8 = (gint8)ATOI(gtk_entry_get_text(GTK_ENTRY(w)));
		item->value = g_memdup(&i8, sizeof(i8));
	    }
	    else if(type == CFG_ITEM_TYPE_UINT8)
	    {
		const guint8 i8 = (guint8)ATOI(gtk_entry_get_text(GTK_ENTRY(w)));
		item->value = g_memdup(&i8, sizeof(i8));
	    }
	    else if(type == CFG_ITEM_TYPE_INT16)
	    {
		const gint16 i16 = (gint16)ATOI(gtk_entry_get_text(GTK_ENTRY(w)));
		item->value = g_memdup(&i16, sizeof(i16));
	    }
	    else if(type == CFG_ITEM_TYPE_UINT16)
	    {
		const guint16 i16 = (guint16)ATOI(gtk_entry_get_text(GTK_ENTRY(w)));
		item->value = g_memdup(&i16, sizeof(i16));
	    }
	    else if(type == CFG_ITEM_TYPE_INT32)
	    {
		const gint32 i32 = (gint32)ATOI(gtk_entry_get_text(GTK_ENTRY(w)));
		item->value = g_memdup(&i32, sizeof(i32));
	    }
	    else if(type == CFG_ITEM_TYPE_UINT32)
	    {
		const guint32 i32 = (guint32)ATOI(gtk_entry_get_text(GTK_ENTRY(w)));
		item->value = g_memdup(&i32, sizeof(i32));
	    }
	    else if(type == CFG_ITEM_TYPE_INT64)
	    {
		const gint64 i64 = (gint64)ATOL(gtk_entry_get_text(GTK_ENTRY(w)));
		item->value = g_memdup(&i64, sizeof(i64));
	    }
	    else if(type == CFG_ITEM_TYPE_UINT64)
	    {
		const guint64 i64 = (guint64)ATOL(gtk_entry_get_text(GTK_ENTRY(w)));
		item->value = g_memdup(&i64, sizeof(i64));
	    }
	}
	else if(GTK_IS_TEXT(w))
	{
	    const CfgItemType type = item->type;

	    /* Delete the existing value */
	    CFGItemResetValue(item);

	    /* Set the new value based on the item's type */
	    if(type == CFG_ITEM_TYPE_STRING)
	    {
		item->value = gtk_editable_get_chars(
		    GTK_EDITABLE(w),
		    0, -1
		);
	    }
	}
}


/*
 *	Sets the GtkPaned's position from the configuration value.
 *
 *	The list specifies the configuration list.
 *
 *	The parm specifies the configuration parameter name.
 *
 *	The w specifies the GtkPaned.
 */
void CfgGtkSetGtkPaned(
	const CfgList *list,
	const gchar *parm,
	GtkWidget *w
)
{
	GtkPaned *paned;

	if(w == NULL)
	    return;

	paned = GTK_PANED(w);
	if(paned == NULL)
	    return;

	/* Set the GtkPaned's position */
	gtk_paned_set_position(
	    paned,
	    CFGItemListGetValueI(list, parm)
	);
}

/*
 *	Sets the configuration value from the GtkPaned's position.
 *
 *	The list specifies the configuration list.
 *
 *	The parm specifies the configuration parameter name.
 *
 *	The w specifies the GtkPaned.
 */
void CfgGtkGetGtkPaned(
	CfgList *list,
	const gchar *parm,
	GtkWidget *w
)
{
	GtkPaned *paned;

	if(w == NULL)
	    return;

	paned = GTK_PANED(w);
	if(paned == NULL)
	    return;

	/* Get the GtkPaned's position */
	CFGItemListSetValueI(
	    list,
	    parm,
	    paned->child1_size,
	    FALSE
	);
}


/*
 *	Sets the GtkSpinButton's value from the configuration value.
 *
 *	The list specifies the configuration list.
 *
 *	The parm specifies the configuration parameter name.
 *
 *	The w specifies the GtkSpinButton..
 */
void CfgGtkSetGtkSpinButton(
	const CfgList *list,
	const gchar *parm,
	GtkWidget *w
)
{
	gint i;
	GtkSpinButton *spin;
	CfgItemType type;
	const CfgItem *item;

	if(w == NULL)
	    return;

	spin = GTK_SPIN_BUTTON(w);
	if(spin == NULL)
	    return;

	i = CFGItemListMatchParameter(list, parm);
	if(i < 0)
	    return;

	item = &list[i];

	/* No value on the item? */
	if(item->value == NULL)
	    return;

	/* Set the GtkSpinButton's value based on the item's type */
	type = item->type;
	if(type == CFG_ITEM_TYPE_STRING)
	{
	    gtk_entry_set_text(
		GTK_ENTRY(w),
		(const gchar *)item->value
	    );
	}
	else if(type == CFG_ITEM_TYPE_INT8)
	{
	    gtk_spin_button_set_value(
		spin,
		(gfloat)*(gint8 *)item->value
	    );
	}
	else if(type == CFG_ITEM_TYPE_UINT8)
	{
	    gtk_spin_button_set_value(
		spin,
		(gfloat)*(guint8 *)item->value
	    );
	}
	else if(type == CFG_ITEM_TYPE_INT16)
	{
	    gtk_spin_button_set_value(
		spin,
		(gfloat)*(gint16 *)item->value
	    );
	}
	else if(type == CFG_ITEM_TYPE_UINT16)
	{
	    gtk_spin_button_set_value(
		spin,
		(gfloat)*(guint16 *)item->value
	    );
	}
	else if(type == CFG_ITEM_TYPE_INT32)
	{
	    gtk_spin_button_set_value(
		spin,
		(gfloat)*(gint32 *)item->value
	    );
	}
	else if(type == CFG_ITEM_TYPE_UINT32)
	{
	    gtk_spin_button_set_value(
		spin,
		(gfloat)*(guint32 *)item->value
	    );
	}
	else if(type == CFG_ITEM_TYPE_INT64)
	{
	    gtk_spin_button_set_value(
		spin,
		(gfloat)*(gint64 *)item->value
	    );
	}
	else if(type == CFG_ITEM_TYPE_UINT64)
	{
	    gtk_spin_button_set_value(
		spin,
		(gfloat)*(guint64 *)item->value
	    );
	}
	else if(type == CFG_ITEM_TYPE_FLOAT)
	{
	    gtk_spin_button_set_value(
		spin,
		*(gfloat *)item->value
	    );
	}
	else if(type == CFG_ITEM_TYPE_DOUBLE)
	{
	    gtk_spin_button_set_value(
		spin,
		(gfloat)*(gdouble *)item->value
	    );
	}
}

/*
 *	Sets the configuration value from the GtkSpinButton's value.
 *
 *	The list specifies the configuration list.
 *
 *	The parm specifies the configuration parameter name.
 *
 *	The w specifies the GtkSpinButton.
 */
void CfgGtkGetGtkSpinButton(
	CfgList *list,
	const gchar *parm,
	GtkWidget *w
)
{
	gint i;
	GtkSpinButton *spin;
	CfgItemType type;
	CfgItem *item;

	if(w == NULL)
	    return;

	spin = GTK_SPIN_BUTTON(w);
	if(spin == NULL)
	    return;

	i = CFGItemListMatchParameter(list, parm);
	if(i < 0)
	    return;

	item = &list[i];

	/* Delete the existing value */
	CFGItemResetValue(item);

	/* Set the new value based on the item's type */
	type = item->type;
	if(type == CFG_ITEM_TYPE_STRING)
	{
	    item->value = STRDUP(gtk_entry_get_text(GTK_ENTRY(w)));
	}
	else if(type == CFG_ITEM_TYPE_INT8)
	{
	    const gint8 i8 = (gint8)gtk_spin_button_get_value_as_int(spin);
	    item->value = g_memdup(&i8, sizeof(i8));
	}
	else if(type == CFG_ITEM_TYPE_UINT8)
	{
	    const guint8 i8 = (guint8)gtk_spin_button_get_value_as_int(spin);
	    item->value = g_memdup(&i8, sizeof(i8));
	}
	else if(type == CFG_ITEM_TYPE_INT16)
	{
	    const gint16 i16 = (gint16)gtk_spin_button_get_value_as_int(spin);
	    item->value = g_memdup(&i16, sizeof(i16));
	}
	else if(type == CFG_ITEM_TYPE_UINT16)
	{
	    const guint16 i16 = (guint16)gtk_spin_button_get_value_as_int(spin);
	    item->value = g_memdup(&i16, sizeof(i16));
	}
	else if(type == CFG_ITEM_TYPE_INT32)
	{
	    const gint32 i32 = (gint32)gtk_spin_button_get_value_as_int(spin);
	    item->value = g_memdup(&i32, sizeof(i32));
	}
	else if(type == CFG_ITEM_TYPE_UINT32)
	{
	    const guint32 i32 = (guint32)gtk_spin_button_get_value_as_int(spin);
	    item->value = g_memdup(&i32, sizeof(i32));
	}
	else if(type == CFG_ITEM_TYPE_INT64)
	{
	    const gint64 i64 = (gint64)gtk_spin_button_get_value_as_int(spin);
	    item->value = g_memdup(&i64, sizeof(i64));
	}
	else if(type == CFG_ITEM_TYPE_UINT64)
	{
	    const guint64 i64 = (guint64)gtk_spin_button_get_value_as_int(spin);
	    item->value = g_memdup(&i64, sizeof(i64));
	}
	else if(type == CFG_ITEM_TYPE_FLOAT)
	{
	    const gfloat f = gtk_spin_button_get_value_as_float(spin);
	    item->value = g_memdup(&f, sizeof(f));
	}
	else if(type == CFG_ITEM_TYPE_DOUBLE)
	{
	    const gdouble d = (gdouble)gtk_spin_button_get_value_as_float(spin);
	    item->value = g_memdup(&d, sizeof(d));
	}
}


/*
 *	Sets the GtkToggleButton's value from the configuration value.
 *
 *	The list specifies the configuration list.
 *
 *	The parm specifies the configuration parameter name.
 *
 *	The w specifies the GtkToggleButton.
 */
void CfgGtkSetGtkToggleButton(
	const CfgList *list,
	const gchar *parm,
	GtkWidget *w
)
{
	GtkToggleButton *tb;

	if(w == NULL)
	    return;

	tb = GTK_TOGGLE_BUTTON(w);
	if(tb == NULL)
	    return;

	/* Set the GtkPaned's position */
	gtk_toggle_button_set_active(
	    tb,
	    CFGItemListGetValueI(list, parm) ? TRUE : FALSE
	);
}

/*
 *	Sets the configuration value from the GtkToggleButton's value.
 *
 *	The list specifies the configuration list.
 *
 *	The parm specifies the configuration parameter name.
 *
 *	The w specifies the GtkToggleButton.
 */
void CfgGtkGetGtkToggleButton(
	CfgList *list,
	const gchar *parm,
	GtkWidget *w
)
{
	GtkToggleButton *tb;

	if(w == NULL)
	    return;

	tb = GTK_TOGGLE_BUTTON(w);
	if(tb == NULL)
	    return;

	/* Get the GtkPaned's position */
	CFGItemListSetValueI(
	    list,
	    parm,
	    gtk_toggle_button_get_active(tb),
	    FALSE
	);
}


/*
 *	Sets the GtkWindow's position and size from the configuration
 *	values.
 *
 *	The list specifies the configuration list.
 *
 *	The parm_x and parm_y specifies the position configuration
 *	parameter names.
 *
 *	The parm_width and parm_height specifies the size configuration
 *	parameter names.
 *
 *	The w specifies the GtkWindow.
 */
void CfgGtkSetGtkWindow(
	const CfgList *list,
	const gchar *parm_x,
	const gchar *parm_y,
	const gchar *parm_width,
	const gchar *parm_height,
	GtkWidget *w
)
{
	if(w == NULL)
	    return;

	/* Set the GtkWindow's position */
	if((parm_x != NULL) && (parm_y != NULL))
	    gtk_widget_set_uposition(
		w,
		CFGItemListGetValueI(list, parm_x),
		CFGItemListGetValueI(list, parm_y)
	    );

	/* Set the GtkWindow's size */
	if((parm_width != NULL) && (parm_height != NULL))
	    gtk_widget_set_usize(
		w,
		CFGItemListGetValueI(list, parm_width),
		CFGItemListGetValueI(list, parm_height)
	    );
}

/*
 *	Sets the configuration values from the GtkWindow's position
 *	and size.
 *
 *	The list specifies the configuration list.
 *
 *	The parm_x and parm_y specifies the position configuration
 *	parameter names.
 *
 *	The parm_width and parm_height specifies the size configuration
 *	parameter names.
 *
 *	The w specifies the GtkWindow.
 */
void CfgGtkGetGtkWindow(
	CfgList *list,
	const gchar *parm_x,
	const gchar *parm_y,
	const gchar *parm_width,
	const gchar *parm_height,
	GtkWidget *w
)
{
	if(w == NULL)
	    return;

	/* Get the GtkWindow's position by obtaining its GdkWindow
	 * and obtaining its root origin
	 */
	if((parm_x != NULL) && (parm_y != NULL))
	{
	    GdkWindow *window = w->window;
	    if(window != NULL)
	    {
		gint x = 0, y = 0;
		gdk_window_get_root_origin(window, &x, &y);
		CFGItemListSetValueI(
		    list,
		    parm_x,
		    x,
		    FALSE
		);
		CFGItemListSetValueI(
		    list,
		    parm_y,
		    y,
		    FALSE
		);
	    }
	}

	/* Get the GtkWindow's size by using it's current GtkAllocation
	 * size
	 */
	if((parm_width != NULL) && (parm_height != NULL))
	{
	    const GtkAllocation *allocation = &w->allocation;
	    CFGItemListSetValueI(
		list,
		parm_width,
		(gint)allocation->width,
		FALSE
	    );
	    CFGItemListSetValueI(
		list,
		parm_height,
		(gint)allocation->height,
		FALSE
	    );
	}
}
