#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <glib.h>

#include "../include/string.h"
#include "../include/fio.h"
#include "../include/disk.h"

#include "cfg.h"
#include "cfg_fio.h"
#include "config.h"


static gchar *CFGStripQuotes(gchar *s);

/* Open */
static void CFGFileOpenGetValue(
	CfgItem *item,
	FILE *fp
);
gint CFGFileOpen(
	const gchar *path,
	CfgList *list
);
gint CFGFileOpenProgress(
        const gchar *path,
        CfgList *list,
        gint (*progress_cb)(
		const gulong,
		const gulong,
		gpointer
	),
        gpointer progress_data
);

/* Save */
static gchar *CfgFileSaveEscapeString(const gchar *s);
gint CFGFileSave(
	const gchar *path,
	const CfgList *list
);
gint CFGFileSaveProgress(
        const gchar *path,
        const CfgList *list,
        gint (*progress_cb)(
		const gulong,
		const gulong,
		gpointer
	),
        gpointer progress_data
);


#ifndef CFG_COMMENT_CHAR
# define CFG_COMMENT_CHAR			'#'
#endif

#ifndef CFG_DELIMINATOR_CHAR
# define CFG_DELIMINATOR_CHAR			'='
#endif

#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


static gchar *CFGStripQuotes(gchar *s)
{
	if(s == NULL)
	    return(NULL);

	if(*s == '"')
	{
	    gchar *s2 = s;
	    while(*s2 != '\0')
	    {
		*s2 = *(s2 + 1);
		s2++;
	    }
	    s2 -= 2;
	    if(s2 >= s)
	    {
		if(*s2 == '"')
		    *s2 = '\0';
	    }
	}

	return(s);
}


/*
 *	Gets the value for the CfgItem at the stream position.
 *
 *	The item specifies the CfgItem.
 *
 *	The fp specifies the stream position which should be
 *	positioned at the start of the value. The fp will be seeked
 *	past the value to the next line after this call.
 */
static void CFGFileOpenGetValue(
	CfgItem *item,
	FILE *fp
)
{
	gchar *s;
	gint vi[10];
	glong vl[10];
	gdouble vd[10];

	gint8		i8;
	guint8		ui8;
	gint16		i16;
	guint16		ui16;
	gint32 		i32;
	guint32		ui32;
	gint64		i64;
	guint64		ui64;
	gfloat		f;
	gdouble		d;
	CfgColor color;

	switch(item->type)
	{
	  case CFG_ITEM_TYPE_INT8:
	    FGetValuesI(fp, (int *)vi, 1);
	    i8 = (gint8)vi[0];
	    CFGItemSetValue(item, &i8);
	    break;
	  case CFG_ITEM_TYPE_UINT8:
	    FGetValuesI(fp, (int *)vi, 1);
	    ui8 = (guint8)vi[0];
	    CFGItemSetValue(item, &ui8);
	    break;

	  case CFG_ITEM_TYPE_INT16:
	    FGetValuesI(fp, (int *)vi, 1);
	    i16 = (gint16)vi[0];
	    CFGItemSetValue(item, &i16);
	    break;
	  case CFG_ITEM_TYPE_UINT16:
	    FGetValuesI(fp, (int *)vi, 1);  
	    ui16 = (guint16)vi[0];
	    CFGItemSetValue(item, &ui16);
	    break;

	  case CFG_ITEM_TYPE_INT32:
	    FGetValuesI(fp, (int *)vi, 1);
	    i32 = (gint32)vi[0];
	    CFGItemSetValue(item, &i32);
	    break;
	  case CFG_ITEM_TYPE_UINT32:
	    FGetValuesI(fp, (int *)vi, 1);
	    ui32 = (guint32)vi[0];
	    CFGItemSetValue(item, &ui32);
	    break;

	  case CFG_ITEM_TYPE_INT64:
	    FGetValuesL(fp, (long *)vl, 1);
	    i64 = (gint64)vl[0];
	    CFGItemSetValue(item, &i64);
	    break;
	  case CFG_ITEM_TYPE_UINT64:
	    FGetValuesL(fp, (long *)vl, 1);
	    ui64 = (guint64)vl[0];
	    CFGItemSetValue(item, &ui64);
	    break;

	  case CFG_ITEM_TYPE_FLOAT:
	    FGetValuesF(fp, (double *)vd, 1);
	    f = (gfloat)vd[0];
	    CFGItemSetValue(item, &f);
	    break;
	  case CFG_ITEM_TYPE_DOUBLE:
	    FGetValuesF(fp, (double *)vd, 1);
	    d = (gdouble)vd[0];
	    CFGItemSetValue(item, &d);
	    break;

	  case CFG_ITEM_TYPE_STRING:
	    g_free(item->value);
	    item->value = FGetString(fp);
	    break;

	  case CFG_ITEM_TYPE_INT_LIST:
	    s = (gchar *)FGetString(fp);
	    if(s != NULL)
	    {
		gchar *s2 = s;
		CfgIntList *intlist = CFGIntListNew(NULL);

		/* Delete existing Cfg Item value (if any) */
		CFGItemResetValue(item);

		/* Format:
		 *
		 * <i1> [i2] [i3] [i4] [...]
		 */
		while(*s2 != '\0')
		{
		    while(ISSPACE(*s2))
			s2++;

		    intlist->list = g_list_append(
			intlist->list,
			(gpointer)ATOI(s2)
		    );

		    /* Seek s2 to next value or end of string */
		    while(!ISSPACE(*s2) && (*s2 != '\0'))
			s2++;
		}

		/* Set the value to the cfg item */
		item->value = intlist;

		/* Delete the string */
		g_free(s);
	    }
	    break;

	  case CFG_ITEM_TYPE_STRING_LIST:
	    s = (gchar *)FGetString(fp);
	    if(s != NULL)
	    {
		gint len;
		gchar	*ss,
			*ss_start = s,
			*ss_end;
		CfgStringList *string_list = CFGStringListNew(NULL);

		/* Delete the existing CfgItem value */
		CFGItemResetValue(item);

		/* Format:
		 *
		 * "<s1>" "[s2]" "[s3]" "[s4]" "[...]"
		 */
		while(*ss_start != '\0')
		{
		    /* Seek ss_start to the starting deliminator */
		    ss_start = (gchar *)strchr(
			(char *)ss_start,
			'"'
		    );
		    if(ss_start == NULL)
			break;

		    ss_start++;		/* Seek past the deliminator */

		    /* Seek ss_end to the ending deliminator */
		    ss_end = ss_start;
		    while(*ss_end != '\0')
		    {
			if(*ss_end == '\\')
			{
			    ss_end++;
			    if(*ss_end == '\0')
				break;
			}
			else if(*ss_end == '"')
			{
			    break;
			}

			ss_end++;
		    }

		    /* Get this string */
		    len = (gint)(ss_end - ss_start);
		    ss = (gchar *)g_malloc((len + 1) * sizeof(gchar));
		    if(ss != NULL)
		    {
			memcpy(
			    ss,
			    ss_start,
			    len * sizeof(gchar)
			);
			ss[len] = '\0';

			/* Remove any escaped quotes */
			substr(
			    (char *)ss,
			    "\\\"",
			    "\""
			);

			string_list->list = g_list_append(
			    string_list->list,
			    ss
			);
		    }

		    /* Seek ss_start past the ending deliminator */
		    ss_start = ss_end;
		    if(*ss_start == '"')
			ss_start++;
		}

		/* Set the value to the cfg item */
		item->value = string_list;

		/* Delete the string */
		g_free(s);
	    }
	    break;

	  case CFG_ITEM_TYPE_COLOR:   
	    /* Format:
	     *
	     * <r> <g> <b> <a>
	     */
	    FGetValuesF(fp, (double *)vd, 4);
	    color.r = (gfloat)vd[0];
	    color.g = (gfloat)vd[1];
	    color.b = (gfloat)vd[2];
	    color.a = (gfloat)vd[3];
	    CFGItemSetValue(item, &color);
	    break;

	  case CFG_ITEM_TYPE_ACCELKEY_LIST:
	    s = (gchar *)FGetString(fp);
	    if(s != NULL)
	    {
		gint op_id;
		guint key, modifiers;
		gchar *s2 = s;
		GList	*glist,
			*akey_glist;
		CfgAccelkey *akey;
		CfgAccelkeyList
		    *akey_list_src = CFG_ACCELKEY_LIST(item->value),
		    *akey_list = CFGAccelkeyListNew(
			(akey_list_src != NULL) ?
			    akey_list_src->list : NULL
		    );

		/* Delete existing Cfg Item value (if any) */
		CFGItemResetValue(item);
		akey_list_src = NULL;

		/* Unable to create new Accelkey List? */
		if(akey_list == NULL)
		{
		    g_free(s);
		    break;
		}

		akey_glist = akey_list->list;

		/* Format:
		 *
		 * <op_id1> <key1> <modifiers1>
		 * [op_id2] [key2] [modifiers2]
		 * [op_id3] [key3] [modifiers3]
		 * [...]
		 */
		while(*s2 != '\0')
		{
#define SEEK_NEXT	{		\
 while(!ISSPACE(*s2) && (*s2 != '\0'))	\
  s2++;					\
 while(ISSPACE(*s2))			\
  s2++;					\
}

		    while(ISSPACE(*s2))
			s2++;

		    op_id = ATOI(s2);
		    SEEK_NEXT
		    key = (guint)ATOI(s2);
		    SEEK_NEXT
		    modifiers = (guint)ATOI(s2);
		    SEEK_NEXT

		    /* Check if this Accelkey's op_id with one
		     * in the Accelkey List
		     */
		    for(glist = akey_glist;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			akey = CFG_ACCELKEY(glist->data);
			if(akey == NULL)
			    continue;

			if(akey->op_id <= 0)
			    continue;

			if(akey->op_id == op_id)
			{
			    akey->key = key;
			    akey->modifiers = modifiers;
			    break;
			}
		    }
#undef SEEK_NEXT
		}

		/* Set the value to the cfg item */
		item->value = akey_list;

		/* Delete the string */
		g_free(s);
	    }
	    break;

	  case CFG_ITEM_TYPE_STYLE:
	    s = (gchar *)FGetString(fp);
	    if(s != NULL)
	    {
		gint n;
		gchar *s2 = s, *s3;
		CfgColor *c;
		CfgStyle *style = CFGStyleNew();

#define SEEK_NEXT	{		\
 while(!ISSPACE(*s2) && (*s2 != '\0'))	\
  s2++;					\
 while(ISSPACE(*s2))			\
  s2++;					\
}
#define GET_COLOR	{		\
 c->r = ATOF(s2);			\
 SEEK_NEXT				\
 c->g = ATOF(s2);			\
 SEEK_NEXT				\
 c->b = ATOF(s2);			\
 SEEK_NEXT				\
 c->a = ATOF(s2);			\
 SEEK_NEXT				\
}

		/* Delete existing Cfg Item value (if any) */
		CFGItemResetValue(item);

		/* Format:
		 *
		 * <font_name>
		 * (following repeats 5 times for the 5 states)
		 * <fg_rgba> <bg_rgba> <text_rgba> <base_rgba>
		 * <bg_pixmap_name>
		 */
		if(!strcasepfx(s2, "*none*"))
		{
		    style->font_name = STRDUP(s2);
		    s3 = strpbrk(style->font_name, " \t");
		    if(s3 != NULL)
			*s3 = '\0';
		}
		SEEK_NEXT

		for(n = 0; n < 5; n++)
		{
		    style->color_flags[n] |= CFG_STYLE_FG |
			CFG_STYLE_BG | CFG_STYLE_TEXT |
			CFG_STYLE_BASE;
		    c = &style->fg[n];
		    GET_COLOR
		    c = &style->bg[n];
		    GET_COLOR
		    c = &style->text[n];
		    GET_COLOR
		    c = &style->base[n];
		    GET_COLOR

		    /* bg_pixmap_name is defined if the first
		     * character is not a '-'
		     */
		    if(!strcasepfx(s2, "*none*"))
		    {
			style->bg_pixmap_name[n] = STRDUP(s2);
			s3 = strpbrk(style->bg_pixmap_name[n], " \t");
			if(s3 != NULL)
			    *s3 = '\0';
		    }
		    SEEK_NEXT
		}

		/* Set the value to the cfg item */
		item->value = style;

		/* Delete the string */
		g_free(s);
#undef GET_COLOR
#undef SEEK_NEXT
	    }
	    break;

	  case CFG_ITEM_TYPE_MENU:
	    s = (gchar *)FGetString(fp);
	    if(s != NULL)
	    {
		gchar **strv;
		CfgMenu	*menu_src = CFG_MENU(item->value),
			*menu = CFGMenuNew(NULL);

		/* Delete the existing Cfg Item value (if any) */
		CFGItemResetValue(item);
		menu_src = NULL;

		/* Unable to create a new Menu? */
		if(menu == NULL)
		{
		    g_free(s);
		    break;
		}

		/* Format:
		 *
		 * <label>;<command>;<icon_file>;<description>;
		 *    <ext_data>;<flags>...
		 *
		 * Each value is deliminated by a ';'
		 * character, each menu item occures every
		 * 5 values
		 */
		strv = g_strsplit(s, ";", -1);
		if(strv != NULL)
		{
		    gint i = 0;
		    CfgMenuItemFlags flags;
		    const gchar *label,
				*command,
				*icon_file,
				*description,
				*ext_data;

		    while(strv[i] != NULL)
		    {
			label = CFGStripQuotes(strv[i]);
			i++;

			if(strv[i] != NULL)
			{
			    command = CFGStripQuotes(strv[i]);
			    i++;
			}
			else
			    command = NULL;

			if(strv[i] != NULL)
			{
			    icon_file = CFGStripQuotes(strv[i]);
			    i++;
			}
			else
			    icon_file = NULL;

			if(strv[i] != NULL)
			{
			    description = CFGStripQuotes(strv[i]);
			    i++;
			}
			else
			    description = NULL;

			if(strv[i] != NULL)
			{
			    ext_data = CFGStripQuotes(strv[i]);
			    i++;
			}
			else
			    ext_data = NULL;

			if(strv[i] != NULL)
			{
			    flags = (CfgMenuItemFlags)ATOI(CFGStripQuotes(strv[i]));
			    i++;
			}
			else
			    flags = 0;

			menu->list = g_list_append(
			    menu->list,
			    CFGMenuItemNew(
				flags,
				label,
				command,
				icon_file,
				description,
				ext_data
			    )
			);
		    }

		    /* Delete the exploded values */
		    g_strfreev(strv);
		}

		/* Set the value to the cfg item */
		item->value = menu;

		/* Delete the string */
		g_free(s);
	    }
	    break;

	  default:
	    /* Unsupported value type */
	    FSeekNextLine(fp);
	    break;
	}
}

/*
 *	Opens the configuration list from the configuration file.
 *
 *	The path specifies the full path to the configuration file to
 *	open from.
 *
 *	The list specifies the configuration list to open to.
 *
 *	Returns:
 *
 *	0	Success.
 *	-1	General error (including no such file).
 *	-2	Invalid value.
 *	-3	System error or memory allocation error.
 */
gint CFGFileOpen(
	const gchar *path,
	CfgList *list
)
{
	return(CFGFileOpenProgress(
	    path,
	    list,
	    NULL, NULL
	));
}

/*
 *	Opens the configuration list from the configuration file.
 *
 *	The path specifies the full path to the configuration file to
 *	open from.
 *
 *	The list specifies the configuration list to open to.
 *
 *	The progress_cb and progress_data specifies the progress
 *	callback. If progress_cb is NULL then no progress callback
 *	will be used. If progress_cb() returns non-zero then the
 *	operation will be aborted.
 *
 *	Returns:
 *
 *	0	Success.
 *	-1	General error (including no such file).
 *	-2	Invalid value.
 *	-3	System error or memory allocation error.
 *	-4	User aborted.
 */
gint CFGFileOpenProgress(
        const gchar *path,
        CfgList *list,
        gint (*progress_cb)(
		const gulong,
		const gulong,
		gpointer
	),
        gpointer progress_data
)
{
	FILE *fp;
	struct stat stat_buf;
	gboolean ignore_all_errors = FALSE;
	gint i;
	gulong total_size;
	gchar *parm;

	if(STRISEMPTY(path) || (list == NULL))
	    return(-2);

	/* Report the initial progress */
	if(progress_cb != NULL)
	{
	    if(progress_cb(
		0l, 0l,
		progress_data
	    ))
		return(-4);
	}

	if(stat((const char *)path, &stat_buf))
	{
	    const gint error_code = (gint)errno;
#ifdef ENOENT
	    if(error_code == ENOENT)
#else
	    if(FALSE)
#endif
	    {

	    }
	    else
	    {
		g_printerr(
		    "%s: %s.\n",
		    path, g_strerror(error_code)
		);
	    }
	    return(-1);
	}
#ifdef S_ISREG
	if(!S_ISREG(stat_buf.st_mode))
	{
	    g_printerr(
		"%s: Error: Not a file.\n",
		path
	    );
	    return(-2);
	}
#endif

	/* Open the configuration file for reading */
	fp = fopen((const char *)path, "rb");
	if(fp == NULL)
	{
	    const gint error_code = (gint)errno;
	    g_printerr(
		"%s: %s.\n",
		path,
		g_strerror(error_code)
	    );
	    return(-1);
	}

	total_size = (gulong)stat_buf.st_size;

	/* Report the progress */
	if(progress_cb != NULL)
	{
	    if(progress_cb(
		0l, total_size,
		progress_data
	    ))
	    {
		(void)fclose(fp);
		return(-4);
	    }
	}

	parm = NULL;

	/* Begin reading the configuration file */
	while(TRUE)
	{
	    /* Report the progress */
	    if(progress_cb != NULL)
	    {
		if(progress_cb(
		    (gulong)ftell(fp),
		    total_size,
		    progress_data
		))
		{
		    g_free(parm);
		    (void)fclose(fp);
		    return(-4);
		}
	    }

	    /* Get the next parameter from the configuration file */
	    parm = (gchar *)FSeekNextParm(
		fp,
		(char *)parm,
		CFG_COMMENT_CHAR,
		CFG_DELIMINATOR_CHAR
	    );
	    if(parm == NULL)
		break;

	    /* Get the CfgItem in the CfgList who's parameter matches
	     * this parameter
	     */
	    i = CFGItemListMatchParameter(list, parm);
	    if(i > -1)
	    {
		/* Get the value for this CfgItem and position the
		 * stream to the next line
		 */
		CFGFileOpenGetValue(
		    &list[i],			/* CfgItem */
		    fp
		);
	    }
	    else
	    {
		/* No such CfgItem */
		if(!ignore_all_errors)
		    g_printerr(
"%s: Warning: Unsupported parameter \"%s\".\n",
			path,
			parm
		    );
		FSeekNextLine(fp);
	    }
	}

	g_free(parm);

	/* Close the configuration file */
	(void)fclose(fp);

	/* Report the final progress */
	if(progress_cb != NULL)
	{
	    if(progress_cb(
		total_size, total_size,
		progress_data
	    ))
		return(-4);
	}

	return(0);
}


/*
 *	Puts escapes sequences ('\') in front of '\' characters.
 */
gchar *CfgFileSaveEscapeString(const gchar *s)
{
	if(s == NULL)
	    return(NULL);

	return((gchar *)strsub(
	    (const char *)s,
	    "\\",
	    "\\\\"
	));
}

/*
 *	Saves the configuration list to the configuration file.
 *
 *	The path specifies the full path to the configuration file to
 *	save to.
 *
 *	The list specifies the configuration list to save from.
 *
 *	Returns:
 *
 *	0	Success.
 *	-1	General error (including unable to open file for writing).
 *	-2	Invalid value.
 *	-3	System error or memory allocation error.
 */
gint CFGFileSave(
	const gchar *path,
	const CfgList *list
)
{
	return(CFGFileSaveProgress(
	    path,
	    list,
	    NULL, NULL
	));
}

/*
 *	Saves the configuration list to the configuration file.
 *
 *	The path specifies the full path to the configuration file to
 *	save to.
 *
 *	The list specifies the configuration list to save from.
 *
 *	The progress_cb and progress_data specifies the progress
 *	callback. If progress_cb is NULL then no progress callback
 *	will be used. If progress_cb() returns non-zero then the
 *	operation will be aborted.
 *
 *	Returns:
 *
 *	0	Success.
 *	-1	General error (including unable to open file for writing).
 *	-2	Invalid value.
 *	-3	System error or memory allocation error.
 *	-4	User aborted.
 */
gint CFGFileSaveProgress(
        const gchar *path,
        const CfgList *list,
        gint (*progress_cb)(
		const gulong,
		const gulong,
		gpointer
	),
        gpointer progress_data
)
{
	FILE *fp;
	struct stat stat_buf;
	gint		i,
			ncfg_items;
	gchar *parent;
	const CfgItem *ci;

	if(STRISEMPTY(path) || (list == NULL))
	    return(-2);

	/* Report the initial progress */
	if(progress_cb != NULL)
	{
	    if(progress_cb(
		0l, 0l,
		progress_data
	    ))
		return(-4);
	}

	/* Check if the file already exists */
	if(!stat((const char *)path, &stat_buf))
	{
#ifdef S_ISDIR
	    /* Check if it is a directory */
	    if(S_ISDIR(stat_buf.st_mode))
	    {
		g_printerr(
"%s: Is a directory.\n",
		    path
		);
		return(-2);
	    }
#endif
	}

	/* Get the parent directory and create it as needed */
	parent = g_dirname(path);
	if(parent != NULL)
	{
	    const mode_t m = umask(0);
	    gint status, error_code;

	    umask(m);

	    status = rmkdir(
		(char *)parent,
		(~m) &
		    (S_IRUSR | S_IWUSR | S_IXUSR |
		     S_IRGRP | S_IWGRP | S_IXGRP |
		     S_IROTH | S_IWOTH | S_IXOTH)
	    );
	    error_code = (gint)errno;
	    if(status != 0)
	    {
		g_printerr(
		    "%s: %s.\n",
		    parent,
		    g_strerror(error_code)
		);
		g_free(parent);
		return(-1);
	    }

	    g_free(parent);
	}

	/* Open the configuration file for writing */
	fp = fopen((const char *)path, "wb");
	if(fp == NULL)
	{
	    const gint error_code = (gint)errno;
	    g_printerr(
		"%s: %s.\n",
		path,
		g_strerror(error_code)
	    );
	    return(-1);
	}

	/* Calculate the number of items in the configuration list */
	i = 0;
	ci = &list[i];
	while((ci->type != 0) &&
	      (ci->parameter != NULL)
	)
	{
            i++;
            ci = &list[i];
	}
	ncfg_items = i;

	/* Report the progress */
	if(progress_cb != NULL)
	{
	    if(progress_cb(
		0l, (gulong)ncfg_items,
		progress_data
	    ))
	    {
		(void)fclose(fp);
		return(-4);
	    }
	}

	/* Begin writing the configuration file */
	i = 0;
	ci = &list[i];
	while((ci->type != 0) &&
	      (ci->parameter != NULL)
	)
	{
	    /* Report the progress */
	    if(progress_cb != NULL)
	    {
		if(progress_cb(
		    (gulong)i, (gulong)ncfg_items,
		    progress_data
		))
		{
		    (void)fclose(fp);
		    return(-4);
		}
	    }

	    if(!STRISEMPTY(ci->parameter))
	    {
		const gchar *parm = ci->parameter;
		const gpointer value = ci->value;
		gchar *s;
		const CfgIntList *intlist;
		const CfgStringList *string_list;
		const CfgColor *color;
		const CfgAccelkeyList *accelkey_list;
		const CfgStyle *style;
		const CfgMenu *menu;

		switch(ci->type)
		{
		  case CFG_ITEM_TYPE_NONE:
		    break;

		  case CFG_ITEM_TYPE_INT8:
		    (void)fprintf(
			fp,
			"%s %c %i\n",
			parm, CFG_DELIMINATOR_CHAR,
			(value != NULL) ?
			    (int)(*(gint8 *)value) : 0
		    );
		    break;
		  case CFG_ITEM_TYPE_UINT8:
		    (void)fprintf(
			fp,
			"%s %c %i\n",
			parm, CFG_DELIMINATOR_CHAR,
			(value != NULL) ?
			    (unsigned int)(*(guint8 *)value) : 0
		    ); 
		    break;

		  case CFG_ITEM_TYPE_INT16:
		    (void)fprintf(
			fp,
			"%s %c %i\n",
			parm, CFG_DELIMINATOR_CHAR,
			(value != NULL) ?
			    (int)(*(gint16 *)value) : 0
		    );
		    break;
		  case CFG_ITEM_TYPE_UINT16:
		    (void)fprintf(
			fp,
			"%s %c %i\n",
			parm, CFG_DELIMINATOR_CHAR,
			(value != NULL) ?
			    (unsigned int)(*(guint16 *)value) : 0
 		    );
		    break;

		  case CFG_ITEM_TYPE_INT32:
		    (void)fprintf(
			fp,
			"%s %c %i\n",
			parm, CFG_DELIMINATOR_CHAR,
			(value != NULL) ?
			    (int)(*(gint32 *)value) : 0
		    );
		    break;
		  case CFG_ITEM_TYPE_UINT32:
		    (void)fprintf(
			fp,
			"%s %c %i\n",
			parm, CFG_DELIMINATOR_CHAR,
			(value != NULL) ?
			    (unsigned int)(*(guint32 *)value) : 0
		    );
		    break;

		  case CFG_ITEM_TYPE_INT64:
		    (void)fprintf(
			fp,
			"%s %c %ld\n",
			parm, CFG_DELIMINATOR_CHAR,
			(value != NULL) ?
			    (long)(*(gint64 *)value) : 0
		    );
		    break;
		  case CFG_ITEM_TYPE_UINT64:
		    (void)fprintf(
			fp,
			"%s %c %ld\n",
			parm, CFG_DELIMINATOR_CHAR,
			(value != NULL) ?
			    (unsigned long)(*(guint64 *)value) : 0
  		    );
		    break;

		  case CFG_ITEM_TYPE_FLOAT:
		    (void)fprintf(
			fp,
			"%s %c %f\n",
			parm, CFG_DELIMINATOR_CHAR,
			(value != NULL) ?
			    (float)(*(gfloat *)value) : 0.0f
 		    );
		    break;
		  case CFG_ITEM_TYPE_DOUBLE:
		    (void)fprintf(
			fp,
			"%s %c %f\n",
			parm, CFG_DELIMINATOR_CHAR,
			(value != NULL) ?
			    (double)(*(gdouble *)value) : 0.0
		    );
		    break;

		  case CFG_ITEM_TYPE_STRING:
		    s = CfgFileSaveEscapeString((const gchar *)value);
		    (void)fprintf(
			fp,
			"%s %c %s\n",
			parm, CFG_DELIMINATOR_CHAR,
			s
		    );
		    g_free(s);
		    break;

		  case CFG_ITEM_TYPE_INT_LIST:
		    /* Format:
		     *
		     * <i>
		     * (repeats...)
		     */
		    intlist = CFG_INT_LIST(value);
		    (void)fprintf(
			fp,
			"%s %c ",
			parm, CFG_DELIMINATOR_CHAR
		    );
		    if(intlist != NULL)
		    {
			GList *glist = intlist->list;
			while(glist != NULL)
			{
			    (void)fprintf(fp, "%i", (int)glist->data);
			    glist = g_list_next(glist);
			    if(glist != NULL)
				(void)fprintf(fp, " ");
			}
		    }
		    (void)fprintf(fp, "\n");
		    break;

		  case CFG_ITEM_TYPE_STRING_LIST:
		    /* Format:
		     *
		     * "<s1>" "[s2]" "[s3]" "[s4]" "[...]"
		     */
		    string_list = CFG_STRING_LIST(value);
		    (void)fprintf(
			fp,
			"%s %c ",
			parm, CFG_DELIMINATOR_CHAR
		    );
		    if(string_list != NULL)
		    {
			gchar	*s,
				*s2;
			GList *glist = string_list->list;
			while(glist != NULL)
			{
			    /* Escape all backslashes */
			    s = CfgFileSaveEscapeString((const gchar *)glist->data);
			    /* Escape all double quotes */
			    s2 = (gchar *)strsub(
				(const char *)s,
				"\"",
				"\\\""
			    );
			    g_free(s);
			    (void)fprintf(fp, "\"%s\"", s2);
			    g_free(s2);
			    glist = g_list_next(glist);
			    if(glist != NULL)
				(void)fprintf(fp, " ");
			}
		    }
		    (void)fprintf(fp, "\n");
		    break;

		  case CFG_ITEM_TYPE_COLOR:
		    /* Format:
		     *
		     * <r> <g> <b> <a>
		     */
		    color = CFG_COLOR(value);
		    (void)fprintf(
			fp,
			"%s %c %f %f %f %f\n",
			parm, CFG_DELIMINATOR_CHAR,
			(value != NULL) ?
			    (float)color->r : 0.0f,
			(value != NULL) ?
			    (float)color->g : 0.0f,
			(value != NULL) ?   
			    (float)color->b : 0.0f,
			(value != NULL) ?
			    (float)color->a : 0.0f
		    );
		    break;

		  case CFG_ITEM_TYPE_ACCELKEY_LIST:
		    /* Format:
		     *
		     * <op_id1> <key1> <modifiers1>
		     * [op_id2] [key2] [modifiers2]
		     * [op_id3] [key3] [modifiers3]
		     * [...]
		     */
		    accelkey_list = CFG_ACCELKEY_LIST(value);
		    (void)fprintf(
			fp,
			"%s %c ",
			parm, CFG_DELIMINATOR_CHAR
		    );
		    if(accelkey_list != NULL)
		    {
			GList *glist;
			CfgAccelkey *akey;

			for(glist = accelkey_list->list;
			    glist != NULL;
			    glist = g_list_next(glist)
			)
			{
			    akey = CFG_ACCELKEY(glist->data);
			    if(akey == NULL)
				continue;

			    if(akey->op_id <= 0)
				continue;

			    (void)fprintf(
				fp, "%i %i %i",
				(int)akey->op_id,
				(int)akey->key,
				(int)akey->modifiers
			    );

			    if(g_list_next(glist) != NULL)
				(void)fprintf(fp, "\\\n ");
			}
		    }
		    (void)fprintf(fp, "\n");  
		    break;

		  case CFG_ITEM_TYPE_STYLE:
		    /* Format:
		     *
		     * <font_name>
		     * (repeat 5 times for 5 states)
		     * <fg_rgba> <bg_rgba> <text_rgba> <base_rgba>
		     * <bg_pixmap_name>
		     */
		    style = CFG_STYLE(value);
		    (void)fprintf(
			fp,
			"%s %c ",
			parm, CFG_DELIMINATOR_CHAR
		    );
		    if(style != NULL)
		    {
			gint n;
			const CfgColor *c;

			(void)fprintf(
			    fp,
			    "%s\\\n ",
			    STRISEMPTY(style->font_name) ?
				"*none*" : (const char *)style->font_name
			);

			for(n = 0; n < CFG_STYLE_STATES; n++)
			{
			    c = &style->fg[n];
			    (void)fprintf(
				fp,
				"%f %f %f %f\\\n ",
				(float)c->r, (float)c->g, (float)c->b, (float)c->a
			    );
			    c = &style->bg[n];
			    (void)fprintf(
				fp,
				"%f %f %f %f\\\n ",
				(float)c->r, (float)c->g, (float)c->b, (float)c->a
			    );
			    c = &style->text[n];
			    (void)fprintf(
				fp,
				"%f %f %f %f\\\n ",
				(float)c->r, (float)c->g, (float)c->b, (float)c->a
			    );
			    c = &style->base[n];
			    (void)fprintf(
				fp,
				"%f %f %f %f\\\n ",
				(float)c->r, (float)c->g, (float)c->b, (float)c->a
			    );
			    (void)fprintf(
				fp,
				"%s",
				STRISEMPTY(style->bg_pixmap_name[n]) ?
				    "*none*" : (const char *)style->bg_pixmap_name[n]
			    );
			    /* Not last state? */
			    if(n < (CFG_STYLE_STATES - 1))
				(void)fprintf(fp, "\\\n ");
			}
		    }
		    (void)fprintf(fp, "\n");
		    break;

		  case CFG_ITEM_TYPE_MENU:
		    /* Format:
		     *
		     * <label>;<command>;<icon_file>;<description>;
		     *     <ext_data>;<flags>...
		     *
		     * Each value is deliminated by a ';' character,
		     * each menu item occures every 5 values
		     */
		    menu = CFG_MENU(value);
		    (void)fprintf(
			fp,
			"%s %c ",
			parm, CFG_DELIMINATOR_CHAR
		    );
		    if(menu != NULL)
		    {
			gchar *s = NULL, *sv;
			GList *glist;
			CfgMenuItem *mi;

			for(glist = menu->list;
			    glist != NULL;
			    glist = g_list_next(glist)
			)
			{
			    mi = CFG_MENU_ITEM(glist->data);
			    if(mi == NULL)
				continue;

			    sv = g_strdup_printf(
				"\"%s\";\"%s\";\"%s\";\"%s\";\"%s\";\"%i\"",
				(mi->label != NULL) ? mi->label : "",
				(mi->command != NULL) ? mi->command : "",
				(mi->icon_file != NULL) ? mi->icon_file : "",
				(mi->description != NULL) ? mi->description : "",
				(mi->ext_data != NULL) ? mi->ext_data : "",
				mi->flags
			    );
			    if(s != NULL)
			    {
				gchar *s_tmp = g_strconcat(
				    s,
				    ";\\\n",
				    sv,
				    NULL
				);
				g_free(sv);
				g_free(s);
				s = s_tmp;
			    }
			    else
			    {
				s = STRDUP(sv);
				g_free(sv);
			    }
			}

			if(s != NULL)
			{
			    (void)fprintf(fp, "%s", s);
			    g_free(s);
			}
		    }
		    (void)fprintf(fp, "\n");  
		    break;

		}
	    }

	    i++;
	    ci = &list[i];
	}

	/* Close the configuration file */
	if(fclose(fp))
	{
	    const gint error_code = (gint)errno;
	    g_printerr(
		"%s: %s.\n",
		path,
		g_strerror(error_code)
	    );
	    return(-1);
	}

	/* Report the final progress */
	if(progress_cb != NULL)
	{
	    if(progress_cb(
		(gulong)ncfg_items, (gulong)ncfg_items,
		progress_data
	    ))
		return(-4);
	}

	return(0);
}
