/*
		       Configuration Items & Lists
 */

#ifndef CFG_H
#define CFG_H

#include <glib.h>


typedef struct _CfgItem			CfgItem;
#define CFG_ITEM(p)			((CfgItem *)(p))

/*
 *	Configurations List:
 *
 *	Same type as the CfgItem, it is not a GList because an
 *	array is easier to initialize using a constant declarative.
 */
#define CfgList				CfgItem


/* Types for cfg_item->value */
typedef struct _CfgIntList		CfgIntList;
#define CFG_INT_LIST(p)			((CfgIntList *)(p))

typedef struct _CfgStringList		CfgStringList;
#define CFG_STRING_LIST(p)		((CfgStringList *)(p))

typedef struct _CfgColor		CfgColor;
#define CFG_COLOR(p)			((CfgColor *)(p))

typedef struct _CfgAccelkey		CfgAccelkey;
#define CFG_ACCELKEY(p)			((CfgAccelkey *)(p))

typedef struct _CfgAccelkeyList		CfgAccelkeyList;
#define CFG_ACCELKEY_LIST(p)		((CfgAccelkeyList *)(p))

typedef struct _CfgStyle		CfgStyle;
#define CFG_STYLE(p)			((CfgStyle *)(p))

typedef struct _CfgMenuItem		CfgMenuItem;
#define CFG_MENU_ITEM(p)		((CfgMenuItem *)(p))

typedef struct _CfgMenu			CfgMenu;
#define CFG_MENU(p)			((CfgMenu *)(p))


/*
 *	CfgItem Value Types:
 *
 *	To determine the data type of cfg_item->value.
 */
typedef enum {
	CFG_ITEM_TYPE_NONE,
	CFG_ITEM_TYPE_INT8,
	CFG_ITEM_TYPE_UINT8,
	CFG_ITEM_TYPE_INT16,
	CFG_ITEM_TYPE_UINT16,
	CFG_ITEM_TYPE_INT32,
	CFG_ITEM_TYPE_UINT32,
	CFG_ITEM_TYPE_INT64,
	CFG_ITEM_TYPE_UINT64,
	CFG_ITEM_TYPE_FLOAT,
	CFG_ITEM_TYPE_DOUBLE,
	CFG_ITEM_TYPE_STRING,
	CFG_ITEM_TYPE_INT_LIST,		/* CfgIntList */
	CFG_ITEM_TYPE_STRING_LIST,	/* CfgStringList */
	CFG_ITEM_TYPE_COLOR,		/* CfgColor */
	CFG_ITEM_TYPE_ACCELKEY_LIST,	/* CfgAccelkeyList */
	CFG_ITEM_TYPE_STYLE,		/* CfgStyle */
	CFG_ITEM_TYPE_MENU		/* CfgMenu */
} CfgItemType;


/*
 *	Configuration Item:
 */
struct _CfgItem {
	CfgItemType	type;		/* Determines member value's type */
	gchar		*parameter;	/* Parameter string */
	gpointer	value;
};

/*
 *	Integer List:
 */
struct _CfgIntList {                 
	GList		*list;		/* (gint) */
};

/*
 *	String List:
 */
struct _CfgStringList {                 
	GList		*list;		/* (gchar *) */
};

/*
 *	Color:
 */
struct _CfgColor {
	gfloat		r, g, b, a;	/* 0.0 to 1.0 */
};

/*
 *	Accelkey:
 */
struct _CfgAccelkey {
	gint		op_id;
	guint		key,
			modifiers;
};

/*
 *	Accelkey List:
 */
struct _CfgAccelkeyList {
	GList		*list;		/* (CfgAccelkey *) */
};


/*
 *	Style Flags:
 */
typedef enum {
	CFG_STYLE_FG			= (1 << 0),
	CFG_STYLE_BG			= (1 << 1),
	CFG_STYLE_TEXT			= (1 << 2),
	CFG_STYLE_BASE			= (1 << 3)
} CfgStyleFlags;

/*
 *	Style:
 *
 *	Mimics the GtkRcStyle structure.
 */
struct _CfgStyle {
#define CFG_STYLE_STATES	5
	gchar		*font_name;
	CfgStyleFlags	color_flags[CFG_STYLE_STATES];
	CfgColor	fg[CFG_STYLE_STATES],
			bg[CFG_STYLE_STATES],
			text[CFG_STYLE_STATES],                      
			base[CFG_STYLE_STATES];
	gchar		*bg_pixmap_name[CFG_STYLE_STATES];
};


/*
 *	Menu Item Flags:
 */
typedef enum {
	CFG_MENU_ITEM_SENSITIVE			= (1 << 0),
	CFG_MENU_ITEM_HAS_FUNCTION		= (1 << 1),	/* Activates a
								 * function */
	CFG_MENU_ITEM_IS_SEPARATOR		= (1 << 2),
	CFG_MENU_ITEM_RUN_IN_TERMINAL		= (1 << 8)
} CfgMenuItemFlags;

/*
 *	Menu Item:
 */
struct _CfgMenuItem {
	CfgMenuItemFlags	flags;
	gchar		*label,
			*command,
			*icon_file,
			*description,
			*ext_data;	/* Application specific data */
};

/*
 *	Menu:
 */
struct _CfgMenu {
	GList		*list;		/* GList of CfgMenuItem * menu items */
};


/* Item Types */
extern CfgIntList *CFGIntListNew(GList *glist);
extern void CFGIntListDelete(CfgIntList *intlist);

extern CfgStringList *CFGStringListNew(GList *glist);
extern void CFGStringListDelete(CfgStringList *string_list);

extern CfgColor *CFGColorNew(
	const gfloat r,
	const gfloat g,
	const gfloat b,
	const gfloat a
);
extern void CFGColorDelete(CfgColor *color);

extern CfgAccelkey *CFGAccelkeyNew(
	const gint op_id,
	const guint key,
	const guint modifiers
);
extern void CFGAccelkeyDelete(CfgAccelkey *accelkey);
extern CfgAccelkeyList *CFGAccelkeyListNew(GList *accelkeys_list);
extern void CFGAccelkeyListDelete(CfgAccelkeyList *accelkeys_list);

extern CfgStyle *CFGStyleNew(void);
extern void CFGStyleDelete(CfgStyle *style);

extern CfgMenuItem *CFGMenuItemNew(
	const CfgMenuItemFlags flags,
	const gchar *label,
	const gchar *command,
	const gchar *icon_file,
	const gchar *description,
	const gchar *ext_data
);
extern void CFGMenuItemDelete(CfgMenuItem *mi);
extern CfgMenu *CFGMenuNew(GList *menu_items_list);
extern void CFGMenuDelete(CfgMenu *m);


/* Cfg Item */
extern CfgItem *CFGItemNew(
	const CfgItemType type,
	const gchar *parameter
);
extern gpointer CFGItemGetValue(
	const CfgItem *item,
	CfgItemType *type_rtn
);
extern void CFGItemSetValue(
	CfgItem *item,
	gconstpointer value
);
extern void CFGItemResetValue(CfgItem *item);
extern void CFGItemReset(CfgItem *item);
extern void CFGItemDelete(CfgItem *item);


/* CfgList */
extern CfgList *CFGItemListCopyList(const CfgList *list);
extern void CFGItemListDeleteList(CfgList *list);
extern gint CFGItemListMatchParameter(
	const CfgList *list,
	const gchar *parameter
);

/* Get CfgItem Value From CfgList */
extern gpointer CFGItemListMatchGetValue(
	const CfgList *list,
	const gchar *parameter,
	CfgItemType *type_rtn
);
extern gint CFGItemListGetValueI(
	const CfgList *list,
	const gchar *parameter
);
extern glong CFGItemListGetValueL(
	const CfgList *list,
	const gchar *parameter
);
extern gulong CFGItemListGetValueUL(
	const CfgList *list,
	const gchar *parameter
);
extern gfloat CFGItemListGetValueF(
	const CfgList *list,
	const gchar *parameter
);
extern gdouble CFGItemListGetValueD(
	const CfgList *list,
	const gchar *parameter
);
extern gchar *CFGItemListGetValueS(
	const CfgList *list,
	const gchar *parameter
);
extern CfgColor *CFGItemListGetValueColor(
	const CfgList *list,
	const gchar *parameter
);
extern CfgIntList *CFGItemListGetValueIntList(
	const CfgList *list,
	const gchar *parameter
);
extern CfgStringList *CFGItemListGetValueStringList(
	const CfgList *list,
	const gchar *parameter
);
extern CfgAccelkeyList *CFGItemListGetValueAccelkeyList(
	const CfgList *list,
	const gchar *parameter
);
extern CfgStyle *CFGItemListGetValueStyle(
	const CfgList *list,
	const gchar *parameter
);
extern CfgMenu *CFGItemListGetValueMenu(
	const CfgList *list,
	const gchar *parameter
);

/* Set CfgItem Value In CfgList */
extern void CFGItemListSetValue(
	CfgList *list,
	const gchar *parameter,
	gconstpointer value,
	const gboolean create_as_needed
);
extern void CFGItemListSetValueI(
	CfgList *list,
	const gchar *parameter,
	const gint value,
	const gboolean create_as_needed
);
extern void CFGItemListSetValueL(
	CfgList *list,
	const gchar *parameter,
	const glong value,
	const gboolean create_as_needed
);
extern void CFGItemListSetValueUL(
	CfgList *list,
	const gchar *parameter,
	const gulong value,
	const gboolean create_as_needed
);
extern void CFGItemListSetValueF(
	CfgList *list,
	const gchar *parameter,
	const gfloat value,
	const gboolean create_as_needed
);
extern void CFGItemListSetValueD(
	CfgList *list,
	const gchar *parameter,
	const gdouble value,
	const gboolean create_as_needed
);
extern void CFGItemListSetValueS(
	CfgList *list,
	const gchar *parameter,
	const gchar *value,
	const gboolean create_as_needed
);
extern void CFGItemListSetValueColor(
	CfgList *list,
	const gchar *parameter,
	const CfgColor *value,
	const gboolean create_as_needed
);
extern void CFGItemListSetValueIntList(
	CfgList *list,
	const gchar *parameter,
	const CfgIntList *value,
	const gboolean create_as_needed
);
extern void CFGItemListSetValueStringList(
	CfgList *list,
	const gchar *parameter,
	const CfgStringList *value,
	const gboolean create_as_needed
);
extern void CFGItemListSetValueAccelkeyList(
	CfgList *list,
	const gchar *parameter,
	const CfgAccelkeyList *value,
	const gboolean create_as_needed
);
extern void CFGItemListSetValueStyle(
	CfgList *list,
	const gchar *parameter,
	const CfgStyle *value,
	const gboolean create_as_needed
);
extern void CFGItemListSetValueMenu(
	CfgList *list,
	const gchar *parameter,
	const CfgMenu *value,
	const gboolean create_as_needed
);


#endif	/* CFG_H */
