#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <gtk/gtk.h>

#include "cfg.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_stream.h"
#include "libendeavour2-base/edv_process.h"
#include "libendeavour2-base/edv_property.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "libendeavour2-base/edv_archive_obj.h"
#include "edv_archive_comment.h"
#include "edv_status_bar.h"
#include "archive_comments_dlg.h"
#include "archiver.h"
#include "archiver_list.h"
#include "archiver_op.h"
#include "archiver_subprocess.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"


/* Subprocess */
EDVArchiverSubprocess *edv_archiver_subprocess_new(void);
void edv_archiver_subprocess_delete(EDVArchiverSubprocess *sp);

/* Start/Stop */
EDVArchiverSubprocess *edv_archiver_subprocess_start(
	EDVArchiver *archiver,
	const EDVArchiverOpID op,
	const gchar *command
);
EDVArchiverSubprocess *edv_archiver_subprocess_start_list_archive(
	EDVArchiver *archiver,
	const gchar *arch_path,
	const gchar *filter,
	const gchar *password,
	const gboolean show_progress,
	const gboolean show_comments
);
gint edv_archiver_subprocess_get_pid(EDVArchiverSubprocess *sp);
void edv_archiver_subprocess_stop(EDVArchiverSubprocess *sp);
gint edv_archiver_subprocess_get_stop_count(EDVArchiverSubprocess *sp);

/* Callbacks */
static gint edv_archiver_subprocess_timeout_cb(gpointer data);

/* Operations */
static EDVArchiveObject *edv_archiver_subprocess_list_parse_list_value(
	EDVArchiver *archiver,
	EDVArchiverSubprocess *sp,
	const gchar *s
);
static gint edv_archiver_subprocess_list_iteration(EDVArchiverSubprocess *sp);
gint edv_archiver_subprocess_iteration(EDVArchiverSubprocess *sp);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)

#define FCLOSE(p)	(((p) != NULL) ? (gint)fclose(p) : -1)
#define INTERRUPT(p)	(((p) > 0) ? (gint)kill((int)(p), SIGINT) : -1)

#define FDISCARD(p)	{			\
 if((p) != NULL) {				\
  const gint fd = (gint)fileno(p);		\
  while(edv_poll_read(fd)) {			\
   if(fgetc(p) == EOF)				\
    break;					\
  }						\
 }						\
}


/*
 *	Subprocess:
 *
 *	Forward declaration is in archiver.h.
 */
struct _EDVArchiverSubprocess {

        EDVArchiverOpID op;
        EDVArchiver     *archiver;
        gint            freeze_count,
                        stop_count;

        guint           timeout_id;

        GTimer          *timer;

        gint            pid;
        FILE            *cstdout,
                        *cstderr;
        gchar           *cstdout_buf,
                        *cstderr_buf;

        gboolean        show_progress;
        gulong          progress_i,
                        progress_n;

        gchar           *arch_path,
                        *filter,
                        *password;

        /* When listing archives and this is TRUE then the archive's
         * comments will be displayed after a successful non-aborted
         * list */
        gboolean        show_comments;
};


/*
 *	Creates a new subprocess.
 */
EDVArchiverSubprocess *edv_archiver_subprocess_new(void)
{
	return(EDV_ARCHIVER_SUBPROCESS(g_malloc0(sizeof(EDVArchiverSubprocess))));
}

/*
 *	Stops and deletes the subprocess.
 *
 *	If the Archiver was set on the subprocess then it will be
 *	marked as no longer passive busy and its display will be
 *	updated.
 */
void edv_archiver_subprocess_delete(EDVArchiverSubprocess *sp)
{
	if(sp == NULL)
		return;

	/* If the archiver was set then set it as no longer passive
	 * busy and update its display
	 */
	if(sp->archiver != NULL)
	{
		EDVArchiver *archiver = sp->archiver;
		edv_archiver_set_passive_busy(archiver, FALSE);
		edv_archiver_update_display(archiver);
	}

	sp->freeze_count++;

	sp->timeout_id = GTK_TIMEOUT_REMOVE(sp->timeout_id);

	if(sp->timer != NULL)
		g_timer_destroy(sp->timer);

	(void)INTERRUPT(sp->pid);

	(void)FCLOSE(sp->cstdout);
	(void)FCLOSE(sp->cstderr);

	g_free(sp->cstdout_buf);
	g_free(sp->cstderr_buf);

	g_free(sp->arch_path);
	g_free(sp->filter);
	g_free(sp->password);

	sp->freeze_count--;

	g_free(sp);
}


/*
 *	Starts a new subprocess on the Archiver.
 *
 *	If the Archiver already has a subprocess set on it then that
 *	subprocess will be stopped and deleted. The Archiver will be
 *	marked as passive busy and its display will be updated.
 *
 *	The archiver specifies the Archiver.
 *
 *	The op specifies the type of subprocess operation. The following
 *	operations are supported:
 *
 *	EDV_ARCHIVER_OP_OPEN		List archive.
 *
 *	The command specifies the command to execute.
 *
 *	Returns the new EDVArchiverSubprocess or NULL on error.
 */
EDVArchiverSubprocess *edv_archiver_subprocess_start(
	EDVArchiver *archiver,
	const EDVArchiverOpID op,
	const gchar *command
)
{
	EDVArchiverSubprocess *sp;

	if(archiver == NULL)
		return(NULL);

	/* If the Archiver already has a subprocess then stop and
	 * delete it
	 */
	edv_archiver_subprocess_delete(archiver->subprocess);

	/* Create a new subprocess on the Archiver */
	archiver->subprocess = sp = edv_archiver_subprocess_new();
	if(sp == NULL)
		return(NULL);

	sp->freeze_count++;

	sp->op = op;
	sp->archiver = archiver;

	/* Execute the command */
	if(command != NULL)
	{
		EDVCore *core = archiver->core;
		CfgList *cfg_list = core->cfg_list;
		gchar *shell_prog;
		const gchar	*shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL),
	                        *shell_args = edv_strarg(
	                        shell_cmd,
	                        &shell_prog,
	                        TRUE,		/* Parse escapes */
	                        TRUE		/* Parse quotes */
	        );
		sp->pid = edv_system_shell_streams(
			command,
	                shell_prog,
	                shell_args,
			NULL,
			&sp->cstdout,
			&sp->cstderr
	        );
	        g_free(shell_prog);
	}

	/* Set the monitor subprocess timeout callback */
	sp->timeout_id = gtk_timeout_add(
		8l,
		edv_archiver_subprocess_timeout_cb, sp
	);

	/* Create a new GTimer */
	sp->timer = g_timer_new();

	/* Set the Archiver as passive busy and update its display */
	edv_archiver_set_passive_busy(archiver, TRUE);
	edv_archiver_update_display(archiver);

	sp->freeze_count--;

	return(sp);
}

/*
 *	Starts a new list archive subprocess on the Archiver.
 *
 *	If the Archiver already has a subprocess set on it then that
 *	subprocess will be stopped and deleted. The Archiver will be
 *	marked as passive busy and its display will be updated.
 *
 *	The archiver specifies the Archiver.
 *
 *	The arch_path specifies the archive to list.
 *
 *	If filter is not NULL then it specifies the filter.
 *
 *	If password is not NULL then it specifies the password.
 *
 *	Returns the new EDVArchiverSubprocess or NULL on error.
 */
EDVArchiverSubprocess *edv_archiver_subprocess_start_list_archive(
	EDVArchiver *archiver,
	const gchar *arch_path,
	const gchar *filter,
	const gchar *password,
	const gboolean show_progress,
	const gboolean show_comments
)
{
	gchar *cmd;
	EDVArchiverSubprocess *sp;
	EDVCore *core;

	if(archiver == NULL)
		return(NULL);

	core = archiver->core;

	cmd = g_strdup_printf(
"\"%s\" --list-archive \"%s\" \"%s\" \"%s\"",
		core->prog_path,
		arch_path,
		(filter != NULL) ? filter : "",
		(password != NULL) ? password : ""
	);

	sp = edv_archiver_subprocess_start(
		archiver,
		EDV_ARCHIVER_OP_OPEN,		/* List archive*/
		cmd
	);

	g_free(cmd);

	if(sp == NULL)
		return(NULL);

	sp->freeze_count++;

	sp->show_progress = show_progress;
	sp->arch_path = STRDUP(arch_path);
	sp->filter = STRDUP(filter);
	sp->password = STRDUP(password);
	sp->show_comments = show_comments;

	sp->freeze_count--;

	return(sp);
}

/*
 *	Checks if the process is running and gets its pid.
 *
 *	Returns the pid of the process if it is running or 0.
 */
gint edv_archiver_subprocess_get_pid(EDVArchiverSubprocess *sp)
{
        if(sp == NULL)
                return(0);

	if(sp->pid == 0)
		return(0);

	if(edv_pid_exists(sp->pid))
		return(sp->pid);
	else
		return(0);
}

/*
 *	Sends a SIGINT to the subprocess and increments the stop count.
 */
void edv_archiver_subprocess_stop(EDVArchiverSubprocess *sp)
{
	if(sp == NULL)
		return;

	sp->freeze_count++;

	if(sp->pid > 0)
	{
		(void)INTERRUPT(sp->pid);
		sp->pid = 0;
	}

	sp->stop_count++;

	sp->freeze_count--;
}

/*
 *	Gets the stop count.
 */
gint edv_archiver_subprocess_get_stop_count(EDVArchiverSubprocess *sp) 
{
        if(sp == NULL)
                return(0);

	return(sp->stop_count);
}


/*
 *	Subprocess monitor timeout callback.
 */
static gint edv_archiver_subprocess_timeout_cb(gpointer data)
{
	gint status;
	EDVArchiver *archiver;
	EDVArchiverSubprocess *sp = EDV_ARCHIVER_SUBPROCESS(data);
	if(sp == NULL)
		return(FALSE);

	if(sp->freeze_count > 0)
		return(TRUE);

	archiver = sp->archiver;
	if(archiver == NULL)
	{
		/* This should never happen, the subprocess should
		 * always have an Archiver set
		 */
		edv_archiver_subprocess_delete(sp);
		return(FALSE);
	}

	sp->freeze_count++;

	/* Handle the subprocess */
	status = edv_archiver_subprocess_iteration(sp);

	sp->freeze_count--;

	/* Keep calling? */
	if(status > 0)
	{
		/* Reschedual the call */
		sp->timeout_id = gtk_timeout_add(
			(status == 1) ? 250l : 8l,
			edv_archiver_subprocess_timeout_cb, sp
		);
		return(FALSE);
	}
	else
	{
		/* Stop calling
		 *
		 * Remove the subprocess from the archiver and stop it
		 */
		archiver->subprocess = NULL;
		edv_archiver_subprocess_delete(sp);
		return(FALSE);
	}
}


/*
 *	Parses the progress and archive object from the string sent
 *	by edv_archive_list_send_object_data() in module
 *	edv_archive_list.c.
 *
 *	Updates the progress on the subprocess and returns a new
 *	dynamically allocated EDVArchiveObject.
 */
static EDVArchiveObject *edv_archiver_subprocess_list_parse_list_value(
	EDVArchiver *archiver,
	EDVArchiverSubprocess *sp,
	const gchar *s
)
{
	gchar *parameter_value_s;
	EDVArchiveObject *obj;

	if(s == NULL)
		return(NULL);

	obj = edv_archive_object_new();
	if(obj == NULL)
		return(NULL);

	/* Begin parsing
	 *
	 * '\n' or '\0' marks the end of string
	 *
	 * '\t' is a value deliminator
	 *
	 * All values are in ASCII
	 */

/* Seeks s past the next deliminator ('\t') character or returns if an
 * newline ('\n') or end of string ('\0') character is encountered */
#define SEEK_NEXT	{			\
 /* Seek past the next deliminator or to the	\
  * end of string				\
  */						\
 while(*s != '\0') {				\
  /* Premature end? */				\
  if(*s == '\n') {				\
   return(obj);					\
  /* Deliminator? */				\
  } else if(*s == '\t') {			\
   s++;						\
   break;					\
  }						\
  s++;						\
 }						\
						\
 /* Premature end? */				\
 if(*s == '\0')					\
  return(obj);					\
}

/* Gets the string value from s and stores it to _s_ (does not delete
 * _s_ first) */
#define GET_STRING(_s_)	{			\
 const gchar *s_end = (const gchar *)strpbrk(	\
  (const char *)s,				\
  "\t\n"					\
 );						\
 if(s_end != NULL) {				\
  const gint len = s_end - s;			\
  if(len > 0) {					\
   (_s_) = (gchar *)g_malloc((len + 1) * sizeof(gchar)); \
   if((_s_) != NULL) {				\
    (void)memcpy(				\
     (_s_),					\
     s,						\
     (size_t)len				\
    );						\
    (_s_)[len] = '\0';				\
   }						\
  } else {					\
   (_s_) = NULL;				\
  }						\
 } else {					\
  (_s_) = g_strdup(s);				\
 }						\
}

	/* Progress I */
	sp->progress_i = (gulong)atol((const char *)s);
	SEEK_NEXT

	/* Progress N */
	sp->progress_n = (gulong)atol((const char *)s);
	SEEK_NEXT

	/* Type */
	obj->type = (EDVObjectType)atoi((const char *)s);
	SEEK_NEXT

	/* Index */
	obj->index = (gulong)atol((const char *)s);
	SEEK_NEXT

	/* Path */
	GET_STRING(obj->path);
	SEEK_NEXT

	/* Name */
	GET_STRING(obj->name);
	SEEK_NEXT

	/* Size */
	obj->size = (gulong)atol((const char *)s);
	SEEK_NEXT

	/* Storage Size */
	obj->storage_size = (gulong)atol((const char *)s);
	SEEK_NEXT

	/* Link */
	GET_STRING(obj->link_target);
	SEEK_NEXT

	/* Permissions */
	obj->permissions = (EDVPermissionFlags)atoi((const char *)s);
	SEEK_NEXT

	/* Access Time */
	obj->access_time = (gulong)atol((const char *)s);
	SEEK_NEXT

	/* Modify Time */
	obj->modify_time = (gulong)atol((const char *)s);
	SEEK_NEXT

	/* Change Time */
	obj->change_time = (gulong)atol((const char *)s);
	SEEK_NEXT

	/* Owner */
	GET_STRING(obj->owner_name);
	SEEK_NEXT

	/* Group */
	GET_STRING(obj->group_name);
	SEEK_NEXT

	/* Device Type */
	obj->device_type = (gint)atoi((const char *)s);
	SEEK_NEXT

	/* Encryption Name */
	GET_STRING(obj->encryption_name);
	SEEK_NEXT

	/* Compression Ratio */
	obj->compression_ratio = (gfloat)atof((const char *)s);
	SEEK_NEXT

	/* Storage Method Name */
	GET_STRING(obj->storage_method);
	SEEK_NEXT

	/* CRC Checksum */
	GET_STRING(obj->crc);
	SEEK_NEXT

	/* Subsequent values are meta datas list properties where
	 * each property consists of a property name and value, the
	 * property name and value are deliminated by a '=', each
	 * property is still deliminated by a '\t', for example:
	 *
	 *	\tname=value\tname2=value2\n
	 */
	do {
		parameter_value_s = NULL;
		GET_STRING(parameter_value_s);
		if(parameter_value_s != NULL)
		{
			/* Check for and seek to the parameter and value
			 * deliminator ('=')
			 */
			gchar *delim_s = (gchar *)strchr(
				(char *)parameter_value_s,
				'='
			);
			if(delim_s != NULL)
			{
				/* Parse the name */
				const gint len = delim_s - parameter_value_s;
				if(len > 0)
				{
					/* Parse the name */
					gchar *name = (gchar *)g_malloc(
						(len + 1) * sizeof(gchar)
					);
					if(name != NULL)
					{
						(void)memcpy(
							name,
							parameter_value_s,
							(size_t)len
						);
						name[len] = '\0';

						/* Append this property to the
						 * meta data list
						 */
						obj->meta_data_list = edv_properties_list_set_s(
							obj->meta_data_list,
							name,
							delim_s + 1,	/* Value */
							TRUE	/* Create as needed */
						);

						g_free(name);
					}
				}
			}
			g_free(parameter_value_s);
		}
		SEEK_NEXT
	} while(*s != '\0');

#undef GET_STRING
#undef SEEK_NEXT

	return(obj);
}


/*
 *	Handles one iteration of a list archive subprocess.
 *
 *	Reads any archive object values from the standard output stream
 *	and appends them to the Archiver's list, updates the Archiver's
 *	status message as needed, and displays the statistics and
 *	comments as needed.
 */
static gint edv_archiver_subprocess_list_iteration(EDVArchiverSubprocess *sp)
{
	gboolean time_exceeded;
	gfloat time_limit_sec;
	FILE *fp = sp->cstdout;
	gchar **line_buf = &sp->cstdout_buf;
	gint		fd,
			nobjs_listed;
	EDVArchiver *archiver = sp->archiver;
	GtkWidget *toplevel = archiver->toplevel;
	GtkCList *clist = GTK_CLIST(archiver->contents_clist);
	EDVCore *core = archiver->core;
	CfgList *cfg_list = core->cfg_list;

	if(fp == NULL)
		return(0);

	fd = (gint)fileno(fp);

	/* Set the time limit to 250 ms */
	time_limit_sec = 0.250f;

	nobjs_listed = 0;
	time_exceeded = FALSE;

	/* Keep reading archive objects from the stream while there
	 * is data in the stream and the time limit has not been
	 * exceeded
	 */
	while(edv_poll_read(fd))
	{
	        if(edv_stream_read_lineptr(
			fp,
			line_buf,
			FALSE			/* Nonblocking */
		))
		{
			/* Parse the object from the buffer */
			EDVArchiveObject *obj = edv_archiver_subprocess_list_parse_list_value(
				archiver,
				sp,
				*line_buf
			);

			/* Reset the read buffer */
			g_free(*line_buf);
			*line_buf = NULL;

			if(obj != NULL)
			{
				/* If this is the first object being
				 * listed for this iteration then
				 * freeze the GtkCList
				 */
				if(nobjs_listed == 0)
					gtk_clist_freeze(clist);

				/* Append this object to the list */
	                        (void)edv_archiver_list_append(
		                        archiver,
	               	                obj
		                );
/*				obj = NULL; */

				/* Mark that we will need to thaw
				 * the GtkCList
				 */
				nobjs_listed++;
			}
		}
		else
		{
			/* Did not get enough data describing a
			 * complete object from the stream so stop
			 * reading
			 */
			break;
		}

		/* Time limit exceeded? */
		if(g_timer_elapsed(sp->timer, NULL) > time_limit_sec)
		{
			/* Stop and mark that the time limit had been
			 * exceeded with at least one attempted read
			 * iteration
			 */
			time_exceeded = TRUE;
			break;
		}
	}

	/* Need to thaw the GtkCList? */
	if(nobjs_listed > 0)
	{
	        /* Resize the columns */
		if(EDV_GET_B(EDV_CFG_PARM_ARCHIVER_CONTENTS_LIST_AUTO_RESIZE_COLUMNS))
			edv_archiver_list_resize_columns_optimul(archiver);
		gtk_clist_thaw(clist);
	}

	/* Do not continue if the time limit was exceeded */
	if(time_exceeded)
		return((nobjs_listed > 0) ? 2 : 1);

	/* Check for any error messages */
	fp = sp->cstderr;
	fd = (gint)fileno(fp);
	line_buf = &sp->cstderr_buf;
	while(edv_poll_read(fd))
	{
	        if(edv_stream_read_lineptr(
			fp,
			line_buf,
			FALSE			/* Nonblocking */
		))
		{
			/* Store all the error messages and display
			 * them when the process has finished
			 */
		}
		else
		{
			/* Did not get a complete error message from
			 * the stream so stop reading
			 */
			break;
		}
	}

	/* Archive list process finished? */
	if(sp->pid == 0)
	{
		/* Update the status bar message */
		edv_status_bar_message(
			archiver->status_bar,
			(sp->stop_count > 0) ?
"List archive interrupted" :
"Archive contents loaded",
			FALSE
		);

		/* If there was no user abort or error then check if
		 * there are comments to be displayed
		 */
		if((sp->stop_count == 0) && sp->show_comments)
		{
			/* Get the current comments */
			gchar *comments = edv_archive_comment_get(
				cfg_list,
				sp->arch_path
			);
			if(!STRISEMPTY(comments))
			{
				EDVArchiveCommentsDlg *d = archiver->comments_dialog;
				edv_archive_comments_dlg_delete(d);
			        archiver->comments_dialog = d = edv_archive_comments_dlg_new(
					core,
					sp->arch_path,
					sp->password,
					comments,
					toplevel
				);
			        if(d != NULL)
			        {
			                edv_archive_comments_dlg_map(d);
			                edv_archive_comments_dlg_update_display(d);
			        }
			}
			g_free(comments);
		}

		return(0);			/* Stop calling */
	}

	return((nobjs_listed > 0) ? 2 : 1);
}

/*
 *	Handles one iteration of the subprocess based on the
 *	subprocess's operation, updates the subprocess's status on
 *	the subprocess, and updates the Archiver's progress.
 *
 *	The sp specifies the subprocess.
 *
 *	Return values:
 *
 *	0	Stop calling.
 *	1	Call again and nothing was processed.
 *	2	Call again and one or more operations were processed.
 */
gint edv_archiver_subprocess_iteration(EDVArchiverSubprocess *sp)
{
	gint status;
	EDVArchiver *archiver;

	if(sp == NULL)
		return(0);

	sp->freeze_count++;

	/* Restart the timer for this iteration */
	g_timer_start(sp->timer);

	/* If the process was marked as running then check if it is
	 * still running
	 */
	if(sp->pid > 0)
	{
		if(!edv_pid_exists(sp->pid))
			sp->pid = 0;
	}

	/* Progress the iteration based on the operation */
	if(sp->op == EDV_ARCHIVER_OP_OPEN)
	{
		status = edv_archiver_subprocess_list_iteration(sp);
	}
	else
	{
		status = 0;
	}

	archiver = sp->archiver;

	/* If the operation wants to be called again then report the
	 * progress, otherwise report the final progress
	 */
	if(status > 0)
	{
		/* Update the progress */
		if((archiver != NULL) && (sp->show_progress))
			edv_status_bar_progress(
				archiver->status_bar,
				(sp->progress_n > 0l) ?
					((gfloat)sp->progress_i / (gfloat)sp->progress_n) : -1.0f,
				FALSE
			);
	}
	else
	{
		/* Report the final progress */
		if((archiver != NULL) && (sp->show_progress))
			edv_status_bar_progress(
				archiver->status_bar,
				0.0f,
				FALSE
			);
	}

	/* Stop the timer */
	g_timer_stop(sp->timer);

	sp->freeze_count--;

	return(status);
}
