#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <sys/stat.h>
#include <gtk/gtk.h>

#include "cfg.h"

#include "guiutils.h"
#include "tool_bar.h"
#include "cdialog.h"
#include "fb.h"
#include "progressdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "libendeavour2-base/edv_archive_obj.h"
#include "libendeavour2-base/edv_id.h"
#include "edv_archive_obj_stat.h"
#include "edv_obj_info_match.h"
#include "edv_utils_gtk.h"
#include "edv_archive_add.h"
#include "edv_archive_extract.h"
#include "edv_archive_delete.h"
#include "edv_archive_comment.h"
#include "edv_archive_check.h"
#include "edv_archive_fix.h"
#include "edv_find_bar.h"
#include "edv_status_bar.h"
#include "edv_confirm.h"
#include "archive_options_dlg.h"
#include "archive_statistics_dlg.h"
#include "archiver.h"
#include "archiver_cb.h"
#include "archiver_op.h"
#include "archiver_list.h"
#include "archiver_subprocess.h"
#include "edv_help.h"
#include "edv_emit.h"
#include "edv_op.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"


gint edv_archiver_op_idle_cb(gpointer data);

void edv_archiver_op_activate_cb(
	ToolBarItem *item,
	const gint id,
	gpointer data
);
void edv_archiver_op_enter_cb(
	ToolBarItem *item,
	const gint id,
	gpointer data
);
void edv_archiver_op_leave_cb(
	ToolBarItem *item,
	const gint id,
	gpointer data
);

void edv_archiver_op(
	EDVArchiver *archiver,
	const EDVArchiverOpID op
);

static void edv_archiver_op_sync_disks(EDVArchiver *archiver);
static void edv_archiver_op_write_protect(EDVArchiver *archiver);
static void edv_archiver_op_delete_method_recycle(EDVArchiver *archiver);
static void edv_archiver_op_delete_method_purge(EDVArchiver *archiver);

static void edv_archiver_op_new(EDVArchiver *archiver);
static void edv_archiver_op_open(EDVArchiver *archiver);

void edv_archiver_op_close(EDVArchiver *archiver);
static void edv_archiver_op_exit(EDVArchiver *archiver);

static GList *edv_archiver_get_selected_objects(EDVArchiver *archiver);
static void edv_archiver_op_add(EDVArchiver *archiver);
void edv_archiver_op_extract(EDVArchiver *archiver);
static void edv_archiver_op_extract_all(EDVArchiver *archiver);
static void edv_archiver_op_delete(EDVArchiver *archiver);
static void edv_archiver_op_comments(EDVArchiver *archiver);
static void edv_archiver_op_check(EDVArchiver *archiver);
static void edv_archiver_op_fix(EDVArchiver *archiver);
static void edv_archiver_op_select_all(EDVArchiver *archiver);
static void edv_archiver_op_unselect_all(EDVArchiver *archiver);
static void edv_archiver_op_invert_selection(EDVArchiver *archiver);
static void edv_archiver_op_properties(EDVArchiver *archiver);

static void edv_archiver_op_stop(EDVArchiver *archiver);
void edv_archiver_op_refresh(EDVArchiver *archiver);
static void edv_archiver_op_refresh_all(EDVArchiver *archiver);
static void edv_archiver_op_statistics(EDVArchiver *archiver);

static void edv_archiver_op_list_filter(EDVArchiver *archiver);
static void edv_archiver_op_mime_types(EDVArchiver *archiver);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Operation Idle callback.
 *
 *	The data must be a EDVArchiverOpQueue *.
 */
gint edv_archiver_op_idle_cb(gpointer data)
{
	EDVArchiver *archiver;
	EDVArchiverOpQueue *q = EDV_ARCHIVER_OP_QUEUE(data);
	if(q == NULL)
		return(FALSE);

	archiver = q->archiver;
	if(archiver == NULL)
		return(FALSE);

	if(EDV_ARCHIVER_IS_PROCESSING(archiver) || (archiver->freeze_count > 0))
		return(FALSE);

	archiver->freeze_count++;

	/* Remove this operation queue from the list */
	archiver->queued_ops_list = g_list_remove(
		archiver->queued_ops_list,
		q
	);

	/* Perform the operation */
	edv_archiver_op(
		archiver,
		q->op_id
	);

	/* Delete the operation queue */
	g_free(q);

	archiver->freeze_count--;

	return(FALSE);
}

/*
 *	Tool Bar Operation ID activate callback.
 *
 *	The data must be a EDVArchiverOpID *.
 */
void edv_archiver_op_activate_cb(
	ToolBarItem *item,
	const gint id,
	gpointer data
)
{
	EDVArchiver *archiver;
	EDVArchiverOp *op = EDV_ARCHIVER_OP(data);
	if(op == NULL)
		return;

	archiver = op->archiver;
	if((archiver == NULL) || (op->flags & EDV_OPID_NO_OP))
		return;

	if(EDV_ARCHIVER_IS_PROCESSING(archiver) || (archiver->freeze_count > 0))
		return;

	archiver->freeze_count++;

	/* Perform the operation */
	edv_archiver_op(
		archiver,
		op->id
	);

	archiver->freeze_count--;
}

/*
 *	Operation ID enter notify callback nexus.
 *
 *	The data must be a EDVArchiverOpID *.
 */
void edv_archiver_op_enter_cb(
	ToolBarItem *item,
	const gint id,
	gpointer data  
)
{
	const gchar *tooltip;
	EDVArchiverOp *op = EDV_ARCHIVER_OP(data);
	EDVArchiver *archiver = (op != NULL) ? op->archiver : NULL;
	if(archiver == NULL)
		return;

	if(archiver->freeze_count > 0)
		return;

	archiver->freeze_count++;

	tooltip = op->tooltip;
	if(!STRISEMPTY(tooltip))
		edv_status_bar_message(archiver->status_bar, tooltip, FALSE);

	archiver->freeze_count--;
}

/*
 *	Operation ID leave notify callback nexus.
 */
void edv_archiver_op_leave_cb(
	ToolBarItem *item,
	const gint id,
	gpointer data
)
{
	EDVArchiverOp *op = EDV_ARCHIVER_OP(data);
	EDVArchiver *archiver = (op != NULL) ? op->archiver : NULL;
	if(archiver == NULL)
		return;

	if(archiver->freeze_count > 0)
		return;

	archiver->freeze_count++;

	edv_status_bar_message(archiver->status_bar, NULL, FALSE);

	archiver->freeze_count--;
}


/*
 *	Perform operation nexus.
 */
void edv_archiver_op(
	EDVArchiver *archiver,
	const EDVArchiverOpID op
)
{
	GtkWidget *toplevel;
	CfgList *cfg_list;
	EDVCore *core;

	if(archiver == NULL)
		return;

	archiver->freeze_count++;

	toplevel = archiver->toplevel;
	core = archiver->core;
	cfg_list = core->cfg_list;

	switch(op)
	{
	  case EDV_ARCHIVER_OP_NONE:
	  case EDV_ARCHIVER_OP_SEPARATOR:
		break;

	  case EDV_ARCHIVER_OP_NEW:
		edv_archiver_op_new(archiver);
		break;

	  case EDV_ARCHIVER_OP_OPEN:
		edv_archiver_op_open(archiver);
		break;


	  case EDV_ARCHIVER_OP_CLOSE:
		edv_archiver_op_close(archiver);
		break;

	  case EDV_ARCHIVER_OP_EXIT:
		edv_archiver_op_exit(archiver);
		break;


	  case EDV_ARCHIVER_OP_ADD:
		edv_archiver_op_add(archiver);
		break;

	  case EDV_ARCHIVER_OP_EXTRACT:
		edv_archiver_op_extract(archiver);
		break;

	  case EDV_ARCHIVER_OP_EXTRACT_ALL:
		edv_archiver_op_extract_all(archiver);
		break;

	  case EDV_ARCHIVER_OP_DELETE:
		edv_archiver_op_delete(archiver);
		break;

	  case EDV_ARCHIVER_OP_COMMENTS:
		edv_archiver_op_comments(archiver);
		break;

	  case EDV_ARCHIVER_OP_CHECK:
		edv_archiver_op_check(archiver);
		break;

	  case EDV_ARCHIVER_OP_FIX:
		edv_archiver_op_fix(archiver);
		break;

	  case EDV_ARCHIVER_OP_SELECT_ALL:
		edv_archiver_op_select_all(archiver);
		break;

	  case EDV_ARCHIVER_OP_UNSELECT_ALL:
		edv_archiver_op_unselect_all(archiver);
		break;

	  case EDV_ARCHIVER_OP_INVERT_SELECTION:
		edv_archiver_op_invert_selection(archiver);
		break;

	  case EDV_ARCHIVER_OP_FIND:
		edv_map_find_archive(core, archiver);
		break;

	  case EDV_ARCHIVER_OP_PROPERTIES:
		edv_archiver_op_properties(archiver);
		break;


	  case EDV_ARCHIVER_OP_HISTORY:
		edv_map_history(
			core,
			-1,
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL,
			toplevel
		);
		break;

	  case EDV_ARCHIVER_OP_SYNC_DISKS:
		edv_archiver_op_sync_disks(archiver);
		break;

	  case EDV_ARCHIVER_OP_WRITE_PROTECT:
		edv_archiver_op_write_protect(archiver);
		break;

	  case EDV_ARCHIVER_OP_DELETE_METHOD_RECYCLE:
		edv_archiver_op_delete_method_recycle(archiver);
		break;

	  case EDV_ARCHIVER_OP_DELETE_METHOD_PURGE:
		edv_archiver_op_delete_method_purge(archiver);
		break;

	  case EDV_ARCHIVER_OP_RUN:
		edv_map_run_dialog_command(
			core,
			NULL,
			NULL,
			toplevel
		);
		break;

	  case EDV_ARCHIVER_OP_RUN_TERMINAL:
		if(core != NULL)
		{
			gchar	*location = STRDUP(edv_archiver_get_location(archiver)),
				*wd = g_dirname(location);
			edv_run_terminal(
				core,
				NULL,
				wd,
				toplevel
			);
			g_free(wd);
			g_free(location);
		}
		break;


	  case EDV_ARCHIVER_OP_STOP:
		edv_archiver_op_stop(archiver);
		break;

	  case EDV_ARCHIVER_OP_REFRESH:
		edv_archiver_op_refresh(archiver);
		break;

	  case EDV_ARCHIVER_OP_REFRESH_ALL:
		edv_archiver_op_refresh_all(archiver);
		break;

	  case EDV_ARCHIVER_OP_STATISTICS:
		edv_archiver_op_statistics(archiver);
		break;

	  case EDV_ARCHIVER_OP_SHOW_TOOL_BAR:
		if(core != NULL)
		{
			const gboolean state = !EDV_GET_B(
				EDV_CFG_PARM_ARCHIVER_SHOW_TOOL_BAR
			);
			EDV_SET_B(
				EDV_CFG_PARM_ARCHIVER_SHOW_TOOL_BAR,
				state
			);
			edv_queue_emit_reconfigured(core);
		}
		break;

	  case EDV_ARCHIVER_OP_SHOW_LOCATION_BAR:
		if(core != NULL)
		{
			const gboolean state = !EDV_GET_B(
				EDV_CFG_PARM_ARCHIVER_SHOW_LOCATION_BAR
			);
			EDV_SET_B(
				EDV_CFG_PARM_ARCHIVER_SHOW_LOCATION_BAR,
				state
			);
			edv_queue_emit_reconfigured(core);
		}
		break;

	  case EDV_ARCHIVER_OP_SHOW_FIND_BAR:
		if(core != NULL)
		{
			const gboolean state = !EDV_GET_B(
				EDV_CFG_PARM_ARCHIVER_SHOW_FIND_BAR
			);
			EDV_SET_B(
				EDV_CFG_PARM_ARCHIVER_SHOW_FIND_BAR,
				state
			);
			edv_queue_emit_reconfigured(core);
		}
		break;

	  case EDV_ARCHIVER_OP_SHOW_STATUS_BAR:
		if(core != NULL)
		{
			const gboolean state = !EDV_GET_B(
				EDV_CFG_PARM_ARCHIVER_SHOW_STATUS_BAR
			);
			EDV_SET_B(
				EDV_CFG_PARM_ARCHIVER_SHOW_STATUS_BAR,
				state
			);
			edv_queue_emit_reconfigured(core);
		}
		break;


	  case EDV_ARCHIVER_OP_LIST_FILTER:
		edv_archiver_op_list_filter(archiver);
		break;

	  case EDV_ARCHIVER_OP_CONTENTS_LIST_AUTO_RESIZE_COLUMNS:
		if(core != NULL)
		{
			const gboolean state = !EDV_GET_B(
				EDV_CFG_PARM_ARCHIVER_CONTENTS_LIST_AUTO_RESIZE_COLUMNS
			);
			EDV_SET_B(
				EDV_CFG_PARM_ARCHIVER_CONTENTS_LIST_AUTO_RESIZE_COLUMNS,
				state
			);
			edv_queue_emit_reconfigured(core);
		}
		break;

	  case EDV_ARCHIVER_OP_MIME_TYPES:
		edv_archiver_op_mime_types(archiver);
		break;


	  case EDV_ARCHIVER_OP_NEW_BROWSER:
		edv_new_vfs_browser(
			core,
			NULL,
			NULL,
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL
		);
		break;

	  case EDV_ARCHIVER_OP_NEW_IMBR:
		edv_new_image_browser(
			core,
			NULL,
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL
		);
		break;

	  case EDV_ARCHIVER_OP_NEW_ARCHIVER:
		edv_new_archiver(
			core,
			NULL,
			NULL,
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL
		);
		break;

	  case EDV_ARCHIVER_OP_RECYCLE_BIN:
		edv_map_recycle_bin(
			core,
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL
		);
		break;


	  case EDV_ARCHIVER_OP_OPTIONS:
		edv_map_options(
			core,
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL,
			toplevel
		);
		break;

	  case EDV_ARCHIVER_OP_CUSTOMIZE:
		edv_map_customize(
			core,
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL,
			toplevel
		);
		break;


	  case EDV_ARCHIVER_OP_HELP_ABOUT:
		edv_map_about_dialog(
			core,
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL,
			toplevel
		);
		break;
	  case EDV_ARCHIVER_OP_HELP_CONTENTS:
		edv_help(core, "Contents", toplevel);
		break;
	  case EDV_ARCHIVER_OP_HELP_FILE_BROWSER:
		edv_help(core, "File Browser", toplevel);
		break;
	  case EDV_ARCHIVER_OP_HELP_IMAGE_BROWSER:
		edv_help(core, "Image Browser", toplevel);
		break;
	  case EDV_ARCHIVER_OP_HELP_ARCHIVER:
		edv_help(core, "Archiver", toplevel);
		break;
	  case EDV_ARCHIVER_OP_HELP_RECYCLE_BIN:
		edv_help(core, "Recycle Bin", toplevel);
		break;
	  case EDV_ARCHIVER_OP_HELP_KEYS_LIST:
		edv_help(core, "Keys List", toplevel);
		break;
	  case EDV_ARCHIVER_OP_HELP_COMMON_OPERATIONS:
		edv_help(core, "Common Operations", toplevel);
		break;
	}

	archiver->freeze_count--;
}


/*
 *	Sync Disks.
 */
static void edv_archiver_op_sync_disks(EDVArchiver *archiver)
{
	GtkWidget *sb;

	if(archiver == NULL)
		return;

	edv_archiver_set_busy(archiver, TRUE);

	sb = archiver->status_bar;

	edv_status_bar_message(sb, "Syncing disks...", TRUE);

	/* Sync disks */
	edv_sync_edv(archiver->core);

	edv_status_bar_message(sb, "Disk sync done", FALSE);
	edv_status_bar_progress(sb, 0.0f, FALSE);

	edv_archiver_set_busy(archiver, FALSE);
}

/*
 *	Write Protect toggle.
 */
static void edv_archiver_op_write_protect(EDVArchiver *archiver)
{
	gboolean state;
	CfgList *cfg_list;
	EDVCore *core;

	if(archiver == NULL)
		return;

	edv_archiver_set_busy(archiver, TRUE);

	core = archiver->core;
	cfg_list = core->cfg_list;

	/* Get the current Master Write Protect state */
	state = EDV_GET_B(EDV_CFG_PARM_WRITE_PROTECT);

	/* Toggle the Master Write Protect */
	state = !state;

	/* Set the new Master Write Protect state */
	EDV_SET_B(
		EDV_CFG_PARM_WRITE_PROTECT,
		state
	);

	/* Notify about the Master Write Protect state change */
	edv_queue_emit_master_write_protect_changed(core);

	edv_archiver_set_busy(archiver, FALSE);
}

/*
 *	Delete Method: Recycle.
 */
static void edv_archiver_op_delete_method_recycle(EDVArchiver *archiver)
{
	CfgList *cfg_list;
	EDVCore *core;

	if(archiver == NULL)
		return;

	edv_archiver_set_busy(archiver, TRUE);

	core = archiver->core;
	cfg_list = core->cfg_list;

	EDV_SET_I(
		EDV_CFG_PARM_DELETE_METHOD,
		EDV_DELETE_METHOD_RECYCLE
	);
	edv_queue_emit_delete_method_changed(core);

	edv_archiver_set_busy(archiver, FALSE);
}

/*
 *	Delete Method: Purge.
 */                          
static void edv_archiver_op_delete_method_purge(EDVArchiver *archiver)
{
	CfgList *cfg_list;
	EDVCore *core;

	if(archiver == NULL)
		return;

	edv_archiver_set_busy(archiver, TRUE);

	core = archiver->core;
	cfg_list = core->cfg_list;

	EDV_SET_I(
		EDV_CFG_PARM_DELETE_METHOD,
		EDV_DELETE_METHOD_PURGE
	);
	edv_queue_emit_delete_method_changed(core);

	edv_archiver_set_busy(archiver, FALSE);
}


/*
 *	New Archive callback.
 */
static void edv_archiver_op_new(EDVArchiver *archiver)
{
	const gchar *ext_list[] = EDV_ARCHIVER_ARCHIVE_FILE_EXTENSIONS;
	gboolean response;
	gint		i,
			nftypes = 0,
			npaths = 0;
	const gchar *arch_path;
	gchar		*parent_path,
			**paths_list = NULL;
	GtkWidget *toplevel;
	fb_type_struct	**ftypes_list = NULL,
			*ftype_rtn = NULL;
	EDVCore *core;

	if((archiver == NULL) || FileBrowserIsQuery())
		return;

	edv_archiver_set_busy(archiver, TRUE);

	toplevel = archiver->toplevel;
	core = archiver->core;

	/* Check and warn if write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
	{
		edv_archiver_set_busy(archiver, FALSE);
		return;
	}

	/* Create the file types list */
	for(i = 0; ext_list[i] != NULL; i += 2)
		FileBrowserTypeListNew(
			&ftypes_list, &nftypes,
			ext_list[i], ext_list[i + 1]
		);
	FileBrowserTypeListNew(
		&ftypes_list, &nftypes,
		"*.*", "All Files"
	);

	/* Get the current archive's parent path */
	arch_path = edv_archiver_get_location(archiver);
	if(STRISEMPTY(arch_path))
	{
		parent_path = NULL;
	}
	else
	{
		parent_path = g_dirname(arch_path);
		if(!edv_path_is_directory(parent_path))
		{
			g_free(parent_path);
			parent_path = NULL;
		}
	}

	/* Query the user for the name of the new archive */
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
		"Create Archive",
		"Create", "Cancel",
		parent_path,
		ftypes_list, nftypes,
		&paths_list, &npaths,
		&ftype_rtn
	);
	g_free(parent_path);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(response)
	{
		gchar *new_path = (npaths > 0) ?
			STRDUP(paths_list[npaths - 1]) : NULL;
		if(new_path != NULL)
		{
			/* Append the extension as needed */
			gchar *s = FileBrowserTypeCompleteExtension(
				new_path,
				ftype_rtn,
				FALSE		/* Do not replace the
						 * existing extension */
			);
			if(s != NULL)
			{
				g_free(new_path);
				new_path = s;
			}

			/* Check if the archive is of a supported
			 * format, query the user to remove the archive
			 * if the archive already exists, create the new
			 * archive, set the Archiver's title, location,
			 * and password, clear the Contents GtkCList,
			 * and get the list of archive objects in the
			 * archive
			 */
			(void)edv_archiver_new_archive(
				archiver,
				new_path,
				edv_archiver_get_password(archiver),
				TRUE		/* List passively */
			);

			g_free(new_path);
		}
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);

	/* Reset the file selector due to changes */
	FileBrowserReset();

	edv_archiver_set_busy(archiver, FALSE);
}

/*
 *	Open archive callback.
 */
static void edv_archiver_op_open(EDVArchiver *archiver)
{
	const gchar *ext_list[] = EDV_ARCHIVER_ARCHIVE_FILE_EXTENSIONS;
	gboolean response;
	gint		nftypes = 0,
			npaths = 0;
	const gchar *arch_path;
	gchar		*parent_path,
			**paths_list = NULL;
	GtkWidget *toplevel;
	fb_type_struct	**ftypes_list = NULL,
			*ftype_rtn = NULL;
	EDVCore *core;

	if((archiver == NULL) || FileBrowserIsQuery())
		return;

	edv_archiver_set_busy(archiver, TRUE);

	toplevel = archiver->toplevel;
	core = archiver->core;

	/* Create the file types list */
	if(ext_list != NULL)
	{
		gint i;
		gchar	*s = g_strdup(""),
			*s2;
		for(i = 0; ext_list[i] != NULL; i += 2)
		{
			s2 = g_strconcat(
				s,
				ext_list[i],
				NULL
			);
			if(s2 != NULL)
			{
				g_free(s);
				s = s2;
			}

			if(ext_list[i + 2] != NULL)
			{
				s2 = g_strconcat(
					s,
					" ",
					NULL
				);
				if(s2 != NULL)
				{
					g_free(s);
					s = s2;
				}
			}
		}
		FileBrowserTypeListNew(
			&ftypes_list, &nftypes,
			s, "All Archives"
		);
		g_free(s);

		for(i = 0; ext_list[i] != NULL; i += 2)
			FileBrowserTypeListNew(
				&ftypes_list, &nftypes,
				ext_list[i], ext_list[i + 1]
			);
	}
	FileBrowserTypeListNew(
		&ftypes_list, &nftypes,
		"*.*", "All Files"
	);

	/* Get the current archive's parent path */
	arch_path = edv_archiver_get_location(archiver);
	if(STRISEMPTY(arch_path))
	{
		parent_path = NULL;
	}
	else
	{
		parent_path = g_dirname(arch_path);
		if(!edv_path_is_directory(parent_path))
		{
			g_free(parent_path);
			parent_path = NULL;
		}
	}

	/* Query user for the archive to open */
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
		"Open Archive",
		"Open", "Cancel",
		parent_path,
		ftypes_list, nftypes,
		&paths_list, &npaths,
		&ftype_rtn
	);
	g_free(parent_path);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(response)
	{
		gchar *new_path = (npaths > 0) ?
			STRDUP(paths_list[npaths - 1]) : NULL;
		if(new_path != NULL)
		{
			/* Check if the archive is of a supported
			 * format, query the user to create a new
			 * archive if the archive does not exist,
			 * sets the Archiver's title, location, and
			 * password, clears the Contents GtkCList, and
			 * gets the list of archive objects in the
			 * archive
			 */
			(void)edv_archiver_open_archive(
				archiver,
				new_path,
				edv_archiver_get_password(archiver),
				TRUE		/* List passively */
			);
			g_free(new_path);
		}
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);

	edv_archiver_set_busy(archiver, FALSE);
}


/*
 *	Close.
 */
void edv_archiver_op_close(EDVArchiver *archiver)
{
	if(archiver == NULL)
		return;

	/* Check if there is a subprocess */
	if(archiver->subprocess != NULL)
	{
		/* Kill the subprocess, remove the subprocess
		 * monitoring timeout callback, set the Archiver as
		 * not passive busy, update the archiver's display,
		 * and delete the subprocess data
		 */
		edv_archiver_subprocess_delete(archiver->subprocess);
		archiver->subprocess = NULL;
	}

	edv_archiver_sync_configuration(archiver);
	edv_archiver_unmap(archiver);
}

/*
 *	Close All Windows.
 */
static void edv_archiver_op_exit(EDVArchiver *archiver)
{
	EDVCore *core;

	if(archiver == NULL)
		return;

	core = archiver->core;

	edv_archiver_sync_configuration(archiver);
	edv_archiver_unmap(archiver);

	/* Schedual a new pending operation on the core to close all
	 * the windows
	 */
	core->pending_flags |= EDV_CORE_PENDING_CLOSE_ALL_WINDOWS;
}


/*
 *	Returns a list of selected objects in the archive.
 *
 *	Returns the list of selected objects, only the returned list
 *	must be deleted and not each object.
 */
static GList *edv_archiver_get_selected_objects(EDVArchiver *archiver)
{
	GList		*glist,
			*obj_list = NULL;
	GtkCList *clist = GTK_CLIST(archiver->contents_clist);
	EDVArchiveObject *obj;

	for(glist = clist->selection;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		obj = EDV_ARCHIVE_OBJECT(gtk_clist_get_row_data(
			clist,
			(gint)glist->data
		));
		if(obj == NULL)
			continue;

		obj_list = g_list_append(
			obj_list,
			obj
		);
	}

	return(obj_list);
}

/*
 *	Add Objects To Archive.
 */
static void edv_archiver_op_add(EDVArchiver *archiver)
{
	gboolean	response,
			archive_existed = FALSE,
			yes_to_all = FALSE,
			recurse = TRUE,
			dereference_links = FALSE;
	gint		status,
			nobjs,
			nobjects_added = 0,
			nftypes = 0,
			npaths = 0,
			compression = 50;	/* 0 to 100 */
	gchar		*msg,
			*arch_path = NULL,
			*password = NULL,
			*parent_path = NULL,
			**paths_list = NULL;
	const gchar *error_msg;
	GList		*src_paths_list = NULL,
			*new_paths_list = NULL;
	GtkWidget *toplevel;
	fb_type_struct	**ftypes_list = NULL,
			*ftype_rtn = NULL;
	EDVVFSObject *obj;
	EDVCore *core;

	if((archiver == NULL) || FileBrowserIsQuery())
		return;

	edv_archiver_set_busy(archiver, TRUE);

	toplevel = archiver->toplevel;
	core = archiver->core;

	/* Check and warn if write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
	{
		edv_archiver_set_busy(archiver, FALSE);
		return;
	}

#define CLEANUP_RETURN	{				\
 if(new_paths_list != NULL) {				\
  g_list_foreach(					\
   new_paths_list, (GFunc)g_free, NULL			\
  );							\
  g_list_free(new_paths_list);				\
 }							\
							\
 if(src_paths_list != NULL) {				\
  g_list_foreach(					\
   src_paths_list, (GFunc)g_free, NULL			\
  );							\
  g_list_free(src_paths_list);				\
 }							\
							\
 g_free(arch_path);					\
 g_free(password);					\
 g_free(parent_path);					\
							\
 /* Delete the file types list */			\
 FileBrowserDeleteTypeList(ftypes_list, nftypes);	\
							\
 return;						\
}

	/* Get the current archive path */
	arch_path = STRDUP(edv_archiver_get_location(archiver));
	if(STRISEMPTY(arch_path))
	{
		/* No current archive specified */
		edv_play_sound_warning(core);
		edv_message_warning(
			"Add Object Failed",
"The current archive location was not specified.\n\
\n\
You must either go to File->New to create a new\n\
archive or go to File->Open to open an existing\n\
archive in order to add objects to it.",
			NULL,
			toplevel
		);
		edv_archiver_set_busy(archiver, FALSE);
		CLEANUP_RETURN;
	}

	/* Create the file types list */
	FileBrowserTypeListNew(
		&ftypes_list, &nftypes,
		"*.*", "All Files"
	);

	/* Get the current archive's parent path */
	parent_path = g_dirname(arch_path);
	if(!edv_path_is_directory(parent_path))
	{
		g_free(parent_path);
		parent_path = NULL;
	}

	/* Query the user for the objects to add */
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
		"Add Objects To Archive",
		"Add", "Cancel",
		parent_path,
		ftypes_list, nftypes,
		&paths_list, &npaths,
		&ftype_rtn
	);
	g_free(parent_path);
	parent_path = NULL;
	FileBrowserSetTransientFor(NULL);

	/* User canceled? */
	if(!response)
	{
		edv_archiver_set_busy(archiver, FALSE);
		CLEANUP_RETURN;
	}

	/* Get the list of objects to add */
	if(npaths > 0)
	{
		gint i;
		const gchar *s;
		gchar *path;

		for(i = 0; i < npaths; i++)
		{
			s = paths_list[i]; 
			if(STRISEMPTY(s))
				continue;

			path = g_strdup(s);
			if(path == NULL)
				continue;

			/* Path from file browser may have tailing
			 * deliminators, get rid of them
			 */
			edv_path_simplify(path);

			src_paths_list = g_list_append(
				src_paths_list,
				path
			);
		}
	}
	/* No objects to add? */
	if(src_paths_list == NULL)
	{
		edv_archiver_set_busy(archiver, FALSE);
		CLEANUP_RETURN;
	}

	nobjs = g_list_length(src_paths_list);

	/* Query the user for the add to archive options */
	password = STRDUP(edv_archiver_get_password(archiver));
	if(!edv_archive_options_dlg_qiery_add(
		core,
		toplevel,
		arch_path,
		&password,
		&recurse,
		&compression,
		&dereference_links
	))
	{
		/* User canceled */
		edv_archiver_set_busy(archiver, FALSE);
		CLEANUP_RETURN;
	}

	/* Record if the archive existed locally */
	archive_existed = edv_path_lexists(arch_path);

	/* Add the selected object(s) to the archive */
	status = edv_archive_add(
		core,
		arch_path,
		src_paths_list,			/* List of objects to be
						 * added to the archive */
		&new_paths_list,		/* Return list of objects
						 * added to the archive */
		password,
		toplevel,
		TRUE,				/* Show progress */
		TRUE,				/* Interactive */
		&yes_to_all,
		recurse,
		compression,
		dereference_links
	);

	/* Unmap the progress dialog which may have been mapped in
	 * the above operation
	 */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Check for errors */
	error_msg = edv_archive_add_get_error(core);
	if(!STRISEMPTY(error_msg))
	{
		/* Report the error */
		edv_play_sound_error(core);
		edv_message_error(
			"Add Object Error",
			error_msg,
			NULL,
			toplevel
		);
	}

	/* Count the number of objects added */
	nobjects_added = g_list_length(new_paths_list);

	/* Report that the archive has been modified, this will cause
	 * the Archiver to reload the archive listing
	 */
	obj = edv_vfs_object_lstat(arch_path);
	if(obj != NULL)
	{
		if(archive_existed)
			edv_emit_vfs_object_modified(
				core,
				arch_path,
				arch_path,
				obj
			);
		else
			edv_emit_vfs_object_added(
				core,
				arch_path,
				obj
			);
		edv_vfs_object_delete(obj);
	}

	/* Update the status bar */
	if(status == -4)
		msg = g_strdup(
"Add operation canceled"
		);
	else if(nobjects_added > 0)
		msg = g_strdup_printf(
"Added %i %s",
			nobjects_added,
			(nobjects_added == 1) ? "object" : "objects"
		);
	else
		msg = g_strdup_printf(
"Unable to add %s",
			(nobjs == 1) ? "object" : "objects"
		);
	edv_status_bar_message(archiver->status_bar, msg, FALSE);
	g_free(msg);

	/* Play the completed sound on success */
	if(status == 0)
		edv_play_sound_completed(core);

	/* Reset the file selector due to changes */
	FileBrowserReset();

	edv_archiver_set_busy(archiver, FALSE);

	CLEANUP_RETURN;
#undef CLEANUP_RETURN
}

/*
 *	Extract Objects From Archive.
 */
void edv_archiver_op_extract(EDVArchiver *archiver)
{
	gboolean	response,
			yes_to_all = FALSE,
			preserve_directories = TRUE,
			preserve_timestamps = TRUE;
	gint		status,
			nftypes = 0,
			npaths = 0,
			nobjs,
			nobjects_extracted = 0;
	gchar		*msg,
			*parent_path,
			*arch_path = NULL,
			*password = NULL,
			*dest_path = NULL,
			**paths_list = NULL;
	const gchar *error_msg;
	GList		*new_paths_list,
			*obj_list = NULL;
	GtkWidget *toplevel;
	fb_type_struct	**ftypes_list = NULL,
			*ftype_rtn = NULL;
	EDVCore *core;

	if((archiver == NULL) || FileBrowserIsQuery())
		return;

	edv_archiver_set_busy(archiver, TRUE);

	toplevel = archiver->toplevel;
	core = archiver->core;

	/* Check and warn if write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
	{
		edv_archiver_set_busy(archiver, FALSE);
		return;
	}

#define CLEANUP_RETURN	{				\
 g_list_free(obj_list);					\
 g_free(arch_path);					\
 g_free(password);					\
 g_free(dest_path);					\
							\
 /* Delete the file types list */			\
 FileBrowserDeleteTypeList(ftypes_list, nftypes);	\
							\
 return;						\
}

	/* Get the current archive path */
	arch_path = STRDUP(edv_archiver_get_location(archiver));
	if(STRISEMPTY(arch_path))
	{
		/* No current archive specified */
		edv_play_sound_warning(core);
		edv_message_warning(
			"Extract Object Failed",
"The current archive location was not specified.\n\
\n\
You must go to File->Open to open an existing\n\
archive in order to extract objects from it.",
			NULL,
			toplevel
		);
		edv_archiver_set_busy(archiver, FALSE);
		CLEANUP_RETURN;
	}

	/* Get the list of selected objects */
	obj_list = edv_archiver_get_selected_objects(archiver);
	if(obj_list == NULL)
	{
		edv_archiver_set_busy(archiver, FALSE);
		CLEANUP_RETURN;
	}

	nobjs = g_list_length(obj_list);

	/* Create the file types list */
	FileBrowserTypeListNew(
		&ftypes_list, &nftypes,
		"*.*", "All Files"
	);

	/* Get the current archive's parent path */
	parent_path = g_dirname(arch_path);
	if(!edv_path_is_directory(parent_path))
	{
		g_free(parent_path);
		parent_path = NULL;
	}

	/* Query user for extract destination */
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
		"Select Extract Destination",
		"Select", "Cancel",
		parent_path,
		ftypes_list, nftypes,
		&paths_list, &npaths,
		&ftype_rtn
	);
	g_free(parent_path);
	FileBrowserSetTransientFor(NULL);

	/* User canceled? */
	if(!response)
	{
		edv_archiver_set_busy(archiver, FALSE);
		CLEANUP_RETURN;
	}

	/* Get the extract location */
	if(npaths > 0)
	{
		const gchar *path = paths_list[npaths - 1];
		if(!STRISEMPTY(path))
		{
			dest_path = g_strdup(path);

			/* Path from file browser may have tailing
			 * deliminators, so we need to get rid of them
			 */
			edv_path_simplify(dest_path);
		}
	}
	/* Invalid extract destination? */
	if(dest_path == NULL)
	{
		edv_play_sound_warning(core);
		edv_message_warning(
"Invalid Extract Destination",
"The selected extract destination directory is invalid",
			NULL,
			toplevel
		);
		edv_archiver_set_busy(archiver, FALSE);
		CLEANUP_RETURN;
	}

	/* Query the user for extract from archive options */
	password = STRDUP(edv_archiver_get_password(archiver));
	if(!edv_archive_options_dlg_query_extract(
		core,
		toplevel,
		arch_path,
		&password,
		&preserve_directories,
		&preserve_timestamps
	))
	{
		/* User canceled */
		edv_archiver_set_busy(archiver, FALSE);
		CLEANUP_RETURN;
	}

	/* Extract the objects from the archive */
	new_paths_list = NULL;
	status = edv_archive_extract(
		core,
		arch_path,
		obj_list,
		FALSE,				/* Not extract all */
		dest_path,
		&new_paths_list,
		password,
		toplevel,
		TRUE,				/* Show progress */
		TRUE,				/* Interactive */
		&yes_to_all,
		preserve_directories,
		preserve_timestamps
	);

	/* Unmap the progress dialog which may have been mapped in
	 * the above operation
	 */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Check for errors */
	error_msg = edv_archive_extract_get_error(core);
	if(!STRISEMPTY(error_msg))
	{
		/* Report the error */
		edv_play_sound_error(core);
		edv_message_error(
			"Extract Object Error",
			error_msg,
			NULL,
			toplevel
		);
	}

	/* Notify about the extracted objects */
	if(new_paths_list != NULL)
	{
		const gchar *path;
		GList *glist;
		EDVVFSObject *obj;

		for(glist = new_paths_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			path = (gchar *)glist->data;
			if(path == NULL)
				continue;

			obj = edv_vfs_object_lstat(path);
			if(obj != NULL)
			{
				edv_emit_vfs_object_added(
					core,
					path,
					obj
				);
				edv_vfs_object_delete(obj);
			}

			nobjects_extracted++;
		}

		/* Delete the list of extracted object paths */
		g_list_foreach(new_paths_list, (GFunc)g_free, NULL);
		g_list_free(new_paths_list);
	}

	/* Update the status bar */
	if(status == -4)
		msg = g_strdup(
			"Extract operation canceled"
		);
	else if(nobjects_extracted > 0)
		msg = g_strdup_printf(
			"Extracted %i %s",
			nobjects_extracted,
			(nobjects_extracted == 1) ? "object" : "objects"
		);
	else
		msg = g_strdup_printf(
			"Unable to extract %s",
			(nobjs == 1) ? "object" : "objects"
		);
	edv_status_bar_message(archiver->status_bar, msg, FALSE);
	g_free(msg);

	/* Play the completed sound on success */
	if(status == 0)                      
		edv_play_sound_completed(core);

	/* Reset the file selector due to changes */
	FileBrowserReset();

	edv_archiver_set_busy(archiver, FALSE);

	CLEANUP_RETURN;
#undef CLEANUP_RETURN
}

/*
 *	Extract All Objects From Archive.
 */
static void edv_archiver_op_extract_all(EDVArchiver *archiver)
{
	gboolean	response,
			yes_to_all = FALSE,
			preserve_directories = TRUE,
			preserve_timestamps = TRUE;
	gint		status,
			nftypes = 0,
			npaths = 0,
			nobjs,
			nobjects_extracted = 0;
	gchar		*msg,
			*parent_path,
			*arch_path = NULL,
			*password = NULL,
			*dest_path = NULL,
			**paths_list = NULL;
	const gchar *error_msg;
	GList		*obj_list = NULL,
			*new_paths_list = NULL;
	GtkWidget *toplevel;
	GtkCList *clist;
	fb_type_struct	**ftypes_list = NULL,
			*ftype_rtn = NULL;
	EDVCore *core;

	if((archiver == NULL) || FileBrowserIsQuery())
		return;

	edv_archiver_set_busy(archiver, TRUE);

	toplevel = archiver->toplevel;
	clist = GTK_CLIST(archiver->contents_clist);
	core = archiver->core;

	/* Check and warn if write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
	{
		edv_archiver_set_busy(archiver, FALSE);
		return;
	}

#define CLEANUP_RETURN	{				\
 g_list_free(obj_list);					\
 g_free(arch_path);					\
 g_free(password);					\
 g_free(dest_path);					\
							\
 /* Delete the file types list */			\
 FileBrowserDeleteTypeList(ftypes_list, nftypes);	\
							\
 return;						\
}

	/* Get the current archive path */
	arch_path = STRDUP(edv_archiver_get_location(archiver));
	if(STRISEMPTY(arch_path))
	{
		/* No current archive specified */
		edv_play_sound_warning(core);
		edv_message_warning(
			"Extract Object Failed",
"The current archive location was not specified.\n\
\n\
You must go to File->Open to open an existing\n\
archive in order to extract objects from it.",
			NULL,
			toplevel
		);
		edv_archiver_set_busy(archiver, FALSE);
		CLEANUP_RETURN;
	}

	/* Get the list of all the objects in the archive */
	if(clist->rows > 0)
	{
		const gint nrows = clist->rows;
		gint i;
		EDVArchiveObject *obj;

		for(i = 0; i < nrows; i++)
		{
			obj = EDV_ARCHIVE_OBJECT(gtk_clist_get_row_data(
				clist,
				i
			));
			if(obj == NULL)
				continue;

			obj_list = g_list_append(
				obj_list,
				obj
			);
		}
	}
	if(obj_list == NULL)
	{
		edv_archiver_set_busy(archiver, FALSE);
		CLEANUP_RETURN;
	}

	nobjs = g_list_length(obj_list);

	/* Create the file types list */
	FileBrowserTypeListNew(
		&ftypes_list, &nftypes,
		"*.*", "All Files"
	);

	/* Get the current archive's parent path */
	parent_path = g_dirname(arch_path);
	if(!edv_path_is_directory(parent_path))
	{
		g_free(parent_path);
		parent_path = NULL;
	}

	/* Query user for extract destination */
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
		"Select Extract Destination",
		"Select", "Cancel",
		parent_path,
		ftypes_list, nftypes,
		&paths_list, &npaths,
		&ftype_rtn
	);
	g_free(parent_path);
	FileBrowserSetTransientFor(NULL);

	/* User canceled? */
	if(!response)
	{
		edv_archiver_set_busy(archiver, FALSE);
		CLEANUP_RETURN;
	}

	/* Get the destination location */
	if(npaths > 0)
	{
		const gchar *path = paths_list[0];
		if(!STRISEMPTY(path))
		{
			dest_path = g_strdup(path);

			/* Path from file browser may have tailing
			 * deliminators, so we need to get rid of them
			 */
			edv_path_simplify(dest_path);
		}
	}
	/* Invalid extract destination? */
	if(dest_path == NULL)
	{
		edv_play_sound_warning(core);
		edv_message_warning(
			"Invalid Extract Destination",
"The selected extract destination directory is invalid",
			NULL,
			toplevel
		);
		edv_archiver_set_busy(archiver, FALSE);
		CLEANUP_RETURN;
	}

	/* Query user for extract from archive options */
	password = STRDUP(edv_archiver_get_password(archiver));
	if(!edv_archive_options_dlg_query_extract(
		core,
		toplevel,
		arch_path,
		&password,
		&preserve_directories,
		&preserve_timestamps
	))
	{
		/* User canceled */
		edv_archiver_set_busy(archiver, FALSE);
		CLEANUP_RETURN;
	}

	/* Extract all the objects from the archive */
	status = edv_archive_extract(
		core,
		arch_path,
		obj_list,
		TRUE,				/* Extract all */
		dest_path,
		&new_paths_list,
		password,
		toplevel,
		TRUE,				/* Show progress */
		TRUE,				/* Interactive */
		&yes_to_all,
		preserve_directories,
		preserve_timestamps
	);

	/* Unmap the progress dialog which may have been mapped in
	 * the above operation
	 */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Check for errors */
	error_msg = edv_archive_extract_get_error(core);
	if(!STRISEMPTY(error_msg))
	{
		/* Report the error */
		edv_play_sound_error(core);
		edv_message_error(
			"Extract Object Error",
			error_msg,
			NULL,
			toplevel
		);
	}

	/* Notify about the extracted objects */
	if(new_paths_list != NULL)
	{
		const gchar *path;
		GList *glist;
		EDVVFSObject *obj;

		for(glist = new_paths_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			path = (const gchar *)glist->data;
			if(path == NULL)
				continue;

			obj = edv_vfs_object_lstat(path);
			if(obj != NULL)
			{
				edv_emit_vfs_object_added(
					core,
					path,
					obj
				);
				edv_vfs_object_delete(obj);
			}

			nobjects_extracted++;
		}

		/* Delete the list of extracted object paths */
		g_list_foreach(new_paths_list, (GFunc)g_free, NULL);
		g_list_free(new_paths_list);
	}

	/* Update the status bar */
	if(status == -4)
		msg = g_strdup(
			"Extract operation canceled"
		);
	else if(nobjects_extracted > 0)
		msg = g_strdup_printf(
			"Extracted %i %s",
			nobjects_extracted,
			(nobjects_extracted == 1) ? "object" : "objects"
		);
	else
		msg = g_strdup_printf(
			"Unable to extract %s",
			(nobjs == 1) ? "object" : "objects"
		);
	edv_status_bar_message(archiver->status_bar, msg, FALSE);
	g_free(msg);

	/* Play the completed sound on success */
	if(status == 0)
		edv_play_sound_completed(core);

	/* Reset the file selector due to changes */
	FileBrowserReset();

	edv_archiver_set_busy(archiver, FALSE);

	CLEANUP_RETURN;
#undef CLEANUP_RETURN
}

/*
 *	Delete Object From Archive.
 */
static void edv_archiver_op_delete(EDVArchiver *archiver)
{
	gboolean yes_to_all = FALSE;
	gint		status,
			nobjs,
			nobjects_deleted = 0;
	gchar		*msg,
			*arch_path = NULL;
	const gchar	*src_path,
			*error_msg;
	GList *obj_list = NULL;
	GtkWidget *toplevel;
	EDVCore *core;

	if(archiver == NULL)
		return;

	edv_archiver_set_busy(archiver, TRUE);

	toplevel = archiver->toplevel;
	core = archiver->core;

	/* Check and warn if write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
	{
		edv_archiver_set_busy(archiver, FALSE);
		return;
	}

#define CLEANUP_RETURN	{			\
 g_list_free(obj_list);				\
 g_free(arch_path);				\
						\
 return;					\
}

	/* Get the current archive's path */
	arch_path = STRDUP(edv_archiver_get_location(archiver));
	if(arch_path == NULL)
	{
		edv_archiver_set_busy(archiver, FALSE);
		CLEANUP_RETURN;
	}

	/* Get the list of selected objects */
	obj_list = edv_archiver_get_selected_objects(archiver);
	if(obj_list == NULL)
	{
		edv_archiver_set_busy(archiver, FALSE);
		CLEANUP_RETURN;
	}

	/* Get the path of the first object only if it is the only
	 * object in the list
	 */
	nobjs = g_list_length(obj_list);
	if(nobjs == 1)
	{
		EDVArchiveObject *obj = EDV_ARCHIVE_OBJECT(obj_list->data);
		src_path = (obj != NULL) ? obj->path : NULL;
	}
	else
		src_path = NULL;

	/* Confirm delete archive objects */
	status = edv_confirm_archive_delete(
		core,
		toplevel,
		src_path,
		nobjs
	);
	switch(status)
	{
	  case CDIALOG_RESPONSE_YES_TO_ALL:
		yes_to_all = TRUE;
	  case CDIALOG_RESPONSE_YES:
		break;
	  default:
		edv_archiver_set_busy(archiver, FALSE);
		CLEANUP_RETURN;
		break;
	}

	/* Iterate through selected objects in the archive */
	status = edv_archive_delete(
		core,
		arch_path,			/* Archive */
		obj_list,			/* List of objects to
						 * delete from the archive */
		edv_archiver_get_password(archiver),
		toplevel,
		TRUE,				/* Show progress */
		TRUE,				/* Interactive */
		&yes_to_all
	);

	/* Unmap the progress dialog which may have been mapped in
	 * the above operation
	 */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Check for errors */
	error_msg = edv_archive_delete_get_error(core);
	if(!STRISEMPTY(error_msg))
	{
		/* Report the error */
		edv_play_sound_error(core);
		edv_message_error(
			"Delete Object From Archive Error",
			error_msg,
			NULL,
			toplevel
		);
	}

	/* Was the archive object successfully delected? */
	if(status == 0)
	{
		/* Report object in archive being removed */
		GList *glist;
		EDVArchiveObject *obj;
		for(glist = obj_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			obj = EDV_ARCHIVE_OBJECT(glist->data);
			if(obj == NULL)
				continue;

#if 0
/* Note that this may not be good since some archive formats have 
 * archive programs that do not report which objects were deleted
 * so it may be more accurate to skip this and just have the
 * edv_emit_vfs_object_modified() below cause this archiver to reload
 */
			edv_emit_archive_object_removed(
				core,
				arch_path,
				obj->path
			);
#endif
			nobjects_deleted++;
		}
	}

	/* Report that the archive has been modified or removed,
	 * this will cause the archiver to reload the archive
	 * listing
	 */
	if(arch_path != NULL)
	{
		EDVVFSObject *obj = edv_vfs_object_lstat(arch_path);
		if(obj != NULL)
		{
			edv_emit_vfs_object_modified(
				core,
				arch_path,
				arch_path,
				obj
			);
			edv_vfs_object_delete(obj);
		}
		else
		{
			const gint error_code = (gint)errno;
#ifdef ENOENT
			if(error_code == ENOENT)
				edv_emit_vfs_object_removed(
					core,
					arch_path
				);
#endif
		}
	}

	/* Update the status bar */
	if(status == -4)
		msg = g_strdup(
"Delete operation canceled"
		);
	else if(nobjects_deleted > 0)
		msg = g_strdup_printf(
"Deleted %i %s",
			nobjects_deleted,
			(nobjects_deleted == 1) ? "object" : "objects"
		);
	else
		msg = g_strdup_printf(
"Unable to delete %s",
			(nobjs == 1) ? "object" : "objects"
		);
	edv_status_bar_message(archiver->status_bar, msg, FALSE);
	g_free(msg);

	/* Play the completed sound on success */
	if(status == 0)
		edv_play_sound_completed(core);
	
	edv_archiver_set_busy(archiver, FALSE);

	CLEANUP_RETURN;
#undef CLEANUP_RETURN
}

/*
 *	Comments.
 */
static void edv_archiver_op_comments(EDVArchiver *archiver)
{
	gchar		*arch_path,
			*comments;
	GtkCList *clist;
	CfgList *cfg_list;
	EDVArchiveCommentsDlg *d;
	EDVCore *core;

	if(archiver == NULL)
		return;

	edv_archiver_set_busy(archiver, TRUE);

	clist = GTK_CLIST(archiver->contents_clist);
	core = archiver->core;
	cfg_list = core->cfg_list;

	arch_path = STRDUP(edv_archiver_get_location(archiver));
	if(arch_path == NULL)
	{
		edv_archiver_set_busy(archiver, FALSE);
		return;
	}

	/* Delete the previous EDVArchiveCommentsDlg */
	d = archiver->comments_dialog;
	if(d != NULL)
	{
		edv_archive_comments_dlg_delete(d);
		archiver->comments_dialog = d = NULL;
	}

	/* Get the current comments */
	comments = edv_archive_comment_get(
		cfg_list,
		arch_path
	);

	/* Create a new EDVArchiveCommentsDlg to view/edit the
	 * comments
	 */
	archiver->comments_dialog = d = edv_archive_comments_dlg_new(
		archiver->core,
		arch_path,
		edv_archiver_get_password(archiver),
		comments,
		archiver->toplevel
	);
	if(d != NULL)
	{
		edv_archive_comments_dlg_map(d);
		edv_archive_comments_dlg_update_display(d);
	}

	g_free(comments);
	g_free(arch_path);

	edv_archiver_set_busy(archiver, FALSE);
}

/*
 *	Check Archive.
 */
static void edv_archiver_op_check(EDVArchiver *archiver)
{
	gboolean yes_to_all = FALSE;
	gint status;
	gchar		*msg,
			*arch_path;
	const gchar *error_msg;
	GtkWidget *toplevel;
	EDVCore *core;

	if(archiver == NULL)
		return;

	edv_archiver_set_busy(archiver, TRUE);

	toplevel = archiver->toplevel;
	core = archiver->core;

	/* Get the current archive's path */
	arch_path = STRDUP(edv_archiver_get_location(archiver));
	if(arch_path == NULL)
	{
		edv_archiver_set_busy(archiver, FALSE);
		return;
	}

	/* Check the archive */
	status = edv_archive_check(
		core,
		arch_path,
		edv_archiver_get_password(archiver),
		toplevel,
		TRUE,
		TRUE,
		&yes_to_all
	);

	/* Unmap the progress dialog which may have been mapped in
	 * the above operation
	 */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Check for errors */
	error_msg = edv_archive_check_get_error(core);
	if(!STRISEMPTY(error_msg))
	{
		gchar *msg = g_strdup_printf(
"The following problem was found:\n\
\n\
%s\n\
\n\
You may attempt to try and fix this problem by going to\n\
Edit->Fix.",
			error_msg
		);
		edv_play_sound_warning(core);
		edv_message_warning(
			"Check Results",
			msg,
			NULL,
			toplevel
		);
		g_free(msg);
	}
	else if(status != -4)
	{
		gchar *msg = g_strdup_printf(
"There were no problems found in the archive\n\
\"%s\".",
			g_basename(arch_path)
		);
		edv_play_sound_completed(core);
		edv_message_info(
			"Check Results",
			msg,
			NULL,
			toplevel
		);
		g_free(msg);
	}

	/* Update the status bar */
	if(status == -4)
		msg = g_strdup(
"Check operation canceled"
		);
	else
		msg = g_strdup(
"Checked archive"
		);
	edv_status_bar_message(archiver->status_bar, msg, FALSE);
	g_free(msg);

	g_free(arch_path);

	edv_archiver_set_busy(archiver, FALSE);
}

/*
 *	Fix Archive.
 */
static void edv_archiver_op_fix(EDVArchiver *archiver)
{
	gboolean yes_to_all = FALSE;
	gint		status,
			response;
	gchar		*msg,
			*arch_path;
	const gchar *error_msg;
	GtkWidget *toplevel;
	EDVCore *core;

	if(archiver == NULL)
		return;

	edv_archiver_set_busy(archiver, TRUE);

	toplevel = archiver->toplevel;
	core = archiver->core;

	/* Check and warn if write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
	{
		edv_archiver_set_busy(archiver, FALSE);
		return;
	}

	/* Get the current archive's path */
	arch_path = STRDUP(edv_archiver_get_location(archiver));
	if(arch_path == NULL)
	{
		edv_archiver_set_busy(archiver, FALSE);
		return;
	}

	/* Confirm fix archive */
	msg = g_strdup_printf(
"Fix archive \"%s\"?",
		g_basename(arch_path)
	);
	edv_play_sound_question(core);
	CDialogSetTransientFor(toplevel);
	response = CDialogGetResponse(
		"Confirm Fix",
		msg,
		NULL,
		CDIALOG_ICON_QUESTION,
		CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
		CDIALOG_BTNFLAG_YES
	);
	CDialogSetTransientFor(NULL);
	g_free(msg);
	if(response != CDIALOG_RESPONSE_YES)
	{
		g_free(arch_path);
		edv_archiver_set_busy(archiver, FALSE);
		return;
	}

	/* Fix the archive */
	status = edv_archive_fix(
		core,
		arch_path,
		edv_archiver_get_password(archiver),
		toplevel,
		TRUE,				/* Show progress */
		TRUE,				/* Interactive */
		&yes_to_all
	);

	/* Unmap the progress dialog which may have been mapped in
	 * the above operation
	 */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Get the fix results */
	error_msg = edv_archive_fix_get_error(core);
	if(!STRISEMPTY(error_msg))
	{
		edv_play_sound_error(core);
		edv_message_object_op_error(
			"Fix Error",
			error_msg,
			arch_path,
			toplevel
		);
	}

	/* Report that the archive has been modified */
	if(arch_path != NULL)
	{
		EDVVFSObject *obj = edv_vfs_object_lstat(arch_path);
		if(obj != NULL)
		{
			edv_emit_vfs_object_modified(
				core,
				arch_path,
				arch_path,
				obj
			);
			edv_vfs_object_delete(obj);
		}
	}

	/* Update the status bar */
	if(status == -4)
		msg = g_strdup(
"Fix operation canceled"
		);
	else if(status == 0)
		msg = g_strdup(
"Fixed archive"
		);
	else
		msg = g_strdup(
"Unable to fix archive"
		);
	edv_status_bar_message(archiver->status_bar, msg, FALSE);
	g_free(msg);

	g_free(arch_path);

	edv_archiver_set_busy(archiver, FALSE);
}

/*
 *	Select All.
 */
static void edv_archiver_op_select_all(EDVArchiver *archiver)
{
	EDVCore *core;
	GtkCList *clist;

	if(archiver == NULL)
		return;

	edv_archiver_set_busy(archiver, TRUE);

	clist = GTK_CLIST(archiver->contents_clist);
	core = archiver->core;

	/* Select all */
	gtk_clist_freeze(clist);
	gtk_clist_select_all(clist);
	gtk_clist_thaw(clist);

	/* Assume highest row index as the last selected row */
	archiver->contents_clist_selected_row = clist->rows - 1;

	edv_status_bar_message(
		archiver->status_bar, "All objects selected", FALSE
	);

	edv_archiver_set_busy(archiver, FALSE);
}

/*
 *	Unselect All.
 */
static void edv_archiver_op_unselect_all(EDVArchiver *archiver)
{
	EDVCore *core;
	GtkCList *clist;

	if(archiver == NULL)
		return;

	edv_archiver_set_busy(archiver, TRUE);

	clist = GTK_CLIST(archiver->contents_clist);
	core = archiver->core;

	/* Unselect all */
	gtk_clist_freeze(clist);
	gtk_clist_unselect_all(clist);
	gtk_clist_thaw(clist);

	/* Mark contents clist's row as unselected */
	archiver->contents_clist_selected_row = -1;

	edv_status_bar_message(
		archiver->status_bar, "All objects unselected", FALSE
	);

	edv_archiver_set_busy(archiver, FALSE);
}

/*
 *	Invert Selection.
 */
static void edv_archiver_op_invert_selection(EDVArchiver *archiver)
{
	gint		row,
			nrows;
	GList		*glist,
			*selection;
	GtkCList *clist;
	EDVCore *core;

	if(archiver == NULL)
		return;

	edv_archiver_set_busy(archiver, TRUE);

	clist = GTK_CLIST(archiver->contents_clist);
	core = archiver->core;

	/* Get a copy of selected rows list */
	selection = (clist->selection != NULL) ?
		g_list_copy(clist->selection) : NULL;

	/* Invert the selection */
	gtk_clist_freeze(clist);
	for(row = 0, nrows = clist->rows;
		row < nrows;
		row++
	)
	{
		for(glist = selection;
			glist != NULL;
			glist = g_list_next(glist)
		)
		{
			if(row == (gint)glist->data)
			{
				gtk_clist_unselect_row(clist, row, 0);
				break;
			}
		}
		/* Row not selected? */
		if(glist == NULL)
			gtk_clist_select_row(clist, row, 0);
	}
	gtk_clist_thaw(clist);

	g_list_free(selection);

	edv_status_bar_message(
		archiver->status_bar, "Selection inverted", FALSE
	);

	edv_archiver_set_busy(archiver, FALSE);
}

/*
 *	Properties.
 */
static void edv_archiver_op_properties(EDVArchiver *archiver)
{
	GtkWidget *toplevel;
	EDVArchiveObject *obj = NULL;
	EDVCore *core;

	if(archiver == NULL)
		return;

	edv_archiver_set_busy(archiver, TRUE);

	toplevel = archiver->toplevel;
	core = archiver->core;

	edv_archiver_sync_data(archiver);

	/* Get the selected object */
	if(archiver->contents_clist_selected_row > -1)
	{
		GtkCList *clist = GTK_CLIST(archiver->contents_clist);
		GList *glist = clist->selection_end;
		if(glist != NULL)
			obj = EDV_ARCHIVE_OBJECT(gtk_clist_get_row_data(
				clist,
				(gint)glist->data
			));
	}

	/* No selected object? */
	if(obj == NULL)
	{
		edv_archiver_set_busy(archiver, FALSE);
		return;
	}

	/* Create a new properties dialog displaying the object */
	edv_new_properties_dialog_archive(
		core,
		obj->path,
		edv_archiver_get_location(archiver),
		NULL,				/* Default page */
		obj->meta_data_list,		/* Extended properties list */
		core->geometry_flags,
		(core->geometry_flags != 0) ? &core->geometry : NULL,
		toplevel
	);

	edv_archiver_set_busy(archiver, FALSE);
}


/*
 *	Stop.
 */
static void edv_archiver_op_stop(EDVArchiver *archiver)
{
	if(archiver == NULL)
		return;

	/* Send an interrupt signal to the subprocess if it is
	 * running
	 *
	 * If there is a subprocess then, when this call returns,
	 * the subprocess monitoring timeout callback will check
	 * if it is still running and if the process has finished
	 * then it will set the Archiver as not passive busy and
	 * update its displayed values and delete the subprocess
	 * information from the Archiver
	 */
	edv_archiver_subprocess_stop(archiver->subprocess);
}

/*
 *	Refresh.
 */
void edv_archiver_op_refresh(EDVArchiver *archiver)
{
	GtkWidget	*toplevel,
			*sb;
	GtkCList *clist;
	CfgList *cfg_list;
	EDVCore *core;

	if(archiver == NULL)
		return;

	toplevel = archiver->toplevel;
	core = archiver->core;
	cfg_list = core->cfg_list;
	sb = archiver->status_bar;

	edv_archiver_set_busy(archiver, TRUE);
	GUIBlockInput(toplevel, TRUE);

	/* Update the Contents GtkList */
	clist = GTK_CLIST(archiver->contents_clist);
	if(clist != NULL)
	{
		/* Record the last scroll position */
		gboolean passive = TRUE;
		const gfloat	last_x = GTK_ADJUSTMENT_GET_VALUE(clist->hadjustment),
				last_y = GTK_ADJUSTMENT_GET_VALUE(clist->vadjustment);

		gtk_clist_freeze(clist);

		/* Reget the listing */
		edv_archiver_list_get(
			archiver,
			passive,
			EDV_GET_B(EDV_CFG_PARM_LISTS_SHOW_PROGRESS),
			FALSE			/* Do not show comments */
		);

		gtk_clist_thaw(clist);

		/* Scroll back to the original position */
		if(!passive)
			edv_clist_scroll_to_position(
				clist,
				last_x, last_y
			);
	}

	edv_archiver_update_display(archiver);
/*	edv_status_bar_message(sb, "Refreshed contents listing", FALSE); */

	GUIBlockInput(toplevel, FALSE);
	edv_archiver_set_busy(archiver, FALSE);
}

/*
 *	Refresh All.
 */
static void edv_archiver_op_refresh_all(EDVArchiver *archiver)
{
	if(archiver == NULL)
		return;

	edv_archiver_set_busy(archiver, TRUE);

	/* Refresh Archiver */
	edv_archiver_op_refresh(archiver);

	edv_archiver_set_busy(archiver, FALSE);
}

/*
 *	Statistics.
 */
static void edv_archiver_op_statistics(EDVArchiver *archiver)
{
	const gchar *filter;
	gchar *arch_path;
	GList *objs_list;
	GtkCList *clist;
	EDVArchiveStatisticsDlg *d;

	if(archiver == NULL)
		return;

	edv_archiver_set_busy(archiver, TRUE);

	clist = GTK_CLIST(archiver->contents_clist);

	arch_path = STRDUP(edv_archiver_get_location(archiver));
	if(arch_path == NULL)
	{
		edv_archiver_set_busy(archiver, FALSE);
		return;
	}

	/* Delete the previous Archive Info Dialog (if any) */
	d = archiver->statistics_dialog;
	if(d != NULL)
	{
		edv_archive_statistics_dlg_delete(d);
		archiver->statistics_dialog = d = NULL;
	}

/* TODO Get the list of archive objects */
	objs_list = NULL;
	filter = archiver->contents_list_filter;
	if((filter != NULL) ?
		(STRISEMPTY(filter) ||
		 !strcmp((const char *)filter, "*")
		) : TRUE
	)
	{
		const gint nrows = clist->rows;
		gint i;
		EDVArchiveObject *obj;

		for(i = 0; i < nrows; i++)
		{
			obj = EDV_ARCHIVE_OBJECT(gtk_clist_get_row_data(
				clist,
				i
			));
			if(obj == NULL)
				continue;

			objs_list = g_list_append(
				objs_list,
				obj
			);
		}
	}
/* TODO list statistics */

	/* Create a new EDVArchiveStatisticsDlg to display the
	 * statistics
	 */
	archiver->statistics_dialog = d = edv_archive_statistics_dlg_new(
		archiver->core,
		arch_path,
		edv_archiver_get_password(archiver),
		objs_list,
		archiver->toplevel
	);
	if(d != NULL)
	{
		edv_archive_statistics_dlg_map(d);
		edv_archive_statistics_dlg_update_display(d);
	}

	g_list_free(objs_list);
	g_free(arch_path);

	edv_archiver_set_busy(archiver, FALSE);
}


/*
 *	Sets the contents list filter.
 */
static void edv_archiver_op_list_filter(EDVArchiver *archiver)
{
	if(archiver == NULL)
		return;

	edv_archiver_set_busy(archiver, TRUE);

	if(edv_query_name_filter(
		archiver->core,
		&archiver->contents_list_filter,
		archiver->toplevel
	))
	{
		edv_archiver_set_title(
			archiver,
			edv_archiver_get_location(archiver)
		);
		edv_archiver_op_refresh(archiver);
	}

	edv_archiver_set_busy(archiver, FALSE);
}

/*
 *	MIME Types.
 */
static void edv_archiver_op_mime_types(EDVArchiver *archiver)
{
	gint i;
	gchar *type_string = NULL;
	GtkWidget *toplevel;
	GtkCList *clist;
	EDVCore *core;

	if(archiver == NULL)
		return;

	edv_archiver_set_busy(archiver, TRUE);

	toplevel = archiver->toplevel;
	clist = GTK_CLIST(archiver->contents_clist);
	core = archiver->core;

	i = edv_clist_get_selected_last(clist, NULL);
	if(i > -1)
	{
		EDVArchiveObject *obj = EDV_ARCHIVE_OBJECT(
			gtk_clist_get_row_data(clist, i)
		);
		if(obj != NULL)
		{
			(void)edv_match_object_type_string(
				core->mime_types_list,
				obj->type,
				obj->path,
				obj->permissions,
				&type_string
			);
		}
	}

	edv_map_mime_types(
		core,
		type_string,
		core->geometry_flags,
		(core->geometry_flags != 0) ? &core->geometry : NULL,
		toplevel
	);

	g_free(type_string);

	edv_archiver_set_busy(archiver, FALSE);
}
