#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>

#include "cfg.h"

#include "guiutils.h"
#include "cdialog.h"
#include "progressdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_archive_obj.h"
#include "libendeavour2-base/edv_id.h"
#include "edv_pixmap.h"
#include "edv_date_format.h"
#include "edv_archive_obj_stat.h"
#include "edv_obj_info_match.h"
#include "edv_mime_type.h"
#include "edv_mime_types_list.h"
#include "edv_utils_gtk.h"
#include "edv_status_bar.h"
#include "archiver.h"
#include "archiver_cb.h"
#include "archiver_list.h"
#include "archiver_subprocess.h"
#include "edv_cb.h"
#include "edv_op.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"


typedef struct _EDVArchiverContentsListProgressData	EDVArchiverContentsListProgressData;
#define EDV_ARCHIVER_CONTENTS_LIST_PROGRESS_DATA(p)	((EDVArchiverContentsListProgressData *)(p))


/* Callbacks */
static gint edv_archiver_list_progress_cb(
	const gchar *arch_path,
	EDVArchiveObject *obj,
	const gulong i, const gulong m,
	gpointer data
);

/* Columns */
gint edv_archiver_list_get_column_index_by_type(
	EDVArchiver *archiver,
	const EDVArchiverColumnType column_type
);
gint edv_archiver_list_get_column_width_by_type(
	EDVArchiver *archiver,
	const EDVArchiverColumnType column_type
);
EDVArchiverColumnType edv_archiver_list_get_column_type_by_index(
	EDVArchiver *archiver,
	const gint column_num
);
void edv_archiver_list_set_column_width_by_type(
	EDVArchiver *archiver,
	const EDVArchiverColumnType column_type,
	const gint width
);
void edv_archiver_list_resize_column_optimul(
	EDVArchiver *archiver,
	const gint column_num
);
void edv_archiver_list_resize_columns_optimul(EDVArchiver *archiver);
void edv_archiver_list_reset_columns(EDVArchiver *archiver);

/* Set Cells */
static void edv_archiver_list_set_cell_index(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column
);
static void edv_archiver_list_set_cell_name(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column
);
static void edv_archiver_list_set_cell_size(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column,
	const gboolean hide_dir_size,
	const gboolean hide_link_size,
	const EDVSizeFormat size_format,
	const gulong block_size
);
static void edv_archiver_list_set_cell_storage_size(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column,
	const gboolean hide_dir_size,
	const gboolean hide_link_size,
	const EDVSizeFormat size_format,
	const gulong block_size
);
static void edv_archiver_list_set_cell_type(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column
);
static void edv_archiver_list_set_cell_permissions(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column,
	const gboolean hide_link_permissions
);
static void edv_archiver_list_set_cell_owner(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column
);
static void edv_archiver_list_set_cell_group(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column
);
static void edv_archiver_list_set_cell_date_access(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column,
	const EDVDateRelativity date_relativity,
	const gchar *date_format
);
static void edv_archiver_list_set_cell_date_modified(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column,
	const EDVDateRelativity date_relativity,
	const gchar *date_format
);
static void edv_archiver_list_set_cell_date_changed(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column,
	const EDVDateRelativity date_relativity,
	const gchar *date_format
);
static void edv_archiver_list_set_cell_location(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column
);
static void edv_archiver_list_set_cell_linked_to(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column
);
static void edv_archiver_list_set_cell_device_type(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column
);
static void edv_archiver_list_set_cell_encryption(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column
);
static void edv_archiver_list_set_cell_compression(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column,
	const gboolean hide_dir_size, const gboolean hide_link_size
);
static void edv_archiver_list_set_cell_storage_method(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column
);
static void edv_archiver_list_set_cell_crc(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column
);
static void edv_archiver_list_set_row(
	EDVArchiver *archiver,
	const gint row,
	EDVArchiveObject *obj
);

gint edv_archiver_list_append(
	EDVArchiver *archiver,
	EDVArchiveObject *obj
);
static void edv_archiver_list_append_listing(
	EDVArchiver *archiver,
	const gchar *arch_path,
	const gchar *filter,
	const gchar *password,
	const gboolean show_progress
);

/* Finding */
gint edv_archiver_list_find_by_path(
	EDVArchiver *archiver,
	const gchar *path
);

/* Realize Listing */
void edv_archiver_list_realize_listing(EDVArchiver *archiver);

/* Get Listing */
void edv_archiver_list_get(
	EDVArchiver *archiver,
	const gboolean passive,
	const gboolean show_progress,
	const gboolean show_comments
);
void edv_archiver_list_clear(EDVArchiver *archiver);

/* Object Callbacks */
void edv_archiver_list_vfs_object_added_cb(
	EDVArchiver *archiver,
	const gchar *path,
	EDVVFSObject *obj
);
void edv_archiver_list_vfs_object_modified_cb(
	EDVArchiver *archiver,
	const gchar *path,
	const gchar *new_path,
	EDVVFSObject *obj
);
void edv_archiver_list_vfs_object_removed_cb(
	EDVArchiver *archiver,
	const gchar *path
);

/* Archive Object Callbacks */
void edv_archiver_list_archive_object_added_cb(
	EDVArchiver *archiver,
	const gchar *arch_path,
	const gchar *path,
	EDVArchiveObject *obj
);
void edv_archiver_list_archive_object_modified_cb(
	EDVArchiver *archiver,
	const gchar *arch_path,
	const gchar *path,
	const gchar *new_path,
	EDVArchiveObject *obj
);
void edv_archiver_list_archive_object_removed_cb(
	EDVArchiver *archiver,
	const gchar *arch_path,
	const gchar *path
);


/*
 *	Archive Stat List Progress Data:
 *
 *	Used in edv_archiver_list_progress_cb().
 */
struct _EDVArchiverContentsListProgressData {
	EDVCore	*core;
	EDVArchiver	*archiver;
	GtkCList	*clist;
	gboolean	show_progress;
	gint		last_progress_percent;

};


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Archive stat list sequential progress callback.
 *
 *	If obj is not NULL then it specifies a new EDVArchiveObject
 *	that we should take and append to the list. The calling
 *	function does not refer to it again after this call.
 */
static gint edv_archiver_list_progress_cb(
	const gchar *arch_path,
	EDVArchiveObject *obj,
	const gulong i, const gulong m,
	gpointer data
)
{
	EDVArchiverContentsListProgressData *d = EDV_ARCHIVER_CONTENTS_LIST_PROGRESS_DATA(data);
	EDVArchiver *archiver = d->archiver;

	/* Got object to append to the Contents List? */
	if(obj != NULL)
		(void)edv_archiver_list_append(
			archiver,
			obj
		);

	/* Report progress */
	if(d->show_progress)
	{
		if(m > 0l)
		{
			const gint progress_percent = (gint)(i * 100l / m);

#if 0
/* This does not work because m is not always the number of objects */
			/* Report the initial number of archive objects being loaded? */
			if(d->last_progress_percent < 0.0f)
			{
				const gint nobjs = (gint)m;
				gchar *msg = g_strdup_printf(
"Loading archive contents (%i %s)...",
					nobjs,
					(nobjs == 1) ? "archive object" : "archive objects"
				);
				edv_status_bar_message(
					archiver->status_bar,
					msg,
					TRUE
				);
				g_free(msg);
			}
#endif

			if(progress_percent > d->last_progress_percent)
			{
				edv_status_bar_progress(
					archiver->status_bar,
					(gfloat)progress_percent / 100.0f,
					TRUE
				);
				d->last_progress_percent = progress_percent;
			}
		}
		else
		{
			edv_status_bar_progress(
				archiver->status_bar,
				-1.0f,
				TRUE
			);
		}
	}

	return(0);
}


/*
 *	Gets the column index displaying the column type.
 *
 *	The column_type specifies the EDVArchiverColumnType.
 *
 *	Returns the column index that is set to column_type or
 *	negative on error.
 */
gint edv_archiver_list_get_column_index_by_type(
	EDVArchiver *archiver,
	const EDVArchiverColumnType column_type
) 
{
	gint column_num;
	GtkCList *clist;
	CfgList *cfg_list;
	CfgIntList *column_types_intlist;
	EDVCore *core;

	if(archiver == NULL)
		return(-2);

	clist = GTK_CLIST(archiver->contents_clist);
	core = archiver->core;
	cfg_list = core->cfg_list;

	column_types_intlist = EDV_GET_INT_LIST(
		EDV_CFG_PARM_ARCHIVER_CONTENTS_COLUMN
	);
	if(column_types_intlist == NULL)
		return(-2);

	column_num = (gint)g_list_index(
		column_types_intlist->list,
		(gpointer)column_type
	);
	if(column_num < 0)
		return(-2);

	if(column_num >= clist->columns)
		return(clist->columns - 1);

	return(column_num);
}

/*
 *	Gets the width of the column displaying the column type.
 *
 *	The column_type specifies the column type.
 *
 *	Returns the column width.
 */
gint edv_archiver_list_get_column_width_by_type(
	EDVArchiver *archiver,
	const EDVArchiverColumnType column_type
)
{
	GtkCList *clist;
	const gint column_num = edv_archiver_list_get_column_index_by_type(
		archiver,
		column_type
	);
	if(column_num < 0)
		return(0);

	clist = GTK_CLIST(archiver->contents_clist);

	return(clist->column[column_num].width);
}

/*
 *	Gets the column type that the column index is displaying.
 *
 *	The column_num specifies the column index.
 *
 *	Returns the EDVArchiverColumnType that column_num is
 *	displaying.
 */
EDVArchiverColumnType edv_archiver_list_get_column_type_by_index(
	EDVArchiver *archiver,
	const gint column_num
)
{
	CfgList *cfg_list;
	CfgIntList *column_types_intlist;
	EDVCore *core;

	if(archiver == NULL)
		return(EDV_ARCHIVER_COLUMN_TYPE_NAME);

	core = archiver->core;
	cfg_list = core->cfg_list;

	column_types_intlist = EDV_GET_INT_LIST(
		EDV_CFG_PARM_ARCHIVER_CONTENTS_COLUMN
	);
	if(column_types_intlist == NULL)
		return(EDV_ARCHIVER_COLUMN_TYPE_NAME);

	return((EDVArchiverColumnType)g_list_nth_data(
		column_types_intlist->list,
		column_num
	));
}

/*
 *	Sets the width of the column that is displaying the column
 *	type.
 *
 *	The column_type specifies the column type.
 * 
 *	The width specifies the width of the column.
 */
void edv_archiver_list_set_column_width_by_type(
	EDVArchiver *archiver,
	const EDVArchiverColumnType column_type,
	const gint width
)
{
	GtkCList *clist;
	const gint column_num = edv_archiver_list_get_column_index_by_type(
		archiver,
		column_type
	);
	if(column_num < 0)
		return;

	clist = GTK_CLIST(archiver->contents_clist);

	gtk_clist_set_column_width(
		clist,
		column_num,
		width
	);
}

/*
 *	Resizes the column to optimul size.
 */
void edv_archiver_list_resize_column_optimul(
	EDVArchiver *archiver,
	const gint column_num
)
{
	gint		ncolumns,
			width;
	GtkCList *clist;

	if(archiver == NULL)
		return;

	clist = GTK_CLIST(archiver->contents_clist);
	ncolumns = clist->columns;
	if((column_num < 0) || (column_num >= ncolumns))
		return;

	gtk_clist_freeze(clist);

	width = gtk_clist_optimal_column_width(
		clist,
		column_num
	);
	if(width > 0)
		gtk_clist_set_column_width(
			clist,
			column_num,
			width
		);

	gtk_clist_thaw(clist);
}

/*
 *	Resizes all the columns to optimul sizes.
 */
void edv_archiver_list_resize_columns_optimul(EDVArchiver *archiver)
{
	gint		i,
			ncolumns;
	GtkCList *clist;

	if(archiver == NULL)
		return;

	clist = GTK_CLIST(archiver->contents_clist);
	ncolumns = clist->columns;

	gtk_clist_freeze(clist);

	for(i = 0; i < ncolumns; i++)
		edv_archiver_list_resize_column_optimul(
			archiver,
			i
		);

	gtk_clist_thaw(clist);
}

/*
 *	Resets the column headings' titles, justifications, and
 *	visibility according to the values in the current configuration.
 */
void edv_archiver_list_reset_columns(EDVArchiver *archiver)
{
	gint		i,
			width,
			ncolumns;
	const gchar *title = NULL;
	GList *glist;
	GtkJustification justify = GTK_JUSTIFY_LEFT;
	GtkRcStyle *lists_rcstyle;
	GtkCList *clist;
	CfgIntList	*column_types_intlist,
			*column_width_intlist;
	CfgList *cfg_list;
	EDVArchiverColumnType column_type;
	EDVCore *core;

	if(archiver == NULL)
		return;

	clist = GTK_CLIST(archiver->contents_clist);
	ncolumns = clist->columns;
	core = archiver->core;
	cfg_list = core->cfg_list;
	lists_rcstyle = core->lists_rcstyle;

	/* Get the column headings' titles and widths from the
	 * configuration
	 */
	column_types_intlist = EDV_GET_INT_LIST(
		EDV_CFG_PARM_ARCHIVER_CONTENTS_COLUMN
	);
	column_width_intlist = EDV_GET_INT_LIST(
		EDV_CFG_PARM_ARCHIVER_CONTENTS_COLUMN_WIDTH
	);

	if((column_types_intlist == NULL) || (column_width_intlist == NULL))
		return;

	gtk_clist_freeze(clist);

	/* Update the GtkCList's column settings */
	gtk_clist_column_titles_active(clist);
	gtk_clist_column_titles_show(clist);
	gtk_clist_set_auto_sort(
		clist,
		FALSE
	);
	gtk_clist_set_sort_type(
		clist,
		GTK_SORT_DESCENDING
	);

	/* Set the column headings' titles and justifications */
	for(glist = column_types_intlist->list, i = 0;
	    glist != NULL;
	    glist = g_list_next(glist), i++
	)
	{
		column_type = (EDVArchiverColumnType)glist->data;

		/* Get the width for this column type */
		width = (gint)g_list_nth_data(
			column_width_intlist->list,
			(guint)column_type
		);

		/* Get column title and justification  based on the
		 * column type
		 */
		switch(column_type)
		{
		  case EDV_ARCHIVER_COLUMN_TYPE_INDEX:
			title = "Index";
			justify = GTK_JUSTIFY_RIGHT;
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_NAME:
			title = "Name";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_SIZE:
			title = "Size";
			justify = GTK_JUSTIFY_RIGHT;
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_STORAGE_SIZE:
			title = "Storage Size";
			justify = GTK_JUSTIFY_RIGHT;
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_TYPE:
			title = "Type";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_PERMISSIONS:
			title = "Permissions";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_OWNER:
			title = "Owner";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_GROUP:
			title = "Group";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_DATE_ACCESS:
			title = "Date Access";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_DATE_MODIFIED:
			title = "Date Modified";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_DATE_CHANGED:
			title = "Date Changed";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_LOCATION:
			title = "Location";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_LINKED_TO:
			title = "Linked To";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_DEVICE_TYPE:
			title = "Device Type";
			justify = GTK_JUSTIFY_RIGHT;
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_ENCRYPTION:
			title = "Encryption";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_COMPRESSION:
			title = "Compression";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_METHOD:
			title = "Method";
			justify = GTK_JUSTIFY_LEFT;
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_CRC:
			title = "CRC";
			justify = GTK_JUSTIFY_LEFT;
			break;
		}

		gtk_clist_set_column_visibility(
			clist,
			i,
			TRUE
		);
		gtk_clist_set_column_auto_resize(
			clist,
			i,
			FALSE
		);
		gtk_clist_set_column_title(
			clist,
			i,
			title
		);
		gtk_clist_set_column_justification(
			clist,
			i,
			justify
		);
		gtk_clist_set_column_width(
			clist,
			i,
			width
		);
	}
	/* Reset any remaining columns */
	while(i < ncolumns)
	{
		gtk_clist_set_column_visibility(
			clist,
			i,
			FALSE
		);
		i++;
	}

	/* Set RC style after column headings have been mapped */
	if(lists_rcstyle != NULL)
		gtk_widget_modify_style_recursive(
			GTK_WIDGET(clist),
			lists_rcstyle
		);

	gtk_clist_thaw(clist);
}


/*
 *	Index.
 */
static void edv_archiver_list_set_cell_index(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column
)
{
	const gint border_minor = 2;
	const gulong index = obj->index;
	gchar *index_str;

	/* Format the index string */
	if(index != 0l)
		index_str = g_strdup_printf(
			"#%ld",
			obj->index
		);
	else
		index_str = NULL;

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist,
		row, column,
		(index_str != NULL) ? index_str : ""
	);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, -border_minor
	);
	gtk_clist_thaw(clist);

	g_free(index_str);
}

/*
 *	Name.
 */
static void edv_archiver_list_set_cell_name(
	EDVCore *core, EDVArchiver *archiver,
	GtkCList *clist, EDVArchiveObject *obj,
	const gint row, const gint column
)
{
	const gchar *name = obj->name;
	EDVPixmap	*icon,
			*icon_hidden;

	if(STRISEMPTY(name))
		name = (obj->path != NULL) ? obj->path : "(null)";

	/* Get pixmap and mask for the object's icon */
	(void)edv_match_object_icon(
		NULL,				/* No devices */
		core->mime_types_list,
		obj->type,
		name,
		TRUE,				/* Assume link valid */
		obj->permissions,
		EDV_ICON_SIZE_20,
		&icon,
		NULL,
		NULL,
		&icon_hidden
	);
	/* Hidden? */
	if(edv_path_is_hidden(name))
	{
		if(edv_pixmap_is_loaded(icon_hidden))
		{
			(void)edv_pixmap_unref(icon);
			icon = edv_pixmap_ref(icon_hidden);
		}
	}

	/* Set the cell */
	gtk_clist_freeze(clist);
	if(edv_pixmap_is_loaded(icon))
		gtk_clist_set_pixtext(
			clist,
			row, column,
			name,
			EDV_LIST_PIXMAP_TEXT_SPACING,
			icon->pixmap, icon->mask
		);
	else
		gtk_clist_set_text(
			clist,
			row, column,
			name
		);
	gtk_clist_set_cell_style(
		clist,
		row, column,
		NULL
	);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, 0
	);
	gtk_clist_thaw(clist);

	(void)edv_pixmap_unref(icon);
	(void)edv_pixmap_unref(icon_hidden);
}

/*
 *	Size.
 */
static void edv_archiver_list_set_cell_size(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column,
	const gboolean hide_dir_size, const gboolean hide_link_size,
	const EDVSizeFormat size_format,
	const gulong block_size
)
{
	const gint border_minor = 2;
	const EDVObjectType type = obj->type;
	gchar *size_str;

	/* Get string for size */
	if(hide_dir_size && (type == EDV_OBJECT_TYPE_DIRECTORY))
		size_str = NULL;
	else if(hide_link_size && (type == EDV_OBJECT_TYPE_LINK))
		size_str = NULL;
	else if((type == EDV_OBJECT_TYPE_DEVICE_BLOCK) ||
			(type == EDV_OBJECT_TYPE_DEVICE_CHARACTER)
	)
		size_str = NULL;
	else
		size_str = STRDUP(edv_str_size_format(
			obj->size,
			size_format,
			block_size,
			',',
			TRUE				/* Allow unit conversions */
		));

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist,
		row, column,
		(size_str != NULL) ? size_str : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, -border_minor
	);
	gtk_clist_thaw(clist);

	g_free(size_str);
}

/*
 *	Storage Size.
 */
static void edv_archiver_list_set_cell_storage_size(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column,
	const gboolean hide_dir_size, const gboolean hide_link_size,
	const EDVSizeFormat size_format,
	const gulong block_size
)
{
	const gint border_minor = 2;
	const EDVObjectType type = obj->type;
	gchar *size_str;

	/* Get string for size */
	if(hide_dir_size && (type == EDV_OBJECT_TYPE_DIRECTORY))
		size_str = NULL;
	else if(hide_link_size && (type == EDV_OBJECT_TYPE_LINK))
		size_str = NULL;
	else if((type == EDV_OBJECT_TYPE_DEVICE_BLOCK) ||
			(type == EDV_OBJECT_TYPE_DEVICE_CHARACTER)
	)
		size_str = NULL;
	else
		size_str = STRDUP(edv_str_size_format(
			obj->storage_size,
			size_format,
			block_size,
			',',
			TRUE
		));

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist,
		row, column,
		(size_str != NULL) ? size_str : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, -border_minor
	);
	gtk_clist_thaw(clist);

	g_free(size_str);
}

/*
 *	Type.
 */
static void edv_archiver_list_set_cell_type(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column
)
{
	gchar *type_string;

	/* Get MIME Type type string for the given object */
	(void)edv_match_object_type_string(
		core->mime_types_list,
		obj->type,
		obj->name,
		obj->permissions,
		&type_string
	);

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist,
		row, column,
		(type_string != NULL) ? type_string : ""
	);
	gtk_clist_set_cell_style(
		clist,
		row, column,
		NULL
	);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, 0
	);
	gtk_clist_thaw(clist);

	g_free(type_string);
}

/*
 *	Permissions.
 */
static void edv_archiver_list_set_cell_permissions(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist, EDVArchiveObject *obj,
	const gint row, const gint column,
	const gboolean hide_link_permissions
)
{
	gchar *permissions_str;

	/* Format the permissions string */
	if(hide_link_permissions && (obj->type == EDV_OBJECT_TYPE_LINK))
		permissions_str = NULL;
	else
		permissions_str = edv_str_permissions(obj->permissions);

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist,
		row, column,
		(permissions_str != NULL) ? permissions_str : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, 0
	);
	gtk_clist_thaw(clist);

	g_free(permissions_str);
}

/*
 *	Owner.
 */
static void edv_archiver_list_set_cell_owner(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column
)
{
	const gchar *owner_name = obj->owner_name;
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist,
		row, column,
		(owner_name != NULL) ? owner_name : ""
	);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, 0
	);
	gtk_clist_thaw(clist);
}

/*
 *	Group.
 */
static void edv_archiver_list_set_cell_group(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist, EDVArchiveObject *obj,
	const gint row, const gint column
)
{
	const gchar *group_name = obj->group_name;
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist,
		row, column,
		(group_name != NULL) ? group_name : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, 0
	);
	gtk_clist_thaw(clist);
}

/*
 *	Date Accessed.
 */
static void edv_archiver_list_set_cell_date_access(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column,
	const EDVDateRelativity date_relativity,
	const gchar *date_format
)
{
	/* Get/format the date string */
	const gulong t = obj->access_time;
	gchar *s = (t > 0l) ?
		edv_date_string_format(t, date_format, date_relativity) :
		NULL;

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist,
		row, column,
		(s != NULL) ? s : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, 0
	);
	gtk_clist_thaw(clist);

	g_free(s);
}

/*
 *	Date Modified.
 */
static void edv_archiver_list_set_cell_date_modified(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column,
	const EDVDateRelativity date_relativity,
	const gchar *date_format
)
{
	/* Get/format the date string */
	const gulong t = obj->modify_time;
	gchar *s = (t > 0l) ?
		edv_date_string_format(t, date_format, date_relativity) :
		NULL;

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist,
		row, column,
		(s != NULL) ? s : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, 0
	);
	gtk_clist_thaw(clist);

	g_free(s);
}

/*
 *	Date Changed.
 */
static void edv_archiver_list_set_cell_date_changed(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column,
	const EDVDateRelativity date_relativity,
	const gchar *date_format
)
{
	/* Get/format the date string */
	const gulong t = obj->change_time;
	gchar *s = (t > 0l) ?
		edv_date_string_format(t, date_format, date_relativity) :
		NULL;

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist,
		row, column,
		(s != NULL) ? s : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, 0
	);
	gtk_clist_thaw(clist);

	g_free(s);
}

/*
 *	Location.
 */
static void edv_archiver_list_set_cell_location(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column
)
{
	gchar		*path,
			*location;
	const gchar *path_in_arch = obj->path;
	EDVPixmap *icon = NULL;

	/* Get the path in the archive with the appropriate prefix */
	if(STRISEMPTY(path_in_arch))
		path = g_strdup("");
	else if(g_path_is_absolute(path_in_arch))
		path = g_strdup(path_in_arch);
	else
		path = g_strconcat(
			".",
			G_DIR_SEPARATOR_S,
			path_in_arch,
			NULL
		);

	/* Get the location */
	location = g_dirname(path);
	if(location == NULL)
		location = STRDUP("");

	/* Get the icon */
	if(!STRISEMPTY(location))
	{
		/* Get the directory icon */
		EDVMIMEType *m = edv_mime_types_list_match_type(
			core->mime_types_list,
			NULL,
			EDV_MIME_TYPE_TYPE_INODE_DIRECTORY,
			FALSE
		);
		if(m != NULL)
		{
			edv_mime_type_realize(m, FALSE);
			icon = edv_pixmap_ref(m->small_icon[
				EDV_MIME_TYPE_ICON_STATE_STANDARD
			]);
		}
	}

	/* Set the cell */
	gtk_clist_freeze(clist);
	if(edv_pixmap_is_loaded(icon))
		gtk_clist_set_pixtext(
			clist,
			row, column,
			location,
			EDV_LIST_PIXMAP_TEXT_SPACING,
			icon->pixmap, icon->mask
		);
	else
		gtk_clist_set_text(
			clist,
			row, column,
			location
		);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, 0
	);
	gtk_clist_thaw(clist);

	(void)edv_pixmap_unref(icon);
	g_free(path);
	g_free(location);
}

/*
 *	Linked To.
 */
static void edv_archiver_list_set_cell_linked_to(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column
)
{
	const gchar *link_target = obj->link_target;
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist,
		row, column,
		(link_target != NULL) ? link_target : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, 0
	);
	gtk_clist_thaw(clist);
}

/*
 *	Device Type.
 */
static void edv_archiver_list_set_cell_device_type(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column
)
{
	const gint border_minor = 2;
	gint major, minor;
	gchar *device_type_str;
	const EDVObjectType type = obj->type;

	/* Is the object a block or character device? */
	if((type == EDV_OBJECT_TYPE_DEVICE_BLOCK) ||
	   (type == EDV_OBJECT_TYPE_DEVICE_CHARACTER)
	)
	{
		edv_device_numbers_parse(
			(dev_t)obj->device_type,
			&major, &minor
		);
		device_type_str = g_strdup_printf(
			"%i, %i",
			major, minor
		);
	}
	else
	{
		device_type_str = NULL;
	}

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist,
		row, column,
		(device_type_str != NULL) ? device_type_str : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, -border_minor
	);
	gtk_clist_thaw(clist);

	g_free(device_type_str);
}

/*
 *	Encryption Name.
 */
static void edv_archiver_list_set_cell_encryption(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column
)
{
	const gchar *encryption_name = obj->encryption_name;;
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist,
		row, column,
		(encryption_name != NULL) ? encryption_name : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, 0
	);
	gtk_clist_thaw(clist);
}

/*
 *	Compression.
 */
static void edv_archiver_list_set_cell_compression(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column,
	const gboolean hide_dir_size,
	const gboolean hide_link_size
)
{
	EDVPixmap *p;

	/* Get compression ratio and compression type string */
	if(hide_dir_size && (obj->type == EDV_OBJECT_TYPE_DIRECTORY))
	{
		p = NULL;
	}
	else if(hide_link_size && (obj->type == EDV_OBJECT_TYPE_LINK))
	{
		p = NULL;
	}
	else
	{
		const gfloat ratio = obj->compression_ratio;
		if(ratio >= 0.0f)
		{
			GtkWidget *w = GTK_WIDGET(clist);
			GtkCListColumn *column_ptr = ((column >= 0) && (column < clist->columns)) ?
				&clist->column[column] : NULL;
			const gint	width = (column_ptr != NULL) ?
						((column_ptr->width_set) ? column_ptr->width : -1) : -1,
					height = GTK_CLIST_ROW_HEIGHT_SET(clist) ?
						(clist->row_height - 4) : -1;

			p = edv_new_progress_pixmap(
				w->window,
				gtk_widget_get_style(w),
				width, height,
				ratio,
				TRUE,		/* Draw value */
				GTK_ORIENTATION_HORIZONTAL,
				FALSE		/* Not reverse */
			);
		}
		else
			p = NULL;
	}

	/* Set the cell */
	gtk_clist_freeze(clist);
	if(edv_pixmap_is_loaded(p))
		gtk_clist_set_pixmap(
			clist,
			row, column,
			p->pixmap, p->mask
		);
	else
		gtk_clist_set_text(
			clist,
			row, column,
			""
		);
	gtk_clist_set_cell_style(
		clist,
		row, column,
		NULL
	);
	gtk_clist_thaw(clist);

	(void)edv_pixmap_unref(p);
}

/*
 *	Storage Method.
 */
static void edv_archiver_list_set_cell_storage_method(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column
)
{
	const gchar *storage_method = obj->storage_method;
	EDVPixmap *icon = STRISEMPTY(obj->encryption_name) ?
		NULL : edv_pixmap_ref(archiver->encrypted_icon);

	/* Set the cell */
	gtk_clist_freeze(clist);
	if(edv_pixmap_is_loaded(icon))
		gtk_clist_set_pixtext(
			clist,
			row, column,
			(storage_method != NULL) ? storage_method : "",
			EDV_LIST_PIXMAP_TEXT_SPACING,
			icon->pixmap, icon->mask
		);
	else
		gtk_clist_set_text(
			clist,
			row, column,
			(storage_method != NULL) ? storage_method : ""
		);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, 0
	);
	gtk_clist_thaw(clist);

	(void)edv_pixmap_unref(icon);
}

/*
 *	CRC.
 */
static void edv_archiver_list_set_cell_crc(
	EDVCore *core,
	EDVArchiver *archiver,
	GtkCList *clist,
	EDVArchiveObject *obj,
	const gint row, const gint column
)
{
	const gchar *crc = obj->crc;
	if(crc == NULL)
		crc = "";

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
		clist,
		row, column,
		crc
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
		clist,
		row, column,
		0, 0
	);
	gtk_clist_thaw(clist);
}

/*
 *	Sets the Contents List's row to the values of the object.
 *
 *	The clist specifies the Contents List.
 *
 *	The obj specifies the archive object.
 *
 *	The row specifies the row.
 *
 *	All inputs assumed valid.
 */
static void edv_archiver_list_set_row(
	EDVArchiver *archiver,
	const gint row,
	EDVArchiveObject *obj
)
{
	gboolean	hide_dir_size,
			hide_link_size,
			hide_link_permissions;
	gint i;
	gulong block_size;
	const gchar *date_format;
	GList *glist;
	GtkCList *clist = GTK_CLIST(archiver->contents_clist);
	CfgIntList *column_types_intlist;
	EDVSizeFormat size_format;
	EDVDateRelativity date_relativity;
	EDVArchiverColumnType column_type;
	EDVCore *core = archiver->core;
	CfgList *cfg_list = core->cfg_list;

	/* Get column types mapping */
	column_types_intlist = EDV_GET_INT_LIST(
		EDV_CFG_PARM_ARCHIVER_CONTENTS_COLUMN
	);
	if(column_types_intlist == NULL)
		return;

	/* Get additional display options */
	size_format = (EDVSizeFormat)EDV_GET_I(
		EDV_CFG_PARM_SIZE_FORMAT
	);
	block_size = EDV_GET_UL(EDV_CFG_PARM_BLOCK_SIZE);
	date_relativity = (EDVDateRelativity)EDV_GET_I(
		EDV_CFG_PARM_DATE_RELATIVITY
	);
	date_format = EDV_GET_S(EDV_CFG_PARM_DATE_FORMAT);
	hide_dir_size = EDV_GET_B(EDV_CFG_PARM_ARCHIVER_CONTENTS_HIDE_DIR_SIZE);
	hide_link_size = EDV_GET_B(EDV_CFG_PARM_ARCHIVER_CONTENTS_HIDE_LINK_SIZE);
	hide_link_permissions = EDV_GET_B(EDV_CFG_PARM_ARCHIVER_CONTENTS_HIDE_LINK_PERMISSIONS);

	gtk_clist_freeze(clist);

	/* Iterate through each column */
	for(glist = column_types_intlist->list, i = 0;
	    glist != NULL;
	    glist = g_list_next(glist), i++
	)
	{
		column_type = (EDVArchiverColumnType)glist->data;
		switch(column_type)
		{
		  case EDV_ARCHIVER_COLUMN_TYPE_INDEX:
			edv_archiver_list_set_cell_index(
				core,
				archiver,
				clist,
				obj,
				row, i
			);
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_NAME:
			edv_archiver_list_set_cell_name(
				core,
				archiver,
				clist,
				obj,
				row, i
			);
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_SIZE:
			edv_archiver_list_set_cell_size(
				core,
				archiver,
				clist,
				obj,
				row, i,
				hide_dir_size,
				hide_link_size,
				size_format, block_size
			);
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_STORAGE_SIZE:
			edv_archiver_list_set_cell_storage_size(
				core,
				archiver,
				clist,
				obj,
				row, i,
				hide_dir_size,
				hide_link_size,
				size_format, block_size
			);
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_TYPE:
			edv_archiver_list_set_cell_type(
				core,
				archiver,
				clist,
				obj,
				row, i
			);
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_PERMISSIONS:
			edv_archiver_list_set_cell_permissions(
				core,
				archiver,
				clist,
				obj,
				row, i,
				hide_link_permissions
			);
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_OWNER:
			edv_archiver_list_set_cell_owner(
				core,
				archiver,
				clist,
				obj,
				row, i
			);
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_GROUP:
			edv_archiver_list_set_cell_group(
				core,
				archiver,
				clist,
				obj,
				row, i
			);
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_DATE_ACCESS:
			edv_archiver_list_set_cell_date_access(
				core,
				archiver,
				clist,
				obj,
				row, i,
				date_relativity, date_format
			);
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_DATE_MODIFIED:
			edv_archiver_list_set_cell_date_modified(
				core,
				archiver,
				clist,
				obj,
				row, i,
				date_relativity, date_format
			);
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_DATE_CHANGED:
			edv_archiver_list_set_cell_date_changed(
				core,
				archiver,
				clist,
				obj,
				row, i,
				date_relativity, date_format
			);
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_LOCATION:
			edv_archiver_list_set_cell_location(
				core,
				archiver,
				clist,
				obj,
				row, i
			);
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_LINKED_TO:
			edv_archiver_list_set_cell_linked_to(
				core,
				archiver,
				clist,
				obj,
				row, i
			);
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_DEVICE_TYPE:
			edv_archiver_list_set_cell_device_type(
				core,
				archiver,
				clist,
				obj,
				row, i
			);
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_ENCRYPTION:
			edv_archiver_list_set_cell_encryption(
				core,
				archiver,
				clist,
				obj,
				row, i
			);
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_COMPRESSION:
			edv_archiver_list_set_cell_compression(
				core,
				archiver,
				clist,
				obj,
				row, i,
				hide_dir_size,
				hide_link_size
			);
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_METHOD:
			edv_archiver_list_set_cell_storage_method(
				core,
				archiver,
				clist,
				obj,
				row, i
			);
			break;
		  case EDV_ARCHIVER_COLUMN_TYPE_CRC:
			edv_archiver_list_set_cell_crc(
				core,
				archiver,
				clist,
				obj,
				row, i
			);
			break;
		}
	}

	gtk_clist_thaw(clist);
}

/*
 *	Appends the object to the Contents List.
 *
 *	The obj specifies the archive object who's values will be used
 *	to append a new row on the Contents List and will be
 *	transfered to the Content List's row data. The obj should not
 *	be referenced again after this call.
 *
 *	Returns the new row index or -1 on error.
 */
gint edv_archiver_list_append(
	EDVArchiver *archiver,
	EDVArchiveObject *obj
)
{
	gint		i,
			ncolumns,
			new_row;
	gchar **strv;
	GtkCList *clist;

	if(archiver == NULL)
	{
		edv_archive_object_delete(obj);
		return(-2);
	}

	clist = GTK_CLIST(archiver->contents_clist);
	ncolumns = MAX(clist->columns, 1);

	gtk_clist_freeze(clist);

	/* Allocate the row cell values */
	strv = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
	for(i = 0; i < ncolumns; i++)
		strv[i] = "";

	/* Append the new row */
	new_row = gtk_clist_append(
		clist,
		strv
	);

	/* Delete the row cell values */
	g_free(strv);

	/* Failed to appended row? */
	if(new_row < 0)
	{
		gtk_clist_thaw(clist);
		edv_archive_object_delete(obj);
		return(-1);
	}

	/* Set the new row's values */
	edv_archiver_list_set_row(
		archiver,
		new_row,
		obj
	);

	/* Set this object as the new row's data */
	gtk_clist_set_row_data_full(
		clist,
		new_row,
		obj, (GtkDestroyNotify)edv_archive_object_delete
	);
/*	obj = NULL; */

	gtk_clist_thaw(clist);

	return(new_row);
}

/*
 *	Appends the objects from the archive in the current location to
 *	the Contents List.
 *
 *	The arch_path specifies the full path to the archive.
 *
 *	The password specifies the password.
 *
 *	If show_progress is TRUE then the progress and messages will
 *	be displayed on the status bar during this operation.
 *
 *	All inputs assumed valid.
 */
static void edv_archiver_list_append_listing(
	EDVArchiver *archiver,
	const gchar *arch_path,
	const gchar *filter,
	const gchar *password,
	const gboolean show_progress
)
{
	GtkCList *clist = GTK_CLIST(archiver->contents_clist);
	EDVCore *core = archiver->core;
	EDVArchiverContentsListProgressData *d = EDV_ARCHIVER_CONTENTS_LIST_PROGRESS_DATA(g_malloc0(
		sizeof(EDVArchiverContentsListProgressData)
	));
	if(d == NULL)
		return;

	d->core = core;
	d->archiver = archiver;
	d->clist = clist;
	d->show_progress = show_progress;
	d->last_progress_percent = -1;

	/* Get the statistics of all the objects in the archive */
	gtk_clist_freeze(clist);
	edv_archive_object_stat_list_sequential(
		core->cfg_list,
		arch_path,
		NULL,				/* Get all the objects */
		filter,
		password,
		edv_archiver_list_progress_cb, d
	);
	gtk_clist_thaw(clist);

	g_free(d);
}

/*
 *	Finds the row by path.
 *
 *	The path specifies the string describing the path to match
 *	with a row's object path.
 *
 *	Returns the matched row or negative on error.
 */
gint edv_archiver_list_find_by_path(
	EDVArchiver *archiver, const gchar *path
)
{
	gint i, n;
	GtkCList *clist;
	EDVArchiveObject *obj;

	if((archiver == NULL) || (path == NULL))
		return(-2);

	clist = GTK_CLIST(archiver->contents_clist);

	n = clist->rows;
	for(i = 0; i < n; i++)
	{
		obj = EDV_ARCHIVE_OBJECT(gtk_clist_get_row_data(clist, i));
		if(obj == NULL)
			continue;

		if(obj->path == NULL)
			continue;

		if(!strcmp((const char *)obj->path, (const char *)path))
			return(i);
	}

	return(-1);
}


/*
 *	Updates all the existing rows on the Contents GtkCList by
 *	getting each row's archive object and updating the row's
 *	cell values with it.
 */
void edv_archiver_list_realize_listing(EDVArchiver *archiver)
{
	gint i, n;
	GtkCList *clist;
	CfgList *cfg_list;
	EDVCore *core;
	EDVArchiveObject *obj;

	if(archiver == NULL)
		return;

	clist = GTK_CLIST(archiver->contents_clist);
	core = archiver->core;
	cfg_list = core->cfg_list;

	gtk_clist_freeze(clist);

	/* Update the columns */
	edv_archiver_list_reset_columns(archiver);

	/* Update all the rows */
	n = clist->rows;
	for(i = 0; i < n; i++)
	{
		obj = EDV_ARCHIVE_OBJECT(gtk_clist_get_row_data(clist, i));
		if(obj == NULL)
			continue;

		/* Update this row's values */
		edv_archiver_list_set_row(
			archiver,
			i,			/* Row */
			obj
		);
	}

	/* Resize the columns */
	if(EDV_GET_B(EDV_CFG_PARM_ARCHIVER_CONTENTS_LIST_AUTO_RESIZE_COLUMNS))
		edv_archiver_list_resize_columns_optimul(archiver);

	gtk_clist_thaw(clist);
}


/*
 *	Clears the Contents List, resets the columns, and gets a new
 *	listing of archive objects.
 *
 *	If passive is TRUE then a list archive subprocess will be
 *	executed to get the list of objects in the archive asyncronously.
 *	Any current list archive subprocess will be stopped and deleted.
 *
 *	If show_progress is TRUE then the progress and messages will
 *	be displayed on the status bar during this operation.
 *
 *	Returns 0 on success or nonzero if there was an error.
 */
void edv_archiver_list_get(
	EDVArchiver *archiver,
	const gboolean passive,
	const gboolean show_progress,
	const gboolean show_comments
)
{
	gboolean auto_resize_columns;
	gchar		*cur_path,
			*filter,
			*password;
	GtkWidget *sb;
	GtkCList *clist;
	CfgList *cfg_list;
	EDVCore *core;

	if(archiver == NULL)
		return;

	/* Stop the current subprocess */
	if(archiver->subprocess != NULL)
	{
		edv_archiver_subprocess_delete(archiver->subprocess);
		archiver->subprocess = NULL;
	}

	clist = GTK_CLIST(archiver->contents_clist);
	sb = archiver->status_bar;
	core = archiver->core;
	cfg_list = core->cfg_list;

	auto_resize_columns = EDV_GET_B(EDV_CFG_PARM_ARCHIVER_CONTENTS_LIST_AUTO_RESIZE_COLUMNS);

	/* Clear all the error messages on the core */
	edv_clear_error_messages(core);

	/* Get the current archive to list */
	cur_path = STRDUP(edv_archiver_get_location(archiver));
	filter = STRDUP(archiver->contents_list_filter);
	password = STRDUP(edv_archiver_get_password(archiver));
	if((cur_path == NULL) || (password == NULL))
	{
		g_free(cur_path);
		g_free(filter);
		g_free(password);
		return;
	}

	/* Report the initial progress? */
	if(show_progress)
	{
		edv_status_bar_message(
			sb,
"Loading archive contents...",
			FALSE
		);
		edv_status_bar_progress(
			sb,
			0.0f,
			!passive
		);
	}

	gtk_clist_freeze(clist);

	/* Clear the list */
	edv_archiver_list_clear(archiver);

	/* Update the columns */
	edv_archiver_list_reset_columns(archiver);

	/* Automatically resize the columns to optimul widths based on
	 * their headings?
	 */
	if(auto_resize_columns)
		edv_archiver_list_resize_columns_optimul(archiver);

	/* Passive listing? */
	if(passive)
	{
		/* Passive listing involves call this program itself
		 * and instruct it to list the archive by using the
		 * --list-archive option which will instruct it to
		 * run in the "list archive mode" and print the list
		 * of archive objects to the standard output stream
		 * which we will read from using the Archiver's
		 * subprocess functions which connects the streams
		 * and monitors the process
		 */
		(void)edv_archiver_subprocess_start_list_archive(
			archiver,
			cur_path,
			filter,
			password,
			show_progress,
			show_comments
		);

		gtk_clist_thaw(clist);
	}
	else
	{
		/* Get the list of objects from the archive */
		edv_archiver_list_append_listing(
			archiver,
			cur_path,
			filter,
			password,
			show_progress
		);

		/* Automatically resize the columns to optimul
		 * widths based on their row values?
		 */
		if(auto_resize_columns)
			edv_archiver_list_resize_columns_optimul(archiver);

		gtk_clist_thaw(clist);

		/* Report the final progress? */
		if(show_progress)
		{
			edv_status_bar_message(
				sb,
"Archive contents loaded",
				FALSE
			);
			edv_status_bar_progress(
				sb,
				0.0f,
				!passive
			);
		}
	}

	g_free(cur_path);
	g_free(filter);
	g_free(password);
}

/*
 *	Deletes all rows in the Contents List.
 */
void edv_archiver_list_clear(EDVArchiver *archiver)
{
	GtkCList *clist;

	if(archiver == NULL)
		return;

	clist = GTK_CLIST(archiver->contents_clist);
	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);
	gtk_clist_thaw(clist);
}


/*
 *	Object added notify callback.
 */
void edv_archiver_list_vfs_object_added_cb(
	EDVArchiver *archiver,
	const gchar *path,
	EDVVFSObject *obj
)
{
	gchar *cur_arch_path;
	CfgList *cfg_list;
	EDVCore *core;

	if((archiver == NULL) || STRISEMPTY(path) || (obj == NULL))
		return;

	cur_arch_path = STRDUP(edv_archiver_get_location(archiver));
	if(cur_arch_path == NULL)
		return;

	core = archiver->core;
	cfg_list = core->cfg_list;

	/* Is the added path the same as the current archive? */
	if(!strcmp((const char *)path, (const char *)cur_arch_path))
	{
		/* Reget the listing */
		edv_archiver_list_get(
			archiver,
			TRUE,			/* Passive */
			EDV_GET_B(EDV_CFG_PARM_LISTS_SHOW_PROGRESS),
			FALSE			/* Do not show comments */
		);
	}

	g_free(cur_arch_path);
}

/*
 *	Object modified notify callback.
 */
void edv_archiver_list_vfs_object_modified_cb(
	EDVArchiver *archiver,
	const gchar *path,
	const gchar *new_path,
	EDVVFSObject *obj
)
{
	gchar *cur_arch_path;
	CfgList *cfg_list;
	EDVCore *core;

	if((archiver == NULL) || STRISEMPTY(path) || (obj == NULL))
		return;

	if(new_path == NULL)
		new_path = path;

	cur_arch_path = STRDUP(edv_archiver_get_location(archiver));
	if(cur_arch_path == NULL)
		return;

	core = archiver->core;
	cfg_list = core->cfg_list;

	/* Is the modified path the current archive? */
	if(!strcmp((const char *)new_path, (const char *)cur_arch_path))
	{
		/* Reget the listing */
		edv_archiver_list_get(
			archiver,
			TRUE,			/* Passive */
			EDV_GET_B(EDV_CFG_PARM_LISTS_SHOW_PROGRESS),
			FALSE			/* Do not show comments */
		);
	}

	g_free(cur_arch_path);
}

/*
 *	Object removed notify callback.
 */
void edv_archiver_list_vfs_object_removed_cb(
	EDVArchiver *archiver, const gchar *path
)
{
	gchar *cur_arch_path;

	if(archiver == NULL)
		return;

	cur_arch_path = STRDUP(edv_archiver_get_location(archiver));
	if(cur_arch_path == NULL)
		return;

	/* Is the removed path the same as the current archive? */
	if(!strcmp((const char *)path, (const char *)cur_arch_path))
	{
		/* Clear the listing */
		edv_archiver_list_clear(archiver);
	}

	g_free(cur_arch_path);
}


/*
 *	Archive object added notify.
 *
 *	The path specifies the previous path of the object in the
 *	archive.
 *
 *	The path specifies the path of the added object in the
 *	archive.
 *
 *	The obj specifies the values of the added object in the
 *	archive.
 */
void edv_archiver_list_archive_object_added_cb(
	EDVArchiver *archiver, const gchar *arch_path,
	const gchar *path,
	EDVArchiveObject *obj
)
{
	gint row;
	gchar *cur_arch_path;
	GtkCList *clist;
	CfgList *cfg_list;
	EDVCore *core;

	if((archiver == NULL) || STRISEMPTY(arch_path) ||
	   STRISEMPTY(path) || (obj == NULL)
	)
		return;

	cur_arch_path = STRDUP(edv_archiver_get_location(archiver));
	if(cur_arch_path == NULL)
		return;

	clist = GTK_CLIST(archiver->contents_clist);
	core = archiver->core;
	cfg_list = core->cfg_list;

	/* Not our archive? */
	if(strcmp((const char *)arch_path, (const char *)cur_arch_path))
	{
		g_free(cur_arch_path);
		return;
	}

	/* Check if the path of the added archive object describes an
	 * archive object that is already in the list
	 */
	row = edv_archiver_list_find_by_path(archiver, path);
	if(row > -1)
	{
		EDVArchiveObject *cur_obj = EDV_ARCHIVE_OBJECT(gtk_clist_get_row_data(
			clist,
			row
		));
		if(cur_obj != NULL)
		{
			/* Update this object */
			edv_archive_object_set_object(
				cur_obj,			/* Target */
				obj				/* Source */
			);

			gtk_clist_freeze(clist);

			/* Update the row cell values */
			edv_archiver_list_set_row(
				archiver,
				row,
				cur_obj
			);

			/* Resize the columns */
			if(EDV_GET_B(EDV_CFG_PARM_ARCHIVER_CONTENTS_LIST_AUTO_RESIZE_COLUMNS))
				edv_archiver_list_resize_columns_optimul(archiver);

			gtk_clist_thaw(clist);
		}
	}
	else
	{
		/* Append this object to the list */
		EDVArchiveObject *new_obj = edv_archive_object_copy(obj);
		if(new_obj != NULL)
		{
			(void)edv_archiver_list_append(
				archiver,
				new_obj
			);

			/* Resize the columns */
			if(EDV_GET_B(EDV_CFG_PARM_ARCHIVER_CONTENTS_LIST_AUTO_RESIZE_COLUMNS))
				edv_archiver_list_resize_columns_optimul(archiver);
		}
	}

	g_free(cur_arch_path);
}

/*
 *	Archive object modified notify.
 *
 *	The arch_path specifies the full_path to the archive.
 *
 *	The path specifies the previous path of the object in the
 *	archive.
 *
 *	The new_path specifies the new path of the object in the
 *	archive.
 *
 *	The obj specifies the values for the modified object in the
 *	archive.
 */
void edv_archiver_list_archive_object_modified_cb(
	EDVArchiver *archiver, const gchar *arch_path,
	const gchar *path, const gchar *new_path,
	EDVArchiveObject *obj
)
{
	gint row;
	gchar *cur_arch_path;
	GtkCList *clist;
	CfgList *cfg_list;
	EDVCore *core;

	if((archiver == NULL) || STRISEMPTY(arch_path) ||
	   STRISEMPTY(path) || (obj == NULL)
	)
		return;

	if(new_path == NULL)
		new_path = path;

	cur_arch_path = STRDUP(edv_archiver_get_location(archiver));
	if(cur_arch_path == NULL)
		return;

	clist = GTK_CLIST(archiver->contents_clist);
	core = archiver->core;
	cfg_list = core->cfg_list;

	/* Not our archive? */
	if(strcmp((const char *)arch_path, (const char *)cur_arch_path))
	{
		g_free(cur_arch_path);
		return;
	}

	/* Check if the old path of the modified archive object is in
	 * the listing
	 */
	row = edv_archiver_list_find_by_path(archiver, path);
	if(row > -1)
	{
		EDVArchiveObject *cur_obj = EDV_ARCHIVE_OBJECT(gtk_clist_get_row_data(
			clist,
			row
		));
		if(cur_obj != NULL)
		{
			/* Update this object */
			edv_archive_object_set_object(
				cur_obj,			/* Target */
				obj				/* Source */
			);

			gtk_clist_freeze(clist);

			/* Update the row cell values */
			edv_archiver_list_set_row(
				archiver,
				row,
				cur_obj
			);

			/* Resize the columns */
			if(EDV_GET_B(EDV_CFG_PARM_ARCHIVER_CONTENTS_LIST_AUTO_RESIZE_COLUMNS))
				edv_archiver_list_resize_columns_optimul(archiver);

			gtk_clist_thaw(clist);
		}
	}

	g_free(cur_arch_path);
}

/*
 *	Archive object removed notify.
 *
 *	The arch_path specifies the full_path to the archive.
 *
 *	The path specifies the object in the archive that has been
 *	removed.
 */
void edv_archiver_list_archive_object_removed_cb(
	EDVArchiver *archiver, const gchar *arch_path,
	const gchar *path
)
{
	gint row;
	gchar *cur_arch_path;
	GtkCList *clist;
	CfgList *cfg_list;
	EDVCore *core;

	if((archiver == NULL) || STRISEMPTY(arch_path) ||
	   STRISEMPTY(path)
	)
		return;

	cur_arch_path = STRDUP(edv_archiver_get_location(archiver));
	if(cur_arch_path == NULL)
		return;

	clist = GTK_CLIST(archiver->contents_clist);
	core = archiver->core;
	cfg_list = core->cfg_list;

	/* Not our archive? */
	if(strcmp((const char *)arch_path, (const char *)cur_arch_path))
	{
		g_free(cur_arch_path);
		return;
	}

	/* Remove all the rows who's object's path matches the removed
	 * path
	 */
	row = edv_archiver_list_find_by_path(archiver, path);
	if(row > -1)
	{
		gtk_clist_freeze(clist);
		do {
			gtk_clist_remove(clist, row);
			row = edv_archiver_list_find_by_path(archiver, path);
		} while(row > -1);
		gtk_clist_thaw(clist);

		/* Resize the columns */
		if(EDV_GET_B(EDV_CFG_PARM_ARCHIVER_CONTENTS_LIST_AUTO_RESIZE_COLUMNS))
			edv_archiver_list_resize_columns_optimul(archiver);
	}

	g_free(cur_arch_path);
}
