#include <errno.h>
#include <gtk/gtk.h>

#include "url.h"

#include "cdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_archive_obj.h"
#include "edv_utils_gtk.h"
#include "edv_status_bar.h"
#include "archiver.h"
#include "archiver_dnd.h"
#include "edv_dnd.h"

#include "config.h"


static gint EDVArchiverDragDataReceivedNexus(
	EDVArchiver *archiver,
	GdkDragContext *dc,
	const guint info,
	GtkSelectionData *selection_data
);

gint edv_archiver_location_bar_icon_crossing_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);

void edv_archiver_list_drag_data_get_cb(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);
void edv_archiver_list_drag_data_received_cb(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);
void edv_archiver_list_drag_data_delete_cb(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Drag data received nextus.
 *
 *	The dc specifies the GdkDragContext.
 *
 *	The info specifies the source object's location type,
 *
 *	The selection_data specifies the DND data describing the
 *	source objects in the form of a null-separated list of
 *	URL strings which will be parsed into a list of URL strings.
 *
 *      Returns 0 on success or non-zero on error.
 */
static gint EDVArchiverDragDataReceivedNexus(
	EDVArchiver *archiver,
	GdkDragContext *dc,
	const guint info,
	GtkSelectionData *selection_data
)
{
	gint status;
	gchar		*arch_path,
			*password;
	GtkWidget *toplevel = archiver->toplevel;
	EDVCore *core = archiver->core;

	/* Decode the data into a list of URLs */
	GList *urls_list = url_decode(
		(const guint8 *)selection_data->data,
		selection_data->length
	);
	const gint nurls = g_list_length(urls_list);

	edv_archiver_set_busy(archiver, TRUE);

#define CLEANUP_RETURN(_v_)	{			\
 const gint error_code = (gint)errno;			\
							\
 if(urls_list != NULL) {				\
  g_list_foreach(					\
   urls_list,						\
   (GFunc)url_delete,					\
   NULL							\
  );							\
  g_list_free(urls_list);				\
 }							\
							\
 errno = (int)error_code;				\
							\
 return(_v_);						\
}

	/* Check if there is exactly one URL and if it is an archive,
	 * in which case we should give the user a choice to open
	 * this archive instead of adding it to the archive
	 */
	if(nurls == 1)
	{
		URLStruct *url = URL(g_list_nth_data(
			urls_list,
			0
		));
		if(url != NULL)
		{
			if(url->path != NULL)
			{
				const gchar *name = g_basename(url->path);
				if(edv_check_archive_format_supported(core, name))
				{
					/* There is exactly one DND object reference
					 * and if is an archive, so query user about
					 * opening it instead of adding it to the
					 * current archive
					 */
					gint response;
					gchar *msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"El archivo abierto \"%s\" en vez de agregarlo al\n\
archivo actual?\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"\"%s\" ouvert d'archive au lieu d'ajouter il \n\
l'archive actuelle?\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Offenes archiv \"%s\" statt hinzufgen es zum\n\
jetzigen archiv?\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"\"%s\" di archivio aperto invece dell'aggiunta esso\n\
all'archivio attuale?\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Open archief \"%s\" in plaats van toevoegen het aan het\n\
huidig archief?\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"\"%s\" aberto de arquivo em vez de adicionar ele ao\n\
arquivo atual?\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"pen arkiv \"%s\" i stedet for  tilfye det til det\n\
nvrendee arkivet?\n"
#elif defined(PROG_LANGUAGE_POLISH)
"Otworzy archiwum \"%s\" zamiast dodawania go do\n\
aktualnego archiwum?\n"
#else
"Open archive \"%s\" instead of adding it to the\n\
current archive?\n"
#endif
						,
						name
					);
					edv_play_sound_question(core);
					CDialogSetTransientFor(toplevel);
					response = CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"El Archivo abierto?"
#elif defined(PROG_LANGUAGE_FRENCH)
"L'Archive ouverte?"
#elif defined(PROG_LANGUAGE_GERMAN)
"Offenes Archiv?"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Archivio Aperta?"
#elif defined(PROG_LANGUAGE_DUTCH)
"Open Archief?"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Arquivo Aberto?"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"pent Arkiv?"
#elif defined(PROG_LANGUAGE_POLISH)
"Otworzy archiwum?"
#else
"Open Archive?"
#endif
						,
						msg,
						NULL,
						CDIALOG_ICON_QUESTION,
						CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
						CDIALOG_BTNFLAG_YES
					);
					g_free(msg);
					CDialogSetTransientFor(NULL);
					if(response == CDIALOG_RESPONSE_YES)
					{
						/* Open the archive */
						(void)edv_archiver_open_archive(
							archiver,
							url->path,
							edv_archiver_get_password(archiver),
							TRUE	/* List passively */
						);
						edv_archiver_set_busy(archiver, FALSE);
						CLEANUP_RETURN(0);
					}
				}
			}
		}
	}

	g_list_foreach(
		urls_list,
		(GFunc)url_delete,
		NULL
	);
	g_list_free(urls_list);
	urls_list = NULL;

	/* Get the current archive path and password */
	arch_path = STRDUP(edv_archiver_get_location(archiver));
	password = STRDUP(edv_archiver_get_password(archiver));

	/* Process the DND operation to the VFS */
	status = edv_dnd_vfs_to_archive(
		core,
		dc,
		info,				/* Source location type */
		selection_data,			/* Source objects */
		arch_path,			/* Target archive path */
		&password,			/* May be modified */
		TRUE,				/* Verbose */
		toplevel,
		archiver->status_bar
	);

	g_free(arch_path);
	g_free(password);

	edv_archiver_set_busy(archiver, FALSE);

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}


/*
 *	Location Bar icon "enter_notify_event" or "leave_notify_event"
 *	signal callback.
 */
gint edv_archiver_location_bar_icon_crossing_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	gint status = FALSE;
	GtkWidget *sb;
	EDVArchiver *archiver = EDV_ARCHIVER(data);
	if((widget == NULL) || (crossing == NULL) || (archiver == NULL))
		return(status);

	sb = archiver->status_bar;

	switch((gint)crossing->type)
	{
	  case GDK_ENTER_NOTIFY:
		if(archiver->location_icon_pm != NULL)
		{
			GdkBitmap *mask;
			GdkPixmap *pixmap;
			GtkWidget *w = archiver->location_icon_pm;
			gtk_pixmap_get(GTK_PIXMAP(w), &pixmap, &mask);
			if(pixmap != NULL)
				GUIDNDSetDragIcon(
					widget,
					pixmap, mask,
					GUI_DND_DRAG_ICON_DEF_HOT_X, GUI_DND_DRAG_ICON_DEF_HOT_Y
				);
		}
		edv_status_bar_message(
			sb,
#if defined(PROG_LANGUAGE_SPANISH)
"Arrastre esto crear una conexin a esta ubicacin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Traner ceci pour crer un lien  cet emplacement"
#elif defined(PROG_LANGUAGE_GERMAN)
"Schleppen sie dies, eine verbindung zu diesem ort zu schaffen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Trascinare questo per creare una maglia a questa posizione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Sleep dit om een schakel aan deze plaats te creren"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Arraste isto criar um elo a esta localidade"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Slep dette skape et ledd til denne plasseringen"
#elif defined(PROG_LANGUAGE_POLISH)
"Przecignij to by utworzy odnonik do tej lokacji"
#else
"Drag this to create a link to this location"
#endif
			, FALSE
		);
		status = TRUE;
		break;

	  case GDK_LEAVE_NOTIFY:
		edv_status_bar_message(sb, NULL, FALSE);
		status = TRUE;
		break;
	}

	return(status);
}


/*
 *	Contents GtkCList "drag_data_get" signal callback.
 */
void edv_archiver_list_drag_data_get_cb(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	gboolean data_sent = FALSE;
	GList		*glist,
					*urls_list = NULL;
	GtkWidget *w;
	GtkCList *clist;
	URLStruct *url;
	EDVArchiveObject *obj;
	EDVCore *core;
	EDVArchiver *archiver = EDV_ARCHIVER(data);
	if((dc == NULL) || (archiver == NULL))
		return;

	w = archiver->contents_clist;
	clist = GTK_CLIST(w);
	core = archiver->core;

	edv_archiver_sync_data(archiver);

	/* Add the first URL as the archive itself */
	url = url_new();
	if(url == NULL)
		return;

	url->path = STRDUP(edv_archiver_get_location(archiver));
	urls_list = g_list_append(
		urls_list,
		url
	);

	/* Add subsequent URLs from the selected rows */
	for(glist = clist->selection;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		obj = EDV_ARCHIVE_OBJECT(gtk_clist_get_row_data(
			clist,
			(gint)glist->data
		));
		if(obj == NULL)
			continue;

		if(STRISEMPTY(obj->path))
			continue;

		url = url_new();
		if(url == NULL)
			break;

		url->path = g_strdup(obj->path);
		urls_list = g_list_append(
			urls_list,
			url
		);
	}

	/* Encode DDE buffer from the URL list */
	if(urls_list != NULL)
	{
		gint buf_len;
		guint8 *buf = url_encode(
			urls_list,
			&buf_len
		);
		if(buf != NULL)
		{
			/* Send out buffer */
			gtk_selection_data_set(
				selection_data,
				GDK_SELECTION_TYPE_STRING,
				8,				/* Bits Per Character */
				buf,			/* Data */
				buf_len			/* Length */
			);
			data_sent = TRUE;
			g_free(buf);
		}
	}

	/* Delete the URLs list */
	g_list_foreach(
		urls_list,
		(GFunc)url_delete,
		NULL
	);
	g_list_free(urls_list);

	/* If failed to send out buffer then send out error response */
	if(!data_sent)
	{
		const char *s = "Error";
		gtk_selection_data_set(
			selection_data,
			GDK_SELECTION_TYPE_STRING,
			8,				/* Bits Per Character */
			s,				/* Data */
			STRLEN(s)			/* Length */
		);
		data_sent = TRUE;
	}
}

/*
 *	Contents GtkCList "drag_data_received" signal callback.
 */
void edv_archiver_list_drag_data_received_cb(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	EDVArchiver *archiver = EDV_ARCHIVER(data);
	if((dc == NULL) || (archiver == NULL))
		return;

	if(EDV_ARCHIVER_IS_PROCESSING(archiver) || (archiver->freeze_count > 0))
		return;

	/* No source data? */
	if(selection_data == NULL)              
		return;
	if(selection_data->length <= 0)
		return;

	edv_archiver_sync_data(archiver);

	/* Perform the drag operation */
	(void)EDVArchiverDragDataReceivedNexus(
		archiver,
		dc,
		info,
		selection_data
	);
}

/*
 *	Contents GtkCList "drag_data_delete" signal callback.
 */
void edv_archiver_list_drag_data_delete_cb(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
)
{

}
