#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "cfg.h"

#include "guiutils.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_archive_obj.h"
#include "edv_list_seek.h"
#include "edv_utils_gtk.h"
#include "edv_status_bar.h"
#include "archiver.h"
#include "archiver_cb.h"
#include "archiver_op.h"
#include "archiver_dnd.h"
#include "archiver_list.h"
#include "archiver_subprocess.h"
#include "edv_cb.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_endeavour_archiver_48x48.xpm"


/* GTK+ Signal Callbacks */
void edv_archiver_realize_cb(GtkWidget *widget, gpointer data);
gint edv_archiver_delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
gint edv_archiver_key_event_cb(
	 GtkWidget *widget, GdkEventKey *key, gpointer data
);
gint edv_archiver_button_press_event_cb(
	GtkWidget *widget, GdkEventButton *button, gpointer data
);

void edv_archiver_handle_child_attached_cb(
	GtkHandleBox *handle_box, GtkWidget *child, gpointer data
);
void edv_archiver_handle_child_detached_cb(
	GtkHandleBox *handle_box, GtkWidget *child, gpointer data
);

void edv_archiver_location_bar_icon_realize_cb(GtkWidget *widget, gpointer data);

static gint edv_archiver_column_sort_index_cb(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint edv_archiver_column_sort_size_cb(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint edv_archiver_column_sort_storage_size_cb(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint edv_archiver_column_sort_compression_cb(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint edv_archiver_column_sort_date_nexus(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2,
	gint sort_code
);
static gint edv_archiver_column_sort_date_access_cb(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint edv_archiver_column_sort_date_modify_cb(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint edv_archiver_column_sort_date_change_cb(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
void edv_archiver_resize_column_cb(
	GtkCList *clist, gint column, gint width, gpointer data
);
void edv_archiver_click_column_cb(
	GtkCList *clist, gint column, gpointer data
);
void edv_archiver_select_row_cb(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
void edv_archiver_unselect_row_cb(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);

void edv_archiver_combo_activate_cb(GtkWidget *widget, gpointer data);

/* Menu Item Callbacks */
void edv_archiver_menu_item_activate_cb(GtkWidget *widget, gpointer data);
gint edv_archiver_menu_item_enter_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);
gint edv_archiver_menu_item_leave_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);

/* Find Bar Callbacks */
const gchar *edv_archiver_find_bar_location_cb(
	GtkWidget *bar,
	gpointer data
);
void edv_archiver_find_bar_start_cb(
	GtkWidget *bar,
	gpointer data
);
void edv_archiver_find_bar_end_cb(
	GtkWidget *bar,
	const gint nmatches,
	gpointer data
);
void edv_archiver_find_bar_match_cb(
	GtkWidget *bar,
	const gchar *path,
	GList *properties_list,
	const gint line_index,
	const gchar *excerpt,
	gpointer data
);

/* Status Bar Callbacks */
void edv_archiver_status_message_cb(
	GtkWidget *widget,
	const gchar *message, 
	gpointer data
);
void edv_archiver_status_progress_cb(
	GtkWidget *widget,
	const gfloat progress, 
	gpointer data
);

/* Window Created/Deleted Callbacks */
void edv_archiver_window_created_cb(
	EDVArchiver *archiver,
	const EDVWindowType win_type,
	gpointer win
);
void edv_archiver_window_deleted_cb(
	EDVArchiver *archiver,
	const EDVWindowType win_type,
	gpointer win
);

/* Master Write Protect Changed Callback */
void edv_archiver_write_protect_changed_cb(
	EDVArchiver *archiver,
	const gboolean state
);

/* Delete Method Changed Callback */
void edv_archiver_delete_method_changed_cb(
	EDVArchiver *archiver,
	const EDVDeleteMethod delete_method
);

/* Object Callbacks */
void edv_archiver_object_added_cb(
	EDVArchiver *archiver,
	const gchar *path,
	EDVVFSObject *obj
);
void edv_archiver_object_modified_cb(
	EDVArchiver *archiver,
	const gchar *path,
	const gchar *new_path,
	EDVVFSObject *obj
);
void EDVArchiverObjectRemovedNotifyCB(
	EDVArchiver *archiver,
	const gchar *path
);

/* Archive Object Callbacks */
void edv_archiver_archive_object_added_cb(
	EDVArchiver *archiver,
	const gchar *arch_path,
	const gchar *path,
	EDVArchiveObject *obj
);
void edv_archiver_archive_object_modified_cb(
	EDVArchiver *archiver,
	const gchar *arch_path,
	const gchar *path,
	const gchar *new_path,
	EDVArchiveObject *obj
);
void edv_archiver_archive_object_removed_cb(
	EDVArchiver *archiver,
	const gchar *arch_path,
	const gchar *path
);

/* Recycled Object Callbacks */
void edv_archiver_recycled_object_added_cb(
	EDVArchiver *archiver,
	const guint index
);
void edv_archiver_recycled_object_modified_cb(
	EDVArchiver *archiver,
	const guint index
);
void edv_archiver_recycled_object_removed_cb(
	EDVArchiver *archiver,
	const guint index
);

/* Reconfigured Callback */
void edv_archiver_reconfigured_cb(EDVArchiver *archiver);

/* MIME Type Callbacks */
void edv_archiver_mime_type_added_cb(
	EDVArchiver *archiver,
	const gint mt_num, EDVMIMEType *mt
);
void edv_archiver_mime_type_modified_cb(
	EDVArchiver *archiver,
	const gint mt_num, EDVMIMEType *mt
);
void edv_archiver_mime_type_removed_cb(
	EDVArchiver *archiver,
	const gint mt_num
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Toplevel GtkWindow "realize" signal callback.
 */
void edv_archiver_realize_cb(GtkWidget *widget, gpointer data)
{
	GdkWindow *window;
	EDVArchiver *archiver = EDV_ARCHIVER(data);
	if((widget == NULL) || (archiver == NULL))
		return;

	window = widget->window;
	if(window != NULL)
	{
		GdkGeometry geo;
		geo.min_width = 320;
		geo.min_height = 240;
		geo.base_width = 0;
		geo.base_height = 0;
		geo.width_inc = 1;
		geo.height_inc = 1;
		gdk_window_set_geometry_hints(
			window,
			&geo,
			GDK_HINT_MIN_SIZE | GDK_HINT_BASE_SIZE |
				GDK_HINT_RESIZE_INC
		);
		gdk_window_set_hints(
			window,
			0, 0,                           /* Position */
			geo.min_width, geo.min_height,  /* Min size */
			geo.max_width, geo.max_height,  /* Max size */
			GDK_HINT_MIN_SIZE
		);
		GUISetWMIcon(
			window,
			(guint8 **)icon_endeavour_archiver_48x48_xpm
		);
	}

	archiver->flags |= EDV_ARCHIVER_REALIZED; 
}

/*
 *	Toplevel GtkWindow "delete_event" signal callback.
 */
gint edv_archiver_delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	EDVArchiver *archiver = EDV_ARCHIVER(data);
	if(archiver == NULL)
		return(FALSE);

	if(EDV_ARCHIVER_IS_PROCESSING(archiver) || (archiver->freeze_count > 0))
		return(TRUE);

	edv_archiver_op_close(archiver);

	return(TRUE);
}

/*
 *	Any GtkWidget "key_press_event" or "key_release_event" signal
 *	callback.
 */
gint edv_archiver_key_event_cb(
	 GtkWidget *widget, GdkEventKey *key, gpointer data
)
{
	gint status = FALSE;
	gboolean is_press;
	gint etype;
	guint keyval, state;
	GtkWidget *w, *focus_widget;
	CfgList *cfg_list;
	EDVCore *core;
	EDVArchiver *archiver = EDV_ARCHIVER(data);
	if((widget == NULL) || (key == NULL) || (archiver == NULL))
		return(status);

	if(EDV_ARCHIVER_IS_PROCESSING(archiver) || (archiver->freeze_count > 0))
		return(status);

	w = archiver->toplevel;
	focus_widget = GTK_WINDOW(w)->focus_widget;
	core = archiver->core;
	cfg_list = core->cfg_list;

	etype = key->type;
	is_press = (etype == GDK_KEY_PRESS) ? TRUE : FALSE;
	keyval = key->keyval;
	state = key->state;

#define STOP_SIGNAL_EMISSION(_w_)	{	\
 gtk_signal_emit_stop_by_name(			\
  GTK_OBJECT(_w_),				\
  is_press ?					\
   "key_press_event" : "key_release_event"	\
 );						\
}

	/* If the focus_widget is not a GtkEditable then check if the
	 * keyval is an accelerator key before all subsequence checks
	 */
	if((focus_widget != NULL) ?
		!GTK_IS_EDITABLE(focus_widget) : TRUE
	)
	{
		EDVArchiverOpID id = (EDVArchiverOpID)edv_match_accel_key_op_id(
			cfg_list, EDV_CFG_PARM_ARCHIVER_ACCELERATOR_KEYS,
			keyval, state
		);
		if(id > 0)
		{
			if(is_press)
			{
				EDVArchiverOp *op = edv_archiver_op_match_by_id(
					archiver,
					id
				);
				if(op != NULL)
					edv_archiver_op_activate_cb(NULL, -1, op);
			}
			STOP_SIGNAL_EMISSION(widget);
			status = TRUE;
			return(status);
		}
	}

	/* Check which widget this signal is for
	 *
	 * Contents GtkCList
	 */
	if(widget == archiver->contents_clist)
	{
		GtkCList *clist = GTK_CLIST(widget);
		gint row = edv_clist_get_selected_last(clist, NULL);

		/* Handle by key value */
		switch(keyval)
		{
		  case GDK_space:
		  case GDK_KP_Space:
			row = clist->focus_row;
			if((row >= 0) && (row < clist->rows) && is_press)
			{
				gboolean already_selected = FALSE;

				/* Check if this row is already selected */
				GList *glist = clist->selection;
				while(glist != NULL)
				{
					if(row == (gint)glist->data)
					{
						already_selected = TRUE;
						break;
					}
					glist = g_list_next(glist);
				}

				gtk_clist_freeze(clist);
				if(already_selected)
					gtk_clist_unselect_row(clist, row, 0);
				else
					gtk_clist_select_row(clist, row, 0);
				gtk_clist_thaw(clist);
			}
			STOP_SIGNAL_EMISSION(widget);
			status = TRUE;
			break;

		  default:
			/* For all other alphanumeric character keys and while
			 * no modifier keys are held, attempt to seek to the
			 * item who's name starts with the letter of the key
			 * that was pressed
			 */
			if(EDV_GET_B(EDV_CFG_PARM_LISTS_KEYBOARD_SCROLL_TO_KEY_NAME) &&
				isalnum((int)keyval) && is_press
			)
			{
				const gboolean backwards = (state & GDK_MOD1_MASK) ? TRUE : FALSE;
				const gint start_row = (backwards) ?
					(clist->rows - 1) : 0;

				/* Find the column that is set to display the name */
				gint column_num_name = edv_archiver_list_get_column_index_by_type(
					archiver,
					EDV_ARCHIVER_COLUMN_TYPE_NAME
				);
				if(column_num_name < 0)
					column_num_name = 0;

				/* Seek to the row who's text matches this
				 * key event's character
				 */
				gtk_clist_freeze(clist);
				edv_clist_seek_character(
					clist,
					column_num_name,	/* Column */
					start_row,
					backwards,
					0,			/* Text index */
					keyval			/* Character */
				);
				gtk_clist_thaw(clist);

				STOP_SIGNAL_EMISSION(widget);
				status = TRUE;
			}
			break;
		}
	}

#undef STOP_SIGNAL_EMISSION

	return(status);
}

/*
 *	Any GtkWidget "button_press_event" signal callback.
 */
gint edv_archiver_button_press_event_cb(
	GtkWidget *widget, GdkEventButton *button, gpointer data
)
{
	gint		etype,
					status = FALSE;
	CfgList *cfg_list;
	EDVCore *core;
	EDVArchiver *archiver = EDV_ARCHIVER(data);
	if((widget == NULL) || (button == NULL) || (archiver == NULL))
		return(status);

	if(EDV_ARCHIVER_IS_PROCESSING(archiver) || (archiver->freeze_count > 0))
		return(status);

	etype = (gint)button->type;
	core = archiver->core;
	cfg_list = core->cfg_list;

	/* Check which widget this signal is for
	 *
	 * Contents GtkCList
	 */
	if(widget == archiver->contents_clist)
	{
		gint	row, column,
					rows_selected = 0, selected_row = -1;
		GList *glist;
		GtkCList *clist = GTK_CLIST(widget);

		/* Find row and column based on given coordinates */
		if(!gtk_clist_get_selection_info(
			clist, button->x, button->y, &row, &column
		))
		{
			row = -1;
			column = 0;
		}

		/* Get the number of selected rows and the last selected row */
		glist = clist->selection;
		while(glist != NULL)
		{
			rows_selected++;
			selected_row = (gint)glist->data;
			glist = g_list_next(glist);
		}

		/* Handle by button number */
		switch(button->button)
		{
		  case GDK_BUTTON2:
			switch((EDVListsPointerOpButton2)EDV_GET_I(EDV_CFG_PARM_LISTS_POINTER_OP_BUTTON2))
			{
			  case EDV_LISTS_POINTER_OP_BUTTON2_NONE:
			  case EDV_LISTS_POINTER_OP_BUTTON2_SCROLL_XY:
				/* Generic list callback handles this */
				break;
			  case EDV_LISTS_POINTER_OP_BUTTON2_RENAME:
				if(etype == GDK_BUTTON_PRESS)
				{
					/* Rename */

				}
				status = TRUE;
				break;
			  case EDV_LISTS_POINTER_OP_BUTTON2_PASTE:
				if(etype == GDK_BUTTON_PRESS)
				{
					/* Paste objects */

				}
				status = TRUE;
				break;
			}
			break;

		  case GDK_BUTTON3:
			if(etype == GDK_BUTTON_PRESS)
			{
				/* Select item before mapping menu? */
				if(EDV_GET_B(EDV_CFG_PARM_RIGHT_CLICK_MENU_SELECTS) &&
					(row >= 0) && (row < clist->rows))
				{
					/* Select the row that the button was pressed over.
					 * if no key modifiers are held then this will also
					 * unselect all previously selected rows.
					 */
					gtk_clist_freeze(clist);
					if(!(button->state & GDK_CONTROL_MASK) &&
					   !(button->state & GDK_SHIFT_MASK)
					)
						gtk_clist_unselect_all(clist);
					clist->focus_row = row;
					gtk_clist_select_row(clist, row, 0);
					gtk_clist_thaw(clist);
				}

				/* Update all menus and map right click menu */
				edv_archiver_update_display(archiver);
				gtk_menu_popup(
					GTK_MENU(archiver->contents_clist_menu),
					NULL, NULL,
					NULL, NULL,
					button->button, button->time
				);
			}
			status = TRUE;
			break;
		}
	}

	return(status);
}


/*
 *	GtkHandleBox "child_attached" signal callback.
 */
void edv_archiver_handle_child_attached_cb(
	GtkHandleBox *handle_box, GtkWidget *child, gpointer data
)
{
	EDVArchiver *archiver = EDV_ARCHIVER(data);
	if((handle_box == NULL) || (archiver == NULL))
		return;

	gtk_widget_queue_resize(
		gtk_widget_get_toplevel(GTK_WIDGET(handle_box))
	);
}

/*
 *	GtkHandleBox "child_detached" signal callback.
 */
void edv_archiver_handle_child_detached_cb(
	GtkHandleBox *handle_box, GtkWidget *child, gpointer data
)
{
	EDVArchiver *archiver = EDV_ARCHIVER(data);
	if((handle_box == NULL) || (archiver == NULL))
		return;

	gtk_widget_queue_resize(
		gtk_widget_get_toplevel(GTK_WIDGET(handle_box))
	);
}


/*
 *	Location Bar Icon GtkFixed "realize" signal callback.
 */
void edv_archiver_location_bar_icon_realize_cb(GtkWidget *widget, gpointer data)
{
	GdkCursor *cursor;
	GdkWindow *window;
	EDVCore *core;
	EDVArchiver *archiver = EDV_ARCHIVER(data);
	if((widget == NULL) || (archiver == NULL))
		return;

	window = widget->window;
	core = archiver->core;

	cursor = edv_get_cursor(core, EDV_CURSOR_CODE_HAND);
	gdk_window_set_cursor(window, cursor);
}


/*
 *	Contents GtkCList column sort index callback.
 */
static gint edv_archiver_column_sort_index_cb(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	EDVArchiveObject	*obj1 = EDV_ARCHIVE_OBJECT(
			((const GtkCListRow *)ptr1)->data
	),
									*obj2 = EDV_ARCHIVE_OBJECT(
			((const GtkCListRow *)ptr2)->data
	);
	if((obj1 == NULL) || (obj2 == NULL))
		return(0);

	if(obj1->index <= obj2->index)
		return(obj1->index < obj2->index);
	else
		return(-1);
}

/*
 *	Contents GtkCList column sort size callback.
 */
static gint edv_archiver_column_sort_size_cb(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	EDVArchiveObject	*obj1 = EDV_ARCHIVE_OBJECT(
			((const GtkCListRow *)ptr1)->data
	),
									*obj2 = EDV_ARCHIVE_OBJECT(
			((const GtkCListRow *)ptr2)->data
	);
	if((obj1 == NULL) || (obj2 == NULL))
		return(0);

	if(obj1->size <= obj2->size)
		return(obj1->size < obj2->size);
	else
		return(-1);
}

/*
 *	Contents GtkCList column sort storage size callback.
 */
static gint edv_archiver_column_sort_storage_size_cb(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	EDVArchiveObject	*obj1 = EDV_ARCHIVE_OBJECT(
			((const GtkCListRow *)ptr1)->data
	),
							*obj2 = EDV_ARCHIVE_OBJECT(
			((const GtkCListRow *)ptr2)->data
	);
	if((obj1 == NULL) || (obj2 == NULL))
		return(0);

	if(obj1->storage_size <= obj2->storage_size)
		return(obj1->storage_size < obj2->storage_size);
	else
		return(-1);
}

/*
 *	Contents GtkCList column sort compression ratio callback.
 */
static gint edv_archiver_column_sort_compression_cb(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	EDVArchiveObject	*obj1 = EDV_ARCHIVE_OBJECT(
			((const GtkCListRow *)ptr1)->data
	),
									*obj2 = EDV_ARCHIVE_OBJECT(
			((const GtkCListRow *)ptr2)->data
	);
	if((clist == NULL) || (obj1 == NULL) || (obj2 == NULL))
		return(0);

	if(obj1->compression_ratio <= obj2->compression_ratio)
		return(obj1->compression_ratio < obj2->compression_ratio);
	else
		return(-1);
}

/*
 *	Contents GtkCList column sort date nexus.
 *
 *	The archive object structures are obtained from each row and
 *	the dates are compared.
 */
static gint edv_archiver_column_sort_date_nexus(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2,
	gint sort_code
)
{
	EDVArchiveObject	*obj1 = EDV_ARCHIVE_OBJECT(
			((const GtkCListRow *)ptr1)->data
	),
									*obj2 = EDV_ARCHIVE_OBJECT(
			((const GtkCListRow *)ptr2)->data
	);
	if((clist == NULL) || (obj1 == NULL) || (obj2 == NULL))
		return(0);

	/* Handle by the sort code */
	switch(sort_code)
	{
	  case 0:	/* Access time */
		if(obj1->access_time <= obj2->access_time)
			return(obj1->access_time < obj2->access_time);
		else
			return(-1);
		break;

	  case 1:	/* Modify time */
		if(obj1->modify_time <= obj2->modify_time)
			return(obj1->modify_time < obj2->modify_time);
		else
			return(-1);
		break;

	  case 2:	/* Change time */
		if(obj1->change_time <= obj2->change_time)
			return(obj1->change_time < obj2->change_time);
		else
			return(-1);
		break;
	}

	return(0);
}

static gint edv_archiver_column_sort_date_access_cb(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	return(edv_archiver_column_sort_date_nexus(
		clist, ptr1, ptr2, 0
	));
}

static gint edv_archiver_column_sort_date_modify_cb(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	return(edv_archiver_column_sort_date_nexus(
		clist, ptr1, ptr2, 1
	));
}

static gint edv_archiver_column_sort_date_change_cb(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	return(edv_archiver_column_sort_date_nexus(
		clist, ptr1, ptr2, 2
	));
}

/*
 *	Contents GtkCList "resize_column" signal callback.
 */
void edv_archiver_resize_column_cb(
	GtkCList *clist, gint column, gint width, gpointer data
)
{
	GtkWidget *widget;
	CfgItem *cfg_list;
	EDVCore *core;
	EDVArchiver *archiver = EDV_ARCHIVER(data);
	if((clist == NULL) || (archiver == NULL))
		return;

	if(EDV_ARCHIVER_IS_PROCESSING(archiver) || (archiver->freeze_count > 0))
		return;

	widget = GTK_WIDGET(clist);
	core = archiver->core;
	cfg_list = core->cfg_list;

	/* Check which clist this signal is for
	 *
	 * Contents GtkCList
	 */
	if(widget == archiver->contents_clist)
	{
		/* Get the column type that the column is displaying */
		const EDVArchiverColumnType column_type = edv_archiver_list_get_column_type_by_index(
			archiver,
			column
		);

		/* Get the column widths intlist */
		CfgIntList *column_width_intlist = EDV_GET_INT_LIST(
			EDV_CFG_PARM_ARCHIVER_CONTENTS_COLUMN_WIDTH
		);
		if((column_type >= 0) && (column_width_intlist != NULL))
		{
			GList *glist = g_list_nth(
				column_width_intlist->list,
				(guint)column_type
			);
			if(glist != NULL)
			{
				glist->data = (gpointer)width;
			}
			else
			{
				/* The column type is not in the column widths
				 * list so create a new column widths list from
				 * the current column widths list and include
				 * the missing column type
				 */
				const gint m = EDV_ARCHIVER_NCOLUMN_TYPES;
				gint i;
				GList *glist_new = NULL;

				glist = column_width_intlist->list;

				for(i = 0; i < m; i++)
				{
					if(glist != NULL)
					{
						glist_new = g_list_append(
							glist_new,
							glist->data
						);				
						glist = g_list_next(glist);
					}
					else
						glist_new = g_list_append(
							glist_new,
							NULL
						);
				}

				g_list_free(column_width_intlist->list);
				column_width_intlist->list = glist_new;

				glist = g_list_nth(
					glist_new,
					(guint)column_type
				);
				if(glist != NULL)
					glist->data = (gpointer)width;
			}
		}
	}
}

/*
 *	Contents GtkCList "click_column" signal callback.
 */
void edv_archiver_click_column_cb(
	GtkCList *clist, gint column, gpointer data
)
{
	GtkWidget *widget, *toplevel;
	const CfgItem *cfg_list;
	EDVCore *core;
	EDVArchiver *archiver = EDV_ARCHIVER(data);
	if((clist == NULL) || (archiver == NULL))
		return;

	if(EDV_ARCHIVER_IS_PROCESSING(archiver) || (archiver->freeze_count > 0))
		return;

	widget = GTK_WIDGET(clist);
	toplevel = archiver->toplevel;
	core = archiver->core;
	cfg_list = core->cfg_list;

	/* Check which clist this signal is for
	 *
	 * Contents GtkCList
	 */
	if(widget == archiver->contents_clist)
	{
		CfgIntList *column_types_intlist;
		GtkCListCompareFunc cmp_func = NULL;
		GtkCListCompareFunc cmp_func_str =
			 (GtkCListCompareFunc)EDVCListColumnSortStringCB;
#if 0
		GtkCListCompareFunc cmp_func_num =
			(GtkCListCompareFunc)edv_clist_column_sort_number_cb;
#endif

		edv_archiver_set_busy(archiver, TRUE);
		GUIBlockInput(toplevel, TRUE);
		archiver->flags |= EDV_ARCHIVER_PROCESSING;

		/* Get the column types mapping list */
		column_types_intlist = EDV_GET_INT_LIST(
			EDV_CFG_PARM_ARCHIVER_CONTENTS_COLUMN
		);
		if(column_types_intlist != NULL)
		{
			const EDVArchiverColumnType column_type = (EDVArchiverColumnType)g_list_nth_data(
				column_types_intlist->list,
				column
			);
			switch(column_type)
			{
			  case EDV_ARCHIVER_COLUMN_TYPE_INDEX:
				cmp_func = edv_archiver_column_sort_index_cb;
				break;
			  case EDV_ARCHIVER_COLUMN_TYPE_NAME:
				cmp_func = cmp_func_str;
				break;
			  case EDV_ARCHIVER_COLUMN_TYPE_SIZE:
				cmp_func = edv_archiver_column_sort_size_cb;
				break;
			  case EDV_ARCHIVER_COLUMN_TYPE_STORAGE_SIZE:
				cmp_func = edv_archiver_column_sort_storage_size_cb;
				break;
			  case EDV_ARCHIVER_COLUMN_TYPE_TYPE:
				cmp_func = cmp_func_str;
				break;
			  case EDV_ARCHIVER_COLUMN_TYPE_PERMISSIONS:
				cmp_func = cmp_func_str;
				break;
			  case EDV_ARCHIVER_COLUMN_TYPE_OWNER:
				cmp_func = cmp_func_str;
				break;
			  case EDV_ARCHIVER_COLUMN_TYPE_GROUP:
				cmp_func = cmp_func_str;
				break;
			  case EDV_ARCHIVER_COLUMN_TYPE_DATE_ACCESS:
				cmp_func = edv_archiver_column_sort_date_access_cb;
				break;
			  case EDV_ARCHIVER_COLUMN_TYPE_DATE_MODIFIED:
				cmp_func = edv_archiver_column_sort_date_modify_cb;
				break;
			  case EDV_ARCHIVER_COLUMN_TYPE_DATE_CHANGED:
				cmp_func = edv_archiver_column_sort_date_change_cb;
				break;
			  case EDV_ARCHIVER_COLUMN_TYPE_LOCATION:
				cmp_func = cmp_func_str;
				break;
			  case EDV_ARCHIVER_COLUMN_TYPE_LINKED_TO:
				cmp_func = cmp_func_str;
				break;
			  case EDV_ARCHIVER_COLUMN_TYPE_DEVICE_TYPE:
				cmp_func = cmp_func_str;
				break;
			  case EDV_ARCHIVER_COLUMN_TYPE_ENCRYPTION:
				cmp_func = cmp_func_str;
				break;
			  case EDV_ARCHIVER_COLUMN_TYPE_COMPRESSION:
				cmp_func = edv_archiver_column_sort_compression_cb;
				break;
			  case EDV_ARCHIVER_COLUMN_TYPE_METHOD:
				cmp_func = cmp_func_str;
				break;
			  case EDV_ARCHIVER_COLUMN_TYPE_CRC:
				cmp_func = cmp_func_str;
				break;
			}
		}


		gtk_clist_freeze(clist);

		/* Set or invert the sort column on the GtkCList based
		 * on if the clicked on column is already the current
		 * sort column
		 */
		if(column != clist->sort_column)
			gtk_clist_set_sort_column(clist, column);
		else
			gtk_clist_set_sort_type(
				clist,
				(clist->sort_type == GTK_SORT_ASCENDING) ?
					GTK_SORT_DESCENDING : GTK_SORT_ASCENDING
			);
		if(cmp_func != NULL)
			gtk_clist_set_compare_func(clist, cmp_func);

		/* Sort the rows, this will call the GtkCList column
		 * sort callbacks
		 */
		gtk_clist_sort(clist);

		gtk_clist_thaw(clist);

		archiver->flags &= ~EDV_ARCHIVER_PROCESSING;
		GUIBlockInput(toplevel, FALSE);
		edv_archiver_set_busy(archiver, FALSE);

		edv_archiver_update_display(archiver);
	}
}

/*
 *	Archiver GtkCList "select_row" signal callback.
 */
void edv_archiver_select_row_cb(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	GtkWidget *widget;
	EDVArchiver *archiver = EDV_ARCHIVER(data);
	if((clist == NULL) || (archiver == NULL))
		return;

	if(EDV_ARCHIVER_IS_PROCESSING(archiver) || (archiver->freeze_count > 0))
		return;

	widget = GTK_WIDGET(clist);

	/* Check which clist this signal is for
	 *
	 * Contents GtkCList
	 */
	if(widget == archiver->contents_clist)
	{
		/* Get the total number of objects selected */
		const gint nselected = g_list_length(clist->selection);

		/* Get the object */
		EDVArchiveObject *obj = EDV_ARCHIVE_OBJECT(
			gtk_clist_get_row_data(clist, row)
		);

		/* Update selected row */
		archiver->contents_clist_selected_row = row;

		/* Set the DND drag icon */
		GUIDNDSetDragIconFromCListSelection(clist);

		if(obj != NULL)
		{
			/* Update status bar message */
			if(!STRISEMPTY(obj->name))
			{
				gchar *s, *size_str = NULL;
				const gchar *type_str = NULL;

				/* Get object type string and size string */
				switch(obj->type)
				{
				  case EDV_OBJECT_TYPE_UNKNOWN:
					type_str = "Object";
					size_str = NULL;
					break;
				  case EDV_OBJECT_TYPE_FILE:
					type_str = "File";
					size_str = g_strdup_printf(
						" (%s %s)",
						edv_str_size_delim(obj->size),
						(obj->size == 1l) ? "byte" : "bytes"
					);
					break;
				  case EDV_OBJECT_TYPE_DIRECTORY:
					type_str = "Directory";
					size_str = NULL;
					break;
				  case EDV_OBJECT_TYPE_LINK:
					type_str = "Link";
					size_str = NULL;
					break;
				  case EDV_OBJECT_TYPE_DEVICE_BLOCK:
					type_str = "Block device";
					size_str = NULL;
					break;
				  case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
					type_str = "Character device";
					size_str = NULL;
					break;
				  case EDV_OBJECT_TYPE_FIFO:
					type_str = "FIFO pipe";
					size_str = NULL;
					break;
				  case EDV_OBJECT_TYPE_SOCKET:
					type_str = "Socket";
					size_str = NULL;
					break;
				  case EDV_OBJECT_TYPE_ERROR:
					type_str = "Error";
					size_str = NULL;
					break;
				}

				/* Set the status bar message */
				if(nselected > 1)
				{
					gulong total_size = 0l;
					EDVArchiveObject *obj;
					GList *glist = clist->selection;
					while(glist != NULL)
					{
						obj = EDV_ARCHIVE_OBJECT(gtk_clist_get_row_data(
							clist, (gint)glist->data
						));
						if(obj != NULL)
							total_size += obj->size;
						glist = g_list_next(glist);
					}
					g_free(size_str);
					size_str = g_strdup_printf(
						"%s %s",
						edv_str_size_delim(total_size),
						(total_size == 1l) ? "byte" : "bytes"
					);
					s = g_strdup_printf(
						"%s objects selected (totaling %s)",
						edv_str_size_delim(nselected),
						size_str
					);
				}
				else if(!strcmp((const char *)obj->name, ".."))
					s = g_strdup_printf(
						"Parent directory selected"
					);
				else
					s = g_strdup_printf(
						"%s \"%s\" selected%s",
						type_str, obj->name,
						(size_str != NULL) ? size_str : ""
					);
				edv_status_bar_message(
					archiver->status_bar,
					s,
					FALSE
				);
				g_free(s);
				g_free(size_str);
			}
			else
			{
				edv_status_bar_message(
					archiver->status_bar,
					"Object with no name name selected",
					FALSE
				);
			}
		}

		/* Check if selected row is fully visible, if not then
		 * adjust scroll position to try and make it visible
		 */
		if(gtk_clist_row_is_visible(clist, row) !=
			GTK_VISIBILITY_FULL
		)
			gtk_clist_moveto(
				clist,
				row, -1,			/* Row, column */
				0.5f, 0.0f			/* Row, column */
			);

		edv_archiver_update_display(archiver);

		/* Double click? */
		if(event != NULL)
		{
			if(event->type == GDK_2BUTTON_PRESS)
			{
				/* Extract */
				edv_archiver_op_extract(archiver);
			}
		}
	}
}

/*
 *	Archiver GtkCList "unselect_row" signal callback.
 */
void edv_archiver_unselect_row_cb(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	GtkWidget *widget;
	EDVArchiver *archiver = EDV_ARCHIVER(data);
	if((clist == NULL) || (archiver == NULL))
		return;

	if(EDV_ARCHIVER_IS_PROCESSING(archiver) || (archiver->freeze_count > 0))
		return;

	widget = GTK_WIDGET(clist);

	/* Check which clist this signal is for
	 *
	 * Contents GtkCList
	 */
	if(widget == archiver->contents_clist)
	{
		gchar *s;

		/* Get total number of objects selected */
		const gint nselected = g_list_length(clist->selection);

		/* Update the status bar message */
		if(nselected > 0)
		{
			gchar *size_str;
			gulong total_size = 0l;
			EDVArchiveObject *obj;
			GList *glist = clist->selection;
			while(glist != NULL)
			{
				obj = EDV_ARCHIVE_OBJECT(gtk_clist_get_row_data(
					clist, (gint)glist->data
				));
				if(obj != NULL)
					total_size += obj->size;
				glist = g_list_next(glist);
			}
			size_str = g_strdup_printf(
				"%s %s",
				edv_str_size_delim(total_size),
				(total_size == 1l) ? "byte" : "bytes"
			);
			s = g_strdup_printf(
				"%s %s selected (totaling %s)",
				edv_str_size_delim(nselected),
				(nselected == 1) ? "object" : "objects",
				size_str
			);
			g_free(size_str);
		}
		else
			s = STRDUP("No objects selected");
		edv_status_bar_message(
			archiver->status_bar, s, FALSE
		);
		g_free(s);

		edv_archiver_update_display(archiver);
	}
}


/*
 *	Location combo "activate" signal callback.
 */
void edv_archiver_combo_activate_cb(GtkWidget *widget, gpointer data)
{
	EDVArchiver *archiver = EDV_ARCHIVER(data);
	if((widget == NULL) || (archiver == NULL))
		return;

	if(EDV_ARCHIVER_IS_PROCESSING(archiver) || (archiver->freeze_count > 0))
		return;

	/* Check which widget was activated
	 *
	 * Location GtkCombo
	 */
	if(widget == archiver->location_combo)
	{
		GtkCombo *combo = GTK_COMBO(widget);
		gchar *path = edv_path_evaluate(
			NULL,           /* No parent path, imply use toplevel */
			gtk_entry_get_text(GTK_ENTRY(combo->entry))
		);
		if(path != NULL)
		{
			/* Open the archive */
			(void)edv_archiver_open_archive(
				archiver,
				path,
				edv_archiver_get_password(archiver),
				TRUE		/* List passively */
			);
			g_free(path);
		}
	}
	/* Password GtkEntry */
	else if(widget == archiver->location_password_entry)
	{
		(void)edv_archiver_open_archive(
			archiver,
			edv_archiver_get_location(archiver),
			edv_archiver_get_password(archiver),
			TRUE			/* List passively */
		);
	}
}


/*
 *	Menu item activate callback.
 *
 *	The data must be a EDVArchiverOpID *.
 */
void edv_archiver_menu_item_activate_cb(GtkWidget *widget, gpointer data)
{
	edv_archiver_op_activate_cb(NULL, -1, data);
}

/*
 *	Menu item "enter_notify_event" signal callback.
 *
 *	The data must be a EDVArchiverOpID *.
 */
gint edv_archiver_menu_item_enter_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	edv_archiver_op_enter_cb(NULL, -1, data);
	return(TRUE);
}

/*
 *	Menu item "leave_notify_event" signal callback.
 *
 *	The data must be a EDVArchiverOpID *.
 */
gint edv_archiver_menu_item_leave_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	edv_archiver_op_leave_cb(NULL, -1, data);
	return(TRUE);
}


/*
 *	Find Bar get current location callback.
 */
const gchar *edv_archiver_find_bar_location_cb(
	GtkWidget *bar,
	gpointer data
)
{
	return(edv_archiver_get_location(EDV_ARCHIVER(data)));
}

/*
 *	Find Bar start find callback.
 */
void edv_archiver_find_bar_start_cb(
	GtkWidget *bar,
	gpointer data
)
{
	GtkCList *clist;
	EDVArchiver *archiver = EDV_ARCHIVER(data);
	if(archiver == NULL)
		return;

	clist = GTK_CLIST(archiver->contents_clist);

	edv_archiver_set_busy(archiver, TRUE);

	gtk_clist_freeze(clist);
	gtk_clist_unselect_all(clist);
	gtk_clist_thaw(clist);

	edv_archiver_update_display(archiver);

	archiver->freeze_count++;
}

/*
 *	Find Bar end find callback.
 */
void edv_archiver_find_bar_end_cb(
	GtkWidget *bar,
	const gint nmatches,
	gpointer data
)
{
	EDVArchiver *archiver = EDV_ARCHIVER(data);
	if(archiver == NULL)
		return;

	archiver->freeze_count--;
	edv_archiver_set_busy(archiver, FALSE);
}


/*
 *	Find Bar match callback.
 */
void edv_archiver_find_bar_match_cb(
	GtkWidget *bar,
	const gchar *path,
	GList *properties_list,
	const gint line_index,
	const gchar *excerpt,
	gpointer data
)
{
	gint row;
	GtkCList *clist;
	EDVArchiver *archiver = EDV_ARCHIVER(data);
	if((path == NULL) || (archiver == NULL))
		return;

	clist = GTK_CLIST(archiver->contents_clist);

	row = edv_archiver_list_find_by_path(archiver, path);
	if((row >= 0) && (row < clist->rows))
		gtk_clist_select_row(clist, row, 0);
}


/*
 *	Status message callback.
 */
void edv_archiver_status_message_cb(
	GtkWidget *widget,
	const gchar *message, 
	gpointer data
)
{
	EDVArchiver *archiver = EDV_ARCHIVER(data);
	if(archiver == NULL)
		return;

	edv_status_bar_message(archiver->status_bar, message, FALSE);
}

/*
 *	Status progress callback.
 */
void edv_archiver_status_progress_cb(
	GtkWidget *widget,
	const gfloat progress, 
	gpointer data
)
{
	EDVArchiver *archiver = EDV_ARCHIVER(data);
	if(archiver == NULL)
		return;

	edv_status_bar_progress(archiver->status_bar, progress, FALSE);
}


/*
 *	Window created callback.
 */
void edv_archiver_window_created_cb(
	EDVArchiver *archiver,
	const EDVWindowType win_type,
	gpointer win
)
{
	if(archiver == NULL)
		return;

	if(archiver == EDV_ARCHIVER(win))
		return;

	edv_archiver_update_display(archiver);
}

/*
 *	Window deleted callback.
 */
void edv_archiver_window_deleted_cb(
	EDVArchiver *archiver,
	const EDVWindowType win_type,
	gpointer win
)
{
	if(archiver == NULL)
		return;

	if(archiver == EDV_ARCHIVER(win))
		return;

	edv_archiver_update_display(archiver);
}


/*
 *	Master Write Protect changed callback.
 */
void edv_archiver_write_protect_changed_cb(
	EDVArchiver *archiver,
	const gboolean state
)
{
	if(archiver == NULL)
		return;

	if(EDV_ARCHIVER_IS_PROCESSING(archiver))
		return;

	edv_archiver_update_display(archiver);
}

/*
 *	Delete Method changed callback.
 */
void edv_archiver_delete_method_changed_cb(
	EDVArchiver *archiver,
	const EDVDeleteMethod delete_method
)
{
	if(archiver == NULL)
		return;

	if(EDV_ARCHIVER_IS_PROCESSING(archiver))
		return;

	edv_archiver_update_display(archiver);
}


/*
 *	Object added callback.
 */
void edv_archiver_object_added_cb(
	EDVArchiver *archiver,
	const gchar *path,
	EDVVFSObject *obj
)
{
	if((archiver == NULL) || STRISEMPTY(path))
		return;

	if(EDV_ARCHIVER_IS_PROCESSING(archiver))
		return;

	edv_archiver_list_vfs_object_added_cb(
		archiver,
		path,
		obj
	);

/*	edv_archiver_update_display(archiver); */
}

/*
 *	Object modified callback.
 */
void edv_archiver_object_modified_cb(
	EDVArchiver *archiver,
	const gchar *path,
	const gchar *new_path,
	EDVVFSObject *obj
)
{
	gchar *cur_arch_path;

	if((archiver == NULL) || STRISEMPTY(path))
		return;

	if(EDV_ARCHIVER_IS_PROCESSING(archiver))
		return;

	if(new_path == NULL)
		new_path = path;

	cur_arch_path = STRDUP(edv_archiver_get_location(archiver));
	if(cur_arch_path == NULL)
		return;

	/* Is the modified path the current archive? */
	if(!strcmp((const char *)cur_arch_path, (const char *)path))
	{
		/* Path changed? */
		if(path != new_path)
		{
			if(strcmp((const char *)path, (const char *)new_path))
			{
				/* Update the current archive path */
				edv_archiver_set_title(
					archiver,
					new_path
				);
				edv_archiver_set_location(
					archiver,
					new_path,
					FALSE
				);
				edv_archiver_location_icon_update(
					archiver,
					new_path
				);
			}
		}
	}

	edv_archiver_list_vfs_object_modified_cb(
		archiver,
		path,
		new_path,
		obj
	);

/*	edv_archiver_update_display(archiver); */

	g_free(cur_arch_path);
}

/*
 *	Object removed callback.
 */
void EDVArchiverObjectRemovedNotifyCB(
	EDVArchiver *archiver,
	const gchar *path
)
{
	gchar *cur_arch_path;

	if((archiver == NULL) || STRISEMPTY(path))
		return;

	if(EDV_ARCHIVER_IS_PROCESSING(archiver))
		return;

	cur_arch_path = STRDUP(edv_archiver_get_location(archiver));
	if(cur_arch_path == NULL)
		return;

	edv_archiver_list_vfs_object_removed_cb(archiver, path);

	/* Is the removed object the current archive? */
	if(!strcmp((const char *)cur_arch_path, (const char *)path))
	{
		/* No need to update the title or the location */
	}

/*	edv_archiver_update_display(archiver); */

	g_free(cur_arch_path);
}


/*
 *	Archive object added callback.
 */
void edv_archiver_archive_object_added_cb(
	EDVArchiver *archiver,
	const gchar *arch_path,
	const gchar *path,
	EDVArchiveObject *obj
)
{
	gchar *cur_arch_path;

	if((archiver == NULL) || STRISEMPTY(arch_path) ||
	   STRISEMPTY(path) || (obj == NULL)
	)
		return;

	if(EDV_ARCHIVER_IS_PROCESSING(archiver))
		return;

	cur_arch_path = STRDUP(edv_archiver_get_location(archiver));
	if(cur_arch_path == NULL)
		return;

	/* Is the added object in our archive? */
	if(!strcmp((const char *)cur_arch_path, (const char *)arch_path))
	{
		/* Notify the contents list */
		edv_archiver_list_archive_object_added_cb(
			archiver, arch_path, path, obj
		);
	}

/*	edv_archiver_update_display(archiver); */

	g_free(cur_arch_path);
}

/*
 *	Archive object modified callback.
 */
void edv_archiver_archive_object_modified_cb(
	EDVArchiver *archiver,
	const gchar *arch_path,
	const gchar *path,
	const gchar *new_path,
	EDVArchiveObject *obj
)
{
	gchar *cur_arch_path;

	if((archiver == NULL) || STRISEMPTY(arch_path) ||
	   STRISEMPTY(path) || (obj == NULL)
	)
		return;

	if(EDV_ARCHIVER_IS_PROCESSING(archiver))
		return;

	if(new_path == NULL)
		new_path = path;

	cur_arch_path = STRDUP(edv_archiver_get_location(archiver));
	if(cur_arch_path == NULL)
		return;

	/* Is the modified object in our archive? */
	if(!strcmp((const char *)cur_arch_path, (const char *)arch_path))
	{
		/* Notify the contents list */
		edv_archiver_list_archive_object_modified_cb(
			archiver, arch_path, path, new_path, obj
		);
	}

/*	edv_archiver_update_display(archiver); */

	g_free(cur_arch_path);
}

/*
 *	Archive object removed callback.
 */
void edv_archiver_archive_object_removed_cb(
	EDVArchiver *archiver,
	const gchar *arch_path,
	const gchar *path
)
{
	gchar *cur_arch_path;

	if((archiver == NULL) || STRISEMPTY(arch_path) ||
	   STRISEMPTY(path)
	)
		return;

	if(EDV_ARCHIVER_IS_PROCESSING(archiver))
		return;

	cur_arch_path = STRDUP(edv_archiver_get_location(archiver));
	if(cur_arch_path == NULL)
		return;

	/* Is the removed object in our archive? */
	if(!strcmp((const char *)cur_arch_path, (const char *)arch_path))
	{
		/* Notify the contents list */
		edv_archiver_list_archive_object_removed_cb(
			archiver, arch_path, path
		);
	}

/*	edv_archiver_update_display(archiver); */

	g_free(cur_arch_path);
}

/*
 *	Recycled object added callback.
 */
void edv_archiver_recycled_object_added_cb(
	EDVArchiver *archiver,
	const guint index
)
{
	EDVCore *core;

	if(archiver == NULL)
		return;

	if(EDV_ARCHIVER_IS_PROCESSING(archiver))
		return;

	core = archiver->core;

	/* There is not much interest when a recycled object has been
	 * added or removed. Only the menus need to be updated if there
	 * is a change in the number of recycled objects
	 */
	if(core->last_nrecycle_bin_items != archiver->last_nrecycle_bin_items)
		edv_archiver_update_display(archiver);
}

/*
 *	Recycled object modified callback.
 */
void edv_archiver_recycled_object_modified_cb(
	EDVArchiver *archiver,
	const guint index
)
{
	EDVCore *core;

	if(archiver == NULL)
		return;

	if(EDV_ARCHIVER_IS_PROCESSING(archiver))
		return;

	core = archiver->core;

	/* There is not much interest when a recycled object has been
	 * added or removed. Only the menus need to be updated if there
	 * is a change in the number of recycled objects
	 */
	if(core->last_nrecycle_bin_items != archiver->last_nrecycle_bin_items)
		edv_archiver_update_display(archiver);
}

/*
 *	Recycled object removed callback.
 */
void edv_archiver_recycled_object_removed_cb(
	EDVArchiver *archiver,
	const guint index
)
{
	EDVCore *core;

	if(archiver == NULL)
		return;

	if(EDV_ARCHIVER_IS_PROCESSING(archiver))
		return;

	core = archiver->core;

	/* There is not much interest when a recycled object has been
	 * added or removed. Only the menus need to be updated if there
	 * is a change in the number of recycled objects
	 */
	if(core->last_nrecycle_bin_items != archiver->last_nrecycle_bin_items)
		edv_archiver_update_display(archiver);
}


/*
 *	Reconfigured callback.
 */
void edv_archiver_reconfigured_cb(EDVArchiver *archiver)
{
	gchar *cur_loc;
	GtkRcStyle *standard_rcstyle, *lists_rcstyle;
	GtkWidget *w;
	CfgItem *cfg_list;
	EDVCore *core;

	if(archiver == NULL)
		return;

	if(EDV_ARCHIVER_IS_PROCESSING(archiver))
		return;

	core = archiver->core;
	cfg_list = core->cfg_list;
	standard_rcstyle = core->standard_rcstyle;
	lists_rcstyle = core->lists_rcstyle;


	/* Reset last state markers so that resources get explicitly
	 * checked due to reconfiguring
	 */
	archiver->last_nrecycle_bin_items = -1;
	archiver->last_write_protect_state = -1;


	/* Get the current location */
	cur_loc = STRDUP(edv_archiver_get_location(archiver));


	/* Update the title */
	edv_archiver_set_title(archiver, cur_loc);

	/* Update the Accelkey Labels */
	edv_archiver_accelkeys_regenerate(archiver);

	/* Regenerate the Tool Bar */
	edv_archiver_tool_bar_regenerate(archiver);

	/* Show/hide the Tool Bar */
	w = archiver->tool_bar_handle;
	if(w != NULL)
	{
		if(EDV_GET_B(EDV_CFG_PARM_ARCHIVER_SHOW_TOOL_BAR))
		{
			gtk_widget_show(w);
			archiver->flags |= EDV_ARCHIVER_TOOL_BAR_MAPPED;
		}
		else
		{
			gtk_widget_hide(w);
			archiver->flags &= ~EDV_ARCHIVER_TOOL_BAR_MAPPED;
		}
	}

	/* Show/hide the Location Bar */
	w = archiver->location_bar_handle;
	if(w != NULL)
	{
		if(EDV_GET_B(EDV_CFG_PARM_ARCHIVER_SHOW_LOCATION_BAR))
		{
			gtk_widget_show(w);
			archiver->flags |= EDV_ARCHIVER_LOCATION_BAR_MAPPED;
		}
		else
		{
			gtk_widget_hide(w);
			archiver->flags &= ~EDV_ARCHIVER_LOCATION_BAR_MAPPED;
		}
	}

	/* Show/hide the Find Bar */
	w = archiver->find_bar_handle;
	if(w != NULL)
	{
		if(EDV_GET_B(EDV_CFG_PARM_ARCHIVER_SHOW_FIND_BAR))
		{
			gtk_widget_show(w);
			archiver->flags |= EDV_ARCHIVER_FIND_BAR_MAPPED;
		}
		else
		{
			gtk_widget_hide(w);
			archiver->flags &= ~EDV_ARCHIVER_FIND_BAR_MAPPED;
		}
	}

	/* Show/hide the Status Bar */
	w = archiver->status_bar;
	if(w != NULL)
	{
		if(EDV_GET_B(EDV_CFG_PARM_ARCHIVER_SHOW_STATUS_BAR))
		{
			gtk_widget_show(w);
			archiver->flags |= EDV_ARCHIVER_STATUS_BAR_MAPPED;
		}
		else
		{
			gtk_widget_hide(w);
			archiver->flags &= ~EDV_ARCHIVER_STATUS_BAR_MAPPED;
		}
	}


	/* Update RC styles */
	w = archiver->toplevel;
	if((w != NULL) && (standard_rcstyle != NULL))
		gtk_widget_modify_style_recursive(w, standard_rcstyle);
	w = archiver->contents_clist;
	if((w != NULL) && (lists_rcstyle != NULL))
		gtk_widget_modify_style_recursive(w, lists_rcstyle);
	w = archiver->contents_clist_menu;
	if((w != NULL) && (standard_rcstyle != NULL))
		gtk_widget_modify_style_recursive(w, standard_rcstyle);


	/* Realize the listings */
	edv_archiver_list_realize_listing(archiver);

	edv_archiver_update_display(archiver);

	/* Notify GTK about possible size changes (such as the tool
	 * bar resizing)
	 */
	gtk_widget_queue_resize(archiver->toplevel);

	g_free(cur_loc);
}


/*
 *	MIME Type added callback.
 */
void edv_archiver_mime_type_added_cb(
	EDVArchiver *archiver,
	const gint mt_num, EDVMIMEType *mt
)
{
	/* Treat a MIME Type added the same as it would be for a MIME
	 * Type modified, forward signal to the MIME Type modified
	 * callback
	 */
	edv_archiver_mime_type_modified_cb(archiver, mt_num, mt);
}

/*
 *	MIME Type modified callback.
 */
void edv_archiver_mime_type_modified_cb(
	EDVArchiver *archiver,
	const gint mt_num, EDVMIMEType *mt
)
{
	gchar *cur_loc;
	EDVCore *core;

	if(archiver == NULL)
		return;

	if(EDV_ARCHIVER_IS_PROCESSING(archiver))
		return;

	core = archiver->core;

	/* Get the current location */
	cur_loc = STRDUP(edv_archiver_get_location(archiver));

	/* Realize listings */
	edv_archiver_list_realize_listing(archiver);

/*	edv_archiver_update_display(archiver); */

	g_free(cur_loc);
}

/*
 *	MIME Type removed callback.
 */
void edv_archiver_mime_type_removed_cb(
	EDVArchiver *archiver,
	const gint mt_num
)
{
	gchar *cur_loc;
	EDVCore *core;

	if(archiver == NULL)
		return;

	if(EDV_ARCHIVER_IS_PROCESSING(archiver))
		return;

	core = archiver->core;

	/* Get the current location */
	cur_loc = STRDUP(edv_archiver_get_location(archiver));

	/* Realize the listings */
	edv_archiver_list_realize_listing(archiver);

/*	edv_archiver_update_display(archiver); */

	g_free(cur_loc);
}
