#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "cfg.h"

#include "guiutils.h"
#include "cfg_gtk.h"
#include "tool_bar.h"
#include "cdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "edv_pixmap.h"
#include "edv_obj_info_match.h"
#include "edv_utils_gtk.h"
#include "edv_list_cb.h"
#include "edv_find_bar.h"
#include "edv_status_bar.h"
#include "archive_comments_dlg.h"
#include "archive_statistics_dlg.h"
#include "archiver.h"
#include "archiver_cb.h"
#include "archiver_op.h"
#include "archiver_list.h"
#include "archiver_dnd.h"
#include "archiver_subprocess.h"
#include "edv_emit.h"
#include "edv_cb.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_hsep_20x20.xpm"
#include "images/icon_new_20x20.xpm"
#include "images/icon_open_20x20.xpm"
#include "images/icon_save_20x20.xpm"
#include "images/icon_edit_20x20.xpm"
#include "images/icon_trace_20x20.xpm"
#include "images/icon_run_20x20.xpm"
#include "images/icon_terminal2_20x20.xpm"
#include "images/icon_close_20x20.xpm"
#include "images/icon_exit_20x20.xpm"
#include "images/icon_archive_add_20x20.xpm"
#include "images/icon_archive_extract_20x20.xpm"
#include "images/icon_aid_20x20.xpm"
#include "images/icon_archive_check_20x20.xpm"
#include "images/icon_security_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"
#include "images/icon_select_all_20x20.xpm"
#include "images/icon_unselect_all_20x20.xpm"
#include "images/icon_invert_selection_20x20.xpm"
#include "images/icon_reload_20x20.xpm"
#include "images/icon_stop_20x20.xpm"
#include "images/icon_info_20x20.xpm"
#include "images/icon_mimetypes_20x20.xpm"
#include "images/icon_search_20x20.xpm"
#include "images/icon_properties_20x20.xpm"
#include "images/icon_tool_bar_20x20.xpm"
#include "images/icon_status_bar_20x20.xpm"
#include "images/icon_wildcards_20x20.xpm"
#include "images/icon_resize_column_20x20.xpm"
#include "images/icon_options_20x20.xpm"
#include "images/icon_customize_20x20.xpm"
#include "images/icon_write_protect_off_20x20.xpm"
#include "images/icon_write_protect_on_20x20.xpm"
#include "images/icon_help_20x20.xpm"
#include "images/icon_about_20x20.xpm"

#include "images/icon_endeavour_file_browser_20x20.xpm"
#include "images/icon_endeavour_image_browser_20x20.xpm"
#include "images/icon_endeavour_archiver_20x20.xpm"
#include "images/icon_purge_20x20.xpm"


/* Utilities */
EDVArchiverOp *edv_archiver_op_match_by_id(
	EDVArchiver *archiver,
	const EDVArchiverOpID id
);

/* Title */
void edv_archiver_set_title(
	EDVArchiver *archiver,
	const gchar *path
);

/* Location */
const gchar *edv_archiver_get_location(EDVArchiver *archiver);
void edv_archiver_set_location(
	EDVArchiver *archiver,
	const gchar *path,
	const gboolean record_history
);
void edv_archiver_location_icon_update(
	EDVArchiver *archiver,
	const gchar *path
);

/* Password */
const gchar *edv_archiver_get_password(EDVArchiver *archiver);
void edv_archiver_set_password(
	EDVArchiver *archiver,
	const gchar *password
);

/* Regenerate Style, Menus, and Tool Bar */
void edv_archiver_icons_regenerate(EDVArchiver *archiver);
void edv_archiver_accelkeys_regenerate(EDVArchiver *archiver);
void edv_archiver_tool_bar_regenerate(EDVArchiver *archiver);
void edv_archiver_list_menu_regenerate(EDVArchiver *archiver);

/* Sync Configuration */
void edv_archiver_sync_configuration(EDVArchiver *archiver);

/* Create Archive */
gint edv_archiver_new_archive(
	EDVArchiver *archiver,
	const gchar *path,
	const gchar *password,
	const gboolean passive
);

/* Open Archive */
gint edv_archiver_open_archive(
	EDVArchiver *archiver,
	const gchar *path,
	const gchar *password,
	const gboolean passive
);

/* EDVArchiverOp */
static EDVArchiverOp *edv_archiver_op_new(void);
static void edv_archiver_op_delete(EDVArchiverOp *op);
GList *edv_archiver_op_ids_list_new(EDVArchiver *archiver);
void edv_archiver_op_ids_list_delete(GList *op_ids_list);

/* Accel Keys */
CfgAccelkeyList *edv_archiver_accelkeys_new(void);

/* EDVArchiverOpQueue */
void edv_archiver_queue_op(
	EDVArchiver *archiver,
	const EDVArchiverOpID id
);

/* EDVArchiver */
static void edv_archiver_create_menu_bar(
	EDVArchiver *archiver,
	GtkWidget *parent
);
EDVArchiver *edv_archiver_new(
	EDVCore *core,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry
);
void edv_archiver_sync_data(EDVArchiver *archiver);
void edv_archiver_update_display(EDVArchiver *archiver);
void edv_archiver_set_busy(
	EDVArchiver *archiver,
	const gboolean busy
);
void edv_archiver_set_passive_busy(
	EDVArchiver *archiver,
	const gboolean busy
);
gboolean edv_archiver_is_mapped(EDVArchiver *archiver);
void edv_archiver_map(EDVArchiver *archiver);
void edv_archiver_unmap(EDVArchiver *archiver);
void edv_archiver_delete(EDVArchiver *archiver);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Matches the EDVArchiverOp by a EDVArchiverOpID.
 */
EDVArchiverOp *edv_archiver_op_match_by_id(
	EDVArchiver *archiver,
	const EDVArchiverOpID id
)
{
	GList *glist;
	EDVArchiverOp *op;

	if(archiver == NULL)
		return(NULL);

	for(glist = archiver->op_ids_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		op = EDV_ARCHIVER_OP(glist->data);
		if(op == NULL)
			continue;

		if(op->id == id)
			return(op);
	}

	return(NULL);
}


/*
 *	Sets the Archiver's title.
 */
void edv_archiver_set_title(
	EDVArchiver *archiver,
	const gchar *path
)
{
	gboolean	show_program_name,
			show_user_name,
			show_location;
	const gchar	*prog_name = "Archiver",
			*user_name,
			*filter;
	gchar *title;
	GtkWidget *w;
	CfgList *cfg_list;
	EDVCore *core;

	if(archiver == NULL)
		return;

	w = archiver->toplevel;
	core = archiver->core;
	cfg_list = core->cfg_list;

	/* Get the values to show in the title */
	if(!STRISEMPTY(core->title))
		prog_name = core->title;

	user_name = core->effective_user_id_host_str;
	filter = archiver->contents_list_filter;

	show_program_name = EDV_GET_B(EDV_CFG_PARM_ARCHIVER_TITLE_WITH_PROGRAM);
	if(show_program_name && STRISEMPTY(prog_name))
		show_program_name = FALSE;

	show_user_name = EDV_GET_B(EDV_CFG_PARM_ARCHIVER_TITLE_WITH_USER);
	if(show_user_name && STRISEMPTY(user_name))
		show_user_name = FALSE;

	show_location = EDV_GET_B(EDV_CFG_PARM_ARCHIVER_TITLE_WITH_LOCATION);
	if(show_location && STRISEMPTY(path))
		show_location = FALSE;

	/* Format the new title string */
	title = g_strdup("");
	if(EDV_GET_B(EDV_CFG_PARM_ARCHIVER_TITLE_REVERSE))
	{
		if(show_location)
		{
			gchar *s;
			if(STRISEMPTY(filter))
				s = g_strconcat(
					title,
					path,
					NULL
				);
			else
				s = g_strconcat(
					title,
					path,
					" - ",
					filter,
					NULL
				);
			g_free(title);
			title = s;
		}
		if(show_user_name)
		{
			gchar *s = g_strconcat(
				title,
				(show_location) ? " - " : "",
				user_name,
				NULL
			);
			g_free(title);
			title = s;
		}
		if(show_program_name)
		{
			gchar *s = g_strconcat(
				title,
				(show_user_name || show_location) ? " - " : "",
				prog_name,
				NULL
			);
			g_free(title);
			title = s;
		}
	}
	else
	{
		if(show_program_name)
		{
			gchar *s = g_strconcat(
				title,
				prog_name,
				(show_user_name || show_location) ? ": " : "",
				NULL
			);
			g_free(title);
			title = s;
		}
		if(show_user_name)
		{
			gchar *s = g_strconcat(
				title,
				user_name,
				(show_location) ? "" : "",
				NULL
			);
			g_free(title);
			title = s;
		}
		if(show_location)
		{
			gchar *s;
			if(STRISEMPTY(filter))
				s = g_strconcat(
					title,
					path,
					NULL
				);
			else
				s = g_strconcat(
					title,
					path,
					": ",
					filter,
					NULL
				);
			g_free(title);
			title = s;
		}
	}

	/* Set the new title */
	gtk_window_set_title(GTK_WINDOW(w), title);

	g_free(title);
}

/*
 *	Gets the current location.
 */
const gchar *edv_archiver_get_location(EDVArchiver *archiver)
{
	if(archiver == NULL)
		return(NULL);

	return(gtk_entry_get_text(GTK_ENTRY(
		GTK_COMBO(archiver->location_combo)->entry
	)));
}

/*
 *	Sets the location.
 *
 *	The path specifies the new location.
 *
 *	If record_history is TRUE then path will be recorded in the
 *	locations history.
 */
void edv_archiver_set_location(
	EDVArchiver *archiver,
	const gchar *path,
	const gboolean record_history
)
{
	const gchar *s;
	gchar *new_path;
	GtkEntry *entry;
	GtkCombo *combo;

	if(archiver == NULL)
		return;

	combo = GTK_COMBO(archiver->location_combo);
	entry = GTK_ENTRY(combo->entry);

	/* Copy the specified location path */
	new_path = STRDUP(path);

	/* Simplify the new path */
	edv_path_simplify(new_path);

	/* No change? */
	s = gtk_entry_get_text(entry);
	if(s != NULL)
	{
		if(!strcmp(s, new_path))
		{
			g_free(new_path);
			return;
		}
	}

	/* Record the new location? */
	if(record_history)
		GUIComboAddItem(
			GTK_WIDGET(combo),
			new_path
		);

	/* Set the new location */
	gtk_entry_set_text(
		entry,
		new_path
	);
	gtk_entry_set_position(
		entry,
		-1
	);

	g_free(new_path);
}

/*
 *	Updates the Location Bar's icon.
 */
void edv_archiver_location_icon_update(
	EDVArchiver *archiver,
	const gchar *path
)
{
	GtkWidget	*w,
			*parent;
	EDVPixmap	*icon,
			*icon_hidden;
	EDVVFSObject *obj;
	EDVCore *core;

	if(archiver == NULL)
		return;

	parent = archiver->location_icon_fixed;
	core = archiver->core;

	obj = edv_vfs_object_lstat(path);
	if(obj == NULL)
	{
		obj = edv_vfs_object_new();
		if(obj != NULL)
		{
			obj->type = EDV_OBJECT_TYPE_FILE;
			edv_vfs_object_set_path(obj, path);
		}
	}
	if(obj == NULL)
		return;

	/* Get the location icon */
	(void)edv_match_object_icon(
		core->devices_list,
		core->mime_types_list,
		obj->type,
		obj->path,
		EDV_VFS_OBJECT_LINK_TARGET_EXISTS(obj),
		obj->permissions,
		EDV_ICON_SIZE_20,
		&icon,
		NULL,
		NULL,
		&icon_hidden
	);
	/* Hidden */
	if(edv_is_object_hidden(obj))
	{
		if(edv_pixmap_is_loaded(icon_hidden))
		{
			(void)edv_pixmap_unref(icon);
			icon = edv_pixmap_ref(icon_hidden);
		}
	}

	edv_vfs_object_delete(obj);

	/* Create or set the location icon */
	w = archiver->location_icon_pm;
	if(w != NULL)
	{
		edv_pixmap_set_gtk_pixmap(icon, w);
	}
	else
	{
		archiver->location_icon_pm = w = edv_pixmap_new_gtk_pixmap(icon);
		if(w != NULL)
		{
			gtk_fixed_put(GTK_FIXED(parent), w, 0, 0);
			gtk_widget_show(w);
		}
	}

	(void)edv_pixmap_unref(icon);
	(void)edv_pixmap_unref(icon_hidden);
}

/*
 *	Gets the current password.
 */
const gchar *edv_archiver_get_password(EDVArchiver *archiver)
{
	if(archiver == NULL)
		return(NULL);

	return(gtk_entry_get_text(GTK_ENTRY(archiver->location_password_entry)));
}

/*
 *	Sets the Archiver's password.
 */
void edv_archiver_set_password(
	EDVArchiver *archiver, const gchar *password
)
{
	GtkEntry *entry;
	gchar *new_password;

	if((archiver == NULL) || (password == NULL))
		return;

	entry = GTK_ENTRY(archiver->location_password_entry);

	/* Copy the specified password */
	new_password = STRDUP(password);

	/* Set the new password */
	gtk_entry_set_text(
		entry,
		new_password
	);
	gtk_entry_set_position(
		entry,
		-1
	);

	g_free(new_password);
}

/*
 *	Reloads the icons.
 */
void edv_archiver_icons_regenerate(EDVArchiver *archiver)
{
	gchar *icons_path;
	CfgList *cfg_list;
	EDVCore *core;

	if(archiver == NULL)
		return;

	core = archiver->core;
	cfg_list = core->cfg_list;

	/* Get the icons directory */
	icons_path = g_strconcat(
		EDV_GET_S(EDV_CFG_PARM_DIR_GLOBAL),
		G_DIR_SEPARATOR_S,
		EDV_NAME_ICONS_SUBDIR,
		NULL
	);

#define RELOAD_ICON(_icon_,_path_)	{		\
 gchar *path;						\
							\
 (_icon_) = edv_pixmap_unref(_icon_);			\
							\
 path = g_strconcat(					\
  icons_path,						\
  G_DIR_SEPARATOR_S,					\
  (_path_),						\
  NULL							\
 );							\
 if(path != NULL) {					\
  (_icon_) = edv_pixmap_new_from_file(path);		\
  g_free(path);						\
 }							\
}

	RELOAD_ICON(
		archiver->encrypted_icon,
		"icon_encrypted_20x20.xpm"
	);

#undef RELOAD_ICON

	g_free(icons_path);
}

/* 
 *	Resets the Archiver's accelkeys displayed on the widgets.
 */
void edv_archiver_accelkeys_regenerate(EDVArchiver *archiver)
{
	GtkAccelGroup *accelgrp = NULL;
	CfgAccelkeyList *ak_list;
	CfgList *cfg_list;
	EDVCore *core;

	if(archiver == NULL)
		return;

	core = archiver->core;
	if(core == NULL)
		return;
			   
	cfg_list = core->cfg_list;

	ak_list = EDV_GET_ACCELKEY_LIST(EDV_CFG_PARM_ARCHIVER_ACCELERATOR_KEYS);
	if(ak_list == NULL)
		return;

#define UPDATE_ACCELKEY(_w_,_id_)	{	\
 if(((_w_) != NULL) && ((_id_) > 0)) {		\
  GList *glist;					\
  CfgAccelkey *ak;				\
						\
  for(glist = ak_list->list;			\
      glist != NULL;				\
      glist = g_list_next(glist)		\
  ) {						\
   ak = CFG_ACCELKEY(glist->data);		\
   if(ak == NULL)				\
    continue;					\
						\
   if(ak->op_id == (gint)(_id_)) {		\
    GUIMenuItemSetAccelKey(			\
     (_w_), accelgrp, ak->key, ak->modifiers	\
    );						\
    break;					\
   }						\
  }						\
 }						\
}

	UPDATE_ACCELKEY(
		archiver->new_mi,
		EDV_ARCHIVER_OP_NEW
	);
	UPDATE_ACCELKEY(
		archiver->open_mi,
		EDV_ARCHIVER_OP_OPEN
	);
	UPDATE_ACCELKEY(
		archiver->sync_disks_mi,
		EDV_ARCHIVER_OP_SYNC_DISKS
	);
	UPDATE_ACCELKEY(
		archiver->history_mi,
		EDV_ARCHIVER_OP_HISTORY
	);
	UPDATE_ACCELKEY(
		archiver->run_mi,
		EDV_ARCHIVER_OP_RUN
	);
	UPDATE_ACCELKEY(
		archiver->close_mi,
		EDV_ARCHIVER_OP_CLOSE
	);

	UPDATE_ACCELKEY(
		archiver->edit_add_mi,
		EDV_ARCHIVER_OP_ADD
	);
	UPDATE_ACCELKEY(
		archiver->edit_extract_mi,
		EDV_ARCHIVER_OP_EXTRACT
	);
	UPDATE_ACCELKEY(
		archiver->edit_extract_all_mi,
		EDV_ARCHIVER_OP_EXTRACT_ALL
	);
	UPDATE_ACCELKEY(
		archiver->edit_delete_mi,
		EDV_ARCHIVER_OP_DELETE
	);

	UPDATE_ACCELKEY(
		archiver->comments_mi,
		EDV_ARCHIVER_OP_COMMENTS
	);
	UPDATE_ACCELKEY(
		archiver->check_mi,
		EDV_ARCHIVER_OP_CHECK
	);
	UPDATE_ACCELKEY(
		archiver->fix_mi,
		EDV_ARCHIVER_OP_FIX
	);

	UPDATE_ACCELKEY(
		archiver->select_all_mi,
		EDV_ARCHIVER_OP_SELECT_ALL
	);
	UPDATE_ACCELKEY(
		archiver->unselect_all_mi,
		EDV_ARCHIVER_OP_UNSELECT_ALL
	);
	UPDATE_ACCELKEY(
		archiver->invert_selection_mi,
		EDV_ARCHIVER_OP_INVERT_SELECTION
	);
	UPDATE_ACCELKEY(
		archiver->find_mi,
		EDV_ARCHIVER_OP_FIND
	);
	UPDATE_ACCELKEY(
		archiver->properties_mi,
		EDV_ARCHIVER_OP_PROPERTIES
	);

	UPDATE_ACCELKEY(
		archiver->refresh_mi,
		EDV_ARCHIVER_OP_REFRESH
	);
	UPDATE_ACCELKEY(
		archiver->refresh_all_mi,
		EDV_ARCHIVER_OP_REFRESH_ALL
	);
	UPDATE_ACCELKEY(
		archiver->stop_mi,
		EDV_ARCHIVER_OP_STOP
	);
	UPDATE_ACCELKEY(
		archiver->statistics_mi,
		EDV_ARCHIVER_OP_STATISTICS
	);
	UPDATE_ACCELKEY(
		archiver->show_tool_bar_micheck,
		EDV_ARCHIVER_OP_SHOW_TOOL_BAR
	);
	UPDATE_ACCELKEY(
		archiver->show_location_bar_micheck,
		EDV_ARCHIVER_OP_SHOW_LOCATION_BAR
	);
	UPDATE_ACCELKEY(
		archiver->show_find_bar_micheck,
		EDV_ARCHIVER_OP_SHOW_FIND_BAR
	);
	UPDATE_ACCELKEY(
		archiver->show_status_bar_micheck,
		EDV_ARCHIVER_OP_SHOW_STATUS_BAR
	);
	UPDATE_ACCELKEY(
		archiver->contents_list_filter_mi,
		EDV_ARCHIVER_OP_LIST_FILTER
	);
	UPDATE_ACCELKEY(
		archiver->contents_list_auto_resize_columns_micheck,
		EDV_ARCHIVER_OP_CONTENTS_LIST_AUTO_RESIZE_COLUMNS
	);
	UPDATE_ACCELKEY(
		archiver->mime_types_mi,
		EDV_ARCHIVER_OP_MIME_TYPES
	);

	UPDATE_ACCELKEY(
		archiver->write_protect_mi,
		EDV_ARCHIVER_OP_WRITE_PROTECT
	);
	UPDATE_ACCELKEY(
		archiver->delete_method_recycle_micheck,
		EDV_ARCHIVER_OP_DELETE_METHOD_RECYCLE
	);
	UPDATE_ACCELKEY(
		archiver->delete_method_purge_micheck,
		EDV_ARCHIVER_OP_DELETE_METHOD_PURGE
	);
	UPDATE_ACCELKEY(
		archiver->options_mi,
		EDV_ARCHIVER_OP_OPTIONS
	);
	UPDATE_ACCELKEY(
		archiver->customize_mi,
		EDV_ARCHIVER_OP_CUSTOMIZE
	);

	UPDATE_ACCELKEY(
		archiver->window_new_browser_mi,
		EDV_ARCHIVER_OP_NEW_BROWSER
	);
	UPDATE_ACCELKEY(
		archiver->window_new_imbr_mi,
		EDV_ARCHIVER_OP_NEW_IMBR
	);
	UPDATE_ACCELKEY(
		archiver->window_new_archiver_mi,
		EDV_ARCHIVER_OP_NEW_ARCHIVER
	);
	UPDATE_ACCELKEY(
		archiver->window_recycle_bin_mi,
		EDV_ARCHIVER_OP_RECYCLE_BIN
	);

	UPDATE_ACCELKEY(
		archiver->help_contents_mi,
		EDV_ARCHIVER_OP_HELP_CONTENTS
	);
	UPDATE_ACCELKEY(
		archiver->help_archiver_mi,
		EDV_ARCHIVER_OP_HELP_ARCHIVER
	);
	UPDATE_ACCELKEY(
		archiver->help_keys_list_mi,
		EDV_ARCHIVER_OP_HELP_KEYS_LIST
	);
	UPDATE_ACCELKEY(
		archiver->help_common_operations_mi,
		EDV_ARCHIVER_OP_HELP_COMMON_OPERATIONS
	);
	UPDATE_ACCELKEY(
		archiver->help_about_mi,
		EDV_ARCHIVER_OP_HELP_ABOUT
	);

	UPDATE_ACCELKEY(
		archiver->contents_clist_add_mi,
		EDV_ARCHIVER_OP_ADD
	);
	UPDATE_ACCELKEY(
		archiver->contents_clist_extract_mi,
		EDV_ARCHIVER_OP_EXTRACT
	);
	UPDATE_ACCELKEY(
		archiver->contents_clist_extract_all_mi,
		EDV_ARCHIVER_OP_EXTRACT_ALL
	);
	UPDATE_ACCELKEY(
		archiver->contents_clist_delete_mi,
		EDV_ARCHIVER_OP_DELETE
	);
	UPDATE_ACCELKEY(
		archiver->contents_clist_properties_mi,
		EDV_ARCHIVER_OP_PROPERTIES
	);

#undef UPDATE_ACCELKEY
}

/*
 *	Recreates the Archiver's Tool Bar.
 */
void edv_archiver_tool_bar_regenerate(EDVArchiver *archiver)
{
	const gint border_minor = 2;
	GList		*glist,
			*items_list;
	GtkReliefStyle tb_relief;
	GtkWidget	*w,
			*parent;
	CfgIntList *intlist;
	CfgList *cfg_list;
	ToolBarButtonDecals tb_button_decals;
	ToolBarItem *item;
	EDVArchiverOp *op;
	EDVCore *core;

	if(archiver == NULL)
		return;

	parent = archiver->tool_bar_handle;
	core = archiver->core;
	cfg_list = core->cfg_list;

	/* Get the Tool Bar configuration */
	tb_button_decals = (ToolBarButtonDecals)EDV_GET_I(
		EDV_CFG_PARM_ARCHIVER_TOOL_BAR_BUTTON_DECALS
	);
	tb_relief = (GtkReliefStyle)EDV_GET_I(
		EDV_CFG_PARM_ARCHIVER_TOOL_BAR_RELIEF
	);
	intlist = EDV_GET_INT_LIST(EDV_CFG_PARM_ARCHIVER_TOOL_BAR);
	if(intlist == NULL)
		return;

	/* Delete the existing Tool Bar */
	GTK_WIDGET_DESTROY(archiver->tool_bar);
	archiver->tool_bar = NULL;

	/* Create the ToolBarItems list */
	items_list = NULL;
	for(glist = intlist->list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		/* Get the OPID */
		op = edv_archiver_op_match_by_id(
			archiver,
			(EDVArchiverOpID)glist->data
		);
		if(op == NULL)
			continue;

		/* Exclude this OPID from Tool Bars? */
		if(op->flags & EDV_OPID_EX_TOOL_BAR)
			continue;

		/* Create a new ToolBarItem from this OPID */
		item = ToolBarItemNew(
			op->tb_item_type,
			op->button_name,
			op->button_icon_data,
			op->tooltip,
			op->id,
			edv_archiver_op_activate_cb, op,
			edv_archiver_op_enter_cb, op,
			edv_archiver_op_leave_cb, op
		);
		if(item == NULL)
			continue;

		/* Append the new ToolBarItem to the list */
		items_list = g_list_append(
			items_list,
			item
		);
	}

	/* Create the Tool Bar */
	archiver->tool_bar = w = ToolBarNew(
		GTK_ORIENTATION_HORIZONTAL,
		tb_button_decals,
		tb_relief,
		GTK_POLICY_AUTOMATIC,
		items_list
	);
	gtk_widget_set_name(w, EDV_WIDGET_NAME_TOOL_BAR);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_minor);
	gtk_widget_show(w);

	/* Delete the Tool Bar items list */
	if(items_list != NULL)
	{
		g_list_foreach(
			items_list,
			(GFunc)ToolBarItemDelete,
			NULL
		);
		g_list_free(items_list);
	}
}

/*
 *	Recreates the Archiver Contents List right click menu.
 */
void edv_archiver_list_menu_regenerate(EDVArchiver *archiver)
{
	edv_pixmap_data *icon_data = NULL;
	EDVArchiverOpID id;
	GtkAccelGroup *accelgrp = NULL;
	GtkWidget	*w,
			*menu;

	if(archiver == NULL)
		return;

	/* Recreate menu */
	GTK_WIDGET_DESTROY(archiver->contents_clist_menu);
	archiver->contents_clist_menu = menu = GUIMenuCreate();

#define ADD_MENU_ITEM_LABEL		{	\
 EDVArchiverOp *op =				\
  edv_archiver_op_match_by_id(archiver, id);	\
 if(op != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu, GUI_MENU_ITEM_TYPE_LABEL, accelgrp,	\
   (guint8 **)((icon_data != NULL) ?		\
    icon_data : op->menu_icon_data),		\
   op->menu_name,				\
   op->accel_key, op->accel_mods,		\
   edv_archiver_menu_item_activate_cb, op	\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)edv_archiver_menu_item_enter_cb, op,	\
   (gpointer)edv_archiver_menu_item_leave_cb, op	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define ADD_MENU_ITEM_SEPARATOR	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_SEPARATOR,			\
  NULL,						\
  NULL, NULL,					\
  0, 0,						\
  NULL, NULL					\
 );						\
}

	id = EDV_ARCHIVER_OP_ADD;
	ADD_MENU_ITEM_LABEL
	GUISetMenuItemSetDefault(w, TRUE);
	archiver->contents_clist_add_mi = w;

	id = EDV_ARCHIVER_OP_EXTRACT;
	ADD_MENU_ITEM_LABEL
	archiver->contents_clist_extract_mi = w;

	id = EDV_ARCHIVER_OP_EXTRACT_ALL;
	ADD_MENU_ITEM_LABEL
	archiver->contents_clist_extract_all_mi = w;

	id = EDV_ARCHIVER_OP_DELETE;
	ADD_MENU_ITEM_LABEL
	archiver->contents_clist_delete_mi = w;

	ADD_MENU_ITEM_SEPARATOR

	id = EDV_ARCHIVER_OP_PROPERTIES;
	ADD_MENU_ITEM_LABEL
	archiver->contents_clist_properties_mi = w;

#undef ADD_MENU_ITEM_LABEL
#undef ADD_MENU_ITEM_SEPARATOR
}


/*
 *	Sets the Archiver's current values to the configuration.
 */
void edv_archiver_sync_configuration(EDVArchiver *archiver)
{
	GtkWidget *w;
	EDVCore *core;
	CfgList *cfg_list;

	if(archiver == NULL)
		return;

	core = archiver->core;
	if(core == NULL)
		return;

	cfg_list = core->cfg_list;

	/* Location history */
	w = archiver->location_combo;
	if(w != NULL)
		edv_save_text_file_glist(
			EDV_GET_S(EDV_CFG_PARM_FILE_ARCHIVER_LOCATION_HISTORY),
			GUIComboGetList(w)
		);

	/* Toplevel GtkWindow position and size */
	CfgGtkGetGtkWindow(
		cfg_list,
		EDV_CFG_PARM_ARCHIVER_X,
		EDV_CFG_PARM_ARCHIVER_Y,
		EDV_CFG_PARM_ARCHIVER_WIDTH,
		EDV_CFG_PARM_ARCHIVER_HEIGHT,
		archiver->toplevel
	);

	/* EDV_CFG_PARM_ARCHIVER_CONTENTS_COLUMN_WIDTH parameter is updated
	 * as they change in edv_archiver_resize_column_cb() or when the
	 * configuration list is changed directly
	 */

	/* EDV_CFG_PARM_ARCHIVER_CONTENTS_COLUMN parameter is updated
	 * when the configuration list is changed directly
	 */

	/* Contents list filter */
	if(STRISEMPTY(archiver->contents_list_filter))
		EDV_SET_S(
			EDV_CFG_PARM_ARCHIVER_CONTENTS_LIST_FILTER,
			"*"
		);
	else
		EDV_SET_S(
			EDV_CFG_PARM_ARCHIVER_CONTENTS_LIST_FILTER,
			archiver->contents_list_filter
		);
}


/*
 *	Checks if the archive is of a supported format, queries the
 *	user to remove the archive if the archive already exists,
 *	creates the new archive, sets the Archiver's title, location,
 *	and password, clears the Contents GtkCList, and gets the list of
 *	archive objects in the archive.
 *
 *	The path specifies the path to the archive. The Archiver's
 *	location history will be recorded prior to setting it with
 *	the value of path.
 *
 *	If the password is not NULL then password specifies the password.
 *
 *	If passive is TRUE then a list archive subprocess will be
 *	executed to get the list of objects in the archive asyncronously.
 *	Any current list archive subprocess will be stopped and deleted.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_archiver_new_archive(
	EDVArchiver *archiver,
	const gchar *path,
	const gchar *password,
	const gboolean passive
)
{
	gchar		*darch_path,
			*dpassword;
	GtkWidget	*toplevel,
			*status_bar;
	GtkCList *clist;
	CfgList *cfg_list;
	EDVVFSObject *obj;
	EDVCore *core;

	if((archiver == NULL) || STRISEMPTY(path))
	{
		errno = EINVAL;
		return(-2);
	}

#define CLEANUP_RETURN(_v_)	{		\
 const gint error_code = (gint)errno;		\
						\
 g_free(darch_path);				\
 g_free(dpassword);				\
 edv_archiver_set_busy(archiver, FALSE);	\
						\
 errno = (int)error_code;			\
						\
 return(_v_);					\
}

	edv_archiver_set_busy(archiver, TRUE);

	toplevel = archiver->toplevel;
	clist = GTK_CLIST(archiver->contents_clist);
	status_bar = archiver->status_bar;
	core = archiver->core;
	cfg_list = core->cfg_list;

        /* Check and warn if write protect is enabled */
        if(edv_check_master_write_protect(core, TRUE, toplevel))
        {
                edv_archiver_set_busy(archiver, FALSE);
                return(-5);
        }

	/* Copy the inputs since their original pointers may get
	 * modified if they refer to any current values on the
	 * Archiver
	 */
	darch_path = g_strdup(path);
	dpassword = STRDUP(password);

	/* Check if the specified path is a directory */
	if(edv_path_is_directory(darch_path))
	{
		gchar *msg = g_strdup_printf(
"Is a directory:\n\
\n\
    %s",
			darch_path
		);
		edv_play_sound_warning(core);
		edv_message_warning(
			"New Archive Failed",
			msg,
			NULL,
			toplevel
		);
		g_free(msg);
		errno = EINVAL;
		CLEANUP_RETURN(-2);
	}

	/* Check if the specified path refers to a non-supported
	 * archive format
	 */
	if(!edv_check_archive_format_supported(
		core,
		g_basename(darch_path)
	))
	{
		gchar *msg = g_strdup_printf(
"Format not supported:\n\
\n\
    %s\n\
\n\
Go to Help->Archiver for information on supported\n\
archive formats.",
			darch_path
		);
		edv_play_sound_warning(core);
		edv_message_warning(
			"New Archive Failed",
			msg,
			NULL,
			toplevel
		);
		g_free(msg);
		errno = EINVAL;
		CLEANUP_RETURN(-7);
	}

	/* Check if the path refers to a existing object */
	if(edv_path_exists(darch_path))
	{
		/* Confirm overwrite */
		gint response;
		gchar *msg = g_strdup_printf(
"Overwrite existing file:\n\
\n\
    %s",
			darch_path
		);
		edv_play_sound_warning(core);
		CDialogSetTransientFor(toplevel);
		response = CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"Confirme Escriba Para Reemplazar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Confirmer craser"
#elif defined(PROG_LANGUAGE_GERMAN)
"Besttigen Sie berschreibt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Confermare Sovrascrivere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bevestiig Beschrijft"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Confirme Overwrite"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Bekreft Overskriver"
#else
"Confirm Overwrite"
#endif
			,
			msg,
	                NULL,
	                CDIALOG_ICON_WARNING,
	                CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
	                CDIALOG_BTNFLAG_NO
		);
		g_free(msg);
		CDialogSetTransientFor(NULL);
		if(response == CDIALOG_RESPONSE_YES)
		{
			/* Remove the existing file */
			if(edv_unlink(darch_path))
			{
				const gint error_code = (gint)errno;
				gchar *msg = g_strdup_printf(
"Unable to remove the existing file:\n\
\n\
    %s\n\
\n\
%s.",
					darch_path,
					g_strerror(error_code)
				);
				edv_play_sound_error(core);
				edv_message_error(
					"New Archive Failed",
					msg,
					NULL,
					toplevel
				);
				g_free(msg);
				errno = error_code;
				CLEANUP_RETURN(-1);
			}

			/* Notify about the existing file being
			 * removed, this may cause the Archiver's
			 * listing to be cleared
			 */
			edv_emit_vfs_object_removed(
				core,
				darch_path
			);
		}
		else
		{
			errno = EINTR;
			CLEANUP_RETURN(-5);
		}
	}

	/* Create the new archive */
	if(edv_touch(
		darch_path,
		(gulong)-1,			/* Current time */
		TRUE				/* Create */
	))
	{
		/* Unable to create a new archive */
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"Unable to create the new archive:\n\
\n\
    %s\n\
\n\
%s.",
			darch_path,
			g_strerror(error_code)
		);
		edv_play_sound_error(core);
		edv_message_error(
			"New Archive Failed",
			msg,
			NULL,
			toplevel
		);
		g_free(msg);
		errno = (int)error_code;
		CLEANUP_RETURN(-1);
	}

	/* Get the new archive's statistics */
	obj = edv_vfs_object_stat(darch_path);
	if(obj == NULL)
	{
		/* Unable to create a new archive */
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"Unable to create the new archive:\n\
\n\
    %s\n\
\n\
%s.",
			darch_path,
			g_strerror(error_code)
		);
		edv_play_sound_error(core);
		edv_message_error(
			"New Archive Failed",
			msg,
			NULL,
			toplevel
		);
		g_free(msg);
		errno = (int)error_code;
		CLEANUP_RETURN(-1);
	}

	/* Set the Archiver's title and location to reflect the new
	 * archive
	 */
	edv_archiver_set_title(
		archiver,
		darch_path
	);
	edv_archiver_set_location(
		archiver,
		darch_path,
		TRUE				/* Record previous location */
	);
	edv_archiver_location_icon_update(
		archiver,
		darch_path
	);

	/* Notify about the new archive being created, this will cause
	 * the Archiver to get the new listing
	 */
	edv_emit_vfs_object_added(
		core,
		darch_path,
		obj
	);

	/* Delete the new archive's statistics */
	edv_vfs_object_delete(obj);

	edv_archiver_update_display(archiver);

	/* Queue an add objects to archive operation afterwards */
	edv_archiver_queue_op(
		archiver,
		EDV_ARCHIVER_OP_ADD
	);

	CLEANUP_RETURN(-1);
#undef CLEANUP_RETURN
}


/*
 *	Checks if the archive is of a supported format, queries the
 *	user to create a new archive if the archive does not exist, sets
 *	the Archiver's title, location, and password, clears the
 *	Contents GtkCList, and gets the list of archive objects in the
 *	archive.
 *
 *	The path specifies the path to the archive. The Archiver's
 *	location history will be recorded prior to setting it with
 *	the value of path.
 *
 *	If the password is not NULL then password specifies the password.
 *
 *	If passive is TRUE then a list archive subprocess will be
 *	executed to get the list of objects in the archive asyncronously.
 *	Any current list archive subprocess will be stopped and deleted.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_archiver_open_archive(
	EDVArchiver *archiver,
	const gchar *path,
	const gchar *password,
	const gboolean passive
)
{
	gchar		*darch_path,
			*dpassword;
	GtkWidget	*toplevel,
			*status_bar;
	GtkCList *clist;
	CfgList *cfg_list;
	EDVCore *core;

	if((archiver == NULL) || STRISEMPTY(path))
	{
		errno = EINVAL;
		return(-2);
	}

#define CLEANUP_RETURN(_v_)	{		\
 const gint error_code = (gint)errno;		\
						\
 g_free(darch_path);				\
 g_free(dpassword);				\
 edv_archiver_set_busy(archiver, FALSE);	\
						\
 errno = (int)error_code;			\
						\
 return(_v_);					\
}

	edv_archiver_set_busy(archiver, TRUE);

	toplevel = archiver->toplevel;
	clist = GTK_CLIST(archiver->contents_clist);
	status_bar = archiver->status_bar;
	core = archiver->core;
	cfg_list = core->cfg_list;

	/* Copy the inputs since their original pointers may get
	 * modified if they refer to any current values on the
	 * Archiver
	 */
	darch_path = g_strdup(path);
	dpassword = STRDUP(password);

	/* Check if the specified path is a directory */
	if(edv_path_is_directory(darch_path))
	{
		gchar *msg = g_strdup_printf(
"Is a directory:\n\
\n\
    %s",
			darch_path
		);
		edv_play_sound_warning(core);
		edv_message_warning(
			"Open Archive Failed",
			msg,
			NULL,
			toplevel
		);
		g_free(msg);
		errno = EINVAL;
		CLEANUP_RETURN(-2);
	}

	/* Check if the specified path refers to a non-supported
	 * archive format
	 */
	if(!edv_check_archive_format_supported(
		core,
		g_basename(darch_path)
	))
	{
		gchar *msg = g_strdup_printf(
"Format not supported:\n\
\n\
    %s\n\
\n\
Go to Help->Archiver for information on supported\n\
archive formats.",
			darch_path
		);
		edv_play_sound_warning(core);
		edv_message_warning(
			"Open Archive Failed",
			msg,
			NULL,
			toplevel
		);
		g_free(msg);
		errno = EINVAL;
		CLEANUP_RETURN(-7);
	}

	/* Check if the path refers to a non-existant object */
	if(!edv_path_exists(darch_path))
	{
		/* Does not exist, query the user to create a new archive */
		gint response;
		gchar *msg = g_strdup_printf(
"The archive \"%s\" does not exist.\n\
\n\
Create it?",
			g_basename(darch_path)
		);
		edv_play_sound_question(core);
		CDialogSetTransientFor(toplevel);
		response = CDialogGetResponse(
			"Create New Archive",
			msg,
			NULL,
			CDIALOG_ICON_QUESTION,
			CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
			CDIALOG_BTNFLAG_YES
		);
		g_free(msg);
		CDialogSetTransientFor(NULL);
		if(response == CDIALOG_RESPONSE_YES)
		{
			/* Check if the archive is of a supported
			 * format, query the user to remove the archive
			 * if the archive already exists, create the new
			 * archive, set the Archiver's title, location,
			 * and password, clear the Contents GtkCList,
			 * and get the list of archive objects in the
			 * archive
			 */
			const gint status = edv_archiver_new_archive(
				archiver,
				darch_path,
				password,
				TRUE		/* List passively */
			);
			CLEANUP_RETURN(status);
		}
		else
		{
			errno = EINTR;
			CLEANUP_RETURN(-5);
		}
	}

	/* Update the title, location, and password */
	edv_archiver_set_title(
		archiver,
		darch_path
	);
	edv_archiver_set_location(
		archiver,
		darch_path,
		TRUE				/* Record history */
	);
	edv_archiver_set_password(
		archiver,
		dpassword
	);
	edv_archiver_location_icon_update(
		archiver,
		darch_path
	);

	/* Clear the Contents GtkCList and get a new listing
	 *
	 * If passive is TRUE then a subprocess will be started to
	 * get the archive list asyncronously, which means that the
	 * Contents GtkCList will not be completely updated after
	 * this call
	 */
	gtk_clist_freeze(clist);
	edv_archiver_list_get(
		archiver,
		passive,
		EDV_GET_B(EDV_CFG_PARM_LISTS_SHOW_PROGRESS),
		EDV_GET_B(EDV_CFG_PARM_ARCHIVER_SHOW_ARCHIVE_COMMENTS_ON_OPEN)
	);
	gtk_clist_thaw(clist);

	edv_archiver_update_display(archiver);

	CLEANUP_RETURN(0);
#undef CLEANUP_RETURN
}


/*
 *	Creates a new EDVArchiverOp.
 */
static EDVArchiverOp *edv_archiver_op_new(void)
{
	return(EDV_ARCHIVER_OP(g_malloc0(sizeof(EDVArchiverOp))));
}

/*
 *	Deletes the EDVArchiverOp.
 */
static void edv_archiver_op_delete(EDVArchiverOp *op)
{
	if(op == NULL)
		return;

	g_free(op->button_name);
	g_free(op->menu_name);
	g_free(op->tooltip);
	g_free(op);
}

/*
 *	Creates a new EDVArchiverOp list with default values.
 */
GList *edv_archiver_op_ids_list_new(EDVArchiver *archiver)
{
	const gchar	*button_name,
			*menu_name,
			*tooltip;
	guint		accel_key,
			accel_mods;
	edv_pixmap_data	*button_icon_data,
			*menu_icon_data;
	GList *op_ids_list = NULL;
	ToolBarItemType tb_item_type;
	EDVOpIDFlags flags;
	EDVArchiverOpID id;
	void (*func_cb)(ToolBarItem *, gint, gpointer);
	void (*enter_func_cb)(ToolBarItem *, gint, gpointer);

#define APPEND_OPID	{				\
 EDVArchiverOp *op = edv_archiver_op_new();		\
 if(op != NULL) {					\
  op->id = id;						\
  op->flags = flags;					\
  op->tb_item_type = tb_item_type;			\
  op->func_cb = func_cb;				\
  op->enter_func_cb = enter_func_cb;			\
  op->button_name = STRDUP(button_name);		\
  op->menu_name = STRDUP(menu_name);			\
  op->tooltip = STRDUP(tooltip);			\
  op->accel_key = accel_key;				\
  op->accel_mods = accel_mods;				\
  op->button_icon_data = button_icon_data;		\
  op->menu_icon_data = menu_icon_data;			\
  op->archiver = archiver;				\
  op_ids_list = g_list_append(				\
   op_ids_list,						\
   op							\
  );							\
 }							\
}

	func_cb = edv_archiver_op_activate_cb;
	enter_func_cb = edv_archiver_op_enter_cb;


	id = EDV_ARCHIVER_OP_SEPARATOR;
	flags = EDV_OPID_ALLOW_MULTIPLE | EDV_OPID_NO_OP;
	tb_item_type = TOOL_BAR_ITEM_SEPARATOR;
	button_icon_data = (edv_pixmap_data *)icon_hsep_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Separador";
	menu_name = "Separador";
	tooltip = "Separador";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Sparateur";
	menu_name = "Sparateur";
	tooltip = "Sparateur";
#elif defined(PROG_LANGUAGE_GERMAN)
	button_name = "Separator";
	menu_name = "Separator";
	tooltip = "Separator";
#elif defined(PROG_LANGUAGE_ITALIAN)
	button_name = "Separatore";
	menu_name = "Separatore";
	tooltip = "Separatore";
#elif defined(PROG_LANGUAGE_DUTCH)
	button_name = "Separator";
	menu_name = "Separator";
	tooltip = "Separator";
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	button_name = "Separador";
	menu_name = "Separador";
	tooltip = "Separador";
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	button_name = "Separator";
	menu_name = "Separator";
	tooltip = "Separator";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Separator";
	menu_name = "Separator";
	tooltip = "Separator";
#else
	button_name = "Separator";
	menu_name = "Separator";
	tooltip = "Separator";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID


	id = EDV_ARCHIVER_OP_CLOSE;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_close_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Cierre";
	menu_name = "Cierre";
	tooltip = "Cierre esta ventana";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Proche";
	menu_name = "Proche";
	tooltip = "Fermer cette fentre";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Zamknij";
	menu_name = "Zamknij";
	tooltip = "Zamknij to okno";
#else
	button_name = "Close";
	menu_name = "Close";
	tooltip = "Close this window";
#endif

	accel_key = GDK_w;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_EXIT;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_exit_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "La Salida";
	menu_name = "La Salida";
	tooltip = "Cierra todas ventanas en la aplicacin y la";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Sortie";
	menu_name = "Sortie";
	tooltip = "Ferme toutes fentres dans l'application et sortie";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Wyjcie";
	menu_name = "Wyjcie";
	tooltip = "Zamknij wszystkie okna tej aplikacji i wyjd";
#else
	button_name = "Exit";
	menu_name = "Exit";
	tooltip = "Close all windows in the application and exit";
#endif
	accel_key = GDK_q;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID


	id = EDV_ARCHIVER_OP_SYNC_DISKS;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_save_20x20_xpm;
	menu_icon_data = button_icon_data;

#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Sincronizar Discos";
	menu_name = "Sincronizar Discos";
	tooltip = "Limpan cualquiera transferencias pendientes de datos al disco";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Disq Syn";
	menu_name = "Les Disques De Syn";
	tooltip = "Rincer des transferts de donnes en attente au disque";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Synchr. dyski";
	menu_name = "Synchronizuj dyski";
	tooltip = "Wykonaj wszystkie oczekujce transfery";
#else
	button_name = "Sync Disks";
	menu_name = "Sync Disks";
	tooltip = "Flush any pending data transfers to disk";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_HISTORY;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_trace_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Historia";
	menu_name = "La Historia...";
	tooltip = "La Historia de rdenes procesadas por esta aplicacin";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "L'Histoire";
	menu_name = "L'Histoire...";
	tooltip = "L'Histoire d'ordres traits par cette application";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Historia";
	menu_name = "Historia...";
	tooltip = "Historia komend wykonanych przez t aplikacj";
#else
	button_name = "History";
	menu_name = "History...";
	tooltip = "History of commands processed by this application";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_RUN;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_run_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Corra";
	menu_name = "Corra...";
	tooltip = "Corra un programa";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Courir";
	menu_name = "Courir...";
	tooltip = "Excute un programme";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Uruchom";
	menu_name = "Uruchom...";
	tooltip = "Uruchom program";
#else
	button_name = "Run";
	menu_name = "Run...";
	tooltip = "Run a program";
#endif
	accel_key = GDK_r;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_RUN_TERMINAL;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_terminal2_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "La Terminal";
	menu_name = "La Terminal";
	tooltip = "Corre una terminal";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Le Terminal";
	menu_name = "Le Terminal";
	tooltip = "Courir un terminal";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Terminal";
	menu_name = "Terminal";
	tooltip = "Uruchom terminal";
#else
	button_name = "Terminal";
	menu_name = "Terminal";
	tooltip = "Run a terminal";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_WRITE_PROTECT;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_write_protect_on_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Proteja";
	menu_name = "Proteja contra escritura";
	tooltip = "Habilite/inhabilita la proteccion contra escritura";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Protge";
	menu_name = "Protge En criture";
	tooltip = "Rend capable/rend infirme protge en criture";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Zabezpieczenie";
	menu_name = "Zabezpieczenie przed zapisem";
	tooltip = "Wcz/wycz zabezpieczenie przed zapisem";
#else
	button_name = "Protect";
	menu_name = "Master Write Protect";
	tooltip = "Turn the master write protect on/off";
#endif
	accel_key = GDK_y;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_DELETE_METHOD_RECYCLE;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = edv_get_recycle_bin_icon_data(
		EDV_ICON_SIZE_20,
		1,
		NULL
	);
	menu_icon_data = button_icon_data;
	button_name = "Recycle";
	menu_name = "Delete Method: Recycle";
	tooltip = "Set the delete method to recycle";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_DELETE_METHOD_PURGE;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_purge_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "Purge";
	menu_name = "Delete Method: Purge";
	tooltip = "Set the delete method to purge";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID


	id = EDV_ARCHIVER_OP_NEW;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_new_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Nuevo";
	menu_name = "Nuevo...";
	tooltip = "Crea un archivo nuevo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Nouveau";
	menu_name = "Nouveau...";
	tooltip = "Crer une nouvelle archive";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Nowe";
	menu_name = "Nowe...";
	tooltip = "Utwrz nowe archiwum";
#else
	button_name = "New";
	menu_name = "New...";
	tooltip = "Create a new archive";
#endif
	accel_key = GDK_n;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_OPEN;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_open_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Abrir";
	menu_name = "Abrir...";
	tooltip = "Abre un archivo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Ouvert";
	menu_name = "Ouvert...";
	tooltip = "Ouvrir une archive";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Otwrz";
	menu_name = "Otwrz...";
	tooltip = "Otwrz archiwum";
#else
	button_name = "Open";
	menu_name = "Open...";
	tooltip = "Open an archive";
#endif
	accel_key = GDK_F3;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_ADD;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_archive_add_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Agregue";
	menu_name = "Agregue...";
	tooltip = "Agregue objetos al archivo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Ajouter";
	menu_name = "Ajouter...";
	tooltip = "Ajoute des objets  l'archive";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Dodaj";
	menu_name = "Dodaj...";
	tooltip = "Dodaj obiekty do archiwum";
#else
	button_name = "Add";
	menu_name = "Add...";
	tooltip = "Add objects to the archive";
#endif
	accel_key = GDK_Insert;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_EXTRACT;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_archive_extract_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Extraer";
	menu_name = "Extraer...";
	tooltip = "Extraer los objetos escogidos del archivo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "L'Extrait";
	menu_name = "L'Extrait...";
	tooltip = "L'Extrait a choisi l'objet (s) de l'archive";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Rozpakuj";
	menu_name = "Rozpakuj...";
	tooltip = "Rozpakuj zaznaczony(e) obiekt(y) z archiwum";
#else
	button_name = "Extract";
	menu_name = "Extract...";
	tooltip = "Extract the selected object(s) from the archive";
#endif
	accel_key = GDK_Return;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_EXTRACT_ALL;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_archive_extract_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "ExtrTodo";
	menu_name = "Extraiga Todo...";
	tooltip = "Extrae todo del archivo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "ExtrTout";
	menu_name = "Extraire Tout...";
	tooltip = "Extraire tous objets de l'archive";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Rozpakuj wszystko";
	menu_name = "Rozpakuj wszystko...";
	tooltip = "Rozpakuj wszystkie obiekty z archiwum";
#else
	button_name = "ExtractAll";
	menu_name = "Extract All...";
	tooltip = "Extract all objects from the archive";
#endif
	accel_key = GDK_Return;
	accel_mods = GDK_SHIFT_MASK;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_DELETE;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_cancel_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Borre";
	menu_name = "Borre";
	tooltip = "Borre los objeto(s) escogido(s) del archivo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Effacer";
	menu_name = "Effacer";
	tooltip = "Efface l'objet choisi (s) de l'archive";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Usu";
	menu_name = "Usu";
	tooltip = "Usu zaznaczony(e) obiekt(y) z archiwum";
#else
	button_name = "Delete";
	menu_name = "Delete";
	tooltip = "Delete the selected object(s) from the archive";
#endif
	accel_key = GDK_Delete;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_COMMENTS;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_edit_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "Comments";
	menu_name = "Comments...";
	tooltip = "View/edit the archive's comments";
	accel_key = GDK_F9;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_CHECK;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_archive_check_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "Check";
	menu_name = "Check";
	tooltip = "Check the archive for problems";
	accel_key = GDK_c;
	accel_mods = GDK_SHIFT_MASK | GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_FIX;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_aid_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "Fix";
	menu_name = "Fix";
	tooltip = "Attempt to fix any problems with the archive";
	accel_key = GDK_f;
	accel_mods = GDK_SHIFT_MASK | GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_PROPERTIES;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_properties_20x20_xpm;;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Props";
	menu_name = "Las Propiedades";
	tooltip = "Modifican escogi las propiedades de objeto";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Props";
	menu_name = "Les Proprits";
	tooltip = "Modifier a choisi les proprits de l'objet";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Waciwoci";
	menu_name = "Waciwoci...";
	tooltip = "Modyfikuj waciwoci zaznaczonego obiektu";
#else
	button_name = "Properties";
	menu_name = "Properties...";
	tooltip = "View the selected object's properties";
#endif
	accel_key = GDK_Return;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID


	id = EDV_ARCHIVER_OP_SELECT_ALL;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_select_all_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "SelAll";
	menu_name = "Escoja Todo";
	tooltip = "Escoge todo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "SelAll";
	menu_name = "Choisir Tout";
	tooltip = "Choisir tous objets";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "ZaznWszyst";
	menu_name = "Zaznacz wszystko";
	tooltip = "Zaznacz wszystkie obiekty";
#else
	button_name = "SelAll";
	menu_name = "Select All";
	tooltip = "Select all objects";
#endif
	accel_key = GDK_a;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_UNSELECT_ALL;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_unselect_all_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "UnSelAll";
	menu_name = "Unescoja Todo";
	tooltip = "Unescoge todo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "UnSelAll";
	menu_name = "Unchoisir Tout";
	tooltip = "Unchoisir tous objets";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "OdznWszyst";
	menu_name = "Odznacz wszystko";
	tooltip = "Usu zaznaczenie ze wszystkich obiektw";
#else
	button_name = "UnSelAll";
	menu_name = "Unselect All";
	tooltip = "Unselect all objects";
#endif
	accel_key = GDK_u;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_INVERT_SELECTION;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_invert_selection_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "InvSel";
	menu_name = "Invierta Seleccin";
	tooltip =
 "Selectos todos objetos y unselect no seleccionados todos objetos escogidos";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "InvSel";
	menu_name = "Inverser Slection";
	tooltip =
 "Choisir tous objets dslections et deslectionner tous objets choisis";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "OdwrZazn";
	menu_name = "Odwr zaznaczenie";
	tooltip =
 "Zaznacza wszystkie niezaznaczone obiekty i usuwa zaznaczenie ze wszystkich zaznaczonych obiektw";
#else
	button_name = "InvSel";
	menu_name = "Invert Selection";
	tooltip =
 "Select all unselected objects and unselect all selected objects";
#endif
	accel_key = GDK_i;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_FIND;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_search_20x20_xpm;;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Buscar";
	menu_name = "Buscar";
	tooltip = "Buscar objeto(s) por atributos especificados";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Dcouverte";
	menu_name = "La Dcouverte";
	tooltip = "L'objet de Dcouverte (s) par les attributs spcifis ";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Znajd";
	menu_name = "Znajd...";
	tooltip = "Znajd obiekt(y) o podanych atrybutach";
#else
	button_name = "Find";
	menu_name = "Find...";
	tooltip = "Find object(s) by specific attributes";
#endif
	accel_key = GDK_f;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID


	id = EDV_ARCHIVER_OP_REFRESH;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_reload_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Refresque";
	menu_name = "Refresque";
	tooltip = "Refresque objetos de archivo que listan";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Rafrachir";
	menu_name = "Rafrachir";
	tooltip = "Rafrachit les objets d'archive numrants";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Odwie";
	menu_name = "Odwie";
	tooltip = "Odwiea list obiektw archiwum";
#else
	button_name = "Refresh";
	menu_name = "Refresh";
	tooltip = "Refresh the archive objects listing";
#endif
	accel_key = GDK_F5;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_REFRESH_ALL;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_reload_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Refresque Todo";
	menu_name = "Refresque Todo";
	tooltip = "Refresca todas listas";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Rafrachir Tout";
	menu_name = "Rafrachir Tout";
	tooltip = "Rafrachir toutes listes";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Odwie wszystko";
	menu_name = "Odwie wszystko";
	tooltip = "Odwiea wszystkie listy";
#else
	button_name = "Refresh All";
	menu_name = "Refresh All";
	tooltip = "Refresh all listings";
#endif

	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_STOP;
	flags = 0;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	button_icon_data = (edv_pixmap_data *)icon_stop_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "Stop";
	menu_name = "Stop Loading";
	tooltip = "Stop the current operation";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_STATISTICS;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_info_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "Stats";
	menu_name = "Statistics...";
	tooltip = "View the archive's statistics";
	accel_key = GDK_F10;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_SHOW_TOOL_BAR;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	flags = EDV_OPID_EX_TOOL_BAR;
	button_icon_data = (edv_pixmap_data *)icon_tool_bar_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "ToolBar";
	menu_name = "Tool Bar";
	tooltip = "La Exposicin/la barra de herramienta de cuero";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "ToolBar";
	menu_name = "Tool Bar";
	tooltip = "La barre d'outil de montre/peau";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PasNarz";
	menu_name = "Pasek narzdzi";
	tooltip = "Poka/ukryj pasek narzdzi";
#else
	button_name = "ToolBar";
	menu_name = "Tool Bar";
	tooltip = "Show/hide tool bar";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_SHOW_LOCATION_BAR;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_tool_bar_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "LocBar";
	menu_name = "Location Bar";
	tooltip = "La exposicin/la barra de la ubicacin del cuero";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "LocBar";
	menu_name = "Location Bar";
	tooltip = "La barre d'emplacement de montre/peau";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Pascie";
	menu_name = "Pasek cieki";
	tooltip = "Poka/ukryj pasek cieki";
#else
	button_name = "LocBar";
	menu_name = "Location Bar";
	tooltip = "Show/hide location bar";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_SHOW_FIND_BAR;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_tool_bar_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "FindBar";
	menu_name = "Find Bar";
	tooltip = "La exposicin/la barra del hallazgo del cuero";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "FindBar";
	menu_name = "Find Bar";
	tooltip = "La barre de dcouverte de montre/peau";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PasSzuk";
	menu_name = "Pasek szukania";
	tooltip = "Poka/ukryj pasek szukania";
#else
	button_name = "FindBar";
	menu_name = "Find Bar";
	tooltip = "Show/hide find bar";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_SHOW_STATUS_BAR;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_status_bar_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "StatusBar";
	menu_name = "Status Bar";
	tooltip = "La exposicin/la barra de la posicin del cuero";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "StatusBar";
	menu_name = "Status Bar";
	tooltip = "La barre de statut de montre/peau";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PasStatus";
	menu_name = "Pasek statusu";
	tooltip = "Poka/ukryj pasek statusu";
#else
	button_name = "StatusBar";
	menu_name = "Status Bar";
	tooltip = "Show/hide status bar";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_LIST_FILTER;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_wildcards_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "Filter";
	menu_name = "Set Name Filter...";
	tooltip = "Set the filter to list only the objects that have\
 a specific name pattern";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_CONTENTS_LIST_AUTO_RESIZE_COLUMNS;
	tb_item_type = TOOL_BAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_resize_column_20x20_xpm;
	menu_icon_data = NULL;
	button_name = "ReSzCol";
	menu_name = "Auto Resize Columns";
	tooltip = "Automatically resize the columns each time the list is updated";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID


	id = EDV_ARCHIVER_OP_MIME_TYPES;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_mimetypes_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "MIMETypes";
	menu_name = "MIME Types...";
	tooltip = "El panorama/modifica MIME Types";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "MIMETypes";
	menu_name = "MIME Types...";
	tooltip = "Vue/modifie MIME Types";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "TypyMIME";
	menu_name = "Typy MIME...";
	tooltip = "Obejrzyj/zmodyfikuj typy MIME";
#else
	button_name = "MIMETypes";
	menu_name = "MIME Types...";
	tooltip = "View/modify MIME Types";
#endif
	accel_key = GDK_m;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID


	id = EDV_ARCHIVER_OP_NEW_BROWSER;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_endeavour_file_browser_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "FBrowser";
	menu_name = "Nuevo Examinador De Archivos";
	tooltip = "Muestra un nuevo examinador de archivos";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "FBrowser";
	menu_name = "Nouveau Browser de Fichier";
	tooltip = "Crer un nouveau browser de fichier";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PrzeglPlik";
	menu_name = "Nowa przegldarka plikw";
	tooltip = "Utwrz now przegldark plikw";
#else
	button_name = "FBrowser";
	menu_name = "New File Browser";
	tooltip = "Create a new file browser";
#endif
	accel_key = GDK_1;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_NEW_IMBR;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_endeavour_image_browser_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "IBrowser";
	menu_name = "Nuevo Examinador De Imagenes";
	tooltip = "Muestra un nuevo examinador de imagenes";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "IBrowser";
	menu_name = "Nouveau Browser D'Image";
	tooltip = "Crer un nouveau browser d'image";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PrzeglObr";
	menu_name = "Nowa przegldarka obrazw";
	tooltip = "Utwrz now przegldark obrazw";
#else
	button_name = "IBrowser";
	menu_name = "New Image Browser";
	tooltip = "Create a new image browser";
#endif
	accel_key = GDK_2;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_NEW_ARCHIVER;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_endeavour_archiver_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Archiver";
	menu_name = "Nuevo Compresor De Archivos";
	tooltip = "Muestra un archivo comprimido nuevo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Archiver";
	menu_name = "Nouveau Archiver";
	tooltip = "Crer un nouveau archiver";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Archiwizator";
	menu_name = "Nowy archwizator";
	tooltip = "Utwrz nowego archwizatora";
#else
	button_name = "Archiver";
	menu_name = "New Archiver";
	tooltip = "Create a new archiver";
#endif
	accel_key = GDK_3;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_RECYCLE_BIN;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = edv_get_recycle_bin_icon_data(
		EDV_ICON_SIZE_20,
		1,
		NULL
	);
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "RecBin";
	menu_name = "El Cajn De Reciclaje";
	tooltip = "Abrir Cajn de Reciclaje (basura)";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "RecBin";
	menu_name = "Recycler L'Huche";
	tooltip = "Recycler l'huche (la poubelle)";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Kosz";
	menu_name = "Kosz";
	tooltip = "Otwrz kosz";
#else
	button_name = "RecBin";
	menu_name = "Recycle Bin";
	tooltip = "Open Recycle Bin";
#endif
	accel_key = GDK_4;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID


	id = EDV_ARCHIVER_OP_OPTIONS;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_options_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Las Opciones";
	menu_name = "Las Opciones...";
	tooltip = "Modifican estas opciones de la aplicacin";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Les Options";
	menu_name = "Les Options";
	tooltip = "Modifier ces options de l'application";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Opcje";
	menu_name = "Opcje...";
	tooltip = "Modyfikuj opcje tej aplikacji";
#else
	button_name = "Options";
	menu_name = "Options...";
	tooltip = "Modify this application's options";
#endif
	accel_key = GDK_o;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_CUSTOMIZE;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_customize_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Personal";
	menu_name = "Personaliza...";
	tooltip = "Personaliza la mirada y se siente de esta aplicacin";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Personnal";
	menu_name = "Personnalise...";
	tooltip = "Personnalise le regard et sens de cette application";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Dostosuj";
	menu_name = "Dostosuj...";
	tooltip = "Dostosuj wygld i zachowanie tej aplikacji";
#else
	button_name = "Custom";
	menu_name = "Customize...";
	tooltip = "Customize the look and feel of this application";
#endif
	accel_key = GDK_t;
	accel_mods = GDK_CONTROL_MASK;
	APPEND_OPID


	id = EDV_ARCHIVER_OP_HELP_CONTENTS;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_help_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "La Ayuda";
	menu_name = "El Contenido";
	tooltip = "El Indice de toda ayuda pagina";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "L'Aide";
	menu_name = "Les Contenus";
	tooltip = "L'Index de toutes pages d'aide";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Pomoc";
	menu_name = "Tematy pomocy";
	tooltip = "Spis treci wszystkich stron pomocy";
#else
	button_name = "Help";
	menu_name = "Contents";
	tooltip = "Index of all help pages";
#endif
	accel_key = GDK_F1;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_HELP_FILE_BROWSER;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_help_20x20_xpm;
	menu_icon_data = NULL;
	button_name = "Help";
	menu_name = "File Browser";
	tooltip = "Help on using the file browser";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_HELP_IMAGE_BROWSER;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_help_20x20_xpm;
	menu_icon_data = NULL;
	button_name = "Help";
	menu_name = "Image Browser";
	tooltip = "Help on using the image browser";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_HELP_ARCHIVER;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_help_20x20_xpm;
	menu_icon_data = NULL;
	button_name = "Help";
	menu_name = "Archiver";
	tooltip = "Help on using the archiver";
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_HELP_RECYCLE_BIN;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_help_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_FRENCH)
	button_name = "Aide";
	menu_name = "Corbeille";
	tooltip = "Aide sur l'utilisation de la Corbeille";
#else
	button_name = "Help";
	menu_name = "Recycle Bin";
	tooltip = "Help on using the recycle bin";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_HELP_KEYS_LIST;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_help_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "HelpKeys";
	menu_name = "La Lista De Llaves";
	tooltip = "La Lista de todo teclado las operaciones claves";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "HelpKeys";
	menu_name = "La Liste De Clefs";
	tooltip = "La Liste de tout clavier toutes oprations principales";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PomocKlaw";
	menu_name = "Lista klawiszy";
	tooltip = "Lista skrtw klawiaturowych";
#else
	button_name = "HelpKeys";
	menu_name = "Keys List";
	tooltip = "List of all keyboard key operations";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_HELP_COMMON_OPERATIONS;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_help_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "HelpMIME";
	menu_name = "Mime Types";
	tooltip = "Ayude a establecer MIME Types";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "HelpMIME";
	menu_name = "Mime Types";
	tooltip = "Aider sur montant MIME Types";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PomocMIME";
	menu_name = "Typy MIME";
	tooltip = "Pomoc w ustawianiu typw MIME";
#else
	button_name = "HelpMIME";
	menu_name = "Mime Types";
	tooltip = "Help on setting up MIME Types";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID

	id = EDV_ARCHIVER_OP_HELP_ABOUT;
	tb_item_type = TOOL_BAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (edv_pixmap_data *)icon_about_20x20_xpm;
	menu_icon_data = (edv_pixmap_data *)icon_about_20x20_xpm;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Acerca";
	menu_name = "Acerca De";
	tooltip = "Acerca de esta aplicacin";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "De";
	menu_name = "De";
	tooltip = "De cette application";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "O programie";
	menu_name = "O programie";
	tooltip = "Informacje o programie";
#else
	button_name = "About";
	menu_name = "About";
	tooltip = "About this application";
#endif
	accel_key = 0;
	accel_mods = 0;
	APPEND_OPID


	return(op_ids_list);
#undef APPEND_OPID
}

/*
 *	Deletes the Archiver Operation IDs list.
 */
void edv_archiver_op_ids_list_delete(GList *op_ids_list)
{
	if(op_ids_list == NULL)
		return;

	g_list_foreach(
		op_ids_list,
		(GFunc)edv_archiver_op_delete,
		NULL
	);
	g_list_free(op_ids_list);
}


/*
 *	Creates a new Accel Key List with default values.
 */
CfgAccelkeyList *edv_archiver_accelkeys_new(void)
{
	GList		*glist,
			*op_ids_list,
			*ak_glist = NULL;
	CfgAccelkeyList *ak_list;
	EDVArchiverOp *op;

	op_ids_list = edv_archiver_op_ids_list_new(NULL);
	for(glist = op_ids_list; glist != NULL; glist = g_list_next(glist))
	{
		op = EDV_ARCHIVER_OP(glist->data);
		if(op == NULL)
			continue;

		if(op->flags & EDV_OPID_NO_OP)
			continue;

		ak_glist = g_list_append(
			ak_glist,
			CFGAccelkeyNew(
				(gint)op->id,
				op->accel_key,
				op->accel_mods
			)
		);
	}
	edv_archiver_op_ids_list_delete(op_ids_list);

	ak_list = CFGAccelkeyListNew(ak_glist);

	g_list_foreach(ak_glist, (GFunc)CFGAccelkeyDelete, NULL);
	g_list_free(ak_glist);

	return(ak_list);
}


/*
 *	Queues an operation.
 */
void edv_archiver_queue_op(
	EDVArchiver *archiver,
	const EDVArchiverOpID id
)
{
	EDVArchiverOpQueue *q;

	if(archiver == NULL)
		return;

	q = EDV_ARCHIVER_OP_QUEUE(g_malloc0(sizeof(EDVArchiverOpQueue)));
	if(q == NULL)
		return;

	archiver->queued_ops_list = g_list_append(
		archiver->queued_ops_list,
		q
	);
	q->archiver = archiver;
	q->op_id = id;
	q->idle_id = gtk_idle_add_priority(
		G_PRIORITY_LOW,
		edv_archiver_op_idle_cb, q
	);
}


/*
 *	Creates the Archiver's Menu Bar.
 *
 *	This function should be called by edv_archiver_new().
 */
static void edv_archiver_create_menu_bar(
	EDVArchiver *archiver,
	GtkWidget *parent
)
{
	edv_pixmap_data *icon_data = NULL;
	EDVArchiverOpID id;
	GtkAccelGroup *accelgrp = NULL;
	GtkWidget	*w,
			*menu,
			*menu_bar;

	/* Create the GtkMenuBar */
	archiver->menu_bar = menu_bar = w = GUIMenuBarCreate(NULL);
	gtk_widget_set_name(w, EDV_WIDGET_NAME_MENU_BAR);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);

#define ADD_MENU_ITEM_LABEL		{	\
 EDVArchiverOp *op =				\
  edv_archiver_op_match_by_id(archiver, id);	\
 if(op != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu, GUI_MENU_ITEM_TYPE_LABEL, accelgrp,	\
   (guint8 **)((icon_data != NULL) ?		\
    icon_data : op->menu_icon_data),		\
   op->menu_name,				\
   op->accel_key, op->accel_mods,		\
   edv_archiver_menu_item_activate_cb, op	\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)edv_archiver_menu_item_enter_cb, op,	\
   (gpointer)edv_archiver_menu_item_leave_cb, op	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define ADD_MENU_ITEM_SUBMENU_LABEL	{	\
 EDVArchiverOp *op =				\
  edv_archiver_op_match_by_id(archiver, id);	\
 if(op != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu, GUI_MENU_ITEM_TYPE_SUBMENU, accelgrp,	\
   (guint8 **)((icon_data != NULL) ?		\
    icon_data : op->menu_icon_data),		\
   op->menu_name,				\
   op->accel_key, op->accel_mods,		\
   NULL, op					\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)edv_archiver_menu_item_enter_cb, op,	\
   (gpointer)edv_archiver_menu_item_leave_cb, op	\
  );						\
  if(w != NULL)					\
   GUIMenuItemSetSubMenu(w, submenu);		\
  if(w_record != NULL)				\
   *w_record = w;				\
 }						\
}

#define ADD_MENU_ITEM_CHECK		{	\
 EDVArchiverOp *op =				\
  edv_archiver_op_match_by_id(archiver, id);	\
 if(op != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu,					\
   GUI_MENU_ITEM_TYPE_CHECK,			\
   accelgrp,					\
   NULL,					\
   op->menu_name,				\
   op->accel_key, op->accel_mods,		\
   edv_archiver_menu_item_activate_cb, op	\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)edv_archiver_menu_item_enter_cb, op,	\
   (gpointer)edv_archiver_menu_item_leave_cb, op	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define ADD_MENU_ITEM_SEPARATOR		{	\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_SEPARATOR,			\
  NULL,						\
  NULL,						\
  NULL,						\
  0, 0,						\
  NULL, NULL					\
 );						\
}

	/* Create Menus */
	/* File */
	menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
		id = EDV_ARCHIVER_OP_NEW;
		ADD_MENU_ITEM_LABEL
		archiver->new_mi = w;

		id = EDV_ARCHIVER_OP_OPEN;
		ADD_MENU_ITEM_LABEL
		archiver->open_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_ARCHIVER_OP_SYNC_DISKS;
		ADD_MENU_ITEM_LABEL
		archiver->sync_disks_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_ARCHIVER_OP_HISTORY;
		ADD_MENU_ITEM_LABEL
		archiver->history_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_ARCHIVER_OP_RUN;
		ADD_MENU_ITEM_LABEL
		archiver->run_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_ARCHIVER_OP_CLOSE;
		ADD_MENU_ITEM_LABEL
		archiver->close_mi = w;
	}
	GUIMenuAddToMenuBar(
		menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Archivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Fichier"
#elif defined(PROG_LANGUAGE_GERMAN)
"Akte"
#elif defined(PROG_LANGUAGE_ITALIAN)
"File"
#elif defined(PROG_LANGUAGE_DUTCH)
"Dossier"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Arquivo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Arkiv"
#elif defined(PROG_LANGUAGE_POLISH)
"Plik"
#else
"File"
#endif
		, GTK_JUSTIFY_LEFT
	);

	/* Edit */
	menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
		id = EDV_ARCHIVER_OP_ADD;
		ADD_MENU_ITEM_LABEL
		archiver->edit_add_mi = w;

		id = EDV_ARCHIVER_OP_EXTRACT;
		ADD_MENU_ITEM_LABEL
		archiver->edit_extract_mi = w;

		id = EDV_ARCHIVER_OP_EXTRACT_ALL;
		ADD_MENU_ITEM_LABEL
		archiver->edit_extract_all_mi = w;

		id = EDV_ARCHIVER_OP_DELETE;
		ADD_MENU_ITEM_LABEL
		archiver->edit_delete_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_ARCHIVER_OP_COMMENTS;
		ADD_MENU_ITEM_LABEL
		archiver->comments_mi = w;

		id = EDV_ARCHIVER_OP_CHECK;
		ADD_MENU_ITEM_LABEL
		archiver->check_mi = w;

		id = EDV_ARCHIVER_OP_FIX;
		ADD_MENU_ITEM_LABEL
		archiver->fix_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_ARCHIVER_OP_SELECT_ALL;
		ADD_MENU_ITEM_LABEL
		archiver->select_all_mi = w;

		id = EDV_ARCHIVER_OP_UNSELECT_ALL;
		ADD_MENU_ITEM_LABEL
		archiver->unselect_all_mi = w;

		id = EDV_ARCHIVER_OP_INVERT_SELECTION;
		ADD_MENU_ITEM_LABEL
		archiver->invert_selection_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_ARCHIVER_OP_FIND;
		ADD_MENU_ITEM_LABEL
		archiver->find_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_ARCHIVER_OP_PROPERTIES;
		ADD_MENU_ITEM_LABEL
		archiver->properties_mi = w;
	}
	GUIMenuAddToMenuBar(
		menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Editar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Editer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Redigieren"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Redigere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bewerking"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Edite"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Rediger"
#elif defined(PROG_LANGUAGE_POLISH)
"Edycja"
#else
"Edit"
#endif
		, GTK_JUSTIFY_LEFT
	);

	/* View */
	menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
		id = EDV_ARCHIVER_OP_STOP;
		ADD_MENU_ITEM_LABEL
		archiver->stop_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_ARCHIVER_OP_REFRESH;
		ADD_MENU_ITEM_LABEL
		archiver->refresh_mi = w;

		id = EDV_ARCHIVER_OP_REFRESH_ALL;
		ADD_MENU_ITEM_LABEL
		archiver->refresh_all_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_ARCHIVER_OP_STATISTICS;
		ADD_MENU_ITEM_LABEL
		archiver->statistics_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_ARCHIVER_OP_SHOW_TOOL_BAR;
		ADD_MENU_ITEM_CHECK
		archiver->show_tool_bar_micheck = w;

		id = EDV_ARCHIVER_OP_SHOW_LOCATION_BAR;
		ADD_MENU_ITEM_CHECK
		archiver->show_location_bar_micheck = w;

		id = EDV_ARCHIVER_OP_SHOW_FIND_BAR;
		ADD_MENU_ITEM_CHECK
		archiver->show_find_bar_micheck = w;

		id = EDV_ARCHIVER_OP_SHOW_STATUS_BAR;
		ADD_MENU_ITEM_CHECK
		archiver->show_status_bar_micheck = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_ARCHIVER_OP_CONTENTS_LIST_AUTO_RESIZE_COLUMNS;
		ADD_MENU_ITEM_CHECK
		archiver->contents_list_auto_resize_columns_micheck = w;

		id = EDV_ARCHIVER_OP_LIST_FILTER;
		ADD_MENU_ITEM_LABEL
		archiver->contents_list_filter_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_ARCHIVER_OP_MIME_TYPES;
		ADD_MENU_ITEM_LABEL
		archiver->mime_types_mi = w;
	}
	GUIMenuAddToMenuBar(
		menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Vista"
#elif defined(PROG_LANGUAGE_FRENCH)
"Vue"
#elif defined(PROG_LANGUAGE_GERMAN)
"Blick"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Veduta"
#elif defined(PROG_LANGUAGE_DUTCH)
"Overzicht"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Vista"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Sikt"
#elif defined(PROG_LANGUAGE_POLISH)
"Widok"
#else
"View"
#endif
		, GTK_JUSTIFY_LEFT
	);

	/* Settings */
	menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
		id = EDV_ARCHIVER_OP_WRITE_PROTECT;
		ADD_MENU_ITEM_LABEL
		archiver->write_protect_mi = w;

		id = EDV_ARCHIVER_OP_DELETE_METHOD_RECYCLE;
		ADD_MENU_ITEM_CHECK
		archiver->delete_method_recycle_micheck = w;

		id = EDV_ARCHIVER_OP_DELETE_METHOD_PURGE;
		ADD_MENU_ITEM_CHECK
		archiver->delete_method_purge_micheck = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_ARCHIVER_OP_OPTIONS;
		ADD_MENU_ITEM_LABEL
		archiver->options_mi = w;

		id = EDV_ARCHIVER_OP_CUSTOMIZE;
		ADD_MENU_ITEM_LABEL
		archiver->customize_mi = w;
	}
	GUIMenuAddToMenuBar(
		menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Configuracion"
#elif defined(PROG_LANGUAGE_FRENCH)
"Montages"
#elif defined(PROG_LANGUAGE_GERMAN)
"Setzen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Montaggi"
#elif defined(PROG_LANGUAGE_DUTCH)
"Zetten"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Por"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Innstillinger"
#elif defined(PROG_LANGUAGE_POLISH)
"Ustawienia"
#else
"Settings"
#endif
		, GTK_JUSTIFY_LEFT
	);

	/* Windows */
	menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
		id = EDV_ARCHIVER_OP_NEW_BROWSER;
		ADD_MENU_ITEM_LABEL
		archiver->window_new_browser_mi = w;

		id = EDV_ARCHIVER_OP_NEW_IMBR;
		ADD_MENU_ITEM_LABEL
		archiver->window_new_imbr_mi = w;

		id = EDV_ARCHIVER_OP_NEW_ARCHIVER;
		ADD_MENU_ITEM_LABEL
		archiver->window_new_archiver_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_ARCHIVER_OP_RECYCLE_BIN;
		ADD_MENU_ITEM_LABEL
		archiver->window_recycle_bin_mi = w;
	}
	GUIMenuAddToMenuBar(
		menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Ventanas"
#elif defined(PROG_LANGUAGE_FRENCH)
"Windows"
#elif defined(PROG_LANGUAGE_GERMAN)
"Fenster"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Finestre"
#elif defined(PROG_LANGUAGE_DUTCH)
"Ramen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Janelas"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Vinduer"
#elif defined(PROG_LANGUAGE_POLISH)
"Okna"
#else
"Windows"
#endif
		, GTK_JUSTIFY_LEFT
	);

	/* Help */
	menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
		id = EDV_ARCHIVER_OP_HELP_CONTENTS;
		ADD_MENU_ITEM_LABEL
		archiver->help_contents_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_ARCHIVER_OP_HELP_ARCHIVER;
		ADD_MENU_ITEM_LABEL
		archiver->help_archiver_mi = w;

		id = EDV_ARCHIVER_OP_HELP_KEYS_LIST;
		ADD_MENU_ITEM_LABEL
		archiver->help_keys_list_mi = w;

		id = EDV_ARCHIVER_OP_HELP_COMMON_OPERATIONS;
		ADD_MENU_ITEM_LABEL
		archiver->help_common_operations_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		id = EDV_ARCHIVER_OP_HELP_ABOUT;
		ADD_MENU_ITEM_LABEL
		archiver->help_about_mi = w;
	}
	GUIMenuAddToMenuBar(
		menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Ayuda"
#elif defined(PROG_LANGUAGE_FRENCH)
"Aide"
#elif defined(PROG_LANGUAGE_GERMAN)
"Hilfe"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Aiuto"
#elif defined(PROG_LANGUAGE_DUTCH)
"Hulp"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Ajuda"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Hjelp"
#elif defined(PROG_LANGUAGE_POLISH)
"Pomoc"
#else
"Help"
#endif
		, GTK_JUSTIFY_RIGHT
	);

#undef ADD_MENU_ITEM_LABEL
#undef ADD_MENU_ITEM_SUBMENU_LABEL
#undef ADD_MENU_ITEM_CHECK
#undef ADD_MENU_ITEM_SEPARATOR
}

/*
 *	Creates a new Archiver.
 */
EDVArchiver *edv_archiver_new(
	EDVCore *core,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry
)
{
	const gint	border_major = 5;
	gpointer	entry_rtn,
					combo_rtn;
	const gchar	*wm_name,
					*wm_class;
	GList *glist;
	GtkRcStyle	*standard_rcstyle,
					*lists_rcstyle;
	GtkWidget	*w,
					*parent, *parent2, *parent3, *parent4,
					*toplevel;
	GtkEntry *entry;
	GtkCombo *combo;
	GtkCList *clist;
	const GtkTargetEntry dnd_tar_types[] = {
{GUI_TARGET_NAME_TEXT_PLAIN,	0,	EDV_DND_INFO_TEXT_PLAIN},
{GUI_TARGET_NAME_TEXT_URI_LIST,	0,	EDV_DND_INFO_TEXT_URI_LIST},
{GUI_TARGET_NAME_STRING,	0,	EDV_DND_INFO_STRING}
	};
	const GtkTargetEntry dnd_src_types[] = {
{EDV_DND_TARGET_ARCHIVE_OBJECT,	0,	EDV_DND_INFO_ARCHIVE_OBJECT}
	};
	CfgList *cfg_list;
	EDVArchiver *archiver;

	if(core == NULL)
		return(NULL);

	cfg_list = core->cfg_list;

	standard_rcstyle = core->standard_rcstyle;
	lists_rcstyle = core->lists_rcstyle;

	wm_name = core->wm_name;
	wm_class = core->wm_class;

	/* Create the Archiver */
	archiver = EDV_ARCHIVER(g_malloc0(sizeof(EDVArchiver)));
	if(archiver == NULL)
		return(NULL);

	archiver->toplevel = toplevel = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	archiver->core = core;
#if 0
	archiver->flags = 0;
	archiver->freeze_count = 0;
	archiver->busy_count = 0;
	archiver->passive_busy_count = 0;
#endif
	archiver->op_ids_list = edv_archiver_op_ids_list_new(archiver);

	if(EDV_GET_B(EDV_CFG_PARM_ARCHIVER_SHOW_TOOL_BAR))
		archiver->flags |= EDV_ARCHIVER_TOOL_BAR_MAPPED;
	if(EDV_GET_B(EDV_CFG_PARM_ARCHIVER_SHOW_LOCATION_BAR))
		archiver->flags |= EDV_ARCHIVER_LOCATION_BAR_MAPPED;
	if(EDV_GET_B(EDV_CFG_PARM_ARCHIVER_SHOW_FIND_BAR))
		archiver->flags |= EDV_ARCHIVER_FIND_BAR_MAPPED;
	if(EDV_GET_B(EDV_CFG_PARM_ARCHIVER_SHOW_STATUS_BAR))
		archiver->flags |= EDV_ARCHIVER_STATUS_BAR_MAPPED;

	archiver->contents_list_filter = STRDUP(EDV_GET_S(
		EDV_CFG_PARM_ARCHIVER_CONTENTS_LIST_FILTER
	));
	archiver->contents_clist_selected_row = -1;
	archiver->last_nrecycle_bin_items = -1;
	archiver->last_write_protect_state = -1;
#if 0
	archiver->subprocess = NULL;
	archiver->archive_info_dialog = NULL;
#endif

	archiver->freeze_count++;

	/* Toplevel GtkWindow */
	w = toplevel;
	if(!STRISEMPTY(wm_name) && !STRISEMPTY(wm_class))
		gtk_window_set_wmclass(GTK_WINDOW(w), wm_name, wm_class);
	else
		gtk_window_set_wmclass(
			GTK_WINDOW(w), EDV_ARCHIVER_WM_CLASS_WINDOW_NAME, PROG_NAME
		);
	gtk_widget_set_name(w, EDV_WIDGET_NAME_ARCHIVER_TOPLEVEL);
	gtk_window_set_policy(GTK_WINDOW(w), TRUE, TRUE, FALSE);
	if((geometry_flags != 0) && (geometry != NULL))
	{
		if((geometry_flags & GDK_GEOMETRY_X) || (geometry_flags & GDK_GEOMETRY_Y))
			gtk_widget_set_uposition(w, geometry->x, geometry->y);

		if((geometry_flags & GDK_GEOMETRY_WIDTH) || (geometry_flags & GDK_GEOMETRY_HEIGHT))
			gtk_widget_set_usize(w, geometry->width, geometry->height);
		else
			CfgGtkSetGtkWindow(
				cfg_list,
				NULL, NULL,
				EDV_CFG_PARM_ARCHIVER_WIDTH,
				EDV_CFG_PARM_ARCHIVER_HEIGHT,
				w
			);
	}
	else
	{
		CfgGtkSetGtkWindow(
			cfg_list,
			EDV_CFG_PARM_ARCHIVER_X,
			EDV_CFG_PARM_ARCHIVER_Y,
			EDV_CFG_PARM_ARCHIVER_WIDTH,
			EDV_CFG_PARM_ARCHIVER_HEIGHT,
			w
		);
	}
	gtk_widget_add_events(
		w,
		GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "realize",
		GTK_SIGNAL_FUNC(edv_archiver_realize_cb), archiver
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "delete_event",
		GTK_SIGNAL_FUNC(edv_archiver_delete_event_cb), archiver
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(edv_archiver_key_event_cb), archiver
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(edv_archiver_key_event_cb), archiver
	);
	parent = w;


	/* Main GtkVBox */
	archiver->main_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;


	/* Tool Bars GtkVBox */
	w = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;


	/* Menu Bar GtkHandleBox */
	archiver->menu_bar_handle = w = gtk_handle_box_new();
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_attached",
		GTK_SIGNAL_FUNC(edv_archiver_handle_child_attached_cb), archiver
	);
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_detached",
		GTK_SIGNAL_FUNC(edv_archiver_handle_child_detached_cb), archiver
	);
	gtk_widget_show(w);
	/* Create menu bar */
	edv_archiver_create_menu_bar(archiver, w);


	/* Tool Bar GtkHandleBox */
	archiver->tool_bar_handle = w = gtk_handle_box_new();
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_attached",
		GTK_SIGNAL_FUNC(edv_archiver_handle_child_attached_cb), archiver
	);
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_detached",
		GTK_SIGNAL_FUNC(edv_archiver_handle_child_detached_cb), archiver
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	if(archiver->flags & EDV_ARCHIVER_TOOL_BAR_MAPPED)
		gtk_widget_show(w);
	/* Build the Tool Bar later */
/*	archiver->tool_bar = NULL; */


	/* Location Bar GtkHandleBox */
	archiver->location_bar_handle = w = gtk_handle_box_new();
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_attached",
		GTK_SIGNAL_FUNC(edv_archiver_handle_child_attached_cb), archiver
	);
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_detached",
		GTK_SIGNAL_FUNC(edv_archiver_handle_child_detached_cb), archiver
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	if(archiver->flags & EDV_ARCHIVER_LOCATION_BAR_MAPPED)
		gtk_widget_show(w);
	parent2 = w;

	/* Location Bar GtkHBox */
	archiver->location_bar_hbox = w = gtk_hbox_new(FALSE, 0);
	gtk_widget_set_name(w, EDV_WIDGET_NAME_LOCATION_BAR);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);
	parent2 = w;

	/* Hbox for location icon and combo (no spacing between icon
	 * and combo needed since combo already has spacing)
	 */
	w = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Vbox to orient location icon fixed parent */
	w = gtk_vbox_new(TRUE, 0);
	gtk_widget_set_usize(w, 20, 20);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Location Icon GtkFixed */
	archiver->location_icon_fixed = w = gtk_fixed_new();
	gtk_widget_add_events(
		w,
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
		GDK_POINTER_MOTION_MASK |
		GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "realize",
		GTK_SIGNAL_FUNC(edv_archiver_location_bar_icon_realize_cb), archiver
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "enter_notify_event",
		GTK_SIGNAL_FUNC(edv_archiver_location_bar_icon_crossing_cb), archiver
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "leave_notify_event",
		GTK_SIGNAL_FUNC(edv_archiver_location_bar_icon_crossing_cb), archiver
	);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
"Arrastre esto crear una conexin a este archivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Traner ceci pour crer un lien  cette archive"
#elif defined(PROG_LANGUAGE_GERMAN)
"Schleppen sie dies, ein kettenglied zu diesem archiv zu schaffen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Trascinare questo per creare una maglia a quest'archivio"
#elif defined(PROG_LANGUAGE_DUTCH)
"Sleep dit om een schakel aan deze archief te creren"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Arraste isto criar um elo a este arquivo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Slep dette skape et ledd til dette arkivet"
#elif defined(PROG_LANGUAGE_POLISH)
"Przecignij to by utworzy odnonik do tego archiwum"
#else
"Drag this to create a link to this archive"
#endif
	);
	gtk_widget_show(w);

	/* Reset the location icon pixmap */
/*	archiver->location_icon_pm = NULL; */

	/* Location GtkCombo */
	glist = edv_open_text_file_glist(
		EDV_GET_S(EDV_CFG_PARM_FILE_ARCHIVER_LOCATION_HISTORY),
		-1,				/* No lines limit */
		TRUE				/* Strip CRs */
	);
	w = GUIComboCreate(
#if defined(PROG_LANGUAGE_SPANISH)
"Archivo:"
#elif defined(PROG_LANGUAGE_FRENCH)
"Archive:"
#elif defined(PROG_LANGUAGE_GERMAN)
"Archiv:"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Archivio:"
#elif defined(PROG_LANGUAGE_DUTCH)
"Archief:"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Arquivo:"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Arkiv:"
#elif defined(PROG_LANGUAGE_POLISH)
"Archiwum:"
#else
"Archive:"
#endif
		, "/",
		glist,
		20,		/* Maximum items */
		&combo_rtn,
		archiver,
		edv_archiver_combo_activate_cb,
		NULL
	);
	if(glist != NULL)
	{
		g_list_foreach(glist, (GFunc)g_free, NULL);
		g_list_free(glist);
		glist = NULL;
	}
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);

	archiver->location_combo = w = (GtkWidget *)combo_rtn;
	combo = GTK_COMBO(w);
	gtk_combo_set_case_sensitive(combo, TRUE);
	if(TRUE)
	{
		/* Need to set different dnd_src_types for the drag source */
		const GtkTargetEntry dnd_src_types[] = {
{GUI_TARGET_NAME_TEXT_PLAIN,	0,	EDV_DND_INFO_TEXT_PLAIN},
{GUI_TARGET_NAME_TEXT_URI_LIST,	0,	EDV_DND_INFO_TEXT_URI_LIST},
{GUI_TARGET_NAME_STRING,	0,	EDV_DND_INFO_STRING}
		};
		GUIDNDSetSrc(
			archiver->location_icon_fixed,
			dnd_src_types,
			sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
			GDK_ACTION_COPY | GDK_ACTION_MOVE |
			GDK_ACTION_LINK,		/* Actions */
			GDK_BUTTON1_MASK,		/* Buttons */
			NULL,
			edv_location_bar_icon_drag_data_get_cb,
			NULL,
			NULL,
			combo
		);
	}
	w = combo->entry;
	edv_entry_set_dnd(core, w);
	edv_entry_set_complete_path(core, w);
	GUIEditableEndowPopupMenu(w, 0);

	/* Password GtkVBox */
	w = gtk_vbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, border_major);
	gtk_widget_show(w);
	parent3 = w;
	/* Password prompt */
	w = GUIPromptBar(
		(guint8 **)icon_security_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Contrasea:"
#elif defined(PROG_LANGUAGE_FRENCH)
"Mot De Passe:"
#elif defined(PROG_LANGUAGE_GERMAN)
"Kennwort:"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Parola D'Ordine:"
#elif defined(PROG_LANGUAGE_DUTCH)
"Wachtwoord:"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Senha:"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Passord:"
#elif defined(PROG_LANGUAGE_POLISH)
"Haso:"
#else
"Password:"
#endif
		, NULL, &entry_rtn
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	archiver->location_password_entry = w = (GtkWidget *)entry_rtn;
	entry = GTK_ENTRY(w);
	gtk_widget_set_usize(w, 100, -1);
	gtk_entry_set_visibility(entry, FALSE);
	gtk_entry_set_editable(entry, TRUE);
	gtk_signal_connect(
		GTK_OBJECT(w), "activate",
		GTK_SIGNAL_FUNC(edv_archiver_combo_activate_cb), archiver
	);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
"Entre la contrasea (si necesit) para el decrypting/codificado\
 el archivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Entrer le mot de passe (si ncessaire) pour le decrypting\
 encrypting l'archive"
#elif defined(PROG_LANGUAGE_GERMAN)
"Tragen das kennwort (wenn hat gebraucht) fr decrypting\
 verschlsseln des archivs zu ein dem sie"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Entrare la parola d'ordine (se bisogno avuto di) per decrypting\
 codificare l'archivio"
#elif defined(PROG_LANGUAGE_DUTCH)
"Ga het wachtwoord binnen (indien nodig) voor decrypting coderen\
 het archief had"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Entre a senha (se necessrio) para decrypting encrypting o arquivo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"G inn i passordet (om trengt) for decrypting encrypting arkivet"
#elif defined(PROG_LANGUAGE_POLISH)
"Podaj haso (jeli potrzebne) by odszyfrowa/zaszyfrowa archiwum"
#else
"Enter the password (if needed) for decrypting/encrypting the archive"
#endif
	);

	/* GtkHandleBox for the Find Bar */
	archiver->find_bar_handle = w = gtk_handle_box_new();
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_attached",
		GTK_SIGNAL_FUNC(edv_archiver_handle_child_attached_cb), archiver
	);
	gtk_signal_connect_after(
		GTK_OBJECT(w), "child_detached",
		GTK_SIGNAL_FUNC(edv_archiver_handle_child_detached_cb), archiver
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	if(archiver->flags & EDV_ARCHIVER_FIND_BAR_MAPPED)
		gtk_widget_show(w);
	parent2 = w;

	/* Create the Find Bar */
	archiver->find_bar = w = edv_find_bar_new(
		core,
		EDV_LOCATION_TYPE_ARCHIVE,
		edv_archiver_find_bar_location_cb,
		archiver,
		edv_archiver_find_bar_start_cb,
		archiver,
		edv_archiver_find_bar_end_cb,
		archiver,
		edv_archiver_find_bar_match_cb,
		archiver,
		edv_archiver_status_message_cb,
		archiver,
		edv_archiver_status_progress_cb,
		archiver
	);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);


	/* GtkScrolledWindow for the GtkCList */
	w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
		GTK_SCROLLED_WINDOW(w),
		GTK_POLICY_AUTOMATIC,
		GTK_POLICY_AUTOMATIC
	);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Contents GtkCList */
	archiver->contents_clist = w = gtk_clist_new(
		EDV_ARCHIVER_NCOLUMN_TYPES
	);
	gtk_widget_set_name(w, EDV_WIDGET_NAME_ARCHIVER_CONTENTS_LIST);
	clist = GTK_CLIST(w);
	gtk_widget_add_events(
		w,
		GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
		GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
		GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(edv_clist_key_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(edv_clist_key_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(edv_clist_button_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(edv_clist_button_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "motion_notify_event",
		GTK_SIGNAL_FUNC(edv_clist_motion_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(edv_archiver_key_event_cb), archiver
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(edv_archiver_key_event_cb), archiver
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(edv_archiver_button_press_event_cb), archiver
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "resize_column",
		GTK_SIGNAL_FUNC(edv_archiver_resize_column_cb), archiver
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "click_column",
		GTK_SIGNAL_FUNC(edv_archiver_click_column_cb), archiver
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "select_row",
		GTK_SIGNAL_FUNC(edv_archiver_select_row_cb), archiver
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "unselect_row",
		GTK_SIGNAL_FUNC(edv_archiver_unselect_row_cb), archiver
	);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_EXTENDED);
	if(clist->columns > 0)
	{
		CfgIntList *column_widths_intlist = EDV_GET_INT_LIST(
			EDV_CFG_PARM_ARCHIVER_CONTENTS_COLUMN_WIDTH
		);
		if(column_widths_intlist != NULL)
		{
			EDVArchiverColumnType column_type = 0;
			GList *glist = column_widths_intlist->list;
			while(glist != NULL)
			{
				edv_archiver_list_set_column_width_by_type(
					archiver,
					column_type,
					MAX((gint)glist->data, 10)      /* Width */
				);
				column_type++;
				glist = g_list_next(glist);
			}
		}
	}
	gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
	gtk_clist_set_row_height(clist, EDV_LIST_ROW_SPACING);
	GUIDNDSetSrc(
		w,
		dnd_src_types,
		sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_COPY,		/* Actions */
		GDK_BUTTON1_MASK,		/* Buttons */
		NULL,
		edv_archiver_list_drag_data_get_cb,
		edv_archiver_list_drag_data_delete_cb,
		NULL,
		archiver
	);
	GUIDNDSetTar(
		w,
		dnd_tar_types,
		sizeof(dnd_tar_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_COPY,		/* Actions */
		GDK_ACTION_COPY,		/* Default action if same */
		GDK_ACTION_COPY,		/* Default action */
		edv_archiver_list_drag_data_received_cb,
		archiver,
		TRUE				/* Highlight */
	);
	gtk_widget_show(w);


	/* Create the Tool Bar */
	edv_archiver_tool_bar_regenerate(archiver);

	/* Create the Contents List popup menu */
	edv_archiver_list_menu_regenerate(archiver);


	/* Status Bar */
	archiver->status_bar = w = edv_status_bar_new(core);
	gtk_box_pack_end(GTK_BOX(parent), w, FALSE, FALSE, 0);
	if(archiver->flags & EDV_ARCHIVER_STATUS_BAR_MAPPED)
		gtk_widget_show(w);


	/* The Archive Info Dialog is created later when needed */
	archiver->comments_dialog = NULL;

	/* Set the initial RC styles */
	if(standard_rcstyle != NULL)
		gtk_widget_modify_style_recursive(
			archiver->toplevel,
			standard_rcstyle
		);
	if(lists_rcstyle != NULL)
		gtk_widget_modify_style_recursive(
			archiver->contents_clist,
			lists_rcstyle
		);
	if(standard_rcstyle != NULL)
		gtk_widget_modify_style_recursive(
			archiver->contents_clist_menu,
			standard_rcstyle
		);

	/* Load the initial icons */
	edv_archiver_icons_regenerate(archiver);

	/* Set the initial Accel Keys */
	edv_archiver_accelkeys_regenerate(archiver);

	archiver->freeze_count--;

	return(archiver);
}

/*
 *	Syncronizes all data in the archiver's widgets to internal data
 *	on the archiver.
 */
void edv_archiver_sync_data(EDVArchiver *archiver)
{
	if(archiver == NULL)
		return;

/* This function may be removed in the future, since there really is
 * no floating data from widgets to be synced to disk on the archiver
 */
}

/*
 *	Updates all widgets on the given archiver to reflect the current
 *	data.
 */
void edv_archiver_update_display(EDVArchiver *archiver)
{
	gboolean	write_protect,
			write_protect_changed = FALSE,
			sensitive,
			state;
	gint		i,
			selected_obj_row = -1;
	guint8		**icon_data,
			**recbin_icon_data;
	const gchar *location;
	GtkWidget	*w,
			*tb;
	GtkCList *clist;
	CfgList *cfg_list;
	EDVDeleteMethod delete_method;
	EDVCore *core;

	if(archiver == NULL)
		return;

	archiver->freeze_count++;

	tb = archiver->tool_bar;
	core = archiver->core;
	cfg_list = core->cfg_list;

#define MENU_ITEM_SET_TOGGLE(_w_,_state_)		\
 GUIMenuItemSetCheck(					\
  (_w_),						\
  (_state_),						\
  FALSE							\
 )
#define TOOL_BAR_ITEM_SET_SENSITIVE(_op_,_sensitive_)	\
 ToolBarSetItemSensitive(				\
  tb,							\
  (_op_),						\
  (_sensitive_)						\
 )
#define TOOL_BAR_ITEM_SET_TOGGLE(_op_,_state_)		\
 ToolBarSetItemToggle(					\
  tb,							\
  (_op_),						\
  (_state_)						\
 )
#define TOOL_BAR_ITEM_UPDATE(_op_,_text_,_icon_data_,_tip_) \
 ToolBarUpdateItem(					\
  tb,							\
  (_op_),						\
  (_text_),						\
  (_icon_data_),					\
  (_tip_)						\
 )

	/* Get the global write protect state */
	write_protect = EDV_GET_B(EDV_CFG_PARM_WRITE_PROTECT);
	if(archiver->last_write_protect_state < 0)
	{
		write_protect_changed = TRUE;
	}
	else
	{
		if((write_protect && !archiver->last_write_protect_state) ||
		   (!write_protect && archiver->last_write_protect_state)
		)
			write_protect_changed = TRUE;
	}
	archiver->last_write_protect_state = write_protect ? 1 : 0;

	/* Delete method */
	delete_method = (EDVDeleteMethod)EDV_GET_I(
		EDV_CFG_PARM_DELETE_METHOD
	);

	/* Get last selected row on the contents clist */
	clist = GTK_CLIST(archiver->contents_clist);
	selected_obj_row = edv_clist_get_selected_last(clist, NULL);

	/* Get the Recycle Bin icon data */
	icon_data = edv_get_recycle_bin_icon_data(
		EDV_ICON_SIZE_20,
		core->last_nrecycle_bin_items,
		NULL
	);
	archiver->last_nrecycle_bin_items = core->last_nrecycle_bin_items;
	recbin_icon_data = icon_data;

	/* Get the current location */
	location = edv_archiver_get_location(archiver);


	/* Update the find bar */
	edv_find_bar_update_display(archiver->find_bar);



	/* Begin updating menu items and tool bar items */

	/* Edit */
	sensitive = !STRISEMPTY(location);
	/* Add */
	GTK_WIDGET_SET_SENSITIVE(archiver->edit_add_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_ARCHIVER_OP_ADD, sensitive);
	/* Extract all */
	GTK_WIDGET_SET_SENSITIVE(archiver->edit_extract_all_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_ARCHIVER_OP_EXTRACT_ALL, sensitive);
	/* Extract */
	sensitive = (selected_obj_row < 0) ? FALSE : TRUE;
	GTK_WIDGET_SET_SENSITIVE(archiver->edit_extract_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_ARCHIVER_OP_EXTRACT, sensitive);
	/* Delete */
	GTK_WIDGET_SET_SENSITIVE(archiver->edit_delete_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_ARCHIVER_OP_DELETE, sensitive);
	/* Properties */
	GTK_WIDGET_SET_SENSITIVE(archiver->properties_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_ARCHIVER_OP_PROPERTIES, sensitive);

	sensitive = !STRISEMPTY(location);
	/* Comments */
	GTK_WIDGET_SET_SENSITIVE(archiver->comments_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_ARCHIVER_OP_COMMENTS, sensitive);
	/* Check */
	GTK_WIDGET_SET_SENSITIVE(archiver->check_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_ARCHIVER_OP_CHECK, sensitive);
	/* Comments */
	GTK_WIDGET_SET_SENSITIVE(archiver->fix_mi, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_ARCHIVER_OP_FIX, sensitive);

	/* View */
	/* Stop */
	sensitive = (archiver->subprocess != NULL) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(
		archiver->stop_mi,
		sensitive
	);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_ARCHIVER_OP_STOP, sensitive);
	/* Refresh and Refresh All */
/*	sensitive = (archiver->subprocess != NULL) ? FALSE : TRUE; */
	sensitive = TRUE;
	GTK_WIDGET_SET_SENSITIVE(
		archiver->refresh_mi,
		sensitive
	);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_ARCHIVER_OP_REFRESH, sensitive);
	GTK_WIDGET_SET_SENSITIVE(
		archiver->refresh_all_mi,
		sensitive
	);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_ARCHIVER_OP_REFRESH_ALL, sensitive);
	/* Statistics */
	sensitive = !STRISEMPTY(location);
	GTK_WIDGET_SET_SENSITIVE(
		archiver->statistics_mi,
		sensitive
	);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_ARCHIVER_OP_STATISTICS, sensitive);
	/* Show Tool Bar */
	state = EDV_GET_B(EDV_CFG_PARM_ARCHIVER_SHOW_TOOL_BAR);
	MENU_ITEM_SET_TOGGLE(archiver->show_tool_bar_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_ARCHIVER_OP_SHOW_TOOL_BAR, state);
	/* Show Location Bar */
	state = EDV_GET_B(EDV_CFG_PARM_ARCHIVER_SHOW_LOCATION_BAR);
	MENU_ITEM_SET_TOGGLE(archiver->show_location_bar_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_ARCHIVER_OP_SHOW_LOCATION_BAR, state);
	/* Show Find Bar */
	state = EDV_GET_B(EDV_CFG_PARM_ARCHIVER_SHOW_FIND_BAR);
	MENU_ITEM_SET_TOGGLE(archiver->show_find_bar_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_ARCHIVER_OP_SHOW_FIND_BAR, state);
	/* Show Status Bar */
	state = EDV_GET_B(EDV_CFG_PARM_ARCHIVER_SHOW_STATUS_BAR);
	MENU_ITEM_SET_TOGGLE(archiver->show_status_bar_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_ARCHIVER_OP_SHOW_STATUS_BAR, state);

	/* Auto Resize Columns */
	state = EDV_GET_B(EDV_CFG_PARM_ARCHIVER_CONTENTS_LIST_AUTO_RESIZE_COLUMNS);
	MENU_ITEM_SET_TOGGLE(archiver->contents_list_auto_resize_columns_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_ARCHIVER_OP_CONTENTS_LIST_AUTO_RESIZE_COLUMNS, state);


	/* Settings */
	/* Master Write Protect */
	if(write_protect_changed)
	{
		w = archiver->write_protect_mi;
		GUIMenuItemSetPixmap(
			w,
			(guint8 **)(write_protect ?
				icon_write_protect_on_20x20_xpm : icon_write_protect_off_20x20_xpm)
		);
		GUIMenuItemSetLabel(
			w,
			write_protect ?
				"Master Write Protect: On" :
				"Master Write Protect: Off"
		);
		TOOL_BAR_ITEM_UPDATE(
			EDV_ARCHIVER_OP_WRITE_PROTECT,
			NULL,
			(guint8 **)(write_protect ?
				icon_write_protect_on_20x20_xpm : icon_write_protect_off_20x20_xpm),
			NULL
		);
	}

	/* Delete Method: Recycle */
	state = (delete_method == EDV_DELETE_METHOD_RECYCLE) ? TRUE : FALSE;
	MENU_ITEM_SET_TOGGLE(archiver->delete_method_recycle_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_ARCHIVER_OP_DELETE_METHOD_RECYCLE, state);

	/* Delete Method: Purge */
	state = (delete_method == EDV_DELETE_METHOD_PURGE) ? TRUE : FALSE;
	MENU_ITEM_SET_TOGGLE(archiver->delete_method_purge_micheck, state);
	TOOL_BAR_ITEM_SET_TOGGLE(EDV_ARCHIVER_OP_DELETE_METHOD_PURGE, state);


	/* Windows
	 *
	 * New Browser
	 */
	w = archiver->window_new_browser_mi;
	i = EDV_GET_I(EDV_CFG_PARM_MAX_BROWSERS);
	sensitive = ((core->total_browsers < i) || (i < 0)) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(w, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_ARCHIVER_OP_NEW_BROWSER, sensitive);

	/* New Image Browser */
	w = archiver->window_new_imbr_mi;
	i = EDV_GET_I(EDV_CFG_PARM_MAX_IMBRS);
	sensitive = ((core->total_imbrs < i) || (i < 0)) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(w, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_ARCHIVER_OP_NEW_IMBR, sensitive);

	/* New Archiver */
	w = archiver->window_new_archiver_mi;
	i = EDV_GET_I(EDV_CFG_PARM_MAX_ARCHIVERS);
	sensitive = ((core->total_archivers < i) || (i < 0)) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(w, sensitive);
	TOOL_BAR_ITEM_SET_SENSITIVE(EDV_ARCHIVER_OP_NEW_ARCHIVER, sensitive);

	/* Recycle Bin */
	if(recbin_icon_data != NULL)
	{
		GUIMenuItemSetPixmap(
			archiver->window_recycle_bin_mi, recbin_icon_data
		);
		TOOL_BAR_ITEM_UPDATE(
			EDV_ARCHIVER_OP_RECYCLE_BIN,
			NULL,
			recbin_icon_data,
			NULL
		);
	}


	/* Contents List popup menu */
	sensitive = (selected_obj_row > -1) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(
		archiver->contents_clist_add_mi,
		TRUE
	);
	GTK_WIDGET_SET_SENSITIVE(
		archiver->contents_clist_extract_mi,
		sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
		archiver->contents_clist_extract_all_mi,
		TRUE
	);
	GTK_WIDGET_SET_SENSITIVE(
		archiver->contents_clist_delete_mi,
		sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
		archiver->contents_clist_properties_mi,
		sensitive
	);


	/* Status Bar */
	edv_status_bar_update_display(archiver->status_bar);

#undef MENU_ITEM_SET_TOGGLE
#undef TOOL_BAR_ITEM_SET_SENSITIVE
#undef TOOL_BAR_ITEM_SET_TOGGLE
#undef TOOL_BAR_ITEM_UPDATE

	archiver->freeze_count--;
}

/*
 *	Sets the Archiver as busy or ready.
 */
void edv_archiver_set_busy(EDVArchiver *archiver, const gboolean busy)
{
	GdkCursor *cur;
	GtkWidget *w;
	EDVCore *core;

	if(archiver == NULL)
		return;

	w = archiver->toplevel;
	core = archiver->core;

	if(busy)
	{
		archiver->busy_count++;
		if(archiver->busy_count > 1)
			return;

		cur = edv_get_cursor(core, EDV_CURSOR_CODE_BUSY);
	}
	else
	{
		archiver->busy_count--;
		if(archiver->busy_count < 0)
			archiver->busy_count = 0;
		if(archiver->busy_count > 0)
			return;
		cur = (archiver->passive_busy_count > 0) ?
			edv_get_cursor(core, EDV_CURSOR_CODE_PASSIVE_BUSY) :
			NULL;
	}

	if(w->window != NULL)
	{
		gdk_window_set_cursor(w->window, cur);
		gdk_flush();
	}
}

/*
 *	Sets the Archiver as passive busy or ready.
 */
void edv_archiver_set_passive_busy(EDVArchiver *archiver, const gboolean busy)
{
	GdkCursor *cur;
	GtkWidget *w;
	EDVCore *core;

	if(archiver == NULL)
		return;

	w = archiver->toplevel;
	core = archiver->core;

	if(busy)
	{
		archiver->passive_busy_count++;
		if((archiver->passive_busy_count > 1) || (archiver->busy_count > 0))
			return;
		cur = edv_get_cursor(core, EDV_CURSOR_CODE_PASSIVE_BUSY);
	}
	else
	{
		archiver->passive_busy_count--;
		if(archiver->passive_busy_count < 0)
			archiver->passive_busy_count = 0;
		if((archiver->passive_busy_count > 0) || (archiver->busy_count > 0))
			return;
		cur = NULL;
	}

	if(w->window != NULL)
	{
		gdk_window_set_cursor(w->window, cur);
		gdk_flush();
	}
}

/*
 *	Checks if the Archiver is mapped.
 */
gboolean edv_archiver_is_mapped(EDVArchiver *archiver)
{
	if(archiver == NULL)
		return(FALSE);

	return(GTK_WIDGET_MAPPED(archiver->toplevel));
}

/*
 *	Maps the Archiver.
 */
void edv_archiver_map(EDVArchiver *archiver)
{
	if(archiver == NULL)
		return;

	gtk_widget_show_raise(archiver->toplevel);
	archiver->flags |= EDV_ARCHIVER_MAPPED;
}

/*
 *	Unmaps the Archiver.
 */
void edv_archiver_unmap(EDVArchiver *archiver)
{
	if(archiver == NULL)
		return;

	gtk_widget_hide(archiver->toplevel);
	archiver->flags &= ~EDV_ARCHIVER_MAPPED;
}

/*
 *	Deletes the Archiver.
 */
void edv_archiver_delete(EDVArchiver *archiver)
{
	if(archiver == NULL)
		return;

	/* Stop any subprocesses */
	edv_archiver_subprocess_delete(archiver->subprocess);
	archiver->subprocess = NULL;

	/* Clear the Contents List */
	edv_archiver_list_clear(archiver);

	/* Reset the last selected item references */
	archiver->contents_clist_selected_row = -1;

	/* Delete the Archive Comments Dialog */
	edv_archive_comments_dlg_delete(archiver->comments_dialog);
	archiver->comments_dialog = NULL;

	/* Delete the Archive Statistics Dialog */
	edv_archive_statistics_dlg_delete(archiver->statistics_dialog);
	archiver->statistics_dialog = NULL;

	edv_archiver_unmap(archiver);

	archiver->freeze_count++;

	/* Delete any queued operations */
	if(archiver->queued_ops_list != NULL)
	{
		GList *glist;
		EDVArchiverOpQueue *q;
		for(glist = archiver->queued_ops_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			q = EDV_ARCHIVER_OP_QUEUE(glist->data);
			if(q == NULL)
				continue;

			q->idle_id = GTK_IDLE_REMOVE(q->idle_id);
			g_free(q);
			glist->data = NULL;
		}
		g_list_free(archiver->queued_ops_list);
		archiver->queued_ops_list = NULL;
	}

	/* Destroy all the GtkWidgets */
	GTK_WIDGET_DESTROY(archiver->contents_clist_menu);
	GTK_WIDGET_DESTROY(archiver->toplevel);

	/* Delete the EDVArchiverOps List */
	edv_archiver_op_ids_list_delete(archiver->op_ids_list);

	/* Unref all the icons */
	(void)edv_pixmap_unref(archiver->encrypted_icon);

	g_free(archiver->contents_list_filter);

	archiver->freeze_count--;

	g_free(archiver);
}
