#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "cfg.h"

#include "guiutils.h"
#include "pie_chart.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "edv_archive_obj_stat.h"
#include "edv_obj_info_match.h"
#include "edv_utils_gtk.h"
#include "archive_statistics_dlg.h"
#include "edv_emit.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_ok_20x20.xpm"


/* Callbacks */
static gint edv_archive_statistics_dlg_delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void edv_archive_statistics_dlg_ok_cb(GtkWidget *widget, gpointer data);

/* EDVArchiveStatisticsDlg */
EDVArchiveStatisticsDlg *edv_archive_statistics_dlg_new(
	EDVCore *core,
	const gchar *arch_path,
	const gchar *password,
	GList *objs_list,
	GtkWidget *toplevel
);
void edv_archive_statistics_dlg_update_display(EDVArchiveStatisticsDlg *d);
void edv_archive_statistics_dlg_set_busy(
	EDVArchiveStatisticsDlg *d,
	const gboolean busy
);
gboolean edv_archive_statistics_dlg_is_mapped(EDVArchiveStatisticsDlg *d);
void edv_archive_statistics_dlg_map(EDVArchiveStatisticsDlg *d);
void edv_archive_statistics_dlg_unmap(EDVArchiveStatisticsDlg *d);
void edv_archive_statistics_dlg_delete(EDVArchiveStatisticsDlg *d);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Toplevel GtkWindow "delete_event" signal callback.
 */
static gint edv_archive_statistics_dlg_delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	edv_archive_statistics_dlg_ok_cb(widget, data);
	return(TRUE);
}


/*
 *	OK callback.
 */
static void edv_archive_statistics_dlg_ok_cb(GtkWidget *widget, gpointer data)
{
	EDVArchiveStatisticsDlg *d = EDV_ARCHIVE_STATISTICS_DLG(data);
	if(d == NULL)
		return;

	if(d->freeze_count > 0)
		return;

	d->freeze_count++;

	edv_archive_statistics_dlg_unmap(d);

	d->freeze_count--;
}


/*
 *	Creates a new EDVArchiveStatisticsDlg.
 *
 *	The arch_path specifies the path to the archive.
 *
 *	The password specifies the password.
 *
 *	The objs_list specifies a GList of EDVArchiveObject * archive
 *	objects.
 *
 *	If toplevel is not NULL then it specifies the relative toplevel
 *	GtkWindow.
 */
EDVArchiveStatisticsDlg *edv_archive_statistics_dlg_new(
	EDVCore *core,
	const gchar *arch_path,
	const gchar *password,
	GList *objs_list,
	GtkWidget *toplevel
)
{
	const gint	border_major = 5,
			border_minor = 2;
	gint		font_size,
			total_objects = 0,
			file_objects = 0,
			directory_objects = 0,
			link_objects = 0,
			other_objects = 0;
	gulong		files_size = 0l,
			directories_size = 0l,
			links_size = 0l,
			others_size = 0l,
			uncompressed_size = 0l;
	const gchar *font_encoding;
	gchar		*s,
			*s2,
			*font_name_h1_bold,
			*font_name_h2,
			*font_name_fixed_width;
	GdkFont *font;
	GdkWindow *window;
	GtkRcStyle *rcstyle;
	GtkStyle *style;
	GtkAccelGroup *accelgrp;
	GtkWidget	*w,
			*parent, *parent2, *parent3, *parent4,
			*parent5,
			*ref_toplevel = toplevel;
	CfgList *cfg_list;
	EDVPixmap *icon;
	EDVVFSObject *obj;
	EDVArchiveStatisticsDlg *d;

	if(core == NULL)
		return(NULL);

	cfg_list = core->cfg_list;

	d = EDV_ARCHIVE_STATISTICS_DLG(g_malloc0(
		sizeof(EDVArchiveStatisticsDlg)
	));
	if(d == NULL)
		return(NULL);

	d->toplevel = toplevel = gtk_window_new(GTK_WINDOW_DIALOG);
	d->accelgrp = accelgrp = gtk_accel_group_new();
#if 0
	d->freeze_count = 0;
	d->busy_count = 0;
#endif
	d->core = core;

	d->arch_path = STRDUP(arch_path);

	d->busy_cur = gdk_cursor_new(GDK_WATCH);

	d->freeze_count++;

	/* Get the statistics of the archive's VFS object */
	obj = edv_vfs_object_lstat(arch_path);
	if(obj != NULL)
	{
		/* Get the icon for the archive */
		(void)edv_match_object_icon(
			core->devices_list,
			core->mime_types_list,
			obj->type,
			obj->path,
			EDV_VFS_OBJECT_LINK_TARGET_EXISTS(obj),
			obj->permissions,
			EDV_ICON_SIZE_32,
			&icon,
			NULL,
			NULL,
			NULL
		);
	}
	else
	{
		icon = NULL;
	}

	/* Gather the statistics from the archive objects */
	if(objs_list != NULL)
	{
		gulong this_obj_size;
		GList *glist;
		EDVArchiveObject *arch_obj;

		for(glist = objs_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			arch_obj = EDV_ARCHIVE_OBJECT(glist->data);
			if(arch_obj == NULL)
				continue;

			total_objects++;
			switch(arch_obj->type)
			{
			    case EDV_OBJECT_TYPE_UNKNOWN:
				break;
			    case EDV_OBJECT_TYPE_FILE:
				file_objects++;
				this_obj_size = arch_obj->size + (
					(arch_obj->path != NULL) ?
					strlen(arch_obj->path) : STRLEN(arch_obj->name)
				);
				files_size += this_obj_size;
				uncompressed_size += this_obj_size;
				break;
			    case EDV_OBJECT_TYPE_DIRECTORY:
				directory_objects++;
				this_obj_size = arch_obj->size + (
					(arch_obj->path != NULL) ?
					strlen(arch_obj->path) : STRLEN(arch_obj->name)
				);
				directories_size += this_obj_size;
				uncompressed_size += this_obj_size;
				break;
			    case EDV_OBJECT_TYPE_LINK:
				link_objects++;
				this_obj_size = arch_obj->size + (
					(arch_obj->path != NULL) ?
					strlen(arch_obj->path) : STRLEN(arch_obj->name)
				) + STRLEN(arch_obj->link_target);
				links_size += this_obj_size;
				uncompressed_size += this_obj_size;
				break;
			    default:
				other_objects++;
				this_obj_size = arch_obj->size + (
					(arch_obj->path != NULL) ?
					strlen(arch_obj->path) : STRLEN(arch_obj->name)
				);
				others_size += this_obj_size;
				uncompressed_size += this_obj_size;
				break;
			}
		}
	}

	/* Toplevel GtkWindow */
	w = toplevel;
	gtk_window_set_wmclass(
		GTK_WINDOW(w), "dialog", PROG_NAME
	);
	gtk_window_set_policy(GTK_WINDOW(w), TRUE, TRUE, TRUE);
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
		gdk_window_set_decorations(
			window,
			GDK_DECOR_BORDER | GDK_DECOR_TITLE | GDK_DECOR_MENU |
			GDK_DECOR_MINIMIZE
		);
		gdk_window_set_functions(
			window,
			GDK_FUNC_MOVE | GDK_FUNC_MINIMIZE | GDK_FUNC_CLOSE
		);
/*		GUISetWMIcon(window, (guint8 **)icon_endeavour_archiver_48x48_xpm); */
	}
	style = gtk_widget_get_style(w);
	gtk_signal_connect(
		GTK_OBJECT(w), "delete_event",
		GTK_SIGNAL_FUNC(edv_archive_statistics_dlg_delete_event_cb), d
	);
	gtk_container_border_width(GTK_CONTAINER(w), 0);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	if(ref_toplevel != NULL)
	{
		if(GTK_IS_WINDOW(GTK_OBJECT(ref_toplevel)))
			gtk_window_set_transient_for(
				GTK_WINDOW(w),
				GTK_WINDOW(ref_toplevel)
			);
	}
	parent = w;

	/* Main GtkVBox */
	d->main_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(w), 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;


	/* Get the base font size */
	font = style->font;
	font_size = GDK_FONT_GET_FONT_NAME_SIZE(font);
	if(font_size < 3)
		font_size = 3;

	/* Format the font names */
#if defined(PROG_LANGUAGE_POLISH)
	font_encoding = "iso8859-2";
#else
	font_encoding = "iso8859-1";
#endif

	font_name_h1_bold = g_strdup_printf(
"-adobe-helvetica-bold-r-normal-*-%i-*-*-*-*-*-%s",
		font_size + 2,
		font_encoding
	);
	font_name_h2 = g_strdup_printf(
"-adobe-helvetica-medium-r-normal-*-%i-*-*-*-*-*-%s",
		font_size,
		font_encoding
	);
	font_name_fixed_width = g_strdup_printf(
"-adobe-courier-medium-r-normal-*-%i-*-*-*-*-*-%s",
		font_size,
		font_encoding
	);


	/* Information & Statistics GtkVBox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(d->main_vbox), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent = w;


	/* GtkHBox to make two columns */
	w = gtk_hbox_new(FALSE, 20);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;


	/* Left column GtkVBox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;


	/* Icon & Name GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Icon */
	d->icon_pm = w = edv_pixmap_new_gtk_pixmap(icon);
	if(w != NULL)
	{
		gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
	}
	/* Label */
	if(obj != NULL)
		s = edv_path_shorten(obj->name, 40);
	else
		s = g_strdup("");
	d->name_label = w = gtk_label_new(s);
	g_free(s);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	rcstyle = gtk_rc_style_new();
	rcstyle->font_name = STRDUP(font_name_h1_bold);
	gtk_widget_modify_style(w, rcstyle);
	GTK_RC_STYLE_UNREF(rcstyle);
	gtk_widget_show(w);


	/* Location GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Label */
	rcstyle = gtk_rc_style_new();
	rcstyle->font_name = STRDUP(font_name_h2);
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"La Ubicacin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Emplacement"
#elif defined(PROG_LANGUAGE_GERMAN)
"Ort"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Posizione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Plaats"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Localidade"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Plassering"
#elif defined(PROG_LANGUAGE_POLISH)
"cieka"
#else
"Location"
#endif
		":"
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_modify_style(w, rcstyle);
	gtk_widget_show(w);
	GTK_RC_STYLE_UNREF(rcstyle);
	/* Location Label */
	if(obj != NULL)
		s2 = g_dirname(obj->path);
	else
		s2 = STRDUP("");
	s = edv_path_shorten(s2, 45);
	d->location_label = w = gtk_label_new(s);
	g_free(s);
	g_free(s2);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	rcstyle = gtk_rc_style_new();
	rcstyle->font_name = STRDUP(font_name_h2);
	gtk_widget_modify_style(w, rcstyle);
	GTK_RC_STYLE_UNREF(rcstyle);
	gtk_widget_show(w);


	/* GtkHBox to separate columns for object and size break-down */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Table for object and size break-down labels */
	w = gtk_table_new(5, 3, FALSE);
	gtk_table_set_row_spacings(GTK_TABLE(w), border_minor);
	gtk_table_set_col_spacings(GTK_TABLE(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

#define NEW_LABEL(_r_,_n_,_i_,_b_)	{		\
 GtkAttachOptions	x_attach_opt = 0,		\
			y_attach_opt = 0;		\
 guint	x_pad = 0, y_pad = 0;				\
							\
 /* Alignment for name label */				\
 w = gtk_alignment_new(1.0f, 0.5f, 0.0f, 0.0f);		\
 gtk_widget_set_usize(w, 100, -1);			\
 gtk_table_attach(					\
  GTK_TABLE(parent4), w,				\
  0, 1, (_r_), (_r_) + 1,				\
  x_attach_opt, y_attach_opt, x_pad, y_pad		\
 );							\
 gtk_widget_show(w);					\
 parent5 = w;						\
 /* Name label */					\
 w = gtk_label_new(_n_);				\
 gtk_container_add(GTK_CONTAINER(parent5), w);		\
 gtk_widget_modify_style(w, rcstyle);			\
 gtk_widget_show(w);					\
							\
 /* Alignment for quantity label */			\
 w = gtk_alignment_new(0.0f, 0.5f, 0.0f, 0.0f);		\
 gtk_widget_set_usize(w, 50, -1);			\
 gtk_table_attach(					\
  GTK_TABLE(parent4), w,				\
  1, 2, (_r_), (_r_) + 1,				\
  x_attach_opt, y_attach_opt, x_pad, y_pad		\
 );							\
 gtk_widget_show(w);					\
 parent5 = w;						\
 /* Quantity label */					\
 s = g_strdup_printf("%i", (gint)(_i_));		\
 w = gtk_label_new(s);					\
 g_free(s);						\
 gtk_container_add(GTK_CONTAINER(parent5), w);		\
 gtk_widget_modify_style(w, rcstyle);			\
 gtk_widget_show(w);					\
							\
 /* Alignmet for size label */				\
 w = gtk_alignment_new(0.0f, 0.5f, 0.0f, 0.0f);		\
 gtk_widget_set_usize(w, 150, -1);			\
 gtk_table_attach(					\
  GTK_TABLE(parent4), w,				\
  2, 3, (_r_), (_r_) + 1,				\
  x_attach_opt, y_attach_opt, x_pad, y_pad		\
 );							\
 gtk_widget_show(w);					\
 parent5 = w;						\
 /* Size label */					\
 s = g_strdup_printf(					\
  "%s %s",						\
  edv_str_size_delim((gulong)(_b_)),			\
  ((gint)(_b_) == 1) ? "byte" : "bytes"			\
 );							\
 w = gtk_label_new(s);					\
 g_free(s);						\
 gtk_container_add(GTK_CONTAINER(parent5), w);		\
 gtk_widget_modify_style(w, rcstyle);			\
 gtk_widget_show(w);					\
}

	rcstyle = gtk_rc_style_new();
	rcstyle->font_name = STRDUP(font_name_h2);
#if defined(PROG_LANGUAGE_SPANISH)
	NEW_LABEL(0, "Los Archivos:", file_objects, files_size);
	NEW_LABEL(1, "Las Guas:", directory_objects, directories_size);
	NEW_LABEL(2, "Las Conexiones:", link_objects, links_size);
	NEW_LABEL(3, "Otros Objetos:", other_objects, others_size);
#elif defined(PROG_LANGUAGE_FRENCH)
	NEW_LABEL(0, "Fichiers:", file_objects, files_size);
	NEW_LABEL(1, "Annuaires:", directory_objects, directories_size);
	NEW_LABEL(2, "Liens:", link_objects, links_size);
	NEW_LABEL(3, "Autres Objets:", other_objects, others_size);
#elif defined(PROG_LANGUAGE_GERMAN)
	NEW_LABEL(0, "Akten:", file_objects, files_size);
	NEW_LABEL(1, "Verzeichnisse:", directory_objects, directories_size);
	NEW_LABEL(2, "Kettenglieder:", link_objects, links_size);
	NEW_LABEL(3, "Andere Objekte:", other_objects, others_size);
#elif defined(PROG_LANGUAGE_ITALIAN)
	NEW_LABEL(0, "I File:", file_objects, files_size);
	NEW_LABEL(1, "Gli Elenchi:", directory_objects, directories_size);
	NEW_LABEL(2, "Le Maglie:", link_objects, links_size);
	NEW_LABEL(3, "L'Altro Obbietta:", other_objects, others_size);
#elif defined(PROG_LANGUAGE_DUTCH)
	NEW_LABEL(0, "Dossiers:", file_objects, files_size);
	NEW_LABEL(1, "Gidzen:", directory_objects, directories_size);
	NEW_LABEL(2, "Schakels:", link_objects, links_size);
	NEW_LABEL(3, "Anderze Voorwerpen:", other_objects, others_size);
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	NEW_LABEL(0, "Os Arquivos:", file_objects, files_size);
	NEW_LABEL(1, "Os Guias:", directory_objects, directories_size);
	NEW_LABEL(2, "Os Elos:", link_objects, links_size);
	NEW_LABEL(3, "Outro Ope-se:", other_objects, others_size);
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	NEW_LABEL(0, "Arkiver:", file_objects, files_size);
	NEW_LABEL(1, "Kataloger:", directory_objects, directories_size);
	NEW_LABEL(2, "Ledd:", link_objects, links_size);
	NEW_LABEL(3, "Annen Object:", other_objects, others_size);
#elif defined(PROG_LANGUAGE_POLISH)
	NEW_LABEL(0, "Pliki:", file_objects, files_size);
	NEW_LABEL(1, "Katalogi:", directory_objects, directories_size);
	NEW_LABEL(2, "Odnoniki:", link_objects, links_size);
	NEW_LABEL(3, "Inne obiekty:", other_objects, others_size);
#else
	NEW_LABEL(0, "Files:", file_objects, files_size);
	NEW_LABEL(1, "Directories:", directory_objects, directories_size);
	NEW_LABEL(2, "Links:", link_objects, links_size);
	NEW_LABEL(3, "Misc Objects:", other_objects, others_size);
#endif
	GTK_RC_STYLE_UNREF(rcstyle);

	rcstyle = gtk_rc_style_new();
	rcstyle->font_name = STRDUP(font_name_h2);
#if defined(PROG_LANGUAGE_SPANISH)
	NEW_LABEL(4, "El Suma:", total_objects, uncompressed_size);
#elif defined(PROG_LANGUAGE_FRENCH)
	NEW_LABEL(4, "Total:", total_objects, uncompressed_size);
#elif defined(PROG_LANGUAGE_GERMAN)
	NEW_LABEL(4, "Summe:", total_objects, uncompressed_size);
#elif defined(PROG_LANGUAGE_ITALIAN)
	NEW_LABEL(4, "Il Totale:", total_objects, uncompressed_size);
#elif defined(PROG_LANGUAGE_DUTCH)
	NEW_LABEL(4, "Totaal:", total_objects, uncompressed_size);
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	NEW_LABEL(4, "O Total:", total_objects, uncompressed_size);
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	NEW_LABEL(4, "Total:", total_objects, uncompressed_size);
#elif defined(PROG_LANGUAGE_POLISH)
	NEW_LABEL(4, "Razem:", total_objects, uncompressed_size);
#else
	NEW_LABEL(4, "Total:", total_objects, uncompressed_size);
#endif
	GTK_RC_STYLE_UNREF(rcstyle);

#undef NEW_LABEL



	/* Right column GtkVBox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_end(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Size pie chart */
	if(obj != NULL)
	{
		gulong v;
		gchar *s2;
		GdkColor *c, uncompressed_size_color, current_size_color;

		/* Uncompressed Size */
		GtkAdjustment *adj = (GtkAdjustment *)gtk_adjustment_new(
			0.0f, 0.0f, (gfloat)MAX(uncompressed_size, obj->size),
			0.0f, 0.0f, 0.0f
		);
		c = &uncompressed_size_color;
		GDK_COLOR_SET_COEFF(c, 0.0f, 0.0f, 1.0f);
		v = MAX(uncompressed_size, obj->size);
		s = g_strdup_printf(
			"%s %s",
			edv_str_size_delim(v),
			(v == 1l) ? "byte" : "bytes"
		);
		s2 = g_strdup_printf(
			"%.0f%% Compressed",
			CLIP(100.0f - (((uncompressed_size > 0) ?
				(gfloat)obj->size / (gfloat)uncompressed_size : 1.0f
			) * 100.0f), 0.0f, 100.0f)
		);

		/* Pie Chart */
		d->pie_chart = w = PieChartNew(
			(EDV_GET_B(EDV_CFG_PARM_LISTS_DOUBLE_BUFFER) ?
				PIE_CHART_DOUBLE_BUFFER : 0) |
				PIE_CHART_SHOW_LABELS |
				PIE_CHART_SHOW_SHADOW |
				PIE_CHART_SHOW_OUTLINE,
			110, 70,
			adj, c,
#if defined(PROG_LANGUAGE_SPANISH)
			"El Tamao", s2,
			"Ensanchado:", s
#elif defined(PROG_LANGUAGE_FRENCH)
			"Taille", s2,
			"Augment:", s
#elif defined(PROG_LANGUAGE_GERMAN)
			"Gre", s2,
			"Ausgebreitet:", s
#elif defined(PROG_LANGUAGE_ITALIAN)
			"La Misura", s2,
			"Allargato:", s
#elif defined(PROG_LANGUAGE_DUTCH)
			"Maat", s2,
			"Uitgebreide:", s
#elif defined(PROG_LANGUAGE_PORTUGUESE)
			"O Tamanho", s2,
			"Expandido:", s
#elif defined(PROG_LANGUAGE_NORWEGIAN)
			"Strrelse", s2,
			"Utvidet:", s
#elif defined(PROG_LANGUAGE_POLISH)
			"Rozmiar", s2,
			"Niezkompresowany:", s
#else
			"Size", s2,		/* Header & Footer */
			"Uncompressed:", s
#endif
		);
		g_free(s);
		g_free(s2);
		adj = (GtkAdjustment *)GTK_OBJECT_UNREF(GTK_OBJECT(adj));
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_show(w);

		/* Current Size */
		adj = (GtkAdjustment *)gtk_adjustment_new(
			0.0f, 0.0f, (gfloat)obj->size,
			0.0f, 0.0f, 0.0f
		);
		c = &current_size_color;
		GDK_COLOR_SET_COEFF(c, 1.0f, 0.0f, 1.0f);
		s = g_strdup_printf(
			"%s %s",
			edv_str_size_delim(obj->size),
			(obj->size == 1l) ? "byte" : "bytes"
		);
		PieChartValueAdd(
			w,
			adj,
			c,
#if defined(PROG_LANGUAGE_SPANISH)
"Comprimido"
#elif defined(PROG_LANGUAGE_FRENCH)
"Serr"
#elif defined(PROG_LANGUAGE_GERMAN)
"Zusammengedrckt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Compresso"
#elif defined(PROG_LANGUAGE_DUTCH)
"Samengedrukte"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Comprimido"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Komprimert"
#elif defined(PROG_LANGUAGE_POLISH)
"Zkompresowany"
#else
"Compressed"
#endif
			":",
			s
		);
		g_free(s);
		adj = (GtkAdjustment *)GTK_OBJECT_UNREF(GTK_OBJECT(adj));
	}


	/* Delete the archive's statistics */
	edv_vfs_object_delete(obj);


	parent = d->main_vbox;

	/* Separator */
	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Buttons GtkHBox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* OK GtkButton */
	d->ok_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_ok_20x20_xpm, "OK", NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_FOCUS | GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_archive_statistics_dlg_ok_cb), d
	);
	gtk_accel_group_add(
		accelgrp,
		GDK_Escape, 0,
		GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
		accelgrp,
		GDK_o, GDK_CONTROL_MASK,
		GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_o);
	gtk_widget_show(w);


	edv_archive_statistics_dlg_update_display(d);


	(void)edv_pixmap_unref(icon);
	g_free(font_name_h1_bold);
	g_free(font_name_h2);
	g_free(font_name_fixed_width);


	d->freeze_count--;

	return(d);
}

/*
 *	Updates the EDVArchiveStatisticsDlg's widgets to reflect current
 *	values.
 */
void edv_archive_statistics_dlg_update_display(EDVArchiveStatisticsDlg *d)
{
	gchar *s;

	if(d == NULL)
		return;

	/* Title */
	s = g_strconcat(
		"Archive Statistics",
		": ",
		(d->arch_path != NULL) ?
			g_basename(d->arch_path) : "Untitled",
		NULL
	);
	gtk_window_set_title(GTK_WINDOW(d->toplevel), s);
	g_free(s);
}


/*
 *	Sets the EDVArchiveStatisticsDlg as busy or ready.
 */
void edv_archive_statistics_dlg_set_busy(
	EDVArchiveStatisticsDlg *d,
	const gboolean busy
)
{
	GdkCursor *cur;
	GtkWidget *w;

	if(d == NULL)
		return;

	w = d->toplevel;

	if(busy)
	{
		d->busy_count++;
		if(d->busy_count > 1)
			return;

		cur = d->busy_cur;
	}
	else
	{
		d->busy_count--;
		if(d->busy_count < 0)
			d->busy_count = 0;

		if(d->busy_count > 0)
			return;

		cur = NULL;
	}

	gdk_window_set_cursor(w->window, cur);

	gdk_flush();
}

/*
 *	Checks if the EDVArchiveStatisticsDlg is mapped.
 */
gboolean edv_archive_statistics_dlg_is_mapped(EDVArchiveStatisticsDlg *d)
{
	if(d == NULL)
		return(FALSE);

	return(GTK_WIDGET_MAPPED(d->toplevel));
}

/*
 *	Maps the EDVArchiveStatisticsDlg.
 */
void edv_archive_statistics_dlg_map(EDVArchiveStatisticsDlg *d)
{
	GtkWidget *w;

	if(d == NULL)
		return;

	gtk_widget_show_raise(d->toplevel);

	/* Grab focus and default on the OK button */
	w = d->ok_btn;
	if(w != NULL)
	{
		gtk_widget_grab_focus(w);
		gtk_widget_grab_default(w);
	}
}

/*
 *	Unmap the EDVArchiveStatisticsDlg.
 */
void edv_archive_statistics_dlg_unmap(EDVArchiveStatisticsDlg *d)
{
	if(d == NULL)
		return;

	gtk_widget_hide(d->toplevel);
}

/*
 *	Deletes the EDVArchiveStatisticsDlg.
 */
void edv_archive_statistics_dlg_delete(EDVArchiveStatisticsDlg *d)
{
	if(d == NULL)
		return;

	edv_archive_statistics_dlg_unmap(d);

	gtk_window_set_transient_for(
		GTK_WINDOW(d->toplevel),
		NULL
	);

	d->freeze_count++;

	gtk_widget_destroy(d->toplevel);
	gtk_accel_group_unref(d->accelgrp);

	GDK_CURSOR_DESTROY(d->busy_cur);

	g_free(d->arch_path);

	d->freeze_count--;

	g_free(d);
}
