#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "cfg.h"

#include "guiutils.h"
#include "pie_chart.h"
#include "tool_bar.h"
#include "cdialog.h"
#include "fb.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "edv_archive_obj_stat.h"
#include "edv_archive_comment.h"
#include "edv_obj_info_match.h"
#include "edv_utils_gtk.h"
#include "archive_comments_dlg.h"
#include "edv_emit.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_cut_20x20.xpm"
#include "images/icon_copy_20x20.xpm"
#include "images/icon_paste_20x20.xpm"
#include "images/icon_folder_opened_20x20.xpm"
#include "images/icon_ok_20x20.xpm"
#include "images/icon_select_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"
#include "images/icon_close_20x20.xpm"


/*
 *	Operation Codes:
 */
typedef enum {
	EDV_ARCHIVE_COMMENTS_DLG_OP_CUT,
	EDV_ARCHIVE_COMMENTS_DLG_OP_COPY,
	EDV_ARCHIVE_COMMENTS_DLG_OP_PASTE,
	EDV_ARCHIVE_COMMENTS_DLG_OP_INSERT_FILE
} EDVArchiveCommentsDlgOP;


/* Callbacks */
static gint edv_archive_comments_dlg_delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static gint edv_archive_comments_dlg_text_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void edv_archive_comments_dlg_any_changed_cb(GtkWidget *widget, gpointer data);
static void edv_archive_comments_dlg_tool_bar_activate_cb(
	ToolBarItem *tool_bar,
	const gint op,
	gpointer data
);
static void edv_archive_comments_dlg_ok_cb(GtkWidget *widget, gpointer data);
static void edv_archive_comments_dlg_apply_cb(GtkWidget *widget, gpointer data);
static void edv_archive_comments_dlg_cancel_cb(GtkWidget *widget, gpointer data);
static void edv_archive_comments_dlg_close_cb(GtkWidget *widget, gpointer data);

/* Operations */
static void edv_archive_comments_dlg_set_comments(EDVArchiveCommentsDlg *d);
static void edv_archive_comments_dlg_cut(EDVArchiveCommentsDlg *d);
static void edv_archive_comments_dlg_copy(EDVArchiveCommentsDlg *d);
static void edv_archive_comments_dlg_paste(EDVArchiveCommentsDlg *d);
static void edv_archive_comments_dlg_insert_file(
	EDVArchiveCommentsDlg *d,
	const gchar *path
);
static void edv_archive_comments_dlg_insert_file_query(EDVArchiveCommentsDlg *d);

/* EDVArchiveCommentsDlg */
EDVArchiveCommentsDlg *edv_archive_comments_dlg_new(
	EDVCore *core,
	const gchar *arch_path,
	const gchar *password,
	const gchar *comments,
	GtkWidget *toplevel
);
void edv_archive_comments_dlg_set_has_changes(
	EDVArchiveCommentsDlg *d,
	const gboolean has_changes
);
void edv_archive_comments_dlg_update_display(EDVArchiveCommentsDlg *d);
void edv_archive_comments_dlg_set_busy(
	EDVArchiveCommentsDlg *d,
	const gboolean busy
);
gboolean edv_archive_comments_dlg_is_mapped(EDVArchiveCommentsDlg *d);
gboolean edv_archive_comments_dlg_has_changes(EDVArchiveCommentsDlg *d);
void edv_archive_comments_dlg_map(EDVArchiveCommentsDlg *d);
void edv_archive_comments_dlg_unmap(EDVArchiveCommentsDlg *d);
void edv_archive_comments_dlg_delete(EDVArchiveCommentsDlg *d);


#define EDV_ARCHIVE_COMMENTS_DLG_DEF_WIDTH	\
					640
#define EDV_ARCHIVE_COMMENTS_DLG_DEF_HEIGHT	\
					480

#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Toplevel GtkWindow "delete_event" signal callback.
 */
static gint edv_archive_comments_dlg_delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	edv_archive_comments_dlg_cancel_cb(widget, data);
	return(TRUE);
}

/*
 *	GtkText event signal callback.
 */
static gint edv_archive_comments_dlg_text_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	GdkEventButton *button;
	GtkText *text;
	EDVArchiveCommentsDlg *d = EDV_ARCHIVE_COMMENTS_DLG(data);
	if((widget == NULL) || (event == NULL) || (d == NULL))
		return(status);

	if(d->freeze_count > 0)
		return(status);

	text = GTK_TEXT(widget);

	switch((gint)event->type)
	{
	  case GDK_BUTTON_PRESS:
		button = (GdkEventButton *)event;
		switch(button->button)
		{
		  case GDK_BUTTON4:
			/* Scroll up */
			if(text->vadj != NULL)
			{
				GtkAdjustment *adj = text->vadj;
				const gfloat inc = MAX(
					(0.25f * adj->page_size),
					adj->step_increment
				);
				gfloat v = adj->value - inc;
				if(v > (adj->upper - adj->page_size))
					v = adj->upper - adj->page_size;
				if(v < adj->lower)
					v = adj->lower;
				gtk_adjustment_set_value(adj, v);
			}
			/* Need to mark the GtkText button as 0 or else it will
			 * keep marking
			 */
			text->button = 0;
			gtk_grab_remove(widget);
			gtk_signal_emit_stop_by_name(
				GTK_OBJECT(widget), "button_press_event"
			);
			status = TRUE;
			break;
		  case GDK_BUTTON5:
			/* Scroll down */
			if(text->vadj != NULL)
			{
				GtkAdjustment *adj = text->vadj;
				const gfloat inc = MAX(
					(0.25f * adj->page_size),
					adj->step_increment
				);
				gfloat v = adj->value + inc;
				if(v > (adj->upper - adj->page_size))
					v = adj->upper - adj->page_size;
				if(v < adj->lower)
					v = adj->lower;
				gtk_adjustment_set_value(adj, v);
			}
			/* Need to mark the GtkText button as 0 or else it will
			 * keep marking
			 */
			text->button = 0;
			gtk_grab_remove(widget);
			gtk_signal_emit_stop_by_name(
				GTK_OBJECT(widget), "button_press_event"
			);
			status = TRUE;
			break;
		}
		break;

	  case GDK_BUTTON_RELEASE:
		button = (GdkEventButton *)event;
		switch(button->button)
		{
		  case GDK_BUTTON4:
			/* Need to mark the GtkText button as 0 or else it will
			 * keep marking
			 */
			text->button = 0;
			gtk_grab_remove(widget);
			gtk_signal_emit_stop_by_name(
				GTK_OBJECT(widget), "button_release_event"
			);
			status = TRUE;
			break;
		  case GDK_BUTTON5:
			/* Need to mark the GtkText button as 0 or else it will
			 * keep marking
			 */
			text->button = 0;
			gtk_grab_remove(widget);
			gtk_signal_emit_stop_by_name(
				GTK_OBJECT(widget), "button_release_event"
			);
			status = TRUE;
			break;
		}
		break;
	}

	return(status);
}

/*
 *	Any GtkWidget "changed" signal callback.
 */
static void edv_archive_comments_dlg_any_changed_cb(GtkWidget *widget, gpointer data)
{
	EDVArchiveCommentsDlg *d = EDV_ARCHIVE_COMMENTS_DLG(data);
	if(d == NULL)
		return;

	if(d->freeze_count > 0)
		return;

	d->freeze_count++;

	if(!d->has_changes)
		edv_archive_comments_dlg_set_has_changes(d, TRUE);

	d->freeze_count--;
}

/*
 *	Tool bar activate callback.
 */
static void edv_archive_comments_dlg_tool_bar_activate_cb(
	ToolBarItem *tool_bar,
	const gint op,
	gpointer data
)
{
	EDVArchiveCommentsDlg *d = EDV_ARCHIVE_COMMENTS_DLG(data);
	if(d == NULL)
		return;

	if(d->freeze_count > 0)
		return;

	d->freeze_count++;

	switch((EDVArchiveCommentsDlgOP)op)
	{
	    case EDV_ARCHIVE_COMMENTS_DLG_OP_CUT:
		edv_archive_comments_dlg_cut(d);
		break;
	    case EDV_ARCHIVE_COMMENTS_DLG_OP_COPY:
		edv_archive_comments_dlg_copy(d);
		break;
	    case EDV_ARCHIVE_COMMENTS_DLG_OP_PASTE:
		edv_archive_comments_dlg_paste(d);
		break;
	    case EDV_ARCHIVE_COMMENTS_DLG_OP_INSERT_FILE:
		edv_archive_comments_dlg_insert_file_query(d);
		break;
	}

	d->freeze_count--;
}

/*
 *	OK callback.
 */
static void edv_archive_comments_dlg_ok_cb(GtkWidget *widget, gpointer data)
{
	EDVArchiveCommentsDlg *d = EDV_ARCHIVE_COMMENTS_DLG(data);
	if(d == NULL)
		return;

	if(d->freeze_count > 0)
		return;

	d->freeze_count++;

	edv_archive_comments_dlg_unmap(d);
	edv_archive_comments_dlg_set_comments(d);
	edv_archive_comments_dlg_set_has_changes(d, FALSE);

	d->freeze_count--;
}

/*
 *	Apply callback.
 */
static void edv_archive_comments_dlg_apply_cb(GtkWidget *widget, gpointer data)
{
	EDVArchiveCommentsDlg *d = EDV_ARCHIVE_COMMENTS_DLG(data);
	if(d == NULL)
		return;

	if(d->freeze_count > 0)
		return;

	d->freeze_count++;

	edv_archive_comments_dlg_set_comments(d);
	edv_archive_comments_dlg_set_has_changes(d, FALSE);

	d->freeze_count--;
}

/*
 *	Cancel callback.
 */
static void edv_archive_comments_dlg_cancel_cb(GtkWidget *widget, gpointer data)
{
	EDVArchiveCommentsDlg *d = EDV_ARCHIVE_COMMENTS_DLG(data);
	if(d == NULL)
		return;

	if(d->freeze_count > 0)
		return;

	d->freeze_count++;

	edv_archive_comments_dlg_unmap(d);
	edv_archive_comments_dlg_set_has_changes(d, FALSE);

	d->freeze_count--;
}

/*
 *	Close callback.
 */
static void edv_archive_comments_dlg_close_cb(GtkWidget *widget, gpointer data)
{
	EDVArchiveCommentsDlg *d = EDV_ARCHIVE_COMMENTS_DLG(data);
	if(d == NULL)
		return;

	if(d->freeze_count > 0)
		return;

	d->freeze_count++;

	edv_archive_comments_dlg_unmap(d);
	edv_archive_comments_dlg_set_has_changes(d, FALSE);

	d->freeze_count--;
}


/*
 *	Set comments.
 */
static void edv_archive_comments_dlg_set_comments(EDVArchiveCommentsDlg *d)
{
	gint status;
	gchar *comment;
	const gchar *arch_path = d->arch_path;
	GtkWidget *toplevel;
	GtkEditable *editable;
	GtkText *text;
	CfgList *cfg_list;
	EDVCore *core;

	if(STRISEMPTY(arch_path))
		return;

	toplevel = d->toplevel;
	editable = GTK_EDITABLE(d->comment_text);
	text = GTK_TEXT(editable);
	core = d->core;
	cfg_list = core->cfg_list;

	edv_archive_comments_dlg_set_busy(d, TRUE);

	/* Get/copy the current comment string */
	comment = gtk_editable_get_chars(
		editable,
		0, -1				/* Get all the characters */
	);

	/* Set the archive's comment */
	status = edv_archive_comment_set(
		cfg_list,
		arch_path,
		comment
	);

	/* Delete the comment string */
	g_free(comment);

	/* Error setting the archive comment? */
	if(status != 0)
	{
		const gchar *msg;

		switch(status)
		{
		  case -3:
			msg =
#if defined(PROG_LANGUAGE_SPANISH)
"El error de sistemas."
#elif defined(PROG_LANGUAGE_FRENCH)
"Erreur de systmes."
#elif defined(PROG_LANGUAGE_GERMAN)
"Systeme fehler."
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'errore di sistemi."
#elif defined(PROG_LANGUAGE_DUTCH)
"Systemen fout."
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O erro de sistemas."
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Systemfeil."
#elif defined(PROG_LANGUAGE_POLISH)
"Bd systemu."
#else
"Systems error."
#endif
			;
			break;
		  case -2:
			msg =
#if defined(PROG_LANGUAGE_SPANISH)
"Este formato del archivo no sostiene los comentarios."
#elif defined(PROG_LANGUAGE_FRENCH)
"Ce format de l'archive pas soutien commente."
#elif defined(PROG_LANGUAGE_GERMAN)
"Das format dieses archivs untersttzt bemerkungen nicht."
#elif defined(PROG_LANGUAGE_ITALIAN)
"Questo formato dell'archivio non sostiene i commenti."
#elif defined(PROG_LANGUAGE_DUTCH)
"Het formaat van deze archief steunt opmerkingen niet."
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Este formato do arquivo nao apoia comentrios."
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Dette arkivs format sttter ikke kommentarer."
#elif defined(PROG_LANGUAGE_POLISH)
"Format tego archiwum nie obsuguje komentarzy."
#else
"This archive's format does not support comments."
#endif
			;
			break;
		  default:
			msg =
#if defined(PROG_LANGUAGE_SPANISH)
"El comentario del archivo de poner de error."
#elif defined(PROG_LANGUAGE_FRENCH)
"Le commentaire d'archive de montage d'erreur."
#elif defined(PROG_LANGUAGE_GERMAN)
"Fehler setzen archiv bemerkung."
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il commento di archivio di montaggio di errore."
#elif defined(PROG_LANGUAGE_DUTCH)
"Fout zetten archief opmerking."
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Erro por o comentrio de arquivo."
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"En feil forekommet whiles etting arkivkommentaren."
#elif defined(PROG_LANGUAGE_POLISH)
"Bd przy ustawianiu komentarza archiwum."
#else
"Error setting archive comment."
#endif
			;
			break;
		}
		edv_play_sound_warning(core);
		edv_message_warning(
#if defined(PROG_LANGUAGE_SPANISH)
"El Comentario Fijo Fall"
#elif defined(PROG_LANGUAGE_FRENCH)
"Rgler Le Commentaire Echou"
#elif defined(PROG_LANGUAGE_GERMAN)
"Feste Bemerkung Hat Versagt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Ha Regolato Il Commento Fallito"
#elif defined(PROG_LANGUAGE_DUTCH)
"Vast Opmerking Verzuimde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Comentrio Fixo Fracassou"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Fast Comment Failed"
#elif defined(PROG_LANGUAGE_POLISH)
"Bd ustawiania komentarza"
#else
"Set Comment Failed"
#endif
			, msg,
			NULL,
			toplevel
		);
	}
	else
	{
		EDVVFSObject *obj;

		/* Reset the has changes flag */
		edv_archive_comments_dlg_set_has_changes(d, FALSE);

		/* Notify about the archive being modified */
		obj = edv_vfs_object_lstat(arch_path);
		if(obj != NULL)
		{
			edv_emit_vfs_object_modified(
				core,
				arch_path,
				arch_path,
				obj
			);
			edv_vfs_object_delete(obj);
		}
		else
		{
			const gint error_code = (gint)errno;
			if(error_code == ENOENT)
				edv_emit_vfs_object_removed(
					core,
					arch_path
				);
		}
	}

	edv_archive_comments_dlg_set_busy(d, FALSE);
}

/*
 *	Cut.
 */
static void edv_archive_comments_dlg_cut(EDVArchiveCommentsDlg *d)
{
	gint len;
	GtkEditable *editable;
	GtkText *text;

	edv_archive_comments_dlg_set_busy(d, TRUE);

	text = GTK_TEXT(d->comment_text);
	editable = GTK_EDITABLE(text);

        /* Calculate the selected region to copy */
        len = (gint)editable->selection_end_pos -
                (gint)editable->selection_start_pos;
        if(len <= 0)
	{
		edv_archive_comments_dlg_set_busy(d, FALSE);
		return;
	}

        gtk_events_process();

	/* Cut */
        gtk_text_freeze(text);
        gtk_editable_cut_clipboard(editable);
       	gtk_text_thaw(text);

        gtk_events_process();

	edv_archive_comments_dlg_set_has_changes(d, TRUE);

	edv_archive_comments_dlg_set_busy(d, FALSE);
}

/*
 *	Copy.
 */
static void edv_archive_comments_dlg_copy(EDVArchiveCommentsDlg *d)
{
	gint len;
	GtkEditable *editable;
	GtkText *text;

	edv_archive_comments_dlg_set_busy(d, TRUE);

	text = GTK_TEXT(d->comment_text);
	editable = GTK_EDITABLE(text);

        /* Calculate the selected region to copy */
        len = (gint)editable->selection_end_pos -
                (gint)editable->selection_start_pos;
        if(len <= 0)
	{
		edv_archive_comments_dlg_set_busy(d, FALSE);
		return;
	}

        gtk_events_process();

	/* Copy */
        gtk_text_freeze(text);
        gtk_editable_copy_clipboard(editable);
       	gtk_text_thaw(text);

        gtk_events_process();

	edv_archive_comments_dlg_set_has_changes(d, TRUE);

	edv_archive_comments_dlg_set_busy(d, FALSE);
}

/*
 *	Paste.
 */
static void edv_archive_comments_dlg_paste(EDVArchiveCommentsDlg *d)
{
	GtkEditable *editable;
	GtkText *text;

	edv_archive_comments_dlg_set_busy(d, TRUE);

	text = GTK_TEXT(d->comment_text);
	editable = GTK_EDITABLE(text);

        gtk_events_process();

        /* Paste */
        gtk_text_freeze(text);
        gtk_editable_paste_clipboard(editable);
        gtk_text_thaw(text);

        gtk_events_process();

	edv_archive_comments_dlg_set_has_changes(d, TRUE);

	edv_archive_comments_dlg_set_busy(d, FALSE);
}

/*
 *	Inserts the file into the comments text.
 */
static void edv_archive_comments_dlg_insert_file(
	EDVArchiveCommentsDlg *d,
	const gchar *path
)
{
	FILE *fp;
	GtkWidget *toplevel;
	GtkText *text;

	if(STRISEMPTY(path))
		return;

	edv_archive_comments_dlg_set_busy(d, TRUE);

	toplevel = d->toplevel;
	text = GTK_TEXT(d->comment_text);

	/* Open the file for reading */
	fp = fopen((const char *)path, "rb");
	if(fp != NULL)
	{
		gulong block_size;
		guint8 *buf;
		EDVVFSObject *obj;

		gtk_text_freeze(text);

		obj = edv_vfs_object_fstat((gint)fileno(fp));
		if(obj != NULL)
		{
			block_size = obj->block_size;
			edv_vfs_object_delete(obj);
		}
		else
		{
			block_size = 0l;
		}

		/* Allocate the read buffer */
		buf = (block_size > 0l) ?
			(guint8 *)g_malloc(block_size * sizeof(guint8)) : NULL;
		if(buf != NULL)
		{
			gint units_read;
			gchar	*s,
					*s_end;

			/* Begin reading the file one block at a time */
			while(!feof(fp))
			{
				/* Read the next block from the file */
				units_read = (gint)fread(
					buf,
					sizeof(guint8),
					(size_t)block_size,
					fp
				);
				if(units_read <= 0)
					break;

				/* Convert non-printable characters into spaces */
				s = (gchar *)buf;
				s_end = s + units_read;
				while(s < s_end)
				{
					/* Not a printable character? */
					if(!isprint((int)(*s)))
					{
						/* Exclude newlines */
						if(*s != '\n')
							*s = ' ';
					}
					s++;
				}

				/* Insert this block to the comment text */
				gtk_text_insert(
					text,
					NULL,
					NULL,
					NULL,
					(const gchar *)buf, units_read * sizeof(gchar)
				);
			}

			/* Delete the read buffer */
			g_free(buf);
		}
		else
		{
			gint c;
			guint8 buf[1];

			while(!feof(fp))
			{
				/* Read the next character from the file */
				c = (gint)fgetc(fp);
				if((int)c == EOF)
					break;

				/* Convert non-printable characters into spaces */
				if(!isprint((int)c) && (c != '\n'))
					c = ' ';

				buf[0] = c;

				/* Insert this character to the comment text */
				gtk_text_insert(
					text,
					NULL,
					NULL, NULL,
					(const gchar *)buf, 1
				);
			}
		}

		/* Close the file */
		(void)fclose(fp);

		gtk_text_thaw(text);

		edv_archive_comments_dlg_set_has_changes(d, TRUE);
	}
	else
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %s",
			g_strerror(error_code),
			path
		);
		edv_play_sound_warning(d->core);
		edv_message_warning(
			"Insert File Failed",
			msg,
			NULL,
			toplevel
		);
		g_free(msg);
	}

	edv_archive_comments_dlg_set_busy(d, FALSE);
}

/*
 *	Queries the user to insert a comment file.
 */
static void edv_archive_comments_dlg_insert_file_query(EDVArchiveCommentsDlg *d)
{
	gboolean response;
	gint		npaths = 0,
			nftypes = 0;
	gchar		*parent_path,
			**paths_list = NULL;
	GtkWidget *toplevel;
	fb_type_struct	**ftype = NULL,
			*ftype_rtn = NULL;

	if(FileBrowserIsQuery())
		return;

	edv_archive_comments_dlg_set_busy(d, TRUE);

	toplevel = d->toplevel;

	/* Get startup path from archive path */
	parent_path = (d->arch_path != NULL) ?
		g_dirname(d->arch_path) : NULL;

	/* Create the file types list */
	FileBrowserTypeListNew(
		&ftype, &nftypes,
		"*.*", "All Files"
	);

	/* Query the user for the file to insert */
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
		"Insert File",
		"Insert", "Cancel",
		parent_path,
		ftype, nftypes,
		&paths_list, &npaths,
		&ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(response && (npaths > 0))
	{
		gint i;
		gchar *path;
		for(i = npaths - 1; i >= 0; i--)
		{
			path = STRDUP(paths_list[i]);
			if(path == NULL)
				continue;

			/* Insert this file */
			edv_archive_comments_dlg_insert_file(
				d,
				path
			);

			g_free(path);
		}
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftype, nftypes);

	g_free(parent_path);

	edv_archive_comments_dlg_set_busy(d, FALSE);
}


/*
 *	Creates a new EDVArchiveCommentsDlg.
 *
 *	The arch_path specifies the path to the archive.
 *
 *	The password specifies the password.
 *
 *	If objs_list is not NULL then it specifies a GList of
 *	EDVArchiveObject * archive objects that will be used to
 *	obtain the statistics from, otherwise the statistics will be
 *	obtained from listing the archive specified by arch_path.
 *
 *	If toplevel is not NULL then it specifies the relative toplevel
 *	GtkWindow.
 */
EDVArchiveCommentsDlg *edv_archive_comments_dlg_new(
	EDVCore *core,
	const gchar *arch_path,
	const gchar *password,
	const gchar *comments,
	GtkWidget *toplevel
)
{
	const gint	border_major = 5,
			border_minor = 2;
	gint font_size;
	const gchar *font_encoding;
	gchar *font_name_fixed_width;
	GList *tool_bar_items_list;
	GdkFont *font;
	GdkWindow *window;
	GtkStyle *style;
	GtkRcStyle *rcstyle;
	GtkAccelGroup *accelgrp;
	GtkWidget	*w,
			*parent, *parent2,
			*ref_toplevel = toplevel;
	GtkEditable *editable;
	GtkText *text;
	CfgList *cfg_list;
	EDVArchiveCommentsDlg *d;

	if(core == NULL)
		return(NULL);

	cfg_list = core->cfg_list;

	d = EDV_ARCHIVE_COMMENTS_DLG(g_malloc0(
		sizeof(EDVArchiveCommentsDlg)
	));
	if(d == NULL)
		return(NULL);

	d->toplevel = toplevel = gtk_window_new(GTK_WINDOW_DIALOG);
	d->accelgrp = accelgrp = gtk_accel_group_new();
#if 0
	d->has_changes = FALSE;
	d->freeze_count = 0;
	d->busy_count = 0;
#endif
	d->core = core;

	d->arch_path = STRDUP(arch_path);

	d->busy_cur = gdk_cursor_new(GDK_WATCH);

	d->freeze_count++;


	/* Toplevel GtkWindow */
	w = toplevel;
	gtk_window_set_wmclass(
		GTK_WINDOW(w), "dialog", PROG_NAME
	);
	
	gtk_window_set_policy(GTK_WINDOW(w), TRUE, TRUE, FALSE);
	gtk_widget_set_usize(
		w,
		EDV_ARCHIVE_COMMENTS_DLG_DEF_WIDTH,
		EDV_ARCHIVE_COMMENTS_DLG_DEF_HEIGHT
	);
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
/*
		gdk_window_set_decorations(
			window,
			GDK_DECOR_BORDER | GDK_DECOR_TITLE | GDK_DECOR_MENU |
			GDK_DECOR_MINIMIZE
		);
		gdk_window_set_functions(
			window,
			GDK_FUNC_MOVE | GDK_FUNC_MINIMIZE | GDK_FUNC_CLOSE
		);
 */
/*		GUISetWMIcon(window, (guint8 **)icon_endeavour_archiver_48x48_xpm); */
	}
	style = gtk_widget_get_style(w);
	gtk_signal_connect(
		GTK_OBJECT(w), "delete_event",
		GTK_SIGNAL_FUNC(edv_archive_comments_dlg_delete_event_cb), d
	);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	if(ref_toplevel != NULL)
	{
		if(GTK_IS_WINDOW(GTK_OBJECT(ref_toplevel)))
			gtk_window_set_transient_for(
				GTK_WINDOW(w),
				GTK_WINDOW(ref_toplevel)
			);
	}
	parent = w;


	/* Get the base font size */
	font = style->font;
	font_size = GDK_FONT_GET_FONT_NAME_SIZE(font);
	if(font_size < 3)
		font_size = 3;

	/* Format the font names */
#if defined(PROG_LANGUAGE_POLISH)
	font_encoding = "iso8859-2";
#else
	font_encoding = "iso8859-1";
#endif

	font_name_fixed_width = g_strdup_printf(
"-adobe-courier-medium-r-normal-*-%i-*-*-*-*-*-%s",
		font_size,
		font_encoding
	);


	/* Main GtkVBox */
	d->main_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;


	/* Tool Bar GtkFrame */
	w = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_OUT);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Tool Bar Items */
	tool_bar_items_list = NULL;
	tool_bar_items_list = g_list_append(
		tool_bar_items_list,
		ToolBarItemNew(
			TOOL_BAR_ITEM_BUTTON,
			"Cut",
			(guint8 **)icon_cut_20x20_xpm,
			"Cut the selected text to the clipboard",
			EDV_ARCHIVE_COMMENTS_DLG_OP_CUT,
			edv_archive_comments_dlg_tool_bar_activate_cb, d,
			NULL, d,
			NULL, d
		)
	);
	tool_bar_items_list = g_list_append(
		tool_bar_items_list,
		ToolBarItemNew(
			TOOL_BAR_ITEM_BUTTON,
			"Copy",
			(guint8 **)icon_copy_20x20_xpm,
			"Copy the selected text to the clipboard",
			EDV_ARCHIVE_COMMENTS_DLG_OP_COPY,
			edv_archive_comments_dlg_tool_bar_activate_cb, d,
			NULL, d,
			NULL, d
		)
	);
	tool_bar_items_list = g_list_append(
		tool_bar_items_list,
		ToolBarItemNew(
			TOOL_BAR_ITEM_BUTTON,
			"Paste",
			(guint8 **)icon_paste_20x20_xpm,
			"Paste the text from the clipboard",
			EDV_ARCHIVE_COMMENTS_DLG_OP_PASTE,
			edv_archive_comments_dlg_tool_bar_activate_cb, d,
			NULL, d,
			NULL, d
		)
	);
	tool_bar_items_list = g_list_append(
		tool_bar_items_list,
		ToolBarItemNewSeparator()
	);
	tool_bar_items_list = g_list_append(
		tool_bar_items_list,
		ToolBarItemNew(
			TOOL_BAR_ITEM_BUTTON,
			"Insert",
			(guint8 **)icon_folder_opened_20x20_xpm,
			"Insert a file",
			EDV_ARCHIVE_COMMENTS_DLG_OP_INSERT_FILE,
			edv_archive_comments_dlg_tool_bar_activate_cb, d,
			NULL, d,
			NULL, d
		)
	);

	/* Tool Bar */
	d->tool_bar = w = ToolBarNew(
		GTK_ORIENTATION_HORIZONTAL,
		TOOL_BAR_BUTTON_DECALS_PICTURES,
		GTK_RELIEF_NONE,
		GTK_POLICY_AUTOMATIC,
		tool_bar_items_list
	);
	g_list_foreach(tool_bar_items_list, (GFunc)ToolBarItemDelete, NULL);
	g_list_free(tool_bar_items_list);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_minor);
	gtk_widget_show(w);


	/* Comments text and scroll bars GtkTable */
	w = gtk_table_new(2, 2, FALSE);
	gtk_table_set_row_spacing(GTK_TABLE(w), 0, border_minor);
	gtk_table_set_col_spacing(GTK_TABLE(w), 0, border_minor);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;
	/* Comments GtkText */
	rcstyle = gtk_rc_style_new();
	rcstyle->font_name = STRDUP(font_name_fixed_width);
	d->comment_text = w = gtk_text_new(NULL, NULL);
	editable = GTK_EDITABLE(w);
	text = GTK_TEXT(w);
/*	gtk_widget_set_usize(w, 520, 150); */
	text->default_tab_width = 8;
	gtk_text_set_editable(text, TRUE);
	gtk_text_set_word_wrap(text, TRUE);
	gtk_table_attach(
		GTK_TABLE(parent2), w,
		0, 1, 0, 1,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		0, 0
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(edv_archive_comments_dlg_text_event_cb), d
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(edv_archive_comments_dlg_text_event_cb), d
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "changed",
		GTK_SIGNAL_FUNC(edv_archive_comments_dlg_any_changed_cb), d
	);
	gtk_widget_modify_style(w, rcstyle);
	GTK_RC_STYLE_UNREF(rcstyle);
	GUIEditableEndowPopupMenu(w, GUI_EDITABLE_POPUP_MENU_UNDO);
	if(!STRISEMPTY(comments))
	{
		gtk_text_freeze(text);
		gtk_text_insert(
			text,
			NULL,
			NULL, NULL,
			comments, -1
		);
		gtk_text_thaw(text);
	}
	gtk_widget_show(w);
	/* Vertical GtkScrollBar */
	w = gtk_vscrollbar_new(text->vadj);
	gtk_table_attach(
		GTK_TABLE(parent2), w,
		1, 2, 0, 1,
		GTK_FILL,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		0, 0
	);
	gtk_widget_show(w);


	/* Separator */
	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Buttons GtkHBox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* OK GtkButton */
	d->ok_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_ok_20x20_xpm, "OK", NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_FOCUS | GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_archive_comments_dlg_ok_cb), d
	);
	gtk_accel_group_add(
		accelgrp,
		GDK_Return, 0,
		GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
		accelgrp,
		GDK_o, GDK_CONTROL_MASK,
		GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_o);

	/* Apply GtkButton */
	d->apply_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_select_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Aplique"
#elif defined(PROG_LANGUAGE_FRENCH)
"Appliquer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Verwenden"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Applicare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Toepas"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Aplique"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Gjeld"
#else
"Apply"
#endif
		, NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_FOCUS | GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_archive_comments_dlg_apply_cb), d
	);
	gtk_accel_group_add(
		accelgrp,
		GDK_a, GDK_CONTROL_MASK,
		GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_a);

	/* Cancel GtkButton */
	d->cancel_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_cancel_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Cancele"
#elif defined(PROG_LANGUAGE_FRENCH)
"Annuler"
#elif defined(PROG_LANGUAGE_GERMAN)
"Heben"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Annullare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Annuleer"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Cancelamento"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kanseller"
#else
"Cancel"
#endif
		, NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_FOCUS | GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_archive_comments_dlg_cancel_cb), d
	);
	gtk_accel_group_add(
		accelgrp,
		GDK_Escape, 0,
		GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
		accelgrp,
		GDK_c, GDK_CONTROL_MASK,
		GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);

	/* Close GtkButton */
	d->close_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_close_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Cierre"
#elif defined(PROG_LANGUAGE_FRENCH)
"Quitter"
#elif defined(PROG_LANGUAGE_GERMAN)
"Nah"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Vicino"
#elif defined(PROG_LANGUAGE_DUTCH)
"Einde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Prximo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Nr"
#else
"Close"
#endif
		, NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_FOCUS | GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_archive_comments_dlg_close_cb), d
	);
	gtk_accel_group_add(
		accelgrp,
		GDK_Escape, 0,
		GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
		accelgrp,
		GDK_c, GDK_CONTROL_MASK,
		GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);


	edv_archive_comments_dlg_update_display(d);


	g_free(font_name_fixed_width);

	d->freeze_count--;

	return(d);
}

/*
 *	Resets the EDVArchiveCommentsDlg has_changes marker.
 */
void edv_archive_comments_dlg_set_has_changes(
	EDVArchiveCommentsDlg *d,
	const gboolean has_changes
)
{
	if(d == NULL)
		return;

	if(d->has_changes == has_changes)
		return;

	d->has_changes = has_changes;
	edv_archive_comments_dlg_update_display(d);
}

/*
 *	Updates the EDVArchiveCommentsDlg's widgets to reflect current
 *	values.
 */
void edv_archive_comments_dlg_update_display(EDVArchiveCommentsDlg *d)
{
	gboolean has_changes;
	gchar *s;

	if(d == NULL)
		return;

	has_changes = d->has_changes;

	/* Title */
	s = g_strconcat(
		"Comments",
		": ",
		(d->arch_path != NULL) ?
			g_basename(d->arch_path) : "Untitled",
		(has_changes) ? " (*)" : "",
		NULL
	);
	gtk_window_set_title(GTK_WINDOW(d->toplevel), s);
	g_free(s);

	if(has_changes)
	{
		gtk_widget_show(d->ok_btn);
		gtk_widget_set_sensitive(d->ok_btn, TRUE);
		gtk_widget_show(d->apply_btn);
		gtk_widget_set_sensitive(d->apply_btn, TRUE);
		gtk_widget_show(d->cancel_btn);
		gtk_widget_set_sensitive(d->cancel_btn, TRUE);
		gtk_widget_hide(d->close_btn);
		gtk_widget_set_sensitive(d->close_btn, FALSE);
	}
	else
	{
		gtk_widget_show(d->ok_btn);
		gtk_widget_set_sensitive(d->ok_btn, FALSE);
		gtk_widget_show(d->apply_btn);
		gtk_widget_set_sensitive(d->apply_btn, FALSE);
		gtk_widget_hide(d->cancel_btn);
		gtk_widget_set_sensitive(d->cancel_btn, FALSE);
		gtk_widget_show(d->close_btn);
		gtk_widget_set_sensitive(d->close_btn, TRUE);
	}
}

/*
 *	Sets the EDVArchiveCommentsDlg as busy or ready.
 */
void edv_archive_comments_dlg_set_busy(
	EDVArchiveCommentsDlg *d,
	const gboolean busy
)
{
	GdkCursor *cur;
	GtkWidget *w;

	if(d == NULL)
		return;

	w = d->toplevel;

	if(busy)
	{
		d->busy_count++;
		if(d->busy_count > 1)
			return;

		cur = d->busy_cur;
	}
	else
	{
		d->busy_count--;
		if(d->busy_count < 0)
			d->busy_count = 0;

		if(d->busy_count > 0)
			return;

		cur = NULL;
	}

	gdk_window_set_cursor(w->window, cur);

	gdk_flush();
}

/*
 *	Checks if the EDVArchiveCommentsDlg is mapped.
 */
gboolean edv_archive_comments_dlg_is_mapped(EDVArchiveCommentsDlg *d)
{
	if(d == NULL)
		return(FALSE);

	return(GTK_WIDGET_MAPPED(d->toplevel));
}

/*
 *	Checks if the EDVArchiveCommentsDlg has changes.
 */
gboolean edv_archive_comments_dlg_has_changes(EDVArchiveCommentsDlg *d)
{
	if(d == NULL)
		return(FALSE);

	return(d->has_changes);
}

/*
 *	Maps the EDVArchiveCommentsDlg.
 */
void edv_archive_comments_dlg_map(EDVArchiveCommentsDlg *d)
{
	GtkWidget *w;

	if(d == NULL)
		return;

	gtk_widget_show_raise(d->toplevel);

	/* Grab focus and default on the OK button */
	w = d->ok_btn;
	if(w != NULL)
	{
		gtk_widget_grab_focus(w);
		gtk_widget_grab_default(w);
	}
}

/*
 *	Unmap the EDVArchiveCommentsDlg.
 */
void edv_archive_comments_dlg_unmap(EDVArchiveCommentsDlg *d)
{
	if(d == NULL)
		return;

	gtk_widget_hide(d->toplevel);
}

/*
 *	Deletes the EDVArchiveCommentsDlg.
 */
void edv_archive_comments_dlg_delete(EDVArchiveCommentsDlg *d)
{
	if(d == NULL)
		return;

	edv_archive_comments_dlg_unmap(d);

	gtk_window_set_transient_for(
		GTK_WINDOW(d->toplevel),
		NULL
	);

	d->freeze_count++;

	gtk_widget_destroy(d->toplevel);
	gtk_accel_group_unref(d->accelgrp);

	GDK_CURSOR_DESTROY(d->busy_cur);

	g_free(d->arch_path);

	d->freeze_count--;

	g_free(d);
}
