#include <time.h>
#include <signal.h>
#include <gtk/gtk.h>

#include "../../include/prochandle.h"

#include "../guiutils.h"
#include "sysinfowin.h"
#include "config.h"


static void SysInfoSignalCB(int s);
static gint SysInfoTimeOutCB(gpointer data);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *      UNIX signal callback.
 */
static void SysInfoSignalCB(int s)
{
	switch(s)
	{
	  case SIGINT:
	  case SIGTERM:
	  case SIGSEGV:
	    exit(1);
	    break;
	}
}

/*
 *	Timeout callback.
 */
static gint SysInfoTimeOutCB(gpointer data)
{
	sysinfo_win_struct *si = SYSINFO_WIN(data);
	if(!SysInfoIsMapped(si))
	{
	    gtk_main_quit();
	    return(FALSE);
	}

	SysInfoUpdate(si);

	return(TRUE);
}


int main(int argc, char *argv[])
{
	gboolean initialized_gtk = FALSE;
	gint i;
	const gchar *arg;
	guint id;
	sysinfo_display_flags display = SYSINFO_DISPLAY_CPU_LOAD |
	    SYSINFO_DISPLAY_CPU_LOADAVG | SYSINFO_DISPLAY_MEMORY |
	    SYSINFO_DISPLAY_APM_POWER;
	sysinfo_bar_style bar_style = SYSINFO_BAR_STYLE_CONTINUOUS;
	sysinfo_bar_orientation bar_orientation = SYSINFO_BAR_ORIENTATION_HORIZONTAL;
	sysinfo_win_struct *si;


	/* Set up time zone */
	tzset();

	/* Set up signal callbacks */
#ifdef SIGINT
	signal(SIGINT, SysInfoSignalCB);
#endif
#ifdef SIGTERM
	signal(SIGTERM, SysInfoSignalCB);
#endif
#ifdef SIGKILL
	signal(SIGKILL, SysInfoSignalCB);
#endif
#ifdef SIGSEGV
	signal(SIGSEGV, SysInfoSignalCB);
#endif
#ifdef SIGSTOP
	signal(SIGSTOP, SysInfoSignalCB);
#endif
#ifdef SIGCONT
	signal(SIGCONT, SysInfoSignalCB);
#endif
#ifdef SIGPIPE
	signal(SIGPIPE, SysInfoSignalCB);
#endif


	/* Parse arguments */
	for(i = 1; i < argc; i++)
	{
	    arg = argv[i];
	    if(arg == NULL)
		continue;

	    /* Help */
	    if(!g_strcasecmp(arg, "--help") ||
	       !g_strcasecmp(arg, "-help")
	    )
	    {
		g_print(PROG_HELP_MESG);
		return(0);
	    }
	    /* Version */
	    else if(!g_strcasecmp(arg, "--version") ||
		    !g_strcasecmp(arg, "-version")
	    )
	    {
		g_print(
		    PROG_NAME " Version " PROG_VERSION "\n"
		    PROG_COPYRIGHT
		);
		return(0);
	    }
	    /* Display */
	    else if(!g_strcasecmp(arg, "-d"))
	    {
		display = 0;		/* Reset display flags */
		i++;
		arg = (i < argc) ? argv[i] : NULL;
		while((arg != NULL) ? (*arg != '-') : FALSE)
		{
		    if(!g_strcasecmp(arg, "load"))
			display |= SYSINFO_DISPLAY_CPU_LOAD;
		    else if(!g_strcasecmp(arg, "loadavg") ||
			    !g_strcasecmp(arg, "avg") ||
			    !g_strcasecmp(arg, "average")
		    )
			display |= SYSINFO_DISPLAY_CPU_LOADAVG;
		    else if(!g_strcasecmp(arg, "memory") ||
			    !g_strcasecmp(arg, "mem")
		    )
			display |= SYSINFO_DISPLAY_MEMORY;
		    else if(!g_strcasecmp(arg, "apmpower") ||
			    !g_strcasecmp(arg, "apm")
		    )
			display |= SYSINFO_DISPLAY_APM_POWER;
		    else if(!g_strcasecmp(arg, "details"))
			display |= SYSINFO_DISPLAY_CPU_DETAILS;
		    else
			g_printerr(
"-d: Warning: Unsupported display flag \"%s\".\n",
			    arg
			);

		    i++;
		    arg = (i < argc) ? argv[i] : NULL;
		}
		i--;
	    }
	    /* Bar Style */
	    else if(!g_strcasecmp(arg, "--style") ||
		    !g_strcasecmp(arg, "-style") ||
		    !g_strcasecmp(arg, "-s")
	    )
	    {
		i++;
		arg = (i < argc) ? argv[i] : NULL;
		if(!STRISEMPTY(arg))
		{
		    if(*arg == 'c')
			bar_style = SYSINFO_BAR_STYLE_CONTINUOUS;
		    else if(*arg == 'd')
			bar_style = SYSINFO_BAR_STYLE_DISCRETE;
		    else if(*arg == 'l')
			bar_style = SYSINFO_BAR_STYLE_LED;
		    else
			g_printerr(
"%s: Warning: Unsupported argument \"%s\".\n",
			    argv[i - 1], arg
			);
		}
	    }
	    /* Bar Orientation */
	    else if(!g_strcasecmp(arg, "--orientation") ||
		    !g_strcasecmp(arg, "-orientation") ||
		    !g_strcasecmp(arg, "-o")
	    )
	    {
		i++;
		arg = (i < argc) ? argv[i] : NULL;
		if(!STRISEMPTY(arg))
		{
		    if(*arg == 'h')
			bar_orientation = SYSINFO_BAR_ORIENTATION_HORIZONTAL;
		    else if(*arg == 'v')
			bar_orientation = SYSINFO_BAR_ORIENTATION_VERTICAL;
		    else
			g_printerr(
"%s: Warning: Unsupported argument \"%s\".\n",
			    argv[i - 1], arg
			);
		}
	    }
	}


	/* Set GTK locale */
	gtk_set_locale();

	/* Initialize GTK as needed */
	if(!initialized_gtk)
	{
	    if(!gtk_init_check(&argc, &argv))
	    {
		g_printerr("Unable to initialize GTK.\n");
		return(1);
	    }
	    initialized_gtk = TRUE;
	}

	/* Initialize GDK RGB buffers system */
	gdk_rgb_init();


	/* Create new SysInfo Window */
	si = SysInfoNew(
	    NULL,
	    display,
	    bar_style,
	    bar_orientation
	);
	gtk_window_apply_args(GTK_WINDOW(si->toplevel), argc, argv);
	SysInfoMap(si);

	id = gtk_timeout_add(
	    250,
	    SysInfoTimeOutCB, si
	);

	gtk_main();


	/* Begin deleting */

	SysInfoDelete(si);

        /* Reset the DND Icon */
        GUIDNDSetDragIcon(NULL, NULL, 0, 0);

	return(0);
}
