#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <unistd.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "cdialog.h"
#include "fb.h"

#include "cfg.h"
#include "edvtypes.h"
#include "rundlg.h"
#include "endeavour.h"
#include "edvop.h"
#include "edvutils.h"
#include "edvutilsgtk.h"
#include "edvcfglist.h"
#include "config.h"

#include "images/icon_run_20x20.xpm"
#include "images/icon_executable_48x48.xpm"
#include "images/icon_close_20x20.xpm"
#include "images/icon_browse_20x20.xpm"
#include "images/icon_clear_20x20.xpm"
#include "images/icon_run_dir_from_cmd_20x20.xpm"
#include "images/icon_options2_20x20.xpm"


static gint EDVRunDlgDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void EDVRunDlgRunCB(
	GtkWidget *widget, gpointer data
);
static gint EDVRunDlgCommandFocusEventCB(
	 GtkWidget *widget, GdkEventFocus *focus, gpointer data
);
static void EDVRunDlgBrowseCommandCB(
	GtkWidget *widget, gpointer data
);
static void EDVRunDlgBrowseWorkingDirCB(
	GtkWidget *widget, gpointer data
);
static void EDVRunDlgClearCB(
	GtkWidget *widget, gpointer data
);
static void EDVRunDlgGetWorkingDirFromCmdCB(
	GtkWidget *widget, gpointer data
);
static void EDVRunDlgOptionsCB(
	GtkWidget *widget, gpointer data
);
static void EDVRunDlgCloseCB(GtkWidget *widget, gpointer data);

/* Utilities */
static void EDVRunDlgCompleteCommand(edv_run_dlg_struct *dlg);

/* Set & Get Values */
void EDVRunDlgSetCommand(
	edv_run_dlg_struct *dlg, const char *command
);
void EDVRunDlgSetWorkingDir(
	edv_run_dlg_struct *dlg, const char *working_dir
);
void EDVRunDlgFetchValues(edv_run_dlg_struct *dlg);

/* Run Dialog */
edv_run_dlg_struct *EDVRunDlgNew(gpointer core_ptr);
void EDVRunDlgSetBusy(edv_run_dlg_struct *dlg, gboolean is_busy);
gboolean EDVRunDlgIsMapped(edv_run_dlg_struct *dlg);
void EDVRunDlgMap(edv_run_dlg_struct *dlg);
void EDVRunDlgUnmap(edv_run_dlg_struct *dlg);
void EDVRunDlgDelete(edv_run_dlg_struct *dlg);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


#if defined(PROG_LANGUAGE_SPANISH)
# define RUNDLG_TITLE		"Corra"
#elif defined(PROG_LANGUAGE_FRENCH)
# define RUNDLG_TITLE		"Course"
#elif defined(PROG_LANGUAGE_GERMAN)
# define RUNDLG_TITLE		"Lauf"
#elif defined(PROG_LANGUAGE_ITALIAN)
# define RUNDLG_TITLE		"Ha Correto"
#elif defined(PROG_LANGUAGE_DUTCH)
# define RUNDLG_TITLE		"Tocht"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
# define RUNDLG_TITLE		"A Corrida"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
# define RUNDLG_TITLE		"Lp"
#else
# define RUNDLG_TITLE		"Run"
#endif

#define RUNDLG_WIDTH		500
#define RUNDLG_HEIGHT		-1


/*
 *	Run dialog "delete_event" signal callback.
 */
static gint EDVRunDlgDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	edv_run_dlg_struct *dlg = EDV_RUN_DLG(data);
	if(dlg == NULL)
	    return(TRUE);

	EDVRunDlgCloseCB(NULL, dlg);

	return(TRUE);
}

/*
 *      Run dialog run callback.
 */
static void EDVRunDlgRunCB(
	GtkWidget *widget, gpointer data
)
{
	gint euid;
	gchar *s, *buf, *prog, *args, *cmd;
	const gchar *cmd_src, *working_dir;
	gchar orig_working_dir[PATH_MAX];
	struct stat stat_buf;
	const cfg_item_struct *cfg_list;
	GtkWidget *toplevel;
	edv_core_struct *core_ptr;
	edv_run_dlg_struct *dlg = EDV_RUN_DLG(data);
	if(dlg == NULL)
	    return;

	toplevel = dlg->toplevel;
	core_ptr = EDV_CORE(dlg->core_ptr);
	cfg_list = core_ptr->cfg_list;
	euid = core_ptr->effective_user_id;

	EDVRunDlgSetBusy(dlg, TRUE);

	/* Complete command as needed */
	EDVRunDlgCompleteCommand(dlg);

	/* Get working directory and original working directory */
	working_dir = gtk_entry_get_text(
	    GTK_ENTRY(dlg->working_dir_entry)
	);
	if(getcwd(orig_working_dir, sizeof(orig_working_dir)) != NULL)
	    orig_working_dir[sizeof(orig_working_dir) - 1] = '\0';
	else
	    strcpy(orig_working_dir, "/");

	/* Get command */
	cmd_src = gtk_entry_get_text(
	    GTK_ENTRY(GTK_COMBO(dlg->run_combo)->entry)
	);
	if(STRISEMPTY(cmd_src))
	{
	    EDVRunDlgSetBusy(dlg, FALSE);
	    return;
	}

	/* Seek past initial spaces */
	while(ISBLANK(*cmd_src))
	    cmd_src++;

	/* Get program from command */
	prog = STRDUP(cmd_src);
	s = strpbrk(prog, " \t");
	if(s != NULL)
	    *s = '\0';

	/* Get arguments from command */
	s = strpbrk(cmd_src, " \t");
	if(s != NULL)
	{
	    while(ISBLANK(*s))
		s++;
	    args = STRDUP(s);
	}
	else
	{
	    args = NULL;
	}

	/* Record this command */
	if(args != NULL)
	    buf = g_strdup_printf("%s %s", prog, args);
	else
	    buf = g_strdup_printf("%s", prog);
	GUIComboAddItem(dlg->run_combo, buf);
	g_free(buf);

	/* Get program stats to see if it exists and is executable */
	if(stat(prog, &stat_buf))
	{
	    s = STRDUP(strerror(errno));
	    if(s == NULL)
		s = STRDUP("no such file");
	    *s = toupper(*s);
	    buf = g_strdup_printf(
"%s:\n\
\n\
    %s",
		s, prog
	    );
	    EDVPlaySoundWarning(core_ptr);
	    EDVMessageWarning(
		"Run Failed", buf, NULL, toplevel
	    );
	    g_free(buf);
	    g_free(s);

	    EDVRunDlgSetBusy(dlg, FALSE);
	    g_free(prog);
	    g_free(args);
	    return;
	}
	else if(((euid == stat_buf.st_uid) || (euid == 0)) ?
		 !(stat_buf.st_mode & S_IXUSR) :
		 !(stat_buf.st_mode & S_IXOTH)
	       )
	{
	    buf = g_strdup_printf(
"Not executable:\n\
\n\
    %s\n\
\n\
The program must be set executable in order to run it.\n",
		prog
	    );
	    EDVPlaySoundWarning(core_ptr);
	    EDVMessageWarning(
		"Run Failed", buf, NULL, toplevel
	    );
	    g_free(buf);

	    EDVRunDlgSetBusy(dlg, FALSE);
	    g_free(prog);
	    g_free(args);
	    return;
	}

	/* Format command */
	if(GTK_TOGGLE_BUTTON_GET_ACTIVE(dlg->run_in_terminal_check))
	{
	    /* Format command for running in a terminal */
	    const gchar *cmd_terminal_run = EDV_GET_S(
		EDV_CFG_PARM_PROG_TERMINAL_RUN
	    );
	    if(!STRISEMPTY(cmd_terminal_run))
	    {
		/* Format command for run in terminal */
		if(args != NULL)
		    cmd = g_strdup_printf(
		        "%s %s %s",
			cmd_terminal_run, prog, args
		    );
		else
		    cmd = g_strdup_printf(
			"%s %s",
			cmd_terminal_run, prog
		    );


	    }
	    else
	    {
		EDVPlaySoundWarning(core_ptr);
		EDVMessageWarning(   
		    "Run Failed",
"The \"Run In Terminal\" command was not set.\n\
\n\
To set the \"Run In Terminal\" command go to\n\
Settings->Options...->Programs.",
		    NULL,
		    toplevel
		);
		EDVRunDlgSetBusy(dlg, FALSE);
		g_free(prog);
		g_free(args);
		return;
	    }
	}
	else
	{
	    /* Format plain command */
	    if(args != NULL)
		cmd = g_strdup_printf(
		    "%s %s &",
		    prog, args
		);
	    else
		cmd = g_strdup_printf(
		    "%s &",
		    prog
		);
	}

	/* Generated command successfully? */
	if(cmd != NULL)
	{
	    gint p;
	    gchar *buf;

	    /* Change working directory before executing command */
	    if(!STRISEMPTY(working_dir))
		chdir(working_dir);

	    /* Execute command */
	    p = EDVSystem(cmd);

	    /* Restore original working directory */
	    if(!STRISEMPTY(orig_working_dir))
		chdir(orig_working_dir);

	    /* Error executing the command? */
	    if(p <= 0)
	    {
		buf = g_strdup_printf(
"Unable to execute command:\n\
\n\
    %s",
		    cmd
		);
		EDVPlaySoundError(core_ptr);
		EDVMessageError(
"Run Failed",
		    buf,
"Please verify that the program exists, is executable, and any\n\
specified arguments are valid.",
		    toplevel
		);
		g_free(buf);
	    }
	}

	/* Do not keep dialog after running? */
	if(!GTK_TOGGLE_BUTTON_GET_ACTIVE(dlg->keep_dialog_check))
	    EDVRunDlgCloseCB(dlg->close_btn, dlg);

	g_free(cmd);
	g_free(prog);
	g_free(args);
	EDVRunDlgSetBusy(dlg, FALSE);
}


/*
 *	Run Dialog Command GtkEntry "focus_in_event" or
 *	"focus_out_event" signal callback.
 */
static gint EDVRunDlgCommandFocusEventCB(
	 GtkWidget *widget, GdkEventFocus *focus, gpointer data
)
{
	gint status = FALSE;
	edv_run_dlg_struct *dlg = EDV_RUN_DLG(data);
	if((widget == NULL) || (focus == NULL) || (dlg == NULL))
	    return(status);

	if(focus->in)
	{
	    status = TRUE;
	}
	else
	{
	    EDVRunDlgCompleteCommand(dlg);
	    status = TRUE;
	}

	return(status);
}


/*
 *	Run Dialog browse command callback.
 */
static void EDVRunDlgBrowseCommandCB(
	GtkWidget *widget, gpointer data
)
{
	gboolean status;
	gchar *s, *cur_prog;
	fb_type_struct **ftype = NULL, *ftype_rtn = NULL;
	gint total_ftypes = 0;
	gchar **path_rtn = NULL;
	gint total_path_rtns = 0;
	GtkWidget *toplevel, *w;
	GtkEntry *entry;
	edv_run_dlg_struct *dlg = EDV_RUN_DLG(data);
	if((dlg == NULL) || FileBrowserIsQuery())
	    return;

	toplevel = dlg->toplevel;
	w = dlg->run_combo;
	if(w == NULL)
	    return;
	entry = GTK_ENTRY(GTK_COMBO(w)->entry);

	cur_prog = STRDUP(gtk_entry_get_text(entry));
	s = strpbrk(cur_prog, " \t");
	if(s != NULL)
	    *s = '\0';


	EDVRunDlgSetBusy(dlg, TRUE);

	/* Create file types list */
	FileBrowserTypeListNew(
	    &ftype, &total_ftypes,
	    "*.*", "All files"
	);

	/* Query user for program */
	FileBrowserSetTransientFor(toplevel);
	status = FileBrowserGetResponse(
	    "Select Program",
	    "Select", "Cancel",
	    cur_prog,
	    ftype, total_ftypes,
	    &path_rtn, &total_path_rtns,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(status)
	{
	    const gchar *new_prog = (total_path_rtns > 0) ?
		path_rtn[0] : NULL;
	    if(!STRISEMPTY(new_prog))
	    {
		gtk_entry_set_text(entry, new_prog);
		gtk_entry_set_position(entry, -1);
	    }
	}

	/* Delete file types list */
	FileBrowserDeleteTypeList(ftype, total_ftypes);

	g_free(cur_prog);

	EDVRunDlgSetBusy(dlg, FALSE);
}

/*
 *	Run Dialog browse working directory callback.
 */
static void EDVRunDlgBrowseWorkingDirCB(
	GtkWidget *widget, gpointer data
)
{
	gboolean status;
	gchar *cur_dir;
	fb_type_struct **ftype = NULL, *ftype_rtn = NULL;
	gint total_ftypes = 0;
	gchar **path_rtn = NULL;
	gint total_path_rtns = 0;
	GtkWidget *toplevel, *w;
	GtkEntry *entry;
	edv_run_dlg_struct *dlg = EDV_RUN_DLG(data);
	if((dlg == NULL) || FileBrowserIsQuery())
	    return;

	toplevel = dlg->toplevel;
	w = dlg->working_dir_entry;
	if(w == NULL)
	    return;
	entry = GTK_ENTRY(w);

	cur_dir = STRDUP(gtk_entry_get_text(entry));


	EDVRunDlgSetBusy(dlg, TRUE);

	/* Create file types list */
	FileBrowserTypeListNew(
	    &ftype, &total_ftypes,
	    "*.*", "All files"
	);

	/* Query user for directory */
	FileBrowserSetTransientFor(toplevel);
	status = FileBrowserGetResponse(
	    "Select Directory",
	    "Select", "Cancel",
	    cur_dir,
	    ftype, total_ftypes,
	    &path_rtn, &total_path_rtns,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(status)
	{
	    const gchar *new_dir = (total_path_rtns > 0) ?
		path_rtn[0] : NULL;
	    if(!STRISEMPTY(new_dir))
	    {
		gtk_entry_set_text(entry, new_dir);
		gtk_entry_set_position(entry, -1);
	    }
	}

	/* Delete file types list */
	FileBrowserDeleteTypeList(ftype, total_ftypes);

	g_free(cur_dir);

	EDVRunDlgSetBusy(dlg, FALSE);
}

/*
 *	Run Dialog clear callback.
 *
 *      The data must be a GtkEntry.
 */
static void EDVRunDlgClearCB(
	GtkWidget *widget, gpointer data
)
{
	GtkWidget *w = (GtkWidget *)data;
	GtkEntry *entry;

	if((w != NULL) ? GTK_IS_ENTRY(w) : FALSE)
	    entry = GTK_ENTRY(w);
	else
	    return;

/*	EDVRunDlgSetBusy(dlg, TRUE); */

	gtk_entry_set_text(entry, "");
	gtk_entry_set_position(entry, 0);

/*	EDVRunDlgSetBusy(dlg, FALSE); */
}

/*
 *	Run dialog get working directory from current run command value
 *	callback.
 */
static void EDVRunDlgGetWorkingDirFromCmdCB(
	GtkWidget *widget, gpointer data
)
{
	const gchar *cmd;
	GtkWidget *w;
	GtkEntry *entry;
	GtkCombo *combo;
	edv_run_dlg_struct *dlg = EDV_RUN_DLG(data);
	if(dlg == NULL)
	    return;

	w = dlg->run_combo;
	if(w != NULL)
	    combo = GTK_COMBO(w);
	else
	    return;

	w = dlg->working_dir_entry;
	if(w != NULL)
	    entry = GTK_ENTRY(w);
	else
	    return;

	EDVRunDlgSetBusy(dlg, TRUE);

	cmd = gtk_entry_get_text(GTK_ENTRY(combo->entry));
	if(cmd != NULL)
	{
	    while(ISBLANK(*cmd))
		cmd++;

	    /* Run command starts with an absolute path? */
	    if(*cmd == DIR_DELIMINATOR)
	    {
		gchar *s, *new_working_dir = STRDUP(cmd);

		/* Seek strptr to first space (if any) in
		 * new_working_dir
		 */
		s = new_working_dir;
		while(!ISBLANK(*s) && (*s != '\0'))
		    s++;
		/* Cap new_working_dir at first blank character */
		*s = '\0';

		gtk_entry_set_text(
		    entry, GetParentDir(new_working_dir)
		);
		gtk_entry_set_position(entry, -1);

		g_free(new_working_dir);
	    }
	}

	EDVRunDlgSetBusy(dlg, FALSE);
}

/*
 *	Options button callback.
 */
static void EDVRunDlgOptionsCB(
	GtkWidget *widget, gpointer data
)
{
	edv_core_struct *core_ptr;
	edv_run_dlg_struct *dlg = EDV_RUN_DLG(data);
	if(dlg == NULL)
	    return;

	core_ptr = EDV_CORE(dlg->core_ptr);

	EDVRunDlgSetBusy(dlg, TRUE);

	EDVDoMapOptionsWinPage(
	    core_ptr,
	    "Programs",		/* Page Name */
	    dlg->toplevel	/* Toplevel */
	);

	EDVRunDlgSetBusy(dlg, FALSE);
}

/*
 *	Run dialog close button callback.
 */
static void EDVRunDlgCloseCB(
	GtkWidget *widget, gpointer data
)
{
	GtkWidget *w;
	edv_core_struct *core_ptr;
	cfg_item_struct *cfg_list;
	edv_run_dlg_struct *dlg = EDV_RUN_DLG(data);
	if(dlg == NULL)
	    return;

	core_ptr = EDV_CORE(dlg->core_ptr);
	cfg_list = (core_ptr != NULL) ? core_ptr->cfg_list : NULL;

	/* Begin recording current widget values to global
	 * configuration
	 */

	/* Run history */
	w = dlg->run_combo;
	if(w != NULL)
	{
	    EDVSaveFileGList(
		EDV_GET_S(EDV_CFG_PARM_FILE_RUNDLG_HISTORY),
		GUIComboGetList(w)
	    );
	}

	/* Toplevel geometry */
	w = dlg->toplevel;
	if(w != NULL)
	{
	    gint x = 0, y = 0;
	    GdkWindow *window = w->window;
	    if(window != NULL)
		gdk_window_get_root_origin(window, &x, &y);

	    CFGItemListSetValueI(
		cfg_list, EDV_CFG_PARM_RUNDLG_X,
		x, FALSE
	    );
	    CFGItemListSetValueI(
		cfg_list, EDV_CFG_PARM_RUNDLG_Y,
		y, FALSE
	    );
	}

	/* Last working directory */
	w = dlg->working_dir_entry;
	if(w != NULL)
	    CFGItemListSetValueS(
		cfg_list, EDV_CFG_PARM_RUNDLG_LAST_WORKING_DIR,
		gtk_entry_get_text(GTK_ENTRY(w)), FALSE
	    );

	/* Run in terminal */
	w = dlg->run_in_terminal_check;
	if(w != NULL)
	    CFGItemListSetValueI(
		cfg_list, EDV_CFG_PARM_RUNDLG_RUN_IN_TERMINAL,
		GTK_TOGGLE_BUTTON(w)->active, FALSE
	    );

	/* Keep dialog */
	w = dlg->keep_dialog_check;
	if(w != NULL)
	    CFGItemListSetValueI(
		cfg_list, EDV_CFG_PARM_RUNDLG_KEEP_DIALOG,
		GTK_TOGGLE_BUTTON(w)->active, FALSE
	    );


	/* Now close dialog */
	EDVRunDlgUnmap(dlg);
}


/*
 *	Completes the Run Dialog's command as needed and updates the
 *	command.
 */
static void EDVRunDlgCompleteCommand(edv_run_dlg_struct *dlg)
{
	const gchar *cmd_src;

	if(dlg == NULL)
	    return;

	/* Get current command */
	cmd_src = gtk_entry_get_text(
	    GTK_ENTRY(GTK_COMBO(dlg->run_combo)->entry)
	);   
	if(!STRISEMPTY(cmd_src))
	{
	    gchar *s, *prog, *args;

	    /* Seek past spaces in command */
	    while(ISBLANK(*cmd_src))
		cmd_src++;

	    /* Get program from command */
	    prog = STRDUP(cmd_src);
	    s = strpbrk(prog, " \t");
	    if(s != NULL)
		*s = '\0';   

	    /* Get arguments from command */
	    s = strpbrk(cmd_src, " \t");
	    if(s != NULL)
	    {
		while(ISBLANK(*s))
		    s++;
		args = STRDUP(s);
	    }
	    else
	    {
		args = NULL;
	    }

	    /* If the program location is not an absolute path then
	     * we need to complete it
	     */
	    s = ISPATHABSOLUTE(prog) ? NULL : EDVWhich(prog);
	    if(s != NULL)
	    {
		/* Set new program with completed path */
		g_free(prog);
		prog = s;

		/* Format and set new command */
		if(args != NULL)
		    s = g_strdup_printf(
			"%s %s",
			prog, args
		    );
		else
		    s = g_strdup_printf(
			"%s",
			prog 
		    );
		gtk_entry_set_text(
		    GTK_ENTRY(GTK_COMBO(dlg->run_combo)->entry),
		    s
		);
		g_free(s);
	    }

	    g_free(prog);
	    g_free(args);
	}
}


/*
 *	Sets the command value.
 */
void EDVRunDlgSetCommand(
	edv_run_dlg_struct *dlg, const char *command
)
{
	GtkEntry *entry;
	GtkCombo *combo;

	if(dlg == NULL)
	    return;

	combo = (GtkCombo *)dlg->run_combo;
	if(combo == NULL)
	    return;

	entry = GTK_ENTRY(combo->entry);
	gtk_entry_set_text(entry, (command != NULL) ? command : "");
	gtk_entry_set_position(entry, -1);
}

/*
 *	Sets the working directory value.
 */
void EDVRunDlgSetWorkingDir(
	edv_run_dlg_struct *dlg, const char *working_dir
)
{
	GtkEntry *entry;

	if(dlg == NULL)
	    return;

	entry = (GtkEntry *)dlg->working_dir_entry;
	if(entry == NULL)
	    return;

	gtk_entry_set_text(entry, (working_dir != NULL) ? working_dir : "");
	gtk_entry_set_position(entry, -1);
}

/*
 *	Regets list of past runned objects and updates the widgets to
 *	match the current configuration.
 */
void EDVRunDlgFetchValues(edv_run_dlg_struct *dlg)
{
	GtkCombo *combo;
	const cfg_item_struct *cfg_list;
	edv_core_struct *core_ptr;

	if(dlg == NULL)
	    return;

	core_ptr = EDV_CORE(dlg->core_ptr);
	if(core_ptr == NULL)
	    return;

	cfg_list = core_ptr->cfg_list;

	/* Begin fetching values from global configuration and
	 * updating the Run Dialog's widgets
	 */

	/* Run History */
	combo = (GtkCombo *)dlg->run_combo;
	if(combo != NULL)
	{
	    const gchar *path = EDV_GET_S(EDV_CFG_PARM_FILE_RUNDLG_HISTORY);
	    GList *glist = EDVOpenFileGList(path, 0);
	    if(glist != NULL)
	    {
		/* Set the new glist to the run combo's history, this
		 * glist will be managed by the call and should not
		 * be referenced afterwards
		 */
		GUIComboSetList(GTK_WIDGET(combo), glist);
		glist = NULL;
	    }
	}

	/* Run In Terminal */
	GTK_TOGGLE_BUTTON_SET_ACTIVE(
	    dlg->run_in_terminal_check,
	    EDV_GET_B(EDV_CFG_PARM_RUNDLG_RUN_IN_TERMINAL)
	);

	/* Keep Dialog */
	GTK_TOGGLE_BUTTON_SET_ACTIVE(
	    dlg->keep_dialog_check,
	    EDV_GET_B(EDV_CFG_PARM_RUNDLG_KEEP_DIALOG)
	);
}


/*
 *	Creates a new Run Dialog.
 */
edv_run_dlg_struct *EDVRunDlgNew(gpointer core_ptr)
{
	const gint	border_major = 5,
			border_minor = 2;
	gint x = 0, y = 0;
	GdkWindow *window;
	GtkAccelGroup *accelgrp;
	GtkStyle *style;
	gpointer combo_rtn;
	GtkWidget *w, *parent, *parent2;
	GtkCombo *combo;
	const cfg_item_struct *cfg_list = (core_ptr != NULL) ?
	    EDV_CORE(core_ptr)->cfg_list : NULL;
	edv_run_dlg_struct *dlg = EDV_RUN_DLG(
	    g_malloc0(sizeof(edv_run_dlg_struct))
	);
	if(dlg == NULL)
	    return(dlg);

	if(cfg_list != NULL)
	{
	    x = EDV_GET_I(EDV_CFG_PARM_RUNDLG_X);
	    y = EDV_GET_I(EDV_CFG_PARM_RUNDLG_Y);
	}

	dlg->accelgrp = accelgrp = gtk_accel_group_new();
	dlg->processing = FALSE;
	dlg->busy_count = 0;
	dlg->core_ptr = core_ptr;


	/* Begin creating widgets */

	/* Toplevel */
	dlg->toplevel = w = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_widget_set_usize(w, RUNDLG_WIDTH, RUNDLG_HEIGHT);
	gtk_widget_set_uposition(w, x, y);
	gtk_window_set_wmclass(
	    GTK_WINDOW(w), "dialog", PROG_NAME
	);
	gtk_widget_realize(w);
	gtk_window_set_title(GTK_WINDOW(w), RUNDLG_TITLE);
	style = gtk_widget_get_style(w);
	window = w->window;
	if(window != NULL)
	{
	    gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_TITLE | GDK_DECOR_MENU |
		GDK_DECOR_MINIMIZE
	    );
	    gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_MINIMIZE | GDK_FUNC_CLOSE
	    );
	    GUISetWMIcon(window, (guint8 **)icon_executable_48x48_xpm);
	}
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(EDVRunDlgDeleteEventCB), dlg
	);
	gtk_container_border_width(GTK_CONTAINER(w), 0);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	parent = w;


	/* Main vbox */
	dlg->main_vbox = w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;


	/* Hbox for run combo, browse button, and clear button */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	w = GUIComboCreate(
#if defined(PROG_LANGUAGE_SPANISH)
"La Orden:"
#elif defined(PROG_LANGUAGE_FRENCH)
"Ordre:"
#elif defined(PROG_LANGUAGE_GERMAN)
"Befehl:"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il Comando:"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bevel:"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Comando:"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kommando:"
#else
"Command:"
#endif
	    , "",
	    NULL,		/* Set list later */
	    20,			/* Maximum items */
	    &combo_rtn,
	    dlg,
	    EDVRunDlgRunCB,
	    NULL
	);
	dlg->run_combo = (GtkWidget *)combo_rtn;
	combo = (GtkCombo *)combo_rtn;
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);

	if(combo != NULL)
	{
	    w = combo->entry;
	    gtk_signal_connect(
		GTK_OBJECT(w), "focus_in_event",
	        GTK_SIGNAL_FUNC(EDVRunDlgCommandFocusEventCB), dlg
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "focus_out_event",
		GTK_SIGNAL_FUNC(EDVRunDlgCommandFocusEventCB), dlg
	    );
	    EDVEntrySetDND(EDV_CORE(core_ptr), w);
	    GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"Entre el nombre del programa ms cualquier argumento"
#elif defined(PROG_LANGUAGE_FRENCH)
"Entrer le nom du programme plus n'importe quels arguments"
#elif defined(PROG_LANGUAGE_GERMAN)
"Tragen Sie den Namen des Programms und irgendeine Argumente ein"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Entrare il nome del programma pi qualunque discussioni"
#elif defined(PROG_LANGUAGE_DUTCH)
"Ga de naam van het programma bovendien enig argumenten binnen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Entre o nome do programa com mais qualquer argumentos"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"G inn i navnet av programet pluss noen argumenter"
#else
"Enter the name of the program plus any arguments"
#endif
	    );
	}

	/* Browse button */
	dlg->browse_btn = w = GUIButtonPixmap(
	    (guint8 **)icon_browse_20x20_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	combo = (GtkCombo *)dlg->run_combo;
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVRunDlgBrowseCommandCB), dlg
	);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"Hojee"
#elif defined(PROG_LANGUAGE_FRENCH)
"Brouter"
#elif defined(PROG_LANGUAGE_GERMAN)
"Brausen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Curiosare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Grasduin"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Olhe"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Browse"
#else
"Browse"
#endif
	);
	gtk_widget_show(w);

	/* Clear button */
	dlg->clear_btn = w = GUIButtonPixmap(
	    (guint8 **)icon_clear_20x20_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVRunDlgClearCB), combo->entry
	);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"Claro"
#elif defined(PROG_LANGUAGE_FRENCH)
"Clair"
#elif defined(PROG_LANGUAGE_GERMAN)
"Klar"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Chiaro"
#elif defined(PROG_LANGUAGE_DUTCH)
"Helder"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Claro"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Klar"
#else
"Clear"
#endif
	);
	gtk_widget_show(w);


	/* Hbox for working directory entry and get from command
	 * button
	 */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Working Directory entry */
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Gua Trabajo:"
#elif defined(PROG_LANGUAGE_FRENCH)
"Annuaire Travaillant:"
#elif defined(PROG_LANGUAGE_GERMAN)
"Arbeitenden Verzeichnis:"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Che Lavorando L'Elenco:"
#elif defined(PROG_LANGUAGE_DUTCH)
"Werkende Gids:"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Que trabalhando Guia:"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Arbeidende Directory:"
#else
"Working Directory:"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	dlg->working_dir_entry = w = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_entry_set_text(
	    GTK_ENTRY(w),
	    EDV_GET_S(EDV_CFG_PARM_RUNDLG_LAST_WORKING_DIR)
	);
	EDVEntrySetDND(EDV_CORE(core_ptr), w);
	GUISetWidgetTip(
	    w,
#if defined(PROG_LANGUAGE_SPANISH)
"Entre la gua de trabajo que se usar cundo correr el programa"
#elif defined(PROG_LANGUAGE_FRENCH)
"Entrer l'annuaire qui travaillant qui sera utilis en courant le\
 programme"
#elif defined(PROG_LANGUAGE_GERMAN)
"Tragen Sie das arbeitende Verzeichnis, das benutzt werden wird,\
 beim Laufen des Programms ein"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Entrare il che lavorando elenco che sar usato quando correre il\
 programma"
#elif defined(PROG_LANGUAGE_DUTCH)
"Ga de werking gids die gebruikt worden zal wanneer lopen het\
 programma binnen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Entre o que trabalhando guia que ser usado quando correr o programa"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"G inn i den arbeidende katalogen som brukt ved kjre av av programet"
#else
"Enter the working directory that will be used when running the\
 program"
#endif
	);
	gtk_widget_show(w);

	/* Browse button */
	dlg->working_dir_browse_btn = w = GUIButtonPixmap(
	    (guint8 **)icon_browse_20x20_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	combo = (GtkCombo *)dlg->run_combo;
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVRunDlgBrowseWorkingDirCB), dlg
	);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"Hojee"
#elif defined(PROG_LANGUAGE_FRENCH)
"Brouter"
#elif defined(PROG_LANGUAGE_GERMAN)
"Brausen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Curiosare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Grasduin"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Olhe"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Browse"
#else
"Browse"
#endif
	);
	gtk_widget_show(w);

	/* Clear button */
	dlg->working_dir_clear_btn = w = GUIButtonPixmap(
	    (guint8 **)icon_clear_20x20_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVRunDlgClearCB), dlg->working_dir_entry
	);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"Claro"
#elif defined(PROG_LANGUAGE_FRENCH)
"Clair"
#elif defined(PROG_LANGUAGE_GERMAN)
"Klar"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Chiaro"
#elif defined(PROG_LANGUAGE_DUTCH)
"Helder"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Claro"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Klar"
#else
"Clear"
#endif
	);
	gtk_widget_show(w);

	/* Get from command button */
	dlg->working_dir_get_from_cmd_btn = w = GUIButtonPixmap(
	    (guint8 **)icon_run_dir_from_cmd_20x20_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVRunDlgGetWorkingDirFromCmdCB), dlg
	);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"Obtenga trabajando gua del valor actual de la orden"
#elif defined(PROG_LANGUAGE_FRENCH)
"Recevoir l'annuaire qui travaillant de la valeur d'ordre actuelle"
#elif defined(PROG_LANGUAGE_GERMAN)
"Erhalten Sie arbeitenden Verzeichnis vom jetzigen Befehl Wert"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Prendere che lavorando l'elenco dal valore di comando attuale"
#elif defined(PROG_LANGUAGE_DUTCH)
"Krijg werkende gids van de huidig bevel waarde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Fique trabalhando guia do valor atual de comando"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Bli arbeidende katalog fra den nvrendee kommandoverdi"
#else
"Get working directory from the current command value"
#endif
	);
	gtk_widget_show(w);


	/* Hbox for run in terminal and keep dialog checks */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	dlg->run_in_terminal_check = w = gtk_check_button_new_with_label(
#if defined(PROG_LANGUAGE_SPANISH)
"Corra En La Terminal"
#elif defined(PROG_LANGUAGE_FRENCH)
"Courir Dans Le Terminal"
#elif defined(PROG_LANGUAGE_GERMAN)
"Laufen Sie In Terminal"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Ha Correto Nell'Estremit"
#elif defined(PROG_LANGUAGE_DUTCH)
"Loop In Eindpunt"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Corra Em Terminal"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kjr In Terminal"
#else
"Run In Terminal"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"Corra el programa especificado en una terminal, esto a\
 menudo se necesita para programas basados de texto"
#elif defined(PROG_LANGUAGE_FRENCH)
"Courir le programme spcifi dans un terminal, ceci\
 est souvent eu besoin de pour les programmes texte-bass"
#elif defined(PROG_LANGUAGE_GERMAN)
"Laufen Sie das angegebene Programm in ein Terminal, dies ist often,\
 der fr Text basiert Programme gebraucht wird"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Ha correto il programma specificato in un'estremit, questo  often\
 ha avuto bisogno di per i programmi testo basati"
#elif defined(PROG_LANGUAGE_DUTCH)
"Loop het gespecificeerd programma in een eindpunt, dit is often nood\
 aan tekst gebaseerd programma's"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Corra o programa especificado num terminal, isto  often necessitou\
 para programas texto baseados"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kjr det spesifiserte programet i en terminal, dette er often som\
 trengt for tekst basert programer"
#else
"Check this to run the specified program in a terminal, this is needed\
 when running text-based programs"
#endif
	);
	gtk_widget_show(w);

	dlg->keep_dialog_check = w = gtk_check_button_new_with_label(
#if defined(PROG_LANGUAGE_SPANISH)
"Mantenga El Dilogo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Garder Le Dialogue"
#elif defined(PROG_LANGUAGE_GERMAN)
"Behalten Sie Dialog"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Tenere Il Dialogo"
#elif defined(PROG_LANGUAGE_DUTCH)
"Houd Dialoog Bij"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Mantenha Dialog"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Behold Dialog"
#else
"Keep Dialog"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"Mantenga este dilogo trazado despus de correr un programa"
#elif defined(PROG_LANGUAGE_FRENCH)
"Garder ce mapped de dialogue aprs avoir excutant un programme"
#elif defined(PROG_LANGUAGE_GERMAN)
"Behalten Sie diesen Dialog, der nach Laufen jedes Programms\
 aufgezeichnet wird"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Tenere questo mapped di dialogo dopo correre ogni programma"
#elif defined(PROG_LANGUAGE_DUTCH)
"Houd deze dialoog bracht in kaart na lopen van elk programma bij"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Mantenha este mapped de dialog depois de correr cada programa"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Behold denne dialogen som kartlagt etter kjring av hver program"
#else
"Check this to keep this dialog mapped after running each program"
#endif
	);
	gtk_widget_show(w);

	/* Options button */
	dlg->options_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_options2_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Las Opciones"
#elif defined(PROG_LANGUAGE_FRENCH)
"Options"
#elif defined(PROG_LANGUAGE_GERMAN)
"Optionen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Opzioni"
#elif defined(PROG_LANGUAGE_DUTCH)
"Opties"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Opes"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Alternativer"
#else
"Options"
#endif
	    , NULL
	);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_end(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVRunDlgOptionsCB), dlg
	);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"El clic para configurar los programas terminales"
#elif defined(PROG_LANGUAGE_FRENCH)
"Le dclic pour configurer des programmes dlimitants"
#elif defined(PROG_LANGUAGE_GERMAN)
"Klicken, letzte Programme zu gestalten"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Lo scatto di configurare i programmi terminali"
#elif defined(PROG_LANGUAGE_DUTCH)
"Klik Eindprogramma's te vormen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O estalido configurar programas terminais"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Klikk forme avsluttende programer"
#else
"Click to configure terminal programs"
#endif
	);
	gtk_widget_show(w);


	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Hbox for buttons */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Run button */
	dlg->run_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_run_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Corra"
#elif defined(PROG_LANGUAGE_FRENCH)
"Courir"
#elif defined(PROG_LANGUAGE_GERMAN)
"Lauf"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Correto"
#elif defined(PROG_LANGUAGE_DUTCH)
"Tocht"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Corrida"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Lp"
#else
"Run"
#endif
	    , NULL
	);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVRunDlgRunCB), dlg
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Return, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_widget_show(w);

	/* Close button */
	dlg->close_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_close_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Cierre"
#elif defined(PROG_LANGUAGE_FRENCH)
"Fin"
#elif defined(PROG_LANGUAGE_GERMAN)
"Nah"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Vicino"
#elif defined(PROG_LANGUAGE_DUTCH)
"Einde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Prximo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Nr"
#else
"Close"
#endif
	    , NULL
	);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVRunDlgCloseCB), dlg
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_widget_show(w);



	return(dlg);
}

/*
 *	Marks the Run Dialog as busy or ready.
 */
void EDVRunDlgSetBusy(edv_run_dlg_struct *dlg, gboolean is_busy)
{
	GdkCursor *cursor;
	GtkWidget *w;
	edv_core_struct *core_ptr;


	if(dlg == NULL)
	    return;

	core_ptr = EDV_CORE(dlg->core_ptr);
	if(core_ptr == NULL)
	    return;

	w = dlg->toplevel;
	if(w != NULL)
	{
	    if(is_busy)
	    {
		/* Increase busy count */
		dlg->busy_count++;

		/* If already busy then don't change anything */
		if(dlg->busy_count > 1)
		    return;

		cursor = EDVGetCursor(core_ptr, EDV_CURSOR_CODE_BUSY);
	    }
	    else
	    {
		/* Reduce busy count */
		dlg->busy_count--;
		if(dlg->busy_count < 0)
		    dlg->busy_count = 0;

		/* If still busy do not change anything */
		if(dlg->busy_count > 0)
		    return;

		cursor = NULL;  /* Use default cursor */
	    }

	    /* Update toplevel window's cursor */
	    if(w->window != NULL)
	    {
		gdk_window_set_cursor(w->window, cursor);
		gdk_flush();
	    }
	}
}

/*
 *	Checks if the Run Dialog is mapped.
 */
gboolean EDVRunDlgIsMapped(edv_run_dlg_struct *dlg)
{
	GtkWidget *w = (dlg != NULL) ? dlg->toplevel : NULL;
	return((w != NULL) ? GTK_WIDGET_MAPPED(w) : FALSE);
}

/*
 *	Maps the Run Dialog.
 */
void EDVRunDlgMap(edv_run_dlg_struct *dlg)
{
	GtkWidget *w = (dlg != NULL) ? dlg->toplevel : NULL;
	if(w == NULL)
	    return;

	gtk_widget_show_raise(w);

	/* Set the Run Combo as the default widget */
	w = dlg->run_combo;
	if(w != NULL)
	{
	    GtkCombo *combo = GTK_COMBO(w);
	    gtk_widget_grab_focus(combo->entry);
/*	    gtk_entry_set_position(GTK_ENTRY(combo->entry), 0); */
	}
}

/*
 *	Unmaps the Run Dialog.
 */
void EDVRunDlgUnmap(edv_run_dlg_struct *dlg)
{
	GtkWidget *w = (dlg != NULL) ? dlg->toplevel : NULL;
	if(w == NULL)
	    return;

	gtk_widget_hide(w);
}

/*
 *	Deletes the Run Dialog.
 */
void EDVRunDlgDelete(edv_run_dlg_struct *dlg)
{
	if(dlg == NULL)
	    return;

	/* Begin destroying widgets */
	GTK_WIDGET_DESTROY(dlg->run_combo)
	dlg->run_combo = NULL;
	GTK_WIDGET_DESTROY(dlg->browse_btn)
	dlg->browse_btn = NULL;
	GTK_WIDGET_DESTROY(dlg->clear_btn)
	dlg->clear_btn = NULL;

	GTK_WIDGET_DESTROY(dlg->working_dir_entry)
	dlg->working_dir_entry = NULL;
	GTK_WIDGET_DESTROY(dlg->working_dir_browse_btn)
	dlg->working_dir_browse_btn = NULL;
	GTK_WIDGET_DESTROY(dlg->working_dir_clear_btn)
	dlg->working_dir_clear_btn = NULL;
	GTK_WIDGET_DESTROY(dlg->working_dir_get_from_cmd_btn)
	dlg->working_dir_get_from_cmd_btn = NULL;

	GTK_WIDGET_DESTROY(dlg->run_in_terminal_check)
	dlg->run_in_terminal_check = NULL;
	GTK_WIDGET_DESTROY(dlg->keep_dialog_check)
	dlg->keep_dialog_check = NULL;
	GTK_WIDGET_DESTROY(dlg->options_btn)
	dlg->options_btn = NULL;

	GTK_WIDGET_DESTROY(dlg->run_btn)
	dlg->run_btn = NULL;
	GTK_WIDGET_DESTROY(dlg->close_btn)
	dlg->close_btn = NULL;

	GTK_WIDGET_DESTROY(dlg->toplevel)
	dlg->toplevel = NULL;

	GTK_ACCEL_GROUP_UNREF(dlg->accelgrp)
	dlg->accelgrp = NULL;

	g_free(dlg);
}
