/****************************************************************************/
/*                                                                          */
/*  Script to create a maintenance partition or a bootable CD               */
/*                                                                          */
/*  Written by Hayo Baan (hrbaan@xs4all.nl)                                 */
/*  http://www.xs4all.nl/~hrbaan                                            */
/*                                                                          */
/*  Please see booAble.html for instructions and version history            */
/*                                                                          */
/*  Version 3.2 (27/03/2004)                                                */
/*                                                                          */
/****************************************************************************/

'@echo off'

/* Make sure environment is local */
rc=setLocal()

/* Load the rexxUtil functions -------------------------------------------- */
call RxFuncAdd 'SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs'
call SysLoadFuncs

/* String variable to insert a newline in a character string -------------- */
CRLF=d2c(13)||d2c(10)

/* Verbose setting -------------------------------------------------------- */
CLEANUP='NO'
IGNOREOUTPUT=''
ARGUMENTS=ARG(1)
ARG1=translate(word(ARGUMENTS,1))
ARG2=word(ARGUMENTS,2)
VERBOSE=65
if (ARG1='/V' | ARG1='-V') then do
    if datatype(ARG2)='NUM' then do
        VERBOSE=ARG2
        ARGUMENTS=delword(ARGUMENTS,1,2)
    end
    else
        call error 'Usage: bootAble [/v <verbose>] [<config file>|/@<config-listfile>...]'
end

/* Create temp directory -------------------------------------------------- */
TEMPDIR=value('TEMP',,'OS2ENVIRONMENT')
if TEMPDIR='' then TEMPDIR=value('TMP',,'OS2ENVIRONMENT')
if right(TEMPDIR,1)<>'\' then TEMPDIR=TEMPDIR'\'
TEMPDIR=TEMPDIR'bootAble.$$$'
rc=createDir(TEMPDIR)
rc=cleanDir(TEMPDIR)
CLEANUP='YES'

/* Read the configuration file(s) ----------------------------------------- */
rc=readConfig(ARGUMENTS)

/* Determine if command output should be shown ---------------------------- */
select
    when bitMask(VERBOSE,128) then IGNOREOUTPUT=''
    when bitMask(VERBOSE,64)  then IGNOREOUTPUT='> /dev/nul'
    otherwise                      IGNOREOUTPUT='> /dev/nul 2>&1'
end


/****************************************************************************/
/**** Bootable CD creation **************************************************/
/****************************************************************************/

if isSet(CREATEBOOTCD) & isSet(CDRECORD) then do
    rc=createDir(WRITERDRIVE)
    if isSet(CDRECORDCLEANDIR) then
        rc=cleanDir(WRITERDRIVE)
end

if isSet(CREATEBOOTCD) & \isSet(CDRECORD) then do
    if isSet(RSJATTACHCD) then do
        call progress 1, 'Attaching CD to drive' WRITERDRIVE
        rc=runCMD(RSJCDATTACHEXE WRITERDRIVE, 'Failed to attach CD to drive' WRITERDRIVE)
    end
    if isSet(ERASECD) then do
        call progress 1, 'Erasing CD in drive' WRITERDRIVE
        rc=runCMD(RSJCDWFMTEXE WRITERDRIVE '/erase', 'Failed to erase CD in drive' WRITERDRIVE)
    end
end

rc=prepareImage(1)
rc=copyImageFiles(1)
rc=createConfigSys()

/* Bootmessages? */
if isSet(BOOTMESSAGES) then
    rc=createFile(VIRTUALDRIVE'\OS2\BOOT\ALTF2ON.$$$', '')

/* Create second disk image if creating a two disk boot */
if isSet(TWODISKBOOT) then do
    rc=writeImage(1)
    rc=prepareImage(2)
end

rc=copyImageFiles(2)

/* HPFS386 */
if isSet(HPFS386) then
    rc=createFile(VIRTUALDRIVE'\IBM386FS\HPFS386.INI', HPFS386INI)

if strip(STARTUPCMD)<>'' then
    rc=createFile(VIRTUALDRIVE'\STARTUP.CMD',parseText(STARTUPCMD))


if isSet(CREATEBOOTCD) & isSet(MEMDISK) then
    DESTDRIVE=VIRTUALDRIVE'\TOTAR.$$$'
else
    DESTDRIVE=VIRTUALDRIVE
rc=copyOtherFiles('IMG')
rc=createFiles('IMG')

if isSet(TWODISKBOOT) then
    rc=writeImage(2)
else
    rc=writeImage(1)

DESTDRIVE=WRITERDRIVE
rc=copyOtherFiles('CD')
rc=createFiles('CD')

call progress 1, 'Done creating' ifSet(CREATEBOOTCD,'bootable CD', 'Maintenance Partition' WRITERDRIVE)

if isSet(CREATEBOOTCD) & \isSet(CDRECORD) & isSet(AUTOBURNCD) then do
    call progress 1, 'Finalizing CD using RSJ'
    rc=runCMD(RSJCDATTACHEXE WRITERDRIVE '-s', 'Failed to finalize CD in drive' WRITERDRIVE)
end

if isSet(CREATEBOOTCD) & isSet(CDRECORD) & (isSet(CDRECORDBURNCMD) | isSet(AUTOBURNCD)) then do 
    CMD=CDRECORDMKISOFSEXE CDRECORDMKISOFSOPTIONS ifSet(MEMDISK,'-no-emul-boot -b BOOTIMGS/cdloader.bin','-b' BOOTIMAGENAME.1) WRITERDRIVE '|' CDRECORDEXE ifSet(ERASECD,CDRECORDERASEOPTION) CDRECORDEXEOPTIONS '-'
    if isSet(AUTOBURNCD) then do
        call progress 1, 'Burning CD using mkisofs and cdrecord'
        rc=runCMD(CMD, 'Failed to burn CD')
    end
    else do
        TXT='@echo off'||CRLF||,
            'REM Command to burn a bootAble CD using cdrecord'||CRLF||CRLF
        TXT=TXT||CMD||CRLF
        if fileExists('burnCD.cmd','Do not Search Path')<>'' then do
            rc=sysFileDelete('burnCD.cmd')
            if rc<>0 then
                call error 'Failed to remove existing burnCD.cmd ('rc')'
        end
        rc=createFile('burnCD.cmd',TXT)
    end
end
exit cleanUp()


/****************************************************************************/
/**** Functions *************************************************************/
/****************************************************************************/

/* Determine if an option is set ------------------------------------------ */
isSet: procedure
    return (left(translate(ARG(1)),1)='Y')


/* Inline version of the if clause ---------------------------------------- */
ifClause: procedure
    if (ARG(1)) then
        return ARG(2)
    else
        return ARG(3)


/* Combination of ifClause and isSet -------------------------------------- */
ifSet: procedure
    if (isSet(ARG(1))) then
        return ARG(2)
    else
        return ARG(3)


/* Sets a variable to a default if not already set ------------------------ */
defaultIfNotSet:
    VARNAME=ARG(1)
    VARVALUE=ARG(2)
    if translate(VARNAME)=value(VARNAME) then do
        rc=value(VARNAME,VARVALUE)
    end
    return value(VARNAME)


/* Return the text portion of an ini entry (i.e., strip of the 0 delimiter) */
getFromIni: procedure
    TXT=sysIni(ARG(1), ARG(2), ARG(3))
    if length(TXT)>0 & c2d(right(TXT,1))=0 then
        TXT=left(TXT,length(TXT)-1)
    return TXT


/* Return the remainder of the first config.sys line starting with argument */
/* Optionally an index number and a delimiter can be supplied, then only    */
/*   the indexed part of the string as delimited is reurned                 */
getFromConfigSys: procedure expose CONFIGSYSFILE
    return getFromFile(CONFIGSYSFILE, ARG(1), ARG(2), ARG(3))

/* Return the remainder of the first line of the specified file starting    */
/*   with argument                                                          */
/* Optionally an index number and a delimiter can be supplied, then only    */
/*   the indexed part of the string as delimited is reurned                 */
getFromFile: procedure
    ORGFILE=ARG(1)
    FILE=fileExists(ORGFILE)
    KEYTXT=translate(ARG(2))
    INDEXTXT=ARG(3)
    DELIMTXT=ARG(4)
    TXT=''
    if FILE='' then
        call error 'File' ORGFILE 'not found'
    do while lines(FILE)
        LINE=translate(space(linein(FILE),1))
        if abbrev(translate(LINE), KEYTXT) then do
            TXT=substr(LINE,length(KEYTXT)+1)
            leave
        end
    end
    rc=stream(FILE, 'C', 'CLOSE')
    if INDEXTXT<>'' then do
        if DELIMTXT<>'' then do 
            LENDELIMTXT=length(DELIMTXT)
            do I=1 to INDEXTXT-1
                POSTXT=pos(DELIMTXT, TXT)
                if POSTXT>0 then
                    TXT=substr(TXT,POSTXT+LENDELIMTXT)
                else do
                    TXT=''
                    leave
                end
            end
            POSTXT=pos(DELIMTXT, TXT)
            if POSTXT>0 then
                TXT=left(TXT,POSTXT-1)
        end
        else
            TXT=word(TXT,INDEXTXT)
    end
    return TXT


/* Return 'YES' if the config.sys file contains a line made up of the first */
/*   argument folowed by the second argument possibly prefixed by a path    */
configSysContains: procedure expose CONFIGSYSFILE
    return ifClause(configSysCount(ARG(1), ARG(2), 'NOCOUNT'),'YES','NO')


/* Return 'YES' if the specifid file contains a line made up of the first   */
/*   argument folowed by the second argument possibly prefixed by a path    */
fileContains: procedure
    return ifClause(fileCountMatches(ARG(1), ARG(2), ARG(3), 'NOCOUNT'),'YES','NO')


/* Return the number of times the config.sys file contains a line made up   */
/*   of the first argument folowed by the second argument possibly prefixed */
/*   by a path                                                              */
/* If the optional third argument is given, counting stops after the first  */
/*   occurence                                                              */ 
configSysCount: procedure expose CONFIGSYSFILE
    return fileCountMatches(CONFIGSYSFILE, ARG(1), ARG(2), ARG(3), 'EMPTY=0')


/* Return the number of times the specified file contains a line made up of */
/*   the first argument folowed by the second argument possibly prefixed by */
/*   a path                                                                 */
/* If the optional third argument is given, counting stops after the first  */
/*   occurence. The optional fourth argument specifies if an empty second   */
/*   argument is to cause 0 matches. The fifth optional argument determines */
/*   if a file not found is to cause 0 matches or fail with an error        */
fileCountMatches: procedure
    ORGFILE=ARG(1)
    FILE=fileExists(ORGFILE)
    KEYTXT=translate(ARG(2))
    FILETXT=translate(ARG(3))
    COUNT=ARG(4)=''
    if FILETXT='' & ARG(5)<>'' then
        return 0
    if FILE='' then
        if ARG(6)='' then
            call error 'File' ORGFILE 'not found'
        else
            return 0
    FOUND=0
    do while lines(FILE)
        LINE=translate(space(linein(FILE),1))
        if abbrev(LINE, KEYTXT) then do
            TXT=''
            if FILETXT<>'' then do
                LINE=substr(LINE,length(KEYTXT)+1)
                IDX=pos(' ',LINE)
                if IDX>0 then
                    TXT=left(LINE,IDX-1)
                else
                    TXT=LINE
                TXT=substr(TXT,lastpos('\',TXT)+1)
            end
            if TXT=FILETXT then do
                FOUND=FOUND+1
                if \COUNT then
                    leave
            end
        end
    end
    rc=stream(FILE, 'C', 'CLOSE')
    return FOUND


/* Return the remainder of the first config.sys line starting with the      */
/*   first argument folowed by the second argument possibly prefixed by a   */
/*   path                                                                   */
configSysOptions: procedure expose CONFIGSYSFILE
    return fileMatchRemainder(CONFIGSYSFILE, ARG(1), ARG(2))


/* Return the remainder of the first line in the specified file starting    */
/*   with the first argument folowed by the second argument possibly        */
/*   prefixed by a path                                                     */
/*   The third optional argument determines if a file not found is to cause */
/*   an empty string to be returned or fail with an error                   */
fileMatchRemainder: procedure
    ORGFILE=ARG(1)
    FILE=fileExists(ORGFILE)
    KEYTXT=translate(ARG(2))
    FILETXT=translate(ARG(3))
    if FILETXT='' then return ''
    if FILE='' then
        if ARG(4)='' then
            call error 'File' ORGFILE 'not found'
        else
            return 0
    OPTIONTXT=''
    do while lines(FILE)
        LINE=translate(space(linein(FILE),1))
        if abbrev(LINE, KEYTXT) then do
            TXT=''
            LINE=substr(LINE,length(KEYTXT)+1)
            IDX=pos(' ',LINE)
            if IDX>0 then
                TXT=left(LINE,IDX-1)
            else
                TXT=LINE
            IDX=lastpos('\',TXT)+1
            TXT=substr(TXT,IDX)
            if TXT=FILETXT then do
                OPTIONTXT=strip(substr(LINE,IDX+length(FILETXT)))
                leave
            end
        end
    end
    rc=stream(FILE, 'C', 'CLOSE')
    return OPTIONTXT


/* Returns the file argument if it exists, an empty string if not --------- */
/* If the second argument is not specified, the SEARCHPATH is used when the */
/*   file can not be found on the specified location                        */
/* Can be used in configuration files to add optional files                 */
fileExists: procedure expose SEARCHPATH
    FILESPEC=stream(ARG(1), 'C', 'QUERY EXISTS')
    if FILESPEC='' & ARG(2)='' & SEARCHPATH<>'' & pos('*', ARG(1))=0 & pos('?', ARG(1))=0 then
        FILESPEC=sysSearchPath('SEARCHPATH', fileSpec('NAME', ARG(1)))
    return FILESPEC


/* Returns the first file argument if it exists, the second if not -------- */
ifExists: procedure expose SEARCHPATH
    FILESPEC=fileExists(ARG(1))
    if FILESPEC<>'' then
        return FILESPEC
    else
        return ARG(2)


/* Returns the 1 if the directory exists, 0 if not  ----------------------- */
dirExists: procedure
    rc=sysFileTree(ARG(1), 'DIRS', 'D')
    return (rc=0 & DIRS.0=1)


/* Return file name portion of a filespec --------------------------------- */
fileName: procedure
    if ARG(2) <> '' then do
        FS=filespec('NAME', ARG(1))
        FSDOT=pos('.', FS)
        if FSDOT>0 then do
            FSNAME=left(FS,min(FSDOT-1,8))
            FSEXT=right(FS,length(FS)-FSDOT)
        end
        else do
            FSNAME=left(FS,min(length(FS),8))
            FSEXT=''
        end
        return FSNAME||ifClause(FSDOT>0,'.')||FSEXT
    end
    else
        return filespec('NAME', ARG(1))


/* Return directory portion of a filespec --------------------------------- */
directoryName: procedure
    return filespec('PATH', ARG(1))


/* Return drive portion of a filespec ------------------------------------- */
driveName: procedure
    return filespec('DRIVE', ARG(1))


/* Return path of a filespec (drive and directory) ------------------------ */
pathName: procedure
    return driveName(ARG(1))||directoryName(ARG(1))


/* Return drive letter based on destination ------------------------------- */
destinationDrive: procedure expose CREATEBOOTCD MEMDISK MEMDISKDRIVE WRITERDRIVE CDROMDRIVE
    DESTINATION=ARG(1)
    if isSet(CREATEBOOTCD) then
        if DESTINATION='CD' then
            return CDROMDRIVE
        else
            return ifSet(MEMDISK,MEMDISKDRIVE,'A:')
    return WRITERDRIVE
    

/* Return the value of a variable or an epmty string of not set ----------- */
varValue:
    if translate(arg(1))<>value(arg(1)) then
        return value(arg(1))
    else
        return '' 


/* Return the UNICODE keyboard layout based on the KEYB layout ------------ */
unicodeKeyboard: procedure
    KEYB=translate(ARG(1))
    UNIKEYB=''
    select
        when KEYB='AR'    then UNIKEYB='AA'      /* Arabic */
        when KEYB='AR470' then UNIKEYB='AA470'   /* Arabic 101 */
        when KEYB='BA'    then UNIKEYB=KEYB      /* Bosnian */
        when KEYB='BE'    then UNIKEYB=KEYB      /* Belgium */
        when KEYB='BG'    then UNIKEYB=KEYB      /* Bulgaria */
        when KEYB='BG241' then UNIKEYB=KEYB      /* Bulgaria 241 */
        when KEYB='BR'    then UNIKEYB=KEYB      /* Brazil */
        when KEYB='BR274' then UNIKEYB=KEYB      /* Brazil 101 */
        when KEYB='BY'    then UNIKEYB=KEYB      /* Belarus */
        when KEYB='CA'    then UNIKEYB=KEYB      /* Canada */
        when KEYB='CF'    then UNIKEYB=KEYB      /* Canada French */
        when KEYB='CZ'    then UNIKEYB=KEYB      /* Czech Republic */
        when KEYB='DE453' then UNIKEYB=KEYB      /* Germany DIN */
        when KEYB='DK'    then UNIKEYB=KEYB      /* Denmark */
        when KEYB='EE'    then UNIKEYB=KEYB      /* Estonia */
        when KEYB='FR'    then UNIKEYB=KEYB      /* France */
        when KEYB='GK'    then UNIKEYB='EL'      /* Greek */
        when KEYB='GK220' then UNIKEYB='EL220'   /* Greek 101 */
        when KEYB='GK459' then UNIKEYB='EL459'   /* Greek 459 */
        when KEYB='GR'    then UNIKEYB='DE'      /* Germany */
        when KEYB='HE'    then UNIKEYB=''        /* Hebrew */
        when KEYB='HR'    then UNIKEYB=KEYB      /* Croatia */
        when KEYB='HU'    then UNIKEYB=KEYB      /* Hungary */
        when KEYB='IS'    then UNIKEYB=KEYB      /* Iceland */
        when KEYB='IS458' then UNIKEYB=KEYB      /* Iceland */
        when KEYB='IT'    then UNIKEYB=KEYB      /* Italy */
        when KEYB='IT142' then UNIKEYB=KEYB      /* Italy 142 */
        when KEYB='LA'    then UNIKEYB=KEYB      /* Latin America */
        when KEYB='LT'    then UNIKEYB=KEYB      /* Lithuania */
        when KEYB='LV'    then UNIKEYB=KEYB      /* Latvia */
        when KEYB='MK'    then UNIKEYB=KEYB      /* FYR Macedonia */
        when KEYB='NL'    then UNIKEYB=KEYB      /* Netherlands */
        when KEYB='NO'    then UNIKEYB=KEYB      /* Norway */
        when KEYB='PL'    then UNIKEYB=KEYB      /* Poland */
        when KEYB='PL457' then UNIKEYB=KEYB      /* Poland Pgmr */
        when KEYB='PO'    then UNIKEYB=KEYB      /* Portugal */
        when KEYB='RO'    then UNIKEYB=KEYB      /* Romania */
        when KEYB='RU'    then UNIKEYB=KEYB      /* Russia */
        when KEYB='RU441' then UNIKEYB=KEYB      /* Russia 441 */
        when KEYB='SF'    then UNIKEYB=KEYB      /* Swiss Frech */
        when KEYB='SG'    then UNIKEYB='SD'      /* Swiss German */
        when KEYB='SK'    then UNIKEYB=KEYB      /* Slovakia */
        when KEYB='SL'    then UNIKEYB=KEYB      /* Slovenia */
        when KEYB='SP'    then UNIKEYB='ES'      /* Spain */
        when KEYB='SQ'    then UNIKEYB=KEYB      /* Albania */
        when KEYB='SR'    then UNIKEYB=KEYB      /* Serbia */
        when KEYB='SU'    then UNIKEYB='FI'      /* Finland */
        when KEYB='SV'    then UNIKEYB=KEYB      /* Sweden */
        when KEYB='TH'    then UNIKEYB=KEYB      /* Thailand */
        when KEYB='TH190' then UNIKEYB=KEYB      /* Thailand 190 */
        when KEYB='TR'    then UNIKEYB=KEYB      /* Turkey */
        when KEYB='TR440' then UNIKEYB=KEYB      /* Turkey 440 */
        when KEYB='TW'    then UNIKEYB=KEYB      /* Taiwan */
        when KEYB='UK'    then UNIKEYB=KEYB      /* United Kingdom */
        when KEYB='UK168' then UNIKEYB=KEYB      /* United Kingdom 168 */
        when KEYB='US'    then UNIKEYB=KEYB      /* United States */
        when KEYB='USDV'  then UNIKEYB=KEYB      /* US Dvorak */
        when KEYB='USDVL' then UNIKEYB=KEYB      /* US Dvorak Left Handed */
        when KEYB='USDVR' then UNIKEYB=KEYB      /* US Dvorak Right Handed */
        when KEYB='UX'    then UNIKEYB='USINTER' /* US International */
        otherwise              UNIKEYB=''        /* 'IL' 'JP' 'KR' 'UA' */
    end
    if UNIKEYB='' then
        call error 'Unknown keyboard layout' KEYB', specify UNIKEYB manually'
    return UNIKEYB'.kbl'


/* Return the drive letter of the CDROM ------------------------------------ */
cdromDriveLetter: procedure expose RESERVEDRIVELETTER CLEANUP TEMPDIR VERBOSE IGNOREOUTPUT
    if RESERVEDRIVELETTER<>'' then
        return d2c(c2d(RESERVEDRIVELETTER)+1)':'
    DRIVEMAP=sysDriveMap(,'LOCAL')
    do I=1 to words(DRIVEMAP)
        DRIVE=word(DRIVEMAP,I)
        DRIVEFS=sysFileSystemType(DRIVE)
        if DRIVEFS='' | DRIVEFS='CDFS' then
            return DRIVE
    end
    call error 'No CDROM drive found, specify CDROMDRIVE manually'

/* Return the writer drive path ------------------------------------------- */
writerDrivePath: procedure expose CREATEBOOTCD CDRECORD RSJATTACHCD CLEANUP TEMPDIR VERBOSE IGNOREOUTPUT
    if isSet(CREATEBOOTCD) then do
        if isSet(CDRECORD) then do
            DRIVE=left(TEMPDIR,lastPos('\',TEMPDIR))'bootAble.IMG'
            return DRIVE
        end
        if isSet(RSJATTACHCD) then do
            DRIVEMAP=sysDriveMap(,'FREE')
            DRIVE=word(DRIVEMAP,1)
            if DRIVE='' then
                call error 'No available drives to attach CD-R(W)'
            return DRIVE
        end
        DRIVEMAP=sysDriveMap(,'REMOTE')
        do I=1 to words(DRIVEMAP)
            DRIVE=word(DRIVEMAP,I)
            if sysFileSystemType(DRIVE)='CDWFS' then
                return DRIVE
        end
        call error 'No attached CD-R(W) found'
    end
    DRIVE='M:'
    if sysFileSystemType(DRIVE)='' then
        call error 'Default maintenance partition' DRIVE 'not found, specify WRITERDRIVE manually'
    return DRIVE


/* Return the name of the boot image -------------------------------------- */
bootImageName: procedure expose CREATEBOOTCD TWODISKBOOT IMAGENR WRITERDRIVE
    return ifSet(CREATEBOOTCD,'Boot Image' ifSet(TWODISKBOOT,IMAGENR),'Maintenance Partition' WRITERDRIVE)


/* Return the name of the destination ------------------------------------- */
destinationName: procedure expose CREATEBOOTCD TWODISKBOOT IMAGENR WRITERDRIVE DEST
    return ifClause(isSet(CREATEBOOTCD) & DEST='CD','CD',bootImageName())


/* Return true if the values have a matching bitmask ---------------------- */
bitMask: procedure
    if datatype(ARG(1))<>'NUM' then
        return 0
    else
        return c2d(bitAnd(d2c(ARG(1)),d2c(ARG(2))))>0


/* Run a command checking return code ------------------------------------- */
runCMD: procedure expose CLEANUP TEMPDIR VERBOSE IGNOREOUTPUT
    CMD=ARG(1)
    if ARG(2)<>'' then
        ERROR=ARG(2)
    else
        ERROR='Failed to run' ARG(1)
    CMD IGNOREOUTPUT
    if rc<>0 then call error ERROR '(rc='rc')'
    return 0

/* Display progress when in given verbose mode ---------------------------- */
progress: procedure expose VERBOSE
    if bitMask(VERBOSE, ARG(1)) then say ARG(2)
    return rc


/* Write an error message to stderr and exit program ---------------------- */
error: procedure expose CLEANUP TEMPDIR VERBOSE IGNOREOUTPUT
    call errorMessage ARG(1)
    rc=cleanUp()
    exit 1


/* Write an error message to stderr --------------------------------------- */
errorMessage: procedure
    call beep 440,150
    call lineout 'STDERR', 'ERROR:' ARG(1)
    return 0


/* Write a warning message to stderr -------------------------------------- */
warningMessage: procedure
    call lineout 'STDERR', 'WARNING:' ARG(1)
    return 0


/* Cleanup ---------------------------------------------------------------- */
cleanUp: procedure expose CLEANUP TEMPDIR VERBOSE IGNOREOUTPUT
    rc=endLocal()
    if \isSet(CLEANUP) then return 0
    rc=cleanDir(TEMPDIR,'YES')
    rc=sysRmDir(TEMPDIR)
    if rc<>0 then
        call errorMessage 'Failed to remove temp directory' TEMPDIR '('rc')'
    return rc


/* Clean a directory tree ------------------------------------------------- */
cleanDir: procedure expose CLEANUP TEMPDIR VERBOSE IGNOREOUTPUT
    call progress 2, 'Cleaning up directory' ARG(1)
    /* First delete all files */
    rc=sysFileTree(ARG(1)'\*', 'FILETREE', 'FOS',, '-*---')
    do I=1 to FILETREE.0 while rc=0
        rc=sysFileDelete(FILETREE.I)
    end
    if rc<>0 then
        ERR='Failed to remove files from directory' ARG(1) '('rc')'
    /* Then delete all directories */
    if rc=0 then do
        drop FILETREE.
        rc=sysFileTree(ARG(1)'\*', 'FILETREE', 'DOS',, '-*---')
        do I=FILETREE.0 to 1 by -1 while rc=0
            rc=sysRmDir(FILETREE.I)
        end
        if rc<>0 then
            ERR='Failed to remove directory structure' ARG(1) '('rc')'
    end
    if rc<>0 then
        if isSet(ARG(2)) then
            call errorMessage ERR
        else
            call error ERR
    return rc


/* Parse text argument (replacing special $$ and %% constructs) ----------- */
parseText:
    PARSETXT=ARG(1)

    PARSECONSTRUCT.0=4
    PARSECONSTRUCT.1='$$'
    PARSECONSTRUCT.2='%%'
    PARSECONSTRUCT.3='$$'
    PARSECONSTRUCT.4='%%'

    do I=1 to PARSECONSTRUCT.0
        STARTPOS=1
        do until STARTPOS=0
            STARTPOS=pos(PARSECONSTRUCT.I, PARSETXT, STARTPOS)
            if (STARTPOS<>0) then do
                ENDPOS=pos(PARSECONSTRUCT.I, PARSETXT, STARTPOS+2)
                if ENDPOS<>0 then do
                    LEFTPARSETXT=left(PARSETXT,STARTPOS-1)
                    RIGHTPARSETXT=substr(PARSETXT,ENDPOS+2)
                    MIDPARSETXT=substr(PARSETXT,STARTPOS+2,ENDPOS-STARTPOS-2)
                    if (I=2|I=4) then do
                        if left(MIDPARSETXT,1)<>'!' & left(MIDPARSETXT,1)<>'~' then
                            VALUEMIDPARSETXT=ifSet(value(MIDPARSETXT),'Y','N')
                        else
                            VALUEMIDPARSETXT=ifSet(value(substr(MIDPARSETXT,2)),'N','Y')
                        MIDPARSETXT=''
                    end
                    else do
                        if left(MIDPARSETXT,1)<>'@' then
                            MIDPARSETXT=value(MIDPARSETXT)
                        else do
                            interpret 'MIDPARSETXT='substr(MIDPARSETXT,2)
                        end
                    end
                    if (I=2|I=4) & \isSet(VALUEMIDPARSETXT) then do
                        CRLFPOSBEFORE=lastpos(CRLF,left(PARSETXT,STARTPOS))
                        CRLFPOSAFTER=pos(CRLF,PARSETXT,ENDPOS)
                        if CRLFPOSAFTER=0 then CRLFPOSAFTER=length(PARSETEXT)
                        if CRLFPOSBEFORE=0 then do
                            PARSETXT=substr(PARSETXT,CRLFPOSAFTER+2)
                            ENDPOS=1
                        end
                        else do
                            PARSETXT=left(PARSETXT,CRLFPOSBEFORE+1)||substr(PARSETXT,CRLFPOSAFTER+2)
                            ENDPOS=CRLFPOSBEFORE+1
                        end
                    end
                    else do
                        PARSETXT=LEFTPARSETXT||MIDPARSETXT||RIGHTPARSETXT
                        ENDPOS=STARTPOS+length(MIDPARSETXT)
                    end
                end
                STARTPOS=ENDPOS
            end
        end
    end
    return PARSETXT


/* Create a file with given content --------------------------------------- */
createFile: procedure expose CLEANUP TEMPDIR VERBOSE IGNOREOUTPUT
    FILE=ARG(1)
    TXT=ARG(2)
    call progress 2, 'Creating file' FILE
    rc=charout(FILE,TXT)
    if rc=0 then
        rc=charout(FILE)
    if rc<>0 then
        call error 'Failed to create' FILE
    return 0


/* Parse a list of files into FILELIST variable ---------------------------- */
parseFileList: procedure expose FILELIST.
    FILE=strip(ARG(1))
    I=0
    do while length(FILE)>0
        if left(FILE,2)='/@' then do
            LISTFILE='X'
            FILE=strip(substr(FILE,3))
        end
        else
            LISTFILE=''
        LFT=left(FILE,1)
        if LFT='"' then
            FILE=substr(FILE,2)
        else
            LFT=' '
        IDX=pos(LFT,FILE)
        I=I+1
        if IDX=0 then do
            FILELIST.I=FILE
            FILE=''
        end
        else do
            FILELIST.I=left(FILE,IDX-1)
            FILE=strip(right(FILE,length(FILE)-IDX))
        end
        IDX=1
        do until IDX=0
            IDX=pos('\\',FILELIST.I)
            if (IDX>0) then do
                FILELIST.I=left(FILELIST.I,IDX-1)||substr(FILELIST.I,IDX+1)
            end
        end
        if LISTFILE<>'' then do
            LISTFILE=fileExists(FILELIST.I)
            if LISTFILE='' then call error 'List file' FILELIST.I 'not found'
            LISTFILES=''
            do while lines(LISTFILE)
                LISTFILES=LISTFILES linein(LISTFILE)
            end
            rc=stream(LISTFILE, 'C', 'CLOSE')
            if rc<>'READY:' then call error 'Failed to close' LISTFILE '('rc')'
            I=I-1
            FILE=strip(LISTFILES FILE)
        end
    end
    FILELIST.0=I
    return 0


/* Copy a list of files --------------------------------------------------- */
copyFiles: procedure expose SEARCHPATH CLEANUP TEMPDIR VERBOSE IGNOREOUTPUT MEMDISK VIRTUALDRIVE REMOVEROFLAG LXLITE LXLITECOMMAND LXLITEOPTIONS
    DESTINATION=ARG(2)
    if right(DESTINATION,1)<>'\' then DESTINATION=DESTINATION'\'
    TRUNCATE=ARG(3)
    IGNORENOTFOUND=ARG(4)
    call progress 2, 'Copying files to' DESTINATION
    rc=parseFileList(ARG(1))
    do I=1 to FILELIST.0
        drop FILETREE.
        rc=sysFileTree(FILELIST.I, 'FILETREE', 'F')
        if rc=0 then do
            if FILETREE.0=0 & IGNORENOTFOUND='' then do
                /* If not found at specified location, search additional path */
                SOURCEFILE=fileExists(FILELIST.I)
                if SOURCEFILE<>'' then do
                    drop FILETREE.
                    rc=sysFileTree(SOURCEFILE, 'FILETREE', 'F')
                end
            end  
            if rc=0 then do 
                if FILETREE.0=0 & IGNORENOTFOUND='' then
                    call error 'File not found' FILELIST.I
                do II=1 to FILETREE.0
                    rc=copyFile(FILETREE.II, DESTINATION, TRUNCATE)
                end II
            end
        end
        if rc<>0 then
            call error 'Not enough memory copying' FILELIST.I
    end I
    return 0


/* Copy a file (removing extended attributes when copying to boot image) -- */
copyFile: procedure expose CLEANUP TEMPDIR VERBOSE IGNOREOUTPUT MEMDISK VIRTUALDRIVE REMOVEROFLAG LXLITE LXLITECOMMAND LXLITEOPTIONS
    parse value ARG(1) with dummy dummy dummy ATTRIBUTES SOURCEFILESPEC
    ATTRIBUTES=space(translate(ATTRIBUTES, ' ', '-'),0)
    SOURCEFILESPEC=strip(SOURCEFILESPEC)
    SOURCEDIR=fileSpec('PATH', SOURCEFILESPEC)
    SOURCEFILE=fileName(SOURCEFILESPEC)
    DESTDIR=ARG(2)
    DESTFILE=fileName(SOURCEFILE, ARG(3))
    COMPLEXCOPY=isSet(LXLITE) | SOURCEFILE<>DESTFILE |,
        (\isSet(MEMDISK) & abbrev(translate(DESTDIR||DESTFILE),translate(VIRTUALDRIVE))) |,
        (isSet(MEMDISK)  & abbrev(translate(DESTDIR||DESTFILE),translate(VIRTUALDRIVE'\TOTAR.$$$')))

    if fileExists(DESTDIR||DESTFILE, 'T')<>'' then do
        call progress 2, 'Ignored' left(ATTRIBUTES,4) SOURCEFILESPEC '(file already copied)'
        return 0
    end
    call progress 2, 'Copying' left(ATTRIBUTES,4) SOURCEFILESPEC '=>' DESTDIR||DESTFILE

    if \isSet(REMOVEROFLAG) | pos('H',ATTRIBUTES)>0 | pos('S',ATTRIBUTES)>0 then
        XCOPYROFLAG='/r'
    else do
        ATTRIBUTES=space(translate(ATTRIBUTES, ' ', 'R'))
        XCOPYROFLAG=''
    end
    if COMPLEXCOPY then
        'xcopy /h/t'XCOPYROFLAG '"'SOURCEFILESPEC'"' TEMPDIR IGNOREOUTPUT  
    else    
        'xcopy /h/t'XCOPYROFLAG '"'SOURCEFILESPEC'"' DESTDIR IGNOREOUTPUT
    select
        when rc=1 then ERR='File not found'
        when rc=2 then ERR='File errors'
        when rc=3 then ERR='Interrupted'
        when rc=4 then ERR='Not enough room for file'
        otherwise ERR=''
    end
    if rc=0 & COMPLEXCOPY then do
        'attrib -a -h -r -s' '"'TEMPDIR'\'SOURCEFILE'"' IGNOREOUTPUT
        if rc<>0 then ERR='Resetting attributes failed ('rc')'
    end
    if rc=0 & COMPLEXCOPY then do
        'eautil /s /r' '"'TEMPDIR'\'SOURCEFILE'"' TEMPDIR'\eautil.$$$' IGNOREOUTPUT
        if rc<>0 then ERR='Removing extended attributes failed'
    end
    if rc=0 & isSet(LXLITE) then do
        LXLITECOMMAND LXLITEOPTIONS TEMPDIR'\'SOURCEFILE IGNOREOUTPUT
        if rc<>0 then ERR='Lxlite failed ('rc')'
    end
    if rc=0 & COMPLEXCOPY then do
        'copy' '"'TEMPDIR'\'SOURCEFILE'"' '"'DESTDIR||DESTFILE'"' IGNOREOUTPUT
        if rc<>0 then ERR='Copy to' DESTDIR||DESTFILE 'failed ('rc')'
    end
    if rc=0 & COMPLEXCOPY then do
        'del' '"'TEMPDIR'\'SOURCEFILE'"' IGNOREOUTPUT
        if rc<>0 then ERR='Deletion of' TEMPDIR'\'SOURCEFILE 'failed ('rc')'
    end
    if rc=0 & ATTRIBUTES<>'' & COMPLEXCOPY then do
        do I=1 to length(ATTRIBUTES)
            SETATTRIBUTES=' +'||substr(ATTRIBUTES,I,1)
        end
        'attrib' SETATTRIBUTES '"'DESTDIR||DESTFILE'"' IGNOREOUTPUT
        if rc<>0 then ERR='Setting' ATTRIBUTES 'attributes failed ('rc')'
    end
    if rc<>0 then
        call error 'Error copying' SOURCEFILESPEC||':' ERR
    return rc


/* Simple copy and rename file -------------------------------------------- */
sCopyFile: procedure expose CLEANUP TEMPDIR VERBOSE IGNOREOUTPUT
    SOURCEFILESPEC=strip(ARG(1))
    DESTFILESPEC=strip(ARG(2))
    SOURCEFILESPEC=ifExists(SOURCEFILESPEC,SOURCEFILESPEC)

    call progress 2, 'sCopying' SOURCEFILESPEC '=>' DESTFILESPEC
    rc=runCMD('copy' '"'SOURCEFILESPEC'"' '"'DESTFILESPEC'"', 'Error copying' SOURCEFILESPEC 'to' DESTFILESPEC)
    return rc


/* Create a directory ----------------------------------------------------- */
createDir: procedure expose CLEANUP TEMPDIR VERBOSE IGNOREOUTPUT
    DIRNAME=ARG(1)
    if right(DIRNAME,1)='\' then
        DIRNAME=left(DIRNAME,length(DIRNAME)-1)

    /* Don't create it if it already exists */
    if right(DIRNAME,1)=':' | dirExists(DIRNAME) then
        return 0

    /* Make sure all higher directory levels are created */
    IDX=lastpos('\',DIRNAME)
    if IDX>3 then rc=createDir(left(DIRNAME,IDX-1))

    /* Create directory */
    call progress 2, 'Creating directory' DIRNAME
    rc=sysMkDir(DIRNAME)
    select
        when rc=2   then ERR='File not found'
        when rc=3   then ERR='Path not found'
        when rc=5   then ERR='Access denied'
        when rc=26  then ERR='Not a DOS disk'
        when rc=87  then ERR='Invalid parameter'
        when rc=108 then ERR='Drive locked'
        when rc=206 then ERR='Filename exceeds range'
        otherwise ERR=''
    end
    if rc<>0 then
        call error 'Error creating directory' DIRNAME||':' ERR
    return rc


/* Write a line to the config.sys file ------------------------------------ */
configLine: procedure expose VIRTUALDRIVE
    rc=lineout(VIRTUALDRIVE'\config.sys',ARG(1))
    return rc


/* Read and parse the configuration file(s) ------------------------------- */
readConfig:
    /* Global initialisation */
    OSBOOTDRIVE=sysBootDrive()
    CONFIGSYSFILE=OSBOOTDRIVE'\config.sys'
    EXTRADEVICE=0
    EXTRABASEDEV=0
    EXTRADIRECTORIES=0
    EXTRACREATEFILES=0

    EXTRAUSERINIRCFILES=1
    EXTRASYSINIRCFILES=1

    SEARCHPATH=OSBOOTDRIVE'\os2\boot;'getFromConfigSys('LIBPATH=')';'value('PATH',,'OS2ENVIRONMENT')';'value('DPATH',,'OS2ENVIRONMENT')';'value('HELP',,'OS2ENVIRONMENT')';'value('BOOKSHELF',,'OS2ENVIRONMENT')';'OSBOOTDRIVE'\PSFONTS;'
    rc=value('SEARCHPATH',SEARCHPATH,'OS2ENVIRONMENT')
    PREVSEARCHPATH=SEARCHPATH

    CREATECLONECDCMDLINEARGS=''
    rc=parseFileList(ARGUMENTS)
    /* Process all specified configuration files */
    do I=1 to FILELIST.0
        CREATECLONECDCMDLINEARGS=CREATECLONECDCMDLINEARGS||' "'FILELIST.I'"'
        rc=readConfigFile(FILELIST.I)
    end
    /* Process the defaults */
    rc=readConfigFile('bootAble.cfg', 'YES')

    /* Make sure the RESERVEDRIVELETTER has no trailing ':' */
    RESERVEDRIVELETTER=left(RESERVEDRIVELETTER,1)

    /* Add optional EXIT to startup.cmd file */
    if STARTUPCMD<>'' & isSet(STARTUPCMDEXIT) then
        STARTUPCMD=STARTUPCMD||CRLF||'EXIT'

    /* Enforce dependant settings */
    if isSet(JJSCDROM) then
        CDROM='YES'
    if isSet(DANIIDE) then
        IDE='YES'
    if isSet(DANIATAPI) then
        ATAPI='YES'
    if isSet(JFS) then
        UNICODE='YES'

    /* Force destination settings */
    OS2APPSDESTINATION=ifClause(translate(OS2APPSDESTINATION)='IMG','IMG','CD')
    REXXDESTINATION=ifClause(translate(REXXDESTINATION)='IMG','IMG','CD')
    WPSDESTINATION=ifClause(translate(WPSDESTINATION)='IMG','IMG','CD')
    WPSAPPSDESTINATION=ifClause(translate(WPSAPPSDESTINATION)='IMG','IMG','CD')
    NETWORKINGDESTINATION=ifClause(translate(NETWORKINGDESTINATION)='IMG','IMG','CD')
    EXTRAFILESDESTINATION=ifClause(translate(EXTRAFILESDESTINATION)='IMG','IMG','CD')
    EXTRADLLFILESDESTINATION=ifClause(translate(EXTRADLLFILESDESTINATION)='IMG','IMG','CD')
    EXTRADEVICEDESTINATION=ifClause(translate(EXTRADEVICEDESTINATION)='IMG','IMG','CD')  

    /* Force serial port settings */
    if isSet(SIO2KSERIALPORT) then do
        SIOSERIALPORT='NO'
        SERIALPORT='NO'
    end
    else do
        SIO2KPATCH='NO'
        if isSet(SIOSERIALPORT) then
            SERIALPORT='NO'
    end

    /* Force MEMDISK settings */
    if \isSet(CREATEBOOTCD) then
        MEMDISK='NO'
    if isSet(MEMDISK) then do
        if isSet(RAMDISK) & translate(RAMDISKDRIVE)=translate(MEMDISKDRIVE) then
            call error 'Both the RAMDISKDRIVE and MEMDISKDRIVE point to the same drive' translate(MEMDISKDRIVE)
        if \isSet(RAMDISK) then
            RAMDISKDRIVE=MEMDISKDRIVE
        VIRTUALDRIVE=TEMPDIR'\MEMDISK'
        rc=createDir(VIRTUALDRIVE)
        TWODISKBOOT='NO'
    end

    /* Force SCSI settings */
    if SCSI=0 then do
        ASPI='NO'
        SCSIDMD='NO'
        REMOVABLESCSI='NO'
    end
    do I=1 to SCSI
        rc=defaultIfNotSet('SCSIOPTIONS.'I,'')
    end

    /* Force USB settings */
    if (USBOHCI>0 | USBUHCI>0 | USBEHCI>0) then
        USB='YES'
    else do
        USB='NO'
        USBKEYB='NO'
        USBCOM='NO'
        USBMOUSE='NO'
        USBPRINTER='NO'
        USBCDROM='NO'
        USBSTORAGE='NO'
    end

    /* Force WPS settings */
    WPSDRIVE=WRITERDRIVE
    if isSet(WPS) then do
        if isSet(CREATEBOOTCD) then do
            if \isSet(MEMDISK) then
                RAMDISK='YES'
            WPSDRIVE=RAMDISKDRIVE
        end
        MOUSE='YES'
        /* Determine display resolution string */
        if VIDEODRIVER<>1 and VIDEORESOLUTION<>0 then do
            /* Video resolution and color depth are encoded in 32bit */
            /* Little-Endian notation                                */
            select
                when VIDEORESOLUTION=2 then VIDEORESOLUTIONSTRING='0x20030000 58020000' /*  800x 600 */
                when VIDEORESOLUTION=3 then VIDEORESOLUTIONSTRING='0x00040000 00030000' /* 1024x 768 */
                when VIDEORESOLUTION=4 then VIDEORESOLUTIONSTRING='0x00050000 00040000' /* 1280x1024 */
                when VIDEORESOLUTION=5 then VIDEORESOLUTIONSTRING='0x40060000 b0040000' /* 1600x1200 */
                otherwise                   VIDEORESOLUTIONSTRING='0x80020000 e0010000' /*  640x 480 */
            end
            select
                when VIDEOCOLORDEPTH=2 then VIDEORESOLUTIONSTRING=VIDEORESOLUTIONSTRING '00800000' /* 32K colors */ 
                when VIDEOCOLORDEPTH=3 then VIDEORESOLUTIONSTRING=VIDEORESOLUTIONSTRING '00000100' /* 64K colors */
                when VIDEOCOLORDEPTH=4 then VIDEORESOLUTIONSTRING=VIDEORESOLUTIONSTRING '00000001' /* 16M colors */
                otherwise                   VIDEORESOLUTIONSTRING=VIDEORESOLUTIONSTRING '00010000' /* 256 colors */
            end
            /* Two extra 32bit integers don't know what they are for (refresh rate?) */
            VIDEORESOLUTIONSTRING=VIDEORESOLUTIONSTRING '01000000 40000000'
        end
    end
    else do
        WPSBOOT='NO'
        WPSAPPS='NO'
    end

    /* Networking? */
    if isSet(NETBIOSTCPIP) then
        TCPIP='YES'
    if isSet(TCPIP) | isSet(NETBIOS) then
        NETWORKING='YES'
    else
        NETWORKING='NO'
    if \isSet(TCPIP) then do
        TCPIPDHCP='NO'
        TCPIP431='NO'
    end

    /* Force extra directory settings */
    do I=1 to EXTRADIRECTORIES
        IDX=pos(":", EXTRADIRECTORY.I)
	if IDX<>0 then EXTRADIRECTORY.I=right(EXTRADIRECTORY.I, length(EXTRADIRECTORY.I)-IDX)
        if left(EXTRADIRECTORY.I,1)<>'\' then EXTRADIRECTORY.I='\'EXTRADIRECTORY.I
        EXTRADIRECTORYADDTOPATHS.I=translate(EXTRADIRECTORYADDTOPATHS.I)
        EXTRADIRECTORYADDTOPATHS.I=translate(EXTRADIRECTORYADDTOPATHS.I, '  ', ',;')
        EXTRADIRECTORYADDTOPATHS.I=strip(EXTRADIRECTORYADDTOPATHS.I)
        EXTRADIRECTORYDESTINATION.I=ifClause(translate(EXTRADIRECTORYDESTINATION.I)='IMG','IMG','CD')
    end

    /* Force extra create file settings */
    do I=1 to EXTRACREATEFILES
        IDX=pos(":", EXTRACREATEFILEDIRECTORY.I)
	if IDX<>0 then EXTRACREATEFILEDIRECTORY.I=right(EXTRACREATEFILEDIRECTORY.I, length(EXTRACREATEFILEDIRECTORY.I)-IDX)
        if left(EXTRACREATEFILEDIRECTORY.I,1)<>'\' then EXTRACREATEFILEDIRECTORY.I='\'EXTRACREATEFILEDIRECTORY.I
        EXTRACREATEFILEDESTINATION.I=ifClause(translate(EXTRACREATEFILEDESTINATION.I)='IMG','IMG','CD')
    end

    /* Force extra INI file settings */
    EXTRAUSERINIRCFILE.1=WPSUSERINIRC
    EXTRASYSINIRCFILE.1=WPSSYSINIRC
    do I=1 to EXTRAUSERINIRCFILES
        rc=defaultIfNotSet('EXTRAUSERINIRCFILEUPDATE.'I,'NO')
    end
    do I=1 to EXTRASYSINIRCFILES
        rc=defaultIfNotSet('EXTRASYSINIRCFILEUPDATE.'I,'NO')
    end

    /* Force settings when creating a maintenance partition */
    if \isSet(CREATEBOOTCD) then do
        VIRTUALDRIVE=WRITERDRIVE
        CDROMDRIVE=WRITERDRIVE
        MAKEROOM='NO'
        TWODISKBOOT='NO'
    end

    /* Only one of HPFS386 and HPFS can be set */
    if isSet(HPFS386) then HPFS='NO'

    /* Force AUTOCHECK options */
    if \isSet(HPFS)    then HPFSAUTOCHECK='NO'
    if \isSet(HPFS386) then HPFS386AUTOCHECK='NO'
    if \isSet(JFS)     then JFSAUTOCHECK='NO'
    if isSet(HPFSAUTOCHECK) | isSet(JFSAUTOCHECK) | isSet(HPFS386AUTOCHECK) then
        AUTOCHECK='YES'
    else
        AUTOCHECK='NO'

    /* Clone CD setup */
    if \isSet(CREATEBOOTCD) then
        CREATECLONECD='NO'
    if isSet(CREATECLONECD) then do
        OS2APPS='YES'
        REXX='YES'
        EXTRAFILES=EXTRAFILES SYSINSTX 'makeini.exe'
        EXTRADIRECTORIES=EXTRADIRECTORIES+1
        EXTRADIRECTORY.EXTRADIRECTORIES=CREATECLONECDDIRECTORY
        EXTRADIRECTORYDESTINATION.EXTRADIRECTORIES='CD'
        EXTRADIRECTORYFILES.EXTRADIRECTORIES='bootAble.cmd bootAble.ico bootAble.cfg' CREATECLONECDCMDLINEARGS
        EXTRADIRECTORYADDTOPATHS.EXTRADIRECTORIES='PATH'
        EXTRADIRECTORYRECURSE.EXTRADIRECTORIES='NO'
        if isSet(WPS) then do
            do I=1 to EXTRAUSERINIRCFILES
                EXTRADIRECTORYFILES.EXTRADIRECTORIES=EXTRADIRECTORYFILES.EXTRADIRECTORIES '"'EXTRAUSERINIRCFILE.I'"'
            end
            do I=1 to EXTRASYSINIRCFILES
                EXTRADIRECTORYFILES.EXTRADIRECTORIES=EXTRADIRECTORYFILES.EXTRADIRECTORIES '"'EXTRASYSINIRCFILE.I'"'
            end
        end
        EXTRACREATEFILES=EXTRACREATEFILES+1
        EXTRACREATEFILEDIRECTORY.EXTRACREATEFILES=CREATECLONECDDIRECTORY
        EXTRACREATEFILEDESTINATION.EXTRACREATEFILES='CD'
        EXTRACREATEFILENAME.EXTRACREATEFILES='cloneCD.cmd'
        EXTRACREATEFILECONTENT.EXTRACREATEFILES='@bootAble /v 'CREATECLONECDVERBOSE CREATECLONECDCMDLINEARGS 'cloneCD.cfg'
        EXTRACREATEFILES=EXTRACREATEFILES+1
        EXTRACREATEFILEDIRECTORY.EXTRACREATEFILES=CREATECLONECDDIRECTORY
        EXTRACREATEFILEDESTINATION.EXTRACREATEFILES='CD'
        EXTRACREATEFILENAME.EXTRACREATEFILES='cloneCD.cfg'
        EXTRACREATEFILECONTENT.EXTRACREATEFILES=''||,
            'SYSINSTX="'fileName(SYSINSTX)'"'||CRLF||,
            'CREATEBOOTCD="NO"'||CRLF||,
            'CREATECLONECD="NO"'||CRLF||,
            'NETWORKCARDFILE="'destinationDrive(NETWORKINGDESTINATION)'\NETWORK\'fileName(NETWORKCARDFILE)'"'||CRLF||,
            'RAMDISK="'CREATECLONECDRAMDISK'"'||CRLF||,
            'WRITERDRIVE="'CREATECLONECDDRIVE'"'||CRLF||,
            'REMOVEROFLAG="'CREATECLONECDREMOVEROFLAG'"'||CRLF||,
            'VERBOSE='CREATECLONECDVERBOSE
    end

    /* Determine tardirs */
    NRTARDIRS=0
    drop TARDIRS.
    if isSet(UNICODE) then do
        NRTARDIRS=NRTARDIRS+1
        TARDIRS.NRTARDIRS='language'
    end
    if isSet(HPFS386) then do
        NRTARDIRS=NRTARDIRS+1
        TARDIRS.NRTARDIRS='ibm386fs'
    end
    if NETWORKINGDESTINATION='IMG' then do
        NRTARDIRS=NRTARDIRS+1
        TARDIRS.NRTARDIRS='network'
    end
    if \isSet(MAKEROOM) |,
       OS2APPSDESTINATION='IMG' |,
       REXXDESTINATION='IMG' |,
       EXTRAFILESDESTINATION='IMG' |,
       EXTRADLLFILESDESTINATION='IMG' |,
       EXTRADEVICEDESTINATION='IMG' |,
       WPSDESTINATION='IMG' |,
       WPSAPPSDESTINATION='IMG'
    then do
        NRTARDIRS=NRTARDIRS+1
        TARDIRS.NRTARDIRS='os2'
    end

    do I=1 to EXTRADIRECTORIES
        if EXTRADIRECTORYDESTINATION.I='IMG' then do
            LENDIR=pos('\',EXTRADIRECTORY.I,2)-2
            if LENDIR=0 then LENDIR=lenght(EXTRADIRECTORY.I)-1
            EXTRADIR=translate(substr(EXTRADIRECTORY.I,2,LENDIR))
            do II=1 to NRTARDIRS
                if translate(TARDIRS.II)=EXTRADIR then do
                    EXTRADIR=''
                    leave
                end
            end
            if EXTRADIR<>'' then do
                NRTARDIRS=NRTARDIRS+1
                TARDIRS.NRTARDIRS=EXTRADIR
            end
        end
    end
    do I=1 to EXTRACREATEFILES
        if EXTRACREATEFILEDESTINATION.I='IMG' then do
            LENDIR=pos('\',EXTRACREATEFILEDIRECTORY.I,2)-2
            if LENDIR=0 then LENDIR=lenght(EXTRACREATEFILEDIRECTORY.I)-1
            EXTRADIR=translate(substr(EXTRACREATEFILEDIRECTORY.I,2,LENDIR))
            do II=1 to NRTARDIRS
                if translate(TARDIRS.II)=EXTRADIR then do
                    EXTRADIR=''
                    leave
                end
            end
            if EXTRADIR<>'' then do
                NRTARDIRS=NRTARDIRS+1
                TARDIRS.NRTARDIRS=EXTRADIR
            end
        end
    end
    TARDIRS.0=NRTARDIRS

    /* Check required executables */
    if fileExists(SYSINSTX)='' then
        call error 'Required executabe SYSINSTX ('SYSINSTX') does not exist'
    if isSet(CREATEBOOTCD) & isSet(CDRECORD) & (isSet(AUTOBURNCD) | isSet(CDRECORDBURNCMD)) then do
        if fileExists(CDRECORDEXE)='' then
            call error 'Required executabe CDRECORDEXE ('CDRECORDEXE') does not exist'
        if fileExists(CDRECORDMKISOFSEXE)='' then
            call error 'Required executabe CDRECORDMKISOFSEXE ('CDRECORDMKISOFSEXE') does not exist'
    end
    if isSet(CREATEBOOTCD) & isSet(CDBOOTMENU) & \isSet(MEMDISK) & fileExists(CDBOOTMENUFILE)='' then
        call error 'Required executabe CDBOOTMENUFILE ('CDBOOTMENUFILE') does not exist'
    if isSet(CREATEBOOTCD) & \isSet(CDRECORD) & fileExists(RSJCDBOOTEXE)='' then
        call error 'Required executabe RSJCDBOOTEXE ('RSJCDBOOTEXE') does not exist'
    if isSet(CREATEBOOTCD) & \isSet(CDRECORD) & isSet(CDBOOTMENU) & RSJCDBOOTEXE=CDBOOTMENUFILE then
        call error 'RSJCDBOOTEXE and CDBOOTMENUFILE point to the same file ('RSJCDBOOTEXE')'
    if isSet(CREATEBOOTCD) & \isSet(CDRECORD) & isSet(RSJATTACHCD) & fileExists(RSJCDATTACHEXE)='' then
        call error 'Required executabe RSJCDATTACHEXE ('RSJCDATTACHEXE') does not exist' 
    if isSet(CREATEBOOTCD) & \isSet(CDRECORD) & isSet(AUTOBURNCD) & isSet(ERASECD) & fileExists(RSJCDWFMTEXE)='' then
        call error 'Required executabe RSJCDWFMTEXE ('RSJCDWFMTEXE') does not exist' 

    return 0


/* Read and parse an individual configuration file ------------------------ */
readConfigFile:
    CONFIGFILE=ARG(1)
    SETDEFAULTS=ARG(2)
    CONFIGFILE=fileExists(CONFIGFILE)
    if CONFIGFILE='' then 
        call error 'Configuration file' ARG(1) 'not found'
    call progress 2, 'Reading configuration file' CONFIGFILE ifSet(SETDEFAULTS, '(defaults)')
    STATEMENT=''
    LINENR=0
    do while lines(CONFIGFILE)>0
        LINE=strip(linein(CONFIGFILE))
        LINENR=LINENR+1
        call progress 4, CONFIGFILE'('LINENR'):' LINE
        CONT=0
        if right(LINE,1)=',' then do
            CONT=1
            LINE=left(LINE,length(LINE)-1)
        end
        if length(LINE)>0 & left(LINE,2)<>'/*' & left(LINE,1)<>';' then
            STATEMENT=strip(STATEMENT LINE)
        if CONT=0 then do
            DOPARSE='YES'
            if isSet(SETDEFAULTS) then do
                EQUALSIGN=pos('=', STATEMENT)
                if EQUALSIGN>0 & words(left(STATEMENT,EQUALSIGN))=1 then do
                    VARNAME=left(STATEMENT,EQUALSIGN-1)
                    if translate(VARNAME)<>'RC' & translate(VARNAME)<>value(VARNAME) then
                        DOPARSE='NO'
                end
            end
            if isSet(DOPARSE) & STATEMENT<>'' then do
                interpret STATEMENT
                /* Searchpath set? */
                if SEARCHPATH<>PREVSEARCHPATH then do
                    PREVSEARCHPATH=SEARCHPATH
                    rc=value('SEARCHPATH',SEARCHPATH,'OS2ENVIRONMENT')
                end
            end
            STATEMENT=''
        end
    end
    rc=stream(CONFIGFILE, 'C', 'CLOSE')
    if rc<>'READY:' then
        call error 'Failed to close' CONFIGFILE 'rc='rc

    return 0


/* List of MAKEROOM files for the os2 directory --------------------------- */
makeRoomFiles:
    FILES_=VITALFILES_ SHELLFILE PROTSHELLFILE
    if isSet(WPS)               then FILES_=FILES_ WPSRUNWORKPLACEFILE
    if isSet(CDFS)              then FILES_=FILES_ CDFSFILES
    if isSet(FAT32)             then FILES_=FILES_ FAT32FILES FAT32CACHEFILE
    if isSet(NTFS)              then FILES_=FILES_ NTFSFILES
    if isSet(UDF)               then FILES_=FILES_ UDFFILES
    if isSet(APM)               then FILES_=FILES_ APMFILE APMDAEMONFILE
    if isSet(SERIALPORT)        then FILES_=FILES_ SERIALPORTFILE
    if isSet(SIOSERIALPORT)     then FILES_=FILES_ SIOSERIALPORTFILE
    if isSet(SIO2KSERIALPORT)   then FILES_=FILES_ SIO2KUARTFILE SIO2KSERIALPORTFILE
    if isSet(SIO2KPATCH)        then FILES_=FILES_ SIO2KPATCHFILE
    if isSet(MOUSE)             then FILES_=FILES_ MOUSEPOINTDDFILE MOUSEFILE MOUSEDEVFILE
    if isSet(USBKEYB)           then FILES_=FILES_ USBKEYBFILE
    if isSet(USBCOM)            then FILES_=FILES_ USBCOMFILE
    if isSet(USBMOUSE)          then FILES_=FILES_ USBMOUSEFILE
    if isSet(USBPRINTER)        then FILES_=FILES_ USBPRINTERFILE
    if isSet(LVM)               then FILES_=FILES_ LVMFILES
    if isSet(RAMDISK)           then FILES_=FILES_ RAMDISKFILE
    if isSet(HPFS)    & \isSet(HPFSAUTOCHECK)    then FILES_=FILES_ HPFSFILES
    if isSet(HPFS386) & \isSet(HPFS386AUTOCHECK) then FILES_=FILES_ HPFS386OS2FILES
    if isSet(JFS)     & \isSet(JFSAUTOCHECK)     then FILES_=FILES_ JFSFILES
    if                  \isSet(AUTOCHECK)        then FILES_=FILES_ VITALFILES__
    return FILES_


/* List of AUTOCHECK files for the os2 directory -------------------------- */
autoCheckFiles:
    FILES__=''
    if isSet(HPFS)    & isSet(HPFSAUTOCHECK)    then FILES__=FILES__ HPFSFILES
    if isSet(HPFS386) & isSet(HPFS386AUTOCHECK) then FILES__=FILES__ HPFS386OS2FILES
    if isSet(JFS)     & isSet(JFSAUTOCHECK)     then FILES__=FILES__ JFSFILES
    if                  isSet(AUTOCHECK)        then FILES__=FILES__ VITALFILES__
    return FILES__


/* List of MAKEROOM files for the os2\dll directory ----------------------- */
makeRoomDLLFiles:
    DLLFILES_=VITALDLLFILES_
    if isSet(CDFS)  then DLLFILES_=DLLFILES_ CDFSDLLFILES
    if isSet(FAT32) then DLLFILES_=DLLFILES_ FAT32DLLFILES
    if isSet(NTFS)  then DLLFILES_=DLLFILES_ NTFSDLLFILES
    if isSet(UDF)   then DLLFILES_=DLLFILES_ UDFDLLFILES
    if isSet(HPFS)    & \isSet(HPFSAUTOCHECK)    then DLLFILES_=DLLFILES_ HPFSDLLFILES
    if isSet(HPFS386) & \isSet(HPFS386AUTOCHECK) then DLLFILES_=DLLFILES_ HPFS386OS2DLLFILES
    if isSet(JFS)     & \isset(JFSAUTOCHECK)     then DLLFILES_=DLLFILES_ JFSDLLFILES
    if isSet(LVM)     & \isSet(AUTOCHECK)        then DLLFILES_=DLLFILES_ LVMDLLFILES
    if isSet(UNICODE) & \isSet(AUTOCHECK)        then DLLFILES_=DLLFILES_ UNICODEDLLFILES
    if                  \isSet(AUTOCHECK)        then DLLFILES_=DLLFILES_ VITALDLLFILES__
    return DLLFILES_
    

/* List of AUTOCHECK files for the os2\dll directory ---------------------- */
autoCheckDLLFiles:
    DLLFILES__=''
    if isSet(HPFS)    & isSet(HPFSAUTOCHECK)    then DLLFILES__=DLLFILES__ HPFSDLLFILES
    if isSet(HPFS386) & isSet(HPFS386AUTOCHECK) then DLLFILES__=DLLFILES__ HPFS386OS2DLLFILES
    if isSet(JFS)     & isset(JFSAUTOCHECK)     then DLLFILES__=DLLFILES__ JFSDLLFILES
    if isSet(LVM)     & isSet(AUTOCHECK)        then DLLFILES__=DLLFILES__ LVMDLLFILES
    if isSet(UNICODE) & isSet(AUTOCHECK)        then DLLFILES__=DLLFILES__ UNICODEDLLFILES
    if                  isSet(AUTOCHECK)        then DLLFILES__=DLLFILES__ VITALDLLFILES__
    return DLLFILES__
    

/* Prepare a boot image --------------------------------------------------- */
prepareImage:
    IMAGENR=ARG(1)

    if isSet(MEMDISK) then
        return 0

    if (\isSet(CREATEBOOTCD) & \isSet(FORMATMAINTENANCE)) then do
        say 'You are creating a Maintenance Partition on drive' WRITERDRIVE
        say 'In order to guarantee its working, bootAble needs to format it.' 
        rc=charout(,'Do you want to format the partition? ([Y]es/[N]o/[Q]uit) ')
        rc=1
        do until rc=0
            KEY=translate(sysGetKey('NOECHO'))
            if pos(KEY,'YNQ')>0 then
                rc=0
            else
                call beep 524,100
        end
        say KEY
        if KEY='Q' then
            call error 'Script aborted'
        if KEY='N' then do
            return 0
        end
    end

    /* Format the (virtual) disk */
    call progress 1, 'Formatting' bootImageName()
    LABEL=ifSet(CREATEBOOTCD,'CDBOOT_'IMAGENR,'MAINTENANCE')
    rc=runCMD('label' VIRTUALDRIVE||'FAILED', 'Failed to reset the label of' VIRTUALDRIVE '(was the drive formatted before?)')
    rc=lineOut(TEMPDIR'\response', 'FAILED'||CRLF||'Y')
    if rc=0 then
        rc=lineout(TEMPDIR'\response')
    if rc<> 0 then
        call error 'Failed to create format response file'
    'format' VIRTUALDRIVE '/fs:'ifSet(CREATEBOOTCD,'fat','hpfs') '/once /v:'LABEL '<' TEMPDIR'\response' IGNOREOUTPUT
    /* Somehow format does not always return non-zero rc in case of error... */
    if rc<>0 | word(sysDriveInfo(VIRTUALDRIVE),4)<>LABEL then
        call error 'Error formatting' bootImageName() '('VIRTUALDRIVE')'

    /* Install system files */
    call progress 1, 'Transfering system files to' bootImageName()
    rc=runCMD(SYSINSTX VIRTUALDRIVE, 'Error transfering system files to' bootImageName())

    return rc


/* Write a boot image to CD ----------------------------------------------- */
writeImage:
    if \isSet(CREATEBOOTCD) then return 0

    IMAGENR=ARG(1)

    if IMAGENR=1 & isSet(CDBOOTMENU) & \isSet(MEMDISK) then do
        rc=runCMD(CDBOOTMENUFILE '/DRIVE' left(VIRTUALDRIVE,1) '/TIMEOUT' CDBOOTMENUTIMEOUT CDBOOTMENUOTHEROPTIONS, 'Failed adding boot menu')
    end

    if isSet(PAUSEBEFOREBOOTIMAGE) then do
        say 'Pausing before writing' bootImageName()
        rc=charout(,'Continue with script? ([Y]es/[N]o/[Q]uit without cleaning up) ')
        rc=1
        do until rc=0
            KEY=translate(sysGetKey('NOECHO'))
            if pos(KEY,'YNQ')>0 then
                rc=0
            else
                call beep 524,100
        end
        say KEY
        if KEY<>'Y' then do
            if KEY='Q' then
                CLEANUP='NO'
            call error 'Script aborted'
        end
    end

    if isSet(MEMDISK) then do
        rc=writeMemDisk()
    end
    else do
        call progress 1, 'Writing' bootImageName()
        rc=runCMD(SAVEDSKFEXE '/a/d' VIRTUALDRIVE WRITERDRIVE'\'BOOTIMAGENAME.IMAGENR, 'Error writing' bootImageName())

        /* The first image will be marked bootable */
        if IMAGENR=1 & \isSet(CDRECORD) then do
            rc=runCMD(RSJCDBOOTEXE WRITERDRIVE||BOOTIMAGENAME.IMAGENR, 'Error marking Boot Image as bootAble')
        end
    end

    /* Copy config.sys to root of CD? */
    if isSet(COPYCONFIGSYS) & IMAGENR=1 then
        rc=copyFiles(VIRTUALDRIVE'\config.sys', WRITERDRIVE)

    return 0


/* Write memdisk images to CD --------------------------------------------- */
writeMemDisk:
    /* CDLOADER.BIN */
    rc=createDir(WRITERDRIVE'\BOOTIMGS')
    rc=sCopyFile(MEMDISKCDLOADERBIN, WRITERDRIVE'\BOOTIMGS\cdloader.bin')
    if \isSet(CDRECORD) then do
        rc=runCMD(RSJCDBOOTEXE WRITERDRIVE'\BOOTIMGS\cdloader.bin', 'Error marking' WRITERDRIVE'\BOOTIMGS\cdloader.bin as bootAble')
    end

    /* MEMDISK.PF */
    PFFILE=TEMPDIR'\memdisk.pf'
    NEXTPFFILE=TEMPDIR'\root.pf'
    PFSFILE=PFFILE's'
    PFSTXT=,
        'newline'||CRLF||,
        'output "Welcome to";'||CRLF||,
        'newline'||CRLF||,
        'output "                                                                ";'||CRLF||,
        'output "                                                            ";'||CRLF||,
        'output "                                                          ";'||CRLF||,
        'output "                                                       ";'||CRLF||,
        'output "                                                      ";'||CRLF||,
        'output "                            ";'||CRLF||,
        'output "                     ";'||CRLF||,
        'output "                                ";'||CRLF||,
        'output "                               ";'||CRLF||,
        'output "                                ";'||CRLF||,
        'output "                                     ";'||CRLF||,
        'output "                                  ";'||CRLF||,
        'output "                        ";'||CRLF||,
        'output "                                     ";'||CRLF||,
        'newline'||CRLF||,
        'newline'||CRLF
    rc=createPFFile(PFSFILE, PFSTXT, PFFILE, NEXTPFFILE)
    
    /* ROOT.PF */
    rc=sCopyFile(MEMDISKERASEFILSYS, VIRTUALDRIVE'\erasefil.sys')
    if (TARDIRS.0>0) then do
        rc=sCopyFile(MEMDISKUNTARDLL,    VIRTUALDRIVE'\untar.dll')
        rc=sCopyFile(MEMDISKUNTARMSG,    VIRTUALDRIVE'\untar.msg')
        rc=sCopyFile(MEMDISKLOADDLLSYS,  VIRTUALDRIVE'\loaddll.sys')
        rc=sCopyFile(MEMDISKLOADDLLMSG,  VIRTUALDRIVE'\loaddll.msg')
    end
    PFFILE=NEXTPFFILE
    NEXTPFFILE=TEMPDIR'\os2.pf'
    PFSFILE=PFFILE's'
    PFSTXT=,
        'output "Loading root     files "'||CRLF||,
        'add file 'VIRTUALDRIVE'\*'||CRLF||,
        'bootsector 'MEMDISKBOOTSECTOR||CRLF
    rc=createPFFile(PFSFILE, PFSTXT, PFFILE, NEXTPFFILE)


    /* OS2.PF */
    PFFILE=NEXTPFFILE
    NEXTPFFILE=TEMPDIR'\os2boot.pf'
    PFSFILE=PFFILE's'
    PFSTXT=,
        'output "Loading os2      files "'||CRLF||,
        'add file 'VIRTUALDRIVE'\os2\*'||CRLF
    rc=createPFFile(PFSFILE, PFSTXT, PFFILE, NEXTPFFILE)


    /* OS2BOOT.PF */
    PFFILE=NEXTPFFILE
    NEXTPFFILE=TEMPDIR'\os2dll.pf'
    PFSFILE=PFFILE's'
    PFSTXT=,
        'output "Loading os2\boot files "'||CRLF||,
        'add file 'VIRTUALDRIVE'\os2\boot\*'||CRLF
    rc=createPFFile(PFSFILE, PFSTXT, PFFILE, NEXTPFFILE)

    /* OS2DLL.PF */
    PFFILE=NEXTPFFILE
    NEXTPFFILE=ifClause(TARDIRS.0>0,TEMPDIR'\'TARDIRS.1'.pft','')
    PFSFILE=PFFILE's'
    PFSTXT=,
        'output "Loading os2\dll  files "'||CRLF||,
        'add file 'VIRTUALDRIVE'\os2\dll\*'||CRLF
    rc=createPFFile(PFSFILE, PFSTXT, PFFILE, NEXTPFFILE)

    /* Tar files */
    do I=1 to TARDIRS.0
        II=I+1
        PFFILE=TEMPDIR'\'TARDIRS.I'.pft'
        NEXTPFFILE=ifClause(TARDIRS.0>I,TEMPDIR'\'TARDIRS.II'.pft','')
        PFSFILE=PFFILE's'
        PFSTXT=,
            'output "Loading 'left(TARDIRS.I,8)' tar file "'||CRLF
        rc=createTARFile(TARDIRS.I)
        rc=createPFFile(PFSFILE, PFSTXT, PFFILE, NEXTPFFILE)
    end

    /* MEMBOOT.BIN setup */
    MEMDISKSIZEMINFREE=0
    MEMDISKSIZEFREED=0
    if left(MEMDISKSIZE,1)='+' then do
        MEMDISKSIZEMINFREE=substr(MEMDISKSIZE,2)*1024
        MEMDISKMAXTARSIZE=0
        /* Determine space needed */
        drop FILETREE.
        rc=sysFileTree(VIRTUALDRIVE'\*', 'FILETREE', 'S')
        do I=1 to FILETREE.0
            FNAME=translate(word(FILETREE.I,5))
            FSIZE=word(FILETREE.I,3)
            FSIZE=max((FSIZE+511)%512*512,512)
            MEMDISKSIZE=MEMDISKSIZE+FSIZE
            if FNAME=translate(VIRTUALDRIVE'\untar.dll') |,
               FNAME=translate(VIRTUALDRIVE'\untar.msg') |,
               FNAME=translate(VIRTUALDRIVE'\loaddll.sys') |,
               FNAME=translate(VIRTUALDRIVE'\loaddll.msg')
            then 
                MEMDISKSIZEFREED=MEMDISKSIZEFREED+FSIZE
            do II=1 to TARDIRS.0
                if FNAME=translate(VIRTUALDRIVE'\'TARDIRS.II'.tar') then do
                    MEMDISKMAXTARSIZE=max(MEMDISKMAXTARSIZE,FSIZE)
                    MEMDISKSIZE=MEMDISKSIZE-FSIZE
                end
            end
        end
        drop FILETREE.
        MEMDISKSIZEFREED=MEMDISKSIZEFREED+MEMDISKMAXTARSIZE
        MEMDISKSIZE=MEMDISKSIZE+MEMDISKMAXTARSIZE
        MEMDISKSIZE=MEMDISKSIZE+max(0,MEMDISKSIZEMINFREE-MEMDISKSIZEFREED)
        MEMDISKSIZE=(MEMDISKSIZE+1023)%1024
        MEMDISKSIZE=MEMDISKSIZE+384 /* Reserve for overhead */
        MEMDISKSIZEFREED=MEMDISKSIZEFREED+max(0,MEMDISKSIZEMINFREE-MEMDISKSIZEFREED)
    end
    rc=sCopyFile(MEMDISKMEMBOOTBIN, TEMPDIR'\memboot.bin')
    RSPFILE=TEMPDIR'\memboot.rsp'
    RSPTXT=''||,
        '$Disk_KiB$='MEMDISKSIZE||CRLF||,
        '$OSminMib$='MEMDISKMINMB||CRLF||,
        '$PartName$=bootAble memdisk'||CRLF||,
        '$VolName_$=bootAble memdisk'||CRLF||,
        '$V_Letter$='MEMDISKDRIVE||CRLF||,
        '$PartType$=true'||CRLF||,
        '$TermEmul$=true'||CRLF||,
        '$BootTarg$='ifSet(CDBOOTMENU,'*','m')||CRLF||,
        '$Update_A$=false'||CRLF||,
        '$Update_B$=false'||CRLF||,
        '$SerialDB$=false'||CRLF||,
        '$Indicato$=1'||CRLF||,
        '$eCS_Menu$=0'||CRLF||,
        '$DebugOpt$=0'||CRLF||,
        '$LoadFont$=false'||CRLF||,
        '$PaletteM$=false'||CRLF||,
        '$CALL_CSM$=false'||CRLF||,
        '$UpdPPrim$=true'||CRLF||,
        '$ProgChar$='||CRLF||,
        '$SaveFile$=yes'||CRLF
    rc=createFile(RSPFILE,RSPTXT)
    call progress 2, 'Configuring' TEMPDIR'\memboot.bin'
    rc=runCMD(MEMDISKMEMCFGEXE TEMPDIR'\memboot.bin' '<' TEMPDIR'\memboot.rsp', 'Failed to configure' TEMPDIR'\memboot.bin')
    rc=sCopyFile(TEMPDIR'\memboot.bin',  WRITERDRIVE'\BOOTIMGS')
    TXT='Created MEMDISK boot image of' trunc(2*MEMDISKSIZE/1024+0.5)/2 'MiB'
    if MEMDISKSIZEMINFREE<>0 then
        TXT=TXT 'leaving approximately' trunc(2*MEMDISKSIZEFREED/1024/1024+0.5)/2 'MiB free'
    call progress 1, TXT
    return rc


/* Create a PF file ------------------------------------------------------- */
createPFFile: procedure expose MEMDISKPACKFILEEXE CRLF WRITERDRIVE CLEANUP TEMPDIR VERBOSE IGNOREOUTPUT
    PFSFILE=ARG(1)
    PFFILE=ARG(3)
    NEXTPFFILE=ARG(4)
    PFSTXT='create archive' PFFILE||CRLF||,
        ARG(2)||,
        ifClause(NEXTPFFILE<>'','next pointer' NEXTPFFILE||CRLF)||,
        'close archive'||CRLF
    rc=createFile(PFSFILE, PFSTXT)
    call progress 2, 'Creating' PFFILE
    rc=runCMD(MEMDISKPACKFILEEXE PFSFILE, 'Failed to create' PFFILE)
    rc=sCopyFile(PFFILE, WRITERDRIVE'\BOOTIMGS')
    return rc


/* Create a TAR file ------------------------------------------------------ */
createTARFile: procedure expose MEMDISKTAREXE PFSTXT CRLF VIRTUALDRIVE CLEANUP TEMPDIR VERBOSE IGNOREOUTPUT
    PREVDIR=directory()
    PREVDIR_=directory(driveName(VIRTUALDRIVE))
    rc=directory(VIRTUALDRIVE'\TOTAR.$$$')
    TARDIR=ARG(1)
    TARFILE=VIRTUALDRIVE'\'TARDIR'.tar'
    call progress 2, 'Creating' TARFILE
    rc=runCMD(MEMDISKTAREXE '-cpvf' TARFILE TARDIR, 'Failed to create' TARFILE)
    rc=directory(PREVDIR_)
    rc=directory(PREVDIR)
    PFSTXT=PFSTXT||'add file' TARFILE||CRLF
    return 0


/* Copy files to boot image ----------------------------------------------- */
copyImageFiles:
    IMAGENR=ARG(1)

    call progress 1, 'Copying boot files to' bootImageName()

    /* Files to copy to root directory of boot image */
    if IMAGENR=1 | (isSet(TWODISKBOOT) & \isSet(MEMDISK)) then
        rc=copyFiles(VITALROOTFILES.IMAGENR, VIRTUALDRIVE'\', 'T')

    /* Files to copy to os2 directory of boot image */
    rc=createDir(VIRTUALDRIVE'\OS2')
    if IMAGENR=2 then do
        FILES=VITALFILES
        if   isSet(CDROM)     then FILES=FILES CDROMDMDFILE
        if   isSet(CDFS)      then FILES=FILES CDFSIFSFILE
        if   isSet(FAT32)     then FILES=FILES FAT32IFSFILE
        if   isSet(NTFS)      then FILES=FILES NTFSIFSFILE
        if   isSet(HPFS)      then FILES=FILES HPFSIFSFILE
        if   isSet(JFS)       then FILES=FILES JFSIFSFILE
        if   isSet(UDF)       then FILES=FILES UDFIFSFILE
        if   isSet(RAMDISK)   then FILES=FILES RAMDISKIFSFILE
        if   isSet(UNICODE)   then FILES=FILES UNICODEFILE
        if   isSet(AUTOCHECK) then FILES=FILES autoCheckFiles()
        rc=copyFiles(FILES, VIRTUALDRIVE'\os2', 'T')
    end
    
    /* Files to copy to os2\boot directory of boot image */
    if IMAGENR=1 then do
        rc=createDir(VIRTUALDRIVE'\OS2\BOOT')
        BOOTFILES=VITALBOOTFILES DASDFILE
        if isSet(CREATEBOOTCD) & \isSet(MEMDISK) then BOOTFILES=BOOTFILES BOOTFLTFILE
        if isSet(WPS)           then BOOTFILES=BOOTFILES WPSBOOTFILES
        if isSet(FLOPPY)        then BOOTFILES=BOOTFILES FLOPPYADDFILE
        if isSet(PARALLELPORT)  then BOOTFILES=BOOTFILES PARALLELPORTFILE
        if isSet(PPZIP)         then BOOTFILES=BOOTFILES PPZIPADDFILE
        if isSet(NON512DASD)    then BOOTFILES=BOOTFILES NON512DASDFILE
        if isSet(DASDFALLBACK)  then BOOTFILES=BOOTFILES DASDFALLBACKFILE ifSet(MEMDISK,MEMDISKDASDFALLBACKFILE)
        if isSet(IDE)           then BOOTFILES=BOOTFILES IDEADDFILE
        if isSet(ATAPI)         then BOOTFILES=BOOTFILES ATAPIFLTFILE ifClause(\isSet(DANIATAPI)&\isSet(LATESTATAPI),ATAPICDFLTFILE)
        if isSet(LVM)           then BOOTFILES=BOOTFILES LVMDMDFILE
        if isSet(MEMDISK)       then BOOTFILES=BOOTFILES MEMDISKADDFILE
        if isSet(SCSIDMD)       then BOOTFILES=BOOTFILES SCSIDMDFILE
        if isSet(ASPI)          then BOOTFILES=BOOTFILES ASPIDMDFILE
        if isSet(REMOVABLESCSI) then BOOTFILES=BOOTFILES REMOVABLESCSIFILE
        do I=1 to SCSI
            BOOTFILES=BOOTFILES SCSIADDFILE.I
        end
        do I=1 to EXTRABASEDEV
            BOOTFILES=BOOTFILES EXTRABASEDEVFILE.I
        end
        if isSet(USB) then do
            BOOTFILES=BOOTFILES USBDFILE USBHIDFILE
            if USBOHCI>0 then BOOTFILES=BOOTFILES USBOHCIFILE
            if USBUHCI>0 then BOOTFILES=BOOTFILES USBUHCIFILE
            if USBEHCI>0 then BOOTFILES=BOOTFILES USBEHCIFILE
            if isSet(USBCDROM)   then BOOTFILES=BOOTFILES USBCDROMFILE
            if isSet(USBSTORAGE) then BOOTFILES=BOOTFILES USBSTORAGEFILE
        end
        rc=copyFiles(BOOTFILES, VIRTUALDRIVE'\os2\boot', 'T')
    end
    if IMAGENR=2 & isSet(APM) & APMGSVDAEMONFILE<>'' then do
        rc=createDir(VIRTUALDRIVE'\OS2\BOOT')
        rc=copyFiles(APMGSVDAEMONFILE, VIRTUALDRIVE'\OS2\BOOT', 'T')
    end

    /* Files to copy to os2\dll directory of boot image */
    if IMAGENR=2 then do
        rc=createDir(VIRTUALDRIVE'\OS2\DLL')
        DLLFILES=VITALDLLFILES
        if   isSet(AUTOCHECK) then DLLFILES=DLLFILES autoCheckDLLFiles()
        rc=copyFiles(DLLFILES, VIRTUALDRIVE'\os2\dll', 'T')
    end
    
    /* Unicode support files */
    if IMAGENR=2 & isSet(UNICODE) then do
        if isSet(CREATEBOOTCD) & isSet(MEMDISK) then
            DESTDRIVE=VIRTUALDRIVE'\TOTAR.$$$'
        else
            DESTDRIVE=VIRTUALDRIVE
        /* Language directory tree */
        rc=createDir(DESTDRIVE'\LANGUAGE')
        rc=createDir(DESTDRIVE'\LANGUAGE\CODEPAGE')
        rc=createDir(DESTDRIVE'\LANGUAGE\FONT')
        rc=createDir(DESTDRIVE'\LANGUAGE\KEYBOARD')
        rc=createDir(DESTDRIVE'\LANGUAGE\LOCALE')
        rc=copyFiles(UNICODECODEPAGEFILE, DESTDRIVE'\language\codepage', 'T')
        rc=copyFiles(UNICODEKEYBOARDFILE, DESTDRIVE'\language\keyboard', 'T')
        rc=copyFiles(UNICODEUSERFILE,     DESTDRIVE'\language\locale',   'T')
    end

    /* HPFS386 support files */
    if IMAGENR=2 & isSet(HPFS386) then do
        if isSet(CREATEBOOTCD) & isSet(MEMDISK) then
            DESTDRIVE=VIRTUALDRIVE'\TOTAR.$$$'
        else
            DESTDRIVE=VIRTUALDRIVE
        rc=createDir(DESTDRIVE'\IBM386FS')
        rc=copyFiles(HPFS386IFSFILE HPFS386FILES, DESTDRIVE'\ibm386fs', 'T')
    end

    return rc


/* Copy files to CD ------------------------------------------------------- */
copyOtherFiles:
    DEST=ARG(1)      
    IMGNR=2
    call progress 1, 'Copying other files to' destinationName()

    /* Files to copy to os2 directory of the CD or image */
    FILES=''
    if                  DEST=EXTRAFILESDESTINATION      then FILES=FILES EXTRAFILES
    if isSet(OS2APPS) & DEST=OS2APPSDESTINATION         then FILES=FILES OS2APPFILES
    if isSet(REXX)    & DEST=REXXDESTINATION            then FILES=FILES REXXFILES ifSet(WPSBOOT,'',REXXCOMMANDLINEFILE)
    if isSet(WPS)     & DEST=WPSDESTINATION             then FILES=FILES WPSFILES VIDEODRIVERFILES.VIDEODRIVER
    if                  DEST=ifSet(MAKEROOM,'CD','IMG') then FILES=FILES makeRoomFiles()
    if                  DEST=EXTRADEVICEDESTINATION     then
        do I=1 to EXTRADEVICE
            FILES=FILES EXTRADEVICEFILE.I
        end
    FILES=strip(FILES)
    if FILES<>'' then do
        rc=createDir(DESTDRIVE'\OS2')
        rc=copyFiles(FILES, DESTDRIVE'\os2',ifClause(isSet(CREATEBOOTCD) & DEST='IMG','T',''))
    end

    /* Files to copy to os2\dll directory of CD or image */
    FILES=''
    if                                                    DEST=EXTRADLLFILESDESTINATION    then FILES=FILES EXTRADLLFILES
    if isSet(OS2APPS)                                   & DEST=OS2APPSDESTINATION          then FILES=FILES OS2APPDLLFILES
    if isSet(REXX)                                      & DEST=REXXDESTINATION             then FILES=FILES REXXDLLFILES
    if isSet(WPS)                                       & DEST=WPSDESTINATION              then FILES=FILES WPSDLLFILES VIDEODRIVERDLLFILES.VIDEODRIVER
    if                                                    DEST=ifSet(MAKEROOM,'CD','IMG')  then FILES=FILES makeRoomDLLFiles()
    if isSet(TCPIP431) & isSet(TCPIPDHCP) & \isSet(WPS) & DEST=NETWORKINGDESTINATION       then FILES=FILES TCPIP431WPSDLLFILES
    FILES=strip(FILES)
    if FILES<>'' then do
        rc=createDir(DESTDRIVE'\OS2\DLL')
        rc=copyFiles(FILES, DESTDRIVE'\os2\dll',ifClause(isSet(CREATEBOOTCD) & DEST='IMG','T',''))
    end

    /* Files to copy to network directory of CD or image */
    if isSet(NETWORKING) & DEST=NETWORKINGDESTINATION then do
        rc=createDir(DESTDRIVE'\NETWORK')
        FILES=NETWORKCARDFILE NETWORKINGFILES
        if isSet(TCPIP)        then FILES=FILES TCPIPFILES
        if isSet(NETBIOS)      then FILES=FILES NETBIOSFILES
        if isSet(NETBIOSTCPIP) then FILES=FILES NETBIOSTCPIPFILES
        rc=copyFiles(FILES,DESTDRIVE'\NETWORK',ifClause(isSet(CREATEBOOTCD) & DEST='IMG','T',''))
    end

    /* Files to copy to network\bin directory of CD or image */
    if isSet(NETWORKING) & DEST=NETWORKINGDESTINATION then do
        rc=createDir(DESTDRIVE'\NETWORK\BIN')
        BINFILES=NETWORKINGBINFILES
        if isSet(TCPIP)        then BINFILES=BINFILES TCPIPBINFILES
        if isSet(NETBIOS)      then BINFILES=BINFILES NETBIOSBINFILES
        if isSet(NETBIOSTCPIP) then BINFILES=BINFILES NETBIOSTCPIPBINFILES
        rc=copyFiles(BINFILES,DESTDRIVE'\NETWORK\BIN',ifClause(isSet(CREATEBOOTCD) & DEST='IMG','T',''))
    end

    /* Files to copy to network\dll directory of CD or image */
    if isSet(NETWORKING) & DEST=NETWORKINGDESTINATION then do
        rc=createDir(DESTDRIVE'\NETWORK\DLL')
        FILES=NETWORKINGDLLFILES
        if isSet(TCPIP)                       then FILES=FILES TCPIPDLLFILES
        if isSet(TCPIP431) & isSet(TCPIPDHCP) then FILES=FILES TCPIP431DLLFILES
        if isSet(NETBIOS)                     then FILES=FILES NETBIOSDLLFILES
        if isSet(NETBIOSTCPIP)                then FILES=FILES NETBIOSTCPIPDLLFILES
        rc=copyFiles(FILES,DESTDRIVE'\NETWORK\DLL',ifClause(isSet(CREATEBOOTCD) & DEST='IMG','T',''))
    end

    /* Files to copy to os2\help directory of CD */
    if isSet(WPS) & DEST=WPSDESTINATION then do
        rc=createDir(DESTDRIVE'\OS2\HELP')
        HELPFILES=WPSHELPFILES VIDEODRIVERHELP.VIDEODRIVER EXTRAHELPFILES
        if isSet(WPSAPPS)      then HELPFILES=HELPFILES WPSAPPSHELPFILES
        if isSet(REXX)         then HELPFILES=HELPFILES REXXHELPFILES
        if isSet(TCPIP)        then HELPFILES=HELPFILES TCPIPHELPFILES
        if isSet(NETBIOS)      then HELPFILES=HELPFILES NETBIOSHELPFILES
        if isSet(NETBIOSTCPIP) then HELPFILES=HELPFILES NETBIOSTCPIPHELPFILES
        rc=copyFiles(HELPFILES, DESTDRIVE'\os2\help')
    end

    /* Files to copy to os2\apps directory of CD */
    if isSet(WPSAPPS) & DEST=WPSAPPSDESTINATION then do
        rc=createDir(DESTDRIVE'\OS2\APPS')
        rc=copyFiles(WPSAPPSFILES, DESTDRIVE'\OS2\APPS')
    end

    /* Files to copy to os2\apps\dll directory of CD */
    if isSet(WPSAPPS) & DEST=WPSAPPSDESTINATION then do
        rc=createDir(DESTDRIVE'\OS2\APPS\DLL')
        rc=copyFiles(WPSAPPSDLLFILES, DESTDRIVE'\OS2\APPS\DLL')
    end

    /* Extra directories */
    do I=1 to EXTRADIRECTORIES
        if DEST=EXTRADIRECTORYDESTINATION.I then do
            rc=createDir(DESTDRIVE||EXTRADIRECTORY.I)
            rc=copyFiles(EXTRADIRECTORYFILES.I,DESTDRIVE||EXTRADIRECTORY.I,,ifSet(EXTRADIRECTORYRECURSE.I,'I',''))
            if isSet(EXTRADIRECTORYRECURSE.I) then do
                rc=parseFileList(EXTRADIRECTORYFILES.I)
                drop EDFILELIST.
                EDFILELIST.0=FILELIST.0
                do II=1 to FILELIST.0
                    EDFILELIST.II=FILELIST.II
                end
                do II=1 to EDFILELIST.0
                    BASEDIR=filespec('Drive', EDFILELIST.II)||filespec('Path', EDFILELIST.II)
                    if substr(BASEDIR,2,2)<>':\' then 
                        call warningMessage 'Ignored invalid recursive extra directory specification' EDFILELIST.II 
                    else do
                        FILEMASK=filespec('Name', EDFILELIST.II)
                        drop DIRTREE.
                        rc=sysFileTree(BASEDIR'*', 'DIRTREE', 'DOS')
                        if rc<>0 then call error 'Failed to retrieve directory tree for recursive extra directory' EDFILELIST.II
                        do III=1 to DIRTREE.0
                            NEWDIR=EXTRADIRECTORY.I||substr(DIRTREE.III,length(BASEDIR))
                            rc=createDir(DESTDRIVE||NEWDIR)
                            rc=copyFiles('"'||DIRTREE.III||'\'||FILEMASK||'"',DESTDRIVE||NEWDIR,,'I')
                        end
                    end
                end
            end
        end
    end

    return rc


/* Create config.sys file ------------------------------------------------- */
createConfigSys:
    call progress 1, 'Creating config.sys file on' bootImageName()
    
    /* Drive and paths for boot image files */
    BASEDRIVE=destinationDrive('IMG')
    if \isSet(MEMDISK) then do
        BASEPATH=BASEDRIVE'\OS2'
        BASEDLLPATH=BASEDRIVE'\OS2\DLL'
    end
    else do
        BASEPATH=BASEDRIVE
        BASEDLLPATH=BASEDRIVE
    end

    /* Drive and paths for files that can reside on boot image or on CD (MAKEROOM) */
    RESTDRIVE=destinationDrive(ifSet(MAKEROOM,'CD','IMG'))
    RESTPATH=RESTDRIVE'\OS2'
    RESTDLLPATH=RESTDRIVE'\OS2\DLL'

    /* Other drives and paths */
    NETWORKPATH=destinationDrive(NETWORKINGDESTINATION)'\NETWORK'
    HPFS386PATH=destinationDrive('IMG')'\IBM386FS'
    EXTRADEVICEPATH=destinationDrive(EXTRADEVICEDESTINATION)'\OS2'
    WPSPATH=destinationDrive(WPSDESTINATION)'\OS2'
    WPSAPPSPATH=destinationDrive(WPSAPPSDESTINATION)'\OS2\APPS'

    /* Truncate file names only if they are written to the boot image */
    if isSet(MAKEROOM) then
        TRUNCATE=''
    else
        TRUNCATE='T'

    /* Show all output of commands executed from the config.sys? */
    if isSet(BOOTMESSAGES) then
        IGNORECMDOUTPUT=''
    else
        IGNORECMDOUTPUT='> /dev/nul'

    /* Allow access to >16MB early in boot? */
    if isSet(EARLYMEMINIT) then
        rc=configLine('EARLYMEMINIT=TRUE')

    /* Memory Disk preparations */
    if isSet(MEMDISK) then do
        if isSet(LVM) then
            rc=configLine('BASEDEV='filename(MEMDISKADDFILE, 'T') MEMDISKADDOPTIONS)
        do I=1 to TARDIRS.0
            rc=configLine('DEVICE='BASEDRIVE'\loaddll.sys'  BASEDRIVE'\untar.dll untar' BASEDRIVE'\'TARDIRS.I'.tar')
            rc=configLine('DEVICE='BASEDRIVE'\erasefil.sys'                             BASEDRIVE'\'TARDIRS.I'.tar')
        end
        if TARDIRS.0>0 then do
            rc=configLine('DEVICE='BASEDRIVE'\erasefil.sys' BASEDRIVE'\untar.dll')
            rc=configLine('DEVICE='BASEDRIVE'\erasefil.sys' BASEDRIVE'\untar.msg')
            rc=configLine('DEVICE='BASEDRIVE'\erasefil.sys' BASEDRIVE'\loaddll.sys')
            rc=configLine('DEVICE='BASEDRIVE'\erasefil.sys' BASEDRIVE'\loaddll.msg')
        end
    end

    /* Reservedriveletter */
    if RESERVEDRIVELETTER<>'' then
        rc=configLine('RESERVEDRIVELETTER='RESERVEDRIVELETTER)

    /* Paths */
    LIBPATHLINE='LIBPATH=.;'||,
        ifSet(CREATEBOOTCD,BASEDLLPATH||ifClause(BASEDLLPATH=BASEDRIVE,'\;'BASEDRIVE'\OS2\DLL')';'ifClause(BASEDRIVE'\OS2\DLL'<>RESTDLLPATH,RESTDLLPATH';'))||,
        CDROMDRIVE'\OS2\DLL;'||,
        ifSet(WPSAPPS,WPSAPPSPATH'\DLL;')||,
        ifSet(NETWORKING,NETWORKPATH'\DLL;')||,
        ifSet(HPFS386,HPFS386PATH';')
    PATHLINE='SET PATH='||,
        ifSet(CREATEBOOTCD,BASEPATH||ifClause(BASEPATH=BASEDRIVE,'\;'BASEDRIVE'\OS2')';'ifClause(BASEDRIVE'\OS2'<>RESTPATH,RESTPATH';'))||,
        CDROMDRIVE'\os2;'||,
        ifSet(WPSAPPS,WPSAPPSPATH';')||,
        ifSet(NETWORKING,NETWORKPATH'\BIN;')||,
        ifSet(HPFS386,HPFS386PATH';')
    DPATHLINE='SET DPATH='||,
        ifSet(CREATEBOOTCD,BASEPATH||ifClause(BASEPATH=BASEDRIVE,'\;'BASEDRIVE'\OS2')';'ifClause(BASEDRIVE'\OS2'<>RESTPATH,RESTPATH';'))||,
        CDROMDRIVE'\os2;'||,
        ifSet(WPSAPPS,WPSAPPSPATH';')||,
        ifSet(NETWORKING,NETWORKPATH';')||,
        ifSet(HPFS386,HPFS386PATH';')
    HELPPATHLINE='SET HELP='WPSPATH'\HELP;'
    BOOKSHELFLINE='SET BOOKSHELF='WPSPATH'\HELP;'

    do I=1 to EXTRADIRECTORIES
        ADDTOPATHS=EXTRADIRECTORYADDTOPATHS.I
        do while words(ADDTOPATHS)>0
            ADDTOPATH=word(ADDTOPATHS,1)
            ADDTOPATHS=strip(delword(ADDTOPATHS,1,1))
            ADDTOPATHTXT=destinationDrive(EXTRADIRECTORYDESTINATION.I)||EXTRADIRECTORY.I||';'
            select
                when ADDTOPATH='LIBPATH'   & pos(translate(ADDTOPATHTXT),translate(LIBPATHLINE))=0   then LIBPATHLINE=LIBPATHLINE||ADDTOPATHTXT
                when ADDTOPATH='PATH'      & pos(translate(ADDTOPATHTXT),translate(PATHLINE))=0      then PATHLINE=PATHLINE||ADDTOPATHTXT
                when ADDTOPATH='DPATH'     & pos(translate(ADDTOPATHTXT),translate(DPATHLINE))=0     then DPATHLINE=DPATHLINE||ADDTOPATHTXT
                when ADDTOPATH='HELP'      & pos(translate(ADDTOPATHTXT),translate(HELPPATHLINE))=0  then HELPPATHLINE=HELPPATHLINE||ADDTOPATHTXT
                when ADDTOPATH='BOOKSHELF' & pos(translate(ADDTOPATHTXT),translate(BOOKSHELFLINE))=0 then BOOKSHELFLINE=BOOKSHELFLINE||ADDTOPATHTXT
                otherwise ;
            end
        end
    end
    rc=configLine(LIBPATHLINE)
    rc=configLine(PATHLINE)
    rc=configLine(DPATHLINE)
    if isSet(WPS) then do
        rc=configLine(HELPPATHLINE)
        rc=configLine(BOOKSHELFLINE)
    end
    

    /* Unicode */
    if isSet(UNICODE) then do
        rc=configLine('SET LANG='UNILANG)
        rc=configLine('SET ULSPATH='BASEDRIVE'\language')
        rc=configLine('DEVICE='BASEPATH'\'fileName(UNICODEFILE,'T'))
    end

    /* File systems */
    if isSet(CDFS)    then rc=configLine('IFS='BASEPATH'\'fileName(CDFSIFSFILE,'T')      CDFSOPTIONS)
    if isSet(FAT32)   then rc=configLine('IFS='BASEPATH'\'fileName(FAT32IFSFILE,'T')     FAT32OPTIONS)
    if isSet(FAT32)   then rc=configLine('CALL='RESTPATH'\'fileName(FAT32CACHEFILE,'T') FAT32CACHEOPTIONS)
    if isSet(NTFS)    then rc=configLine('IFS='BASEPATH'\'fileName(NTFSIFSFILE,'T')      NTFSOPTIONS)
    if isSet(HPFS)    then rc=configLine('IFS='BASEPATH'\'fileName(HPFSIFSFILE,'T')      HPFSOPTIONS)
    if isSet(HPFS386) then rc=configLine('IFS='HPFS386PATH'\'fileName(HPFS386IFSFILE,'T')  HPFS386OPTIONS)
    if isSet(JFS)     then rc=configLine('IFS='BASEPATH'\'fileName(JFSIFSFILE,'T')       JFSOPTIONS)
    if isSet(UDF)     then rc=configLine('IFS='BASEPATH'\'fileName(UDFIFSFILE,'T')       UDFOPTIONS)

    /* Ramdisk */
    if isSet(RAMDISK) then do
        rc=configLine('IFS='BASEPATH'\'fileName(RAMDISKIFSFILE,'T') RAMDISKOPTIONS)
        rc=configLine('CALL='RESTPATH'\'filename(RAMDISKFILE,TRUNCATE) RAMDISKDRIVE 'RAMDISK' IGNORECMDOUTPUT)
    end

    /* Base device driver for keyboard */
    rc=configLine('BASEDEV=ibmkbd.sys')

    /* Base device for chkdsk large disk support */
    rc=configLine('BASEDEV=chkdsk.sys')

    /* DASD Fallback */
    if isSet(DASDFALLBACK)                  then rc=configLine('BASEDEV='fileName(DASDFALLBACKFILE,'T')        DASDFALLBACKOPTIONS)
    if isSet(DASDFALLBACK) & isSet(MEMDISK) then rc=configLine('BASEDEV='fileName(MEMDISKDASDFALLBACKFILE,'T') MEMDISKDASDFALLBACKOPTIONS)

    /* Non-512 bytes/sector removable media support */
    if isSet(NON512DASD) then rc=configLine('BASEDEV='fileName(NON512DASDFILE) NON512DASDOPTIONS)

    /* Removable SCSI devices geometry filter support */
    if isSet(REMOVABLESCSI) then rc=configLine('BASEDEV='fileName(REMOVABLESCSIFILE,'T') REMOVABLESCSIOPTIONS)

    /* Base device drivers for storage devices */
    if isSet(FLOPPY) & (isSet(LVM) | \isSet(MEMDISK)) then rc=configLine('BASEDEV='fileName(FLOPPYADDFILE,'T') FLOPPYOPTIONS)
    if isSet(IDE) & \isSet(SCSIBEFOREIDE)             then rc=configLine('BASEDEV='fileName(IDEADDFILE,'T')    IDEOPTIONS)
    if isSet(SCSIDMD)                                 then rc=configLine('BASEDEV='fileName(SCSIDMDFILE,'T')   SCSIDMDOPTIONS)
    do I=1 to SCSI
        rc=configLine('BASEDEV='fileName(SCSIADDFILE.I,'T') SCSIOPTIONS.I)
    end
    if isSet(IDE) & isSet(SCSIBEFOREIDE) then rc=configLine('BASEDEV='fileName(IDEADDFILE,'T') IDEOPTIONS)
    if isSet(ATAPI) then do
        rc=configLine('BASEDEV='fileName(ATAPIFLTFILE,'T') ATAPIOPTIONS)
        if \isSet(DANIATAPI) & \isSet(LATESTATAPI) then rc=configLine('BASEDEV='fileName(ATAPICDFLTFILE,'T') ATAPICDOPTIONS)
    end
    if \isSet(LVM) & isSet(MEMDISK) then do
        rc=configLine('BASEDEV='filename(MEMDISKADDFILE, 'T') MEMDISKADDOPTIONS)
        if isSet(FLOPPY)   then rc=configLine('BASEDEV='fileName(FLOPPYADDFILE,'T')    FLOPPYOPTIONS)
    end
    if isSet(PPZIP)        then rc=configLine('BASEDEV='fileName(PPZIPADDFILE,'T')     PPZIPOPTIONS)
                                rc=configLine('BASEDEV='fileName(DASDFILE,'T')         DASDOPTIONS)
    if isSet(LVM)          then rc=configLine('BASEDEV='fileName(LVMDMDFILE,'T')       LVMOPTIONS)

    /* Extra base device drivers */
    do I=1 to EXTRABASEDEV
        rc=configLine('BASEDEV='fileName(EXTRABASEDEVFILE.I,'T') EXTRABASEDEVOPTIONS.I)
    end
    
    /* Base device driver providing bootable CD support */
    if isSet(CREATEBOOTCD) & \isSet(MEMDISK) then rc=configLine('BASEDEV='fileName(BOOTFLTFILE,'T') BOOTOPTIONS)

    /* Codepage, display, and keyboard layout */
    rc=configLine('COUNTRY='COUNTRY','BASEPATH'\country.sys')
    rc=configLine('CODEPAGE='CODEPAGE)
    rc=configLine('DEVINFO=KBD,'KEYBOARD','BASEPATH'\keyboard.dcp')
    rc=configLine('DEVINFO=SCR,VGA,'BASEPATH'\viotbl.dcp')

    /* CDROM support */
    if isSet(CDROM) then rc=configLine('DEVICE='BASEPATH'\'fileName(CDROMDMDFILE,'T') CDROMOPTIONS)

    /* ASPI support */
    if isSet(ASPI)  then rc=configLine('BASEDEV='fileName(ASPIDMDFILE,'T') ASPIOPTIONS)

    /* Test device driver (required?) */
    rc=configLine('DEVICE='RESTPATH'\testcfg.sys')

    /* DOS.SYS device driver */
    rc=configLine('DEVICE='RESTPATH'\dos.sys')

    /* Mouse support */
    if isSet(MOUSE) then do
        rc=configLine('DEVICE='RESTPATH'\'fileName(MOUSEPOINTDDFILE,TRUNCATE))
        if MOUSEDEVFILE<>'' then
            rc=configLine('DEVICE='RESTPATH'\'fileName(MOUSEDEVFILE,TRUNCATE) MOUSEDEVFILEOPTIONS)
        rc=configLine('DEVICE='RESTPATH'\'fileName(MOUSEFILE,TRUNCATE) MOUSEFILEOPTIONS)
    end
    
    /* USB */
    if isSet(USB) then do
        do USBOHCI
            rc=configLine('BASEDEV='fileName(USBOHCIFILE,'T') USBOHCIOPTIONS)
        end
        do USBUHCI
            rc=configLine('BASEDEV='fileName(USBUHCIFILE,'T') USBUHCIOPTIONS)
        end
        do USBEHCI
            rc=configLine('BASEDEV='fileName(USBEHCIFILE,'T') USBEHCIOPTIONS)
        end
        rc=configLine('BASEDEV='fileName(USBDFILE,  'T') USBDOPTIONS)
        rc=configLine('BASEDEV='fileName(USBHIDFILE,'T') USBHIDOPTIONS)
        if isSet(USBKEYB)    then rc=configLine('DEVICE='RESTPATH'\'fileName(USBKEYBFILE,   TRUNCATE) USBKEYBOPTIONS)
        if isSet(USBCOM)     then rc=configLine('DEVICE='RESTPATH'\'fileName(USBCOMFILE,    TRUNCATE) USBCOMOPTIONS)
        if isSet(USBMOUSE)   then rc=configLine('DEVICE='RESTPATH'\'fileName(USBMOUSEFILE,  TRUNCATE) USBMOUSEOPTIONS)
        if isSet(USBPRINTER) then rc=configLine('DEVICE='RESTPATH'\'fileName(USBPRINTERFILE,TRUNCATE) USBPRINTEROPTIONS)
        if isSet(USBCDROM)   then rc=configLine('BASEDEV='fileName(USBCDROMFILE,'T')   USBCDROMOPTIONS)
        if isSet(USBSTORAGE) then rc=configLine('BASEDEV='fileName(USBSTORAGEFILE,'T') USBSTORAGEOPTIONS)
    end
    
    /* Serial port support */
    if isSet(SERIALPORT) then
        rc=configLine('DEVICE='RESTPATH'\'fileName(SERIALPORTFILE,TRUNCATE) SERIALPORTOPTIONS)

    /* SIO Serial port support */
    if isSet(SIOSERIALPORT) then
        rc=configLine('DEVICE='RESTPATH'\'fileName(SIOSERIALPORTFILE,TRUNCATE) SIOSERIALPORTOPTIONS)

    /* SIO2K Serial port support */
    if isSet(SIO2KSERIALPORT) then do
        if isSet(SIO2KPATCH) then
            rc=configLine('DEVICE='RESTPATH'\'fileName(SIO2KPATCHFILE,TRUNCATE))
        rc=configLine('DEVICE='RESTPATH'\'fileName(SIO2KUARTFILE,TRUNCATE) SIO2KUARTOPTIONS)
        rc=configLine('DEVICE='RESTPATH'\'fileName(SIO2KSERIALPORTFILE,TRUNCATE) SIO2KSERIALPORTOPTIONS)
    end

    /* Parallel port support */
    if isSet(PARALLELPORT) then do
        rc=configLine('BASEDEV='filename(PARALLELPORTFILE,TRUNCATE) PARALLELPORTOPTIONS)
        rc=configLine('PRINTMONBUFSIZE=134,134,134')
    end

    /* APM */
    if isSet(APM) then do
        rc=configLine('DEVICE='RESTPATH'\'filename(APMFILE,TRUNCATE) APMOPTIONS)
        if APMDAEMONFILE<>'' then rc=configLine('RUN='RESTPATH'\'filename(APMDAEMONFILE,TRUNCATE) APMDAEMONOPTIONS)
    end

    /* Temp directory */
    if \isSet(CREATEBOOTCD) | isSet(RAMDISK) | isSet(MEMDISK) then do
        if \isSet(CREATEBOOTCD) then do
            TEMPPATH=WRITERDRIVE'\Temp'
            rc=createDir(TEMPPATH)
        end
        else do
            TEMPPATH=RAMDISKDRIVE'\Temp'
            rc=configLine('CALL='filename(SHELLFILE,TRUNCATE) '/c mkdir' TEMPPATH IGNORECMDOUTPUT)
        end
        rc=configLine('SET TEMP='TEMPPATH)
        rc=configLine('SET TMP='TEMPPATH)
        rc=configLine('SUPPRESSPOPUPS='left(TEMPPATH,1))
    end

    /* WPS */
    if isSet(WPS) then do
        rc=configLine('DEVICE='WPSPATH'\pmdd.sys')
        if isSet(CREATEBOOTCD) & (WPSDESTINATION<>'IMG' | isSet(RAMDISK)) then do
            rc=configLine('CALL='filename(SHELLFILE,TRUNCATE) '/c if not exist' RAMDISKDRIVE'\OS2 mkdir' RAMDISKDRIVE'\OS2' IGNORECMDOUTPUT)
            rc=configLine('CALL='filename(SHELLFILE,TRUNCATE) '/c copy'  WPSPATH'\*.ini' RAMDISKDRIVE'\OS2' IGNORECMDOUTPUT)
        end
        rc=configLine('SET USER_INI='WPSDRIVE'\os2\os2.ini')
        rc=configLine('SET SYSTEM_INI='WPSDRIVE'\os2\os2sys.ini')
        rc=configLine('SET AUTOSTART='AUTOSTART)
        rc=configLine('SET RESTARTOBJECTS=STARTUPFOLDERSONLY')
        if isSet(WPSAPPS) then rc=configLine('SET EPMPATH='WPSAPPSPATH)
    end

    /* Networking */
    if isSet(NETWORKING) then do
        rc=configLine('CALL='NETWORKPATH'\netbind.exe')
        rc=configLine('DEVICE='NETWORKPATH'\protman.os2  /I:'NETWORKPATH'')

        /* TCP/IP */
        if isSet(TCPIP) then do
            rc=configLine('DEVICE='NETWORKPATH'\sockets.sys')
            rc=configLine('DEVICE='NETWORKPATH'\aflean.sys')
            rc=configLine('RUN='NETWORKPATH'\bin\cntrl.exe')
            rc=configLine('SET HOSTNAME='TCPIPHOSTNAME)
            rc=configLine('SET USE_HOSTS_FIRST=1')
            ETCDIR=NETWORKPATH'\etc'
            if isSet(CREATEBOOTCD) & isSet(RAMDISK) & NETWORKINGDESTINATION='CD' then do
                rc=configLine('CALL='filename(SHELLFILE,TRUNCATE) '/c mkdir' RAMDISKDRIVE'\ETC' IGNORECMDOUTPUT)
                rc=configLine('CALL='filename(SHELLFILE,TRUNCATE) '/c xcopy' ETCDIR'\*' RAMDISKDRIVE'\ETC' IGNORECMDOUTPUT)
                ETCDIR=RAMDISKDRIVE'\ETC'
            end
            rc=configLine('SET ETC='ETCDIR)
            rc=configLine('CALL='filename(SHELLFILE,TRUNCATE) '/c' NETWORKPATH'\bin\tcpip.cmd' IGNORECMDOUTPUT)
        end

        /* NETBIOS */
        if isSet(NETBIOS) then do
            rc=configLine('DEVICE='NETWORKPATH'\netbeui.os2')
        end

        /* NETBIOS over TCP/IP */
        if isSet(NETBIOSTCPIP) then do
            rc=configLine('DEVICE='NETWORKPATH'\tcpbeui.os2')
            rc=configLine('RUN='NETWORKPATH'\nbtcp.exe')
        end

        /* NETBIOS or NETBIOS over TCP/IP */
        if isSet(NETBIOS) | isSet(NETBIOSTCPIP) then do
            rc=configLine('DEVICE='NETWORKPATH'\netbios.os2')
        end

        /* Card driver */
        rc=configLine('DEVICE='NETWORKPATH'\'fileName(NETWORKCARDFILE))
    end

    /* Extra device drivers */
    do I=1 to EXTRADEVICE
        rc=configLine('DEVICE='EXTRADEVICEPATH'\'||fileName(EXTRADEVICEFILE.I,'T') EXTRADEVICEOPTIONS.I)
    end
    
    /* System settings */
    rc=configLine('MEMMAN='MEMMAN)
    if pos('NOSWAP',translate(MEMMAN))=0 then
        rc=configLine('SWAPPATH='SWAPPATH)
    rc=configLine('IOPL=YES')
    rc=configLine('PROTECTONLY='PROTECTONLY)
    rc=configLine('AUTOFAIL=YES') /* Necessary for WPS CD boot! */
    rc=configLine('THREADS='THREADS)
    rc=configLine('DISKCACHE='FATCACHEOPTIONS)
    rc=configLine('MAXWAIT=1')

    /* Display settings */    
    if isSet(WPS) then rc=configLine('SET DISPLAYTYPE=VGA')
    select
        when isSet(WPS) & VIDEODRIVER=2 then do
            rc=configLine('SET VIDEO_DEVICES=VIO_SVGA')
            rc=configLine('SET VIO_SVGA=DEVICE(BVHVGA,BVHSVGA)')
            rc=configLine('SET C1=GENGRADD')
            rc=configLine('SET GRADD_CHAINS=C1')
            rc=configLine('DEVICE='WPSPATH'\gradd.sys')
        end
        otherwise do
            rc=configLine('SET VIDEO_DEVICES=VIO_VGA')
            rc=configLine('SET VIO_VGA=DEVICE(BVHVGA)')
        end
    end

    /* Command environment */
    TZ=value('TZ',,'OS2ENVIRONMENT')
    if TZ<>'' then rc=configLine('SET TZ='TZ)
    rc=configLine('SET PROMPT='SHELLPROMPT)
    rc=configLine('SET KEYS=ON')

    /* Shell */
    rc=configLine('PROTSHELL='ifSet(HPFS386,HPFS386PATH'\bootsh.exe ')RESTPATH'\'fileName(PROTSHELLFILE,TRUNCATE) PROTSHELLFILEOPTIONS)
    if isSet(WPS) then do
        rc=configLine('SET RUNWORKPLACE='RESTPATH'\'fileName(WPSRUNWORKPLACEFILE,TRUNCATE) WPSRUNWORKPLACEFILEOPTIONS)
    end
    rc=configLine('SET OS2_SHELL='RESTPATH'\'fileName(SHELLFILE,TRUNCATE) SHELLFILEOPTIONS)
    rc=configLine('SET COMSPEC='RESTPATH'\'fileName(SHELLFILE,TRUNCATE))

    /* REXX */
    if isSet(REXX) & \isSet(WPSBOOT) then
        rc=configLine('RUN='destinationDrive(REXXDESTINATION)'\os2\'filename(REXXCOMMANDLINEFILE))

    /* Additional config.sys lines */
    rc=configLine(parseText(EXTRACONFIGSYS))
    
    /* Close config.sys */
    rc=lineout(VIRTUALDRIVE'\config.sys')
    
    return rc


/* Create INI files on the CD --------------------------------------------- */
createINIFiles:
    IMGNR=2
    call progress 1, 'Creating INI files on' destinationName()

    /* Create ini file from EXTRAxxxINI settings */
    if strip(EXTRAUSERINI)<>'' then do
        EXTRAUSERINIRCFILES=EXTRAUSERINIRCFILES+1
        EXTRAUSERINIRCFILE.EXTRAUSERINIRCFILES=TEMPDIR'\inirc.xtr'
        EXTRAUSERINIRCFILEUPDATE.EXTRAUSERINIRCFILES='YES'
        rc=createFile(EXTRAUSERINIRCFILE.EXTRAUSERINIRCFILES, EXTRAUSERINI)
    end
    if strip(EXTRASYSINI)<>'' then do
        EXTRASYSINIRCFILES=EXTRASYSINIRCFILES+1
        EXTRASYSINIRCFILE.EXTRASYSINIRCFILES=TEMPDIR'\sysinirc.xtr'
        EXTRASYSINIRCFILEUPDATE.EXTRASYSINIRCFILES='YES'
        rc=createFile(EXTRASYSINIRCFILE.EXTRASYSINIRCFILES, EXTRASYSINI)
    end

    /* Create the ini files based on the templates */
    do IRC=1 to EXTRAUSERINIRCFILES
        rc=parseINIRC(ifExists(EXTRAUSERINIRCFILE.IRC, EXTRAUSERINIRCFILE.IRC), TEMPDIR'\inirc.'||IRC, TEMPDIR'\os2.ini', EXTRAUSERINIRCFILEUPDATE.IRC)
    end

    do IRC=1 to EXTRASYSINIRCFILES
        rc=parseINIRC(ifExists(EXTRASYSINIRCFILE.IRC, EXTRASYSINIRCFILE.IRC), TEMPDIR'\sysinirc.'||IRC, TEMPDIR'\os2sys.ini', EXTRASYSINIRCFILEUPDATE.IRC)
    end

    /* Move the ini files to destination */
    rc=sCopyFile(TEMPDIR'\os2.ini', DESTDRIVE'\OS2')
    rc=sCopyFile(TEMPDIR'\os2sys.ini', DESTDRIVE'\OS2')

    return 0


/* Parse INI file template ------------------------------------------------ */
parseINIRC:
    SOURCERC=ARG(1)
    DESTINATIONRC=ARG(2)
    DESTINATIONINI=ARG(3)
    UPDATEINI=ARG(4)

    call progress 2, 'Parsing INI file template' SOURCERC ifSet(UPDATEINI,'(updating)')

    PREINITXT='CODEPAGE $$CODEPAGE$$'||CRLF||CRLF||,
            'STRINGTABLE'||CRLF||,
            'BEGIN'||CRLF||,
            '    ""'||CRLF||,
            '    ""'||CRLF||,
            '    ""'||CRLF||,
            '    ""'||CRLF||,
            '    ""'||CRLF||,
            '    ""'||CRLF||,
            '    ""'||CRLF||,
            '    ""'||CRLF||,
            '    ""'||CRLF||,
            '    ""'||CRLF||CRLF
    POSTINITXT=CRLF||'END'||CRLF

    drop INIXAPPLICATIONS.
    drop INIXKEYS.
    drop INIXVALUES.
    NRINIX=0

    drop INIAPPLICATIONS.
    drop INIKEYS.
    drop INIVALUES.
    NRINI=0

    if lines(SOURCERC)=0 then
        call error SOURCERC 'is empty or does not exist'

    if \isSet(UPDATEINI) then
        rc=createFile(DESTINATIONRC, PREINITXT)

    rc=0
    RCLINENR=0
    do while lines(SOURCERC)>0
        RCLINENR=RCLINENR+1
        ORGRCLINE=linein(SOURCERC)
        RCLINE=ORGRCLINE
        call progress 8, SOURCERC '('RCLINENR'):' RCLINE
        if \(left(strip(RCLINE),2)='/*' | left(strip(RCLINE),1)=';') then do
            RCLINE=parseText(RCLINE)

           if strip(RCLINE)<>'' then do
                parse value RCLINE with '"'RCAPPL'"' '"'RCKEY'"' RCVALUE
                RCVALUE=strip(RCVALUE)
                if RCVALUE='""' then RCVALUE='0x00' /* Convert empty string to hex 0x00 */
                RCLINEOK=(left(RCVALUE,1)='"' & right(RCVALUE,1)='"')
                RCLINEOK=RCLINEOK | (translate(left(RCVALUE,2))='0X' & length(RCVALUE)>2 & strip(translate(RCVALUE,'XX','xX 01234567890abcdefABCDEF',' '))='X')
                RCLINEOK=RCLINEOK & RCAPPL<>'' & RCKEY<>''
                if \RCLINEOK then
                    call error 'Syntax error in ini file template' SOURCERC 'Line:' RCLINENR||CRLF||'==>'||ORGRCLINE

                if translate(left(RCVALUE,2))='0X' then do
                    NRINIX=NRINIX+1
                    INIXAPPLICATIONS.NRINIX=RCAPPL
                    INIXKEYS.NRINIX=RCKEY
                    INIXVALUES.NRINIX=substr(RCVALUE,3)
                end
                else
                    if isSet(UPDATEINI) then do
                        NRINI=NRINI+1
                        INIAPPLICATIONS.NRINI=RCAPPL
                        INIKEYS.NRINI=RCKEY
                        INIVALUES.NRINI=strip(RCVALUE,,'"')
                    end
                    else do
                        RCAPPL='"'RCAPPL'"'
                        RCKEY='"'RCKEY'"'
                        if length(RCAPPL)<31 then RCAPPL=left(RCAPPL,31)
                        if length(RCKEY)<31  then RCKEY=left(RCKEY,31)
                        RCLINE='    'RCAPPL' 'RCKEY' 'RCVALUE
                        rc=lineout(DESTINATIONRC, RCLINE)
                    end
           end
        end
    end

    /* Close files */
    if \isSet(UPDATEINI) then do
        rc=lineout(DESTINATIONRC, POSTINITXT)
        rc=lineout(DESTINATIONRC)
        if rc<>0 then
            call error 'Failed to close file' DESTINATIONRC 'rc('rc')'
    end
    rc=stream(SOURCERC, 'C', 'CLOSE')
    if rc<>'READY:' & rc<>'' then
        call error 'Failed to close file' SOURCERC 'rc('rc')'

    /* Make ini */
    if \isSet(UPDATEINI) then do
        rc=runCMD('makeini' DESTINATIONINI DESTINATIONRC, 'Failed to create ini file using template' SOURCERC)
    end

    /* Set hex values */
    do I=1 to NRINIX
        /* Only write X value if in update mode, or not already existing */
        if isSet(UPDATEINI) | sysIni(DESTINATIONINI, INIXAPPLICATIONS.I, INIXKEYS.I)='ERROR:' then do
            rc=sysIni(DESTINATIONINI, INIXAPPLICATIONS.I, INIXKEYS.I, X2C(INIXVALUES.I))
            if rc<>'' then
                call error 'Failed to add hex value' INIXAPPLICATIONS.I||'('||INIXKEYS.I||')=0x'||INIXVALUES.I 'from template' SOURCERC 'to ini file' DESTINATIONINI', RC='RC
        end
    end

    /* Update values */
    do I=1 to NRINI
        rc=sysIni(DESTINATIONINI, INIAPPLICATIONS.I, INIKEYS.I, INIVALUES.I)
        if rc<>'' then
            call error 'Failed to add/update value' INIAPPLICATIONS.I||'('||INIKEYS.I||')='||INIVALUES.I 'from template' SOURCERC 'to ini file' DESTINATIONINI', RC='RC
    end
    
    return 0


/* Create networking files on the CD -------------------------------------- */
createNetworkFiles:
    call progress 1, 'Creating network files on' destinationName()

    if isSet(TCPIP) then do
        /* tcpip.cmd */
        TXT=''||,
          'inetwait'||CRLF||,
          'route -fh'||CRLF||,
          'arp -f'||CRLF||,
          'ifconfig lo 127.0.0.1'||CRLF
        if isSet(TCPIPDHCP) then
            TXT=TXT||'dhcpstrt -i lan0 -d 'TCPIPDHCPTIMEOUT||CRLF
        else
            TXT=TXT||'ifconfig lan0' TCPIPHOSTADDRESS||CRLF
        if  strip(TCPIPGATEWAYADDRESS)<>'' then
            TXT=TXT||'route add default' TCPIPGATEWAYADDRESS||CRLF
        rc=createFile(DESTDRIVE'\network\bin\tcpip.cmd',TXT)

        /* ETC Directory */
        rc=createDir(DESTDRIVE'\NETWORK\ETC')

        /* dhcpcd.cfg */
        if isSet(TCPIPDHCP) then do
            TXT=''||,
              'clientid   MAC'||CRLF||,
              'option 1                     # Subnet Mask'||CRLF||,
              'option 3                     # Router'||CRLF||,
              'option 6                     # Domain Name Server'||CRLF||,
              ifSet(TCPIPDHCPUPDATEHOST,'option 12 "'||TCPIPHOSTNAME||'" # Host Name'||CRLF)||,
              'option 15                    # Domain Name'||CRLF||,
              'option 28                    # Broadcast Address'||CRLF||,
              'option 33                    # Static Routes'||CRLF||,
              'option 60 "IBMWARP_V4.1"     # Vendor Class'||CRLF||,
              'option 77 "IBMWARP_V4.1"     # User Class'||CRLF
            rc=createFile(DESTDRIVE'\network\etc\dhcpcd.cfg',TXT)
        end

        /* protocol */
        TXT=''||,
          'icmp  1 ICMP'||CRLF||,
          'ip    4 IP'||CRLF||,
          'tcp   6 TCP'||CRLF||,
          'udp  17 UDP'||CRLF
        rc=createFile(DESTDRIVE'\network\etc\protocol',TXT)

        /* services */
        TXT=''||,
          'ftp-data         20/tcp'||CRLF||,
          'ftp-data         20/udp'||CRLF||,
          'ftp              21/tcp'||CRLF||,
          'ftp              21/udp'||CRLF||,
          'telnet           23/tcp'||CRLF||,
          'telnet           23/udp'||CRLF||,
          'name             42/tcp'||CRLF||,
          'name             42/udp'||CRLF||,
          'nameserver       42/tcp'||CRLF||,
          'nameserver       42/udp'||CRLF||,
          'domain           53/tcp'||CRLF||,
          'domain           53/udp'||CRLF||,
          'tftp             69/tcp'||CRLF||,
          'tftp             69/udp'||CRLF||,
          'finger           79/tcp'||CRLF||,
          'finger           79/udp'||CRLF||,
          'hostname        101/tcp'||CRLF||,
          'hostname        101/udp'||CRLF||,
          'exec            512/tcp'||CRLF||,
          'login           513/tcp'||CRLF||,
          'shell           514/tcp'||CRLF
        rc=createFile(DESTDRIVE'\network\etc\services',TXT)

        /* hosts */
        TXT=''||,
          '127.0.0.1 localhost'||CRLF||,
          ifClause(TCPIPHOSTADDRESS<>'' & \isSet(TCPIPDHCP), TCPIPHOSTADDRESS TCPIPHOSTNAME'.'TCPIPDOMAINNAME||CRLF)||,
          TCPIPHOSTS
        rc=createFile(DESTDRIVE'\network\etc\hosts',TXT)

        /* resolv2 */
        if TCPIPDNSADDRESS<>'' then do
            TXT=''||,
              'domain' TCPIPDOMAINNAME||CRLF||,
              'nameserver' TCPIPDNSADDRESS||CRLF
            rc=createFile(DESTDRIVE'\network\etc\resolv2',TXT)
        end
    end

    /* protocol.ini */
    NAME=translate(fileName(NETWORKCARDFILE))
    IDX=pos('.',NAME)
    if IDX>0 then NAME=left(NAME,IDX-1)
    TXT=''||,
      '[PROTMAN]'||CRLF||CRLF||,
      '  DRIVERNAME = PROTMAN$'||CRLF||CRLF||,
      '[IBMLXCFG]'||CRLF||CRLF||,
      ifSet(TCPIP,       '  tcpip_nif = tcpip.nif'||CRLF)||,
      ifSet(NETBIOS,     '  netbeui_nif = netbeui.nif'||CRLF)||,
      ifSet(NETBIOSTCPIP,'  tcpbeui_nif = tcpbeui.nif'||CRLF)||,
      '  NETWORK_nif = NETWORK.nif'||CRLF
    if isSet(TCPIP) then 
        TXT=TXT||CRLF||,
          '[tcpip_nif]'||CRLF||CRLF||,
          '  DriverName = TCPIP$'||CRLF||,
          '  Bindings = NETWORK_nif'||CRLF
    if isSet(NETBIOS) | isSet(NETBIOSTCPIP) then
        TXT=TXT||CRLF||,
          '[NETBIOS]'||CRLF||CRLF||,
          '  DriverName = netbios$'||CRLF
    ADAPTER=0
    if isSet(NETBIOS) then do
        TXT=TXT||'  ADAPTER'ADAPTER' = netbeui$,'ADAPTER||CRLF
        ADAPTER=ADAPTER+1
    end
    if isSet(NETBIOSTCPIP) then
        TXT=TXT||'  ADAPTER'ADAPTER' = tcpbeui$,'ADAPTER||CRLF
    if isSet(NETBIOS) then
        TXT=TXT||CRLF||,
          '[netbeui_nif]'||CRLF||CRLF||,
          '  DriverName = netbeui$'||CRLF||,
          '  Bindings = NETWORK_nif'||CRLF
    if isSet(NETBIOSTCPIP) then
        TXT=TXT||CRLF||,
          '[tcpbeui_nif]'||CRLF||,
          ''||CRLF||,
          '   DriverName = tcpbeui$'||CRLF||,
          '   Bindings = NETWORK_nif'||CRLF||,
          '   NODETYPE = "B-Node"'||CRLF
    TXT=TXT||CRLF||,
      '[NETWORK_nif]'||CRLF||CRLF||,
      '  DriverName = 'NAME'$'||CRLF
    rc=createFile(DESTDRIVE'\network\protocol.ini',TXT)
    
    return 0


/* Create files on CD or image -------------------------------------------- */
createFiles:
    DEST=ARG(1)
    IMGNR=2

    if isSet(NETWORKING) & DEST=NETWORKINGDESTINATION then rc=createNetworkFiles()
    if isSet(WPS)        & DEST=WPSDESTINATION        then rc=createINIFiles()

    if isSet(WPS) & isSet(LVM) & DEST=WPSDESTINATION then do
        call progress 1, 'Creating lvm icons and command-files on' destinationName()
        TXT='lvm.exe /RediscoverPRM'
        rc=createFile(DESTDRIVE'\OS2\lvmrfrsh.cmd',TXT)
        rc=sCopyFile('bootAble.lvmrefresh.ico',DESTDRIVE'\OS2\lvmrfrsh.ico')
        TXT='lvm.exe'
        rc=createFile(DESTDRIVE'\OS2\lvmgui.cmd',TXT)
        rc=sCopyFile('bootAble.lvmgui.ico',DESTDRIVE'\OS2\lvmgui.ico')
    end

    FIRSTCREATED=0
    do I=1 to EXTRACREATEFILES
        if DEST=EXTRACREATEFILEDESTINATION.I then do
            if \FIRSTCREATED then do
                FIRSTCREATED=1
                call progress 1, 'Creating extra files on' destinationName()
            end
            rc=createDir(DESTDRIVE||EXTRACREATEFILEDIRECTORY.I)
            rc=createFile(DESTDRIVE||EXTRACREATEFILEDIRECTORY.I||'\'||EXTRACREATEFILENAME.I,parseText(EXTRACREATEFILECONTENT.I))
        end
    end

    return rc
