#include <ctype.h>
#include <search.h>
#include <stdio.h>                       
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>

#define INCL_DOSERRORS
#define INCL_DOSMEMMGR
#include <os2.h>

#include <uconv.h>

#define INCL_RXFUNC
#define INCL_RXSHV
#include <rexxsaa.h>

#include <ldap.h>

#include "rxstring.h"

#define CallingValid 0
#define CallingInvalid 40
#define Good 0
#define Bad 40

#define RxldapDll "RXLDAP"
#define RxldapVer "0.21" 

#define SetSignedInt2Rx(rxvar, intval) rxset_length( &(rxvar), strlen( _ltoa( (ULONG)intval, RXSTRPTR(rxvar), 10 ) ) )
#define SetUnsignedInt2Rx(rxvar, intval) rxset_length( &(rxvar), strlen( _ultoa( (ULONG)intval, RXSTRPTR(rxvar), 10 ) ) )
#define SetRetStr(rxvar, str) rxreturn_value(&(rxvar), (unsigned char*)str, strlen(str))
#define SetLdapCode(rxstr, code) setcodename( rxstr, code, LdapReturnCodes, sizeof(LdapReturnCodes) / sizeof(Name2Int), 0 )
#define SetIntFromName( n, t, v, i, o ) \
   v = findintfromname( n, t, sizeof(t) / sizeof(Name2Int), 1 ); \
   if ( !v ) { \
      return ( CallingInvalid ); \
   } /* endif */ \
   i = v->intval; \
   o = &i;

extern const char *RxldapDate;
extern const char *RxldapTime;

RexxFunctionHandler RxldapDropFuncs;
RexxFunctionHandler RxldapLoadFuncs;
RexxFunctionHandler RxldapVersion;
RexxFunctionHandler RxldapToUTF8;
RexxFunctionHandler RxldapFromUTF8;
RexxFunctionHandler Rxldap_abandon;
RexxFunctionHandler Rxldap_abandon_ext;
RexxFunctionHandler Rxldap_add;
RexxFunctionHandler Rxldap_add_ext;
RexxFunctionHandler Rxldap_add_ext_s;
RexxFunctionHandler Rxldap_add_s;
RexxFunctionHandler Rxldap_ber_free;
RexxFunctionHandler Rxldap_compare;
RexxFunctionHandler Rxldap_compare_ext;
RexxFunctionHandler Rxldap_compare_ext_s;
RexxFunctionHandler Rxldap_compare_s;
RexxFunctionHandler Rxldap_count_entries;
RexxFunctionHandler Rxldap_count_messages;
RexxFunctionHandler Rxldap_count_references;
RexxFunctionHandler Rxldap_delete;
RexxFunctionHandler Rxldap_delete_ext;
RexxFunctionHandler Rxldap_delete_ext_s;
RexxFunctionHandler Rxldap_delete_s;
RexxFunctionHandler Rxldap_err2string;
RexxFunctionHandler Rxldap_explode_dn;
RexxFunctionHandler Rxldap_first_attribute;
RexxFunctionHandler Rxldap_first_entry;
RexxFunctionHandler Rxldap_first_message;
RexxFunctionHandler Rxldap_first_reference;
RexxFunctionHandler Rxldap_get_dn;
RexxFunctionHandler Rxldap_get_errno;
RexxFunctionHandler Rxldap_get_option;
RexxFunctionHandler Rxldap_get_values;
RexxFunctionHandler Rxldap_get_values_len;
RexxFunctionHandler Rxldap_init;
RexxFunctionHandler Rxldap_is_ldap_url;
RexxFunctionHandler Rxldap_modify;
RexxFunctionHandler Rxldap_modify_ext;
RexxFunctionHandler Rxldap_modify_ext_s;
RexxFunctionHandler Rxldap_modify_s;
RexxFunctionHandler Rxldap_msgfree;
RexxFunctionHandler Rxldap_msgid;
RexxFunctionHandler Rxldap_msgtype;
RexxFunctionHandler Rxldap_next_attribute;
RexxFunctionHandler Rxldap_next_entry;
RexxFunctionHandler Rxldap_next_message;
RexxFunctionHandler Rxldap_next_reference;
RexxFunctionHandler Rxldap_parse_result;
RexxFunctionHandler Rxldap_rename;
RexxFunctionHandler Rxldap_rename_s;
RexxFunctionHandler Rxldap_result;
RexxFunctionHandler Rxldap_search;
RexxFunctionHandler Rxldap_search_ext;
RexxFunctionHandler Rxldap_search_ext_s;
RexxFunctionHandler Rxldap_search_s;
RexxFunctionHandler Rxldap_search_st;
RexxFunctionHandler Rxldap_set_option;
RexxFunctionHandler Rxldap_simple_bind;
RexxFunctionHandler Rxldap_simple_bind_s;
RexxFunctionHandler Rxldap_unbind;
RexxFunctionHandler Rxldap_unbind_s;
RexxFunctionHandler Rxldap_url_parse;
RexxFunctionHandler Rxldap_url_search;
RexxFunctionHandler Rxldap_url_search_s;
RexxFunctionHandler Rxldap_url_search_st;
RexxFunctionHandler Rxldap_version;

static const PSZ RxldapFuncsTable[] = {
   "RxldapDropFuncs",
   "RxldapFromUTF8",
   "RxldapLoadFuncs",
   "RxldapToUTF8",
   "RxldapVersion",
   "Rxldap_abandon",
   "Rxldap_abandon_ext",
   "Rxldap_add",
   "Rxldap_add_ext",
   "Rxldap_add_ext_s",
   "Rxldap_add_s",
   "Rxldap_ber_free",
   "Rxldap_compare",
   "Rxldap_compare_ext",
   "Rxldap_compare_ext_s",
   "Rxldap_compare_s",
   "Rxldap_count_entries",
   "Rxldap_count_messages",
   "Rxldap_count_references",
   "Rxldap_delete",
   "Rxldap_delete_ext",
   "Rxldap_delete_ext_s",
   "Rxldap_delete_s",
   "Rxldap_err2string",
   "Rxldap_explode_dn",
   "Rxldap_first_attribute",
   "Rxldap_first_entry",
   "Rxldap_first_message",
   "Rxldap_first_reference",
   "Rxldap_get_dn",
   "Rxldap_get_errno",
   "Rxldap_get_option",
   "Rxldap_get_values",
   "Rxldap_get_values_len",
   "Rxldap_init",
   "Rxldap_is_ldap_url",
   "Rxldap_modify",
   "Rxldap_modify_ext",
   "Rxldap_modify_ext_s",
   "Rxldap_modify_s",
   "Rxldap_msgfree",
   "Rxldap_msgid",
   "Rxldap_msgtype",
   "Rxldap_next_attribute",
   "Rxldap_next_entry",
   "Rxldap_next_message",
   "Rxldap_next_reference",
   "Rxldap_parse_result",
   "Rxldap_rename",
   "Rxldap_rename_s",
   "Rxldap_result",
   "Rxldap_search",
   "Rxldap_search_ext",
   "Rxldap_search_ext_s",
   "Rxldap_search_s",
   "Rxldap_search_st",
   "Rxldap_set_option",
   "Rxldap_simple_bind",
   "Rxldap_simple_bind_s",
   "Rxldap_unbind",
   "Rxldap_unbind_s",
   "Rxldap_url_parse",
   "Rxldap_url_search",
   "Rxldap_url_search_s",
   "Rxldap_url_search_st",
   "Rxldap_version",
};

typedef struct {
   char *name;
   int intval;
} Name2Int;   

typedef struct {
   char *name;
   char *strval;
} Name2Str;   

static const Name2Int ResAllNames[] = {
   {"LDAP_MSG_ALL", LDAP_MSG_ALL},
   {"LDAP_MSG_ONE", LDAP_MSG_ONE},
   {"LDAP_MSG_RECEIVED", LDAP_MSG_RECEIVED},
};   

static const Name2Int OpNames[] = {
   {"LDAP_MOD_ADD", LDAP_MOD_ADD},
   {"LDAP_MOD_ADD|LDAP_MOD_BVALUES", LDAP_MOD_ADD|LDAP_MOD_BVALUES},
   {"LDAP_MOD_DELETE", LDAP_MOD_DELETE},
   {"LDAP_MOD_DELETE|LDAP_MOD_BVALUES", LDAP_MOD_DELETE|LDAP_MOD_BVALUES},
   {"LDAP_MOD_REPLACE", LDAP_MOD_REPLACE},
   {"LDAP_MOD_REPLACE|LDAP_MOD_BVALUES", LDAP_MOD_REPLACE|LDAP_MOD_BVALUES},
};

static const Name2Int ScopeNames[] = {
   {"LDAP_SCOPE_DEFAULT", LDAP_SCOPE_DEFAULT},
   {"LDAP_SCOPE_BASE", LDAP_SCOPE_BASE},
   {"LDAP_SCOPE_ONELEVEL", LDAP_SCOPE_ONELEVEL},
   {"LDAP_SCOPE_SUBTREE", LDAP_SCOPE_SUBTREE}
};   

static const Name2Int ResultTypes[] = {
   {"LDAP_RES_ANY", LDAP_RES_ANY},
   {"LDAP_RES_UNSOLICITED", LDAP_RES_UNSOLICITED},
   {"LDAP_RES_BIND", LDAP_RES_BIND},
   {"LDAP_RES_SEARCH_ENTRY", LDAP_RES_SEARCH_ENTRY},
   {"LDAP_RES_SEARCH_RESULT", LDAP_RES_SEARCH_RESULT},
   {"LDAP_RES_MODIFY", LDAP_RES_MODIFY},
   {"LDAP_RES_ADD", LDAP_RES_ADD},
   {"LDAP_RES_DELETE", LDAP_RES_DELETE},
   {"LDAP_RES_RENAME", LDAP_RES_MODRDN},
   {"LDAP_RES_COMPARE", LDAP_RES_COMPARE},
   {"LDAP_RES_SEARCH_REFERENCE", LDAP_RES_SEARCH_REFERENCE},
   {"LDAP_RES_EXTENDED", LDAP_RES_EXTENDED},
   {"LDAP_RES_EXTENDED_PARTIAL", LDAP_RES_EXTENDED_PARTIAL},
};   

static const Name2Int LdapReturnCodes[] = {
   {"LDAP_SUCCESS", LDAP_SUCCESS},
   {"LDAP_OPERATIONS_ERROR", LDAP_OPERATIONS_ERROR},
   {"LDAP_PROTOCOL_ERROR", LDAP_PROTOCOL_ERROR},
   {"LDAP_TIMELIMIT_EXCEEDED", LDAP_TIMELIMIT_EXCEEDED},
   {"LDAP_SIZELIMIT_EXCEEDED", LDAP_SIZELIMIT_EXCEEDED},
   {"LDAP_COMPARE_FALSE", LDAP_COMPARE_FALSE},
   {"LDAP_COMPARE_TRUE", LDAP_COMPARE_TRUE},
   {"LDAP_AUTH_METHOD_NOT_SUPPORTED", LDAP_AUTH_METHOD_NOT_SUPPORTED},
   {"LDAP_STRONG_AUTH_REQUIRED", LDAP_STRONG_AUTH_REQUIRED},
   {"LDAP_PARTIAL_RESULTS", LDAP_PARTIAL_RESULTS},
   {"LDAP_REFERRAL", LDAP_REFERRAL},
   {"LDAP_ADMINLIMIT_EXCEEDED", LDAP_ADMINLIMIT_EXCEEDED},
   {"LDAP_UNAVAILABLE_CRITICAL_EXTENSION", LDAP_UNAVAILABLE_CRITICAL_EXTENSION},
   {"LDAP_CONFIDENTIALITY_REQUIRED", LDAP_CONFIDENTIALITY_REQUIRED},
   {"LDAP_SASL_BIND_IN_PROGRESS", LDAP_SASL_BIND_IN_PROGRESS},
   {"LDAP_NO_SUCH_ATTRIBUTE", LDAP_NO_SUCH_ATTRIBUTE},
   {"LDAP_UNDEFINED_TYPE", LDAP_UNDEFINED_TYPE},
   {"LDAP_INAPPROPRIATE_MATCHING", LDAP_INAPPROPRIATE_MATCHING},
   {"LDAP_CONSTRAINT_VIOLATION", LDAP_CONSTRAINT_VIOLATION},
   {"LDAP_TYPE_OR_VALUE_EXISTS", LDAP_TYPE_OR_VALUE_EXISTS},
   {"LDAP_INVALID_SYNTAX", LDAP_INVALID_SYNTAX},
   {"LDAP_NO_SUCH_OBJECT", LDAP_NO_SUCH_OBJECT},
   {"LDAP_ALIAS_PROBLEM", LDAP_ALIAS_PROBLEM},
   {"LDAP_INVALID_DN_SYNTAX", LDAP_INVALID_DN_SYNTAX},
   {"LDAP_IS_LEAF", LDAP_IS_LEAF},
   {"LDAP_ALIAS_DEREF_PROBLEM", LDAP_ALIAS_DEREF_PROBLEM},
   {"LDAP_INAPPROPRIATE_AUTH", LDAP_INAPPROPRIATE_AUTH},
   {"LDAP_INVALID_CREDENTIALS", LDAP_INVALID_CREDENTIALS},
   {"LDAP_INSUFFICIENT_ACCESS", LDAP_INSUFFICIENT_ACCESS},
   {"LDAP_BUSY", LDAP_BUSY},
   {"LDAP_UNAVAILABLE", LDAP_UNAVAILABLE},
   {"LDAP_UNWILLING_TO_PERFORM", LDAP_UNWILLING_TO_PERFORM},
   {"LDAP_LOOP_DETECT", LDAP_LOOP_DETECT},
   {"LDAP_NAMING_VIOLATION", LDAP_NAMING_VIOLATION},
   {"LDAP_OBJECT_CLASS_VIOLATION", LDAP_OBJECT_CLASS_VIOLATION},
   {"LDAP_NOT_ALLOWED_ON_NONLEAF", LDAP_NOT_ALLOWED_ON_NONLEAF},
   {"LDAP_NOT_ALLOWED_ON_RDN", LDAP_NOT_ALLOWED_ON_RDN},
   {"LDAP_ALREADY_EXISTS", LDAP_ALREADY_EXISTS},
   {"LDAP_NO_OBJECT_CLASS_MODS", LDAP_NO_OBJECT_CLASS_MODS},
   {"LDAP_RESULTS_TOO_LARGE", LDAP_RESULTS_TOO_LARGE},
   {"LDAP_AFFECTS_MULTIPLE_DSAS", LDAP_AFFECTS_MULTIPLE_DSAS},
   {"LDAP_OTHER", LDAP_OTHER},
   {"LDAP_SERVER_DOWN", LDAP_SERVER_DOWN},
   {"LDAP_LOCAL_ERROR", LDAP_LOCAL_ERROR},
   {"LDAP_ENCODING_ERROR", LDAP_ENCODING_ERROR},
   {"LDAP_DECODING_ERROR", LDAP_DECODING_ERROR},
   {"LDAP_TIMEOUT", LDAP_TIMEOUT},
   {"LDAP_AUTH_UNKNOWN", LDAP_AUTH_UNKNOWN},
   {"LDAP_FILTER_ERROR", LDAP_FILTER_ERROR},
   {"LDAP_USER_CANCELLED", LDAP_USER_CANCELLED},
   {"LDAP_PARAM_ERROR", LDAP_PARAM_ERROR},
   {"LDAP_NO_MEMORY", LDAP_NO_MEMORY},
   {"LDAP_CONNECT_ERROR", LDAP_CONNECT_ERROR},
   {"LDAP_NOT_SUPPORTED", LDAP_NOT_SUPPORTED},
   {"LDAP_CONTROL_NOT_FOUND", LDAP_CONTROL_NOT_FOUND},
   {"LDAP_NO_RESULTS_RETURNED", LDAP_NO_RESULTS_RETURNED},
   {"LDAP_MORE_RESULTS_TO_RETURN", LDAP_MORE_RESULTS_TO_RETURN},
   {"LDAP_CLIENT_LOOP", LDAP_CLIENT_LOOP},
   {"LDAP_REFERRAL_LIMIT_EXCEEDED", LDAP_REFERRAL_LIMIT_EXCEEDED},
};

static const Name2Int LdapOptNames[] = {
   {"LDAP_OPT_API_FEATURE_INFO", LDAP_OPT_API_FEATURE_INFO},
   {"LDAP_OPT_API_INFO", LDAP_OPT_API_INFO},
   {"LDAP_OPT_CLIENT_CONTROLS", LDAP_OPT_CLIENT_CONTROLS},
   {"LDAP_OPT_DEREF", LDAP_OPT_DEREF},
   {"LDAP_OPT_ERROR_NUMBER", LDAP_OPT_ERROR_NUMBER},
   {"LDAP_OPT_ERROR_STRING", LDAP_OPT_ERROR_STRING},
   {"LDAP_OPT_HOST_NAME", LDAP_OPT_HOST_NAME},
   {"LDAP_OPT_MATCHED_DN", LDAP_OPT_API_FEATURE_INFO},
   {"LDAP_OPT_PROTOCOL_VERSION", LDAP_OPT_PROTOCOL_VERSION},
   {"LDAP_OPT_REFERRALS", LDAP_OPT_REFERRALS},
   {"LDAP_OPT_RESTART", LDAP_OPT_RESTART},
   {"LDAP_OPT_SERVER_CONTROLS", LDAP_OPT_SERVER_CONTROLS},
   {"LDAP_OPT_SIZELIMIT", LDAP_OPT_SIZELIMIT},
   {"LDAP_OPT_TIMELIMIT", LDAP_OPT_TIMELIMIT},
};

static const Name2Int LdapOptValDeref[] = {
   {"LDAP_DEREF_ALWAYS", LDAP_DEREF_ALWAYS},
   {"LDAP_DEREF_FINDING", LDAP_DEREF_FINDING},
   {"LDAP_DEREF_NEVER", LDAP_DEREF_NEVER},
   {"LDAP_DEREF_SEARCHING", LDAP_DEREF_SEARCHING},
};   
    
static const Name2Int LdapOptValOnOff[] = {
   {"LDAP_OPT_OFF", (int)LDAP_OPT_OFF},
   {"LDAP_OPT_ON", (int)LDAP_OPT_ON},
};   
    
static BOOL isnumstr( const PCHAR str );
static VOID rxappendstr( PRXSTRING rxstr, const PCHAR str );
static int getvalue( const PRXSTRING name, PRXSTRING value);
static int setvalue( const PRXSTRING name, const PRXSTRING value);
static int parsetimeval( const PRXSTRING rxstr, struct timeval *tv );
static void setcodename( PRXSTRING rxstr, int code, const Name2Int *base, unsigned int num, int lflag );
static int _Optlink compname2intname( const void *item1, const void *item2 );
static int _Optlink compname2intint( const void *item1, const void *item2 );
static int _Optlink compname2strstr( const void *item1, const void *item2 );
static Name2Int *findintfromname( char *key, const Name2Int *base, unsigned int num, int lflag );
static Name2Int *findnamefromint( int key, const Name2Int *base, unsigned int num, int lflag );
static Name2Str *findnamefromstr( const char *key, const Name2Str *base, unsigned int num, int lflag );
static void freestrs( char **a );
static LDAPMod **getldapmods( PRXSTRING stemname );
static char **getldapattrs( PRXSTRING stemname );
static LDAPControl **getldapcontrols( PRXSTRING stemname );
static int setstrs( PRXSTRING stemname, char **strs, int utf8 );
static int setldapctrls( PRXSTRING stemname, LDAPControl **ldapctrl );
static unsigned char *strtoutf8( const unsigned char *str );
static unsigned char *utf8tostr( const unsigned char *utf8str );
static void freemem( void *p );

ULONG _Export
Rxldap_version(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   int code = 0;
   LDAPAPIInfo info;
   RXSTRING varname;
   RXSTRING stem;
   RXSTRING value;
   if ( argc != 1 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[0]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[0]) ) {
      return ( CallingInvalid );
   } /* endif */
   info.ldapai_info_version = LDAP_API_INFO_VERSION;
   code = ldap_get_option( NULL, LDAP_OPT_API_INFO, &info );
   varname = rxalloc( RXSTRLEN(argv[0]) + BUFSIZ );
   if ( RXNULLSTRING( varname ) ) {
      return ( CallingInvalid );
   } /* endif */
   stem = rxalloc( RXSTRLEN(argv[0]) + BUFSIZ );
   if ( RXNULLSTRING( stem ) ) {
      rxfree( varname );
      return ( CallingInvalid );
   } /* endif */
   value = rxalloc( BUFSIZ );
   if ( RXNULLSTRING( value ) ) {
      rxfree( varname );
      rxfree( stem );
      return ( CallingInvalid );
   } /* endif */
   rxstrcpy( &stem, &argv[0] );
   if ( stem.strptr[RXSTRLEN(stem)-1] != '.' ) {
      rxappendstr( &stem, "." );
   } /* endif */
   rxappendstr( &stem, "!" );
   rxstrcpy( &varname, &stem );
   rxappendstr( &varname, "SDK_VERSION" );
   rxset_length( &value, strlen( _ltoa( (ULONG)info.ldapai_vendor_version, RXSTRPTR(value), 10 ) ) );
   if ( setvalue( &varname, &value ) != Good ) {
      rxfree( varname );
      rxfree( stem );
      rxfree( value );
      return ( CallingInvalid );
   } /* endif */
   rxstrcpy( &varname, &stem );
   rxappendstr( &varname, "PROTOCOL_VERSION" );
   rxset_length( &value, strlen( _ltoa( (ULONG)info.ldapai_protocol_version, RXSTRPTR(value), 10 ) ) );
   if ( setvalue( &varname, &value ) != Good ) {
      rxfree( varname );
      rxfree( stem );
      rxfree( value );
      return ( CallingInvalid );
   } /* endif */
   rxfree( varname );
   rxfree( stem );
   rxfree( value );
   SetUnsignedInt2Rx( *retstr, code );
   return ( CallingValid );
}

ULONG _Export
RxldapToUTF8(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   unsigned char *utf8str = NULL;
   if ( argc != 1 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[0]) || !RXSTRLEN(argv[0]) ) {
      return ( CallingInvalid );
   } /* endif */
   utf8str = strtoutf8( RXSTRPTR(argv[0]) );
   if ( utf8str ) {
      rxreturn_value( retstr, utf8str, strlen(utf8str) );
      free( utf8str );
   } else {
      rxset_zerolen( retstr );
   } /* endif */
   return ( CallingValid );
}

ULONG _Export
RxldapFromUTF8(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   unsigned char *str = NULL;
   if ( argc != 1 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[0]) || !RXSTRLEN(argv[0]) ) {
      return ( CallingInvalid );
   } /* endif */
   str = utf8tostr( RXSTRPTR(argv[0]) );
   if ( str ) {
      rxreturn_value( retstr, str, strlen(str) );
      free( str );
   } else {
      rxset_zerolen( retstr );
   } /* endif */
   return ( CallingValid );
}

ULONG _Export
Rxldap_init(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   int port = LDAP_PORT;
   LDAP *ld = NULL;
   if ( argc < 1 || argc > 2 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[0]) || !RXSTRLEN(argv[0]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( 2 == argc && !isnumstr(RXSTRPTR(argv[1])) ) {
      if ( !isnumstr(RXSTRPTR(argv[1])) ) {
         return ( CallingInvalid );
      } /* endif */
      port = rxtoi( argv[1] );
   } /* endif */
   ld = ldap_init( RXSTRPTR(argv[0]), port );
   SetUnsignedInt2Rx( *retstr, ld );
   return ( CallingValid );
}

ULONG _Export 
RxldapLoadFuncs(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   static const size_t n = sizeof(RxldapFuncsTable) / sizeof(RxldapFuncsTable[0]);
   RXSTRING a;
   int i = 0;
   if ( argc > 0 ) {
      return ( CallingInvalid );
   } /* endif */
   for ( i = 0; i < n; i += 1 ) {
      ULONG code = RexxDeregisterFunction(RxldapFuncsTable[i]);
      switch (code) {
      case RXFUNC_OK:
      case RXFUNC_DEFINED:
      case RXFUNC_NOTREG:
         break;
      default:
         return ( CallingInvalid );
      } /* endswitch */
      code = RexxRegisterFunctionDll(RxldapFuncsTable[i], RxldapDll, RxldapFuncsTable[i]);
      if ( code != RXFUNC_OK ) {
         return ( CallingInvalid );
      } /* endif */
   } /* endfor */
   i = sprintf(RXSTRPTR(*retstr), "RXLDAP Version: %s (%s %s)", RxldapVer, RxldapDate, RxldapTime);
   if ( i + 1 > RXSTRLEN(*retstr) ) {
      return ( CallingInvalid );
   } /* endif */
   rxset_length( retstr, i );
   return ( CallingValid );
}

ULONG _Export  
RxldapDropFuncs(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   static const size_t n = sizeof(RxldapFuncsTable) / sizeof(RxldapFuncsTable[0]);
   int i = 0;
   if (argc > 0) {
      return ( CallingInvalid );
   } /* endif */
   for ( i = 0; i < n; i += 1 ) {
      ULONG code = RexxDeregisterFunction(RxldapFuncsTable[i]);
      switch (code) {
      case RXFUNC_OK:
      case RXFUNC_DEFINED:
      case RXFUNC_NOTREG:
         break;
      default:
         return ( CallingInvalid );
      } /* endswitch */
   } /* endfor */
   rxset_zerolen( retstr );
   return ( CallingValid );
}

ULONG _Export  
RxldapVersion(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   int i = 0;
   if (argc > 0) {
      return ( CallingInvalid );
   } /* endif */
   i = sprintf(RXSTRPTR(*retstr), "RXLDAP Version: %s (%s %s)", RxldapVer, RxldapDate, RxldapTime);
   if ( i + 1 > RXSTRLEN(*retstr) ) {
      return ( CallingInvalid );
   } /* endif */
   rxset_length( retstr, i );
   return ( CallingValid );
}

ULONG _Export
Rxldap_simple_bind(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   char *dn = NULL;
   char *passwd = NULL;
   int code = 0;
   int freedn = 0;
   if ( argc < 1 || argc > 3 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP*)rxtoul( argv[0] );
   if ( argc > 1 && !RXNULLSTRING(argv[1]) && RXSTRLEN(argv[1]) ) {
      dn = strtoutf8(RXSTRPTR(argv[1]));
      if ( dn ) {
         freedn = 1;
      } else {
         dn = RXSTRPTR(argv[1]);
      } /* endif */
   } /* endif */
   if ( argc > 2 && !RXNULLSTRING(argv[2]) && RXSTRLEN(argv[2]) ) {
      passwd = RXSTRPTR(argv[2]);
   } /* endif */
   code = ldap_simple_bind( ld, dn, passwd );
   if ( freedn ) {
      free( dn );
   } /* endif */
   SetSignedInt2Rx( *retstr, code );
   return ( CallingValid );
}   

ULONG _Export
Rxldap_simple_bind_s(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   char *dn = NULL;
   char *passwd = NULL;
   int code = 0;
   int freedn = 0;
   if ( argc < 1 || argc > 3 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP*)rxtoul( argv[0] );
   if ( argc > 1 && !RXNULLSTRING(argv[1]) && RXSTRLEN(argv[1]) ) {
      dn = strtoutf8(RXSTRPTR(argv[1]));
      if ( dn ) {
         freedn = 1;
      } else {
         dn = RXSTRPTR(argv[1]);
      } /* endif */
   } /* endif */
   if ( argc > 2 && !RXNULLSTRING(argv[2]) && RXSTRLEN(argv[2]) ) {
      passwd = RXSTRPTR(argv[2]);
   } /* endif */
   code = ldap_simple_bind_s( ld, dn, passwd );
   if ( freedn ) {
      free( dn );
   } /* endif */
   SetLdapCode( retstr, code );
   return ( CallingValid );
}   

ULONG _Export
Rxldap_unbind(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   int code = 0;
   if ( argc != 1 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP*)rxtoul( argv[0] );
   code = ldap_unbind( ld );
   SetSignedInt2Rx( *retstr, code );
   return ( CallingValid );
}   

ULONG _Export
Rxldap_unbind_s(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   int code = 0;
   if ( argc != 1 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP*)rxtoul( argv[0] );
   code = ldap_unbind_s( ld );
   SetLdapCode( retstr, code );
   return ( CallingValid );
}   

ULONG _Export
Rxldap_rename(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   int code = 0;
   char *newparent = NULL;
   int deleteoldrdn = 0;
   LDAPControl **serverctrls = NULL;
   LDAPControl **clientctrls = NULL;
   RXSTRING value;
   int msgid = 0;
   unsigned char *dn = NULL;
   unsigned char *newrdn = NULL;
   int freedn = 0;
   int freenewrdn = 0;
   int freenewparent = 0;
   if ( argc != 8 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[1]) || RXNULLSTRING(argv[2]) || RXNULLSTRING(argv[7]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[1]) || !RXSTRLEN(argv[2]) || !RXSTRLEN(argv[7]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) || !isnumstr(RXSTRPTR(argv[4])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   dn = strtoutf8(RXSTRPTR(argv[1]));
   if ( dn ) {
      freedn = 1;
   } else {
      dn = RXSTRPTR(argv[1]);
   } /* endif */
   newrdn = strtoutf8(RXSTRPTR(argv[2]));
   if ( newrdn ) {
      freenewrdn = 1;
   } else {
      newrdn = RXSTRPTR(argv[2]);
   } /* endif */
   if ( !RXNULLSTRING(argv[3]) && RXSTRLEN(argv[3]) ) {
      newparent = strtoutf8(RXSTRPTR(argv[3]));
      if ( newparent) {
         freenewparent = 1;
      } else {
         newparent = RXSTRPTR(argv[3]);
      } /* endif */
   } /* endif */
   deleteoldrdn = rxtoi( argv[4] );
   if ( !RXNULLSTRING(argv[5]) && RXSTRLEN(argv[5]) ) {
      serverctrls = getldapcontrols( &argv[5] );
      if ( !serverctrls ) {
         return ( CallingInvalid );
      } /* endif */
   } /* endif */
   if ( !RXNULLSTRING(argv[6]) && RXSTRLEN(argv[6]) ) {
      clientctrls = getldapcontrols( &argv[6] );
      if ( !clientctrls ) {
         return ( CallingInvalid );
      } /* endif */
   } /* endif */
   code = ldap_rename( ld, dn, newrdn, newparent, deleteoldrdn, serverctrls, clientctrls, &msgid );
   value = rxalloc( BUFSIZ );
   if ( RXNULLSTRING( value ) ) {
      if ( freedn ) {
         free( dn );
      } /* endif */
      if ( freenewrdn ) {
         free( newrdn );
      } /* endif */
      if ( freenewparent ) {
         free( newparent );
      } /* endif */
      return ( CallingInvalid );
   } /* endif */
   SetUnsignedInt2Rx( value, msgid );
   if ( setvalue( &argv[7], &value ) != Good ) {
      rxfree( value );
      if ( freedn ) {
         free( dn );
      } /* endif */
      if ( freenewrdn ) {
         free( newrdn );
      } /* endif */
      if ( freenewparent ) {
         free( newparent );
      } /* endif */
      return ( CallingInvalid );
   } /* endif */
   rxfree( value );
   ldap_controls_free( serverctrls );
   ldap_controls_free( clientctrls );
   if ( freedn ) {
      free( dn );
   } /* endif */
   if ( freenewrdn ) {
      free( newrdn );
   } /* endif */
   if ( freenewparent ) {
      free( newparent );
   } /* endif */
   SetLdapCode( retstr, code );
   return ( CallingValid );
}   

ULONG _Export
Rxldap_rename_s(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   int code = 0;
   char *newparent = NULL;
   int deleteoldrdn = 0;
   LDAPControl **serverctrls = NULL;
   LDAPControl **clientctrls = NULL;
   unsigned char *dn = NULL;
   unsigned char *newrdn = NULL;
   int freedn = 0;
   int freenewrdn = 0;
   int freenewparent = 0;
   if ( argc < 5 || argc > 7 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[1]) || RXNULLSTRING(argv[2]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[1]) || !RXSTRLEN(argv[2]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) || !isnumstr(RXSTRPTR(argv[4])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   dn = strtoutf8(RXSTRPTR(argv[1]));
   if ( dn ) {
      freedn = 1;
   } else {
      dn = RXSTRPTR(argv[1]);
   } /* endif */
   newrdn = strtoutf8(RXSTRPTR(argv[2]));
   if ( newrdn ) {
      freenewrdn = 1;
   } else {
      newrdn = RXSTRPTR(argv[2]);
   } /* endif */
   if ( !RXNULLSTRING(argv[3]) && RXSTRLEN(argv[3]) ) {
      newparent = strtoutf8(RXSTRPTR(argv[3]));
      if ( newparent) {
         freenewparent = 1;
      } else {
         newparent = RXSTRPTR(argv[3]);
      } /* endif */
   } /* endif */
   deleteoldrdn = rxtoi( argv[4] );
   if ( !RXNULLSTRING(argv[5]) && RXSTRLEN(argv[5]) ) {
      serverctrls = getldapcontrols( &argv[5] );
      if ( !serverctrls ) {
         return ( CallingInvalid );
      } /* endif */
   } /* endif */
   if ( !RXNULLSTRING(argv[6]) && RXSTRLEN(argv[6]) ) {
      clientctrls = getldapcontrols( &argv[6] );
      if ( !clientctrls ) {
         return ( CallingInvalid );
      } /* endif */
   } /* endif */
   code = ldap_rename_s( ld, dn, newrdn, newparent, deleteoldrdn, serverctrls, clientctrls );
   if ( freedn ) {
      free( dn );
   } /* endif */
   if ( freenewrdn ) {
      free( newrdn );
   } /* endif */
   if ( freenewparent ) {
      free( newparent );
   } /* endif */
   SetLdapCode( retstr, code );
   return ( CallingValid );
}   

ULONG _Export       
Rxldap_add(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   LDAPMod **mods = NULL;
   int code = 0;
   if ( argc != 3 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[1]) || RXNULLSTRING(argv[2]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[1]) || !RXSTRLEN(argv[2]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   mods = getldapmods( &argv[2] );
   if ( !mods ) {
      return ( CallingInvalid );
   } /* endif */
   code = ldap_add( ld, RXSTRPTR(argv[1]), mods );
   ldap_mods_free( mods, 1 );
   SetSignedInt2Rx( *retstr, code );
   return ( CallingValid );
}   

ULONG _Export       
Rxldap_add_ext(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   LDAPMod **mods = NULL;
   int code = 0;
   LDAPControl **serverctrls = NULL;
   LDAPControl **clientctrls = NULL;
   int msgid = 0;
   RXSTRING value;
   if ( argc != 6 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[1]) || RXNULLSTRING(argv[2]) || RXNULLSTRING(argv[5]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[1]) || !RXSTRLEN(argv[2]) || !RXSTRLEN(argv[5]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   mods = getldapmods( &argv[2] );
   if ( !mods ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXNULLSTRING(argv[3]) && RXSTRLEN(argv[3]) ) {
      serverctrls = getldapcontrols( &argv[6] );
      if ( !serverctrls ) {
         ldap_mods_free( mods, 1 );
         return ( CallingInvalid );
      } /* endif */
   } /* endif */
   if ( !RXNULLSTRING(argv[4]) && RXSTRLEN(argv[4]) ) {
      clientctrls = getldapcontrols( &argv[7] );
      if ( !clientctrls ) {
         ldap_mods_free( mods, 1 );
         return ( CallingInvalid );
      } /* endif */
   } /* endif */
   code = ldap_add_ext( ld, RXSTRPTR(argv[1]), mods, serverctrls, clientctrls, &msgid );
   ldap_mods_free( mods, 1 );
   value = rxalloc( BUFSIZ );
   if ( RXNULLSTRING( value ) ) {
      return ( CallingInvalid );
   } /* endif */
   SetUnsignedInt2Rx( value, msgid );
   if ( setvalue( &argv[5], &value ) != Good ) {
      rxfree( value );
      return ( CallingInvalid );
   } /* endif */
   rxfree( value );
   ldap_controls_free( serverctrls );
   ldap_controls_free( clientctrls );
   SetLdapCode( retstr, code );
   return ( CallingValid );
}   

ULONG _Export       
Rxldap_add_s(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   LDAPMod **mods = NULL;
   int code = 0;
   if ( argc != 3 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[1]) || RXNULLSTRING(argv[2]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[1]) || !RXSTRLEN(argv[2]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   mods = getldapmods( &argv[2] );
   if ( !mods ) {
      return ( CallingInvalid );
   } /* endif */
   code = ldap_add_s( ld, RXSTRPTR(argv[1]), mods );
   ldap_mods_free( mods, 1 );
   SetLdapCode( retstr, code );
   return ( CallingValid );
}   

ULONG _Export       
Rxldap_add_ext_s(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   LDAPMod **mods = NULL;
   int code = 0;
   LDAPControl **serverctrls = NULL;
   LDAPControl **clientctrls = NULL;
   if ( argc < 3 || argc > 5 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[1]) || RXNULLSTRING(argv[2]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[1]) || !RXSTRLEN(argv[2]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   mods = getldapmods( &argv[2] );
   if ( !mods ) {
      return ( CallingInvalid );
   } /* endif */
   if ( argc > 3 && !RXNULLSTRING(argv[3]) && RXSTRLEN(argv[3]) ) {
      serverctrls = getldapcontrols( &argv[6] );
      if ( !serverctrls ) {
         ldap_mods_free( mods, 1 );
         return ( CallingInvalid );
      } /* endif */
   } /* endif */
   if ( argc > 4 && !RXNULLSTRING(argv[4]) && RXSTRLEN(argv[4]) ) {
      clientctrls = getldapcontrols( &argv[7] );
      if ( !clientctrls ) {
         ldap_mods_free( mods, 1 );
         return ( CallingInvalid );
      } /* endif */
   } /* endif */
   code = ldap_add_ext_s( ld, RXSTRPTR(argv[1]), mods, serverctrls, clientctrls );
   ldap_mods_free( mods, 1 );
   ldap_controls_free( serverctrls );
   ldap_controls_free( clientctrls );
   SetLdapCode( retstr, code );
   return ( CallingValid );
}   

ULONG _Export       
Rxldap_modify(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   LDAPMod **mods = NULL;
   int code = 0;
   if ( argc != 3 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[1]) || RXNULLSTRING(argv[2]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[1]) || !RXSTRLEN(argv[2]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   mods = getldapmods( &argv[2] );
   if ( !mods ) {
      return ( CallingInvalid );
   } /* endif */
   code = ldap_modify( ld, RXSTRPTR(argv[1]), mods );
   ldap_mods_free( mods, 1 );
   SetSignedInt2Rx( *retstr, code );
   return ( CallingValid );
}   

ULONG _Export       
Rxldap_modify_ext(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   LDAPMod **mods = NULL;
   int code = 0;
   LDAPControl **serverctrls = NULL;
   LDAPControl **clientctrls = NULL;
   int msgid = 0;
   RXSTRING value;
   if ( argc != 6 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[1]) || RXNULLSTRING(argv[2]) || RXNULLSTRING(argv[5]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[1]) || !RXSTRLEN(argv[2]) || !RXSTRLEN(argv[5]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   mods = getldapmods( &argv[2] );
   if ( !mods ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXNULLSTRING(argv[3]) && RXSTRLEN(argv[3]) ) {
      serverctrls = getldapcontrols( &argv[6] );
      if ( !serverctrls ) {
         ldap_mods_free( mods, 1 );
         return ( CallingInvalid );
      } /* endif */
   } /* endif */
   if ( !RXNULLSTRING(argv[4]) && RXSTRLEN(argv[4]) ) {
      clientctrls = getldapcontrols( &argv[7] );
      if ( !clientctrls ) {
         ldap_mods_free( mods, 1 );
         return ( CallingInvalid );
      } /* endif */
   } /* endif */
   code = ldap_modify_ext( ld, RXSTRPTR(argv[1]), mods, serverctrls, clientctrls, &msgid );
   ldap_mods_free( mods, 1 );
   value = rxalloc( BUFSIZ );
   if ( RXNULLSTRING( value ) ) {
      return ( CallingInvalid );
   } /* endif */
   SetUnsignedInt2Rx( value, msgid );
   if ( setvalue( &argv[5], &value ) != Good ) {
      rxfree( value );
      return ( CallingInvalid );
   } /* endif */
   rxfree( value );
   ldap_controls_free( serverctrls );
   ldap_controls_free( clientctrls );
   SetLdapCode( retstr, code );
   return ( CallingValid );
}   

ULONG _Export       
Rxldap_modify_ext_s(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   LDAPMod **mods = NULL;
   int code = 0;
   LDAPControl **serverctrls = NULL;
   LDAPControl **clientctrls = NULL;
   if ( argc < 3 || argc > 5 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[1]) || RXNULLSTRING(argv[2]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[1]) || !RXSTRLEN(argv[2]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   mods = getldapmods( &argv[2] );
   if ( !mods ) {
      return ( CallingInvalid );
   } /* endif */
   if ( argc > 3 && !RXNULLSTRING(argv[3]) && RXSTRLEN(argv[3]) ) {
      serverctrls = getldapcontrols( &argv[6] );
      if ( !serverctrls ) {
         ldap_mods_free( mods, 1 );
         return ( CallingInvalid );
      } /* endif */
   } /* endif */
   if ( argc > 4 && !RXNULLSTRING(argv[4]) && RXSTRLEN(argv[4]) ) {
      clientctrls = getldapcontrols( &argv[7] );
      if ( !clientctrls ) {
         ldap_mods_free( mods, 1 );
         return ( CallingInvalid );
      } /* endif */
   } /* endif */
   code = ldap_modify_ext_s( ld, RXSTRPTR(argv[1]), mods, serverctrls, clientctrls );
   ldap_mods_free( mods, 1 );
   ldap_controls_free( serverctrls );
   ldap_controls_free( clientctrls );
   SetLdapCode( retstr, code );
   return ( CallingValid );
}   

ULONG _Export       
Rxldap_modify_s(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   LDAPMod **mods = NULL;
   int code = 0;
   if ( argc != 3 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[1]) || RXNULLSTRING(argv[2]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[1]) || !RXSTRLEN(argv[2]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   mods = getldapmods( &argv[2] );
   if ( !mods ) {
      return ( CallingInvalid );
   } /* endif */
   code = ldap_modify_s( ld, RXSTRPTR(argv[1]), mods );
   ldap_mods_free( mods, 1 );
   SetLdapCode( retstr, code );
   return ( CallingValid );
}   

ULONG _Export
Rxldap_compare_s(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   int code = 0;
   unsigned char *dn = NULL;
   unsigned char *va = NULL;
   int freedn = 0;
   int freeva = 0;
   if ( argc != 4 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[1]) || RXNULLSTRING(argv[2]) || RXNULLSTRING(argv[3]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[1]) || !RXSTRLEN(argv[2]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   dn = strtoutf8( RXSTRPTR(argv[1]) );
   if ( dn ) {
      freedn = 1;
   } else {
      dn = RXSTRPTR(argv[1]);
   } /* endif */
   va = strtoutf8( RXSTRPTR(argv[3]) );
   if ( va ) {
      freeva = 1;
   } else {
      va = RXSTRPTR(argv[3]);
   } /* endif */
   code = ldap_compare_s( ld, dn, RXSTRPTR(argv[2]), va );
   if ( freedn ) {
      free( dn );
   } /* endif */
   if ( freeva ) {
      free( va );
   } /* endif */
   SetLdapCode( retstr, code );
   return ( CallingValid );
}   

ULONG _Export
Rxldap_compare(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   int code = 0;
   unsigned char *dn = NULL;
   unsigned char *va = NULL;
   int freedn = 0;
   int freeva = 0;
   if ( argc != 4 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[1]) || RXNULLSTRING(argv[2]) || RXNULLSTRING(argv[3]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[1]) || !RXSTRLEN(argv[2]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   dn = strtoutf8( RXSTRPTR(argv[1]) );
   if ( dn ) {
      freedn = 1;
   } else {
      dn = RXSTRPTR(argv[1]);
   } /* endif */
   va = strtoutf8( RXSTRPTR(argv[3]) );
   if ( va ) {
      freeva = 1;
   } else {
      va = RXSTRPTR(argv[3]);
   } /* endif */
   code = ldap_compare( ld, dn, RXSTRPTR(argv[2]), va );
   if ( freedn ) {
      free( dn );
   } /* endif */
   if ( freeva ) {
      free( va );
   } /* endif */
   SetSignedInt2Rx( *retstr, code );
   return ( CallingValid );
}   
          
ULONG _Export
Rxldap_compare_ext_s(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   LDAPControl **serverctrls = NULL;
   LDAPControl **clientctrls = NULL;
   int code = 0;
   struct berval bvalue;
   unsigned char *dn = NULL;
   int freedn = 0;
   if ( argc < 4 || argc > 6 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[1]) || RXNULLSTRING(argv[2]) || RXNULLSTRING(argv[3]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[1]) || !RXSTRLEN(argv[2]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   dn = strtoutf8( RXSTRPTR(argv[1]) );
   if ( dn ) {
      freedn = 1;
   } else {
      dn = RXSTRPTR(argv[1]);
   } /* endif */
   bvalue.bv_val = RXSTRPTR(argv[3]);
   bvalue.bv_len = RXSTRLEN(argv[3]);
   if ( argc > 4 && !RXNULLSTRING(argv[4]) && RXSTRLEN(argv[4]) ) {
      serverctrls = getldapcontrols( &argv[4] );
      if ( !serverctrls ) {
         if ( freedn ) {
            free( dn );
         } /* endif */
         return ( CallingInvalid );
      } /* endif */
   } /* endif */
   if ( argc > 5 && !RXNULLSTRING(argv[5]) && RXSTRLEN(argv[5]) ) {
      clientctrls = getldapcontrols( &argv[5] );
      if ( !clientctrls ) {
         if ( freedn ) {
            free( dn );
         } /* endif */
         return ( CallingInvalid );
      } /* endif */
   } /* endif */
   code = ldap_compare_ext_s( ld, dn, RXSTRPTR(argv[2]), &bvalue, serverctrls, clientctrls );
   ldap_controls_free( serverctrls );
   ldap_controls_free( clientctrls );
   if ( freedn ) {
      free( dn );
   } /* endif */
   SetLdapCode( retstr, code );
   return ( CallingValid );
}   
          
ULONG _Export
Rxldap_compare_ext(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   LDAPControl **serverctrls = NULL;
   LDAPControl **clientctrls = NULL;
   int code = 0;
   struct berval bvalue;
   RXSTRING value;
   int msgid = 0;
   unsigned char *dn = NULL;
   int freedn = 0;
   if ( argc != 7 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[1]) || RXNULLSTRING(argv[2]) || RXNULLSTRING(argv[3]) || RXNULLSTRING(argv[6]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[1]) || !RXSTRLEN(argv[2]) || !RXSTRLEN(argv[6]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   dn = strtoutf8( RXSTRPTR(argv[1]) );
   if ( dn ) {
      freedn = 1;
   } else {
      dn = RXSTRPTR(argv[1]);
   } /* endif */
   bvalue.bv_val = RXSTRPTR(argv[3]);
   bvalue.bv_len = RXSTRLEN(argv[3]);
   if ( !RXNULLSTRING(argv[4]) && RXSTRLEN(argv[4]) ) {
      serverctrls = getldapcontrols( &argv[4] );
      if ( !serverctrls ) {
         if ( freedn ) {
            free( dn );
         } /* endif */
         return ( CallingInvalid );
      } /* endif */
   } /* endif */
   if ( !RXNULLSTRING(argv[5]) && RXSTRLEN(argv[5]) ) {
      clientctrls = getldapcontrols( &argv[5] );
      if ( !clientctrls ) {
         if ( freedn ) {
            free( dn );
         } /* endif */
         return ( CallingInvalid );
      } /* endif */
   } /* endif */
   code = ldap_compare_ext( ld, dn, RXSTRPTR(argv[2]), &bvalue, serverctrls, clientctrls, &msgid );
   value = rxalloc( BUFSIZ );
   if ( RXNULLSTRING( value ) ) {
      if ( freedn ) {
         free( dn );
      } /* endif */
      return ( CallingInvalid );
   } /* endif */
   SetUnsignedInt2Rx( value, msgid );
   if ( setvalue( &argv[6], &value ) != Good ) {
      if ( freedn ) {
         free( dn );
      } /* endif */
      rxfree( value );
      return ( CallingInvalid );
   } /* endif */
   rxfree( value );
   ldap_controls_free( serverctrls );
   ldap_controls_free( clientctrls );
   if ( freedn ) {
      free( dn );
   } /* endif */
   SetLdapCode( retstr, code );
   return ( CallingValid );
}   
          
ULONG _Export
Rxldap_delete(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   int code = 0;
   unsigned char *dn = NULL;
   int freedn = 0;
   if ( argc != 2 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[1]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[1]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   dn = strtoutf8( RXSTRPTR(argv[1]) );
   if ( dn ) {
      freedn = 1;
   } else {
      dn = RXSTRPTR(argv[1]);
   } /* endif */
   code = ldap_delete( ld, dn );
   if ( freedn ) {
      free( dn );
   } /* endif */
   SetSignedInt2Rx( *retstr, code );
   return ( CallingValid );
}   

ULONG _Export
Rxldap_delete_s(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   int code = 0;
   unsigned char *dn = NULL;
   int freedn = 0;
   if ( argc != 2 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[1]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[1]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   dn = strtoutf8( RXSTRPTR(argv[1]) );
   if ( dn ) {
      freedn = 1;
   } else {
      dn = RXSTRPTR(argv[1]);
   } /* endif */
   code = ldap_delete_s( ld, dn );
   if ( freedn ) {
      free( dn );
   } /* endif */
   SetLdapCode( retstr, code );
   return ( CallingValid );
}   

ULONG _Export
Rxldap_delete_ext_s(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   int code = 0;
   LDAPControl **serverctrls = NULL;
   LDAPControl **clientctrls = NULL;
   unsigned char *dn = NULL;
   int freedn = 0;
   if ( argc < 2 || argc > 4 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[1]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[1]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   dn = strtoutf8( RXSTRPTR(argv[1]) );
   if ( dn ) {
      freedn = 1;
   } else {
      dn = RXSTRPTR(argv[1]);
   } /* endif */
   if ( argc > 2 && !RXNULLSTRING(argv[2]) && RXSTRLEN(argv[2]) ) {
      serverctrls = getldapcontrols( &argv[2] );
      if ( !serverctrls ) {
         return ( CallingInvalid );
      } /* endif */
   } /* endif */
   if ( argc > 3 && !RXNULLSTRING(argv[3]) && RXSTRLEN(argv[3]) ) {
      clientctrls = getldapcontrols( &argv[3] );
      if ( !clientctrls ) {
         return ( CallingInvalid );
      } /* endif */
   } /* endif */
   code = ldap_delete_ext_s( ld, dn, serverctrls, clientctrls );
   if ( freedn ) {
      free( dn );
   } /* endif */
   SetLdapCode( retstr, code );
   return ( CallingValid );
}   

ULONG _Export
Rxldap_delete_ext(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   int code = 0;
   LDAPControl **serverctrls = NULL;
   LDAPControl **clientctrls = NULL;
   int msgid = 0;
   RXSTRING value;
   unsigned char *dn = NULL;
   int freedn = 0;
   if ( argc != 5 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[1]) || RXNULLSTRING(argv[4]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[1]) || !RXSTRLEN(argv[4]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   dn = strtoutf8( RXSTRPTR(argv[1]) );
   if ( dn ) {
      freedn = 1;
   } else {
      dn = RXSTRPTR(argv[1]);
   } /* endif */
   if ( !RXNULLSTRING(argv[2]) && RXSTRLEN(argv[2]) ) {
      serverctrls = getldapcontrols( &argv[2] );
      if ( !serverctrls ) {
         if ( freedn ) {
            free( dn );
         } /* endif */
         return ( CallingInvalid );
      } /* endif */
   } /* endif */
   if ( !RXNULLSTRING(argv[3]) && RXSTRLEN(argv[3]) ) {
      clientctrls = getldapcontrols( &argv[3] );
      if ( !clientctrls ) {
         if ( freedn ) {
            free( dn );
         } /* endif */
         return ( CallingInvalid );
      } /* endif */
   } /* endif */
   code = ldap_delete_ext( ld, dn, serverctrls, clientctrls, &msgid );
   value = rxalloc( BUFSIZ );
   if ( RXNULLSTRING( value ) ) {
      if ( freedn ) {
         free( dn );
      } /* endif */
      return ( CallingInvalid );
   } /* endif */
   SetUnsignedInt2Rx( value, msgid );
   if ( setvalue( &argv[4], &value ) != Good ) {
      if ( freedn ) {
         free( dn );
      } /* endif */
      rxfree( value );
      return ( CallingInvalid );
   } /* endif */
   rxfree( value );
   ldap_controls_free( serverctrls );
   ldap_controls_free( clientctrls );
   if ( freedn ) {
      free( dn );
   } /* endif */
   SetLdapCode( retstr, code );
   return ( CallingValid );
}   

ULONG 
Rxldap_search_s(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   int scope = 0;
   int attrsonly = 0;
   LDAPMessage *msg = NULL;
   int code = 0;
   char **attrs = NULL;
   Name2Int *v = NULL;
   RXSTRING value;
   unsigned char *basedn = NULL;
   unsigned char *filter = NULL;
   int freebasedn = 0;
   int freefilter = 0;
   if ( argc != 7 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[2]) || RXNULLSTRING(argv[6]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[2]) || !RXSTRLEN(argv[6]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) || !isnumstr(RXSTRPTR(argv[5])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP*)rxtoul( argv[0] );
   basedn = strtoutf8( RXSTRPTR(argv[1]) );
   if ( basedn ) {
      freebasedn = 1;
   } else {
      basedn = RXSTRPTR(argv[1]);
   } /* endif */
   v = findintfromname( RXSTRPTR(argv[2]), ScopeNames, sizeof(ScopeNames) / sizeof(Name2Int), 0 );
   if ( !v ) {
      return ( CallingInvalid );
   } /* endif */
   scope = v->intval;
   filter = strtoutf8( RXSTRPTR(argv[3]) );
   if ( filter ) {
      freefilter = 1;
   } else {
      filter = RXSTRPTR(argv[3]);
   } /* endif */
   if ( !RXNULLSTRING(argv[4]) && RXSTRLEN(argv[4]) ) {
      attrs = getldapattrs( &argv[4] );
      if ( !attrs ) {
         return ( CallingInvalid );
      } /* endif */
   } /* endif */
   attrsonly = rxtoi( argv[5] );
   code = ldap_search_s( ld, basedn, scope, filter, attrs, attrsonly, &msg );
   freestrs( attrs );
   if ( freebasedn ) {
      free( basedn );
   } /* endif */
   if ( freefilter ) {
      free( filter );
   } /* endif */
   value = rxalloc( BUFSIZ );
   if ( RXNULLSTRING( value ) ) {
      return ( CallingInvalid );
   } /* endif */
   SetUnsignedInt2Rx( value, msg );
   if ( setvalue( &argv[6], &value ) != Good ) {
      rxfree( value );
      return ( CallingInvalid );
   } /* endif */
   rxfree( value );
   SetLdapCode( retstr, code );
   return ( CallingValid );
}

ULONG 
Rxldap_search_st(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   int scope = 0;
   int attrsonly = 0;
   LDAPMessage *msg = NULL;
   int code = 0;
   RXSTRING value;
   Name2Int *v = NULL;
   char **attrs = NULL;
   struct timeval timeout;
   unsigned char *basedn = NULL;
   unsigned char *filter = NULL;
   int freebasedn = 0;
   int freefilter = 0;
   if ( argc != 8 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[2]) || RXNULLSTRING(argv[6]) || RXNULLSTRING(argv[7])) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[2]) || !RXSTRLEN(argv[6]) || !RXSTRLEN(argv[7])) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) || !isnumstr(RXSTRPTR(argv[5])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP*)rxtoul( argv[0] );
   basedn = strtoutf8( RXSTRPTR(argv[1]) );
   if ( basedn ) {
      freebasedn = 1;
   } else {
      basedn = RXSTRPTR(argv[1]);
   } /* endif */
   v = findintfromname( RXSTRPTR(argv[2]), ScopeNames, sizeof(ScopeNames) / sizeof(Name2Int), 0 );
   if ( !v ) {
      return ( CallingInvalid );
   } /* endif */
   scope = v->intval;
   filter = strtoutf8( RXSTRPTR(argv[3]) );
   if ( filter ) {
      freefilter = 1;
   } else {
      filter = RXSTRPTR(argv[3]);
   } /* endif */
   if ( !RXNULLSTRING(argv[4]) && RXSTRLEN(argv[4]) ) {
      attrs = getldapattrs( &argv[4] );
      if ( !attrs ) {
         return ( CallingInvalid );
      } /* endif */
   } /* endif */
   attrsonly = rxtoi( argv[5] );
   if ( parsetimeval( &argv[6], &timeout ) != Good ) {
      freestrs( attrs );
      return ( CallingInvalid );
   } /* endif */
   code = ldap_search_st( ld, basedn, scope, filter, attrs, attrsonly, &timeout, &msg );
   freestrs( attrs );
   if ( freebasedn ) {
      free( basedn );
   } /* endif */
   if ( freefilter ) {
      free( filter );
   } /* endif */
   value = rxalloc( BUFSIZ );
   if ( RXNULLSTRING( value ) ) {
      return ( CallingInvalid );
   } /* endif */
   SetUnsignedInt2Rx( value, msg );
   if ( setvalue( &argv[7], &value ) != Good ) {
      rxfree( value );
      return ( CallingInvalid );
   } /* endif */
   rxfree( value );
   SetLdapCode( retstr, code );
   return ( CallingValid );
}

ULONG 
Rxldap_search(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   int scope = 0;
   int attrsonly = 0;
   int code = 0;
   RXSTRING value;
   Name2Int *v = NULL;
   char **attrs = NULL;
   unsigned char *basedn = NULL;
   unsigned char *filter = NULL;
   int freebasedn = 0;
   int freefilter = 0;
   if ( argc != 6 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[2]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[2]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) || !isnumstr(RXSTRPTR(argv[5])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP*)rxtoul( argv[0] );
   basedn = strtoutf8( RXSTRPTR(argv[1]) );
   if ( basedn ) {
      freebasedn = 1;
   } else {
      basedn = RXSTRPTR(argv[1]);
   } /* endif */
   v = findintfromname( RXSTRPTR(argv[2]), ScopeNames, sizeof(ScopeNames) / sizeof(Name2Int), 0 );
   if ( !v ) {
      return ( CallingInvalid );
   } /* endif */
   scope = v->intval;
   filter = strtoutf8( RXSTRPTR(argv[3]) );
   if ( filter ) {
      freefilter = 1;
   } else {
      filter = RXSTRPTR(argv[3]);
   } /* endif */
   if ( !RXNULLSTRING(argv[4]) && RXSTRLEN(argv[4]) ) {
      attrs = getldapattrs( &argv[4] );
      if ( !attrs ) {
         return ( CallingInvalid );
      } /* endif */
   } /* endif */
   attrsonly = rxtoi( argv[5] );
   code = ldap_search( ld, basedn, scope, filter, attrs, attrsonly );
   freestrs( attrs );
   if ( freebasedn ) {
      free( basedn );
   } /* endif */
   if ( freefilter ) {
      free( filter );
   } /* endif */
   SetSignedInt2Rx( *retstr, code );
   return ( CallingValid );
}

ULONG 
Rxldap_search_ext(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   int scope = 0;
   int attrsonly = 0;
   int code = 0;
   RXSTRING value;
   Name2Int *v = NULL;
   char **attrs = NULL;
   LDAPControl **serverctrls = NULL;
   LDAPControl **clientctrls = NULL;
   struct timeval timeout;
   int sizelimit = 0;
   int msgid = 0;
   unsigned char *basedn = NULL;
   unsigned char *filter = NULL;
   int freebasedn = 0;
   int freefilter = 0;
   if ( argc != 11 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[2]) || RXNULLSTRING(argv[9]) || RXNULLSTRING(argv[10]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[2]) || !RXSTRLEN(argv[9]) || !RXSTRLEN(argv[10]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) || !isnumstr(RXSTRPTR(argv[5])) || !isnumstr(RXSTRPTR(argv[9])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP*)rxtoul( argv[0] );
   basedn = strtoutf8( RXSTRPTR(argv[1]) );
   if ( basedn ) {
      freebasedn = 1;
   } else {
      basedn = RXSTRPTR(argv[1]);
   } /* endif */
   v = findintfromname( RXSTRPTR(argv[2]), ScopeNames, sizeof(ScopeNames) / sizeof(Name2Int), 0 );
   if ( !v ) {
      if ( freebasedn ) {
         free( basedn );
      } /* endif */
      return ( CallingInvalid );
   } /* endif */
   scope = v->intval;
   filter = strtoutf8( RXSTRPTR(argv[3]) );
   if ( filter ) {
      freefilter = 1;
   } else {
      filter = RXSTRPTR(argv[3]);
   } /* endif */
   if ( !RXNULLSTRING(argv[4]) && RXSTRLEN(argv[4]) ) {
      attrs = getldapattrs( &argv[4] );
      if ( !attrs ) {
         if ( freebasedn ) {
            free( basedn );
         } /* endif */
         if ( freefilter ) {
            free( filter );
         } /* endif */
         return ( CallingInvalid );
      } /* endif */
   } /* endif */
   attrsonly = rxtoi( argv[5] );
   if ( !RXNULLSTRING(argv[6]) && RXSTRLEN(argv[6]) ) {
      serverctrls = getldapcontrols( &argv[6] );
      if ( !serverctrls ) {
         if ( freebasedn ) {
            free( basedn );
         } /* endif */
         if ( freefilter ) {
            free( filter );
         } /* endif */
         freestrs( attrs );
         return ( CallingInvalid );
      } /* endif */
   } /* endif */
   if ( !RXNULLSTRING(argv[7]) && RXSTRLEN(argv[7]) ) {
      clientctrls = getldapcontrols( &argv[7] );
      if ( !clientctrls ) {
         ldap_controls_free( serverctrls );
         if ( freebasedn ) {
            free( basedn );
         } /* endif */
         if ( freefilter ) {
            free( filter );
         } /* endif */
         freestrs( attrs );
         return ( CallingInvalid );
      } /* endif */
   } /* endif */
   if ( parsetimeval( &argv[8], &timeout ) != Good ) {
      ldap_controls_free( serverctrls );
      ldap_controls_free( clientctrls );
      if ( freebasedn ) {
         free( basedn );
      } /* endif */
      if ( freefilter ) {
         free( filter );
      } /* endif */
      freestrs( attrs );
      return ( CallingInvalid );
   } /* endif */
   sizelimit = rxtoi( argv[9] );
   code = ldap_search_ext( ld, basedn, scope, filter, attrs, attrsonly, serverctrls, clientctrls, &timeout, sizelimit, &msgid );
   freestrs( attrs );
   if ( freebasedn ) {
      free( basedn );
   } /* endif */
   if ( freefilter ) {
      free( filter );
   } /* endif */
   ldap_controls_free( serverctrls );
   ldap_controls_free( clientctrls );
   value = rxalloc( BUFSIZ );
   if ( RXNULLSTRING( value ) ) {
      return ( CallingInvalid );
   } /* endif */
   SetUnsignedInt2Rx( value, msgid );
   if ( setvalue( &argv[10], &value ) != Good ) {
      rxfree( value );
      return ( CallingInvalid );
   } /* endif */
   rxfree( value );
   SetLdapCode( retstr, code );
   return ( CallingValid );
}

ULONG 
Rxldap_search_ext_s(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   int scope = 0;
   int attrsonly = 0;
   int code = 0;
   RXSTRING value;
   Name2Int *v = NULL;
   char **attrs = NULL;
   LDAPControl **serverctrls = NULL;
   LDAPControl **clientctrls = NULL;
   struct timeval timeout;
   int sizelimit = 0;
   LDAPMessage *msg = NULL;
   unsigned char *basedn = NULL;
   unsigned char *filter = NULL;
   int freebasedn = 0;
   int freefilter = 0;
   if ( argc != 11 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[2]) || RXNULLSTRING(argv[9]) || RXNULLSTRING(argv[10]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[2]) || !RXSTRLEN(argv[9]) || !RXSTRLEN(argv[10]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) || !isnumstr(RXSTRPTR(argv[5])) || !isnumstr(RXSTRPTR(argv[9])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP*)rxtoul( argv[0] );
   basedn = strtoutf8( RXSTRPTR(argv[1]) );
   if ( basedn ) {
      freebasedn = 1;
   } else {
      basedn = RXSTRPTR(argv[1]);
   } /* endif */
   v = findintfromname( RXSTRPTR(argv[2]), ScopeNames, sizeof(ScopeNames) / sizeof(Name2Int), 0 );
   if ( !v ) {
      if ( freebasedn ) {
         free( basedn );
      } /* endif */
      return ( CallingInvalid );
   } /* endif */
   scope = v->intval;
   filter = strtoutf8( RXSTRPTR(argv[3]) );
   if ( filter ) {
      freefilter = 1;
   } else {
      filter = RXSTRPTR(argv[3]);
   } /* endif */
   if ( !RXNULLSTRING(argv[4]) && RXSTRLEN(argv[4]) ) {
      attrs = getldapattrs( &argv[4] );
      if ( !attrs ) {
         if ( freebasedn ) {
            free( basedn );
         } /* endif */
         if ( freefilter ) {
            free( filter );
         } /* endif */
         return ( CallingInvalid );
      } /* endif */
   } /* endif */
   attrsonly = rxtoi( argv[5] );
   if ( !RXNULLSTRING(argv[6]) && RXSTRLEN(argv[6]) ) {
      serverctrls = getldapcontrols( &argv[6] );
      if ( !serverctrls ) {
         if ( freebasedn ) {
            free( basedn );
         } /* endif */
         if ( freefilter ) {
            free( filter );
         } /* endif */
         freestrs( attrs );
         return ( CallingInvalid );
      } /* endif */
   } /* endif */
   if ( !RXNULLSTRING(argv[7]) && RXSTRLEN(argv[7]) ) {
      clientctrls = getldapcontrols( &argv[7] );
      if ( !clientctrls ) {
         ldap_controls_free( serverctrls );
         if ( freebasedn ) {
            free( basedn );
         } /* endif */
         if ( freefilter ) {
            free( filter );
         } /* endif */
         freestrs( attrs );
         return ( CallingInvalid );
      } /* endif */
   } /* endif */
   if ( parsetimeval( &argv[8], &timeout ) != Good ) {
      ldap_controls_free( serverctrls );
      if ( freebasedn ) {
         free( basedn );
      } /* endif */
      if ( freefilter ) {
         free( filter );
      } /* endif */
      freestrs( attrs );
      return ( CallingInvalid );
   } /* endif */
   sizelimit = rxtoi( argv[9] );
   code = ldap_search_ext_s( ld, basedn, scope, filter, attrs, attrsonly, serverctrls, clientctrls, &timeout, sizelimit, &msg );
   freestrs( attrs );
   if ( freebasedn ) {
      free( basedn );
   } /* endif */
   if ( freefilter ) {
      free( filter );
   } /* endif */
   ldap_controls_free( serverctrls );
   ldap_controls_free( clientctrls );
   value = rxalloc( BUFSIZ );
   if ( RXNULLSTRING( value ) ) {
      return ( CallingInvalid );
   } /* endif */
   SetUnsignedInt2Rx( value, msg );
   if ( setvalue( &argv[10], &value ) != Good ) {
      rxfree( value );
      return ( CallingInvalid );
   } /* endif */
   rxfree( value );
   SetLdapCode( retstr, code );
   return ( CallingValid );
}

ULONG 
Rxldap_is_ldap_url(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   int code = 0;
   if ( argc != 1 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[0]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[0]) ) {
      return ( CallingInvalid );
   } /* endif */
   code = ldap_is_ldap_url( RXSTRPTR(argv[0]) );
   SetSignedInt2Rx( *retstr, code );
   return ( CallingValid );
}

ULONG 
Rxldap_url_parse(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   int code = 0;
   LDAPURLDesc *desc = NULL;
   if ( argc != 2 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[0]) || RXNULLSTRING(argv[1]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[0]) || !RXSTRLEN(argv[1]) ) {
      return ( CallingInvalid );
   } /* endif */
   code = ldap_url_parse( RXSTRPTR(argv[0]), &desc );
   if ( !code ) {
      RXSTRING varname;
      RXSTRING stem;
      RXSTRING value;
      RXSTRING t;
      varname = rxalloc( RXSTRLEN(argv[1]) + BUFSIZ );
      if ( RXNULLSTRING( varname ) ) {
         ldap_free_urldesc( desc );
         return ( CallingInvalid );
      } /* endif */
      stem = rxalloc( RXSTRLEN(argv[1]) + BUFSIZ );
      if ( RXNULLSTRING( stem ) ) {
         rxfree( varname );
         ldap_free_urldesc( desc );
         return ( CallingInvalid );
      } /* endif */
      rxstrcpy( &stem, &argv[1] );
      if ( stem.strptr[RXSTRLEN(stem)-1] != '.' ) {
         rxappendstr( &stem, "." );
      } /* endif */
      rxappendstr( &stem, "!" );
      value = rxalloc( BUFSIZ );
      if ( RXNULLSTRING( value ) ) {
         rxfree( varname );
         rxfree( stem );
         ldap_free_urldesc( desc );
         return ( CallingInvalid );
      } /* endif */
      rxstrcpy( &varname, &stem );
      rxappendstr( &varname, "LUD_HOST" );
      MAKERXSTRING( t, desc->lud_host, strlen( desc->lud_host ) );
      if ( setvalue( &varname, &t ) != Good ) {
         rxfree( varname );
         rxfree( stem );
         rxfree( value );
         ldap_free_urldesc( desc );
         return ( CallingInvalid );
      } /* endif */
      rxstrcpy( &varname, &stem );
      rxappendstr( &varname, "LUD_PORT" );
      rxset_length( &value, strlen( _ltoa( desc->lud_port, RXSTRPTR(value), 10 ) ) );
      if ( setvalue( &varname, &value ) != Good ) {
         rxfree( varname );
         rxfree( stem );
         rxfree( value );
         ldap_free_urldesc( desc );
         return ( CallingInvalid );
      } /* endif */
      setcodename( &value, desc->lud_scope, ScopeNames, sizeof(ScopeNames) / sizeof(Name2Int), 0 );
      rxstrcpy( &varname, &stem );
      rxappendstr( &varname, "LUD_SCOPE" );
      if ( setvalue( &varname, &value ) != Good ) {
         rxfree( varname );
         rxfree( stem );
         rxfree( value );
         ldap_free_urldesc( desc );
         return ( CallingInvalid );
      } /* endif */
      rxstrcpy( &varname, &stem );
      rxappendstr( &varname, "LUD_DN" );
      MAKERXSTRING( t, desc->lud_dn, strlen( desc->lud_dn ) );
      if ( setvalue( &varname, &t ) != Good ) {
         rxfree( varname );
         rxfree( stem );
         rxfree( value );
         ldap_free_urldesc( desc );
         return ( CallingInvalid );
      } /* endif */
      rxstrcpy( &varname, &stem );
      rxappendstr( &varname, "LUD_FILTER" );
      MAKERXSTRING( t, desc->lud_filter, strlen( desc->lud_filter ) );
      if ( setvalue( &varname, &t ) != Good ) {
         rxfree( varname );
         rxfree( stem );
         rxfree( value );
         ldap_free_urldesc( desc );
         return ( CallingInvalid );
      } /* endif */
      rxstrcpy( &varname, &stem );
      rxappendstr( &varname, "LUD_ATTRS." );
      if ( setstrs( &varname, desc->lud_attrs, 0 ) < 0 ) {
         rxfree( varname );
         rxfree( stem );
         rxfree( value );
         return ( CallingInvalid );
      } /* endif */
      rxfree( varname );
      rxfree( stem );
      rxfree( value );
      ldap_free_urldesc( desc );
   } /* endif */
   SetLdapCode( retstr, code );
   return ( CallingValid );
}

ULONG 
Rxldap_url_search(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   int attrsonly = 0;
   int code = 0;
   if ( argc < 2 || argc > 3 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[1]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[1]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP*)rxtoul( argv[0] );
   if ( argc > 2 ) {
      if ( !isnumstr(RXSTRPTR(argv[2])) ) {
         return ( CallingInvalid );
      } /* endif */
      attrsonly = rxtoi( argv[2] );
   } /* endif */
   code = ldap_url_search( ld, RXSTRPTR(argv[1]), attrsonly );
   SetSignedInt2Rx( *retstr, code );
   return ( CallingValid );
}

ULONG 
Rxldap_url_search_s(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   int attrsonly = 0;
   int code = 0;
   LDAPMessage *msg = NULL;
   RXSTRING value;
   if ( argc != 4 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[1]) || RXNULLSTRING(argv[3]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[1]) || !RXSTRLEN(argv[3]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) || !isnumstr(RXSTRPTR(argv[2])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP*)rxtoul( argv[0] );
   attrsonly = rxtoi( argv[2] );
   code = ldap_url_search_s( ld, RXSTRPTR(argv[1]), attrsonly, &msg );
   value = rxalloc( BUFSIZ );
   if ( RXNULLSTRING( value ) ) {
      return ( CallingInvalid );
   } /* endif */
   SetUnsignedInt2Rx( value, msg );
   if ( setvalue( &argv[3], &value ) != Good ) {
      rxfree( value );
      return ( CallingInvalid );
   } /* endif */
   rxfree( value );
   SetLdapCode( retstr, code );
   return ( CallingValid );
}

ULONG 
Rxldap_url_search_st(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   int attrsonly = 0;
   int code = 0;
   LDAPMessage *msg = NULL;
   RXSTRING value;
   struct timeval timeout;
   if ( argc != 5 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[1]) || RXNULLSTRING(argv[3]) || RXNULLSTRING(argv[4]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[1]) || !RXSTRLEN(argv[3]) || !RXSTRLEN(argv[4]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) || !isnumstr(RXSTRPTR(argv[2])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP*)rxtoul( argv[0] );
   attrsonly = rxtoi( argv[2] );
   if ( parsetimeval( &argv[3], &timeout ) != Good ) {
      return ( CallingInvalid );
   } /* endif */
   code = ldap_url_search_st( ld, RXSTRPTR(argv[1]), attrsonly, &timeout, &msg );
   value = rxalloc( BUFSIZ );
   if ( RXNULLSTRING( value ) ) {
      return ( CallingInvalid );
   } /* endif */
   SetUnsignedInt2Rx( value, msg );
   if ( setvalue( &argv[4], &value ) != Good ) {
      rxfree( value );
      return ( CallingInvalid );
   } /* endif */
   rxfree( value );
   SetLdapCode( retstr, code );
   return ( CallingValid );
}

ULONG     
Rxldap_result(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   int msgid = 0;
   int all = 0;
   LDAPMessage *msg = NULL;
   int code = 0;
   RXSTRING value;
   struct timeval timeout;
   struct timeval *timeoutp = NULL;
   Name2Int *v = NULL;
   if ( argc != 5 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[1]) || RXNULLSTRING(argv[4]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[1]) || !RXSTRLEN(argv[4]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP*)rxtoul( argv[0] );
   v = findintfromname( RXSTRPTR(argv[1]), ResultTypes, sizeof(ResultTypes) / sizeof(Name2Int), 1 );
   if ( v ) {
      msgid = v->intval;
   } else if ( isnumstr(RXSTRPTR(argv[1])) ) {
      msgid = rxtoi( argv[1] );
   } else {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXNULLSTRING(argv[2]) && RXSTRLEN(argv[2]) ) {
      if ( isnumstr(RXSTRPTR(argv[2])) ) {
         all = rxtoi( argv[2] );
      } else {
         v = findintfromname( RXSTRPTR(argv[2]), ResAllNames, sizeof(ResAllNames) / sizeof(Name2Int), 0 );
         if ( !v ) {
            return ( CallingInvalid );
         } /* endif */
         all = v->intval;
      } /* endif */
   } /* endif */
   if ( !RXNULLSTRING(argv[3]) && RXSTRLEN(argv[3]) ) {
      if ( parsetimeval( &argv[3], &timeout ) != Good ) {
         return ( CallingInvalid );
      } /* endif */
      timeoutp = &timeout;
   } /* endif */
   code = ldap_result( ld, msgid, all, timeoutp, &msg );
   value = rxalloc( BUFSIZ );
   if ( RXNULLSTRING( value ) ) {
      return ( CallingInvalid );
   } /* endif */
   SetUnsignedInt2Rx( value, msg );
   if ( setvalue( &argv[4], &value ) != Good ) {
      rxfree( value );
      return ( CallingInvalid );
   } /* endif */
   rxfree( value );
   setcodename( retstr, code, ResultTypes, sizeof(ResultTypes) / sizeof(Name2Int), 0 );
   return ( CallingValid );
}

ULONG _Export
Rxldap_msgtype(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAPMessage *msg = NULL;
   int code = 0;
   if ( argc != 1 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) ) {
      return ( CallingInvalid );
   } /* endif */
   msg = (LDAPMessage *)rxtoul( argv[0] );
   code = ldap_msgtype( msg );
   setcodename( retstr, code, ResultTypes, sizeof(ResultTypes) / sizeof(Name2Int), 0 );
   return ( CallingValid );
}   

ULONG _Export
Rxldap_msgid(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAPMessage *msg = NULL;
   int msgid = 0;
   int code = 0;
   if ( argc != 1 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) ) {
      return ( CallingInvalid );
   } /* endif */
   msg = (LDAPMessage *)rxtoul( argv[0] );
   msgid = ldap_msgid( msg );
   SetUnsignedInt2Rx( *retstr, msgid );
   return ( CallingValid );
}   

ULONG     
Rxldap_get_option(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   int code = 0;
   Name2Int *v = NULL;
   int opt = 0;
   void *optval = NULL;
   int intval = 0;
   char *strval = NULL;
   LDAPAPIInfo apiinfo;
   LDAPAPIFeatureInfo finfo;
   int i = 0;
   char *w = NULL;
   RXSTRING varname;
   RXSTRING stem;
   RXSTRING value;
   RXSTRING t;
   if ( argc != 3 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[1]) || RXNULLSTRING(argv[2]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[1]) || !RXSTRLEN(argv[2]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP*)rxtoul( argv[0] );
   v = findintfromname( RXSTRPTR(argv[1]), LdapOptNames, sizeof(LdapOptNames) / sizeof(Name2Int), 0 );
   if ( !v ) {
      return ( CallingInvalid );
   } /* endif */
   opt = v->intval;
   switch ( opt ) {
   case LDAP_OPT_SIZELIMIT:
   case LDAP_OPT_TIMELIMIT:
   case LDAP_OPT_DEREF:   
   case LDAP_OPT_REFERRALS:   
   case LDAP_OPT_PROTOCOL_VERSION:
   case LDAP_OPT_ERROR_NUMBER:
      optval = &intval;
      break;
   case LDAP_OPT_HOST_NAME:
   case LDAP_OPT_ERROR_STRING:
   case LDAP_OPT_MATCHED_DN:
      optval = &strval;
      break;
   case LDAP_OPT_API_INFO:
      apiinfo.ldapai_info_version = LDAP_API_INFO_VERSION;
      apiinfo.ldapai_extensions = NULL;
      optval = &apiinfo;
      break;
   case LDAP_OPT_API_FEATURE_INFO:
      finfo.ldapaif_info_version = LDAP_FEATURE_INFO_VERSION;
      optval = &finfo;
      break;
   default:
     return ( CallingInvalid );
     break;
   } /* endswitch */
   if ( !optval ) {
      return ( CallingInvalid );
   } /* endif */
   code = ldap_get_option( ld, opt, optval );
   value = rxalloc( BUFSIZ );
   if ( RXNULLSTRING( value ) ) {
      return ( CallingInvalid );
   } /* endif */
   switch ( opt ) {
   case LDAP_OPT_SIZELIMIT:
   case LDAP_OPT_TIMELIMIT:
   case LDAP_OPT_PROTOCOL_VERSION:
   case LDAP_OPT_ERROR_NUMBER:
      SetUnsignedInt2Rx( value, intval );
      break;
   case LDAP_OPT_DEREF:   
      setcodename( &value, intval, LdapOptValDeref, sizeof(LdapOptValDeref) / sizeof(Name2Int), 1 );
      break;
   case LDAP_OPT_REFERRALS:   
      setcodename( &value, intval, LdapOptValOnOff, sizeof(LdapOptValOnOff) / sizeof(Name2Int), 1 );
      break;
   case LDAP_OPT_HOST_NAME:   
   case LDAP_OPT_ERROR_STRING:   
   case LDAP_OPT_MATCHED_DN:
      strcpy2rx( &value, strval );
      ldap_memfree( strval );
      break;
   case LDAP_OPT_API_INFO:
   case LDAP_OPT_API_FEATURE_INFO:
      varname = rxalloc( RXSTRLEN(argv[2]) + BUFSIZ );
      if ( RXNULLSTRING( varname ) ) {
         ldap_value_free( apiinfo.ldapai_extensions );
         rxfree( value );
         return ( CallingInvalid );
      } /* endif */
      stem = rxalloc( RXSTRLEN(argv[2]) + BUFSIZ );
      if ( RXNULLSTRING( stem ) ) {
         ldap_value_free( apiinfo.ldapai_extensions );
         rxfree( value );
         rxfree( varname );
         return ( CallingInvalid );
      } /* endif */
      value = rxalloc( BUFSIZ );
      if ( RXNULLSTRING( value ) ) {
         ldap_value_free( apiinfo.ldapai_extensions );
         rxfree( value );
         rxfree( varname );
         rxfree( stem );
         return ( CallingInvalid );
      } /* endif */
      rxstrcpy( &stem, &argv[2] );
      if ( stem.strptr[RXSTRLEN(stem)-1] != '.' ) {
         rxappendstr( &stem, "." );
      } /* endif */
      rxappendstr( &stem, "!" );
      w = malloc( BUFSIZ );
      if ( !w ) {
         ldap_value_free( apiinfo.ldapai_extensions );
         rxfree( value );
         rxfree( varname );
         rxfree( stem );
      } /* endif */
      break;
   } /* endswitch */
   switch ( opt ) {
   case LDAP_OPT_SIZELIMIT:
   case LDAP_OPT_TIMELIMIT:
   case LDAP_OPT_PROTOCOL_VERSION:
   case LDAP_OPT_ERROR_NUMBER:
   case LDAP_OPT_DEREF:   
   case LDAP_OPT_REFERRALS:   
   case LDAP_OPT_HOST_NAME:   
   case LDAP_OPT_ERROR_STRING:   
   case LDAP_OPT_MATCHED_DN:
      if ( setvalue( &argv[2], &value ) != Good ) {
         rxfree( value );
         return ( CallingInvalid );
      } /* endif */
      rxfree( value );
      break;
   case LDAP_OPT_API_INFO:
      rxstrcpy( &varname, &stem );
      rxappendstr( &varname, "LDAPAI_VENDOR_VERSION" );
      rxset_length( &value, strlen( _ltoa( (ULONG)apiinfo.ldapai_vendor_version, RXSTRPTR(value), 10 ) ) );
      if ( setvalue( &varname, &value ) != Good ) {
         ldap_value_free( apiinfo.ldapai_extensions );
         rxfree( varname );
         rxfree( stem );
         rxfree( value );
         free( w );
         return ( CallingInvalid );
      } /* endif */
      rxstrcpy( &varname, &stem );
      rxappendstr( &varname, "LDAPAI_API_VERSION" );
      rxset_length( &value, strlen( _ltoa( (ULONG)apiinfo.ldapai_api_version, RXSTRPTR(value), 10 ) ) );
      if ( setvalue( &varname, &value ) != Good ) {
         ldap_value_free( apiinfo.ldapai_extensions );
         rxfree( varname );
         rxfree( stem );
         rxfree( value );
         free( w );
         return ( CallingInvalid );
      } /* endif */
      rxstrcpy( &varname, &stem );
      rxappendstr( &varname, "LDAPAI_PROTOCOL_VERSION" );
      rxset_length( &value, strlen( _ltoa( (ULONG)apiinfo.ldapai_protocol_version, RXSTRPTR(value), 10 ) ) );
      if ( setvalue( &varname, &value ) != Good ) {
         ldap_value_free( apiinfo.ldapai_extensions );
         rxfree( varname );
         rxfree( stem );
         rxfree( value );
         free( w );
         return ( CallingInvalid );
      } /* endif */
      rxstrcpy( &varname, &stem );
      rxappendstr( &varname, "LDAPAI_VENDOR_NAME" );
      MAKERXSTRING( t, apiinfo.ldapai_vendor_name, strlen(apiinfo.ldapai_vendor_name) );
      if ( setvalue( &varname, &t ) != Good ) {
         ldap_value_free( apiinfo.ldapai_extensions );
         rxfree( varname );
         rxfree( stem );
         rxfree( value );
         free( w );
         return ( CallingInvalid );
      } /* endif */
      if ( apiinfo.ldapai_extensions ) {
         for ( i = 0; apiinfo.ldapai_extensions[i]; i += 1) {
            rxstrcpy( &varname, &stem );
            rxappendstr( &varname, "LDAPAI_EXTENSIONS." );
            rxappendstr( &varname, _itoa(i + 1, w, 10) );
            MAKERXSTRING( t, apiinfo.ldapai_extensions[i], strlen(apiinfo.ldapai_extensions[i]) );
            if ( setvalue( &varname, &t ) != Good ) {
               ldap_value_free( apiinfo.ldapai_extensions );
               rxfree( value );
               rxfree( varname );
               rxfree( stem );
               free( w );
               return ( CallingInvalid );
            } /* endif */
         } /* endfor */
      } /* endif */
      rxstrcpy( &varname, &stem );
      rxappendstr( &varname, "LDAPAI_EXTENSIONS." );
      rxappendstr( &varname, "0" );
      rxset_length( &value, strlen( _ltoa( i, RXSTRPTR(value), 10 ) ) );
      if ( setvalue( &varname, &value ) != Good ) {
         ldap_value_free( apiinfo.ldapai_extensions );
         rxfree( value );
         rxfree( varname );
         rxfree( stem );
         free( w );
         return ( CallingInvalid );
      } /* endif */
      ldap_value_free( apiinfo.ldapai_extensions );
      rxfree( varname );
      rxfree( stem );
      rxfree( value );
      free( w );
      break;
   case LDAP_OPT_API_FEATURE_INFO:
      rxstrcpy( &varname, &stem );
      rxappendstr( &varname, "LDAPAIF_VERSION" );
      rxset_length( &value, strlen( _ltoa( (ULONG)finfo.ldapaif_version, RXSTRPTR(value), 10 ) ) );
      if ( setvalue( &varname, &value ) != Good ) {
         rxfree( varname );
         rxfree( stem );
         rxfree( value );
         free( w );
         return ( CallingInvalid );
      } /* endif */
      rxstrcpy( &varname, &stem );
      rxappendstr( &varname, "LDAPAIF_NAME" );
      MAKERXSTRING( t, finfo.ldapaif_name, strlen(finfo.ldapaif_name) );
      if ( setvalue( &varname, &t ) != Good ) {
         rxfree( varname );
         rxfree( stem );
         rxfree( value );
         free( w );
         return ( CallingInvalid );
      } /* endif */
      rxfree( varname );
      rxfree( stem );
      rxfree( value );
      free( w );
      break;
   } /* endswitch */
   SetSignedInt2Rx( *retstr, code );
   return ( CallingValid );
}

ULONG     
Rxldap_set_option(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   int code = 0;
   Name2Int *v = NULL;
   int opt = 0;
   void *optval = NULL;
   int intval = 0;
   char *strval = NULL;
   if ( argc != 3 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[1]) || RXNULLSTRING(argv[2]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[1]) || !RXSTRLEN(argv[2]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP*)rxtoul( argv[0] );
   v = findintfromname( RXSTRPTR(argv[1]), LdapOptNames, sizeof(LdapOptNames) / sizeof(Name2Int), 0 );
   if ( !v ) {
      return ( CallingInvalid );
   } /* endif */
   opt = v->intval;
   switch ( opt ) {
   case LDAP_OPT_SIZELIMIT:
   case LDAP_OPT_TIMELIMIT:
   case LDAP_OPT_PROTOCOL_VERSION:
      if ( !isnumstr(RXSTRPTR(argv[2])) ) {
         return ( CallingInvalid );
      } /* endif */
      intval = rxtoi( argv[2] );
      optval = &intval;
      break;
   case LDAP_OPT_DEREF:   
      SetIntFromName( RXSTRPTR(argv[2]), LdapOptValDeref, v, intval, optval );
      break;
   case LDAP_OPT_REFERRALS:   
      SetIntFromName( RXSTRPTR(argv[2]), LdapOptValOnOff, v, intval, optval );
      break;
   case LDAP_OPT_SERVER_CONTROLS:
   case LDAP_OPT_CLIENT_CONTROLS:
      optval = getldapcontrols( &argv[2] );
      break;
   default:
     return ( CallingInvalid );
     break;
   } /* endswitch */
   if ( !optval ) {
      return ( CallingInvalid );
   } /* endif */
   code = ldap_set_option( ld, opt, optval );
   switch ( opt ) {
   case LDAP_OPT_SERVER_CONTROLS:
   case LDAP_OPT_CLIENT_CONTROLS:
      ldap_controls_free( optval );
      break;
   } /* endswitch */
   SetSignedInt2Rx( *retstr, code );
   return ( CallingValid );
}

ULONG _Export
Rxldap_get_errno(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   int code = 0;
   if ( argc != 1 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP*)rxtoul( argv[0] );
   ldap_get_option( ld, LDAP_OPT_ERROR_NUMBER, &code );
   SetLdapCode( retstr, code );
   return ( CallingValid );
}   

ULONG _Export
Rxldap_err2string(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   int err = 0;
   char *str = NULL;
   if ( argc != 1 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[0]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[0]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( isnumstr(RXSTRPTR(argv[0])) ) {
      err = rxtoi( argv[0] );
   } else {   
      Name2Int *v = findintfromname( RXSTRPTR(argv[0]), LdapReturnCodes, sizeof(LdapReturnCodes) / sizeof(Name2Int), 1 );
      if ( !v ) {
         rxreturn_value( retstr, RXSTRPTR(argv[0]), RXSTRLEN(argv[0]) );
         return ( CallingValid );
      } /* endif */
      err = v->intval;
   } /* endif */
   str = ldap_err2string( err );
   if ( str ) {
      rxreturn_value( retstr, str, strlen(str) );
   } else { 
      rxset_zerolen( retstr );
   } /* endif */
   return ( CallingValid );
}   

ULONG _Export
Rxldap_count_entries(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   LDAPMessage *msg = NULL;
   int count = 0;
   if ( argc != 2 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) || !isnumstr(RXSTRPTR(argv[1])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   msg = (LDAPMessage *)rxtoul( argv[1] );
   count = ldap_count_entries( ld, msg );
   SetSignedInt2Rx( *retstr, count );
   return ( CallingValid );
}   

ULONG _Export
Rxldap_parse_result(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   LDAPMessage *msg = NULL;
   char *matcheddn = NULL;
   char **matcheddnp = NULL;
   int errcode = 0;
   int *errcodep = NULL;
   char *errmsg = NULL;
   char **errmsgp = NULL;
   LDAPControl **serverctrls = NULL;
   LDAPControl ***serverctrlsp = NULL;
   char **referrals = NULL;
   char ***referralsp = NULL;
   int freeit = 0;
   int code = 0;
   RXSTRING value;
   RXSTRING varname;
   RXSTRING stem;
   RXSTRING t;
   if ( argc != 8 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) || !isnumstr(RXSTRPTR(argv[1])) || !isnumstr(RXSTRPTR(argv[7])) ) {
      return ( CallingInvalid );
   } /* endif */  
   ld = (LDAP *)rxtoul( argv[0] );
   msg = (LDAPMessage *)rxtoul( argv[1] );
   if ( !RXNULLSTRING(argv[2]) && RXSTRLEN(argv[2]) ) {
      errcodep = &errcode;
   } /* endif */
   if ( !RXNULLSTRING(argv[3]) && RXSTRLEN(argv[3]) ) {
      matcheddnp = &matcheddn;
   } /* endif */
   if ( !RXNULLSTRING(argv[4]) && RXSTRLEN(argv[4]) ) {
   errmsgp = &errmsg;
   } /* endif */
   if ( !RXNULLSTRING(argv[5]) && RXSTRLEN(argv[5]) ) {
      referralsp = &referrals;
   } /* endif */
   if ( !RXNULLSTRING(argv[6]) && RXSTRLEN(argv[6]) ) {
      serverctrlsp = &serverctrls;
   } /* endif */
   freeit = rxtoi( argv[7] );
   code = ldap_parse_result( ld, msg, errcodep, matcheddnp, errmsgp, referralsp, serverctrlsp, freeit );
   if ( errcodep ) {
      char *w = malloc( BUFSIZ );
      if ( !w ) {
         return ( CallingInvalid );
      } /* endif */
      MAKERXSTRING( value, w, BUFSIZ );
      SetLdapCode( &value, errcode );
      if ( setvalue( &argv[2], &value ) != Good ) {
         free( w );
         return ( CallingInvalid );
      } /* endif */
      free( w );
   } /* endif */
   if ( matcheddnp ) {
      if ( matcheddn ) {
         MAKERXSTRING( value, matcheddn, strlen(matcheddn) );
         if ( setvalue( &argv[3], &value ) != Good ) {
            return ( CallingInvalid );
         } /* endif */
         ldap_memfree( matcheddn );
      } else {
         rxset_zerolen( &value );
         if ( setvalue( &argv[3], &value ) != Good ) {
            return ( CallingInvalid );
         } /* endif */
      } /* endif */
   } /* endif */
   if ( errmsgp ) {
      if ( errmsg ) {
         MAKERXSTRING( value, errmsg, strlen(errmsg) );
         if ( setvalue( &argv[4], &value ) != Good ) {
            return ( CallingInvalid );
         } /* endif */
         ldap_memfree( errmsg );
      } else {
         rxset_zerolen( &value );
         if ( setvalue( &argv[4], &value ) != Good ) {
            return ( CallingInvalid );
         } /* endif */
      } /* endif */
   } /* endif */
   if ( setstrs( &argv[5], referrals, 1 ) < 0 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( setldapctrls( &argv[6], serverctrls ) != Good ) {
      return ( CallingInvalid );
   } /* endif */
   ldap_value_free( referrals );
   ldap_controls_free( serverctrls );
   SetLdapCode( retstr, code );
   return ( CallingValid );
}   

ULONG _Export
Rxldap_first_entry(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   LDAPMessage *msg = NULL;
   if ( argc != 2 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) || !isnumstr(RXSTRPTR(argv[1])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   msg = (LDAPMessage *)rxtoul( argv[1] );
   msg = ldap_first_entry( ld, msg );
   SetUnsignedInt2Rx( *retstr, msg );
   return ( CallingValid );
}   

ULONG _Export
Rxldap_next_entry(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   LDAPMessage *msg = NULL;
   if ( argc != 2 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) || !isnumstr(RXSTRPTR(argv[1])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   msg = (LDAPMessage *)rxtoul( argv[1] );
   msg = ldap_next_entry( ld, msg );
   SetUnsignedInt2Rx( *retstr, msg );
   return ( CallingValid );
}   

ULONG _Export
Rxldap_count_references(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   LDAPMessage *msg = NULL;
   int count = 0;
   if ( argc != 2 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) || !isnumstr(RXSTRPTR(argv[1])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   msg = (LDAPMessage *)rxtoul( argv[1] );
   count = ldap_count_references( ld, msg );
   SetSignedInt2Rx( *retstr, count );
   return ( CallingValid );
}   

ULONG _Export
Rxldap_first_reference(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   LDAPMessage *msg = NULL;
   if ( argc != 2 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) || !isnumstr(RXSTRPTR(argv[1])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   msg = (LDAPMessage *)rxtoul( argv[1] );
   msg = ldap_first_reference( ld, msg );
   SetUnsignedInt2Rx( *retstr, msg );
   return ( CallingValid );
}   

ULONG _Export
Rxldap_next_reference(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   LDAPMessage *msg = NULL;
   if ( argc != 2 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) || !isnumstr(RXSTRPTR(argv[1])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   msg = (LDAPMessage *)rxtoul( argv[1] );
   msg = ldap_next_reference( ld, msg );
   SetUnsignedInt2Rx( *retstr, msg );
   return ( CallingValid );
}   

ULONG _Export
Rxldap_abandon(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   int msgid = 0;
   int code = 0;
   if ( argc != 2 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) || !isnumstr(RXSTRPTR(argv[1])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   msgid = rxtoi( argv[1] );
   code = ldap_abandon( ld, msgid );
   SetSignedInt2Rx( *retstr, code );
   return ( CallingValid );
}   

ULONG _Export
Rxldap_abandon_ext(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   int msgid = 0;
   int code = 0;
   LDAPControl **serverctrls = NULL;
   LDAPControl **clientctrls = NULL;
   if ( argc < 2 || argc > 4 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) || !isnumstr(RXSTRPTR(argv[1])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   msgid = rxtoi( argv[1] );
   if ( argc > 2 && !RXNULLSTRING(argv[2]) && RXSTRLEN(argv[2]) ) {
      serverctrls = getldapcontrols( &argv[2] );
      if ( !serverctrls ) {
         return ( CallingInvalid );
      } /* endif */
   } /* endif */
   if ( argc > 3 && !RXNULLSTRING(argv[3]) && RXSTRLEN(argv[3]) ) {
      serverctrls = getldapcontrols( &argv[3] );
      if ( !serverctrls ) {
         return ( CallingInvalid );
      } /* endif */
   } /* endif */
   code = ldap_abandon_ext( ld, msgid, serverctrls, clientctrls );
   SetLdapCode( retstr, code );
   return ( CallingValid );
}   

ULONG _Export
Rxldap_msgfree(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAPMessage *msg = NULL;
   int code = 0;
   if ( argc != 1 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr( RXSTRPTR(argv[0])) ) {
      return ( CallingInvalid );
   } /* endif */
   msg = (LDAPMessage *)rxtoul( argv[0] );
   code = ldap_msgfree( msg );
   setcodename( retstr, code, ResultTypes, sizeof(ResultTypes) / sizeof(Name2Int), 0 );
   return ( CallingValid );
}   

ULONG _Export
Rxldap_ber_free(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   BerElement *ber = NULL;
   if ( argc != 1 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) ) {
      return ( CallingInvalid );
   } /* endif */
   ber = (BerElement *)rxtoul( argv[0] );
   ber_free( ber, 1 );
   rxset_zerolen( retstr );
   return ( CallingValid );
}   

ULONG _Export
Rxldap_get_dn(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   LDAPMessage *msg = NULL;
   char *dn;
   if ( argc != 2 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) || !isnumstr(RXSTRPTR(argv[1])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   msg = (LDAPMessage *)rxtoul( argv[1] );
   dn = ldap_get_dn( ld, msg );
   if ( dn ) {
      SetRetStr( *retstr, dn );
      ldap_memfree( dn );
   } else {
      rxset_zerolen( retstr );
   } /* endif */
   return ( CallingValid );
}   

ULONG _Export
Rxldap_explode_dn(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   int notypes = 0;
   char **v = NULL;
   int i = 0;
   int stem0 = 0;
   if ( argc != 3 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[2]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[2]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[1])) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[0]) || !RXSTRLEN(argv[0]) ) {
      if ( setstrs( &argv[2], NULL, 0 ) < 0 ) {
         return ( CallingInvalid );
      } /* endif */
      SetSignedInt2Rx( *retstr, 0 );
      return ( CallingValid );
   } /* endif */
   notypes = rxtoi( argv[1] );
   v = ldap_explode_dn( RXSTRPTR(argv[0]), notypes );
   stem0 = setstrs( &argv[2], v, 0 );
   if ( stem0 < 0 ) {
      ldap_value_free( v );
      return ( CallingInvalid );
   } /* endif */
   ldap_value_free( v );
   SetSignedInt2Rx( *retstr, stem0 );
   return ( CallingValid );
}   

ULONG _Export
Rxldap_first_attribute(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   LDAPMessage *msg = NULL;
   BerElement *ber = NULL;
   char *an = NULL;
   RXSTRING value;
   if ( argc != 3 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[2]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[2]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) || !isnumstr(RXSTRPTR(argv[1])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   msg = (LDAPMessage *)rxtoul( argv[1] );
   an = ldap_first_attribute( ld, msg, &ber );
   if ( an ) {
      SetRetStr( *retstr, an );
      ldap_memfree( an );
   } else {
      rxset_zerolen( retstr );
   } /* endif */
   value = rxalloc( BUFSIZ );
   if ( RXNULLSTRING( value ) ) {
      return ( CallingInvalid );
   } /* endif */
   SetUnsignedInt2Rx( value, ber );
   if ( setvalue( &argv[2], &value ) != Good ) {
      rxfree( value );
      return ( CallingInvalid );
   } /* endif */
   return ( CallingValid );
}   

ULONG _Export
Rxldap_next_attribute(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   LDAPMessage *msg = NULL;
   BerElement *ber = NULL;
   char *an = NULL;
   if ( argc != 3 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) || !isnumstr(RXSTRPTR(argv[1])) || !isnumstr(RXSTRPTR(argv[2])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   msg = (LDAPMessage *)rxtoul( argv[1] );
   ber = (BerElement *)rxtoul( argv[2] );
   an = ldap_next_attribute( ld, msg, ber );
   if ( an ) {
      SetRetStr( *retstr, an );
      ldap_memfree( an );
   } else {
      rxset_zerolen( retstr );
   } /* endif */
   return ( CallingValid );
}   

ULONG _Export
Rxldap_get_values(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   LDAPMessage *msg = NULL;
   char **v = NULL;
   int stem0 = 0;
   if ( argc != 4 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[2]) || RXNULLSTRING(argv[3]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[2]) || !RXSTRLEN(argv[3]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) || !isnumstr(RXSTRPTR(argv[1])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   msg = (LDAPMessage *)rxtoul( argv[1] );
   v = ldap_get_values( ld, msg, RXSTRPTR(argv[2]) );
   stem0 = setstrs( &argv[3], v, 1 );
   if ( stem0 < 0 ) {
      ldap_value_free( v );
      return ( CallingInvalid );
   } /* endif */
   ldap_value_free( v );
   SetSignedInt2Rx( *retstr, stem0 );
   return ( CallingValid );
}   

ULONG _Export
Rxldap_get_values_len(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   LDAPMessage *msg = NULL;
   struct berval **v = NULL;
   RXSTRING varname;
   RXSTRING stem;
   RXSTRING value;
   int i = 0;
   char *w = NULL;
   int stem0 = 0;
   if ( argc != 4 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( RXNULLSTRING(argv[2]) || RXNULLSTRING(argv[3]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !RXSTRLEN(argv[2]) || !RXSTRLEN(argv[3]) ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) || !isnumstr(RXSTRPTR(argv[1])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   msg = (LDAPMessage *)rxtoul( argv[1] );
   varname = rxalloc( RXSTRLEN(argv[3]) + BUFSIZ );
   if ( RXNULLSTRING( varname ) ) {
      return ( CallingInvalid );
   } /* endif */
   stem = rxalloc( RXSTRLEN(argv[3]) + BUFSIZ );
   if ( RXNULLSTRING( stem ) ) {
      rxfree( varname );
      return ( CallingInvalid );
   } /* endif */
   rxstrcpy( &stem, &argv[3] );
   if ( stem.strptr[RXSTRLEN(stem)-1] != '.' ) {
      rxappendstr( &stem, "." );
   } /* endif */
   w = malloc( BUFSIZ );
   if ( !w ) {
      rxfree( varname );
      rxfree( stem );
      return ( CallingInvalid );
   } /* endif */
   v = ldap_get_values_len( ld, msg, RXSTRPTR(argv[2]) );
   if ( v ) {
      for ( i = 0; v[i]; i += 1) {
         rxstrcpy( &varname, &stem );
         rxappendstr( &varname, _itoa(i + 1, w, 10) );
         MAKERXSTRING( value, v[i]->bv_val, v[i]->bv_len );
         if ( setvalue( &varname, &value ) != Good ) {
            rxfree( varname );
            rxfree( stem );
            free( w );
            return ( CallingInvalid );
         } /* endif */
      } /* endfor */
      ldap_value_free_len( v );
      stem0 = i;
   } /* endif */
   rxstrcpy( &varname, &stem );
   rxappendstr( &varname, "0" );
   _itoa( i, w, 10 );
   MAKERXSTRING( value, w, strlen(w) );
   if ( setvalue( &varname, &value ) != Good ) {
      rxfree( varname );
      rxfree( stem );
      free( w );
      return ( CallingInvalid );
   } /* endif */
   rxfree( varname );
   rxfree( stem );
   free( w );
   SetSignedInt2Rx( *retstr, stem0 );
   return ( CallingValid );
}   

ULONG _Export
Rxldap_count_messages(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   LDAPMessage *msg = NULL;
   int count = 0;
   if ( argc != 2 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) || !isnumstr(RXSTRPTR(argv[1])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   msg = (LDAPMessage *)rxtoul( argv[1] );
   count = ldap_count_messages( ld, msg );
   SetSignedInt2Rx( *retstr, count );
   return ( CallingValid );
}   

ULONG _Export
Rxldap_first_message(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   LDAPMessage *msg = NULL;
   if ( argc != 2 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) || !isnumstr(RXSTRPTR(argv[1])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   msg = (LDAPMessage *)rxtoul( argv[1] );
   msg = ldap_first_message( ld, msg );
   SetUnsignedInt2Rx( *retstr, msg );
   return ( CallingValid );
}   

ULONG _Export
Rxldap_next_message(
PUCHAR name, 
ULONG argc,
RXSTRING argv[],
PSZ queuename,
PRXSTRING retstr) {
   LDAP *ld = NULL;
   LDAPMessage *msg = NULL;
   if ( argc != 2 ) {
      return ( CallingInvalid );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(argv[0])) || !isnumstr(RXSTRPTR(argv[1])) ) {
      return ( CallingInvalid );
   } /* endif */
   ld = (LDAP *)rxtoul( argv[0] );
   msg = (LDAPMessage *)rxtoul( argv[1] );
   msg = ldap_next_message( ld, msg );
   SetUnsignedInt2Rx( *retstr, msg );
   return ( CallingValid );
}   

static int 
getvalue(
const PRXSTRING name,
PRXSTRING value) {
   APIRET code = 0;
   SHVBLOCK shvblock;
   shvblock.shvnext = NULL;
   shvblock.shvname = *name;
   shvblock.shvnamelen = 0;
   rxset_null(&shvblock.shvvalue);
   shvblock.shvvaluelen = 0;
   shvblock.shvcode = RXSHV_SYFET;
   RexxVariablePool(&shvblock);
   *value = rxstrdup( shvblock.shvvalue );
   code = DosFreeMem(RXSTRPTR(shvblock.shvvalue));
   if ( code != NO_ERROR ) {
      return ( Bad );
   } /* endif */
   return ( shvblock.shvret );
}

static int 
setvalue(
const PRXSTRING name,
const PRXSTRING value) {
   ULONG rc = 0;
   SHVBLOCK shvblock;
   shvblock.shvnext = NULL;
   shvblock.shvname = *name;
   shvblock.shvnamelen = RXSTRLEN(*name);
   shvblock.shvvalue = *value;
   shvblock.shvvaluelen = RXSTRLEN(*value);
   shvblock.shvcode = RXSHV_SYSET;
   rc = RexxVariablePool(&shvblock);
   if (RXSHV_NOAVL == rc || ~(RXSHV_OK | RXSHV_NEWV) & shvblock.shvret) {
      return ( Bad );
   } /* endif */
   return ( Good );
}

static VOID
rxappendstr(
PRXSTRING rxstr,
const PCHAR str ) {
   RXSTRING t;
   MAKERXSTRING( t, str, strlen( str ) );
   rxstrcat( rxstr, &t );
}

static int _Optlink 
compname2intname( 
const void *item1, 
const void *item2 ) {
   const Name2Int *i1 = (Name2Int*)item1;
   const Name2Int *i2 = (Name2Int*)item2;
   return ( stricmp( i1->name, i2->name ) );
}

static int _Optlink 
compname2intint( 
const void *item1, 
const void *item2 ) {
   const Name2Int *i1 = (Name2Int*)item1;
   const Name2Int *i2 = (Name2Int*)item2;
   if ( i1->intval < i2->intval ) {
      return ( -1 );
   } else if ( i1->intval > i2->intval ) {
      return ( 1 );
   } /* endif */
   return ( 0 );
}

static int _Optlink 
compname2strstr( 
const void *item1, 
const void *item2 ) {
   const Name2Str *i1 = (Name2Str*)item1;
   const Name2Str *i2 = (Name2Str*)item2;
   return ( strcmp( i1->strval, i2->strval ) );
}

static Name2Int *
findintfromname( 
char *key, 
const Name2Int *base, 
unsigned int num,
int lflag ) {
   Name2Int k;
   Name2Int *v = NULL;
   const unsigned int w = sizeof(Name2Int);
   k.name = key;
   if ( num < 10 || lflag ) {
      v = (Name2Int *)lfind( (const char *)&k, (const char *)base, &num, w, compname2intname );
   } else {
      v = (Name2Int *)bsearch( &k, base, num, w, compname2intname );
   } /* endif */
   return ( v );
}

static Name2Int *
findnamefromint( 
int key, 
const Name2Int *base, 
unsigned int num,
int lflag ) {
   Name2Int k;
   Name2Int *v = NULL;
   const unsigned int w = sizeof(Name2Int);
   k.intval = key;
   if ( num < 10 || lflag ) {
      v = (Name2Int *)lfind( (const char *)&k, (const char *)base, &num, w, compname2intint );
   } else {
      v = (Name2Int *)bsearch( &k, base, num, w, compname2intint );
   } /* endif */
   return ( v );
}

static Name2Str *
findnamefromstr( 
const char *key, 
const Name2Str *base, 
unsigned int num,
int lflag ) {
   Name2Str k;
   Name2Str *v = NULL;
   const unsigned int w = sizeof(Name2Str);
   k.strval = (char *)key;
   if ( num < 10 || lflag ) {
      v = (Name2Str *)lfind( (const char *)&k, (const char *)base, &num, w, compname2strstr );
   } else {
      v = (Name2Str *)bsearch( &k, base, num, w, compname2strstr );
   } /* endif */
   return ( v );
}

static void 
freestrs( 
char **a ) {
   int i;
   if ( !a ) {
      return;
   } /* endif */
   for ( i = 0; a[i]; i += 1) {
      free(a[i]);
   } /* endfor */
   free(a);
}   

static int 
parsetimeval( 
const PRXSTRING rxstr, 
struct timeval *tv ) {
   PUCHAR sec = NULL;
   PUCHAR usec = NULL;
   PUCHAR p = NULL;
   if ( isnumstr(RXSTRPTR(*rxstr)) ) {
      tv->tv_sec = rxtoi( *rxstr );
      tv->tv_usec = 0;
      return ( Good );
   } /* endif */
   sec = strdup( RXSTRPTR(*rxstr) );
   if ( !sec ) {
      return ( Bad );
   } /* endif */
   p = strchr( sec, '.' );
   if ( !p ) {
      free( sec );
      return ( Bad );
   } /* endif */
   *p = '\0';
   usec = p + 1;
   if ( !isnumstr(sec) || !isnumstr(usec) ) {
      free( sec );
      return ( Bad );
   } /* endif */
   tv->tv_sec = atoi( sec );
   tv->tv_usec = atoi( usec );
   free( sec );
   return ( Good );
}

static void 
setcodename( 
PRXSTRING rxstr, 
int code, 
const Name2Int *base, 
unsigned int num, 
int lflag ) {
   Name2Int *t = findnamefromint( code, base, num, lflag );
   if ( !t ) {
      SetSignedInt2Rx( *rxstr, code );
      return;
   } /* endif */
   SetRetStr( *rxstr, t->name );
}

static void 
setstrname( 
PRXSTRING rxstr, 
const char *code, 
const Name2Str *base, 
unsigned int num, 
int lflag ) {
   Name2Str *t = findnamefromstr( code, base, num, lflag );
   if ( !t ) {
      SetSignedInt2Rx( *rxstr, code );
      return;
   } /* endif */
   SetRetStr( *rxstr, t->name );
}

static BOOL
isnumstr( 
const PCHAR s ) {
   int i;
   if ( !s ) {
      return ( FALSE );
   } /* endif */
   i = strspn( s, "0123456789" );
   if ( i < strlen( s ) ) {
      return ( FALSE );
   } /* endif */
   return ( TRUE );
}

static LDAPMod **
getldapmods( PRXSTRING stemname ) {
   RXSTRING stem;
   RXSTRING varname;
   RXSTRING value;
   char *w = NULL;
   LDAPMod **mods = NULL;
   int n = 0;
   int i = 0;
   stem = rxalloc( RXSTRLEN(*stemname) + BUFSIZ );
   if ( RXNULLSTRING( stem ) ) {
      return ( NULL );
   } /* endif */ 
   varname = rxalloc( RXSTRLEN(*stemname) + BUFSIZ );
   if ( RXNULLSTRING( varname ) ) {
      rxfree( stem );
      return ( NULL );
   } /* endif */
   rxstrcpy( &stem, stemname );
   if ( stem.strptr[RXSTRLEN(stem)-1] != '.' ) {
      rxappendstr( &stem, "." );
   } /* endif */
   rxstrcpy( &varname, &stem );
   rxappendstr( &varname, "0" );
   if ( getvalue(&varname, &value) != Good ) {
      rxfree( stem );
      rxfree( varname );
      return ( NULL );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(value)) ) {
      rxfree( stem );
      rxfree( varname );
      rxfree( value );
      return ( NULL );
   } /* endif */
   n = rxtoi( value );
   rxfree( value );
   w = malloc( BUFSIZ );
   if ( !w ) {
      rxfree( stem );
      rxfree( varname );
      return ( NULL );
   } /* endif */
   mods = calloc( n + 1, sizeof(LDAPMod*) );
   if ( !mods ) {
      rxfree( stem );
      rxfree( varname );
      free( w );
      return ( NULL );
   } /* endif */
   for ( i = 0; i < n; i += 1) {
      int m = 0;
      int j = 0;
      Name2Int *v = NULL;
      mods[i] = calloc( 1, sizeof(LDAPMod) );
      if ( !mods[i] ) {
         ldap_mods_free( mods, 1 );
         rxfree( stem );
         rxfree( varname );
         free( w );
         return ( NULL );
      } /* endif */
      rxstrcpy( &varname, &stem );
      rxappendstr( &varname, _itoa(i + 1, w, 10) );
      rxappendstr( &varname, ".!MOD_OP" );
      if ( getvalue(&varname, &value) != Good ) {
         ldap_mods_free( mods, 1 );
         rxfree( stem );
         rxfree( varname );
         free( w );
         return ( NULL );
      } /* endif */
      if ( isnumstr( RXSTRPTR(value) ) ) {
         mods[i]->mod_op = rxtoi(value); 
      } else {
         v = findintfromname( RXSTRPTR(value), OpNames, sizeof(OpNames) / sizeof(Name2Int), 0 );
         rxfree( value );
         if ( !v ) {
            rxfree( stem );
            rxfree( varname );
            free( w );
            return ( NULL );
         } /* endif */
         mods[i]->mod_op = v->intval;
      } /* endif */
      rxstrcpy( &varname, &stem );
      rxappendstr( &varname, _itoa(i + 1, w, 10) );
      rxappendstr( &varname, ".!MOD_TYPE" );
      if ( getvalue(&varname, &value) != Good ) {
         ldap_mods_free( mods, 1 );
         rxfree( stem );
         rxfree( varname );
         free( w );
         return ( NULL );
      } /* endif */
      mods[i]->mod_type = strdup( RXSTRPTR(value) );
      rxfree( value );
      if ( !mods[i]->mod_type ) {
         ldap_mods_free( mods, 1 );
         rxfree( stem );
         rxfree( varname );
         free( w );
         return ( NULL );
      } /* endif */
      rxstrcpy( &varname, &stem );
      rxappendstr( &varname, _itoa(i + 1, w, 10) );
      rxappendstr( &varname, ".!MODV.0" );
      if ( getvalue(&varname, &value) != Good ) {
         ldap_mods_free( mods, 1 );
         rxfree( stem );
         rxfree( varname );
         free( w );
         return ( NULL );
      } /* endif */
      if ( !isnumstr(RXSTRPTR(value)) ) {
         ldap_mods_free( mods, 1 );
         rxfree( stem );
         rxfree( varname );
         rxfree( value );
         free( w );
         return ( NULL );
      } /* endif */
      m = rxtoi( value );
      rxfree( value );
      mods[i]->mod_bvalues = calloc( m + 1, sizeof(struct berval *) );
      if ( !mods[i]->mod_bvalues ) {
         ldap_mods_free( mods, 1 );
         rxfree( stem );
         rxfree( varname );
         free( w );
         return ( NULL );
      } /* endif */
      for ( j = 0; j < m; j += 1) {
         mods[i]->mod_bvalues[j] = calloc( 1, sizeof(struct berval) );
         if ( !mods[i]->mod_bvalues[j] ) {
            ldap_mods_free( mods, 1 );
            rxfree( stem );
            rxfree( varname );
            free( w );
            return ( NULL );
         } /* endif */
         rxstrcpy( &varname, &stem );
         rxappendstr( &varname, _itoa(i + 1, w, 10) );
         rxappendstr( &varname, ".!MODV." );
         rxappendstr( &varname, _itoa(j + 1, w, 10) );
         if ( getvalue(&varname, &value) != Good ) {
            ldap_mods_free( mods, 1 );
            rxfree( stem );
            rxfree( varname );
            free( w );
            return ( NULL );
         } /* endif */
         if ( mods[i]->mod_op & LDAP_MOD_BVALUES ) {
            mods[i]->mod_bvalues[j]->bv_val = malloc( RXSTRLEN(value) );
            if ( !mods[i]->mod_bvalues[j]->bv_val ) {
               ldap_mods_free( mods, 1 );
               rxfree( stem );
               rxfree( varname );
               rxfree( value );
               free( w );
               return ( NULL );
            } /* endif */
            mods[i]->mod_bvalues[j]->bv_len = RXSTRLEN(value);
            memcpy( mods[i]->mod_bvalues[j]->bv_val, RXSTRPTR(value), RXSTRLEN(value) );
         } else {
            mods[i]->mod_values[j] = strtoutf8( RXSTRPTR( value ) );
            if ( !mods[i]->mod_values[j] ) {
               mods[i]->mod_values[j] = RXSTRPTR( value );
            } /* endif */
         } /* endif */
         rxfree( value );
      } /* endfor */
   } /* endfor */
   rxfree( stem );
   rxfree( varname );
   free( w );
   return ( mods );
}                                                

static char **
getldapattrs( 
PRXSTRING stemname ) {
   RXSTRING varname;
   RXSTRING stem;
   RXSTRING value;
   char *w = NULL;
   int i = 0;
   int n = 0;
   char **attrs = NULL;
   varname = rxalloc( RXSTRLEN(*stemname) + BUFSIZ );
   if ( RXNULLSTRING( varname ) ) {
      return ( NULL );
   } /* endif */
   stem = rxalloc( RXSTRLEN(*stemname) + BUFSIZ );
   if ( RXNULLSTRING( stem ) ) {
      rxfree( varname );
      return ( NULL );
   } /* endif */
   rxstrcpy( &stem, stemname );
   if ( stem.strptr[RXSTRLEN(stem)-1] != '.' ) {
      rxappendstr( &stem, "." );
   } /* endif */
   rxstrcpy( &varname, &stem );
   rxappendstr( &varname, "0" );
   if ( getvalue(&varname, &value) != Good ) {
      rxfree( stem );
      rxfree( varname );
      return ( NULL );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(value)) ) {
      rxfree( stem );
      rxfree( varname );
      rxfree( value );
      return ( NULL );
   } /* endif */
   n = rxtoi( value );
   rxfree( value );
   attrs = calloc( n + 1, sizeof(char*) );
   if ( !attrs ) {
      rxfree( stem );
      rxfree( varname );
      return ( NULL );
   } /* endif */
   w = malloc( BUFSIZ );
   if ( !w ) {
      rxfree( stem );
      rxfree( varname );
      return ( NULL );
   } /* endif */
   for ( i = 0; i < n; i += 1) {
      rxstrcpy( &varname, &stem );
      rxappendstr( &varname, _itoa(i + 1, w, 10) );
      if ( getvalue(&varname, &value) != Good ) {
         rxfree( stem );
         rxfree( varname );
         free( w );
         freestrs( attrs );
         return ( NULL );
      } /* endif */
      attrs[i] = malloc( RXSTRLEN(value) + 1 );
      if ( !attrs[i] ) {
         rxfree( stem );
         rxfree( varname );
         rxfree( value );
         free( w );
         freestrs( attrs );
         return ( NULL );
      } /* endif */
      memcpy( attrs[i], RXSTRPTR(value), RXSTRLEN(value) + 1 );
      rxfree( value );
   } /* endfor */
   free( w );
   rxfree( varname );
   rxfree( stem );
   return ( attrs );
}

static int 
setstrs( 
PRXSTRING stemname, 
char **strs,
int utf8 ) {
   RXSTRING varname;
   RXSTRING stem;
   RXSTRING value;
   char *w = NULL;
   int i = 0;
   char *v = NULL;
   if ( RXNULLSTRING(*stemname) || !RXSTRPTR(*stemname) ) {
      return ( 0 );
   } /* endif */
   w = malloc( BUFSIZ );
   if ( !w ) {
      return ( -1 );
   } /* endif */
   varname = rxalloc( RXSTRLEN(*stemname) + BUFSIZ );
   if ( RXNULLSTRING( varname ) ) {
      return ( -1 );
   } /* endif */
   stem = rxalloc( RXSTRLEN(*stemname) + BUFSIZ );
   if ( RXNULLSTRING( stem ) ) {
      rxfree( varname );
      return ( -1 );
   } /* endif */
   rxstrcpy( &stem, stemname );
   if ( stem.strptr[RXSTRLEN(stem)-1] != '.' ) {
      rxappendstr( &stem, "." );
   } /* endif */
   if ( strs ) {
      for ( i = 0; strs[i]; i += 1) {
         rxstrcpy( &varname, &stem );
         rxappendstr( &varname, _itoa(i + 1, w, 10) );
         if ( utf8 ) {
            v = utf8tostr( strs[i] );
         } /* endif */
         if ( v ) {
            MAKERXSTRING( value, v, strlen(v) );
         } else {
            MAKERXSTRING( value, strs[i], strlen(strs[i]) );
         } /* endif */
         if ( setvalue( &varname, &value ) != Good ) {
            freemem( v );
            rxfree( varname );
            rxfree( stem );
            free( w );
            return ( -1 );
         } /* endif */
         freemem( v );
      } /* endfor */
   } /* endif */
   rxstrcpy( &varname, &stem );
   rxappendstr( &varname, "0" );
   _ltoa( i, w, 10 );
   MAKERXSTRING( value, w, strlen(w) );
   if ( setvalue( &varname, &value ) != Good ) {
      rxfree( stem );
      rxfree( varname );
      free( w );
      return ( -1 );
   } /* endif */
   rxfree( varname );
   rxfree( stem );
   free( w );
   return ( i );
}

static LDAPControl **
getldapcontrols( 
PRXSTRING stemname ) {
   RXSTRING varname;
   RXSTRING stem;
   RXSTRING value;
   char *w = NULL;
   int i = 0;
   int n = 0;
   LDAPControl **controls = NULL;
   varname = rxalloc( RXSTRLEN(*stemname) + BUFSIZ );
   if ( RXNULLSTRING( varname ) ) {
      return ( NULL );
   } /* endif */
   stem = rxalloc( RXSTRLEN(*stemname) + BUFSIZ );
   if ( RXNULLSTRING( stem ) ) {
      rxfree( varname );
      return ( NULL );
   } /* endif */
   rxstrcpy( &stem, stemname );
   if ( stem.strptr[RXSTRLEN(stem)-1] != '.' ) {
      rxappendstr( &stem, "." );
   } /* endif */
   rxstrcpy( &varname, &stem );
   rxappendstr( &varname, "0" );
   if ( getvalue(&varname, &value) != Good ) {
      rxfree( stem );
      rxfree( varname );
      return ( NULL );
   } /* endif */
   if ( !isnumstr(RXSTRPTR(value)) ) {
      rxfree( stem );
      rxfree( varname );
      rxfree( value );
      return ( NULL );
   } /* endif */
   n = rxtoi( value );
   rxfree( value );
   controls = calloc( n + 1, sizeof(LDAPControl*) );
   if ( !controls ) {
      rxfree( stem );
      rxfree( varname );
      return ( NULL );
   } /* endif */
   w = malloc( BUFSIZ );
   if ( !w ) {
      rxfree( stem );
      rxfree( varname );
      return ( NULL );
   } /* endif */
   for ( i = 0; i < n; i += 1) {
      controls[i] = calloc( 1, sizeof(LDAPControl) );
      if ( !controls[i] ) {
         rxfree( stem );
         rxfree( varname );
         free( w );
         ldap_controls_free( controls );
         return ( NULL );
      } /* endif */
      rxstrcpy( &varname, &stem );
      rxappendstr( &varname, _itoa(i + 1, w, 10) );
      rxappendstr( &varname, ".!LDCTL_OID" );
      if ( getvalue(&varname, &value) != Good ) {
         rxfree( stem );
         rxfree( varname );
         free( w );
         ldap_controls_free( controls );
         return ( NULL );
      } /* endif */
      memcpy( controls[i]->ldctl_oid, RXSTRPTR(value), RXSTRLEN(value) + 1 );
      rxstrcpy( &varname, &stem );
      rxappendstr( &varname, _itoa(i + 1, w, 10) );
      rxappendstr( &varname, ".!LDCTL_VALUE" );
      if ( getvalue(&varname, &value) != Good ) {
         rxfree( stem );
         rxfree( varname );
         free( w );
         ldap_controls_free( controls );
         return ( NULL );
      } /* endif */
      controls[i]->ldctl_value.bv_val = malloc( RXSTRLEN(value) );
      if ( !controls[i]->ldctl_value.bv_val ) {
         rxfree( stem );
         rxfree( varname );
         rxfree( value );
         free( w );
         ldap_controls_free( controls );
         return ( NULL );
      } /* endif */
      memcpy( controls[i]->ldctl_value.bv_val, RXSTRPTR(value), RXSTRLEN(value) );
      controls[i]->ldctl_value.bv_len = RXSTRLEN(value);
      rxfree( value );
      rxstrcpy( &varname, &stem );
      rxappendstr( &varname, _itoa(i + 1, w, 10) );
      rxappendstr( &varname, ".!LDCTL_ISCRITICAL" );
      if ( getvalue(&varname, &value) != Good ) {
         rxfree( stem );
         rxfree( varname );
         free( w );
         ldap_controls_free( controls );
         return ( NULL );
      } /* endif */
      if ( !isnumstr(RXSTRPTR(value)) ) {
         rxfree( stem );
         rxfree( varname );
         rxfree( value );
         free( w );
         ldap_controls_free( controls );
         return ( NULL );
      } /* endif */
      controls[i]->ldctl_iscritical = rxtoi( value );
      rxfree( value );
   } /* endfor */
   free( w );
   rxfree( varname );
   rxfree( stem );
   return ( controls );
}

static int 
setldapctrls( 
PRXSTRING stemname, 
LDAPControl **ctrls ) {
   RXSTRING varname;
   RXSTRING stem;
   RXSTRING value;
   char *w = NULL;
   int i = 0;
   if ( RXNULLSTRING(*stemname) || !RXSTRPTR(*stemname) ) {
      return ( 0 );
   } /* endif */
   w = malloc( BUFSIZ );
   if ( !w ) {
      return ( Bad );
   } /* endif */
   varname = rxalloc( RXSTRLEN(*stemname) + BUFSIZ );
   if ( RXNULLSTRING( varname ) ) {
      return ( CallingInvalid );
   } /* endif */
   stem = rxalloc( RXSTRLEN(*stemname) + BUFSIZ );
   if ( RXNULLSTRING( stem ) ) {
      rxfree( varname );
      return ( CallingInvalid );
   } /* endif */
   rxstrcpy( &stem, stemname );
   if ( stem.strptr[RXSTRLEN(stem)-1] != '.' ) {
      rxappendstr( &stem, "." );
   } /* endif */
   if ( ctrls ) {
      for ( i = 0; ctrls[i]; i += 1) {
         rxstrcpy( &varname, &stem );
         rxappendstr( &varname, _itoa(i + 1, w, 10) );
         rxappendstr( &varname, ".!LDCTL_OID" );
         if ( ctrls[i]->ldctl_oid ) {
            MAKERXSTRING( value, ctrls[i]->ldctl_oid, strlen(ctrls[i]->ldctl_oid) );
         } else {
            rxset_zerolen( &value );
         } /* endif */
         if ( setvalue( &varname, &value ) != Good ) {
            rxfree( varname );
            rxfree( stem );
            free( w );
            return ( CallingInvalid );
         } /* endif */
         rxstrcpy( &varname, &stem );
         rxappendstr( &varname, _itoa(i + 1, w, 10) );
         rxappendstr( &varname, ".!LDCTL_VALUE" );
         if ( ctrls[i]->ldctl_value.bv_len && ctrls[i]->ldctl_value.bv_val ) {
            MAKERXSTRING( value, ctrls[i]->ldctl_value.bv_val, ctrls[i]->ldctl_value.bv_len );
         } else {
            rxset_zerolen( &value );
         } /* endif */
         if ( setvalue( &varname, &value ) != Good ) {
            rxfree( varname );
            rxfree( stem );
            free( w );
            return ( CallingInvalid );
         } /* endif */
         rxstrcpy( &varname, &stem );
         rxappendstr( &varname, _itoa(i + 1, w, 10) );
         rxappendstr( &varname, ".!LDCTL_ISCRITICAL" );
         _itoa( ctrls[i]->ldctl_iscritical, w, 10 );
         MAKERXSTRING( value, w, strlen(w) );
         if ( setvalue( &varname, &value ) != Good ) {
            rxfree( varname );
            rxfree( stem );
            free( w );
            return ( CallingInvalid );
         } /* endif */
      } /* endfor */
   } /* endif */
   rxstrcpy( &varname, &stem );
   rxappendstr( &varname, "0" );
   _ltoa( i, w, 10 );
   MAKERXSTRING( value, w, strlen(w) );
   if ( setvalue( &varname, &value ) != Good ) {
      rxfree( stem );
      rxfree( varname );
      free( w );
      return ( CallingInvalid );
   } /* endif */
   rxfree( varname );
   rxfree( stem );
   free( w );
   return ( Good );
}

static unsigned char *
strtoutf8( 
const unsigned char *str ) {
   UconvObject uconvobj = NULL;
   int code = 0;
   UniChar *ucsstr = NULL;
   UniChar *ucsbuf = NULL;
   size_t strsize = strlen( str );
   size_t ucsbufsize = strsize + BUFSIZ;
   size_t ucsbufleft = ucsbufsize;
   size_t utf8bufsize = 3 * ucsbufsize;
   size_t nonident = 0;
   int i = 0;
   int j = 0;
   unsigned char *utf8str = NULL;
   if ( !str ) {
      return ( NULL );
   } /* endif */
   code = UniCreateUconvObject( (UniChar *)L"", &uconvobj );
   if ( code != ULS_SUCCESS ) {
      return ( NULL );
   } /* endif */
   ucsstr = ucsbuf = calloc( ucsbufsize, sizeof( ucsstr[0] ) );
   if ( !ucsstr ) {
      return ( NULL );
   } /* endif */
   code = UniUconvToUcs( uconvobj, (void **)&str, &strsize, &ucsbuf, &ucsbufleft, &nonident );
   if ( code != ULS_SUCCESS ) {
      free( ucsstr );
      return ( NULL );
   } /* endif */
   code = UniFreeUconvObject( uconvobj );
   if ( code != ULS_SUCCESS ) {
      free( ucsstr );
      return ( NULL );
   } /* endif */
   utf8str = calloc( utf8bufsize, sizeof( utf8str[0] ) );
   if ( !utf8str ) {
      free( ucsstr );
      return ( NULL );
   } /* endif */
   for ( i = 0, j = 0; ucsstr[i]; i += 1) {
      if ( j >= utf8bufsize ) {
         free( ucsstr );
         free( utf8str );
         return ( NULL );
      } /* endif */
      if ( ucsstr[i] < 0x80U ) {
         utf8str[j++] = ucsstr[i];
      } else if (ucsstr[i] < 0x7ffU ) {
         utf8str[j++] = ((ucsstr[i] & ~0x3fU) >> 6) | 0xc0U;
         utf8str[j++] = (ucsstr[i] & 0x3fU) | 0x80U;
      } else {
         utf8str[j++] = ((ucsstr[i] & 0xf000U) >> 12) | 0xe0U;
         utf8str[j++] = ((ucsstr[i] & 0x0fc0U) >> 6) | 0x80U;
         utf8str[j++] = ((ucsstr[i] & 0x003fU)) | 0x80U;
      } /* endif */
   } /* endfor */
   free( ucsstr );
   return ( utf8str );
}

static unsigned char *
utf8tostr( 
const unsigned char *utf8str ) {
   UconvObject  uconvobj = NULL;
   int code = 0;
   UniChar *ucsstr = NULL;
   UniChar *ucsbuf = NULL;
   size_t ucsstrsize = 0;
   int i = 0;
   int j = 0;
   unsigned char *str = NULL;
   unsigned char *buf = NULL;
   size_t bufsize = 0;
   size_t bufleft = 0;
   size_t nonident = 0;
   size_t ucsstrlen = 0;
   if ( !utf8str ) {
      return ( NULL );
   } /* endif */
   ucsstrsize = strlen( utf8str ) + BUFSIZ;
   ucsstr = ucsbuf = calloc( ucsstrsize, sizeof( ucsstr[0] ) );
   if ( !ucsstr ) {
      return ( NULL );
   } /* endif */
   bufsize = bufleft = 2 * ucsstrsize + BUFSIZ;
   for (i = 0, j = 0; utf8str[j]; ) {
      if ( i >= ucsstrsize ) {
         free( ucsstr );
         return ( NULL );
      } /* endif */
      if ( !(utf8str[j] & 0x80U) ) {
         ucsstr[i++] = utf8str[j++];
      } else if ( (utf8str[j] & 0xe0U) == 0xc0U ) {
         if ( utf8str[j+1] & 0xc0U != 0x80U ) {
            free( ucsstr );
            return ( NULL );
         } /* endif */
         ucsstr[i++] = ((utf8str[j] & 0x1cU) << 6) | ((utf8str[j] & 0x3U) << 6) | (utf8str[j+1] & 0x3fU);
         j += 2;
      } else if ( (utf8str[j] & 0xf0U) == 0xe0U ) {
         if ( (utf8str[j+1] & 0xc0U) != 0x80U || (utf8str[j+2] & 0xc0U) != 0x80U ) {
            free( ucsstr );
            return ( NULL );
         } /* endif */
         ucsstr[i++] = ((utf8str[j] & 0x0fU) << 12) | ((utf8str[j+1] & 0x3cU) << 6) | ((utf8str[j+1] & 0x3U) << 6) | (utf8str[j+2] & 0x3fU);
         j += 3;
      } else {
         free( ucsstr );
         return ( NULL );
      } /* endif */
   } /* endfor */
   code = UniCreateUconvObject( (UniChar *)L"", &uconvobj );
   if ( code != ULS_SUCCESS ) {
      free( ucsstr );
      return ( NULL );
   } /* endif */
   str = buf = calloc( bufsize, sizeof( str[0] ) );
   if ( !str ) {
      free( ucsstr );
      return ( NULL );
   } /* endif */
   ucsstrlen = UniStrlen( ucsstr );
   code = UniUconvFromUcs( uconvobj, &ucsbuf, &ucsstrlen, (void **)&buf, &bufleft, &nonident );
   if ( code != ULS_SUCCESS ) {
      free( ucsstr );
      free( str );
      return ( NULL );
   } /* endif */
   code = UniFreeUconvObject( uconvobj );
   if ( code != ULS_SUCCESS ) {
      free( ucsstr );
      free( str );
      return ( NULL );
   } /* endif */
   free( ucsstr );
   return ( str );
}

static void
freemem(
void *p ) {
   if ( p ) {
      free( p );
   } /* endif */
}   
