/* REXX */
/* 4print2.cmd -- OS/2 shell for DOS 4Print.exe */

versionstring =  '4print2 version 0.3, for OS/2   26-oct-2002'

/* author: Pieter Bras
           Cambridge, MA (USA)
           pbras@pobox.com
*/
/*                COPYRIGHT NOTICE AND USER LICENSE

   Copyright (C) 2002, Pieter Bras

   The 4print2 program is copyrighted freeware.  A user may modify or
   customize this program for his own use, to suit his own preferences.

   This program may be redistributed subject to the following conditions:
   - The program may be redistributed only in its original, unmodified form;
     Specifically, user customizations may not be redistributed;
     This Copyright Notice and User License must be retained unmodified.
   - No charge (monetary or otherwise) may be made for any redistribution.

   Exceptions to the above require prior written approval from the author.
*/
/* PLEASE NOTE that the 4Print for DOS program itself is copyrighted
   commercial software. The final version of 4Print for DOS is 4.16.
   Registered copies of 4Print may be ordered from its copyright holder:

      Korenthal Associates, Inc.
      511 Ave. of the Americas #400
      New York, NY  10011
      USA

      phone: 212-242-1790
      fax:   212-242-2599

   (No, they don't have a web site.)
*/

/* This program allows OS/2 use of the DOS utility 4Print.exe with
   pathnames/filenames which are not restricted to the FAT 8.3 rules.
   It also circumvents a Y2K file-date bug in 4Print.

   Works in conjuction with 4Print.exe, which must be on the DOS PATH.
   Tested with 4Print Version 4.15 (registered and shareware versions).

   Usage:  4print2.cmd [options] [filespecs]

      options:    valid 4Print.exe options (see 4Print documentation).
      filespecs:  valid HPFS filespecs including ? and * wildcard chars.

   Options and filespecs can appear in any order.
   Options are global and apply to all files (see 4Print documentation).

   Run-time configuration of 4print2.cmd can be done via OS/2 environment
   entries. It is not necessary to set these as default values are provided.

   4PRINT2_DEBUG   default: 0          if 1, enable diagnostic output
   4PRINT2_REG     default: 0          if 1, assume 4Print.exe is registered
   4PRINT2_OPT     default: -oLPT1     default options passed to 4Print.exe
   4PRINT2_TMP     default: C:\TMP     8.3-compatible working directory
    
   Restrictions (with respect to 4Print.exe):
      - A filespec may not start with the '-' character, which is reserved
        for introducing options. Use "filespec" or .\filespec if necessary.
      - This program uses the -h option to define the page header line.
        The -h option may still be invoked from the command line, and will
        take priority. However the $fn, $fd, $ft macros will then refer to
        a meaningless temporary file.

   Other differences:
      - Processing time may differ as a seperate instance of 4Print.exe
        is invoked for each file, and these can run in parallel.
      - If multiple files are selected, they may be sent to the printer
        in a different order.
      - Multiple instances of this program (4print2.cmd) are serialized.

   Return codes:
      0     successful completion
      1     fatal error
*/
/**********************************************************************/

hmutex = ''				    /* initialized in case of error */
signal on halt
signal on novalue
signal on syntax
say versionstring

/* initialize OS/2 RexxUtil functions */
if LoadLib( 'RexxUtil', 'SysLoadFuncs') then
   call Fatal 'RexxUtil.dll not found'

/* get run-time configuration options from OS/2 environment */
env = 'OS2ENVIRONMENT'
debug = (value( '4PRINT2_DEBUG',, env) = '1')
registered = (value( '4PRINT2_REG',, env) = '1')
dfltopts = strip( value( '4PRINT2_OPT',, env))
if dfltopts = '' then
   dfltopts = '-oLPT1'					  /* output to LPT1 */
tempdir = strip( value( '4PRINT2_TMP',, env))
if tempdir = '' then			 /* if not defined then use default */
   tempdir = 'C:\TMP'			/* 8.3-compatible working directory */
p = pos( ':', tempdir)
if (p \= 0) & (p \= 2) then			/* ':' may be 2nd char only */
   call Fatal 'Invalid temp. directory:' quote( tempdir)
tempdir = strip( tempdir, 'T', '\')		  /* strip off trailing '\' */
if (tempdir = '') | (right( tempdir, 1) = ':') then
   tempdir = tempdir'\'			/* restore trailing '\' if root dir */
if debug then do
   say 'registered:     ' yesno( registered)
   say 'default options:' quote( dfltopts)
   say 'temp. directory:' quote( tempdir)
   end

/* create working directory */
rc = SysMkDir( tempdir)			/* rc = 5 if tempdir already exists */
if (rc \= 0) & (rc \= 5) then		       /* 0 = OK, 5 = access denied */
   call Fatal 'Unable to create temp. directory' quote( tempdir) ' SysMkDir: rc =' rc
template = tempdir
if right( template, 1) \= '\' then
   template = template'\'
template = template'4P_?????.BAT'		 /* temporary DOS .bat file */
if debug then
   say 'template:       ' quote( template)

/* get 4Print user options and filespecs from command line */
useropts = ''
fspec.0 = 0
parse arg commandline
call checkargs commandline
useropts = strip( useropts)
options = dfltopts useropts
unattended = (pos( ' -U ', translate( options' ')) > 0)
if debug then do
   say 'user options:   ' quote( useropts)
   say 'unattended:     ' yesno( unattended)
   end

/* find all files matching the filespec(s) */
fname.0 = 0
do i = 1 to fspec.0
   rc = SysFileTree( fspec.i, 'files', 'F')
   if rc \= 0 then
      call Fatal 'SysFileTree: rc =' rc
   do j = 1 to files.0
      k = fname.0 + 1
      parse var files.j fdate.k ftime.k fsize.k fattr.k fname.k
      fname.k = strip( fname.k, 'L')   /* strip leading blanks */
      parse var fdate.k mm '/' dd '/' yy
      mmm = substr( 'janfebmaraprmayjunjulaugsepoctnovdec', 3*mm - 2, 3)
      fdate.k = dd'-'mmm'-'yy
      fname.0 = k
      end  /* do j = ...*/
   if debug then
      say 'filespec:       ' quote( fspec.i) ' matching files:' files.0
   end   /* do i = ... */

/* serialize 4print2 program execution (indefinite wait) */
if rxFuncQuery( 'SysCreateMutexSem') = 0 then do	    /* if available */
   hmutex = SysCreateMutexSem( '\SEM32\4PRINT2\4PRINT2.MUTEX')
   if hmutex = '' then
      call Fatal 'Unable to create mutex semaphore'
   rc = SysOpenMutexSem( hmutex)
   if rc \= 0 then
      call Fatal 'SysOpenMutexSem: rc =' rc
   rc = SysRequestMutexSem( hmutex)
   if rc \= 0 then
      call Fatal 'SysRequestMutexSem: rc =' rc
   end
else do
   call beep 1500, 250
   say 'Semaphore functions unavailable -- please update your version of REXX'
   end

/* if no files found, display 4Print help */
if fname.0 = 0 then do
   say 'No files selected'				     /* print usage */
   tempfile = newtemp()
   /* write DOS batch file, then execute it */
   rc = stream( batfile, 'C', 'OPEN')
   rc = lineout( batfile, 'tkt.com')
   rc = lineout( batfile, '4Print' useropts)
   rc = lineout( batfile, 'del' batfile)
   rc = lineout( batfile, 'exit')
   rc = stream( batfile, 'C', 'CLOSE')
   '@start /dos /f /c' batfile
   signal done
   end

/* print each file matching the filespec(s) */
do i = 1 to fname.0
   /* copy source to a temp file which follows FAT 8.3 rules */
   tempfile = newtemp()			       /* generate unique file name */
   '@copy' fname.i tempfile '>nul 2>nul'

   /* make HPFS filenames compatible with 4Print page header (-h option) */
   name = substr( fname.i, lastpos( '\', fname.i) + 1)
   if pos( ' ', name) > 0 then		    /* surround with double quotes  */
      name = quote( name)
   say 'Printing:' name
   name = tr2( name, '$$', '$')			 /* quote '$' and '_' chars */
   name = tr2( name, '$_', '_')
   name = translate( name, '_', ' ')	     /* convert spaces to '_' chars */
   header = name'__'fdate.i'__'ftime.i'_//Page_$pn[_of_$pp]'

   nag = \registered & (i = fname.0)	 /* display nag screen once, at end */
   rc = stream( batfile, 'C', 'OPEN')
   rc = lineout( batfile, 'set 4PRINT=-h_'header)
   rc = lineout( batfile, 'tkt.com')
   if \nag then
      rc = lineout( batfile, 'stuffkey P"press any key..." "NN"')
   rc = lineout( batfile, '4Print' options tempfile)
   rc = lineout( batfile, 'del' tempfile)
   rc = lineout( batfile, 'del' batfile)
   rc = lineout( batfile, 'exit')
   rc = stream( batfile, 'C', 'CLOSE')
   if unattended & \nag then
      fbmode = '/b /min'	/* unattended: run in background, minimized */
   else
      fbmode = '/f'			    /* otherwise, run in foreground */
   '@start /dos /c' fbmode batfile

   /* if 4Print job is running in the foreground, wait for it to finish */
   if fbmode = '/f' then do forever
      call SysSleep 1					   /* wait 1 second */
      rc = SysFileTree( batfile, 'files', 'F')
      if rc \= 0 then
         call Fatal 'SysFileTree: rc =' rc
      if files.0 = 0 then	   /* wait until batfile has deleted itself */
         leave
      end   /* do forever */
   end   /* do i = ... */

done:
   if hmutex \= '' then
      rc = SysReleaseMutexSem( hmutex)
   exit 0

/* return a string with quotes around it */
quote:
   return '"' || arg( 1) || '"'

/* return 'YES' or 'NO' for Boolean value */
yesno:
   if arg( 1) = '1' then
      return 'YES'
   else
      return 'NO'

/* generate a new temporary filename */
newtemp:
   batfile = SysTempFileName( template)
   return left( batfile, length( batfile) - 3) || 'TMP'

/* substitute new substring for old, multiple instances */
tr2: procedure
   parse arg str, new, old
   d = length( new) - length( old)
   p = 1
   do forever
      p = pos( old, str, p)
      if p = 0 then
         leave
      str = left( str, p - 1) || new || substr( str, p + length( old))
      p = p + d + 1
      end   /* do forever */
   return str

/* Conditionally load a DLL and register its functions.
   returns: 1 if error, 0 otherwise
*/
LoadLib: procedure
   parse arg lib, proc
   if RxFuncQuery( proc) <> 0 then do
      call RxFuncAdd proc, lib, proc
      str = 'call' proc
      signal on syntax name LoadLibFail
      interpret str
      end
   return 0

LoadLibFail:
   call RxFuncDrop proc
   return 1

/**********************************************************************/
/* parse command line into options and filespecs */

checkargs: procedure expose useropts fspec.
   SP = ' '
   SQ = "'"
   DQ = '"'
   parse arg arglist
   do while arglist \= ''
      arglist = strip( arglist)
      /* if it starts with '-' or '/' then assume it's a 4Print option */
      if abbrev( arglist, '-') | abbrev( arglist, '/') then do
         arg2 = translate( substr( arglist, 2))

         /* special case: -H"..." and -H'...' options */
         if abbrev( arg2, 'H'DQ) | abbrev( arg2, 'H'SQ) then do
            q = substr( arglist, 3, 1)		  /* single or double quote */
            p = pos( q, arglist, 4)		 /* look for matching quote */
            if p < 4 then
               p = pos( SP, arglist, 4)
            if p < 4 then
               p = length( arglist)
            argx = strip( left( arglist, p))
            arglist = substr( arglist, p + 1)
            end   /* -H"..." and -H'...' options */

         /* special case: -FMT... option */
         else if abbrev( arg2, 'FMT') then do
            p = 1
            do forever
               /* locate the first space or quote char */
               p0 = pos( SP, arglist, p)
               p1 = pos( SQ, arglist, p)
               p2 = pos( DQ, arglist, p)
               if fmt2( SQ, p1, p2) | fmt2( DQ, p2, p1) then
                  iterate				  /* found SQ or DQ */
               else do					   /* found neither */
                  p = p0
                  if p = 0 then
                     p = length( arglist)
                  argx = left( arglist, p)
                  arglist = substr( arglist, p + 1)
                  leave
                  end
               end   /* do forever */
            end   /* -FMT option */

         /* general case: all other options */
         else
            parse var arglist argx arglist

         /* all options */
         useropts = useropts argx
         end

      /* if it isn't a 4Print option then assume it's an HPFS filespec */
      else do
         if abbrev( arglist, DQ) then do
            p = pos( DQ, arglist, 2)
            if p < 2 then	  /* if no matching quote go to end of line */
               p = length( arglist)
            argx = strip( left( arglist, p), DQ)
            arglist = substr( arglist, p + 1)
            end
         else
            parse var arglist argx arglist
         if argx \= '' then do
            i = fspec.0 + 1
            fspec.i = argx
            fspec.0 = i
            end
         end
      end   /* do while arglist \= '' */
   return

/* look for matching quotes in -FMT option */
fmt2:
   parse arg qchr, x, y
   if x = 0 then				      /* if quote not found */
      return 0
   if (y > 0) & (y < x) then	       /* if other style quote comes before */
      return 0
   if (p0 > 0) & (p0 < x) then			   /* if space comes before */
      return 0
   p = pos( qchr, arglist, x + 1)		     /* find matching quote */
   if p = 0 then
      p = length( arglist)		      /* -> end of line if no match */
   else
      p = p + 1				     /* -> char after closing quote */
   return 1

/**********************************************************************/
/* and just in case... */

Fatal:
   if hmutex \= '' then
      rc = SysReleaseMutexSem( hmutex)
   say arg( 1)
   exit 1

halt:
   call Fatal 'Program halted by operator'

novalue:
   say 'Uninitialized variable on line:' sigl
   say sourceline( sigl)
   call Fatal

syntax:
   say 'Syntax error on line:' sigl  ' error code:' rc
   say sourceline( sigl)
   call Fatal

/* <eof> */
