/*
 * Decompiled with CFR 0.152.
 */
package gov.nasa.giss.time;

import gov.nasa.giss.time.EarthTime;
import gov.nasa.giss.time.JulianDay;
import gov.nasa.giss.time.MissionTime;
import gov.nasa.giss.time.Point3DCartesian;
import gov.nasa.giss.time.TimeConstants;
import gov.nasa.giss.time.TimeFormat;
import java.awt.geom.Point2D;

public abstract class AbstractPlanetTime
extends JulianDay
implements TimeConstants {
    public static final int DISTANCE_LIGHTMIN = 1;
    public static final int DISTANCE_LIGHTSEC = 2;
    public static final int DISTANCE_AU = 3;
    public static final int DISTANCE_KM = 4;
    public static final int MEAN_SOLAR = 1;
    public static final int TRUE_SOLAR = 2;
    public static final int MEAN_ZONAL = 3;
    public static final int PLANETARY = 4;
    public static final int PLANETOGRAPHIC = 1;
    public static final int PLANETOCENTRIC = 2;
    protected double pgFactor = 0.0;
    protected double semimajorAxisAU;
    protected double meanAnomaly;
    protected double meanAnomalyRad;
    protected double trueAnomaly;
    protected double obliquity;
    protected double eccentricity;
    protected double veeM;
    protected double ellS;
    protected double ellSRad;
    protected double cosEllS;
    protected double sinEllS;
    protected double ellSP;
    protected double alphaFMS;
    protected double subsolarLonPC;
    protected double subsolarLonPG;
    protected double declinSPC;
    protected double declinSPG;
    protected double rHeliocentricAU;
    protected double lonEcliptic;
    protected double latEcliptic;
    protected double eotDegrees;
    protected double eotHours;
    protected double mst0;
    protected double tst0;
    protected String pmTimeSuffix = "XT";
    protected double lambdaRot;
    protected double betaRot;
    protected EarthTime earthTime;
    protected double distEM;
    protected double subEarthLonPG;
    protected double subEarthLonPC;
    protected double declinEPG;
    protected double declinEPC;
    protected boolean gotOrbitParams = false;
    protected boolean gotEarthParams = false;

    public AbstractPlanetTime() {
        this(System.currentTimeMillis());
    }

    public AbstractPlanetTime(long t) {
        super(t);
        this.calculate();
    }

    public void setTime(long t) {
        super.setTime(t);
        this.calculate();
    }

    protected abstract void calculate();

    protected abstract void calculateOrbitParams();

    protected void calculateEarthParams() {
        double lambda;
        if (!this.gotOrbitParams) {
            this.calculateOrbitParams();
        }
        double latEclipticRad = this.latEcliptic * (Math.PI / 180);
        double cosLatEcliptic = Math.cos(latEclipticRad);
        double sinLatEcliptic = Math.sin(latEclipticRad);
        this.earthTime = new EarthTime(this.getMillis());
        double lambdaE = this.earthTime.getHeliocentricLongitude();
        double rE = this.earthTime.getDistanceToSun(3);
        for (lambda = this.lonEcliptic - lambdaE; lambda > 180.0; lambda -= 360.0) {
        }
        while (lambda < -180.0) {
            lambda += 360.0;
        }
        double cosLambda = Math.cos(lambda * (Math.PI / 180));
        double rMu = this.rHeliocentricAU * cosLatEcliptic;
        double zEcliptic = this.rHeliocentricAU * sinLatEcliptic;
        double rE2 = rE * rE;
        double rM2 = this.rHeliocentricAU * this.rHeliocentricAU;
        double rMu2 = rMu * rMu;
        double distEM2 = rE2 + rM2 - 2.0 * rE * this.rHeliocentricAU * cosLambda * cosLatEcliptic;
        this.distEM = Math.sqrt(distEM2);
        double distEMu2 = rE2 + rM2 * cosLatEcliptic * cosLatEcliptic - 2.0 * rE * this.rHeliocentricAU * cosLambda * cosLatEcliptic;
        double distEMu = Math.sqrt(distEMu2);
        double cosPhase = (rMu2 + distEMu2 - rE2) / (2.0 * rMu * distEMu);
        double phaseRad = Math.acos(cosPhase);
        if (lambda < 0.0) {
            phaseRad = -phaseRad;
        }
        double sinPhase = Math.sin(phaseRad);
        double cosChi = zEcliptic / this.distEM;
        double chiNRad = Math.PI - Math.acos(cosChi);
        double cosChiN = Math.cos(chiNRad);
        double sinChiN = Math.sin(chiNRad);
        Point3DCartesian pointE = new Point3DCartesian(sinChiN * cosPhase, sinChiN * sinPhase, Math.cos(chiNRad));
        Point3DCartesian pointS = new Point3DCartesian(cosLatEcliptic, 0.0, -sinLatEcliptic);
        pointE.rotateAxis(3, this.lambdaRot - this.lonEcliptic, 1);
        pointS.rotateAxis(3, this.lambdaRot - this.lonEcliptic, 1);
        pointE.rotateAxis(2, -(90.0 - this.betaRot), 1);
        pointS.rotateAxis(2, -(90.0 - this.betaRot), 1);
        this.declinEPC = 90.0 - pointE.getPhi();
        this.declinEPG = this.transformLatitudePC2PG(this.declinEPC);
        this.subEarthLonPG = pointS.getTheta() - pointE.getTheta() + this.subsolarLonPG;
        this.subEarthLonPC = -this.subEarthLonPG;
        while (this.subEarthLonPG > 360.0) {
            this.subEarthLonPG -= 360.0;
        }
        while (this.subEarthLonPG < 0.0) {
            this.subEarthLonPG += 360.0;
        }
        this.gotEarthParams = true;
    }

    public double transformLatitudePC2PG(double pc) {
        if (this.pgFactor == 0.0) {
            return pc;
        }
        if (pc == 0.0 || pc == 90.0 || pc == -90.0) {
            return pc;
        }
        if (pc > 90.0 || pc < -90.0) {
            throw new IllegalArgumentException("Latitude out of range.");
        }
        return 57.29577951308232 * Math.atan(Math.tan(Math.PI / 180 * pc) / ((1.0 - this.pgFactor) * (1.0 - this.pgFactor)));
    }

    public double transformLatitudePG2PC(double pg) {
        if (this.pgFactor == 0.0) {
            return pg;
        }
        if (pg == 0.0 || pg == 90.0 || pg == -90.0) {
            return pg;
        }
        if (pg > 90.0 || pg < -90.0) {
            throw new IllegalArgumentException("Latitude out of range.");
        }
        return 57.29577951308232 * Math.atan(Math.tan(Math.PI / 180 * pg) * (1.0 - this.pgFactor) * (1.0 - this.pgFactor));
    }

    public abstract double getVernalEquinox();

    public double getEccentricity() {
        return this.eccentricity;
    }

    public double getMeanAnomaly() {
        return this.meanAnomaly;
    }

    public double getTrueAnomaly() {
        return this.trueAnomaly;
    }

    public double getHeliocentricLongitude() {
        if (!this.gotOrbitParams) {
            this.calculateOrbitParams();
        }
        return this.lonEcliptic;
    }

    public abstract double getHeliocentricLongitudeAtPerihelion(boolean var1);

    public double getHeliocentricLatitude() {
        if (!this.gotOrbitParams) {
            this.calculateOrbitParams();
        }
        return this.latEcliptic;
    }

    public double getDistanceToSun(int units) {
        if (!this.gotOrbitParams) {
            this.calculateOrbitParams();
        }
        if (units == 3) {
            return this.rHeliocentricAU;
        }
        if (units == 4) {
            return this.rHeliocentricAU * 1.49597870691E8;
        }
        if (units == 2) {
            return this.rHeliocentricAU * 499.0047838061357;
        }
        if (units == 1) {
            return this.rHeliocentricAU * 499.0047838061357 / 60.0;
        }
        throw new IllegalArgumentException("Unknown distance units.");
    }

    public double getDistanceToSun(int units, double d) {
        double e = this.getEccentricity();
        double result = this.semimajorAxisAU * (1.0 - e * e) / (1.0 + e * Math.cos(d * (Math.PI / 180)));
        if (units == 3) {
            return result;
        }
        if (units == 4) {
            return result * 1.49597870691E8;
        }
        if (units == 2) {
            return result * 499.0047838061357;
        }
        if (units == 1) {
            return result * 499.0047838061357 / 60.0;
        }
        throw new IllegalArgumentException("Unknown distance units.");
    }

    public double getEarthDistance(int units) {
        if (!this.gotEarthParams) {
            this.calculateEarthParams();
        }
        if (units == 3) {
            return this.distEM;
        }
        if (units == 4) {
            return this.distEM * 1.49597870691E8;
        }
        if (units == 2) {
            return this.distEM * 499.0047838061357;
        }
        if (units == 1) {
            return this.distEM * 499.0047838061357 / 60.0;
        }
        throw new IllegalArgumentException("Unknown distance units.");
    }

    public double getAlphaFMS() {
        return this.alphaFMS;
    }

    public double getSolarLongitude() {
        while (this.ellS < 0.0) {
            this.ellS += 360.0;
        }
        while (this.ellS > 360.0) {
            this.ellS -= 360.0;
        }
        return this.ellS;
    }

    public double getSolarLongitudeAtPerihelion() {
        while (this.ellSP < 0.0) {
            this.ellSP += 360.0;
        }
        while (this.ellSP > 360.0) {
            this.ellSP -= 360.0;
        }
        return this.ellSP;
    }

    public Point2D.Double getSubsolarPoint(int how) {
        return new Point2D.Double(this.getSubsolarLongitude(how), this.getSolarDeclination(how));
    }

    public double getSolarDeclination(int how) {
        if (how == 1) {
            return this.declinSPG;
        }
        if (how == 2) {
            return this.declinSPC;
        }
        throw new IllegalArgumentException("Coord. system must be either p'centric or p'graphic.");
    }

    public double getSubsolarLongitude(int how) {
        if (how == 2) {
            return this.subsolarLonPC;
        }
        if (how == 1) {
            return this.subsolarLonPG;
        }
        throw new IllegalArgumentException("Coord. system must be either p'centric or p'graphic.");
    }

    public Point2D.Double getSolarAngleAt(double lon, double lat) {
        return new Point2D.Double(this.getSolarAzimuthAngle(lon, lat), this.getSolarZenithAngle(lon, lat));
    }

    public double getSolarZenithAngle(double lon, double lat) {
        double cosDeclination = Math.cos(AbstractPlanetTime.toRadians(this.declinSPG));
        double sinDeclination = Math.sin(AbstractPlanetTime.toRadians(this.declinSPG));
        double hourAngle = lon - this.subsolarLonPG;
        double cosHourAngle = Math.cos(AbstractPlanetTime.toRadians(hourAngle));
        double result = AbstractPlanetTime.toDegrees(Math.acos(sinDeclination * Math.sin(AbstractPlanetTime.toRadians(lat)) + cosDeclination * Math.cos(AbstractPlanetTime.toRadians(lat)) * cosHourAngle));
        return result;
    }

    public double getSolarAzimuthAngle(double lon, double lat) {
        double azimuthAngle;
        double tanDec = Math.tan(AbstractPlanetTime.toRadians(this.declinSPG));
        double hourAngle = lon - this.subsolarLonPG;
        if (hourAngle > 180.0) {
            hourAngle -= 360.0;
        }
        if (hourAngle < -180.0) {
            hourAngle += 360.0;
        }
        double hourAngleRad = AbstractPlanetTime.toRadians(hourAngle);
        double sinH = Math.sin(hourAngleRad);
        double cosH = Math.cos(hourAngleRad);
        double cosLat = Math.cos(AbstractPlanetTime.toRadians(lat));
        double sinLat = Math.sin(AbstractPlanetTime.toRadians(lat));
        double denominator = cosLat * tanDec - sinLat * cosH;
        for (azimuthAngle = AbstractPlanetTime.toDegrees(Math.atan2(sinH, denominator)); azimuthAngle < 0.0; azimuthAngle += 360.0) {
        }
        while (azimuthAngle >= 360.0) {
            azimuthAngle -= 360.0;
        }
        return azimuthAngle;
    }

    public Point2D.Double getSubEarthPoint(int how) {
        if (how != 1 && how != 2) {
            throw new IllegalArgumentException("Coord. system must be either p'centric or p'graphic.");
        }
        if (!this.gotEarthParams) {
            this.calculateEarthParams();
        }
        if (how == 1) {
            return new Point2D.Double(this.subEarthLonPG, this.declinEPG);
        }
        return new Point2D.Double(-this.subEarthLonPG, this.declinEPC);
    }

    public Point2D.Double getEarthAngleAt(double lon, double lat) {
        double azimuthAngle;
        double hourAngle;
        Point2D.Double earthAngle = this.getSubEarthPoint(1);
        double subLon = earthAngle.getX();
        double subLatRad = earthAngle.getY() * (Math.PI / 180);
        double cosDeclination = Math.cos(subLatRad);
        double sinDeclination = Math.sin(subLatRad);
        double tanDeclination = sinDeclination / cosDeclination;
        for (hourAngle = lon - subLon; hourAngle > 180.0; hourAngle -= 360.0) {
        }
        while (hourAngle < -180.0) {
            hourAngle += 360.0;
        }
        double cosLat = Math.cos(lat * (Math.PI / 180));
        double sinLat = Math.sin(lat * (Math.PI / 180));
        double cosHourAngle = Math.cos(hourAngle * (Math.PI / 180));
        double sinHourAngle = Math.sin(hourAngle * (Math.PI / 180));
        double zenithAngle = AbstractPlanetTime.toDegrees(Math.acos(sinDeclination * sinLat + cosDeclination * cosLat * cosHourAngle));
        for (azimuthAngle = AbstractPlanetTime.toDegrees(Math.atan2(sinHourAngle, cosLat * tanDeclination - sinLat * cosHourAngle)); azimuthAngle < 0.0; azimuthAngle += 360.0) {
        }
        while (azimuthAngle >= 360.0) {
            azimuthAngle -= 360.0;
        }
        return new Point2D.Double(azimuthAngle, zenithAngle);
    }

    public double getTimeHours(double lon, int timeType) {
        if (timeType == 1 && lon == 0.0 || timeType == 4) {
            return this.mst0;
        }
        if (timeType == 2 && lon == 0.0) {
            return this.tst0;
        }
        double xst = 0.0;
        if (timeType == 3) {
            xst = this.mst0 + (double)this.getTimezoneOffset(lon);
        } else if (timeType == 1) {
            xst = this.mst0 - lon / 15.0;
        } else if (timeType == 2) {
            xst = this.tst0 - lon / 15.0;
        } else {
            throw new IllegalArgumentException("Unknown time type");
        }
        while (xst < 0.0) {
            xst += 24.0;
        }
        while (xst >= 24.0) {
            xst -= 24.0;
        }
        return xst;
    }

    public String getTimeStringMinutes(double lon, int timeType, boolean appendSuffix) {
        return this.getTimeString(lon, timeType, 2, appendSuffix);
    }

    public String getTimeStringSeconds(double lon, int timeType, boolean appendSuffix) {
        return this.getTimeString(lon, timeType, 3, appendSuffix);
    }

    public String getTimeString(double lon, int timeType, int smallest, boolean appendSuffix) {
        double hours = this.getTimeHours(lon, timeType);
        if (!appendSuffix) {
            return TimeFormat.format24Hours(hours, smallest, null, null);
        }
        if (timeType == 1) {
            return TimeFormat.format24Hours(hours, smallest, null, "LMST");
        }
        if (timeType == 2) {
            return TimeFormat.format24Hours(hours, smallest, null, "LTST");
        }
        if (timeType == 4) {
            return TimeFormat.format24Hours(hours, smallest, null, this.pmTimeSuffix);
        }
        if (timeType == 3) {
            int tz = this.getTimezoneOffset(lon);
            if (tz < 0) {
                return TimeFormat.format24Hours(hours, smallest, null, this.pmTimeSuffix + tz);
            }
            return TimeFormat.format24Hours(hours, smallest, null, this.pmTimeSuffix + "+" + tz);
        }
        throw new IllegalArgumentException("Unknown time type");
    }

    public double getEOTHours() {
        return this.eotHours;
    }

    public int getTimezoneOffset(double lon) {
        while (lon > 180.0) {
            lon -= 360.0;
        }
        while (lon < -180.0) {
            lon += 360.0;
        }
        return (int)Math.round(-lon / 15.0);
    }

    public String getSeason(boolean inNorth) {
        int seasonNo;
        StringBuffer theSeason = new StringBuffer();
        if (inNorth) {
            if (this.ellS >= 0.0 && this.ellS <= 0.5 || this.ellS >= 359.5 && this.ellS <= 360.0) {
                return "NH Vernal Equinox";
            }
            if (this.ellS >= 89.5 && this.ellS <= 90.5) {
                return "NH Summer Solstice";
            }
            if (this.ellS >= 179.5 && this.ellS <= 180.5) {
                return "NH Autumnal Equinox";
            }
            if (this.ellS >= 269.0 && this.ellS <= 270.5) {
                return "NH Winter Solstice";
            }
            seasonNo = (int)(this.ellS / 90.0);
            switch (seasonNo) {
                case 0: {
                    theSeason.append("NH Spring");
                    break;
                }
                case 1: {
                    theSeason.append("NH Summer");
                    break;
                }
                case 2: {
                    theSeason.append("NH Autumn");
                    break;
                }
                case 3: {
                    theSeason.append("NH Winter");
                }
            }
        } else {
            if (this.ellS >= 0.0 && this.ellS <= 0.5 || this.ellS >= 359.5 && this.ellS <= 360.0) {
                return "SH Autumnal Equinox";
            }
            if (this.ellS >= 89.5 && this.ellS <= 90.5) {
                return "SH Winter Solstice";
            }
            if (this.ellS >= 179.5 && this.ellS <= 180.5) {
                return "SH Vernal Equinox";
            }
            if (this.ellS >= 269.0 && this.ellS <= 270.5) {
                return "SH Summer Solstice";
            }
            seasonNo = (int)(this.ellS / 90.0);
            switch (seasonNo) {
                case 0: {
                    theSeason.append("SH Autumn");
                    break;
                }
                case 1: {
                    theSeason.append("SH Winter");
                    break;
                }
                case 2: {
                    theSeason.append("SH Spring");
                    break;
                }
                case 3: {
                    theSeason.append("SH Summer");
                }
            }
        }
        double section = this.ellS % 90.0;
        if (section < 30.0) {
            theSeason.insert(0, "Early ");
        } else if (section < 60.0) {
            theSeason.insert(0, "Mid ");
        } else {
            theSeason.insert(0, "Late ");
        }
        return theSeason.toString();
    }

    public abstract MissionTime getMissionTime(String var1);

    protected static double toDegrees(double d) {
        return d * 57.29577951308232;
    }

    protected static double toRadians(double d) {
        return d * (Math.PI / 180);
    }
}

