#! /bin/perl

my $usage = <<HERE;
xlat.pl - Translation help for texts.properties files
Version 1.0 - 24 Mar 2005
(C) Olivier Biot, 2005
Anyone is given permission to freely use and modify this file.

Initial purpose: help maintaining the language files for the ExhibitPlus skin
for the JAlbum web album generator.

This perl script will generate a new language file based on the initial
(typically English) file and a previous language file.

The principle is simple: load all "key = value" entries from each language file
in an associative array (a 'hash'), then reprocess the initial file replacing
the initial "key = value" entries with a commented-out initial entry and the
translated value.

Benefits:
1. Get rid automatically of duplicates and obsolete entries.
2. All entries are sorted the same way in the processed files.
3. Original and translated key-value pair can be displayed.
4. If desired, the items lacking a translation can have a "TODO" label,
   allowing easy location of these entries (and easy commenting-out if desired)
5. The boilerpate (first block of comments ended with a blank line) of the
   translated file is copied, and the boilerplate from the original file is
   skipped.

Usage:
1. Output to the console:
  ./xlat.pl original_texts.properties translated_texts.properties
2. Output to a file:
  ./xlat.pl original_texts.properties translated_texts.properties > new_file
  
HERE

if ($#ARGV != 1) {
	print STDERR $usage;
	exit 0;
}

my $props_original = $ARGV[0];
my $props_translated = $ARGV[1];


# Read the initial language file (e.g., English) into the hash "%original"
my %original;
open(PROPS_ORIGINAL, $props_original) or die "Can't open original props file '$props_original'!";
while (<PROPS_ORIGINAL>) {
	my $line = $_;
	$line =~ s/[\r\n]//g;
	if ($line =~ /^([^\/].*) = (.*)/) {
		$original{$1} = $2;
	}
}
close PROPS_ORIGINAL;


# Read in a translated language file into the hash "%translated"
my %translated;
open(PROPS_TRANSLATED, $props_translated) or die "Can't open translated props file '$props_translated'!";
while (<PROPS_TRANSLATED>) {
	my $line = $_;
	$line =~ s/[\r\n]//g;
	if ($line =~ /^([^\/].*) = (.*)/) {
		$translated{$1} = $2;
	}
}
close PROPS_TRANSLATED;



open(PROPS_ORIGINAL, $props_original) or die "Can't open original props file '$props_original'!";

# Copy the existing boilerplate (ends at first blank line!) of a translated file
open(PROPS_TRANSLATED, $props_translated) or die "Can't open translated props file '$props_translated'!";
my $line;
while (($line = <PROPS_TRANSLATED>) && ($line !~ /^[\s]*$/) && ($line !~ /^([^\/].*) = (.*)/)) {
	$line =~ s/[\r\n]//g;
	print "$line\n";
}
if ($line =~ /^[\s]*$/) {
	print "\n";
}
close PROPS_TRANSLATED;


#Skip the original boilerplate (ends at first blank line!)
while (($line = <PROPS_ORIGINAL>) && ($line !~ /^[\s]*$/) && ($line !~ /^([^\/].*) = (.*)/)) {
	; # Skip that boilerplate line
}


# Paste the translations
while (<PROPS_ORIGINAL>) {
	my $line = $_;
	$line =~ s/[\r\n]//g;
	if ($line =~ /^([^\/].*) = (.*)/) {
		my $k = $1;

		if (length($translated{$k}) > 0) {
			print "// Original: $k = $original{$k}\n$k = $translated{$k}\n";
		} else {
			print "// Original: $k = $original{$k}\n//$k = TODO\n";
		}
	} else {
		print "$line\n";
	}
}

close PROPS_ORIGINAL;
