#define DEBUG_FIFO2

#include "basics.hxx"

#include "audio/audio_buffer.hxx"
#include "audio/audio_fifo.hxx"
#include "in/input_file.hxx"
#include "mathhelp.hxx"
#include "param_init.hxx"

#include "debug.hxx"

using namespace std;

static const unsigned int testfile_rate = 1024;
// The floating point constant is not allowed inside the class.
static const float zerolevel = 0.1;
// Make that match the above.
const string prebuffer_zerolevel = "0.1";
static const size_t lead_in = 15;
static const size_t lead_out = 20;
static const off_t length = 333;
// Manually prepared the string form of parameters...
// lead_in+lead_out+30 = 
const string prebuffer_seconds = "0.0634765625";
// max(leadin_in, lead_out)+20
const string prebuffer_zerorange = "0.0390625";

// A test waveform, some below-zero wobbling at beginning and end, louder wobbling in the middle.
class test_file: public input_file
{
private:
	off_t mypos;
	void synth(audio_io_type &left, audio_io_type &right)
	{
		// Works for rather small positions... but eases debugging.
		audio::scale_to_sample(zerolevel, audio::audio_io_code, &left);
		left -= mypos;
		if(mypos >= lead_in && (size_t)(length-mypos) > lead_out)
		left *= 2;

		right = -left;
	};

public:

	test_file(): input_file(input::invalid), mypos(0) {};
	~test_file(){};
	bool do_read(void *abuf, size_t wanted_bytes, size_t &got_bytes)
	{
		size_t wanted_samples = format.bytes2samples(wanted_bytes);
		size_t got_samples = 0;
		audio_io_type* buf = (audio_io_type*)abuf;
		while(mypos < length && got_samples < wanted_samples)
		{
			synth(buf[got_samples*2], buf[got_samples*2+1]);
			++mypos;
			++got_samples;
		}
		got_bytes = format.samples2bytes(got_samples);
		return true;
	};
	bool do_open(const string location)
	{
		mypos = 0;
		format.channels = 2;
		format.rate = testfile_rate;
		return true;
	};
	void do_close(void){ };
	bool do_seek(off_t thepos){ mypos = position = min(length, thepos); return true; };
	off_t do_length(void){ return length; };
};


void print_buffer(audio_buffer* abo)
{
	for(unsigned int t = 0; t < abo->size; ++t) cout << t << ":(" << abo->ptr[2*t] << "|" << abo->ptr[2*t+1] << ")" << endl; 
}


int main()
{
	int err = 0;
	test_file tf;
	dmd::audio_fifo* afi;
	audio_buffer abu(2,4);
	audio_buffer whole(2, length), whole2(2, length);

	afi = new dmd::audio_fifo();

	cerr << "This is the test data we are dealing with:" << endl;
	tf.do_open("someplace");
	tf.buf_read(whole);
	whole.printout(cerr);

	afi->set_source(&tf);
	afi->do_open_safe("someplace");
	if(!afi->buf_read(whole2))
	{
		MERROR("buf_read failed... %s (%i)\n", afi->err.text.c_str(), afi->err.code);
		++err;
	}
	// No zeroscan occured, everything must match!
	if(whole2.fill != (size_t)length)
	{
		MERROR("Did not get what I want (%zu != %" PRIiMAX ")!", whole2.fill, (intmax_t)length);
		++err;
	}
	else
	for(size_t i=0; i<(size_t)length; ++i)
	{
		if(   whole.ptr[2*i]   != whole2.ptr[2*i]
		   || whole.ptr[2*i+1] != whole2.ptr[2*i+1] )
		{
			SERROR("Audio data from file and fifo does not match!");
			++err;
			fprintf(stderr, "This is what I got instead:\n");
			whole2.printout(cerr);
			break;
		}
	}

	if(err) goto end;
	else cerr << "Full read OK." << endl;

cerr << "deleting afi" << endl;
	delete afi;
cerr << "continuing ..." << endl;
	param.concurrent();
	param("input", "zeroscan") = "on";
	param("input", "prebuffer") = prebuffer_seconds;
	param("input", "zerorange") = prebuffer_zerorange;
	param("input", "zerolevel") = prebuffer_zerolevel;
	afi = new dmd::audio_fifo();
	afi->set_source(&tf);

	tf.do_seek(0);
	SDEBUG("starting fifo");
	afi->do_open_safe("someplace");
	whole2.fill = 0;
	afi->buf_read(whole2);
	// Now check if we got the body of the track.
	if(whole2.fill != length-lead_in-lead_out)
	{
		MERROR("Did not get what I want (%zu != %"PRIiMAX")!", whole2.fill, (intmax_t)(length-lead_in-lead_out));
		++err;
	}
	else
	for(size_t i=0; i<whole2.fill; ++i)
	{
		if(   whole.ptr[2*(i+lead_in)]   != whole2.ptr[2*i]
		   || whole.ptr[2*(i+lead_in)+1] != whole2.ptr[2*i+1] )
		{
			SERROR("Audio data from file and fifo does not match!");
			++err;
			break;
		}
	}

	if(err) goto end;
	else cerr << "Zeroscanned read OK." << endl;

end:
	delete afi;
	cout << (err ? "FAIL" : "PASS") << endl;
	return err;
}

