// This test shall help digging for a bug I see with thread synchronization.
// It's the output processor pushing buffers to the output writer... somehow the fill value of the buffers isn't communicated properly (and possibly contents?). Trying to mimick that using a plain bufferchain and two threads synthetically working on it.
// So far it did _not_ reproduce the issue. Now, is that good?

#include "basics.hxx"
#include "threads/threads.hxx"
#include "threads/semaphore.hxx"
#include "audio/audio_bufferpool.hxx"

#include "param_init.hxx"

#include "debug.hxx"

static size_t bufsize;
static size_t bufcount;

#define SIZEERR(buf, count) MERROR("Size of avail buffer %p is bad (%zu != %zu at %p error %u)", buf, buf->size, bufsize, &buf->size, count)


class writer: public dmd::thread
{
	public:
	dmd::audio_bufferpool pool;
	unsigned int errcount;
	semaphore sem; // That's poked on buffer completion.
	 writer(unsigned int buffers): pool(), errcount(0), sem() { pool.add(buffers); };
	~writer() {};
	void thread_work()
	{
		printf("Writer alive.\n");
		while(1)
		{
			audio_buffer *buf = pool.fetch_used(this);
			//printf("Fetched buffer %p\n", buf);
			// buf->fill = buf->size;
			if(buf->size != bufsize)
			{
				++errcount;
				SIZEERR(buf, errcount);
			}
			else if(buf->fill != buf->size)
			{
				++errcount;
				MERROR("Buffer has only fill of %zu/%zu (error %u)", buf->fill, buf->size, errcount);
			}
			buf->fill = 0;
			fprintf(stderr, "Returning buffer %p (%zu/%zu)\n", buf, buf->fill, buf->size);
			pool.store_avail(buf);
			sem.post();
		}
	};
};

class processor: public dmd::thread
{
	public:
	dmd::audio_bufferpool *pool;
	unsigned int errcount;
	semaphore sem; // Poke that to make it process a buffer.
	 processor(dmd::audio_bufferpool &pool_): pool(&pool_), errcount(0), sem() {};
	~processor() {};
	void thread_work()
	{
		printf("Processor alive.\n");
		while(sem.wait(this))
		{
			audio_buffer *buf = pool->fetch_avail(this);
			if(buf->size != bufsize)
			{
				++errcount;
				SIZEERR(buf,errcount);
			}
			//printf("Fetched free buffer %p and stuffing it back.\n", buf);
			for(size_t i=0; i<buf->size*buf->channels; ++i) (*buf)[i] = i;

			buf->fill = buf->size;
			fprintf(stderr, "Processed buffer %p (%zu/%zu)\n", buf, buf->fill, buf->size);
			pool->store_used(buf);
		}
	};
};

// Fetch all buffers in the pool and check if their size is correct.
unsigned int check_sizes(dmd::audio_bufferpool &pool)
{
	unsigned int errcount = 0;
	// Extract and count.
	for(size_t i=0; i<bufcount; ++i)
	{
		audio_buffer *buf = pool.fetch_avail();
		if(buf->size != bufsize)
		{
			++errcount;
			SIZEERR(buf, errcount);
		}
		else{ fprintf(stderr, "Size of %p at %p is good.\n", buf, &buf->size); }
		pool.store_used(buf);
	}

	// Give them back.
	for(size_t i=0; i<bufcount; ++i)
	{
		audio_buffer *buf = pool.fetch_used();
		pool.store_avail(buf);
	}

	return errcount;
}

int main(int argc, char **argv)
{
	bufsize = param.as_size("buffer");
	bufcount = param.as_size("output","buffers");
	writer worf(bufcount);
	processor picard(worf.pool);
	unsigned int tries = 20;
	unsigned int errs = 0;

	if(argc > 1) tries = atoi(argv[1]);

	// Verify the buffer sizes in the main thread.
	fprintf(stderr, "Checking sizes beforehand...\n");
	errs += check_sizes(worf.pool);

	worf.thread_create();
	picard.thread_create();

	fprintf(stderr, "Starting work.\n");
	for(unsigned int i=0; i<tries; ++i)
	picard.sem.post();

	fprintf(stderr,"Waiting for work to end.\n");
	for(unsigned int i=0; i<tries; ++i)
	worf.sem.wait();

	picard.thread_kill();
	worf.thread_kill();

	// Verify sizes again.
	fprintf(stderr, "Checking sizes after work...\n");
	errs += check_sizes(worf.pool);

	printf("Errors: %u\n", errs+worf.errcount+picard.errcount);
	printf("%s\n", errs+worf.errcount+picard.errcount ? "FAIL" : "PASS");
	return 0;
}
