#ifndef H_DMD_THREADS
#define H_DMD_THREADS
/*
	threads: Data structures to ease working with threads.

	part of DerMixD
	(c)2007-2012 Thomas Orgis, licensed under GPLv2

	Initial spark of creation:
	This class currently only exists to ease gathering of thread info.
	I want to know who's there to explain the thread/process count.

	The new whole picture:
	My thread functions do quite similar things. They are run from a thread container, they take a pointer to a specific class to work on. They typically do not create own data structures, they work with what they got. The specific class they get handed also often is of type thread; serving to settle priorities or whatnot. The main body is a loop, either waiting for a specific message to stop existing or waiting endlessly for cancellation.

	That can be structured in a common approach.

	Let's define one generic thread function that takes as argument some class that fulfills a generic interface for thread work. The thread creation itself can be wrapped up, too. All that I should do to create a new type of working thread is to flesh out a derived class with the actual work and let the common infrastructure mess with the actual thread creation and termination.

	All my threads are ended via cancellation - be it inside the work method of the derived class or after it in the common internal function that is used for all threads (it sleeps forever there, until cancelled).
	But: Per default, cancellation is disabled while thread_work() is run, so this method either must return normally or you must use thread_cancelable(true) some place to make cancellations possible during thread_work().
	The companion classes for semaphores and related things support this model,
	for the standard scheme of having the thread waiting for work and being
	cancellable while doing so. You absolutely must ensure that you do not
	hold any non-primitive data on the stack while the thread can be cancelled.
	There won't be any destructors run. Things will break horribly.
	Oh, and of course you shouldn't hold a mutex while being cancelled ...

	Thing for the future: Should the container cancel / join threads?

	About signals: You have to do special work to get signals and threads work together (namely, make them go to a specific thread to get some consistency). Threads created with this code by default will ignore all signals.
*/

#include "mutex.hxx"
#include <vector>
#include <string>
#include <signal.h> // Portability trouble?

namespace dmd {

// You are supposed to inherit from the thread class and flesh out thread_work().
// All specifics of the underlying multithreading implementation are wrapped here.
class thread
{
	private:
	struct my_private;
	struct my_private *parts;

	public:
	std::string thread_name; // of the thread (type)
	// Optional arguments are the name and the priority offset to use for factory threads.
	// Other data fields like thread ID can only be filled in after thread construction, which is after construction of the thread instance.
	thread(const std::string name = "some thread", int priority=0);
	thread(const thread &b);
	// Caution: Make sure there is no thread active anymore when destructing (call thread_kill() before).
	// If there is a thread active, the destructor will abort()!
	virtual ~thread();

	// Functionality for the thread itself.

	// This is the actual work that the thread does.
	// Typically, this is a loop waiting and acting on certain events.
	// Note that even if you return out of that method, the thread will continue to run, sleeping, solely to wait for cancellation.
	virtual void thread_work() = 0;
	// Do something to shift the priority of the calling thread down/up.
	// Only does something on linux for now.
	void thread_priority(int priority);
	// Actually just a hook to call thread_workout with the correct argument, from the thread's private parts.
	void thread_setup();

	// Enable/disable cancelability
	// You can use this to create sections of code that should (not) be interrupted by a cancellation request.
	// Default is disabled cancelability, to give you some safety for the thread's work. But where the thread blocks for I/O, one should consider enabling cancellability.
	// The rule is: Your thread should always respond to a cancellation request in a sensible amount of time.
	// CAUTION: Do never make the thread cancelable for a time it could take hold of some sort of mutex -- this is calling for deadlocks!
	// Essentially, whenever you call external code, assume that it does not like being cancelled; an nice example is simple work with stdio -- access to stdout/stderr is (often?) serialized using locks in the C library. You can block output forever if your thread is cancelled during fprintf()!
	// I wonder: Is this really how it is supposed to be or did I hit a bug in glibc 2.11? If fprintf() is a cancellation point, should it leave mutexes dangling?
	// I suppose it is a bug, but it's out there, in many versions of glibc, apparently. So we need to handle it.
	void thread_cancelable(bool val=true);

	// Insert a cancellation point, use that inside thread_work() if you want.
	void thread_cancelpoint();

	// Functionality for the outside.

	// Make the thread go away. That includes cancellation and joining.
	// Even if this returns false to indicate trouble, the thread is marked as dead -- reasoning that there's nothing better to do.
	bool thread_kill();

	// Diagnostic method to generate thread descrition list for the container.
	void thread_add_info(std::vector<std::string*> &tl);
	// Start the thread and have fun (if this returns true).
	bool thread_create();
};

// The thread container serves as a tool to have some ordering of threads.
// It can start the threads for you... it takes prepared thread objects (well, pointers) to keep track of. It does _not_ take care of destructing these thread objects.
class thread_container
{
	public:
	std::string thread_container_name;
	int thread_errcode; // Verbatim from pthread_create().
	std::vector<thread*> threads;
	real_mutex threads_lock;
	thread_container *sub_thread_container;
	thread_container(const std::string name_="");
	virtual ~thread_container();
	// returns thread count, appends descriptions
	virtual size_t thread_info(std::vector<std::string*> &tl);
	// Take a prepared thread data structure, try to start the corresponding thread, if successful, add to thread list.
	// Returns true if starting of thread succeeded.
	bool create_thread(thread* ted);
	// Add a thread that has already been created by some other means.
	void add_thread(thread *ted);
	// Removes a thread from the list and kills it.
	// Return value is verbatim copy of ted->thread_kill() or false if not found in list.
	bool kill_thread(thread *ted);
	// Kill all threads, empty the list. Returns false if any kill did not work.
	bool kill_threads();
	// Kill and delete the objects.
	bool erase_threads();
	// Just forget about all threads (they have been killed outside).
	void forget_threads();
};

// Those are here because thread-safe signal functionality is platform-dependent.
// On POSIX, this means using pthread functions... though, API very much tied to signal.h, which is still openly exposed as long I use the plain sigset_t here.
namespace threads {

// Set the set of signals to be blocked, all by default.
// Returns the old mask, if non-null pointer provided.
bool block_signals(sigset_t *mask = NULL, sigset_t *oldmask = NULL);
// Wait for a set of signals, any by default.
bool wait_for_signal(sigset_t *mask = NULL, int *signum = NULL);

}}
#endif
