/*
	threads: Data structures to ease working with threads.

	part of DerMixD
	(c)2007-9 Thomas Orgis, licensed under GPLv2

	I settled for cancellation of threads as the one and only way to end them. Any dynamic memory the thread allocates shall be contained within the thread class instance, which will be cleaned up by whoever created it.
	That way, I do not have to deal with cleanup handlers.
*/

#ifdef LINUX
#	ifndef _GNU_SOURCE // Need it for syscall.
#		define _GNU_SOURCE
#	endif
#	include <sys/syscall.h>
#	include <sys/time.h>
#	include <sys/resource.h>
#endif

// Hack for solaris headers that contain a struct mutex.
#define mutex solaris_mutex
#include <stdlib.h>
#include <unistd.h>
#include <pthread.h>
#include <signal.h>

#undef mutex

#include "threads/threads.hxx"
#include "syserror.hxx"
#include "tstring.hxx"
#include "shortcuts.hxx"
#include "debug.hxx"

#define CMDEBUG(s, ...) MDEBUG( "[thread %s %p] " s, thread_name.c_str(), this, __VA_ARGS__)
#define CSDEBUG(s) MDEBUG( "[thread %s %p] " s, thread_name.c_str(), this)

using namespace std;

// Some might like a extern "C" function here... not sure about that myself.

// The actual function used for the thread.
// It sets the priority stuff, then calls your work method.
extern "C" {
static void* thread_worker_func(void *p)
{
	dmd::thread *td = (dmd::thread*) p;
	// Remember: No stdio calls while being cancelable!
	td->thread_cancelable(false);
	MDEBUG("[thread %p] startup", td);
	td->thread_setup();
	MDEBUG("[thread %p] work", td);
	td->thread_work();
	// Make sure we are not cancelable during printing.
	td->thread_cancelable(false);
	MDEBUG("[thread %p] natural end, waiting to be cancelled", td);
	td->thread_cancelable(true);
	// Yeah, I rely on cancellation instead of simple return.
	// It is not trivial to know when not to cancel otherwise. Or is it?
	while(1) sleep(100); 
	return NULL;
}}

namespace dmd {

struct thread::my_private
{
	pthread_t id; // thread ID ... opaque identifier, actually.
	// That field could mean something outside the Linux world when I add support for proper thread scheduling priorities. Does NPTL actually support these now?
	int base_priority;
	bool alive; // The thread is started.
#ifdef LINUX
	pid_t pid;
	int priority;
#endif
};

thread::thread(const string myname, int priority):
	parts(new my_private), thread_name(myname)
{
	CONSTRUCT("thread");
	parts->base_priority = priority;
	parts->alive = false;
#ifdef LINUX
	parts->pid = -1;
	parts->priority = 0;
#endif
}

thread::thread(const thread &b): parts(new my_private), thread_name(b.thread_name)
{
	*parts = *b.parts;
}

thread::~thread()
{
	DESTBEGIN("thread");
	if(parts->alive)
	{
		MERROR("Programmer slap: The thread %p (\"%s\") is still alive after the derived classes destructor has been run. This is very unhealthy. This is fatal, even. You MUST kill the thread before destructing the data it works on!", this, thread_name.c_str());
		abort(); // Yes, it is that fatal.
	}

	delete parts;
	DESTEND("thread");
}

void thread::thread_setup()
{
#ifdef LINUX
	parts->pid = syscall(__NR_gettid); // gettid();
#endif
	if(!dmd::threads::block_signals())
	{
		MERROR( "[thread %s %p] cannot block signals!", thread_name.c_str(), this);
		// Can I react to this? Should I? I don't really expect that call to fail.
	}
	thread_priority(parts->base_priority);
}

void thread::thread_priority(int priority)
{
#ifdef LINUX
	syserror serr;
	int old_priority = getpriority(PRIO_PROCESS, 0); // or parts->pid? should not matter
	if(serr.problem())
	{
		MERROR("...getting priority: %s",serr.str());
		return;
	}

	MDEBUG("[thread %s %p] Experiment with Linux/NTPL oddities: Shift my niceness to %i from %i.", thread_name.c_str(), this, priority, old_priority);

	if(setpriority(PRIO_PROCESS, 0, priority) < 0)
	MERROR("Failed to set priority: %s", serr.str());

	serr.clear();
	parts->priority = getpriority(PRIO_PROCESS, 0);
	if(serr.problem())
	{
		MERROR("...getting priority again: %s",serr.str());
		return;
	}
	MDEBUG("[thread %s %p] Experiment with Linux/NTPL oddities: Shift from %i done, niceness now %i.", thread_name.c_str(), this, old_priority, parts->priority);
#endif
}

void thread::thread_cancelpoint()
{
	pthread_testcancel();
}

void thread::thread_cancelable(bool val)
{
	int newstate = val ? PTHREAD_CANCEL_ENABLE : PTHREAD_CANCEL_DISABLE;
	int oldstate; // One should always provide this for portability.
	MDEBUG("[thread %p] making \"%s\" %s...", this, thread_name.c_str(), val ? "cancelable" : "uncancelable");
	if(pthread_setcancelstate(newstate, &oldstate) != 0)
	{
		MERROR("[thread %p] Unexpected error in setting cancelability of \"%s\"!", this, thread_name.c_str());
	}
}

bool thread::thread_kill()
{
	if(!parts->alive)
	{
		MDEBUG("You want to kill thread %p of \"%s\" that is not alive? Well, dead is dead.", this, thread_name.c_str());
		return false;
	}
	MDEBUG("[thread %p] Cancel/join of \"%s\".", this, thread_name.c_str());
	// In any case, we assume that the thread will be dead, or somehow unrecoverable soon.
	parts->alive = false;
	int err = pthread_cancel(parts->id);
	if(err == 0)
	{
		void *res;
		MDEBUG("[thread %p] Cancelled \"%s\", joining.", this, thread_name.c_str());
		err = pthread_join(parts->id, &res);
		if(err == 0)
		{
			MDEBUG("[thread %p] Joined \"%s\".", this, thread_name.c_str());
			if(res != PTHREAD_CANCELED)
			{
				MERROR("[thread %p] \"%s\" has not finished via cancellation? That is bogus.", this, thread_name.c_str());
				return false;
			}
			// That's the only true path to enlightment.
			return true;
		}
		else
		{
			syserror serr;
			MERROR("[thread %p] Joining of \"%s\" failed: %s", this, thread_name.c_str(), serr.str());
			// Again, it is tricky to do any safe recovery here... so just write it off...
			// We are fucked if joining fails.
			return false;
		}
	}
	else
	{
		// If the thread is actually present and cancel failes for whatever strange reason, we can only hope that the thread does no harm when the instance of the thread class is removed.
		// Well... it should not happen, eh?
		syserror serr;
		MERROR("[thread %p] Unable to cancel \"%s\": %s!", this, thread_name.c_str(), serr.str(err));
		return false;
	}
}

void thread::thread_add_info(vector<std::string*> &tl)
{
	std::string *info = new std::string;
	// Memory address is an universal identifier...
	strprintf(*info, "%p", this);
#ifdef LINUX
	strprintf(*info, "(%i @ %i)", (int)parts->pid, parts->priority);
#endif
	*info += ": ";
	*info += thread_name;
	tl.push_back(info);
}

bool thread::thread_create()
{
	CSDEBUG("wanna create myself");
	if(parts->alive)
	{
		MERROR("You want to create a second thread of \"%s\"?!", thread_name.c_str());
		return false;
	}
	int err = pthread_create(&parts->id, NULL, thread_worker_func, (void*) this);
	if(err == 0)
	{
		CSDEBUG("created");
		parts->alive = true;
		return true;
	}
	else
	{
		syserror serr;
		MERROR("Failed to create thread: %s.", serr.str(err));
		return false;
	}
}

#undef CMDEBUG
#undef CSDEBUG
#define CMDEBUG(s, ...) MDEBUG( "[thread_container %s %p] " s, thread_container_name.c_str(), this,  __VA_ARGS__)
#define CSDEBUG(s) MDEBUG( "[thread_container %s %p] s, ", thread_container_name.c_str(), this)

thread_container::thread_container(const string name_):
	thread_container_name(name_), sub_thread_container(NULL)
{
	CONSTRUCT("thread_container");
}

thread_container::~thread_container()
{
	DESTBEGIN("thread_container");
	// Locking doesn't make sense ... if the destructor is running, nobody should be messing anymore.
	if(!threads.empty())
	{
		MERROR("[thread_container %s %p] Still contains %zu threads during destruction. That indicates that some hreads have been forgotten. This is fatal breakage.", thread_container_name.c_str(), this, threads.size());
		abort();
	}
	DESTEND("thread_container");
}

bool thread_container::create_thread(thread* ted)
{
	CMDEBUG("creating %p", ted);
	if(ted->thread_create())
	{
		add_thread(ted);
		return true;
	}
	else return false;
}

void thread_container::add_thread(thread *ted)
{
	CMDEBUG("adding %p", ted);
	threads_lock.lock();
	threads.push_back(ted);
	threads_lock.unlock();
}

bool thread_container::kill_thread(thread *ted)
{
	bool succ = false;
	CMDEBUG("killing %p", this, ted);

	threads_lock.lock();
	// Only do any action if the thread is still in the list.
	// Rationale: Sloppyness and concurrency of actions could mean that the thread has already been killed by someone else and also the data already deleted...
	FOR_VECTOR(thread*, threads, teddy)
	{
		if(*teddy == ted)
		{
			succ = true;
			threads.erase(teddy);
			break;
		}
	}
	threads_lock.unlock();

	// Rationale for killing the thread outside the lock:
	// We might want to kill a thread that is responsible to kill other threads and thus might be waiting for the very same lock we hold here, while being in an uncancelable state.
	if(succ)
	{
		MDEBUG("[thread_container %p] found %p in list, now really killing", this, ted);
		succ = ted->thread_kill();
	}

	MDEBUG("[thread_container %p] kill %p on finished", this, ted);
	return succ;
}

bool thread_container::kill_threads()
{
	bool succ = true;
	CSDEBUG("killing all threads");
	threads_lock.lock();
	FOR_VECTOR(thread*, threads, teddy)
	{
		bool lsucc = (*teddy)->thread_kill();
		if(succ) succ = lsucc;
	}
	threads.clear();
	threads_lock.unlock();
	CMDEBUG("all threads killed (%i)", (int)succ);
	return succ;
}

bool thread_container::erase_threads()
{
	bool succ = true;
	CSDEBUG("erasing all threads");
	threads_lock.lock();
	FOR_VECTOR(thread*, threads, teddy)
	{
		bool lsucc = (*teddy)->thread_kill();
		if(succ) succ = lsucc;

		delete(*teddy);
	}
	threads.clear();
	threads_lock.unlock();
	CMDEBUG("all threads erased (%i)", (int)succ);
	return succ;
}

void thread_container::forget_threads()
{
	CSDEBUG("forgetting all threads");
	threads_lock.lock();
	threads.clear();
	threads_lock.unlock();
	CSDEBUG("all threads forgotten");
}

size_t thread_container::thread_info(vector<std::string*> &tl)
{
	size_t count = threads.size();
	for(vector<thread*>::iterator i = threads.begin(); i != threads.end(); ++i)
	(*i)->thread_add_info(tl);

	if(sub_thread_container != NULL) count += sub_thread_container->thread_info(tl);
	return count;
}

namespace threads {

bool block_signals(sigset_t *mask, sigset_t *oldmask)
{
	sigset_t *themask;
	sigset_t fullmask;

	if(mask != NULL) themask = mask;
	else
	{
		// Ignore all signals by default.
		sigfillset(&fullmask);
		themask = &fullmask;
	}
	return (pthread_sigmask(SIG_SETMASK, themask, oldmask) == 0);
}

bool wait_for_signal(sigset_t *mask, int *signum)
{
	int signumm;
	sigset_t *themask;
	sigset_t fullmask;

	if(mask != NULL) themask = mask;
	else
	{
		// Watch for all signals by default.
		sigfillset(&fullmask);
		themask = &fullmask;
	}

	if(sigwait(themask, &signumm) == 0)
	{
		if(signum) *signum = signumm;
		return true;
	}
	else return false;
}

}
} // namespace dmd
