// One might imagine ditching that one for syserror functionality.
#include "basics.hxx"
#include <errno.h>
#include <semaphore.h>
#include "threads/semaphore.hxx"
#include "threads/threads.hxx" // For cancelable wait.

#include "syserror.hxx"
#include "debug.hxx"


struct semaphore::my_private
{
	sem_t phore;
};

semaphore::semaphore(): parts(new my_private)
{
	if(sem_init(&parts->phore, 0, 0) != 0)
	{
		syserror serr;
		MERROR("This is really fatal -- failed to initialize a semaphore: %s.", serr.str());
	}
}

semaphore::~semaphore()
{
	sem_destroy(&parts->phore);
	delete parts;
}

bool semaphore::post()
{
	if(sem_post(&parts->phore) != 0)
	{
		syserror serr;
		MERROR("Posting on a semaphore failed: %s.", serr.str());
		return false;
	}
	else return true;
}

bool semaphore::wait(dmd::thread *ted)
{
	int r = 0;
	errno = 0;
	// Remember: Don't use stdio while being cancelable.
	if(ted != NULL) ted->thread_cancelable(true);

	do { r=sem_wait(&parts->phore); } while(r != 0 && errno == EINTR);

	if(ted != NULL) ted->thread_cancelable(false);

	if(r != 0)
	{
		syserror serr;
		MERROR("Waiting on semaphore failed: %s.", serr.str());
	}

	return (r == 0);
}

bool semaphore::trywait()
{
	int r;
	errno = 0;
	// Same as above, we loop as long as we get interrupted.
	// The case errno==EAGAIN is a normal end, just triggers return false, which is a normal result here.
	do { r=sem_trywait(&parts->phore); } while(r != 0 && errno == EINTR);

	if(r != 0 && errno != EAGAIN)
	{
		syserror serr;
		MERROR("Trywait on semaphore failed: %s.", serr.str());
	}

	return (r==0);
}

bool semaphore::getvalue(int &val)
{
	if(sem_getvalue(&parts->phore, &val) == 0)
	{
		// If we got a waiter count, the actual value is 0.
		if(val < 0) val = 0;

		return true;
	}
	else
	{
		syserror serr;
		MERROR("Cannot get semaphore value: %s.", serr.str());
		return false;
	}
}

bool semaphore::zero()
{
	errno = 0;
	do {} while(sem_trywait(&parts->phore) == 0 || errno == EINTR);

	if(errno != EAGAIN)
	{
		syserror serr;
		MERROR("Cannot zero semaphore: %s.", serr.str());
	}

	return (errno == EAGAIN); // I zeroed successfully if the sem_wait() would block.
}
