/*
	syserror: Safely retrieve an error string for system errors. Multithreading style.

	part of DerMixD
	(c)2009 Thomas Orgis, licensed under GPLv2

	We want to use strerror_r() for multithreaded apps, providing the storage for the string. Also, since we do not know how much storage is really needed, we need to increment the storage size and reiterate...
	But: We cannot rely on strerror_r() being there, at least it's non-trivial to know _which_ strerror_r() is there. The GNU C++ compiler defines _GNU_SOURCE so that with glibc, I do not get the XSI-compliant strerror_r() but the (inferior) GNU variant instead.
	So, I am doing something drastic: I simply use the non-threadsafe strerror(), protected by a mutex. This is not too nice, but we are dealing with special error situations here already.
*/

#include <cerrno>
#include <cstdlib>
#include <cstring>
#include "syserror.hxx"
#include "mutex.hxx"

using namespace std;

// This one is used to serialize access to strerror().
// Valgrind complains about lost memory, but I _see_ that the destructor is run!
static real_mutex errmut;

syserror::syserror(): buf(NULL), bufsize(0){ }

syserror::~syserror()
{
	if(buf != NULL) free(buf);
}

void syserror::clear(){ errno = 0; }

bool syserror::problem(){ return (errno != 0); }

const char* syserror::str()
{
	return str(errno);
}

// Prepare storage, try to get error string. Increase storage size and try again.
const char* syserror::str(int errcode)
{
	errmut.lock();
	const char *msg = get_errtext(errcode);
	errmut.unlock();
	return msg;
}

// The plain error text retrieval / copying.
const char* syserror::get_errtext(int errcode)
{
	char *errtext = strerror(errcode);
	size_t newsize = strlen(errtext)+1;

	// Prepare storage.
	if(newsize > bufsize)
	{
		// Safe realloc... some reallocs don't work for buf == NULL.

		if(buf == NULL)
		{
			buf = (char*) malloc(newsize);
			bufsize = newsize;
		}

		if(buf == NULL) return "Not enough memory for error string!";

		if(newsize > bufsize)
		{
			char* more_buf = (char*) realloc(buf, newsize);
			if(more_buf == NULL) return "Not enough Memory to resize error string storage.";
			else
			{
				buf = more_buf;
				bufsize = newsize;
			}
		}
	}

	// Copy the string and be anal about zero termination.
	memcpy(buf, errtext, bufsize);
	buf[bufsize-1] = 0;

	return buf;
}
