/*
	parsenum: simple parsing of strings to numbers of specific type.

	part of DerMixD
	(c)2009 Thomas Orgis, licensed under GPLv2
*/

// This is self-contained, do not need basics.hxx
#include "parsenum.hxx"

// Really ensure that we use a safe locale that prevents surprises in strtoimax.
#include "safelocale.hxx"

#include <ctype.h>
// We need to use intmax_t, hopefully getting that one here.
#include <inttypes.h>
#include <string.h>

#include <errno.h>

// The actual conversion happens using intmax_t / uintmax_t, then downscaling to the desired type with overflow checking.

using std::string;

// Yeah, octet machines ftw!
#ifndef CHAR_BIT
#define CHAR_BIT 8
#endif

// Type min/max macros from:
// system-dependent definitions for fileutils, textutils, and sh-utils packages.
// Copyright (C) 1989, 1991-2004 Free Software Foundation, Inc.
// GPLv2 licensed
// This assumes a bit bout binary format... but what the heck, how else should I get OFF_T_MIN?

/* The extra casts work around common compiler bugs.  */
#define TYPE_SIGNED(t) (! ((t) 0 < (t) -1))
/* The outer cast is needed to work around a bug in Cray C 5.0.3.0.
   It is necessary at least when t == time_t.  */
#define TYPE_MINIMUM(t) ((t) (TYPE_SIGNED (t) \
                              ? ~ (t) 0 << (sizeof (t) * CHAR_BIT - 1) : (t) 0))
#define TYPE_MAXIMUM(t) ((t) (~ (t) 0 - TYPE_MINIMUM (t)))

const int int_min = TYPE_MINIMUM(int);
const int int_max = TYPE_MAXIMUM(int);
const long long_min = TYPE_MINIMUM(long);
const long long_max = TYPE_MAXIMUM(long);
const off_t off_t_min = TYPE_MINIMUM(off_t);
const off_t off_t_max = TYPE_MAXIMUM(off_t);
const size_t size_t_max = TYPE_MAXIMUM(size_t);

extern const int int_min;

static enum strnum_err str_to_intmax(const string s, intmax_t &value)
{
	const char* begin = s.c_str();
	const char* end = begin;
	size_t len = strlen(begin);
	errno = 0;
	// We require at least something. Not sloppy about taking nothing for zero here.
	if(len < 1) return strnum_invalid;

	// Stupid cast needed to get rid of const-ness of pointer that should still be const in practice.
	value = strtoimax(begin, const_cast<char**>(&end), 10);

	if((size_t)(end - begin) != len) return strnum_invalid;

	if(errno != 0) return strnum_overflow;

	return strnum_good;
}

static enum strnum_err str_to_uintmax(const string s, uintmax_t &value)
{
	const char* begin = s.c_str();
	const char* end = begin;
	size_t len = strlen(begin);
	errno = 0;

	// Beware of negative numbers!
	// strtoumax does handle negative numbers by converting them to positive numbers.
	// That is not what I intended. So, I must check first if there is a leading minus sign.
	{
		size_t i;
		for(i=0; i<len; ++i)
		{
			// If the first non-whitespace character is '-', the number is considered negative.
			if(begin[i] == '-') return strnum_overflow;
			if(!isspace(begin[i])) break;
		}
	}

	// Stupid cast needed to get rid of const-ness of pointer that should still be const in practice.
	value = strtoumax(begin, const_cast<char**>(&end), 10);

	if((size_t)(end - begin) != len) return strnum_invalid;

	if(errno != 0) return strnum_overflow;

	return strnum_good;
}

// The lazy wrapper functions that make the error retrieval optional.

#define lazy_wrap(type, func) \
type func(const std::string s, int *err) \
{ \
	int myerr; \
	type ret = func(s, myerr); \
	if(err != NULL) *err = myerr; \
	return ret; \
}

lazy_wrap(int, str_to_int)
lazy_wrap(long, str_to_long)
lazy_wrap(off_t, str_to_offset)
lazy_wrap(size_t, str_to_size)

// Yeah, could use strtol instead...

int str_to_int(const string s, int &err)
{
	intmax_t tmp;
	err = str_to_intmax(s, tmp);
	if(err != strnum_good) return 0;

	if(tmp < int_min)
	{
		err = strnum_overflow;
		return int_min;
	}

	if(tmp > int_max)
	{
		err = strnum_overflow;
		return int_max;
	}

	return (int)tmp;
}

long str_to_long(const string s, int &err)
{
	intmax_t tmp;
	err = str_to_intmax(s, tmp);
	if(err != strnum_good) return 0;

	if(tmp < long_min)
	{
		err = strnum_overflow;
		return long_min;
	}

	if(tmp > long_max)
	{
		err = strnum_overflow;
		return long_max;
	}

	return (long)tmp;
}

off_t str_to_offset(const string s, int &err)
{
	intmax_t tmp;
	err = str_to_intmax(s, tmp);
	if(err != strnum_good) return 0;

	if(tmp < off_t_min)
	{
		err = strnum_overflow;
		return off_t_min;
	}

	if(tmp > off_t_max)
	{
		err = strnum_overflow;
		return off_t_max;
	}

	return (long)tmp;
}

size_t str_to_size(const string s, int &err)
{
	uintmax_t tmp = 0;
	err = str_to_uintmax(s, tmp);
	if(err != strnum_good) return 0;

	if(tmp > size_t_max)
	{
		err = strnum_overflow;
		return size_t_max;
	}

	return (size_t)tmp;
}
