/*
	param: some parameters in the environment...
	
	part of DerMixD
	(c)2005-2012 Thomas Orgis, licensed MIT-style
*/

//not depending on dermixd's common header, this stuff can easily tranfserred to other projects or into an independent library
#include <cstdlib>
// #include <cstdio>
#include <string>
#include <cstring>
#include <strings.h> // no cstrings
#include <vector>

#include "param.hxx"
#include "debug.hxx"

using namespace std;

void param_space::prepare()
{
	maxlen = 0;
	maxshort = 0;
	separator = "--";
	program = "";
	define("help","no","give help");
	define("param", "print", "no", "print out current config (only considering command line parameters before this one), formatted as config file");
	shortcut("-h", "help=1");
	shortcut("--help", "help=1");
}

param_space::param_space()
{
	CONSTRUCT("param_space");
	prepare();
}

param_space::param_space(const char **pardef)
{
	CONSTRUCT("param_space (with definitions)");
	prepare();
	define_groups(pardef);
}

param_space::~param_space()
{
	DESTBEGIN("param_space");
	for(vector<parm_group*>::iterator g = pgv.begin(); g != pgv.end(); ++g)
	delete (*g);
	pgv.clear();
	single(); // That deletes the real mutex.
	DESTEND("param_space");
}

string& param_space::val(const string group, const string name)
{
	parm* p = NULL;
	if(get_parm_struct(group,name,p) == 0) return p->value;
	else
	{
		MERROR("Access to non-existing parameter %s.%s!", group.c_str(), name.c_str());
		return dummy;
	}
}

bool param_space::concurrent()
{
#ifdef PARAM_MULTITHREAD
	if(the_lock.mutti == NULL)
	the_lock.mutti = new real_mutex();

	return (the_lock.mutti != NULL);
#else
	return false;
#endif
}

void param_space::single()
{
#ifdef PARAM_MULTITHREAD
	if(the_lock.mutti != NULL)
	{
		delete the_lock.mutti;
		the_lock.mutti = NULL;
	}
#endif
}

void param_space::lock()
{
#ifdef PARAM_MULTITHREAD
	the_lock.lock();
#endif
}

void param_space::unlock()
{
#ifdef PARAM_MULTITHREAD
	the_lock.unlock();
#endif
}

parm_group* param_space::get_group(const string name)
{
	for(vector<parm_group*>::iterator i = pgv.begin(); i != pgv.end(); ++i)
	{
		if((*i)->name == name) return *i;
	}
	return NULL;
}

int param_space::get_parm_struct(const string group, const string name, parm* &pmp)
{
	parm_group* grp = get_group(group);
	if( grp != NULL )
	{
		if( (pmp = grp->get(name)) != NULL ) return 0; //found
		else
		{
			MERROR("no param \"%s\" in group \"%s\"", name.c_str(), group.c_str());
			return -2;
		}
	}
	else
	{
		MERROR("no group \"%s\"", group.c_str());
		return -1;
	}
}

void param_space::define(const string group, const string name, const string value, const string help)
{
	lock();
	parm_group* grp = get_group(group);
	if(grp == NULL)
	{
		pgv.push_back(new parm_group(group));
		grp = pgv.back();
	}
	parm* p = grp->get(name);
	if(p == NULL)
	{
		grp->parms.push_back(new parm);
		p = grp->parms.back();
	}
	else	MERROR("Redefining \"%s\" in group \"%s\" - the coder is serious?", name.c_str(), group.c_str());
	unsigned int nl = group.length() + name.length() + 1;
	if(nl > maxlen) maxlen = nl;
	p->name = name;
	p->help = help;
	p->value = value;
	unlock();
}

void param_space::define(const char **pardef, const string group)
{
	size_t n = 0;
	const char *key[3]; /* name, value, help */
	if(pardef == NULL) return;
	while(*pardef != NULL)
	{
		key[n] = *pardef;
		if(++n == 3)
		{
			define(group, key[0], key[1], key[2]);
			n = 0;
		}
		++pardef;
	}
	if(n != 0)	SERROR("parameter list size was no multiple of 3 - programmer's fault?");
}

void param_space::define_groups(const char **pardef)
{
	size_t n = 0;
	const char *key[4]; /* group, name, value, help */
	if(pardef == NULL) return;
	while(*pardef != NULL)
	{
		key[n] = *pardef;
		if(++n == 4)
		{
			define(key[0], key[1], key[2], key[3]);
			n = 0;
		}
		++pardef;
	}
	if(n != 0)	SERROR("parameter list size was no multiple of 3 - programmer's fault?");
}

bool param_space::get(const string group, const string name, string& value)
{
	parm* p = NULL;
	lock();
	int error = get_parm_struct(group,name,p);
	if(error == 0) value = p->value;
	unlock();
	return error == 0;
}

bool param_space::set(const string group, const string name, const string value)
{
	parm* p = NULL;
	lock();
	int error = get_parm_struct(group,name,p);
	if(error == 0) p->value = value;
	unlock();
	return error == 0;
}

void param_space::help(FILE *out)
{
	size_t spacelen = maxlen > maxshort ? maxlen : maxshort;
	char* spaces = new char[spacelen+1];
	memset(spaces, ' ', spacelen);
	spaces[spacelen] = 0;

	if(usage != "") fprintf(out, "%s\n\n", usage.c_str());

	fprintf(out, "\nParameter input: group.name=value; you can omit group \"main\"\n");
	fprintf(out, "\"%s\" stops parameter parsing (for file names or similar following)\n", separator.c_str());
	fprintf(out, "Example: program name=value main.another=anothervalue -- file1 file2\n\n");
	fprintf(out, "The parameters:\n");
	lock();
	for(vector<parm_group*>::iterator g = pgv.begin(); g != pgv.end(); ++g)
	for(vector<parm*>::iterator p = (*g)->parms.begin(); p != (*g)->parms.end(); ++p)
	{
		fprintf(out, "%s.%s:%s%s\t[%s]\n", (*g)->name.c_str(), (*p)->name.c_str(),
			spaces + spacelen - maxlen + (*g)->name.length() + (*p)->name.length(),
			(*p)->help.c_str(), (*p)->value.c_str());
	}
	fprintf(out, "\nAvailable shortcuts:\n");
	for(std::map<string, string>::iterator i = shortcuts.begin(); i != shortcuts.end(); ++i)
	fprintf(out, "\t%s%s%s\n", i->first.c_str(), spaces + spacelen - maxshort + i->first.length(), i->second.c_str());

	unlock();
	delete[] spaces;
}

void param_space::shortcut(const string short_one, const string long_one)
{
	unsigned int nl = short_one.length() + 1;
	if(nl > maxshort) maxshort = nl;

	shortcuts[short_one] = long_one;
}

bool param_space::write(FILE *out)
{
	fprintf(out, "# configuration file for %s\n", program != "" ? program.c_str() : "some program");
	fprintf(out, "# Syntax is strictly group.name=value, one per line. No white space!\n");
	for(vector<parm_group*>::iterator g = pgv.begin(); g != pgv.end(); ++g)
	{
		if((*g)->name == "param") continue;
		for(vector<parm*>::iterator p = (*g)->parms.begin(); p != (*g)->parms.end(); ++p)
		{
			if((*g)->name == "main" && (*p)->name == "help") continue;

			fprintf(out, "\n# %s\n", (*p)->help.c_str());
			fprintf(out, "%s.%s=%s\n", (*g)->name.c_str(), (*p)->name.c_str(), (*p)->value.c_str());
		}
	}
	// Could catch error...
	return true;
}

// like the GNU libc extension
size_t param_getline(FILE *in, char *&out, size_t &size)
{
	if(size == 0 || out==NULL)
	{
		size = 10;
		out = (char*)malloc(size); // Increase that later.
		if(out == NULL)
		{
			SERROR("D'OOM!");
			return 0;
		}
	}
	size_t fill = 0; // length of the string, without \0
	while(1)
	{
		MDEBUG("size %zu, fill %zu\n", size, fill);
		if(fgets(out+fill, size-fill, in) == NULL)
		return fill;

		fill = strlen(out);
		MDEBUG("read (partial) line of length %zu: %s\n", fill, out);

		// Check if we really got a line or need to increase the buffer and continue.
		// The last byte is always \0, before that we want a \n or \r.
		if(out[fill-1] == '\n' || out[fill-1] == '\r') // a half \r\n sequence could be there...
		return fill;

		SDEBUG("Line not fully there, increasing buffer.");
		// Else, realloc to get more data in.
		if(size > ((unsigned int)-1)/4)
		{
			SERROR("Maximum line length reached!");
			return 0;
		}
		char *tmp = (char*)realloc(out, 2*size);
		if(tmp == NULL){	SERROR("D'OOM!"); return 0; }

		out = tmp;
		size *= 2;
	}
}

bool param_space::read(FILE *in)
{
	char *work = NULL;
	size_t ws = 0;
	bool ret = true;
	size_t len;
	while( (len = param_getline(in, work, ws)) > 0 )
	{
		if(len < 3) continue; // just \r\n or \n or \r ...

		if(work[0] == '#') continue;

		// Cut off the line end.
		for(size_t pos = len-1; pos > 0; --pos)
		if(work[pos] == '\n' || work[pos] == '\r') work[pos] = 0;

		if(!parse(work))
		{
			MERROR("Cannot parse: %s\n", work);
			ret = false;
			goto psr_end;
		}
	}
psr_end:
	free(work);
	return ret;
}

bool param_space::parse(const string narg)
{
	unsigned int i = 0;
	unsigned int j = 0;
	bool ret = false;
	string group = "";
	string name = "";
	string value = "";
	const string* arg = &narg;
	// I'd like to just call it recursively with the shortcut expansion, but that has the possibility to enter endless loops with some smart ass wiring up recursive shortcuts.
	if(shortcuts.find(narg) != shortcuts.end())
	{
		// I could have stored the iterator from the test above but I cannot be arsed into the tedious syntactic clutter of writing the iterator's name.
		// Heck, I even wrote a much longer comment instead.
		// With C++11 I could use "auto".
		arg = &(shortcuts[narg]);
	}
	for(i = 0; i < arg->length(); ++i)
	{
		if(j == 0)
		{
			// arg->[i] does not work ... bah, C++ is dumb
			if((*arg)[i] == '.')
			{
				if(i == 0) group = "main";
				else group = arg->substr(0,i);
				j = i+1; //just after the dot
			}
		}
		
		if((*arg)[i] == '=')
		{
			if(i != j)
			{
				name = arg->substr(j,i-j);
				value = arg->length() > (i+1) ? arg->substr(i+1) : "";
			}
			if(j == 0) group = "main";
			break;
		}
	}
	if(group.empty() || name.empty()) MERROR("error parsing %s", arg->c_str());
	else
	{
		parm* p = NULL;
		lock();
		int error = get_parm_struct(group,name,p);
		if(error == 0) { p->value = value; ret = true; }
		unlock();
	}
	return ret;
}

bool param_space::parse(int &argc, char **&argv, const bool parse_program)
{
	if(parse_program && argc > 0)
	{
		program = argv[0];
		--argc;
		++argv;
	}
	int oldc = argc;
	for(int c = 0; c < oldc; ++c)
	{
		string w = argv[0];
		MDEBUG("parsing arg %i: %s", c, w.c_str());
		if(w == separator) // skip separator and stop
		{
			++argv; --argc;
			break;
		}
		// No "=", no shortcut match -> end.
		if(w.find("=") == string::npos && shortcuts.find(w) == shortcuts.end())
		break;

		// now try to parse
		if(!parse(w)) return false;
		// we're done with this one
		++argv; --argc;
	}
	return true;
}

bool param_space::as_bool(const string group, const string name)
{
	const char *truism[] = { "true", "on", "y", "yes", "ya", "j", "ja", "please", "yup", "yeah", "Alrighty then!", NULL };
	string val;
	if(get(group,name,val))
	{
		// any non-null number is true
		if(as_double(group, name) != 0) return true;
		// any entry of the positive list is true
		int t = -1;
		while(truism[++t] != NULL)
		if( strcasecmp(val.c_str(), truism[t]) == 0 ) return true;
	}
	return false;
}

unsigned int param_space::as_uint(const string group, const string name)
{
	string val;
	int temp = 0;
	if(get(group,name,val)) temp = atoi(val.c_str());
	if(temp > 0) return (unsigned int) temp;
	else return 0;
}

int param_space::as_int(const string group, const string name)
{
	string val;
	if(get(group,name,val)) return atoi(val.c_str());
	else return 0;
}

long int param_space::as_lint(const string group, const string name)
{
	string val;
	if(get(group,name,val)) return atol(val.c_str());
	else return 0;
}

unsigned long int param_space::as_ulint(const string group, const string name)
{
	string val;
	long int temp = 0;
	if(get(group,name,val)) temp = atol(val.c_str());
	if(temp > 0) return (unsigned long int) temp;
	else return 0;
}

size_t param_space::as_size(const string group, const string name)
{
	// This is paranoid, but _has_ long to be <= size_t?
	unsigned long tmp = as_ulint(group, name);
	if(tmp <= ((size_t)-1)) return (size_t) tmp;
	else return 0;
}

double param_space::as_double(const string group, const string name)
{
	string val;
	if(get(group,name,val)) return atof(val.c_str());
	else return 0;
}

bool param_space::is_equal(const string group, const string name, const string value)
{
	string val;
	if(get(group,name,val)) return val == value;
	else return false;
}

const char *param_space::c_str(const string group, const string name)
{
	parm* p;
	const char* val = "";  // could return NULL ... am I that strict?
	lock();
	int error = get_parm_struct(group,name,p);
	if(error == 0) val = p->value.c_str();
	unlock();
	return val;
}

//implementation of sub types

parm::parm(const string &n, const string &h, const string &v)
{
	name = n;
	help = h;
	value = v;
}

parm::parm()
{
	CONSTRUCT("parm");
}

parm::~parm()
{
	DESTRUCT("parm");
}

parm_group::parm_group(const string &n)
{
	CONSTRUCT("parm_group");
	name = n;
}

parm_group::~parm_group()
{
	DESTBEGIN("parm_group");
	for(vector<parm*>::iterator i = parms.begin(); i != parms.end(); ++i)
	delete (*i); 	
	parms.clear();
	DESTEND("parm_group");
}

parm* parm_group::get(const string &n)
{
	for(vector<parm*>::iterator i = parms.begin(); i != parms.end(); ++i)
	{
		if((*i)->name == n) return *i;
	}
	return NULL;
}
