/*
	outchannel: mixer output channels

	part of DerMixD
	(c)2004-2011 Thomas Orgis, licensed under GPLv2
*/

#include "basics.hxx"
#include <strings.h>
#include "audio/audio_buffer.hxx"

#include "action.hxx"
#include "mixer/mixer_data.hxx"
#include "shortcuts.hxx"

#include "param_init.hxx"
#include "tstring.hxx"

#include <string>
#include <vector>

//#define DEBUG_SCRIPT
//#define DEBUG_MIXER2
//#define DEBUG_RESET

#include "debug.hxx"
#define ME "outchannel"

using std::string;
using std::vector;

namespace dmd {

outchannel::outchannel(size_t id_, const string& name_, semaphore &playsem_, dmd::actionlist &mixact):
	channel(id_, name_), worker(mixact), procor(worker, buffer, playsem_)
{
	CONSTRUCT("outchannel");
	size_t bufs = param.as_size("buffer");
	buffer.reallocate(param.as_uint("channels"), bufs);
	if(buffer.size != bufs)
	{
		MERROR("[outch %zu] Cannot allocate mixer buffer.", id);
		status = DEAD;
	}
	else if(create_thread(&worker) && create_thread(&procor))
	{
		MDEBUG("[outch %zu] started worker thread %p and processor thread", id, &worker, &procor);
		status = IDLE;
	}
	else
	{
		MERROR("[outch %zu] Failed to create the output worker thread!", id);
		status = DEAD;
	}
}

// Nothing special to destruct... the thread will be co
outchannel::~outchannel()
{
	DESTBEGIN("outchannel");
	kill_threads();
	delscript();
	DESTEND("outchannel");
}

void outchannel::play()
{
	// One might want to increment the position... but it's not possible here anymore.
	//state.position += outbuffer->fill;
	procor.cycle();
}

void outchannel::statline(std::string &ts)
{
	strprintf(ts, "out %zu:", id);
	ts += name + " <- (";
	if(!sources.empty())
	{
		strprintf(ts, "%zu", sources[0]->id);
		for(vector<inchannel*>::iterator ini = sources.begin()+1; ini != sources.end(); ++ini)
		strprintf(ts, ",%zu", (*ini)->id);
	}
	ts += ") ";
	ts += state_text();
	if( !script.empty() ) strprintf(ts, ", %zu actions", script.size());
	if(state.type != output::invalid)
	{
		ts += " dev: ";
		ts += output::id_to_name(state.type);
		ts += " file: ";
		ts += state.filename;
	}
}

// Bind an input channel to an output channel.
void outchannel::bind(inchannel* inch)
{
	// Only bind if not already bound...
	bool bound = false;
	FOR_VECTOR(inchannel*, sources, ini)
	if(*ini == inch){ bound = true; break; }

	if(!bound) sources.push_back(inch);
}

// Release the bond.
void outchannel::unbind(inchannel* inch)
{
	FOR_VECTOR(inchannel*, sources, ini)
	if(*ini == inch){ sources.erase(ini); break; }
}

// Output channels are fixed to the mixer output rate.
void outchannel::now_and_then(size_t chunksize, float &now, float &then)
{
	now  = state.format.seconds(state.position);
	then = state.format.seconds(state.position+chunksize);
}

// The grand central action dispatcher.
// Any real work an output channel does by itself is triggered from here.
// Well, apart from actually playing audio;-)

void outchannel::handle_action(action *act)
{
	if(act->def->id == api::NOTIFY)
	{
		sync_state();
		switch(act->subaction->def->id)
		{
			case api::OUT_LOAD:
				if(!act->have_errors()) status = PLAYING;
				else
				{
					status = IDLE; // NODEV ??
					act->push_error(new error(ME, dmd::err::LOADERR, "Channel load failed."));
				}
				MDEBUG("[outch %zu] load done", id);
			break;
			default:
			act->push_error(new error(ME, dmd::err::UNKNOWN_COMMAND, "Notification for _what_?"));
		}
		act->notify();
	}
	// Dropped the "working" flag. Actions are issued strictly serial, the response can be quickly outdated if someone else dropped by with interfering commands, but the flag did not prevent that anyway.
	else
	switch(act->def->id)
	{
		case api::OUT_EJECT:
			status = IDLE;
			worker.nodev(act);
		break;
		case api::OUT_LOAD:
			status = LOADING;
			worker.load(act);
		break;
		default:
		act->push_error(new error(ME, dmd::err::UNKNOWN_COMMAND, "Some unknown output command arrived."));
		act->notify();
	}
}

void outchannel::sync_state()
{
	state.sync(worker.state, 0);
}

// **********************
// Cruft that will vanish

void outchannel::set_id(size_t newid)
{
	id = newid;
}

}
