/*
	output_worker: The actual work happening for output channels, in a thread.

	part of DerMixD
	(c)2010-2011 Thomas Orgis, licensed under GPLv2
*/


#include "basics.hxx"
#include "out/output_worker.hxx"
#include "action.hxx"
#include "coms/client_api.hxx"
#include "param_init.hxx"

#include "debug.hxx"

#define ME "output_worker"

namespace dmd
{

// The output actions are only used internally here.

// Temorary hack for settling the flags.
#define OUTWORK_FILE    INWORK_FILE
#define OUTWORK_NOTHING NOTHING

namespace api
{
	const def outwork_nothing = { OUTWORK_NOTHING, "<internal outwork nothing>", "", 0, { }, { }, 0 };
	const def outwork_play    = { OUTWORK_PLAY,    "<internal outwork play>", "", 0, { }, { OUTWORK_FILE }, 0 };
	const def outwork_close   = { OUTWORK_CLOSE,   "<internal outwork close>", "", 0, { }, { }, 0 };
	const def outwork_nodec   = { OUTWORK_NODEC,   "<internal outwork nodec>", "", 0, { }, { }, 0 };
}

output_worker::output_worker(dmd::actionlist &mixact): thread("output worker", nice::output_worker),
	outfile(NULL), worklist(), pool(), state(&worklist), mixer_actions(&mixact)
	
{
	CONSTRUCT(ME);
	pool.add(param.as_size("output", "buffers"));
}

output_worker::~output_worker()
{
	DESTBEGIN(ME);

	thread_kill();
	if(outfile != NULL) delete outfile;

	DESTEND(ME);
}

// Take a prepared buffer, push to the list of buffers to write out.
// The action for the worker is just a notification.
void output_worker::play(audio_buffer *buf)
{
	pool.store_used(buf);
	action *chore = new action(NULL, &api::outwork_play);
	chore->comdat = NULL;
	chore->disposable = true;
	worklist.push(chore);
}

audio_buffer* output_worker::next_buffer()
{
	return pool.fetch_avail();
}

void output_worker::load(action *act)
{
	worklist.push(act);
}

void output_worker::eject(action *act)
{
	action *chore = new action(NULL, &api::outwork_close);
	chore->comdat = act->comdat;
	chore->disposable = true;
	worklist.push(chore);
}

void output_worker::nodev(action *act)
{
	action *chore = new action(NULL, &api::outwork_nodec);
	chore->comdat = act->comdat;
	chore->disposable = true;
	worklist.push(chore);
}

// Code sharing with the input worker? It's very similar.
void output_worker::thread_work()
{
	MDEBUG("[output_worker %p] startup", this);
	while(1)
	{
		MDEBUG("[output_worker %p] waiting", this);
		action *act = worklist.pull(this);
		if(act == NULL || act->def == NULL)
		{
			SERROR("Pulled a null action! Should this happen?");
			continue;
		}

		bool dispose = act->disposable;
		work_process_action(act);
		if(dispose) delete act;
	}
	MDEBUG("[output_worker %p] leaving", this);
}

void output_worker::work_process_action(action *act)
{
	locker room(act->comdat);
	int aid = act->def->id;
	MDEBUG("[output_worker %p] processing action %i (%s)", this, aid, act->def->name);

	// Ensure that we got the action parameters we expect.
	if(!act->valid())
	{
		MDEBUG("[output_worker %p] Action is invalid!", this);
		act->push_error(new error(ME, dmd::err::BAD_PAR, "This action does not carry the parameters it requires (internal programming error?)."));
	}
	// Common check for actions that assume that there is an existing outfile.
	else if(act->def->flags & api::OUTWORK_FILE && (outfile == NULL))
	act->push_error(new error(ME, dmd::err::NO_DEVICE, "Cannot execute this action without a loaded input file."));
	else
	switch(aid)
	{
		case api::OUTWORK_PLAY:
			do_play();
		break;
		case api::OUT_LOAD:
			do_open(act);
		break;
		case api::OUTWORK_CLOSE:
			do_close(act);
		break;
		case api::OUTWORK_NODEC:
			do_nodev(act);
		break;
		default:
			// Well, this is a programming error.
			MERROR("[output_worker %p] What to do for action %i?", this, aid);
	}

	if(act->def->flags & api::SOCK_NOTIFY)
	{
		MDEBUG("[output_worker %p] lnotifying completion of action %p (id %i)", this, act, act->def->id);
		action *notify_act = api::make_notify_act(act);
		mixer_actions->push(notify_act);
	}

	act->notify();
}


// After being notified of new stuff for playback, pull the new buffer and write it out.
// The buffer is then free again for more mixing work.
void output_worker::do_play()
{
	audio_buffer *buf = pool.fetch_used(this);
	MXDEBUG("[output_worker %p] playing buffer of %zu/%zu samples", this, buf->fill, buf->size);
	outfile->buf_write(*buf);
	state.increment_position(buf->fill);
	pool.store_avail(buf);
}

void output_worker::do_nodev(action *act)
{
	if(outfile != NULL)
	{
		delete outfile;
		outfile = NULL;
	}
}

void output_worker::do_open(action *act)
{
	std::string type     = act->strings[0];
	std::string resource = act->strings[1];

	if(type == "default")
	{
		type = param("default_output");
		resource = param("default_outfile");
	}

	if(!output::switch_file(outfile, output::name_to_id(type)))
	{
		act->push_error(new error(ME, dmd::err::BAD_PAR, "Bad output type " + type + " given (cannot change output file type)."));
		state.newfile(output::invalid); // No valid output file anymore.
		return;
	}

	if(!outfile->do_open_safe(resource))
	{
		act->push_error(new error(ME, dmd::err::IO, "Cannot open resource " + resource + "."));
		state.newfile(outfile->type); // Just change to the new type, without loaded file.
		return;
	}

	// Everything is good: Valid file with actually loaded resource.
	state.newfile(outfile->type, resource);
}

void output_worker::do_close(action *act)
{
	if(outfile != NULL)
	{
		outfile->do_close();
		state.newfile(outfile->type);
	}
}


}
