/*
	out_alsa: alsa output device
	
	part of DerMixD
	(c)2005-2012 Thomas Orgis, licensed under GPLv2
*/

#include "basics.hxx"
#include <alsa/asoundlib.h>

#include "out/drv/out_alsa.hxx"
#include "param_init.hxx"

using std::string;

#include "debug.hxx"
#define ME "alsa_output"

// All ALSA stuff in here.
struct alsa_output::my_private
{
	snd_pcm_hw_params_t *hwparams;
	snd_pcm_t *handle;
	snd_pcm_stream_t stream;
};


alsa_output::alsa_output(): output_file(output::alsa), parts(NULL)
{
	CONSTRUCT(ME);
	parts = new struct my_private;

	snd_pcm_hw_params_malloc(&parts->hwparams);
	parts->stream = SND_PCM_STREAM_PLAYBACK;
	parts->handle = NULL;
}

alsa_output::~alsa_output()
{
	DESTBEGIN(ME);
	do_close();
	snd_pcm_hw_params_free(parts->hwparams);
	delete parts;
	DESTEND(ME);
}

bool alsa_output::do_write(const void *abuf, size_t bytes)
{
	int count = format.bytes2samples(bytes);
	const audio_io_type *buf = (const audio_io_type*)abuf;

	while(count > 0)
	{
		MXDEBUG("["ME" %p] serial write loop with %i samples at %p", this, count, buf); 

		int ret = 0;
		while( (ret = snd_pcm_writei(parts->handle, buf, count)) < 0 && snd_pcm_recover(parts->handle, ret, 0) == 0 )
		{
			MERROR("["ME" %p] ALSA recovery!", this);
		}

		if(ret < 0)
		{
			err.occur(dmd::err::IO, std::string("Cannot write to ALSA output: ") + snd_strerror(ret));
			return false;
		}

		buf += ret * format.channels;
		count -= ret;
	}
	return true;
}

bool alsa_output::do_open(const std::string location)
{
	int errora = 0;
	bool succ = false;
	std::string problem;
	snd_pcm_uframes_t bufsize = (snd_pcm_uframes_t) (param.as_double("alsa","buffer")*format.rate);
	snd_pcm_uframes_t perisize = bufsize / 4;

	// I still hate this deep if... but flattening would add more code...
	if((errora = snd_pcm_open(&parts->handle, location != "" ? location.c_str() : "default", parts->stream, 0)) < 0)
	problem = "device open failed";
	else if((errora = snd_pcm_hw_params_any(parts->handle, parts->hwparams)) < 0)
	problem = "param init failed";
	else if((errora = snd_pcm_hw_params_set_access(parts->handle, parts->hwparams, SND_PCM_ACCESS_RW_INTERLEAVED)) < 0)
	problem = "access setting failed";
	else if((errora = snd_pcm_hw_params_set_format(parts->handle, parts->hwparams, SND_PCM_FORMAT_S16)) < 0)
	problem = "format setting failed";
	else if((errora = snd_pcm_hw_params_set_rate(parts->handle, parts->hwparams, format.rate, 0)) < 0)
	problem = "rate setting failed";
	else if((errora = snd_pcm_hw_params_set_channels(parts->handle, parts->hwparams, (unsigned int)format.channels)) < 0)
	problem = "channel setting failed";
	else if((errora = snd_pcm_hw_params_set_buffer_size_near(parts->handle, parts->hwparams, &bufsize)) < 0)
	problem = "buffer size setting failed";
	else if((errora = snd_pcm_hw_params_set_period_size_near(parts->handle, parts->hwparams, &perisize, NULL)) < 0)
	problem = "periods setting failed";
	else if((errora = snd_pcm_hw_params(parts->handle, parts->hwparams)) < 0)
	problem = "params setting failed";
	else if((errora = snd_pcm_prepare(parts->handle)) < 0) //|| ((errora = snd_pcm_start(parts->handle)) < 0))
	problem = "prepare failed";
	else succ = true;

	MDEBUG("["ME" %p] activate done; error value %i", this, errora);

	if(!succ)
	{
		err.occur(dmd::err::IO, problem);
		do_close();
	}

	return succ;
}

void alsa_output::do_close()
{
	MDEBUG("["ME" %p] closing", this);
	if(parts->handle == NULL) return;

	snd_pcm_drain(parts->handle);
	snd_pcm_close(parts->handle);
	parts->handle = NULL;
}
