#ifndef DMD_H_NUMPARAM
#define DMD_H_NUMPARAM

/*
	numparam: a simplified parameter class for an ordered list of floating point values
	
	part of DerMixD
	(c)2010 Thomas Orgis, licensed under GPLv2
	
	This might be rather superfluous; perhaps the param class is better to be adapted for this purpose. I am doing this as a prototype to be used for audio filter parameters.
	It might occur in future that this spin-off will be reintegrated into the main class again. For now the big hierarchical parameter space just looks like overkill -- especially since it includes scary stuff like a mutex.
*/

#include <string>
#include <vector>

class numparm
{
	public:
	float value;
	std::string name;
	std::string help;
	bool changed; // Flag to notify if the value changed recently.
	float bounds[2]; // Lower/upper boundary for sensible values.
	bool bounded[2]; // Flags to tell if lower/upper bound actually applies.
	// The default unscaled value.
	float defval;
	// A scale to apply when setting the value, used as divisor when displaying.
	float scale;

	numparm(const std::string n, float v, const std::string h);
	~numparm();
	// We also hide integer and boolean values.
	int as_longint(); // Rounded value.
	bool as_bool(); // Rounded value is != 0.
	// Change the scale factor, convert current value to reflect that change.
	// I.e., old value is 50 with scale of 100, new scale is 80
	// -> new value is 50/100*80=40
	// This refuses too small scales.
	bool change_scale(float nscale);
	// Get external value ... meaning: without scaling.
	float get_ext(){ return value/scale; }
	// Set new value, handle scaling internally.
	// Honour the bounds or not? I choose to do so.
	void set(float newval);
};

// You get this one when making an invalid request.
// It's also a good idea not to re-use its name.
extern numparm bad_numparm;


class numparm_group
{
	public:

	std::string name; // Of the corresponding entity.
	std::string help; // General info about what we have parameters for.
	std::vector<numparm> parms;
	bool changed; // Flag to notify any change in values.

	numparm_group(const std::string n, const std::string h);
	~numparm_group();

	// Adding a parameter definition. This is not allowed to fail (I don't really handle out of memory situations).
	// Returns index.
	size_t define(const std::string name, float value, const std::string help);

	// Access by index and name.
	numparm& operator[](size_t i);
	numparm& operator()(const std::string n);
	// Parse string, modify values.
	// String is either space-separated list of numerical values or of name=value pairs.
	// Fun for Future: Allow operators for increment/decrement/multiplication.
	// Returns false on invalid strings ... changes up to the failing position will be applied.
	// The only error is wrong given name; you can always provide more or less numbers than needed.
	// Special: name=  means reset to default.
	bool parse(const std::string parline);
	// Convert current values to string, name=value pairs.
	void namestring(std::string &parline);
	// Just list of numbers.
	void plainstring(std::string &parline);
	// Append a line of help/explanation for each parameter.
	void helptext(std::vector<std::string*> &parlines);
	// Forget all definitons (prepare for new round).
	// Yeah, one could just destruct and create anew.
	void reset(const std::string newname, const std::string newhelp);

	private:
	bool apply(const std::string token, size_t &numcount); // Apply a single setting.
};

#endif
