/*
	numparam: a simplified parameter class for an ordered list of floating point values
	
	part of DerMixD
	(c)2010 Thomas Orgis, licensed under GPLv2
*/

//not depending on dermixd's common header, this stuff can easily tranfserred to other projects or into an independent library
#include <cstdlib>
// #include <cstdio>
#include <string>
#include <cstring>
#include <strings.h> // no cstrings
#include <vector>

#include "numparam.hxx"
#include "tstring.hxx"
#include "shortcuts.hxx"
#include "mathhelp.hxx"

#include "debug.hxx"

using namespace std;

numparm bad_numparm("INVALID", 0., "This is an parameter that does not exist. You tried to get something you cannot get."); // Dummy parameter for invalid retrievals.

numparm::numparm(const std::string n, float v, const std::string h):
	 value(v), name(n), help(h), changed(false)
	,defval(0.), scale(1.)
{
	CONSTRUCT("numparm");
	bounded[0] = bounded[1] = false;
}

numparm::~numparm()
{
	DESTRUCT("numparm");
}

int numparm::as_longint()
{
	return round2long(value);
}

bool numparm::as_bool()
{
	return (as_longint() != 0);
}

bool numparm::change_scale(float nscale)
{
	if(nscale < 1.e-6) return false;

	MDEBUG("changing scale of %s: old value %g, scale %g", name.c_str(), value, scale);
	value /= scale;
	value *= nscale;
	scale  = nscale;
	MDEBUG("changing scale of %s: new value %g, scale %g", name.c_str(), value, scale);
	changed = true;
	return true;
}

void numparm::set(float newval)
{
	MDEBUG("set value of %s to %g", name.c_str(), newval);
	if(bounded[0] && newval < bounds[0]) newval = bounds[0];
	if(bounded[1] && newval > bounds[1]) newval = bounds[1];

	MDEBUG("set value of %s to %g (after bounds)", name.c_str(), newval);
	value = newval*scale;
	MDEBUG("set value of %s to %g (final with scale)", name.c_str(), value);
	changed = true;
}

numparm_group::numparm_group(const string n, const string h): name(n), help(h), changed(false)
{
	CONSTRUCT("numparm_group");
}

numparm_group::~numparm_group()
{
	DESTRUCT("numparm_group");
}

void numparm_group::reset(const std::string newname, const std::string newhelp)
{
	parms.clear();
	name = newname;
	help = newhelp;
	changed = false;
}

numparm& numparm_group::operator[](size_t i)
{
	if(i >= parms.size()) return bad_numparm;

	return parms[i];
}

numparm& numparm_group::operator()(const string name)
{
	FOR_VECTOR(numparm, parms, pi)
	{
		if(pi->name == name) return *pi;
	}
	return bad_numparm;
}

size_t numparm_group::define(const string name, float value, const string help)
{
	// This is not allowed to fail ... Who can take care of OOM, really?
	parms.push_back(numparm(name, value, help));
	return parms.size()-1;
}

void numparm_group::helptext(vector<string*> &parlines)
{
	parlines.push_back(new string(name + ": " + help));

	size_t i = 0;
	FOR_VECTOR(numparm, parms, pi)
	{
		string *line = new string();
		strprintf(*line, "%zu %s=%g",i++, pi->name.c_str(), pi->get_ext());
		if(pi->bounded[0]) strprintf(*line, " >=%g", pi->bounds[0]);
		if(pi->bounded[1]) strprintf(*line, " <=%g", pi->bounds[1]);
		strprintf(*line, "; %s", pi->help.c_str());
		parlines.push_back(line);
	}
}

void numparm_group::namestring(string &parline)
{
	bool first = true;
	FOR_VECTOR(numparm, parms, pi)
	{
		if(first) first=false;
		else parline += " ";

		strprintf(parline, "%s=%g", pi->name.c_str(), pi->get_ext());
	}
}

void numparm_group::plainstring(string &parline)
{
	bool first = true;
	FOR_VECTOR(numparm, parms, pi)
	{
		if(first) first=false;
		else parline += " ";

		strprintf(parline, "%g", pi->get_ext());
	}
}

// name=value or plain number, or just name=
// Future: operators, too.
bool numparm_group::apply(const string token, size_t &numcount)
{
	MDEBUG("parsing token %s; numcount=%zu", token.c_str(), numcount);
	string::size_type sep = token.find_first_of("=");

	if(sep != string::npos)
	{
		MDEBUG("name-value pair, = at %zu", (size_t)sep);
		numparm *par = &((*this)(token.substr(0, sep)));
		if(par == &bad_numparm)
		{
			SDEBUG("Hit the bad parameter. Wrong name.");
			return false;
		}
		float val = par->defval;
		if((size_t)sep+1 < token.length())
		val = atof(token.substr(sep+1).c_str());

		par->set(val);
	}
	else
	{
		// This can happily hit the bad parameter. Silently ignored.
		MDEBUG("plain number: %g", atof(token.c_str()));
		(*this)[numcount++].set(atof(token.c_str()));
	}

	changed = true;
	return true;
}

bool numparm_group::parse(const string parline)
{
	// One could just use a tokenizer... but this one-pass parsing does the trick, too.
	// Skip delimiters at beginning.
	string::size_type lastPos = parline.find_first_not_of(" \t", 0);
	// Find first "non-delimiter".
	string::size_type pos = parline.find_first_of(" \t", lastPos);
	size_t parpos = 0;
	while(string::npos != pos || string::npos != lastPos) //while one of these position is valid...
	{
		// Found a token, work on it.
		if(!apply(parline.substr(lastPos, pos - lastPos), parpos)) return false;

		// Skip delimiters.  Note the "not_of"
		lastPos = parline.find_first_not_of(" \t", pos); //finds first non-delimiter... lastPos = start of new stuff or npos
		// Find next "non-delimiter"
		pos = parline.find_first_of(" \t", lastPos); //next delimiter after start
	}

	if(lastPos != string::npos)
	return apply(parline.substr(lastPos), parpos);

	return true;
}
