/*
	mixer: the mixer, the master, the gate, the keeper
	
	part of DerMixD
	(c)2004-2013 Thomas Orgis, licensed under GPLv2
*/

#include "basics.hxx"
#include "param_init.hxx"

#include "audio/audio_buffer.hxx"

#include "coms/socket_writer.hxx"
#include "action.hxx"
#include "coms/socketeer.hxx"
#include "coms/client_action_handler.hxx"
#include "sigfried.hxx"

#include "secsleep.hxx"
#include "tstring.hxx"

#include "mixer.hxx"

#ifdef LINUX
#include <sys/time.h>
#include <sys/resource.h>
#include <unistd.h>
#endif

#include "debug.hxx"
#include "shortcuts.hxx"

using std::vector;
using std::string;

#define ME "mixer"

#define GLOBAL_DEAD 2
#define GLOBAL_PAUSED 1
#define GLOBAL_RUNNING 0

namespace dmd
{

mixer::mixer(semaphore &clients_go):
	globalstate(GLOBAL_DEAD), jules(actions, clients_go), hermes()
{
	CONSTRUCT(ME);
}

mixer::~mixer()
{
	DESTBEGIN(ME);
	exit(); // Damn, should I really override this function name? Change it!
	DESTEND(ME);
}

bool mixer::init()
{
#ifdef LINUX
	base_pid      = getpid();
	base_priority = getpriority(PRIO_PROCESS, 0); // Could do error check via errno...
#endif

	format.rate = param.as_ulint("audio_rate");
	format.channels = param.as_uint("channels");
	buffer_samples = param.as_size("buffer");

	if(format.channels > 2)
	{
		MERROR("[mixer %p] Sorry, I can only do mono or stero, not %u channels.", this, format.channels);
		return false;
	}

	if(buffer_samples < 2)
	{
		MERROR("[mixer %p] Minimum buffer size is 2 (and you don't expect pitch shifting to work in that area, do you?)!", this);
		return false;
	}

	if(!create_thread(&hermes))
	{
		MERROR("[mixer %p] Cannot create message sink thread.", this);
		return false;
	}

	if(!create_thread(&jules))
	{
		MERROR("[mixer %p] Cannot create worker thread.", this);
		return false;
	}

	return true;
}

void mixer::exit()
{
	scribe bob(rack.the_wall);
	// In that order! Jules needs Hermes for work.
	kill_thread(&jules);
	kill_thread(&hermes);

	// Delete input devices.
	FOR_VECTOR(inchannel*, rack.inners, ini)
	{
		MDEBUG("[mixer %p] would like to delete inchannel %zu (%p)", this, (*ini)->id, *ini);
		delete (*ini);
		MDEBUG("[mixer %p] indeed deleted %p", this, *ini);
	}
	rack.inners.clear();

	MDEBUG("[mixer %p] cleared inners", this);

	// Delete output devices.
	FOR_VECTOR(outchannel*, rack.outers, outi)
	{
		MDEBUG("[mixer %p] would like to delete outchannel %zu (%p)", this, (*outi)->id, *outi);
		delete (*outi);
		MDEBUG("[mixer %p] indeed deleted %p", this, *outi);
	}
	rack.outers.clear();
	MDEBUG("[mixer %p] cleared outers", this);
}

bool mixer::run()
{
	globalstate = GLOBAL_RUNNING;
	while(globalstate == GLOBAL_RUNNING)
	{
		//stage1: admin, timed scripting now there!

		//only if actions are there
		// If this call is interrupted by a signal (thus not sucessful), the actions just will be a bit delayed.
		if(actions.got_work())
		{
			// TODO:  Grab actions, then release the list before processing
			// Other folks can continue pushing actions for the next turn, then.
			actions.pulling();
			MDEBUG("[mixer] executing actions %p (size: %zu)", &actions, actions.get_list().size());
// Let that stay for a while. Really should indicate broken code.
if(actions.get_list().size() < 1)
{
	SERROR("[mixer] Why I am here? Somebody woke me but there are no actions! This is breakage in the program.");
	//actions.done_pulling();
	//break;
}
			process_actions(actions.get_list());
			SDEBUG("[mixer] actions done");
			actions.done_pulling();
			SDEBUG("[mixer] lock released");
			// Handle death asap, without scripts.
			if(globalstate == GLOBAL_DEAD) break;
		}

		// Script actions can be due for this round.
		handle_script(buffer_samples);

		// Scripts might have changed the state... suspend, be dead, or just continue.
		if(globalstate == GLOBAL_PAUSED) wait_for_action();
		else if(globalstate == GLOBAL_DEAD) break; //could delete all inputs and outputs in actions instead...

		// Three stages:
		// 1. Fetch input.
		// 2. Mix into output buffers.
		// 3. Push buffers to output.
		// The first two are combined now, making the mixing happen in the input threads.
		dmd::playlock.lock(); // Make sure scanners don't interfere.

		unsigned int active_inputs = fetch_input_and_mix(buffer_samples);

		dmd::playlock.unlock(); // Now the input scanners can do I/O again...

		//stage4: activate output; double buffering being inherent now with threads and waiting for last chunk inside the device object

		/*
			Babble about open issues:

			All PLAYING outputs are played, even when there is no actual data there.
			The resulting silent output (with no music) keeps the device
			silently busy and prevents any on/off sounds that are not desired or 
			some nasty buffer from keeping a chunk of audio until playback starts again.
			Heard that. Not nice.

			When you want a device to really not be busy then set the channel to PAUSING
			(device still open but no output) or SUSPEND (keep channel around but close
			audio device).
			
			With the new output_device-level multithreading it should be possible for the one output needing silence to push pending audio really out before pauses - OSS - to issue the writing of some silent buffers (post the semaphore multiple times; but be careful with the play() side of waiting)
		*/

		unsigned int active_outputs = dispatch_output();

		//some precautions for unemployment
		if(active_outputs == 0)
		{
			if(active_inputs == 0)
			{
				//really nothing to do: no input, no output
				//wait for some action
				//socketeers trigger semac on every mixer command
				SDEBUG("[mixer] waiting for some action...");
				actions.idle();
				SDEBUG("[mixer] got some action...");
			}
			else
			{
				//no output, but inputs running... that means decode/read stuff as fast as possible into the void or wait a bit here to allow user to react to this quite senseless state without struggling with a very busy cpu...
				//the delay time 
				SXDEBUG("[mixer] just sleeping a bit...");
				//usleep((useconds_t) ((float) buffer_samples / audio_rate * 900000));
				secsleep((double) buffer_samples / format.rate * 0.9);
			}
		}
	} // Le big end.

	return true;
}

// All those funny action handlers... strictly internal.

// Produce overall status info on request... timely updates are better served via the watch mechanism.
void mixer::set_stats(vector<string*> &stats)
{
	scholar sam(rack.the_wall);
	std::string ts;

	rack.the_wall.reading();
	FOR_VECTOR(inchannel*, rack.inners, ini)
	{
		(*ini)->statline(ts);
		stats.push_back(new string(ts));
		ts = "";
	}

	FOR_VECTOR(outchannel*, rack.outers, outi)
	{
		(*outi)->statline(ts);
		stats.push_back(new string(ts));
		ts = "";
	}
	rack.the_wall.done_reading();
}

void mixer::add_input(action *act)
{
	scribe bob(rack.the_wall);
	MDEBUG("[mixer] creating inchannel %zu", rack.inners.size());
	inchannel *newbie = new inchannel(rack.inners.size(), act->strings[0], &insem, format, buffer_samples, actions, hermes.comdat);
	if(newbie->status == dmd::channel::DEAD)
	{
		MDEBUG("[mixer] creation of inchannel %zu failed", newbie->id);
		delete newbie;
		act->push_error(new error(ME, dmd::err::SETUP, "Cannot create input channel."));
	}
	else rack.inners.push_back(newbie);
}

void mixer::add_output(action *act)
{
	scribe bob(rack.the_wall);
	MDEBUG("[mixer] creating outchannel %zu", rack.outers.size());
	outchannel *newbie = new outchannel(rack.outers.size(), act->strings[0], insem, actions);
	if(newbie->status == dmd::channel::DEAD)
	{
		MDEBUG("[mixer] creation of outchannel %zu failed", newbie->id);
		delete newbie;
		act->push_error(new error(ME, dmd::err::SETUP, "Cannot create output channel."));
	}
	else rack.outers.push_back(newbie);
}

// It needs to be ensured that the input is not busy with anything ...
// the working flag is checked before calling this ... or is it not?
void mixer::remove_input(action *act)
{
	scribe bob(rack.the_wall);
	size_t num = act->sizes[0];
	if(num >= rack.inners.size())
	{
		act->push_error(new error(ME, dmd::err::BAD_CHANNEL, "Bad channel ID given."));
		return;
	} 
	if(   (rack.inners[num]->status == dmd::channel::LOADING)
	   || (rack.inners[num]->status == dmd::channel::SEEKING_PLAY)
	   || (rack.inners[num]->status == dmd::channel::SEEKING_PAUSE) )
	{
		act->push_error(new error(ME, dmd::err::BUSY, "Channel is busy, try removal later."));
		return;
	}

	// Sever all connections to other channels.
	rack.inners[num]->detach();

	// Set all following ids to match their future position.
	// U-G-L-L-Y!
	size_t num2 = num;
	for(vector<inchannel*>::iterator ini = rack.inners.begin() + num + 1; ini != rack.inners.end(); ++ini)
	{
		MDEBUG("[mixer] removing input %zu ... inchannel %zu is becoming %zu", num, (*ini)->id, num2);
		(*ini)->set_id(num2++);
	}
	//finally, remove it
	delete rack.inners[num];
	rack.inners.erase(rack.inners.begin() + num);
}

void mixer::remove_output(action* act)
{
	scribe bob(rack.the_wall);
	size_t num = act->sizes[0];
	MDEBUG("[mixer] removing outchannel %li", num);
	if(num < rack.outers.size())
	{
		outchannel *outch = rack.outers[num];
		if( rack.outers[num]->status == dmd::channel::LOADING )
		{
			act->push_error(new error(ME, dmd::err::BUSY, "Cannot remove output while it is busy loading."));
		}
		else
		{
			// Delete from all sources' targets.
			// That is the only reason for storing the connections in the output,
			// too. But one could also just loop over all inputs ...
			FOR_VECTOR(inchannel*, outch->sources, ini) (*ini)->unbind(outch);

			// Set all following ids to match their future position
			size_t num2 = num;
			for(vector<outchannel*>::iterator outi = rack.outers.begin() + num + 1; outi != rack.outers.end(); ++outi)
			{
				MDEBUG("[mixer] deleting output %zu... outchannel %zu is becoming %zu", num, (*outi)->id, num2);
				(*outi)->set_id(num2++);
			}
			// Finally, remove it.
			delete outch;
			rack.outers.erase(rack.outers.begin() + num);
		}
	}
	else
	act->push_error(new error(ME, dmd::err::BAD_CHANNEL, "No such output channel."));

	act->notify();
}

void mixer::exorcise_socket(socket_writer* swr)
{
	// We modify channels individually, so only want to have the grand
	// picture to look at.
	scholar sam(rack.the_wall);
	MDEBUG("[mixer] exorcising socket writer %p", swr);
	FOR_VECTOR(inchannel*, rack.inners, ini) (*ini)->unwatch(swr);
	// Need to check _all_ channels, as postscripts are covered by the scripters list.
	FOR_VECTOR(inchannel*, rack.inners, chi) (*chi)->script_nirvana(swr);
	// In theory, outchannels could have script actions, too...
//	FOR_VECTOR(outchannel*, rack.outers, chi) (*chi)->script_nirvana(swr);
}

static void notify_and_dispose(action *act)
{
	bool dispose = act->disposable;
	act->notify();
	if(dispose)
	{ // If it's disposable, we are allowed to access its contents here.
		MDEBUG("[mixer] disposing of action %p (%s)", act, act->def->name);
		delete act;
	}
}

// Let's play: master and servant.
void mixer::process_actions(vector<action*> &actlist)
{
	FOR_VECTOR(action*, actlist, k)
	{
		action *act = *k;
		if(act->def == NULL)
		{
			SERROR("[mixer] I got an action with NULL definition! That is serious breakage. Fix the thread code that issued that action!");
			globalstate = GLOBAL_DEAD;
			return;
		}
		MDEBUG("[mixer] action loop, act=%p def=%p (%s) comdat=%p %s", act, act->def, act->def->name, act->comdat, act->disposable ? "disposable" : "re-usable");
		// If the action has a backchannel, we use it for communication,
		// else we drop errors/responses to the mixer worker/messenger.
		// I have doubts about that! comm_data *backchannel = act->comdat != NULL ? act->comdat : &hermes.comdat;
		// Note: We're always waking _someone_ on action completion.

		// Not _really_ necessary to cache those here?
		unsigned int aid = act->def->id;
		int flags = act->def->flags;
		MDEBUG("[mixer] executing action type %u", aid);

		if(flags & api::SOCK_IN)
		{
			scholar sam(rack.the_wall); // Need to access channel list.
			// Every single input action carries the input channel ID as first size parameter.
			// In case of notifications, it's the subaction carrying the ID.
			size_t num = ((act)->def->id != api::NOTIFY) ? (act)->sizes[0] : (act)->subaction->sizes[0];
			if( num < rack.inners.size() )
			{
				inchannel *in = rack.inners[num];
				// Let the channel handle the action itself.
				// But there can be need for some administrative postprocessing.
				switch(in->handle_action(act, &hermes.comdat))
				{
					case inchannel::BLISS: break; // Just happy, nothing to do.
					case inchannel::NEW_SCRIPTER:
						scripters.push_back((channel*)in);
					break;
					case inchannel::NO_SCRIPTER:
						FOR_VECTOR(channel*, scripters, sci)
						{
							if((*sci) == (channel*) in){ scripters.erase(sci); break; }
						}
					break;
					default:
					SERROR("Unknown inchannel postaction code!");
				}
			}
			else
			{
				act->push_error(new error(ME, dmd::err::BAD_CHANNEL, "No such input channel"));
				notify_and_dispose(act);
			}
		}
		else if(flags & api::SOCK_OUT)
		{
			scholar sam(rack.the_wall); // Need to access channel list.
			size_t num = (aid != api::NOTIFY) ? act->sizes[0] : act->subaction->sizes[0];
			if( num < rack.outers.size() )
			{
				rack.outers[num]->handle_action(act);
			}
			else
			{
				act->push_error(new error(ME, dmd::err::BAD_CHANNEL, "No such output channel"));
				notify_and_dispose(act);
			}
		}
		else //general action
		{
			//could've been quit
			if(aid == api::QUIT)
			{
				globalstate = GLOBAL_DEAD;
				// Just break out... all socketeers will be cancelled while waiting for I/O or while waiting for their mixer response semaphore.
				break;
			}
			else handle_mixer_action(act);
		}
	}
}

void mixer::handle_mixer_action(action *act)
{
	locker room(act->comdat);

	switch(act->def->id)
	{
		// First thing to move to the worker thread!
		// Need to settle channel state handling first.
		case api::FULLSTAT:
			{
				//I used to have a bool to remember if I already have prepared the current stat_string, but how often will unnecessary double creation happen to justify this additional bloat of creating and initializing one bool...
				SXDEBUG("[mixer] giving stats");
				// Stats are added to the return lines.
				if(act->comdat != NULL)
				set_stats(act->comdat->retlines);
			}
		break;
		case api::THREADSTAT:
		{
			scholar sam(rack.the_wall);
			// collect thread info from all input and output channels
			// You have to add the socket threads and "[threadstat] -end" later - I don't want to do the socketeer locking here.
			unsigned long count = 1; // mixer is one thread.

			act->comdat->retlines.push_back(new std::string());
#ifdef LINUX
			strprintf(*act->comdat->retlines.back(), "%i(%i @ %i): miXer", 0, (int)base_pid, base_priority);
#else
			strprintf(*act->comdat->retlines.back(), "%i: miXer", 0);
#endif
			count += sigfried::thread_info(act->comdat->retlines);
			count += thread_info(act->comdat->retlines);
			FOR_VECTOR(inchannel*, rack.inners, i)
			count += (*i)->thread_info(act->comdat->retlines);

			FOR_VECTOR(outchannel*, rack.outers, i)
			count += (*i)->thread_info(act->comdat->retlines);

			act->comdat->retval_ulong = count;
		}
		break;

		case api::SLEEP:
			globalstate = GLOBAL_PAUSED;
		break;

		case api::ADD_IN:
			add_input(act);
		break;
		case api::ADD_OUT:
			add_output(act);
		break;
		case api::REM_IN:
			remove_input(act);
		break;
		case api::REM_OUT:
			remove_output(act);
		break;

		case api::BIND:
		{
			scholar sam(rack.the_wall);
			if(act->sizes[0] < rack.inners.size() && act->sizes[1] < rack.outers.size())
			{
				rack.inners[act->sizes[0]]->bind(rack.outers[act->sizes[1]]);
				rack.outers[act->sizes[1]]->bind(rack.inners[act->sizes[0]]);
			}
			else act->push_error(new error(ME, dmd::err::BAD_CHANNEL, "Bad channel ID."));
		}
		break;

		case api::UNBIND:
		{
			scholar sam(rack.the_wall);
			if(act->sizes[0] < rack.inners.size() && act->sizes[1] < rack.outers.size())
			{
				rack.inners[act->sizes[0]]->unbind(rack.outers[act->sizes[1]]);
				rack.outers[act->sizes[1]]->unbind(rack.inners[act->sizes[0]]);
			}
			else act->push_error(new error(ME, dmd::err::BAD_CHANNEL, "Bad channel ID."));
		}
		break;

		// This input action is here because I need to look up two input channels.
		// The reverse action works without lookup and is thus contained in the inchannel class.
		case api::FOLLOW:
		{
			scholar sam(rack.the_wall);
			if(act->sizes[0] < rack.inners.size() && act->sizes[1] < rack.inners.size())
			rack.inners[act->sizes[0]]->set_follow(act, rack.inners[act->sizes[1]]);
			else
			act->push_error(new error(ME, dmd::err::BAD_CHANNEL, "No such input channel for following."));
		}
		break;

		case api::SAY:
			//suboptimal... should use list of pointers and let swr delete
			MDEBUG("[mixer] saying %s on socket_writer %p",act->strings[0].c_str(), (void*) act->swr);
			if(act->swr != NULL) act->swr->write_copy("[saying] " + act->strings[0]);
			else MERROR("[mixer %p] homeless message: %s", this, act->strings[0].c_str());
		break;
		case api::CLOSE:
			//no talking to the dead...
			exorcise_socket(act->swr);
		break;
		default:
		act->push_error(new error(ME, dmd::err::UNKNOWN_COMMAND, "Some unknown general command arrived."));
	}
	SXDEBUG("[mixer] end general switch");

	// After a general action, we always notify here.
	notify_and_dispose(act);
}

// Check for script actions, execute them.
void mixer::handle_script(size_t buffer_samples)
{
	vector<action*> script;
	// This is a dumb, slow approach to scripting... it would be better to store in how many samples the next script action is due instead of asking each channel for each buffer "Do you have actions?".
	FOR_VECTOR(channel*, scripters, c)
	{
		MXDEBUG("[mixer] ch %zu:%s is a scripter with status %s", (*c)->id, (*c)->name.c_str(), (*c)->state_text());
		if((*c)->status == channel::PLAYING)
		{
			//getactions fills any current actions into script and returns true if there are still actions left
			if(! (*c)->getactions(buffer_samples,script))
			{
				scripters.erase(c);
				--c;
			}
		}
	}
	if(!script.empty()) //found an action
	{
		SDEBUG("[mixer] executing script actions");
		// Process the actions... and delete them automatically (they're disposable).
		process_actions(script);
	}
}

void mixer::wait_for_action()
{
	//take a rest and wait for the doorbell
	SDEBUG("[mixer] sleeping and waiting for some action (devices still open!)...");
	actions.idle();
	SDEBUG("[mixer] good morning - got some action...");
	globalstate = GLOBAL_RUNNING;
}

// Makes the channels produce data and mix it into the outputs, in parallel.
// Returning the number of input channels that are on air.
unsigned int mixer::fetch_input_and_mix(size_t buffer_samples)
{
	scholar sam(rack.the_wall);
	unsigned int active_inputs = 0;
	FOR_VECTOR(inchannel*, rack.inners, ini)
	{
		//play(number) means start reading of <number> samples into device buffer, this is sizeof(audio_type)*in.format.channels bytes!!!
		if((*ini)->on_air())
		{
			(*ini)->play(buffer_samples);
			++active_inputs;
		}
	}

	// Wait until all inputs have finished pulling data.
	for(unsigned int i = 0; i < active_inputs; ++i){ insem.wait(); } 		

	return active_inputs;
}

unsigned int mixer::dispatch_output()
{
	scholar sam(rack.the_wall);
	unsigned int active_outputs = 0;
	FOR_VECTOR(outchannel*, rack.outers, outi)
	{
		if
		(
			(*outi)->status == channel::PLAYING // If the channel is ready for playback and ...
			&& ( (*outi)->buffer.fill != 0  // Either the buffer got some real audio input... 
				  ||
				  (*outi)->state.live ) // ...or the output needs to be kept alive with silence.
		)
		{
			++active_outputs;
			// Play silence if nothing else.
			// Buffer is at least initialized to all-zero.
			if((*outi)->buffer.fill == 0) (*outi)->buffer.fill = (*outi)->buffer.size;

			(*outi)->play();
		}
	}

	// Wait until all outputs have finished pushing data.
	for(unsigned int i = 0; i < active_outputs; ++i){ insem.wait(); }

	return active_outputs;
}

} // namespace
