#ifndef DMD_H_INCHANNEL
#define DMD_H_INCHANNEL
/*
	inchannel: mixer input channels
	
	part of DerMixD
	(c)2004-2013 Thomas Orgis, licensed under GPLv2
*/

#include "channel.hxx"
#include "audio/audio_buffer.hxx"
#include "in/input_state.hxx"
#include "in/input_file.hxx"
#include "audio/audio_fifo.hxx"
#include "action_queue.hxx"
#include "actionlist.hxx"
#include "file_type.hxx"
#include "audio/effectchain.hxx"

#include <vector>
#include <string>

class semaphore;

namespace dmd {

// Synchronization hack to avoid background work stomping on playback (fetching
// from input channels): Grab this lock to ensure that background work from
// input channels does not interfere with your important work (mixing next
// chunk). Background work is divided into chunks and before each chunk is
// processed, this lock is aquired/released once.
extern real_mutex playlock;

// Forward declarations to avoid header dependency.
class socket_writer;
class outchannel;

class inchannel: public channel, thread
{
	public:

	enum postact { BLISS, NEW_SCRIPTER, NO_SCRIPTER };

	input_state state;
	inchannel* follower;
	inchannel* leader;
	mixer_buffer inbuffer;  // Converted samples from iobuffer.
	mixer_buffer buffer;    // Result of resampling/whatever.
	audio_type startval[2]; // Hack for the resampler.
	size_t mixer_rate;
	// Need the mixer action list to insert scripting actions.
	actionlist *mixer_actions;
	comm_data *mixer_comdat;

	double speed;
	float volume[2];
	float preamp; // Pre-amplification factor.

	std::vector<outchannel*> targets; //pointers to output objects... just make sure that on deletion of those the pointers here are deleted first (?)

	// TODO: Is bufsize really necessary as parameter?
	inchannel(const size_t idv, const std::string namev, semaphore* playsem, audio_format &, size_t bufsize, dmd::actionlist &mixact, comm_data &mixer_comdat_);

	~inchannel();

	bool on_air(){ return (!working && status == channel::PLAYING); };

	void statline(std::string &ts); // Print status line to ts.
	void set_id(size_t newid); // Change ID and notify watchers.

	// needed??? bool device_sane();
	void nodev();
	// Not float and offer configurable rounding?
	off_t position_in_seconds(){ return (off_t) state.format.seconds(state.position); };
	bool resize_buffer(unsigned int channels, size_t s);

	//script helpers... could be generic?
	
	double sampleratio(){ return speed*state.format.rate / mixer_rate; };
	// Get time in seconds for _now_ and the time after extracting a chunk of output samples. 
	void now_and_then(size_t chunksize, float &now, float &then);
	void get_script_commands(std::vector<std::string*> &list);

	// Synchronize channel state with input device state.
	// The input device state is modified by a separate thread... possibly anytime.
	void sync_state();

	// Manage output channel connection.
	void bind(outchannel* outch); // Bind to output.
	void unbind(outchannel *outch); // Release the bond.

	enum postact handle_action(action *act, comm_data *extra_cd = NULL);

	// Some action handler hooks need to be accesible from the mixer.
	// Manage watchers.
	void watch(socket_writer *swr); // Add watcher.
	void unwatch(socket_writer *swr); // Remove watcher.
	// Set following relation.
	void set_follow(action *act, inchannel *imp);
	// Send updated status info, if any.
	void telltime();

	// Play a given number of samples into the output buffers,
	// starting at given offset. This includes chesamplify and followers.
	// This will modify status to PLAYING or PAUSED, respectively.
	// WARNING: Always wait for signal on playsem before issuing the next play!
	// It's a dickheaded optimization, avoiding dynamic allocation of multiple actions for communication with the worker thread.
	void play(size_t n_samples, size_t offset=0); 

	// Only here to keep the former input_device_decode test working.
	// That might get changed to use action handling code instead.
	void load(action *act);

	// Sever all connections to other channels.
	void detach();

	// I am the worker thread, remember?
	void thread_work();

	private:
	// A bit hacky: One static action that gets passed for playback.
	action playact;
	// Those two are for telltime().
	off_t hadseconds;
	enum statecode hadstatus;
	real_mutex watchlock;
	std::vector<socket_writer*> watchers;

	// Notify on the action and get rid of it if it's disposable.
	void notify_and_dispose(action *);
	void start(action *act);
	void eject(action *act);

	// Follwership of channels. This channel is the leader.
	void no_follow();
	// Send some message, prefixed with channel identification.
	void tellsomething(const std::string msg);
	// Mixer actions... with optional extra communication path for implicit actions.
	// Some pull parameters directly out of the action, some rely on mixer to precompute.
	void set_volume(action *act, comm_data *extra_cd = NULL);
	void set_speed(action *act, double speedval, comm_data *extra_cd = NULL);
	void set_preamp(action *act, comm_data *extra_cd = NULL);
	void issue_seeklike(action *act);
	void pause(action *act);

	// Mixer got notified that the load operation is done.
	void finished_load(action *act, comm_data *extra_cd);
	// Same for seek-like operation...
	void finished_seeklike(action *act);

	// This is the worker thread with its data.
	// It's the contained thread as well as a sub thread container for the prebuffer option.
	std::string problem;

	// Close action is in the worker, but not used...
	// void close(comm_data* cd); // Close file.

	// Stuff from the erstwhile input_worker....

	// The state the worker manages, with lock.
	action_queue worklist;
	input_state workstate;
	semaphore *playsem; // Semaphore to signal on play action.

// The worker thread works with these... prebuffer is constructed in the device, also destruction is handled outside the worker thread.
// But changing of underlying input file, that is allowed in the worker.

	// The input device wraps over some input file.
	input_file *infile;
	// The prebuffer is an option, if != NULL, it wraps over the file specified in input_worker. Input work
	audio_fifo *prebuffer;
	// Since infile and prebuffer can be interchangeable most of the times (both implement the input_file API), we also use a work pointer to access either.
	input_file *workfile;
	dmd::effect::chain effectchain;
	input::file_type typer; // To guess input file types...
	// Would be nice to have these totally hidden, but this is C++...
	// These are the actual working routines the worker thread calls.
	// And yes, probably I'll figure out how to properly put this into a
	// separate class. This needs repartitioning of the whole channel thing.
	void work_process_action(action *act);
	void work_play(action *act);
	void work_seek(action *act);
	void work_open(action *act, const std::string filename, const std::string type);
	void work_close(action *act);
	// Set a single or all (band==EQ_BANDS) equalizer channels.
	void work_eq(action *act, enum input_state::eqband band);
	void work_length(action *act);
	void work_scan(action *act);
	void work_add_effect(action *act);
	void work_remove_effect(action *act);
	void work_bypass_effect(action *act);
	void work_tweak_effect(action *act);
	void work_list_effects(action *act);
	void work_help_for_effect(action *act);
	void work_kill_decoder(action *act);
	bool work_format_change();
};

}
#endif
