/*
	inchannel: mixer input channels
	
	part of DerMixD
	(c)2004-2013 Thomas Orgis, licensed under GPLv2
*/

#include "basics.hxx"
#include "inchannel.hxx"
#include "outchannel.hxx"

#include "coms/socket_writer.hxx"

#include "action.hxx"
#include "coms/client_api.hxx"
#include "errorcodes.hxx"
#include "errorchain.hxx"
#include "audio/scanner.hxx"
#include "comparse.hxx"

#include "param_init.hxx"
#include "tstring.hxx"

#include <vector>
#include <string>
using std::string;
using std::vector;

#include "shortcuts.hxx"
#include "debug.hxx"
#define CMDEBUG(s, ...) MDEBUG( "[inch %zu:%s] " s, id, name.c_str(),  __VA_ARGS__)
#define CMXDEBUG(s, ...) MXDEBUG( "[inch %zu:%s] " s, id, name.c_str(),  __VA_ARGS__)
#define CSDEBUG(s) MDEBUG( "[inch %zu:%s] " s, id, name.c_str())
#define CSXDEBUG(s) MXDEBUG( "[inch %zu:%s] " s, id, name.c_str())

#define ME "inchannel"

using namespace dmd;

// Some special-purpose action definitions to communicate with the worker.
namespace dmd {

real_mutex playlock;

namespace api {

static const def inwork_nodec =
{ INWORK_NODEC, "", "", 0,
	{ },
	{ },
	0
};

static const def inwork_play =
{ INWORK_PLAY, "", "", 2,
	{ "wanted samples", "out offset"   },
	{ ACT_SIZE, ACT_SIZE },
	0
};

}

//reconstruct script commands
void inchannel::get_script_commands(vector<string*> &list)
{
	action scriptac(dmd::api::client_world);
	scriptac.good = true;
	scriptac.floats.push_back(0);
	scriptac.sizes.push_back(id);
	scriptac.longs.push_back(1);
	for(vector<timeaction>::iterator t = script.begin(); t != script.end(); ++t)
	{
		string* bully = new string;
		scriptac.subaction = (*t).act;
		scriptac.floats[0] = (*t).time;
		scriptac.longs[1] = (*t).count;
		scriptac.def = (*t).count > 1 ? &api::in_nscript : &api::in_script;
		scriptac.print(*bully);
		list.push_back(bully);
	}
	for(vector<timeaction>::iterator t = postscript.begin(); t != postscript.end(); ++t)
	{
		string* bully = new string;
		scriptac.subaction = (*t).act;
		scriptac.floats[0] = (*t).time;
		scriptac.longs[1] = (*t).count;
		scriptac.def = (*t).count > 1 || (*t).count < 0 ? &api::in_nscript : &api::in_script;
		scriptac.print(*bully);
		list.push_back(bully);
	}
	scriptac.subaction = NULL; //don't delete action on destruct!
}

// This class is too big!
inchannel::inchannel
(
	  const size_t idv
	, const string namev
	, semaphore* theplaysem
	, audio_format &format
	, size_t bufsize
	, dmd::actionlist &mixact
	, comm_data &mixer_comdat_
)
	: channel(idv, namev)
	, thread("input worker", nice::input)
	, follower(NULL), leader(NULL)
	, mixer_actions(&mixact)
	, mixer_comdat(&mixer_comdat_)
	, speed(1.), preamp(0.)
	, playact(NULL, &api::inwork_play)
	, hadseconds(-1), hadstatus(NOTHING)
	, worklist(), workstate(&worklist) // Worklist in double-function as mutex for the state, too.
	, playsem(theplaysem)
	// Input worker stuff (some of the above, too)
	, infile(NULL), prebuffer(NULL), workfile(NULL)
	, effectchain()
{
	CONSTRUCT("inchannel");
	startval[0] = startval[1] = 0;
	volume[0] = volume[1] = 1.;
	resize_buffer(format.channels, bufsize);
	CMDEBUG("I am %p.", this);
	// If prebuffer is demanded, we create it.
	// Note that this does (and should) happen before creating the actual worker thread.
	if(param.as_double("input", "prebuffer") > 0.)
	{
		prebuffer = new audio_fifo(nice::input_prebuffer);
		if(prebuffer == NULL)
		SERROR("Failed to create prebuffer! Continuing without...");
		else if(prebuffer->have_error())
		{
			SERROR("Prebuffer did not initialize, deleting it.");
			delete prebuffer;
			prebuffer = NULL;
		}
		else sub_thread_container = prebuffer;
	}
	// Funny... now I arrived at a thread container that contains itself. Starts itself. Kills itself.
	if(!create_thread(this))
	{
		problem = "cannot create worker thread";
		status = DEAD;
	}
	mixer_rate = format.rate;
	// Prepare storage for the play action parameters.
	playact.sizes.push_back(0);
	playact.sizes.push_back(0);
}

// Using the single play action to avoid allocating a new one repeatedly.
void inchannel::play(size_t n_samples, size_t offset)
{
	playact.sizes[0] = n_samples;
	playact.sizes[1] = offset;
	worklist.push(&playact);
}

void inchannel::statline(std::string &ts)
{
	sync_state(); // Play safe.
	ts = "";
	strprintf(ts, "in %zu:", id);
	ts += name;
	ts += " -> (";
	if(!targets.empty())
	{
		strprintf(ts, "%zu", targets[0]->id);
		for(vector<outchannel*>::iterator outi = targets.begin()+1; outi != targets.end(); ++outi)
		strprintf(ts, ",%zu", (*outi)->id);
	}
	ts += ") ";
	ts += state_text();
	strprintf(ts, " at %gs/%gs (buf: %lu) vol: %g speed: %g",
		         state.format.seconds(state.position),
		         state.format.seconds(state.length),
		         (unsigned long)buffer.fill, 0.5*(volume[0]+volume[1]), speed);
	if(input_state::EQ_BANDS > 0)
	{
		strprintf(ts, " eq: %g", state.eqval[0]);
		for(int band=1; band<input_state::EQ_BANDS; ++band)
		strprintf(ts, "|%g", state.eqval[band]);
	}
	//leader, script and device/file
	if(leader != NULL) strprintf(ts, ", following %zu:%s", leader->id, leader->name.c_str());
	if( !script.empty() ) strprintf(ts, ", %zu actions", script.size());
	if(state.type != input::invalid)
	{
		ts += ", dev: ";
		ts += input::id_to_name(state.type);
		ts += " file: ";
		ts += state.filename;
	}
}

void inchannel::set_id(size_t newid)
{
	string bla;
	strprintf(bla, "change ID to %zu", newid);
	tellsomething(bla);
	id = newid;
}

void inchannel::nodev()
{
	SWARNING("Communication!");
	action *chore = new action(NULL, &api::inwork_nodec);
	chore->comdat = NULL;
	chore->disposable = true;
	worklist.push(chore);
	// Hm, actually the status is "initiated entering of NODEV state, to happen soon".
	status = NODEV;
}

// Funny little helper for the confused business about disposable actions.
// Wait ... it can do the disposal itself!
void inchannel::notify_and_dispose(action *act)
{
	bool dispose = act->disposable;
	act->notify();
	if(dispose)
	{ // If it's disposable, we are allowed to access its contents here.
		CMDEBUG("disposing of action %p (%s)", act, act->def->name);
		delete act;
	}
}


void inchannel::load(action *act)
{
	if(leader != NULL) leader->no_follow();

	worklist.push(act);

	status = LOADING;
}

void inchannel::eject(action *act)
{
	if(leader != NULL) leader->no_follow();

	worklist.push(act);

	status = IDLE; // Actually, that status should be set after eject completed...
}

bool inchannel::resize_buffer(unsigned int ch, size_t s)
{
	if(buffer.reallocate(ch,s) == 0)
	{
		MERROR("[ch %zu] Buffer resize failed! D'OOM!", id);
		status = DEAD;
		return false;
	}
	else return true;
}

inchannel::~inchannel()
{
	DESTBEGIN("inchannel");
	delscript();
	// Polite inquiry to quit is not safe unless we have a backchannel to properly wait until the worker has processed that action.
	// Under extreme timing conditions, the thread could get cancelled before being started up properly -- and so it would be cancelled without doing anything and our pushed action is leaked memory.
	// So, just cancel it. Works.
	kill_thread(this);
	if(prebuffer != NULL) delete prebuffer;
	if(infile    != NULL) delete infile;
	DESTEND("inchannel");
}

void inchannel::now_and_then(size_t chunksize, float &now, float &then)
{
	now  = state.format.seconds(state.position);
	then = state.format.seconds(state.position + (off_t)(sampleratio()*chunksize));
}

void inchannel::sync_state()
{
	state.sync(workstate, buffer.fill);
}

// Bind an input channel to an output channel.
void inchannel::bind(outchannel* outch)
{
	// Only bind if not already bound...
	bool bound = false;
	FOR_VECTOR(outchannel*, targets, outi)
	if(*outi == outch){ bound = true; break; }

	if(!bound)
	{
		targets.push_back(outch);
		string bla;
		strprintf(bla, "bind %zu", outch->id);
		tellsomething(bla);
	}
}

// Release the bond.
void inchannel::unbind(outchannel* outch)
{
	FOR_VECTOR(outchannel*, targets, outi)
	if(*outi == outch){ targets.erase(outi); break; }

	string bla;
	strprintf(bla, "unbind %zu", outch->id);
	tellsomething(bla);
}

// The bond between two channels following each other.
// Always from the viewpoint of the leader: Set and unset the follower relation.
void inchannel::set_follow(action *act, inchannel *imp)
{
	// Remove existing follower.
	if(follower != NULL) no_follow();

	// Prevent multiple following and invalid follower.
	if(imp->leader != NULL)
	{
		act->push_error(new error(ME, dmd::err::IN_FOLLOW_ANOTHER, "This follower has a leader already."));
	}
	else if(imp->status == PAUSED)
	{
		follower = imp;
		follower->leader = this;
		string bla;
		strprintf(bla, "follow %zu",follower->id);
		tellsomething(bla);
	}
	else
	act->push_error(new error(ME, dmd::err::BAD_STATE, "This follower is not in an appropriate state."));
}

void inchannel::no_follow()
{
	CSDEBUG("removing follower");
	if(follower != NULL)
	{
		CMDEBUG("...which is %zu:%s", follower->id, follower->name.c_str());
		follower->leader = NULL;

		string bla;
		strprintf(bla, "nofollow %zu",follower->id);
		tellsomething(bla);
		follower = NULL;
	}
}

// Add swr to the list of sockets that get playback info.
void inchannel::watch(socket_writer* swr)
{
	// Look out for the watcher already being there...
	bool watched = false;
	watchlock.lock();
	FOR_VECTOR(socket_writer*, watchers, swi)
	if(*swi == swr){ watched = true; break; }

	// Now only push it if not found in the list.
	if(!watched) watchers.push_back(swr);
	watchlock.unlock();
}

// Remove the one single enry of swr. Only one is possible. Should be.
void inchannel::unwatch(socket_writer* swr)
{
	watchlock.lock();
	FOR_VECTOR(socket_writer*, watchers, swi)
	if(*swi == swr){ watchers.erase(swi); break; }
	watchlock.unlock();
}

void inchannel::telltime()
{
	//we have the last told time in hadseconds
	//get the current secs by position_in_seconds()
	//if different, print and save
	off_t ntime = position_in_seconds();
	if(ntime != hadseconds || status != hadstatus)
	{
		hadseconds = ntime;
		hadstatus = status;
		watchlock.lock();
		if(!watchers.empty())
		{
			std::string ts;
			strprintf(ts, "[ch%zu] %"OFF_P"s, ", id, ntime);

			ts += state_text();
			// Distribute copies to all watchers.
			FOR_VECTOR(socket_writer*, watchers, swi) (*swi)->write_copy(ts);
		}
		watchlock.unlock();
	}
}

// Can be called from different threads...
void inchannel::tellsomething(const string msg)
{
	watchlock.lock();
	if(!watchers.empty())
	{
		std::string ts;
		strprintf(ts, "[ch%zu] ", id);
		ts += msg;
		// Distribute copies to all watchers.
		FOR_VECTOR(socket_writer*, watchers, swi) (*swi)->write_copy(ts);
	}
	watchlock.unlock();
}

enum inchannel::postact inchannel::handle_action(action *act, comm_data *extra_cd)
{
	enum postact ret = BLISS;
	if(act->def->id == api::NOTIFY)
	{
		// Some longer-term work has been finished.
		// Any notification means there could be input state change.
		sync_state();
		// Notifications need specific postprocessing...
		switch(act->subaction->def->id)
		{
			case api::IN_LOAD:
			case api::IN_INLOAD:
				finished_load(act, extra_cd);
			break;
			case api::IN_SEEK:
			case api::IN_RSEEK:
			case api::IN_LENGTH:
			case api::IN_SCAN:
				finished_seeklike(act);
			break;
			default:
			act->push_error(new error(ME, dmd::err::UNKNOWN_COMMAND, "Notification for _what_?"));
			notify_and_dispose(act);
		}
	}
	else if(working)
	{
		act->push_error(new error(ME, dmd::err::BUSY, "Input channel is busy -- it will not accept any action until the mixer got the notification that the work is done."));
		notify_and_dispose(act);
	}
	else switch(act->def->id)
	{
		case api::IN_VOLUME:
			set_volume(act, extra_cd);
		break;
		case api::IN_PREAMP:
			set_preamp(act, extra_cd);
		break;
		case api::IN_SPEED:
			set_speed(act, act->floats[0], extra_cd);
		break;
		case api::IN_PITCH:
			set_speed(act, speed + act->floats[0], extra_cd);
		break;

		case api::IN_START:
			start(act);
		break;
		case api::IN_PAUSE:
			pause(act);
		break;
		case api::IN_LOAD:
		case api::IN_INLOAD:
			load(act);
		break;
		case api::IN_EJECT:
			eject(act);
		break;
		case api::IN_SEEK:
		case api::IN_RSEEK:
		case api::IN_LENGTH: // this is a self-controlled seek, of sorts
		case api::IN_SCAN:   // This, too.
			issue_seeklike(act);
		break;

		case api::IN_NOFOLLOW:
			no_follow();
			notify_and_dispose(act);
		break;

		case api::IN_WATCH:
			watch(act->swr);
			notify_and_dispose(act);
		break;
		case api::IN_UNWATCH:
			unwatch(act->swr);
			notify_and_dispose(act);
		break;

		case api::IN_SCRIPT:
			//add script action to inchannel, checking channel validity (there must be a track loaded)
			{
				action* sact = new action();
				*sact = *act->subaction; //this for sure won't work with a further subaction!
				CMDEBUG("adding script action at %p; copied from %p", sact, act->subaction);
				sact->comdat = NULL; // swr stays there; not the direct thread comunication
				if(act->floats[0] < 0) addpostaction(act->floats[0], act->longs[0], sact);
				else
				if( addaction(act->floats[0], act->longs[0], sact) == 1 )
				ret = NEW_SCRIPTER;
			}
			notify_and_dispose(act);
		break;

		case api::IN_CLEARSCRIPT:
			delscript();
			ret = NO_SCRIPTER;
			notify_and_dispose(act);
		break;

		case api::IN_SHOWSCRIPT:
		{
			CSDEBUG("showscript!");
			get_script_commands(act->comdat->retlines);
			notify_and_dispose(act);
		}
		break;

		default: // worker always disposes
			worklist.push(act);
	}

	// Any action could mean that there is something new to say, so just let's trigger that here.
	// Better say too much than to omit something.
	telltime();

	return ret;
}

void inchannel::set_volume(action *act, comm_data *extra_cd)
{
	CMDEBUG("setting volume to %g %g", act->floats[0], act->floats[1]);
	volume[0] = act->floats[0];
	volume[1] = act->floats[1];

	// return value is mean volume... yeah, return value stuff needs to be generalized like input parameters.
	if(act->comdat != NULL)
	act->comdat->retval = 0.5*(volume[0]+volume[1]);

	string bla;
	strprintf(bla, "volume %g %g", volume[0], volume[1]);
	tellsomething(bla);

	notify_and_dispose(act);
}

void inchannel::set_preamp(action *act, comm_data *extra_cd)
{
	CMDEBUG("setting preamp to %g dBFS", act->floats[0]);
	preamp = act->floats[0];

	if(act->comdat != NULL)
	act->comdat->retval = preamp;

	string bla;
	strprintf(bla, "preamp %g dBFS", preamp);
	tellsomething(bla);

	notify_and_dispose(act);
}

void inchannel::set_speed(action *act, double speedval, comm_data *extra_cd)
{
	// Backwards play comes later... perhaps.
	if(speedval < 0) speed = 0;
	else speed = speedval;

	if(act->comdat != NULL) act->comdat->retval = speed;

	string bla;
	strprintf(bla, "speed %g", speed);
	tellsomething(bla);

	notify_and_dispose(act);
}

// Once a seek operation has been handed down to the worker, the worker is responsible for notifying for the completion of the action.
void inchannel::issue_seeklike(action *act)
{
	CSDEBUG("wanna seek/scan");
	if(leader != NULL) leader->no_follow();

	if( (state.length > 0) && 
	    (    (status == PLAYING)
	      || (status == PAUSED) ) )
	{
		buffer.fill = 0; //just discard it now... maybe retain in future for length scan?

		CMDEBUG("%s", act->def->name);
		worklist.push(act);

		// Even if we just returned an error, the the socketeer is still supposed to send a followup notification to the mixer to release the channel from the "working" state.
		working = true;
		if(status == PLAYING) status = SEEKING_PLAY;
		else status = SEEKING_PAUSE;
	}
	else
	{
		act->push_error(new error(ME, dmd::err::NONSENSE, "Seek/scan operation does not make sense right now."));
		notify_and_dispose(act);
	}
}


void inchannel::start(action *act)
{
	CSDEBUG("wanna start");
	if(leader != NULL) leader->no_follow();

	//basically we can start if there is a file loaded... even if it is already playing, the start should succeed, I guess... because we want that it is playing afterwards and this is the case, then.
	if(    (status == PLAYING)
	    || (status == PAUSED)  )
	{
		status = PLAYING;
	}
	else act->push_error(new error(ME, dmd::err::NONSENSE, "Starting playback from the current state makes no sense."));

	notify_and_dispose(act);
}

void inchannel::pause(action *act)
{
	if(    (status == PLAYING)
	    || (status == PAUSED)  )
	{
		status = PAUSED;
	}
	else act->push_error(new error(ME, dmd::err::NONSENSE, "Pausing from the current state makes no sense."));

	notify_and_dispose(act);
}

void inchannel::finished_load(action *act, comm_data *extra_cd)
{
	working = false;
	// If the channel had a real load to do (instead of nodev), we update the status for success.
	status = act->have_errors() ? IDLE : PAUSED;
	tellsomething(status == PAUSED ? "loaded" : "nofile");
	CSDEBUG("load finished (success or failure)");
	notify_and_dispose(act);
}

void inchannel::finished_seeklike(action *act)
{
	// We don't do much error handling here anymore. Just note that the operation finished.
	working = false;
	if(status == SEEKING_PAUSE) status = PAUSED;
	else if(status == SEEKING_PLAY) status = PLAYING;
	else
	{
		act->push_error(new error(ME, dmd::err::NONSENSE, "Finished a seek, but in what state am I? Might be a programming error here..."));
		status = PAUSED;
	}
	notify_and_dispose(act);
}

// It is safe to call this when not in main playback action.
// All other work the thread does does not concern these members.
void inchannel::detach()
{
	// Delete from all targets' sources
	FOR_VECTOR(outchannel*, targets, outi)
	{
		(*outi)->unbind(this);
	}
	targets.clear();
	no_follow();
	if(leader) leader->no_follow();
}

// ==================
// The worker thread.
// ==================


// The worker thread creates infile / prebuffer, the destructor of inchannel deletes them in the mother thread.

void inchannel::thread_work()
{
	CSDEBUG("startup");
	// Do file type database initialization here in the separate thread.
	// It would be implicit on the first file load, but I want it prepared as soon as possible.
	typer.prepare();
	// Now loop until getting the command to quit.
	while(1)
	{
		CSXDEBUG("thread waiting for work");
		// Pull next action from queue, that includes waiting for an action becoming available.
		// ... and the possibility to get cancelled.
		action *act = worklist.pull(this);
		if(act == NULL || act->def == NULL)
		{
			SERROR("Pulled a null action! Should this happen?");
			continue;
		}

		// After processing, the action is not mine anymore.
		// It could even be gone (client takes over after act->notify()).
		// But if it is a disposable action, there is nobody else watching.
		bool dispose = act->disposable;
		work_process_action(act);
		if(dispose) delete act; // Done with the action, delete it.
	}
	// That will never happen. Cancellation is what happens.
	CSDEBUG("leaving");
}

// Process non-NULL action. Outsourced for scoping the comdat lock
// and having one indendation level less.
void inchannel::work_process_action(action *act)
{
	locker room(act->comdat);
	enum api::aid aid; // the action's ID
	aid = act->def->id;
	if(aid == api::INWORK_PLAY) CMXDEBUG("processing play action %i", aid);
	else CMDEBUG("processing action %i", aid);

	// Ensure that we got the action parameters we expect.
	if(!act->valid())
	{
		CSDEBUG("Action is invalid!");
		act->push_error(new error(ME, dmd::err::BAD_PAR, "This action does not carry the parameters it requires (internal programming error?)."));
	}
	// Common check for actions that assume that there is an existing workfile.
	else if(act->def->flags & api::INWORK_FILE && (infile == NULL || workfile == NULL))
	act->push_error(new error(ME, dmd::err::NO_DEVICE, "Cannot execute this action without a loaded input file."));
	else
	// Get down to work.
	switch(aid)
	{
		case api::INWORK_PLAY:
			work_play(act);
		break;
		case api::IN_SEEK:
		case api::IN_RSEEK:
			work_seek(act);
		break;
		case api::IN_LOAD:
			work_open(act, act->strings[0], "auto");
		break;
		case api::IN_INLOAD:
			work_open(act, act->strings[1], act->strings[0]);
		break;
		case api::IN_EJECT:
			work_close(act);
		break;
		case api::IN_BASS:
			work_eq(act, input_state::BASS);
		break;
		case api::IN_MID:
			work_eq(act, input_state::MID);
		break;
		case api::IN_TREBLE:
			work_eq(act, input_state::TREBLE);
		break;
		case api::IN_EQ:
			work_eq(act, input_state::EQ_BANDS);
		break;
		case api::IN_LENGTH:
			work_length(act);
		break;
		case api::IN_SCAN:
			work_scan(act);
		break;
		case api::IN_EFFECT_ADD:
			work_add_effect(act);
		break;
		case api::IN_EFFECT_REMOVE:
			work_remove_effect(act);
		break;
		case api::IN_EFFECT_BYPASS:
			work_bypass_effect(act);
		break;
		case api::IN_EFFECT_PARAM:
			work_tweak_effect(act);
		break;
		case api::IN_EFFECT_LIST:
			work_list_effects(act);
		break;
		case api::IN_EFFECT_HELP:
			work_help_for_effect(act);
		break;
		case api::INWORK_NODEC:
			work_kill_decoder(act);
		break;
		default:
			act->push_error(new error(ME, dmd::err::UNKNOWN_COMMAND, "Some unknown input command arrived."));
	}
	// Always wake up. ... except for play action ... fix that mess!
	if(aid != api::INWORK_PLAY)
	{
		if(act->def->flags & api::SOCK_NOTIFY)
		{
			CMDEBUG("notifying completion of action %p (id %i)", act, act->def->id);
			action *notify_act = api::make_notify_act(act);
			mixer_actions->push(notify_act);
		}

		CMDEBUG("input worker notifies on action %p", act);
		act->notify();
	}
}


//
// The actual functionality. Little methods, one for each action executed in the worker thread.
//

// The actual playback: Extract audio data from the workfile and put it into the device buffer.
// There is no usual back-communication, just the posting of a semaphore when done and the fill of the device buffer.
void inchannel::work_play(action *act)
{
	bool hit_end = false;
	inbuffer.fill = 0; // Fill it from beginning.
	buffer.fill = 0;
	size_t ordered_samples = act->sizes[0];
	size_t offset = act->sizes[1];
	// Hm, the goto shortcuts don't allow declaring variables in between.
	// Yeah, goto is evil and such, I know.
	size_t input_samples, got_samples;

	if(workfile == NULL)
	{
		MERROR("[inch %zu:%s] Game without a player? There is no input file to work on.", id, name.c_str());
		status = IDLE;
		goto play_end;
	}

	status = PLAYING; // When we play, we are playing.

	input_samples = round2size(sampleratio() * ordered_samples);
	CMXDEBUG("ordering %zu input samples for %zu in mixer (speed: %g)", input_samples, ordered_samples, speed);
	if(!input_samples) goto play_end;

	// Now, assure that the device buffer is big enough, then read into it.
	// Actual reallocation will only happen if the size changed.
	// Oh... an error here should get more handling...
	if(inbuffer.reallocate(effectchain.format()->channels, input_samples) != input_samples)
	{
		MERROR("[inch %zu:%s] unable to resize buffer!", id, name.c_str());
		hit_end = true; // At least stop the playback here and give followers a chance. Don't stop the music.
		goto play_end;
	}
	if(!(effectchain.receive(inbuffer, input_samples)))
	{
		// This is an extraordinary error, so some fuzzing is allowed to get the message out.
		locker room(mixer_comdat);
		effectchain.collect_errors(mixer_comdat->errors);
		mixer_comdat->waitress->post();
		// Still use the data received up to now. This is an end, not the apocalypse.
	}

	// Advance stream position.
	// Also possibly update the length.
	workstate.set_position(workfile->position);

	got_samples = ordered_samples;
	if(inbuffer.fill < input_samples)
	{
		// We hit the end.
		hit_end = true;
		workstate.set_length(workfile->position);
		// If we got less input samples, we need to scale the resulting mixer samples accordingly.
		got_samples = round2size((double)inbuffer.fill / input_samples * ordered_samples);
		CMDEBUG("reached end, got only %zu < %zu insamples (%zu < %zu for mixer)", inbuffer.fill, input_samples, got_samples, ordered_samples);
	}

	// Got the input data, prepare for further consumption.
	//resample + amplify from input buffer to mixer buffer.
	if(buffer.size < got_samples && buffer.reallocate(got_samples) != got_samples)
	{
		MERROR("[inch %zu:%s] failed to resize mixer buffer", id, name.c_str());
		goto play_end;
	}
	else
	{
		float realvol[2];
		realvol[0] = volume[0] * audio_dbfs2vol(preamp);
		realvol[1] = volume[1] * audio_dbfs2vol(preamp);
		buffer.chesamplify(inbuffer, got_samples, realvol, startval);
		CMXDEBUG("have: %u", buffer.fill);
	}

	// Now mix to output.
	FOR_VECTOR(outchannel*, targets, outi)
	{
		// Buffer smoothing might occur here... or not at all.
		if( (*outi)->status == channel::PLAYING )
		(*outi)->buffer.mix_in(buffer, offset);
	}

play_end:
	// If we weren't adequate, a follower might try to push more data.
	if(hit_end)
	{
		status = PAUSED;
		// Issue postscript
		if(!postscript.empty())
		{
			std::vector<action*> pushscript;
			getpostactions(pushscript);
			CMDEBUG("adding %zu postactions to worklist", pushscript.size());
			mixer_actions->push_some(pushscript);
			CSDEBUG("done adding");
		}
		if(follower != NULL)
		{
			// No official announcement about the divorce between leader and follower, because this is no abortion, it's the fulfillment of the promise.
			follower->leader = NULL;
			CMDEBUG("triggering follower for filling from %zu to %zu samples.", got_samples, ordered_samples);
			// The number of samples we are going to order is going to be noted as history for next scripting action.
			size_t nextblock = ordered_samples - got_samples;
			// Issue script commands for beginning of follower, they're going to be executed a bit late.
			if(!follower->script.empty())
			{
				std::vector<action*> pushscript;
				follower->getactions(nextblock, pushscript);
				CMDEBUG("adding follower (%zu:%s) script with %zu actions", follower->id, follower->name.c_str(), pushscript.size());
				mixer_actions->push_some(pushscript);
				CSDEBUG("done adding");
			}
			follower->play(nextblock, offset+got_samples);
			follower = NULL;
			goto play_postend; // The follower will ring the semaphore.
		}
	}

	playsem->post(); // Always posting the same playback semaphore instead of one in the action...
play_postend:
	// This is wrong! 
	sync_state();
	telltime();
}

void inchannel::work_open(action *act, const std::string filename, const std::string type)
{
	// This also covers the optional prebuffer -- it will go into halt state until the next open is done.
	if(workfile != NULL) workfile->do_close();

	workstate.nofile(); // The file is gone...

	// Tell that the attempt to load is undertaken.
	tellsomething("load" + filename);

	if(type == "nodev")
	{
		// Intentional deletion of the file. No real load.
		delete infile;
		infile = NULL;
	}
	else
	{
		// Now determine the type of the new file and prepare for opening it...
		input::type_id usetype = input::invalid;
		// User can force a file type with "inload", otherwise, we guess.
		if(type != "auto")
		usetype = input::name_to_id(type);
		else
		usetype = typer.guess(filename);

		// Anticipate an announce trouble.
		if(usetype == input::invalid)
		act->push_error(new error(ME, dmd::err::BAD_FILE, "Cannot guess a file type or bad type provided."));
		else if(!input::type_available(usetype))
		act->push_error(new error(ME, dmd::err::BAD_FILE, "This file type is not available in this build."));

		// Switch the real underlying input file to the correct type.
		// For invalid types, we still rely on this to delete the old file.
		if(!input::switch_file(infile, usetype))
		act->push_error(new error(ME, dmd::err::BAD_FILE, "Unable to create a decoder for this file."));

		if(infile != NULL && (!infile->ready() || infile->have_error()))
		{
			CSDEBUG("input file not alive");
			infile->explain_error(act);
			act->push_error(new error(ME, dmd::err::BAD_FILE, "The input file did not come to life."));
			delete infile;
			infile = NULL;
		}
	}

	// File changed to something...
	workstate.newfile(infile != NULL ? infile->type : input::invalid); // That might be _no_ file.

	// Work out the roles of the two input_files.
	if(prebuffer != NULL)
	{
		prebuffer->set_source(infile);
		workfile = prebuffer;
	}
	else workfile = infile;

	// Now, if infile is NULL, that is on purpose or an error, in either case, we leave here.
	if(infile == NULL)
	{
		tellsomething("nodev");
		return;
	}

	// Actually open the file!
	if(workfile->do_open_safe(filename))
	{
		// Emphasize that we might want to mess with infile here.
		if(workfile == prebuffer) prebuffer->stop();

		// Apply equalizer settings that have been stored.
		workfile->do_eq(workstate.eqval);

		if(!effectchain.set_source(workfile))
		{
			if(act->comdat != NULL) effectchain.collect_errors(act->comdat->errors);
			workfile->do_close(); // Stops prebuffer, too.
		}
		else
		{
			if(work_format_change())
			{
				// Yeah! Load suceeded!
				// We use the underlying file's type.
				workstate.newfile(infile->type, effectchain.format(), filename);
				// But the length comes filtered through the buffer.
				// ... do we want to get length info modified by effect chain?
				workstate.set_length(workfile->do_length());
			}
			else act->push_error(new error(ME, dmd::err::NOMEM, "Cannot resize device buffer."));

			// Now the buffer shall start buffering.
			if(workfile == prebuffer) prebuffer->start();
		}
	}
	else
	{
		act->push_error(new error(workfile->err));
		act->push_error(new error(ME, dmd::err::BAD_FILE, "Cannot open the file."));
	}
}

// Format can change due to loaded file or changed effect setup.
// This is for channel count, currently... sampling rate change is more tricky, as that messes up the sample position seeks.
bool inchannel::work_format_change()
{
	if(inbuffer.reallocate(effectchain.format()->channels, param.as_size("buffer")))
	{
		workstate.newformat(effectchain.format());
		return true;
	}
	else return false;
}

void inchannel::work_seek(action *act)
{
	off_t offset;
	float aimed_seconds = (act->def->id==api::IN_SEEK)
		? act->floats[0]
		: workstate.format.seconds(workstate.position) + act->floats[0];
	if(aimed_seconds < 0) aimed_seconds = 0;

	offset = workstate.format.samples(aimed_seconds);
	// In the past ... and perhaps again in the future: Seek to definite sample offset.
	// offset = act->offsets[0];

	if(workfile->do_seek(offset))
	{
		workstate.set_position(workfile->position);;
		if(act->comdat != NULL) act->comdat->retval = workfile->format.seconds(workfile->position);

		if(!effectchain.reset())
		{
			if(act->comdat != NULL) effectchain.collect_errors(act->comdat->errors);
		}
	}
	else
	{
		workfile->explain_error(act);
		act->push_error(new error(ME, dmd::err::IO, "Seek failed."));
	}
}

void inchannel::work_length(action *act)
{
	off_t len;
	if((len=workfile->do_length()) >= 0)
	{
		workstate.set_length(len);
		if(act->comdat != NULL)
		{
			act->comdat->retval = workfile->format.seconds(len);
			// Those integer types need cleanup.
			act->comdat->retval_ulong = len;
			act->comdat->retval_ulong2 = workfile->format.rate;
		}
	}
	else
	{
		workfile->explain_error(act);
		act->push_error(new error(ME, dmd::err::NOT_SUPPORTED, "Length query not available, apparently."));
	}
}

void inchannel::work_scan(action *act)
{
	CSDEBUG("start scanning");
	if(act->comdat == NULL)
	{
		MERROR("[inch %p] Scanning without communication channel is lame!", this);
		return;
	}

	vector<string> scans;
	vector<scanner*> scanners;
	mixer_buffer scanb;

	// Determine the scanners and get them up.
	split(act->strings[0], scans);
	FOR_VECTOR(string, scans, si)
	scanners.push_back(create_scanner(*si, &workstate.format));

	// We need to go back to there.
	off_t oldpos = workfile->position;

	CSDEBUG("seek to begin");
	// Seek to beginning.
	if(!workfile->do_seek(0))
	{
		CSDEBUG("begin seek failed");
		workfile->explain_error(act);
		act->push_error(new error(ME, dmd::err::IO, "Seek-0 did not work."));

		workstate.set_position(workfile->position);
		goto scan_end;
	}

	// Then slurp it all through a private buffer and scan.

	// Don't be too greedy with I/O here, trying same buffer size as mixer...
	if(!scanb.reallocate(workfile->format.channels, param.as_size("buffer")))
	{
		act->push_error(new error(ME, dmd::err::NOMEM, "Cannot size up the scan buffer."));
		goto scan_end;
	}
	CSDEBUG("scan loop");
	// Now read through in a loop, breaking when the source is empty.
	// If scheduling works properly, the hack with playlock should not be needed.
	thread_priority(nice::input_background);
	while(1)
	{
		// Try to get the lock once here... actually just to block when mixer wants to get the important data.
		// This way the other input threads, possibly sharing the same priority, get a chance to do their work uninterrupted.
		// It is a hard measure here, introducing all these mutex operations... but it seems to work!
		// Of course, on a 16-core machine, this hurts scalability...
		playlock.lock();
		playlock.unlock();
		if(!workfile->mixbuf_read(scanb))
		{
			act->push_error(new error(ME, dmd::err::IO, "Cannot read from file for scanning."));
			goto scan_end;
		}
		// Now... SCAN!
		FOR_VECTOR(scanner*, scanners, i) if(*i != NULL) (*i)->scan(scanb);

		// It was the last round if the buffer was not full.
		if(scanb.fill < scanb.size || act->have_errors()) break;

		scanb.fill = 0;
	}

	// Gather the reports. We have been successful up to here.
	FOR_VECTOR(scanner*, scanners, i)
	{
		act->comdat->retlines.push_back(new std::string);
		if(*i != NULL)
		{
			(*i)->report(*act->comdat->retlines.back());
			(*i)->update_input(&state);
		}
		else act->push_error(new error(ME, dmd::err::BAD_STATE, "A NULL-scanner in the list?"));
	}
scan_end:
	thread_priority(nice::input);
	// No sane way to handle that error ... we are in a grey area here since scanning as such worked.
	// So, just update the position value -- it will at least be consistent.
	CSDEBUG("after scan, seek back");
	workfile->do_seek(oldpos);
	workstate.set_position(workfile->position);
	// Cleanup, always in need...
	FOR_VECTOR(scanner*, scanners, i) delete (*i);
	CSDEBUG("scanning done");
}

void inchannel::work_eq(action *act, enum input_state::eqband band)
{
	// Always store the settings.
	if(band < input_state::EQ_BANDS) workstate.eqval[band] = act->floats[0];
	else if(act->floats.size() == input_state::EQ_BANDS)
	workstate.eqval = act->floats;
	else
	act->push_error(new error(ME, dmd::err::BAD_PAR, "Bad number of values for all-band EQ setting."));

	// Just ignore equalizer settings if there is no underlying file.
	// The values are stored in the input channel and get handed in on next load operation.
	if(workfile == NULL) return;

	if(!act->have_errors())
	{
		if(!workfile->do_eq(workstate.eqval))
		{
			workfile->explain_error(act);
			act->push_error(new error(ME, dmd::err::DEVERR, "Equalizer setting just failed (not supported?)."));
		}
		else
		{
			std::string ts = "eq";
			FOR_VECTOR(float, workstate.eqval, v) strprintf(ts, " %g", *v);

			tellsomething(ts);
		}
	}
}

void inchannel::work_add_effect(action *act)
{
	// Need to handle partially optional argument lists...
	size_t pos = 0;
	if(act->sizes.size() > 1) pos = act->sizes[1];

	if(!effectchain.insert(act->strings[0], pos))
	{
		if(act->comdat == NULL) return;

		effectchain.collect_errors(act->comdat->errors);
	}
	else
	if(!work_format_change()) act->push_error(new error(ME, dmd::err::SOMETHING, "Cannot adapt to new format."));
}

void inchannel::work_remove_effect(action *act)
{
	size_t pos = 0;
	if(act->sizes.size() > 1) pos = act->sizes[1];

	if(!effectchain.remove(pos))
	{
		if(act->comdat == NULL) return;

		effectchain.collect_errors(act->comdat->errors);
	}
	else
	if(!work_format_change()) act->push_error(new error(ME, dmd::err::SOMETHING, "Cannot adapt to new format."));
}

void inchannel::work_bypass_effect(action *act)
{
	size_t pos = 0;
	if(act->sizes.size() > 1) pos = act->sizes[1];
	bool bypass = true;
	if(act->longs.size() > 0) bypass = act->longs[0] != 0;

	if(!effectchain.set_bypass(bypass, pos))
	{
		if(act->comdat == NULL) return;

		effectchain.collect_errors(act->comdat->errors);
	}

	if(!work_format_change()) act->push_error(new error(ME, dmd::err::SOMETHING, "Cannot adapt to new format."));
}

void inchannel::work_tweak_effect(action *act)
{
	size_t pos = 0;
	if(act->sizes.size() > 1) pos = act->sizes[1];

	effectchain.set_pars(act->strings[0], pos);
}

void inchannel::work_list_effects(action *act)
{
	if(act->comdat == NULL) return;

	effectchain.list(act->comdat->retlines);
}

void inchannel::work_help_for_effect(action *act)
{
	if(act->comdat == NULL) return;

	size_t pos = 0;
	if(act->sizes.size() > 1) pos = act->sizes[1];

	effectchain.help(act->comdat->retlines, pos);
}

void inchannel::work_close(action *act)
{
	tellsomething("eject");

	// Without workfile it is as closed as it gets already.
	if(workfile == NULL) return;

	// The normal case: We have a decoder instance, just let the file go.
	workfile->do_close(); // Stops prebuffer thread, if needed.
	// We have no file anymore... just a decoder type.
	workstate.newfile(infile != NULL ? infile->type : input::invalid);
}

void inchannel::work_kill_decoder(action *act)
{
	tellsomething("nodev");

	// If there is no decoder active, we are done already.
	if(workfile == NULL) return;

	// Close any open file, delete the real underlying input file.
	// And yes, I do mean workfile and infile here... that takes care of prebuffer.
	workfile->do_close();
	delete infile;
	infile = NULL;
	// We have nothing to work on now, communicate that.
	workfile = NULL;
	workstate.newfile();
}

} // namespace dmd
