#ifndef DMD_H_INPUT_STATE
#define DMD_H_INPUT_STATE

/*
	input_state: State of an input channel, communicated across threads.
	
	part of DerMixD
	(c)2004-2011 Thomas Orgis, licensed under GPLv2
*/

#include "audio/audio_format.hxx"
#include "mutex.hxx"

#include "in/input_file_types.hxx"
#include <vector>
#include <string>

// The input channel has one copy of the state, the input device another one.
// Reason: The background worker thread can mess with the copy of the input device, the mixer with the one in the channel.
// At opportune times, the mixer can sync its copy, secured with a mutex.
class input_state: public optional_mutex
{
	public:

	enum eqband { BASS=0, MID, TREBLE, EQ_BANDS };
	// The payload data is directly worked on by the worker thread on the input device copy.
	// The mixer copy needs to be synced on occasion.
	input::type_id type; // of underlying input file.
	std::string filename; // Stored filename for reference.
	off_t position; // Mixer can count that one according to the number of samples it got, but must update when being notified of seek success.
	off_t length; // Stored once after loading... also when actually reading the end. Result of scan action may be notified through clients, too.
	audio_format format;
	// Equalizer values (3 of them).
	std::vector<float> eqval;

	// Initialize with a mutex to enable locking, or not.
	 input_state(mutex * locker = NULL);
	~input_state();
	// Import state from some source, ensuring action by locking both sides.
	// (Usually, only the source will actually contain a mutex).
	// Meaning of position_offset: A number of samples not to count yet into the position of _this_ state. In other words: It's an offset in positions between the two states.
	void sync(input_state &source, off_t position_offset=0);
	// Clear everything except the type.
	void nofile();
	// A new file has been loaded, possibly.
	// The format and file name are only stored when format != NULL.
	void newfile(input::type_id ntype=input::invalid, audio_format *nformat=NULL, const std::string name="");
	void newformat(audio_format *nformat);
	// Update the position, possibly setting length, too, if position is larger.
	void set_position(off_t pos);
	// Just set the length.
	void set_length(off_t len);
};

#endif
