/*
	in_vorbisfile: vorbisfile input device (OGG Vorbis files)

	initially written by Sven Oliver Moll

	part of DerMixD
	(c)2007 Sven Oliver Moll and Thomas Orgis, licensed under GPLv2
*/

#include "in/drv/in_vorbisfile.hxx"
#include "debug.hxx"

#include <vorbis/vorbisfile.h>

// #include <cstdio>
#include <cstring>
#include <string>

using namespace std;

#define ME "vorbis_file"

struct vorbis_file::my_private
{
	bool opened;
	FILE *file;
	OggVorbis_File handle;
};

#define opened parts->opened
#define handle parts->handle
#define file   parts->file

vorbis_file::vorbis_file(): input_file(input::vorbis), parts(new struct my_private)
{
	CONSTRUCT(ME);
	memset(&handle, 0, sizeof(handle));
	opened = false;
	i_am_fine = true;
}

vorbis_file::~vorbis_file()
{
	DESTBEGIN(ME);
	do_close();
	DESTEND(ME);
}

void vorbis_file::do_close()
{
	// file gets closed by vorbisfile... (?)
	if(opened) ov_clear(&handle);
	opened = false;
}

bool vorbis_file::do_open(const string location)
{
	MDEBUG("going to open %s", location.c_str());
	file = fopen(location.c_str(), "rb");
	if(file)
	{
		SDEBUG("success");
		if(ov_open(file, &handle, 0, 0) == 0)
		{
			vorbis_info *vi = ov_info(&handle, 0);
			//its easy here to get the file parameters
			format.rate = vi->rate;
			format.channels = vi->channels;
			opened = true;
			return true;
		}
		else
		{
			fclose(file);
			file = NULL;
			return err.occur(dmd::err::BAD_FILE, "Vorbisfile open failed.");
		}
	}
	else
	return err.occur(dmd::err::BAD_FILE, "Basic file opening failed.");
}

bool vorbis_file::do_read(void* buf, size_t wanted_bytes, size_t &got_bytes)
{
	int ret = true;
	long decoded = -1; // ov_read really returns long
	size_t size_left = wanted_bytes;
	int current_section = 0; // Initialize?
	char *cbuffer = (char*)buf;

	got_bytes = 0;

	if(!opened) return false; // Unnecessary paranoia?
	while(size_left > 0)
	{
		decoded = ov_read(&handle, cbuffer, size_left, 0, 2, 1/*sgned*/, &current_section);
		if(decoded <= 0)
		{
			MDEBUG("play: got eof (%li)", decoded);
			break;
		}
		if(current_section != 0)	MERROR("current_section changed to %i ... this may include a format change, which I do not support.", current_section);

		cbuffer += decoded;
		size_left -= decoded;
	}
	if(decoded < 0) ret = false;

	SDEBUG("filled");
	got_bytes = wanted_bytes - size_left;
	return ret;
}

bool vorbis_file::do_seek(off_t pos)
{
	int ret = true;

	SDEBUG("do_seek called");
	if(opened)
	{
		off_t len = do_length();
		if(pos >= len) pos = len;
		int seekret = ov_pcm_seek(&handle, pos);
		MDEBUG("seek returned %i", seekret);
		switch(seekret)
		{
			case 0:
			position = pos;
			break;
			case OV_ENOSEEK:  // Bitstream is not seekable.
			case OV_EINVAL:   // Invalid argument value; possibly called with an Ogghandle structure that isn't open.
			case OV_EREAD:    // A read from media returned an error.
			case OV_EFAULT:   // Internal logic fault; indicates a bug or heap/stack corruption.
			case OV_EBADLINK: // Invalid stream section supplied to libvorbisfile, or the requested link is corrupt.
			default: ret = false;
		}
	}
	else ret = false;

	return ret;
}

off_t vorbis_file::do_length()
{
	if(opened) return ov_pcm_total(&handle,-1);
	else return 0;
}
