/*
	in_sine: sine tone generator input device
	
	part of DerMixD
	(c)2006-2013 Thomas Orgis, licensed under GPLv2
*/

#include "basics.hxx"
#include "in/drv/in_sine.hxx"
#include "debug.hxx"

#include <cmath>
#include <cstring>

using namespace std;

#define ME "sine_file"

sine_file::sine_file(): input_file(input::sine)
{
	CONSTRUCT(ME);
	frequency = 440;
	format.rate = 44100;
	format.channels = 1;
	table_pos = 0;
	i_am_fine = true;
}

sine_file::~sine_file()
{
	DESTRUCT(ME);
}

bool sine_file::do_open(const string location)
{
	MDEBUG("loading resource %s", location.c_str());
	size_t toffset = strlen(input::names[input::sine])+3;
	// sine://<freq>@<sampling>
	if(location.length() >= toffset)
	{
		if(location.length() > toffset)
		{
			string::size_type i;
			i = location.find("@", toffset);
			if(i != string::npos)
			{
				if(i > toffset) frequency = strtoul(location.substr(toffset, i-toffset).c_str(), NULL, 10);
				unsigned long temp = strtoul(location.substr(i+1, location.length()-i).c_str(), NULL, 10);
				if(temp > 0) format.rate = temp;
			}
			else frequency = strtoul(location.c_str()+toffset, NULL, 10);
		}
		// Use currently active settings if nothing specified in URL.

		format.channels = 1;
		MDEBUG("have new settings: %luHz tone sampled with %luHz", frequency, format.rate);
		if(!build_table())
		return err.occur(dmd::err::NOMEM, "error building sine table");
	}
	else
	return err.occur(dmd::err::BAD_PAR, "no good sine pseudo-URL");

	return true;
}

// one should consider periodicity of sine
// sin(2pi * freq * pos/rate) == sin(2pi * (freq*pos/rate-floor(freq*pos/rate)) )
// with rate discrete values of the fractional term
// so, a table 0..rate-1, lookup index: pos % rate
// Bollocks! You weren't awake when writing this, right?
// I need to sample one period of the wave, rate determines how many samples that is.
// freq = periods per second
// rate = samples per second
// rate/freq = samples per period
// example: 100 Hz at 44100 Hz sampling; 100 periods in one second, so 441 samples make one period
// What with extreme cases? 20000 Hz ... 44100/20000=2.2050
// Hah, I need to take a multiple of that, being resonably close to an integer. Need to define a maximal phase error.
bool sine_file::build_table()
{
	table_pos = 0;
	// at 1/error samples a period, dropping/adding one sample is OK
	const double phase_error = 0.0001;
	// Have some limit ...
	const unsigned int max_periods = 100;
	double samples_per_period = (double)format.rate/(double)frequency;
	size_t tablesize;
	unsigned int buffer_periods = 1; // number of periods in buffer
	while( fabs((double)(tablesize = round2size(buffer_periods*samples_per_period)) - buffer_periods*samples_per_period)/buffer_periods > phase_error*samples_per_period)
	{
		if(buffer_periods < max_periods)
		buffer_periods++;
		else break;
	}
	if(table.reallocate(1, tablesize) == tablesize)
	{
		// The actual frequency gets modified.
		samples_per_period = (double)table.size/buffer_periods;
		MDEBUG("actual freq: %g Hz", (double)format.rate/samples_per_period);
		// compromise: smooth onset for low frequencies,
		// but also good sampling of high freqs
		// extreme case: 2 samples per period should trigger max/min amplitude, not both zero
		// so, center the samples within one sine period
		// That's achieved by adding 0.5 to the counter.
		SDEBUG("filling allocated table");
		for(unsigned long i = 0; i < table.size; ++i)
		{

			float scale = sin(2.0*3.14159265358979323846 * ((double)i + 0.5) / samples_per_period);
			audio::scale_to_sample(scale, audio::audio_io_code, table.ptr+i);
		}

		SDEBUG("done filling table");
		return true;
	}
	else return false;
}

// only error that can occur is missing / wrong table... I suppose a successful load prevents this
// and without load you shall not play...
bool sine_file::do_read(void* buf, size_t wanted_bytes, size_t &got_bytes)
{
	size_t wanted_samples = format.bytes2samples(wanted_bytes);
	audio_io_type *abuf = (audio_io_type*)buf;
	MDEBUG("playing with %u channels", format.channels);
	for(size_t i = 0; i < wanted_samples; ++i)
	abuf[i] = table.ptr[(table_pos+i) % table.size];

	table_pos += wanted_samples;
	table_pos %= table.size;
	SDEBUG("filled");
	got_bytes = wanted_bytes;
	return true;
}

bool sine_file::do_seek(off_t pos)
{
	position = pos;
	table_pos = pos % table.size;
	return true;
}

void sine_file::do_close()
{
	table_pos = 0;
	// Not clearing table here... memory can be reused.
	// If you want to get rid of the table, delete the sine_file.
}
