/*
	in_libmpg123: libmpg123 input device
	
	part of DerMixD
	(c)2007-2011 Thomas Orgis, licensed under GPLv2
*/

#include "basics.hxx"
#include "in/drv/in_libmpg123.hxx"
#include "param_init.hxx"
#include "tstring.hxx"

#include <mpg123.h>

#define ME "libmpg123_file"
#define HANDLE ((mpg123_handle*)handle)

#include "debug.hxx"

using std::string;
using std::vector;

// Helper class to facilitate automatic single-threaded initialization of mpg123 library on program startup (and cleanup on end).
class mpg123_ignition
{
	private:
	int errcode;
	public:
	mpg123_ignition()   { errcode = mpg123_init();               };
	~mpg123_ignition()  { mpg123_exit();                         };
	bool ready()        { return (errcode == MPG123_OK);         };
	const char* errmsg(){ return mpg123_plain_strerror(errcode); };
};
// One static instance here and the C++ runtime does the job for us.
static mpg123_ignition mpg123_lib;

// Here is the real work.

libmpg123_file::libmpg123_file(): input_file(input::mpg123)
{
	int merr;
	long rva;

	// Get handle with chosen decoder.
	CONSTRUCT("libmpg123_file");
	handle = mpg123_new(param("mpg123", "decoder") != "" ? param("mpg123", "decoder").c_str() : NULL, &merr);
	MDEBUG("[libmpg123_file %p] handle %p, err %i", this, HANDLE, merr);
	if(handle == NULL)
	{
		string errmsg;
		strprintf(errmsg, "mpg123 handle creation failed: %s", mpg123_plain_strerror(merr));
		if(!mpg123_lib.ready()) strprintf(errmsg, " (mpg123 ignition failed, even: %s)", mpg123_lib.errmsg());

		err.occur(dmd::err::SETUP, errmsg);
		return;
	}

	// Settle volume adjustment choice.
	rva = (param("mpg123", "rva") == "")
	    ? MPG123_RVA_OFF
	    : ((param("mpg123", "rva") == "album") ? MPG123_RVA_ALBUM : MPG123_RVA_MIX);
	mpg123_param(HANDLE, MPG123_RVA, rva, 0);
	mpg123_param(HANDLE, MPG123_RESYNC_LIMIT, param.as_lint("mpg123", "resync_limit"), 0);

	i_am_fine = true;
}

libmpg123_file::~libmpg123_file()
{
	DESTBEGIN("libmpg123_file");
	do_close();
	mpg123_delete(HANDLE);
	DESTEND("libmpg123_file");
}

bool libmpg123_file::do_read(void* buf, size_t order_bytes, size_t &got_bytes)
{
	int merr = mpg123_read(HANDLE, (unsigned char *)buf, order_bytes, &got_bytes);

	// Err could be MPG123_DONE, that should be fine, only MPG123_ERR is really bad.
	if(merr == MPG123_ERR) return err.occur(dmd::err::IO, string("mpg123 read failed: ") + mpg123_strerror(HANDLE));

	return true;
}

bool libmpg123_file::do_open(const string location)
{
	int merr;
	int ch;
	long int rt;
	int enc;
	const long *rates;
	size_t rnum;

	mpg123_format_none(HANDLE);
	mpg123_rates(&rates, &rnum);
	for(size_t r=0; r<rnum; ++r) mpg123_format(HANDLE, rates[r], MPG123_STEREO|MPG123_MONO, MPG123_ENC_SIGNED_16);

	MDEBUG("[libmpg123_file %p] opening %s", this, location.c_str());
	merr = mpg123_open(HANDLE, location.c_str());
	if(merr != MPG123_OK)
	{
		MERROR("[libmpg123_file %p] failed to open!", this);
		return err.occur(dmd::err::BAD_FILE, string("failed to open file: ") + mpg123_strerror(HANDLE));
	}
	if(mpg123_getformat(HANDLE, &rt, &ch, &enc) != MPG123_OK)
	{
		return err.occur(dmd::err::BAD_FILE, string("cannot peek for format: ") + mpg123_strerror(HANDLE));
	}
	format.rate = rt > 0 ? rt : 0;
	format.channels = ch > 0 ? ch : 0;
	// Format is not allowed to change!
	mpg123_format_none(HANDLE);
	mpg123_format(HANDLE, (long)format.rate, format.channels == 2 ? MPG123_STEREO : MPG123_MONO, MPG123_ENC_SIGNED_16);

	MDEBUG("[libmpg123_file %p] opened %s", this, location.c_str());
	return true;
}

void libmpg123_file::do_close()
{
	mpg123_close(HANDLE);
}

bool libmpg123_file::do_seek(off_t pos)
{
	position = mpg123_seek(HANDLE, pos, SEEK_SET);
	if(position >= 0)
	{
		MDEBUG("mpg123 seek to %lli", (long long)position);
		return true;
	}
	else
	{
		return err.occur(dmd::err::IO, string("mpg123 could not seek: ") + mpg123_strerror(HANDLE));
	}
}

off_t libmpg123_file::do_length(void)
{
	MDEBUG("[libmpg123_file %p] scanning", this);
	mpg123_scan(HANDLE);
	return mpg123_length(HANDLE);
}

bool libmpg123_file::do_eq(const vector<float> &val)
{
	// Wow... do I really have have only one band for bass and mid and all the rest for treble?
	// It worked like this for quite some time in mpg123 itself...

	for(int cn=0; cn <  1; ++cn)
	{
		mpg123_eq(HANDLE, MPG123_LEFT,  cn, val[0]);
		mpg123_eq(HANDLE, MPG123_RIGHT, cn, val[0]);
	}
	for(int cn=1; cn <  2; ++cn)
	{
		mpg123_eq(HANDLE, MPG123_LEFT,  cn, val[1]);
		mpg123_eq(HANDLE, MPG123_RIGHT, cn, val[1]);
	}
	for(int cn=2; cn < 32; ++cn)
	{
		mpg123_eq(HANDLE, MPG123_LEFT,  cn, val[2]);
		mpg123_eq(HANDLE, MPG123_RIGHT, cn, val[2]);
	}

	return true;
}
